/*
 * Copyright 2021 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software implementation of MIFARE DUOX application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7464 $
 * $Date: 2025-08-29 16:56:35 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 14. Sept 2021
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_MFDUOX_SW
#include <string.h>
#include <stdlib.h>
#include <phTools.h>
#include <phKeyStore.h>
#include <phCryptoASym.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>

#ifdef NXPBUILD__PHAL_VCA
#include <phalVca.h>
#endif /* NXPBUILD__PHAL_VCA */
#include <phalMfDuoX.h>
#include "../phalMfDuoX_Int.h"
#include "phalMfDuoX_Sw.h"
#include "phalMfDuoX_Sw_Int.h"





phStatus_t phalMfDuoX_Sw_Init(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams,
    void * pKeyStoreDataParams, void * pCryptoDataParamsASym, void * pCryptoDataParamsEnc, void * pCryptoDataParamsMac,
    void * pCryptoRngDataParams, void * pTMIDataParams, void * pVCADataParams, uint8_t * pCmdBuf, uint16_t wCmdBufSize,
    uint8_t * pPrsBuf, uint16_t wPrsBufSize)
{
    /* DataParams Size Validation */
    if(sizeof(phalMfDuoX_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDUOX);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pPalMifareDataParams, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pKeyStoreDataParams, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsASym, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsEnc, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsMac, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoRngDataParams, PH_COMP_AL_MFDUOX);
#ifdef NXPBUILD__PH_TMIUTILS
    PH_ASSERT_NULL_DATA_PARAM(pTMIDataParams, PH_COMP_AL_MFDUOX);
#endif /* NXPBUILD__PH_TMIUTILS */
#ifdef NXPBUILD__PHAL_VCA
    PH_ASSERT_NULL_DATA_PARAM(pVCADataParams, PH_COMP_AL_MFDUOX);
#endif /* NXPBUILD__PHAL_VCA */

    PH_ASSERT_NULL_PARAM(pCmdBuf, PH_COMP_AL_MFDUOX);
    PH_ASSERT_NULL_PARAM(pPrsBuf, PH_COMP_AL_MFDUOX);

    if(wCmdBufSize < PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_MFDUOX);

    if(wPrsBufSize < PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_MFDUOX);

    if(wPrsBufSize < wCmdBufSize)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_MFDUOX);

    pDataParams->wId = PH_COMP_AL_MFDUOX | PHAL_MFDUOX_SW_ID;

    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
    pDataParams->pCryptoDataParamsASym = pCryptoDataParamsASym;
    pDataParams->pCryptoDataParamsEnc = pCryptoDataParamsEnc;
    pDataParams->pCryptoDataParamsMac = pCryptoDataParamsMac;
    pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
    PH_UNUSED_VARIABLE(pCryptoDataParamsASym);
    PH_UNUSED_VARIABLE(pCryptoDataParamsEnc);
    PH_UNUSED_VARIABLE(pCryptoDataParamsMac);
    PH_UNUSED_VARIABLE(pCryptoRngDataParams);

#ifdef NXPBUILD__PH_TMIUTILS
    pDataParams->pTMIDataParams = pTMIDataParams;
#endif /* NXPBUILD__PH_TMIUTILS */
#ifdef NXPBUILD__PHAL_VCA
    pDataParams->pVCADataParams = pVCADataParams;
#endif /* NXPBUILD__PHAL_VCA */

    pDataParams->pCmdBuf = pCmdBuf;
    pDataParams->wCmdBufSize = wCmdBufSize;
    pDataParams->wCmdBufLen = 0;
    pDataParams->wCmdBufOffset = 0;

    pDataParams->pPrsBuf = pPrsBuf;
    pDataParams->wPrsBufSize = wPrsBufSize;
    pDataParams->wPrsBufLen = 0;
    pDataParams->wPrsBufOffset = 0;

    (void) memset(pCmdBuf, 0x00, PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM);
    (void) memset(pPrsBuf, 0x00, PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM);
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    (void) memset(phalMfDuoX_Sw_ZeroIv, 0x00, sizeof(phalMfDuoX_Sw_ZeroIv));
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));
    (void) memset(pDataParams->aTi, 0x00, PHAL_MFDUOX_SIZE_TI);
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    (void) memset(pDataParams->aAid, 0x00, 3U);

    pDataParams->wCmdCtr = 0;
    pDataParams->wAdditionalInfo = 0x0000;
    pDataParams->bWrappedMode = PH_OFF;
    pDataParams->bShortLenApdu = PH_ON;                        /* By default, short length APDU format is used. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_INVALID;
    pDataParams->bAuthState = PHAL_MFDUOX_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFFU;
    pDataParams->bPICCDataComplete = PH_OFF;
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    pDataParams->bHasMACProcessed = PH_OFF;
    pDataParams->bIsENCPending = PH_OFF;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_DeInit(phalMfDuoX_Sw_DataParams_t * pDataParams)
{
    phalMfDuoX_Sw_ResetAuthentication(pDataParams);

    pDataParams->wCmdBufSize = 0;
    pDataParams->wPrsBufSize = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





#ifdef NXPBUILD__PHAL_MFDUOX_NDA
/* MIFARE DUOX Authentication commands ---------------------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_ISOGeneralAuthenticate(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bCARootKeyNo,
    uint16_t wKeyNo_PrivA, uint16_t wKeyPos_PrivA, uint8_t * pOptsA, uint8_t bOptsALen, uint8_t * pCertA, uint16_t wCertALen,
    uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t * pE_PubB, uint16_t * pE_PubBLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wAPDU_Mode = 0;
    uint16_t    PH_MEMLOC_REM wIsShortLen = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;
    uint8_t     PH_MEMLOC_REM bLE_Len = 0;

    uint8_t     PH_MEMLOC_REM aLE[2U];
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Clear LE buffer. */
    (void) memset(aLE, 0x00, sizeof(aLE));

    /* Get the current APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_GetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, &wAPDU_Mode));
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_GetConfig(pDataParams, PHAL_MFDUOX_SHORT_LENGTH_APDU, &wIsShortLen));

    /* Validate Option parameter. */
    if((wOption != PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE) && (wOption != PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL))
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);

    /* Update LE Length. */
    if(pExpRspLen != NULL)
    {
        bLE_Len = bExpRspLen;
        (void) memcpy(aLE, pExpRspLen, bLE_Len);
    }
    else
    {
        if(wIsShortLen == PH_ON)
        {
            bLE_Len = 1U;
        }
        else
        {
            bLE_Len = 2U;
        }
    }

    /* Update Command code to DataParams. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_GENERAL;

    /* Update AuthState to Not Authenticated. */
    pDataParams->bAuthState = PHAL_MFDUOX_NOT_AUTHENTICATED;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Get the CurveID from Known Private Key (Priv.A). */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetCurveID(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_PrivA,
        wKeyPos_PrivA,
        &bCurveID));

    /* Generate Ephemeral Key pair (E.Priv.A and E.Pub.A). */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_GenerateKeyPair(
        pDataParams->pCryptoDataParamsASym,
        bCurveID));

    /* Disable Wrapped format.
     * This is required for internal PICC exchange interface to stop from framing the ISO7816 format.
     */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_SetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, PH_OFF));

    /* Perform ISOGeneral Authentication Final ----------------------------------------------------------------------------------------- */
    if(wOption & PHAL_MFDUOX_ISO_GENERAL_AUTHENTICATE_FINAL)
    {
        wStatus = phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Final(
            pDataParams,
            wKeyNo_PrivA,
            wKeyPos_PrivA,
            bCARootKeyNo,
            bCurveID,
            pOptsA,
            bOptsALen,
            pE_PubB,
            *pE_PubBLen,
            pCertA,
            wCertALen,
            aLE,
            bLE_Len,
            ppResponse,
            pRspLen);
    }

    /* Perform ISOGeneral Authentication ----------------------------------------------------------------------------------------------- */
    else
    {
        /* Perform ISOGeneral Authentication Part 1 ------------------------------------------------------------------------------------ */
        wStatus = phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Part1(
            pDataParams,
            bCARootKeyNo,
            pOptsA,
            bOptsALen,
            aLE,
            bLE_Len,
            &pResponse,
            &wRspLen);

        /* Copy Received Ephemeral Public Key to the parameter. */
        (void) memcpy(pE_PubB, pResponse, wRspLen);
        *pE_PubBLen = wRspLen;

        /* Proceed if ISOGeneral Authenticate Part1 is Success. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Perform ISOGeneral Authentication Part 2 -------------------------------------------------------------------------------- */
            wStatus = phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Part2(
                pDataParams,
                wKeyNo_PrivA,
                wKeyPos_PrivA,
                bCurveID,
                pOptsA,
                bOptsALen,
                pResponse,
                wRspLen,
                pCertA,
                wCertALen,
                aLE,
                bLE_Len,
                ppResponse,
                pRspLen);
        }
        else
        {
            /* Nothing to do here */
        }
    }

#ifdef NXPBUILD__PHAL_VCA
    /* Update the authentication state if VCA PC feature is required by the application. */
    if((pDataParams->pVCADataParams != NULL) && ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Set the Session key and IV for Virtual Card which is valid for this authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_SetSessionKeyUtility(
            (phalVca_Sw_DataParams_t *) pDataParams->pVCADataParams,
            pDataParams->aSesAuthMACKey,
            PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_GENERAL));
    }
    else
    {
        /* No VCA Feature usage. */
    }
#endif /* NXPBUILD__PHAL_VCA */

    /* Revert back the APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phalMfDuoX_Sw_SetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, wAPDU_Mode));

    return wStatus;
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

phStatus_t phalMfDuoX_Sw_ISOInternalAuthenticate(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bPrivKeyNo, uint8_t bCurveID,
    uint8_t * pPubBKey, uint16_t wPubBKeyLen, uint8_t * pOptsA, uint8_t bOptsALen, uint8_t * pExpRspLen, uint8_t bExpRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wAPDU_Mode = 0;
    uint16_t    PH_MEMLOC_REM wIsShortLen = 0;
    uint16_t    PH_MEMLOC_REM wLC = 0;
    uint8_t     PH_MEMLOC_REM bLE_Len = 0;
    uint8_t     PH_MEMLOC_REM bRndALen = 0;
    uint8_t     PH_MEMLOC_REM bRndBLen = 0;
    uint8_t     PH_MEMLOC_REM bSigLen = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    uint8_t     PH_MEMLOC_REM aLE[2];
    uint8_t     PH_MEMLOC_REM aCont[2] = { 0xF0U, 0xF0U };
    uint8_t     PH_MEMLOC_REM *pRndA = NULL;
    uint8_t     PH_MEMLOC_REM *pRndB = NULL;
    uint8_t     PH_MEMLOC_REM *pSignature = NULL;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Validate Curve ID */
    PHAL_MFDUOX_VALIDATE_CURVE(bCurveID);

    /* Clear LE buffer. */
    (void) memset(aLE, 0x00, sizeof(aLE));

    /* Get the current APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_GetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, &wAPDU_Mode));
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_GetConfig(pDataParams, PHAL_MFDUOX_SHORT_LENGTH_APDU, &wIsShortLen));

    /* Disable Wrapped format.
     * This is required for internal PICC exchange interface to stop from framing the ISO7816 format.
     */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_SetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, PH_OFF));

    /* Compute LC */
    wLC = (uint16_t) (bOptsALen + 2U /* AuthDOHdr */ + 18U /* RndA TLV */);

    /* Update LE Length. */
    if(pExpRspLen != NULL)
    {
        bLE_Len = bExpRspLen;
        (void) memcpy(aLE, pExpRspLen, bLE_Len);
    }
    else
    {
        if(wIsShortLen == PH_ON)
        {
            bLE_Len = 1U;
        }
        else
        {
            bLE_Len = 2U;
        }

        /* Force Extend LE information to be exchanged if greater than 255 bytes. */
        if(wLC > 255U)
        {
            bExpRspLen = 2U;
        }
    }

    /* Update Command code to DataParams. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_INTERNAL;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the pointer. */
    pRndA = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 4U)];
    pRndB = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];
    pSignature = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    /* Frame Cmd.ISOInternal Authenticate command -------------------------------------------------------------------------------------- */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_ISO7816_GENERIC_CLA;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_INTERNAL;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_P1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bPrivKeyNo;

    /* Frame Extended LC. */
    if(bLE_Len > 1U)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wLC & 0xFF00U) >> 8U);
    }
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wLC & 0x00FFU));

    /* Add Command information for exchange. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        0,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Add OptsA information for exchange. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        0,
        PH_OFF,
        pOptsA,
        bOptsALen,
        NULL,
        NULL,
        NULL));

    /* Reset Command Buffer */
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Generate RndA */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Seed(pDataParams->pCryptoRngDataParams, pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE));
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, pRndA));
    bRndALen = PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Add AuthDOHdr (Authentication Data Object Header) to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_INTERNAL_AUTH_DO_HDR_TAG;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (2U /* Tag (T) + Length (L) */ + bRndALen /* Length of RndA */);

    /* Add Authentication Data Object for RndA. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_INTERNAL_RND_TAG;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bRndALen;

    /* Copy RndA to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRndA, bRndALen);
    PHAL_MFDUOX_CMD_BUF_LEN += bRndALen;

    /* Add LE. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], aLE, bLE_Len);
    PHAL_MFDUOX_CMD_BUF_LEN += bLE_Len;

    /* Exchange command to PICC ---------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_PICC_STATUS_WRAPPED),
        0,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRspLen,
        NULL));

    /* Validate AuthDOHdr from response. */
    if(pResponse[bOffset] != PHAL_MFDUOX_AUTH_ISO_INTERNAL_AUTH_DO_HDR_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);
    bOffset = 2 /* AuthDOHdr TL */;

    /* Validate Tag information from RndB TLV */
    if(pResponse[bOffset++] != PHAL_MFDUOX_AUTH_ISO_INTERNAL_RND_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Extract RndB information. */
    bRndBLen = pResponse[bOffset++];
    (void) memcpy(pRndB, &pResponse[bOffset], bRndBLen);
    bOffset += bRndBLen;

    /* Validate Tag information from Signature TLV */
    if(pResponse[bOffset++] != PHAL_MFDUOX_AUTH_ISO_INTERNAL_SIGNATURE_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Extract Signature information. */
    bSigLen = pResponse[bOffset++];
    (void) memcpy(pSignature, &pResponse[bOffset], bSigLen);

    /* Load public Key to ASymmetric Crypto component ---------------------------------------------------------------------------------- */
    /* Update CurveID based on CryptoASym component. */
    bCurveID = (uint8_t) ((bCurveID == PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256) ? PH_CRYPTOASYM_CURVE_ID_SECP256R1 :
        PH_CRYPTOASYM_CURVE_ID_BRAINPOOL256R1);

    /* Load Public Key for verification. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKeyDirect(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | bCurveID),
        pPubBKey,
        wPubBKeyLen));

    /* Verify the Signature ------------------------------------------------------------------------------------------------------------ */

    /* Add Constant for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        aCont,
        2U,
        NULL,
        0));

    /* Add Opts.A for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pOptsA,
        bOptsALen,
        NULL,
        0));

    /* Add RndB for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pRndB,
        bRndBLen,
        NULL,
        0));

    /* Add RndA for Hashing and verify the signature. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pRndA,
        bRndALen,
        pSignature,
        bSigLen));

    /* Revert back the APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phalMfDuoX_Sw_SetConfig(pDataParams, PHAL_MFDUOX_WRAPPED_MODE, wAPDU_Mode));

    return wStatus;
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_AuthenticateEv2(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bFirstAuth, uint16_t wOption, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen, uint8_t * pPcdCapsIn, uint8_t bLenPcdCapsIn,
    uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;
    uint8_t     PH_MEMLOC_REM *pRndA = NULL;
    uint8_t     PH_MEMLOC_REM *pRndB = NULL;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    /* Validate AuthType flag. */
    if(bFirstAuth > PHAL_MFDUOX_AUTH_FIRST)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Validate Diversification options. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_OPTIONS(wOption);

    /* Validate Diversification Length. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivLen);
#endif

    /* Clear buffers */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];
    pRndA = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pRndB = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate Keytype */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Diversify the Key. */
    if((wOption != PHAL_MFDUOX_NO_DIVERSIFICATION) && (bDivLen != 0x00))
    {
        /* Key is diversified and put back in bKey */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
            pDataParams->pCryptoDataParamsEnc,
            wOption,
            pKey,
            wKeyType,
            pDivInput,
            bDivLen,
            pKey));
    }

    /* Load the key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey,
        wKeyType));

    /* Frame Command PART1 ------------------------------------------------------------------------------------------------------------- */

    /* Frame Command Code */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (bFirstAuth ? PHAL_MFDUOX_CMD_AUTHENTICATE_EV2_FIRST :
        PHAL_MFDUOX_CMD_AUTHENTICATE_EV2_NON_FIRST);

    /* Frame Card's key number to use. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNoCard;

    /* Frame Input PCD Capabilities. */
    if(bFirstAuth)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bLenPcdCapsIn;

        /* Copy PCD Capabilities to Command buffer. */
        if(pPcdCapsIn != NULL)
        {
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pPcdCapsIn, bLenPcdCapsIn);
            PHAL_MFDUOX_CMD_BUF_LEN += bLenPcdCapsIn;
        }

        pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_EV2_FIRST;
    }
    else
    {
        /* No PCD Capabilities for Following Auth. */

        pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_EV2_NON_FIRST;
    }

    /* Exchange Part1 Command. */
    wStatus = phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_COMPLETE,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
        return wStatus;
    }

    /* Validate Response Buffer length. */
    if(wRespLen != PH_CRYPTOSYM_AES_BLOCK_SIZE)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);
    }

    /* Process PART1 Response ---------------------------------------------------------------------------------------------------------- */
    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Decrypt the RndB received from PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        pResponse,
        wRespLen,
        pRndB));

    /* Frame Command PART2 ------------------------------------------------------------------------------------------------------------- */
    /* Clear command buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Add Command Code to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_RESP_ADDITIONAL_FRAME;

    /* Generate RndA */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Seed(pDataParams->pCryptoRngDataParams, pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE));
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, pRndA));

    /* Copy RndA to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRndA, PH_CRYPTOSYM_AES_BLOCK_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Copy RndB to Command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Shift RndB by 8 bits. */
    phalMfDuoX_Int_RotateLeft(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], PH_CRYPTOSYM_AES_BLOCK_SIZE, 1U);

    /* Increment command buffer. */
    PHAL_MFDUOX_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Encrypt RndA + RndB' */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        &PHAL_MFDUOX_CMD_BUF[1U],
        (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN - 1U),
        &PHAL_MFDUOX_CMD_BUF[1U]));

    /* Exchange Part2 Command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_COMPLETE,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL));

    /* Process PART2 Response ---------------------------------------------------------------------------------------------------------- */
    /* Validate Response Length. */
    if(wRespLen != (bFirstAuth ? (2U * PH_CRYPTOSYM_AES_BLOCK_SIZE) : PH_CRYPTOSYM_AES_BLOCK_SIZE))
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);
    }

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Decrypt Response to extract Decrypt TI || RndA' || PDCaps || PCDCaps */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        pResponse,
        wRespLen,
        pResponse));

    /* Define the offset to use. */
    bOffset = (uint8_t) (bFirstAuth ? 4 : 0);

    /* Shift RndA by 8 bits. */
    phalMfDuoX_Int_RotateRight(&pResponse[bOffset], PH_CRYPTOSYM_AES_BLOCK_SIZE, 1);

    /* Validate RndA ------------------------------------------------------------------------------------------------------------------- */
    if(memcmp(pRndA, &pResponse[bOffset], PH_CRYPTOSYM_AES_BLOCK_SIZE) != 0)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);

        /* Authentication failed */
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFDUOX);
    }

    /* Extract TI, PCD and PD Capabilities */
    if(bFirstAuth)
    {
        (void) memcpy(pDataParams->aTi, &pResponse[0], PHAL_MFDUOX_SIZE_TI);
        (void) memcpy(pPdCapsOut, &pResponse[PHAL_MFDUOX_SIZE_TI + PH_CRYPTOSYM_AES_BLOCK_SIZE], PHAL_MFDUOX_SIZE_CAPS);
        (void) memcpy(pPcdCapsOut, &pResponse[PHAL_MFDUOX_SIZE_TI + PH_CRYPTOSYM_AES_BLOCK_SIZE + PHAL_MFDUOX_SIZE_CAPS],
            PHAL_MFDUOX_SIZE_CAPS);
    }
    else
    {
        /* Nothing to do for Following Auth */
    }

    /* Generate the Session Keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateSessionKeys_Sym(pDataParams, wKeyType, pRndA, pRndB));

    /* Session key is generated. */
    if(bFirstAuth)
    {
        pDataParams->wCmdCtr = 0x00;
    }
    else
    {
        /* Nothing to do for Following Auth */
    }

    pDataParams->bAuthState = PHAL_MFDUOX_EV2_AUTHENTICATED;
    pDataParams->bKeyNo = bKeyNoCard;
    pDataParams->wKeyType = wKeyType;

    /* Load the Session ENC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pDataParams->aSesAuthENCKey,
        wKeyType));

    /* Load the Session MAC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pDataParams->aSesAuthMACKey,
        wKeyType));

    /* Set the keep Iv ON */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_KEEP_IV,
        PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

    /* Update the authentication state if VCA PC feature is required by the application. */
    if(pDataParams->pVCADataParams != NULL)
    {
        /* Set the Session key and IV for Virtual Card which is valid for this authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_SetSessionKeyUtility(
            (phalVca_Sw_DataParams_t *) pDataParams->pVCADataParams,
            pDataParams->aSesAuthMACKey,
            PHAL_MFDUOX_CMD_AUTHENTICATE_EV2_FIRST));
    }
    else
    {
        /* No VCA Feature usage. */
    }

    /* Clear the local buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;
    PHAL_MFDUOX_CMD_BUF_OFFSET = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;
    PHAL_MFDUOX_PRS_BUF_OFFSET = 0;

    PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_OFF;
    PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */





/* MIFARE DUOX Memory and Configuration management commands ---------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_FreeMem(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t ** ppMemInfo, uint16_t * pMemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_FREE_MEM;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Command Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Exchange Cmd.FreeMem information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        &pDataParams->bCmdCode,
        1U,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppMemInfo,
        pMemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_Format(phalMfDuoX_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_FORMAT;

    /* Clear Command Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Exchange Cmd.Format information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        &pDataParams->bCmdCode,
        1U,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_SetConfiguration(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_SET_CONFIGURATION;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_SET_CONFIGURATION;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;

    /* Exchange Cmd.SetConfiguration information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        PHAL_MFDUOX_COMMUNICATION_MAC,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        bDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetConfiguration(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bExchangeOption,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Validate ExchangeOption value. */
    if((bExchangeOption != PH_OFF) && (bExchangeOption != PH_ON))
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_CONFIGURATION;

    /* Clear Command Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame Command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    if(bExchangeOption == PH_ON)
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;

    /* Exchange Cmd.GetConfiguration information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_MAC,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

phStatus_t phalMfDuoX_Sw_GetVersion(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppVerInfo,
    uint16_t * pVerInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_VERSION;

    /* Clear Command Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame GetVersion command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_VERSION;

    /* Append Option information. */
    if(bOption != PHAL_MFDUOX_GET_VERSION_EXCLUDE_FAB_ID)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;
    }

    /* Exchange Cmd.GetVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppVerInfo,
        pVerInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_GetCardUID(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t ** ppUid, uint16_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_CARD_UID;

    /* Clear Command Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Exchange Cmd.GetVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        &pDataParams->bCmdCode,
        1U,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppUid,
        pUidLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */





/* MIFARE DUOX Symmetric Key management commands --------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_ChangeKey(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo,
    uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput,
    uint8_t bDivLen)
{
    uint16_t    PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwCRC = 0;
    uint16_t    PH_MEMLOC_REM wOldKeyType;
    uint16_t    PH_MEMLOC_REM wNewKeyType;
    uint16_t    PH_MEMLOC_REM wDivOption = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t     PH_MEMLOC_REM bKeySize = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bResetAuth = PH_OFF;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = PH_OFF;

    uint8_t     PH_MEMLOC_REM * pCmdData = NULL;
    uint8_t     PH_MEMLOC_REM * pOldKey = NULL;
    uint8_t     PH_MEMLOC_REM * pNewKey = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM aAppId[3U] = { 0x00, 0x00, 0x00 };
    /* Change key should also take care of changing other keys at PICC level like the Proximity check key, Transaction MAC key */
    if(memcmp(pDataParams->aAid, aAppId, 3U) == 0x00)
    {
        /* Only if selected Aid is 0x000000, PICC level key change is targeted. */
        if(((bKeyNoCard & 0x3FU) != 0x00) &&

            /* PICC DAMAuthKey, DAMMACKey, DAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x10U) && ((bKeyNoCard & 0x3FU) != 0x11U) && ((bKeyNoCard & 0x3FU) != 0x12U) &&

            /* PICC NXPDAMAuthKey, NXPDAMMACKey, NXPDAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x18U) && ((bKeyNoCard & 0x3FU) != 0x19U) && ((bKeyNoCard & 0x3FU) != 0x1AU) &&

            /* PICC VCConfigurationKey, VCProximityKey */
            ((bKeyNoCard & 0x3FU) != 0x20U) && ((bKeyNoCard & 0x3FU) != 0x21U))
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }

        /* If PICC MasterKey holds AES128 or AES256 information. */
        else if(((bKeyNoCard & 0xC0U) != 0xC0U) && ((bKeyNoCard & 0xC0U) != 0x80U) && ((bKeyNoCard & 0xC0U) != 0x00))
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }
        else
        {
            /*Do Nothing. This is for PRQA compliance */
        }
    }
    else
    {
        /* Key numbers between 0D and 20 are not allowed for APP level, also key numbers above 22 are not allowed.
         * if AID 0x000000 is not selected.
         */
        if(IS_INVALID_APP_KEY(bKeyNoCard) && ((bKeyNoCard & 0x3FU) != 0x21U))
        {
            /* Invalid application key specified */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }
    }

    /* Validate Diversification Length. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivLen);

    /* Validate Option. */
    if((wOption == 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear buffer. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer for the keys. */
    pNewKey = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pOldKey = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Set buffer to use for Command Data. */
    pCmdData = pNewKey;

    /* Frame ChangeKey command information --------------------------------------------------------------------------------------------- */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CHANGE_KEY;

    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CHANGE_KEY;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNoCard;

    /* Get Old and New Keys ------------------------------------------------------------------------------------------------------------ */
    /* Get the Current Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wCurrKeyNo,
        wCurrKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pOldKey,
        &wOldKeyType));

    /* Verify KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wOldKeyType);

    /* Get the New Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wNewKeyNo,
        wNewKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pNewKey,
        &wNewKeyType));

    /* Verify KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wNewKeyType);

    /* Set the Size of the Key. */
    bKeySize = phKeyStore_GetKeySize(wNewKeyType);

    /* Diversify Keys ------------------------------------------------------------------------------------------------------------------ */
    /* Diversify Current and New Keys. */
    if((wOption != PHAL_MFDUOX_NO_DIVERSIFICATION) && (bDivLen != 0x00))
    {
        /* Diversify Old Key */
        if(wOption & PHAL_MFDUOX_CHGKEY_DIV_OLD_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_MFDUOX_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pOldKey,
                wOldKeyType,
                pDivInput,
                bDivLen,
                pOldKey));
        }

        /* Diversify New Key */
        if(wOption & PHAL_MFDUOX_CHGKEY_DIV_NEW_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_MFDUOX_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pNewKey,
                wNewKeyType,
                pDivInput,
                bDivLen,
                pNewKey));
        }

        /* Reload the IV and key since the diversify function has invalidated the key */
        /* Load the Session key which is valid for this authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pDataParams->aSesAuthENCKey,
            pDataParams->wKeyType));

        /* Enable storing of Initialization vector. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_CONFIG_KEEP_IV,
            PH_CRYPTOSYM_VALUE_KEEP_IV_ON));
    }

    /* Generate Cryptogram ------------------------------------------------------------------------------------------------------------- */
    /* New Key is required in Crypto framing.
     * The new key already part of processing buffer so no need or further processing.
     * Increment the processing buffer include Key size.
     */
    wCmdDataLen += bKeySize;

    /* Copy Version */
    pCmdData[wCmdDataLen++] = (uint8_t) wNewKeyVer;

    /* Append CRC if,
     *  - Targeted key different from authenticated key
     *  - If DUOX Authentication is performed.
     */
    if((pDataParams->bKeyNo & 0x3FU) != (bKeyNoCard & 0x3FU) ||
        (pDataParams->bAuthState == PHAL_MFDUOX_ECC_AUTHENTICATED))
    {
        /* Compute CRC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTools_CalculateCrc32(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC32_PRESET_DF8,
            PH_TOOLS_CRC32_POLY_DF8,
            pCmdData,
            (uint16_t) (wCmdDataLen - 1U),
            &dwCRC));

        /* Add CRC to command buffer. */
        (void) memcpy(&pCmdData[wCmdDataLen], &dwCRC, 4U);
        wCmdDataLen += 4U;

        /* Compute XOR between New Key and Old Key. */
        for(bIndex = 0; bIndex < bKeySize; bIndex++)
        {
            pNewKey[bIndex] = pOldKey[bIndex] ^ pNewKey[bIndex];
        }
    }

    /* Targeted key equal to authenticated key */
    else
    {
        /* Nothing to do here as the cryptogram is already available as part of command buffer. */
    }

    /* Should the Authenticate be reset. */
    bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;
    if((pDataParams->bKeyNo & 0x3FU) == (bKeyNoCard & 0x3FU))
    {
        bResetAuth = PH_ON;
        bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_PLAIN;
    }

    /* Exchange Cmd.ChangeKey to PICC -------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        bRsp_ComMode,
        bResetAuth,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ChangeKeyEv2(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo,
    uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeySetNo, uint8_t bKeyNoCard,
    uint8_t * pDivInput, uint8_t bDivLen)
{
    uint16_t    PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwCRC = 0;
    uint16_t    PH_MEMLOC_REM wOldKeyType;
    uint16_t    PH_MEMLOC_REM wNewKeyType;
    uint16_t    PH_MEMLOC_REM wDivOption = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t     PH_MEMLOC_REM bKeySize = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bResetAuth = PH_OFF;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = PH_OFF;

    uint8_t     PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };

    uint8_t     PH_MEMLOC_REM * pCmdData = NULL;
    uint8_t     PH_MEMLOC_REM * pOldKey = NULL;
    uint8_t     PH_MEMLOC_REM * pNewKey = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    /* Change key should also take care of changing other keys at PICC level like the Proximity check key, Transaction MAC key */
    if(memcmp(pDataParams->aAid, aAppId, 3U) == 0x00)
    {
        /* Only if selected Aid is 0x000000, PICC level key change is targeted. */
        if(((bKeyNoCard & 0x3FU) != 0x00) &&

            /* PICC DAMAuthKey, DAMMACKey, DAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x10U) && ((bKeyNoCard & 0x3FU) != 0x11U) && ((bKeyNoCard & 0x3FU) != 0x12U) &&

            /* PICC NXPDAMAuthKey, NXPDAMMACKey, NXPDAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x18U) && ((bKeyNoCard & 0x3FU) != 0x19U) && ((bKeyNoCard & 0x3FU) != 0x1AU) &&

            /* PICC VCConfigurationKey, VCProximityKey */
            ((bKeyNoCard & 0x3FU) != 0x20U) && ((bKeyNoCard & 0x3FU) != 0x21U))
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }

        /* If PICC MasterKey holds AES128 or AES256 information. */
        else if(((bKeyNoCard & 0xC0U) != 0xC0U) && ((bKeyNoCard & 0xC0U) != 0x80U) && ((bKeyNoCard & 0xC0U) != 0x00))
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }
        else
        {
            /*Do Nothing. This is for PRQA compliance */
        }
    }
    else
    {
        /* Key numbers between 0D and 20 are not allowed for APP level, also key numbers above 22 are not allowed.
         * if AID 0x000000 is not selected.
         */
        if(IS_INVALID_APP_KEY(bKeyNoCard) && ((bKeyNoCard & 0x3FU) != 0x21U))
        {
            /* Invalid application key specified */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }
    }

    /* Validate KeySet number. */
    if(bKeySetNo > 0x0FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Validate Diversification Length. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivLen);

    /* Validate Option. */
    if((wOption == 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear buffer. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer for the keys. */
    pNewKey = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pOldKey = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Set buffer to use for Command Data. */
    pCmdData = pNewKey;

    /* Frame ChangeKeyEv2 command information ------------------------------------------------------------------------------------------ */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CHANGE_KEY_EV2;

    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CHANGE_KEY_EV2;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNoCard;

    /* Get Old and New Keys ------------------------------------------------------------------------------------------------------------ */
    /* Get the Current Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wCurrKeyNo,
        wCurrKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pOldKey,
        &wOldKeyType));

    /* Verify KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wOldKeyType);

    /* Get the New Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wNewKeyNo,
        wNewKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pNewKey,
        &wNewKeyType));

    /* Verify KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wNewKeyType);

    /* Set the Size of the Key. */
    bKeySize = phKeyStore_GetKeySize(wNewKeyType);

    /* Diversify Keys ------------------------------------------------------------------------------------------------------------------ */
    /* Diversify Current and New Keys. */
    if((wOption != PHAL_MFDUOX_NO_DIVERSIFICATION) && (bDivLen != 0x00))
    {
        /* Diversify Old Key */
        if(wOption & PHAL_MFDUOX_CHGKEY_DIV_OLD_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_MFDUOX_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pOldKey,
                wOldKeyType,
                pDivInput,
                bDivLen,
                pOldKey));
        }

        /* Diversify New Key */
        if(wOption & PHAL_MFDUOX_CHGKEY_DIV_NEW_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_MFDUOX_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pNewKey,
                wNewKeyType,
                pDivInput,
                bDivLen,
                pNewKey));
        }

        /* Reload the IV and key since the diversify function has invalidated the key */
        /* Load the Session key which is valid for this authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pDataParams->aSesAuthENCKey,
            pDataParams->wKeyType));

        /* Enable storing of Initialization vector. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_CONFIG_KEEP_IV,
            PH_CRYPTOSYM_VALUE_KEEP_IV_ON));
    }

    /* Generate Cryptogram ------------------------------------------------------------------------------------------------------------- */
    /* New Key is required in Crypto framing.
     * The new key already part of processing buffer so no need or further processing.
     * Increment the processing buffer include Key size.
    */
    wCmdDataLen += bKeySize;

    /* Copy Version */
    pCmdData[wCmdDataLen++] = (uint8_t) wNewKeyVer;

    /* Append CRC if,
     *  - Targeted key different from authenticated key
     *  - If DUOX Authentication is performed.
     *  - If Master Application is not selected and KeySetNo is not zero.
     */
    if((pDataParams->bKeyNo & 0x3FU) != (bKeyNoCard & 0x3FU) ||
        (pDataParams->bAuthState == PHAL_MFDUOX_ECC_AUTHENTICATED)||
        ((memcmp(pDataParams->aAid, aAppId, 3U) != 0) && (bKeySetNo != 0)))
    {
        /* Compute CRC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTools_CalculateCrc32(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC32_PRESET_DF8,
            PH_TOOLS_CRC32_POLY_DF8,
            pCmdData,
            (uint16_t) (wCmdDataLen - 1U),
            &dwCRC));

        /* Add CRC to command buffer. */
        (void) memcpy(&pCmdData[wCmdDataLen], &dwCRC, 4U);
        wCmdDataLen += 4U;

        /* Compute XOR between New Key and Old Key. */
        for(bIndex = 0; bIndex < bKeySize; bIndex++)
        {
            pNewKey[bIndex] = pOldKey[bIndex] ^ pNewKey[bIndex];
        }
    }

    /* Targeted key equal to authenticated key */
    else
    {
        /* Nothing to do here as the cryptogram is already available as part of command buffer. */
    }

    /* Should the Authenticate be reset. */
    bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;
    if(((pDataParams->bKeyNo & 0x3FU) == (bKeyNoCard & 0x3FU)) && ((bKeySetNo & 0x0F) == 0))
    {
        bResetAuth = PH_ON;
        bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_PLAIN;
    }

    /* Exchange Cmd.ChangeKeyEV2 to PICC ----------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        bRsp_ComMode,
        bResetAuth,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_InitializeKeySet(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bKeySetNo, uint8_t bKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(((bKeySetNo & 0x7FU) > 0x0FU) || ((bKeyType != PHAL_MFDUOX_KEY_TYPE_AES128) &&
       (bKeyType != PHAL_MFDUOX_KEY_TYPE_AES256)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear buffer. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_INITIALIZE_KEY_SET;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_INITIALIZE_KEY_SET;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyType;

    /* Exchange Cmd.InitializeKeySet information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_FinalizeKeySet(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bKeySetNo, uint8_t bKeySetVersion)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(((bKeySetNo & 0x7FU) > 0x0FU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear buffer. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_FINALIZE_KEY_SET;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_FINALIZE_KEY_SET;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetVersion;

    /* Exchange Cmd.FinalizeKeySet information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_RollKeySet(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bKeySetNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(((bKeySetNo & 0x7F) > 0x0FU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear buffer. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_ROLL_KEY_SET;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_ROLL_KEY_SET;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetNo;

    /* Exchange Cmd.RollKeySet information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        PHAL_MFDUOX_COMMUNICATION_PLAIN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

phStatus_t phalMfDuoX_Sw_GetKeySettings(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_KEY_SETTINGS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_KEY_SETTINGS;

    /* Add Option information to command buffer. */
    if(bOption != PHAL_MFDUOX_KEY_SETTING_PICC_APPLICATION)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;
    }

    /* Exchange Cmd.GetKeySettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_ChangeKeySettings(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bKeySettings)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CHANGE_KEY_SETTINGS;

    /* Exchange Cmd.ChangeKeySettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_FULL,
        PHAL_MFDUOX_COMMUNICATION_MAC,
        PH_OFF,
        &pDataParams->bCmdCode,
        1U,
        &bKeySettings,
        1U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetKeyVersion(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };
    if(memcmp(pDataParams->aAid, aAppId, 3U) && ((bKeyNo & 0x0FU) > 0x0d))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif /* RDR_LIB_PARAM_CHECK */

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_KEY_VERSION;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_KEY_VERSION;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNo;

    /* Append KeySet Number */
    if(bKeyNo & 0x40U)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySetNo;
    }

    /* Exchange Cmd.GetKeyVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /*NXPBUILD__PHAL_MFDUOX_NDA */





/* MIFARE DUOX ASymmetric Key management commands -------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_ManageKeyPair(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bKeyNo,
    uint8_t bOption, uint8_t bCurveID, uint8_t * pKeyPolicy, uint8_t bWriteAccess, uint32_t dwKUCLimit,
    uint16_t wPrivKey_No, uint16_t wPrivKey_Pos, uint8_t ** ppResponse, uint16_t *pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdData_Len = 0;

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_MANAGE_KEY_PAIR;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_MANAGE_KEY_PAIR;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bCurveID;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pKeyPolicy[0U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pKeyPolicy[1U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bWriteAccess;

    /* Append the Key Usage Counter Limit. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], (uint8_t *) &dwKUCLimit, 4U);
    PHAL_MFDUOX_CMD_BUF_LEN += 4U;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Add PrivateKey to command buffer. */
    if(bOption == PHAL_MFDUOX_TARGET_ACTION_IMPORT_PRIVATE_KEY)
    {
        /* Get the Private Key from KeyStore. */
        PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKeyASym(
            pDataParams->pKeyStoreDataParams,
            wPrivKey_No,
            wPrivKey_Pos,
            PH_KEYSTORE_KEY_PAIR_PRIVATE,
            &wKeyType,
            &bCurveID,
            pCmdData,
            &wCmdData_Len));

        /* Verify the Key information. */
        if(wKeyType != PH_KEYSTORE_KEY_TYPE_ECC)
        {
            /* Clear Buffers. */
            (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
            PHAL_MFDUOX_CMD_BUF_LEN = 0;

            /* Clear Buffers. */
            (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
            PHAL_MFDUOX_PRS_BUF_LEN = 0;

            return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDUOX);
        }
        else
        {
            /* Do Nothing */
        }
    }

    /* Exchange Cmd.ManageKeyPair information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdData_Len,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ManageCARootKey(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bKeyNo,
    uint8_t bCurveID, uint8_t * pAccessRights, uint8_t bWriteAccess, uint8_t bReadAccess, uint8_t bCRLFile,
    uint8_t * pCRLFileAID, uint16_t wPubKey_No, uint16_t wPubKey_Pos, uint8_t * pIssuer, uint8_t bIssuerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdData_Len = 0;

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Check for Issuer pointer. */
    if(bIssuerLen)
        PH_ASSERT_NULL_PARAM(pIssuer, PH_COMP_AL_MFDUOX);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_MANAGE_CA_ROOT_KEY;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_MANAGE_CA_ROOT_KEY;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bCurveID;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pAccessRights[0U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pAccessRights[1U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bWriteAccess;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bReadAccess;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bCRLFile;

    /* Add Application ID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pCRLFileAID, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Add PublicKey to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKeyASym(
        pDataParams->pKeyStoreDataParams,
        wPubKey_No,
        wPubKey_Pos,
        PH_KEYSTORE_KEY_PAIR_PUBLIC,
        &wKeyType,
        &bCurveID,
        pCmdData,
        &wCmdData_Len));

    /* Verify the Key information. */
    if(wKeyType != PH_KEYSTORE_KEY_TYPE_ECC)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
        PHAL_MFDUOX_PRS_BUF_LEN = 0;

        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDUOX);
    }
    else
    {
        /* Do nothing */
    }

    /* Add Issuer to Command buffer. */
    pCmdData[wCmdData_Len++] = bIssuerLen;

    (void) memcpy(&pCmdData[wCmdData_Len], pIssuer, bIssuerLen);
    wCmdData_Len += bIssuerLen;

    /* Exchange Cmd.ManageCARootKey information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdData_Len,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ExportKey(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bOption,
    uint8_t bKeyNo, uint8_t ** ppResponse, uint16_t *pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_MANAGE_EXPORT_KEY;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bCmd_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_MANAGE_EXPORT_KEY;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOption;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeyNo;

    /* Exchange Cmd.ExportKey information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bCmd_ComMode,
        bComOption,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX Application management commands ----------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_CreateApplication(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pAid,
    uint8_t bKeySettings1, uint8_t bKeySettings2, uint8_t bKeySettings3, uint8_t * pKeySetValues, uint8_t bKeySetValuesLen,
    uint8_t * pISOFileId, uint8_t * pISODFName, uint8_t bISODFNameLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(((bOption & PHAL_MFDUOX_ISO_DF_NAME_AVAILABLE) && (bISODFNameLen > PHAL_MFDUOX_ISO_DFNAME_LEN))
        || (bOption > PHAL_MFDUOX_ISO_FILE_ID_DF_NAME_AVAILABLE))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_APPLICATION;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_APPLICATION;

    /* Add Application ID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAid, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    /* Add Key Settings to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings2;

    /* Add Key Settings 3 and Key Set values to command buffer. */
    if(bKeySettings2 & PHAL_MFDUOX_KEYSETT3_PRESENT)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings3;
        if(bKeySettings3 & PHAL_MFDUOX_KEYSETVALUES_PRESENT)
        {
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pKeySetValues, bKeySetValuesLen);
            PHAL_MFDUOX_CMD_BUF_LEN += bKeySetValuesLen;
        }
    }

    /* Add ISO File ID to command buffer. */
    if(bOption & PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[0];
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[1];
    }

    /* Add ISO DFName to command buffer. */
    if(bOption & PHAL_MFDUOX_ISO_DF_NAME_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISODFName, bISODFNameLen);
        PHAL_MFDUOX_CMD_BUF_LEN += bISODFNameLen;
    }

    /* Exchange Cmd.CreateApplication information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_DeleteApplication(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pAid, uint8_t * pDAMMAC,
    uint8_t bDAMMAC_Len)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResetAuth = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_DELETE_APPLICATION;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bCmd_ComMode);

    /*
     * At APP level, the MAC is not returned. The authenticate state should be reset and command counter needs to be incremented.
     * At PICC level, 8 bytes MAC is returned. The authenticate state should not be reset.
     *
     * So to check whether its in APP level or PICC level, check for pDataParams->pAid. If its 0x00, then its PICC level
     * else its in APP level.
     */
    bResetAuth = PH_ON;
    bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_PLAIN;
    if((pDataParams->aAid[0] == 0) && (pDataParams->aAid[1U] == 0) &&
        (pDataParams->aAid[2U] == 0))
    {
        bResetAuth = PH_OFF;
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
        bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    }

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_DELETE_APPLICATION;

    /* Add Application ID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAid, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    /* Add DAM MAC to command buffer. */
    if(bDAMMAC_Len)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pDAMMAC, bDAMMAC_Len);
        PHAL_MFDUOX_CMD_BUF_LEN += bDAMMAC_Len;
    }

    /* Exchange Cmd.DeleteApplication information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bCmd_ComMode,
        bRsp_ComMode,
        bResetAuth,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_CreateDelegatedApplication(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pAid,
    uint8_t * pDamParams, uint8_t bDamParamsLen, uint8_t bKeySettings1, uint8_t bKeySettings2, uint8_t bKeySettings3,
    uint8_t * pKeySetValues, uint8_t bKeySetValuesLen, uint8_t * pISOFileId, uint8_t * pISODFName, uint8_t bISODFNameLen,
    uint8_t * pEncK, uint8_t bEncKLen, uint8_t * pDAMMAC, uint8_t bDAMMACLen, uint8_t * pInitK, uint8_t bInitKLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;

    uint8_t     PH_MEMLOC_REM *pSMBuff = 0;
    uint16_t    PH_MEMLOC_REM wSmBufLen = 0;

    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdData_Len = 0;

    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(((bOption & PHAL_MFDUOX_ISO_DF_NAME_AVAILABLE) && (bISODFNameLen > PHAL_MFDUOX_ISO_DFNAME_LEN))
        || (bOption > PHAL_MFDUOX_ISO_FILE_ID_DF_NAME_AVAILABLE))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    bCmd_ComMode = PHAL_MFDUOX_COMMUNICATION_PLAIN;
    bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_PLAIN;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_DELEGATED_APPLICATION;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_DELEGATED_APPLICATION;

    /* Add Application ID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAid, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    /* Add DamParams to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pDamParams, bDamParamsLen);
    PHAL_MFDUOX_CMD_BUF_LEN += bDamParamsLen;

    /* Add Key Settings to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings2;

    /* Add Key Settings 3 and Key Set values to command buffer. */
    if(bKeySettings2 & PHAL_MFDUOX_KEYSETT3_PRESENT)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings3;
        if(bKeySettings3 & PHAL_MFDUOX_KEYSETVALUES_PRESENT)
        {
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pKeySetValues, bKeySetValuesLen);
            PHAL_MFDUOX_CMD_BUF_LEN += bKeySetValuesLen;
        }
    }

    /* Add ISO File ID to command buffer. */
    if(bOption & PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[0];
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[1];
    }

    /* Add ISO DFName to command buffer. */
    if(bOption & PHAL_MFDUOX_ISO_DF_NAME_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISODFName, bISODFNameLen);
        PHAL_MFDUOX_CMD_BUF_LEN += bISODFNameLen;
    }

    /* Buffer CreateDelegatedApplication Part1 command --------------------------------------------------------------------------------- */
    wStatus = phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        (uint8_t) (PHAL_MFDUOX_OPTION_PENDING | PHAL_MFDUOX_RETURN_PICC_STATUS),
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse_PICC,
        &wRspLen_PICC,
        &bPiccErrCode);

    /* Check for Chaining status. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        /* Reset Authentication State. */
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);

        return wStatus;
    }

    /* Add EncK and DAMMAC to command buffer. */
    if(pDataParams->bAuthState == PHAL_MFDUOX_EV2_AUTHENTICATED)
    {
        /* Set communication mode as MAC for command. */
        bCmd_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;
        bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;

        /* Set buffer to use for Command Data. */
        pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

        /* Add EncK to processing buffer. */
        (void) memcpy(&pCmdData[wCmdData_Len], pEncK, bEncKLen);
        wCmdData_Len = bEncKLen;

        /* Add DAMMAC to processing buffer. */
        (void) memcpy(&pCmdData[wCmdData_Len], pDAMMAC, bDAMMACLen);
        wCmdData_Len += bDAMMACLen;
    }

    /* Add InitK if DUOX Authenticated. */
    if(pDataParams->bAuthState == PHAL_MFDUOX_ECC_AUTHENTICATED)
    {
        /* Set communication mode as FULL for command. */
        bCmd_ComMode = PHAL_MFDUOX_COMMUNICATION_FULL;
        bRsp_ComMode = PHAL_MFDUOX_COMMUNICATION_MAC;

        /* Add InitK for Encryption. */
        pCmdData = pInitK;
        wCmdData_Len = bInitKLen;
    }

    /* Generate the Secure Messaging --------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ApplySM(
        pDataParams,
        PH_ON,
        PH_ON,
        bCmd_ComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdData_Len,
        &pSMBuff,
        &wSmBufLen));

    /* Exchange CreateDelegatedApplication Part2 command ------------------------------------------------------------------------------- */

    /* Add Chaining command code */
    PHAL_MFDUOX_CMD_BUF_LEN = 0;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_ADDITIONAL_FRAME;

    /* Add SM Buffer to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pSMBuff, wSmBufLen);
    PHAL_MFDUOX_CMD_BUF_LEN += wSmBufLen;

    wStatus = phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_RETURN_PICC_STATUS),
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse_PICC,
        &wRspLen_PICC,
        &bPiccErrCode);

    /* Remove the Secure Messaging ----------------------------------------------------------------------------------------------------- */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        /* Reset Authentication State. */
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        if(bRsp_ComMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_RemoveSM(
                pDataParams,
                PH_OFF,
                PH_ON,
                PH_ON,
                bRsp_ComMode,
                pResponse_PICC,
                wRspLen_PICC,
                bPiccErrCode,
                &pSMBuff,
                &wSmBufLen));
        }
        else
        {
            /* In case of SUCCESS and Communication mode as PLAIN, increment the command counter. */
            pDataParams->wCmdCtr++;
        }
    }

    return wStatus;
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

phStatus_t phalMfDuoX_Sw_SelectApplication(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pAppId,
    uint8_t * pAppId2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_SELECT_APPLICATION;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_SELECT_APPLICATION;

    /* Add Application ID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAppId, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    /* Add Secondary Application to command buffer. */
    if(bOption)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAppId2, PHAL_MFDUOX_APP_ID_LEN);
        PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;
    }

    /* Exchange Cmd.SelectApplication information to PICC. */
    wStatus = phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        PHAL_MFDUOX_COMMUNICATION_PLAIN,
        PHAL_MFDUOX_COMMUNICATION_PLAIN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL);

    /* Validate Status */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) &&
        ((wStatus & PH_ERR_MASK) != PHAL_MFDUOX_ERR_OPERATION_OK_LIM))
        return wStatus;

    /* Store the currently selected application Id */
    (void) memcpy(pDataParams->aAid, pAppId, PHAL_MFDUOX_APP_ID_LEN);

    return PH_ADD_COMPCODE(wStatus, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetApplicationIDs(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppAidBuff,
    uint16_t * pAidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Validate Exchange Options. */
    PHAL_MFDUOX_VALIDATE_RX_EXCHANGE_OPTIONS(bOption);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_APPLICATION_IDS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame the command information. */
    if(bOption != PH_EXCHANGE_RXCHAINING)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_APPLICATION_IDS;
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.GetApplicationIDs information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        bOption,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppAidBuff,
        pAidLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetDFNames(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppDFBuffer,
    uint16_t * pDFInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Validate Exchange Options. */
    PHAL_MFDUOX_VALIDATE_RX_EXCHANGE_OPTIONS(bOption);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_DF_NAMES;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame the command information. */
    if(bOption != PH_EXCHANGE_RXCHAINING)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_DF_NAMES;
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.GetDFNames information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        bOption,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppDFBuffer,
        pDFInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_GetDelegatedInfo(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pDAMSlot, uint8_t * pDamSlotVer,
    uint8_t * pQuotaLimit, uint8_t * pFreeBlocks, uint8_t * pAid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_DELEGATED_INFO;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_DELEGATED_INFO;

    /* Add DAM Slot to command buffer . */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pDAMSlot, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Exchange Cmd.GetDelegatedInfo information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        &pResponse,
        &wRspLen));

    /* Received data length should be equal to 8 bytes. */
    if(wRspLen != 0x08U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);
    }

    (void) memcpy(pDamSlotVer, &pResponse[bIndex], 1U);
    bIndex = 1U;

    (void) memcpy(pQuotaLimit, &pResponse[bIndex], 2U);
    bIndex += 2U;

    (void) memcpy(pFreeBlocks, &pResponse[bIndex], 2U);
    bIndex += 2U;

    (void) memcpy(pAid, &pResponse[bIndex], PHAL_MFDUOX_APP_ID_LEN);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */




/* MIFARE DUOX File management commands ------------------------------------------------------------------------------------------------ */
phStatus_t phalMfDuoX_Sw_CreateStdDataFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pFileSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bOption > PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_STANDARD_DATA_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_STANDARD_DATA_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add ISO File ID to command buffer. */
    if(bOption == PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISOFileId, 2U);
        PHAL_MFDUOX_CMD_BUF_LEN += 2U;
    }

    /* Add File Options to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add File Size to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pFileSize, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.CreateStdDataFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CreateBackupDataFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pFileSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bOption > PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_BACKUP_DATA_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_BACKUP_DATA_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add ISO File ID to command buffer. */
    if(bOption == PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISOFileId, 2U);
        PHAL_MFDUOX_CMD_BUF_LEN += 2U;
    }

    /* Add File Options to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add File Size to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pFileSize, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.CreateBackupDataFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CreateValueFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pAccessRights, uint8_t * pLowerLmit, uint8_t * pUpperLmit, uint8_t * pValue, uint8_t bLimitedCredit)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_VALUE_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_VALUE_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add LowerLimit to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pLowerLmit, 4U);
    PHAL_MFDUOX_CMD_BUF_LEN += 4U;

    /* Add UpperLimit to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pUpperLmit, 4U);
    PHAL_MFDUOX_CMD_BUF_LEN += 4U;

    /* Add Current Value to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pValue, 4U);
    PHAL_MFDUOX_CMD_BUF_LEN += 4U;

    /* Add Limited Credit to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bLimitedCredit;

    /* Exchange Cmd.CreateValueFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CreateLinearRecordFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bOption > PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_LINEAR_RECORD_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_LINEAR_RECORD_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add ISO File ID to command buffer. */
    if(bOption == PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISOFileId, 2U);
        PHAL_MFDUOX_CMD_BUF_LEN += 2U;
    }

    /* Add File Options to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add Record Size to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRecordSize, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Add Maximum Number of Records to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pMaxNoOfRec, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.CreateLinearRecordFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CreateCyclicRecordFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bOption > PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_CYCLIC_RECORD_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_CYCLIC_RECORD_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add ISO File ID to command buffer. */
    if(bOption == PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISOFileId, 2U);
        PHAL_MFDUOX_CMD_BUF_LEN += 2U;
    }

    /* Add File Options to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add Record Size to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRecordSize, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Add Maximum Number of Records to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pMaxNoOfRec, 3);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.CreateCyclicRecordFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_CreateTransactionMacFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bFileNo,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t bTMKeyOption, uint8_t * pTMKey, uint8_t bTMKeyVer, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t * pTMIExclFileMap, uint8_t bTSIGKeyNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRWAccess = 0;
    uint8_t     PH_MEMLOC_REM * pCmdData = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREATE_TRANSACTION_MAC_FILE;

    /* Frame the communication mode to be applied. */
    bCmd_ComMode = (uint8_t) ((pDataParams->bAuthState == PHAL_MFDUOX_NOT_AUTHENTICATED) ?
        PHAL_MFDUOX_COMMUNICATION_PLAIN : PHAL_MFDUOX_COMMUNICATION_FULL);

    bRsp_ComMode = (uint8_t) ((pDataParams->bAuthState == PHAL_MFDUOX_NOT_AUTHENTICATED) ?
        PHAL_MFDUOX_COMMUNICATION_PLAIN : PHAL_MFDUOX_COMMUNICATION_MAC);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer for the keys. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_SIZE - ((PH_CRYPTOSYM_AES256_KEY_SIZE * 1U) + 2U)];

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_TRANSACTION_MAC_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_MFDUOX_CMD_BUF_LEN += 2U;

    /* Add TMI Exclusion File Map to command buffer. */
    if(bFileOption & PHAL_MFDUOX_FILE_OPTION_TMI_EXCLUSION_FILEMAP)
    {
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pTMIExclFileMap, 4U);
        PHAL_MFDUOX_CMD_BUF_LEN += 4U;
    }

    /* Add TMKey option to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bTMKeyOption;

    /* Extract ReadWrite Access. */
    bRWAccess = (uint8_t) ((pAccessRights[0] & 0xF0U) >> 4U);

    /* Add TMKey and TMKeyVer to command buffer for the following conditions.
     *  - TmKeyOption has Transaction MAC as Mode
     *  - ReadWrite Access is configured for CommitReader ID.
     */
    if(!(bTMKeyOption & PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_SIGNATURE) || (bRWAccess < 0x0EU))
    {
        /* Extract KeyType from TMKey Options. */
        wKeyType = (uint16_t) (((bTMKeyOption & 0x03U) == PHAL_MFDUOX_KEY_TYPE_AES128) ? PH_KEYSTORE_KEY_TYPE_AES128 :
            PH_KEYSTORE_KEY_TYPE_AES256);

        /* Copy Key to Processing buffer. */
        wCmdDataLen = phKeyStore_GetKeySize(wKeyType);
        (void) memcpy(pCmdData, pTMKey, wCmdDataLen);

        /* Diversify the Key. */
        if(wOption == PHAL_MFDUOX_CREATE_TMAC_FILE_DIVERSIFICATION_ON)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS,
                pCmdData,
                wKeyType,
                pDivInput,
                bDivInputLen,
                pCmdData));

            /* Reload the IV and key since the diversify function has invalidated the key */
            /* Load the Session key which is valid for this authentication */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
                pDataParams->pCryptoDataParamsEnc,
                pDataParams->aSesAuthENCKey,
                pDataParams->wKeyType));

            /* Enable storing of Initialization vector. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
                pDataParams->pCryptoDataParamsEnc,
                PH_CRYPTOSYM_CONFIG_KEEP_IV,
                PH_CRYPTOSYM_VALUE_KEEP_IV_ON));
        }

        /* Add Key version to processing buffer. */
        pCmdData[wCmdDataLen++] = bTMKeyVer;
    }

    /* Add TSigKeyNo to command buffer. */
    if(bTMKeyOption & PHAL_MFDUOX_TMKEY_MODE_TRANSACTION_SIGNATURE)
    {
        pCmdData[wCmdDataLen++] = bTSIGKeyNo;
    }

    /* Exchange Cmd.CreateTransactionMacFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

phStatus_t phalMfDuoX_Sw_DeleteFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_DELETE_FILE;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_DELETE_FILE;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.DeleteFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetFileIDs(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t ** ppFileId, uint16_t * pFileIdLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_FILE_IDS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_FILE_IDS;

    /* Exchange Cmd.GetFileIDs information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppFileId,
        pFileIdLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetISOFileIDs(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t ** ppISOFileId, uint16_t * pISOFileIdLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_ISO_FILE_IDS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_ISO_FILE_IDS;

    /* Exchange Cmd.GetISOFileIDs information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppISOFileId,
        pISOFileIdLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetFileSettings(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t ** ppFSBuffer,
    uint16_t * pFSBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_FILE_SETTINGS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_FILE_SETTINGS;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.GetFileSettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppFSBuffer,
        pFSBufLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetFileCounters(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t ** ppFileCounters, uint16_t * pFileCounterLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_FILE_COUNTERS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_GET_FILE_COUNTERS;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.GetFileCounters information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bCmd_ComMode,
        bOption,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppFileCounters,
        pFileCounterLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ChangeFileSettings(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pAddInfo, uint8_t bAddInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));
    PHAL_MFDUOX_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CHANGE_FILE_SETTINGS;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CHANGE_FILE_SETTINGS;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Add FileOption to processing buffer. */
    pCmdData[wCmdDataLen++] = bFileOption;

    /* Add Access Rights to processing buffer. */
    (void) memcpy(&pCmdData[wCmdDataLen], pAccessRights, 2U);
    wCmdDataLen += 2U;

    /* Add File Size to processing buffer. */
    (void) memcpy(&pCmdData[wCmdDataLen], pAddInfo, bAddInfoLen);
    wCmdDataLen += bAddInfoLen;

    /* Exchange Cmd.ChangeFileSettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX Data management commands ------------------------------------------------------------------------------------------------ */
phStatus_t phalMfDuoX_Sw_ReadData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
    uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bExchange_Option = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bTMIOption = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication and Exchange Options. */
    bRsp_ComMode = (uint8_t) (bOption & PHAL_MFDUOX_COMM_OPTIONS_MASK);
    bExchange_Option = (uint8_t) (bOption & PH_EXCHANGE_MODE_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bRsp_ComMode));

    /* Validate Exchange Options. */
    PHAL_MFDUOX_VALIDATE_RX_EXCHANGE_OPTIONS(bExchange_Option);

    /* Validate Chaining Options. */
    PHAL_MFDUOX_VALIDATE_CHAINING_OPTIONS(bIns);

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Compute the length. */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0U];

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = (uint8_t) (bIns ? PHAL_MFDUOX_CMD_READ_DATA_ISO : PHAL_MFDUOX_CMD_READ_DATA_NATIVE);

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Frame the command information. */
    if(bExchange_Option != PH_EXCHANGE_RXCHAINING)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        /* Add command code. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

        /* Add File Number. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

        /* Add offset. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOffset, 3U);
        PHAL_MFDUOX_CMD_BUF_LEN += 3U;

        /* Add length. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pLength, 3U);
        PHAL_MFDUOX_CMD_BUF_LEN += 3U;

        /* Perform TMI Collection for Command Header. */
        if(dwTMIStatus)
        {
            /* Frame the Option. */
            bTMIOption = (uint8_t) (PH_TMIUTILS_ZEROPAD_CMDBUFF | (dwLength ? 0 : PH_TMIUTILS_READ_INS));

            /* Buffer the Command information to TMI buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
                bTMIOption, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, NULL, 0, PHAL_MFDUOX_BLOCK_SIZE));
        }
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.ReadData information to PICC. */
    wStatus = phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        bExchange_Option,
        bIns,
        bCmd_ComMode,
        bRsp_ComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        dwLength,
        ppResponse,
        pRspLen);

    /* Validate Status. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        return wStatus;

    /* Perform TMI Collection for Response. */
    if(dwTMIStatus)
    {
        /* Frame the Option. */
        bTMIOption = (uint8_t) (dwLength ? 0 : PH_TMIUTILS_READ_INS);
        bTMIOption = (uint8_t) ((wStatus == PH_ERR_SUCCESS) ? (bTMIOption | PH_TMIUTILS_ZEROPAD_DATABUFF) : bTMIOption);

        /* Buffer the Command information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            bTMIOption, NULL, 0, *ppResponse, *pRspLen, PHAL_MFDUOX_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(!dwLength && ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
        {
            /* Reset Offset in TMI. */
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_SetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_WriteData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
    uint16_t wCRLVer, uint8_t * pOffset, uint8_t * pData, uint8_t * pLength)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bIsCRLFile = PH_OFF;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_MFDUOX_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bCmd_ComMode));

    /* Validate Chaining Options. */
    PHAL_MFDUOX_VALIDATE_CHAINING_OPTIONS(bIns);

    /* Check if its a CRLFile. */
    bIsCRLFile = (uint8_t) (bOption & PHAL_MFDUOX_CRLFILE_MASK);

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Compute the Total length (Data + [CRL Signature]). */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0U];

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = (uint8_t) (bIns ? PHAL_MFDUOX_CMD_WRITE_DATA_ISO : PHAL_MFDUOX_CMD_WRITE_DATA_NATIVE);

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, NULL, 0, PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Add CLR Information */
    if(bIsCRLFile == PH_ON)
    {
        /* Add CLRVersion. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], (uint8_t *) &wCRLVer, 2U);
        PHAL_MFDUOX_CMD_BUF_LEN += 2U;
    }

    /* Add offset. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOffset, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Add length. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pLength, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        /* Add Offset information to TMICollection. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, pOffset, 3, NULL, 0, PHAL_MFDUOX_BLOCK_SIZE));

        /* Add Length and Data information to TMICollection. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), pLength, 3, pData, dwLength,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        (uint8_t) !bIns,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        dwLength,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GetValue(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t ** ppValue,
    uint16_t * pValueLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication and Exchange Options. */
    bRsp_ComMode = (uint8_t) (bOption & PHAL_MFDUOX_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bRsp_ComMode));

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_GET_VALUE;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Collect TMI for command header. */
    if(dwTMIStatus)
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, NULL, 0, PHAL_MFDUOX_BLOCK_SIZE));

    /* Exchange Cmd.GetValue information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bCmd_ComMode,
        bRsp_ComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        ppValue,
        pValueLen));

    /* Perform TMI Collection for Response. */
    if(dwTMIStatus)/* Buffer the Command information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, *ppValue, *pValueLen, PHAL_MFDUOX_BLOCK_SIZE));

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Credit(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CREDIT;

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, pData, 4U,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.Credit information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        4U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Debit(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_DEBIT;

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, pData, 4U,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.Debit information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        4U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_LimitedCredit(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_LIMITED_CREDIT;

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, pData, 4U,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.LimitedCredit information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        4U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ReadRecords(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
    uint8_t * pRecNo, uint8_t * pRecCount, uint8_t * pRecSize, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint32_t    PH_MEMLOC_REM dwTMIOffset_Len = 0;
    uint32_t    PH_MEMLOC_REM dwTMIBuf_Index = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;
    uint32_t    PH_MEMLOC_REM dwNumRec = 0;
    uint32_t    PH_MEMLOC_REM dwNumRec_Calc = 0;
    uint32_t    PH_MEMLOC_REM dwRecLen = 0;
    uint8_t     PH_MEMLOC_REM bExchange_Option = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bTMIOption = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication and Exchange Options. */
    bRsp_ComMode = (uint8_t) (bOption & PHAL_MFDUOX_COMM_OPTIONS_MASK);
    bExchange_Option = (uint8_t) (bOption & PH_EXCHANGE_MODE_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bRsp_ComMode));

    /* Validate Exchange Options. */
    PHAL_MFDUOX_VALIDATE_RX_EXCHANGE_OPTIONS(bExchange_Option);

    /* Validate Chaining Options. */
    PHAL_MFDUOX_VALIDATE_CHAINING_OPTIONS(bIns);

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = (uint8_t) (bIns ? PHAL_MFDUOX_CMD_READ_RECORD_ISO : PHAL_MFDUOX_CMD_READ_RECORD_NATIVE);

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Frame the command information. */
    if(bExchange_Option != PH_EXCHANGE_RXCHAINING)
    {
        /* Compute the number of records. */
        dwNumRec = pRecCount[2U];
        dwNumRec = dwNumRec << 8U | pRecCount[1U];
        dwNumRec = dwNumRec << 8U | pRecCount[0U];

        /* Compute the record length. */
        dwRecLen = pRecSize[2U];
        dwRecLen = dwRecLen << 8U | pRecSize[1U];
        dwRecLen = dwRecLen << 8U | pRecSize[0U];

        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        /* Add command code. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

        /* Add File Number. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

        /* Add Record Number. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRecNo, 3U);
        PHAL_MFDUOX_CMD_BUF_LEN += 3U;

        /* Add Record Count. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRecCount, 3U);
        PHAL_MFDUOX_CMD_BUF_LEN += 3U;

        /* Perform TMI Collection for Command Header. */
        if(dwTMIStatus)
        {
            /* Should provide at least wRecLen / wNumRec to update in TMI collection */
            if((0 == dwRecLen) && (0 == dwNumRec))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
            }

            /* Frame the Option. */
            bTMIOption = (uint8_t) (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF);

            /* Buffer the Command information to TMI buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
                bTMIOption, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, NULL, 0, PHAL_MFDUOX_BLOCK_SIZE));
        }
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.ReadData information to PICC. */
    wStatus = phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        bExchange_Option,
        bIns,
        bCmd_ComMode,
        bRsp_ComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        dwNumRec * dwRecLen,
        ppResponse,
        pRspLen);

    /* Validate Status. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        return wStatus;

    /* Perform TMI Collection for Response. */
    if(dwTMIStatus)
    {
        /* Update Actual Record Length in case of zeros. */
        if((dwNumRec == 0) && (wStatus == PH_ERR_SUCCESS))
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, &dwTMIOffset_Len));

            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_BUFFER_INDEX, &dwTMIBuf_Index));

            /* calculate Rx length in case of chaining */
            dwDataLen = *pRspLen + dwTMIBuf_Index - (dwTMIOffset_Len + 11);

            /* If user updates wrong RecSize, we cant calculate RecCnt */
            if(dwDataLen % dwRecLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);
            }

            /* Calculate actual number of records */
            dwNumRec_Calc = dwDataLen / dwRecLen;

            /* Update actual number of records to TMI buffer */
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_SetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_VALUE, dwNumRec_Calc));
        }

        /* Frame the Option. */
        bTMIOption = (uint8_t) ((wStatus == PH_ERR_SUCCESS) ? PH_TMIUTILS_ZEROPAD_DATABUFF : PH_TMIUTILS_NO_PADDING);

        /* Buffer the Command information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            bTMIOption, NULL, 0, *ppResponse, *pRspLen, PHAL_MFDUOX_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) && (dwNumRec == 0))
        {
            /* Reset Offset in TMI. */
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_SetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_WriteRecord(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
    uint8_t * pOffset, uint8_t * pData, uint8_t * pLength)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));

    /* Validate Chaining Options. */
    PHAL_MFDUOX_VALIDATE_CHAINING_OPTIONS(bIns);

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Compute the length. */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0U];

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = (uint8_t) (bIns ? PHAL_MFDUOX_CMD_WRITE_RECORD_ISO : PHAL_MFDUOX_CMD_WRITE_RECORD_NATIVE);

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add offset. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOffset, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Add length. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pLength, 3U);
    PHAL_MFDUOX_CMD_BUF_LEN += 3U;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), PHAL_MFDUOX_CMD_BUF,
            PHAL_MFDUOX_CMD_BUF_LEN, pData, dwLength, PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.WriteRecord information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        (uint8_t) !bIns,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        dwLength,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_UpdateRecord(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
    uint8_t * pRecNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pLength)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bOption));

    /* Validate Chaining Options. */
    PHAL_MFDUOX_VALIDATE_CHAINING_OPTIONS(bIns);

    /* Frame the communication mode to be verified. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Compute the length. */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0U];

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = (uint8_t) (bIns ? PHAL_MFDUOX_CMD_UPDATE_RECORD_ISO : PHAL_MFDUOX_CMD_UPDATE_RECORD_NATIVE);

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Add Record Number. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRecNo, 3);
    PHAL_MFDUOX_CMD_BUF_LEN += 3;

    /* Add offset. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOffset, 3);
    PHAL_MFDUOX_CMD_BUF_LEN += 3;

    /* Add length. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pLength, 3);
    PHAL_MFDUOX_CMD_BUF_LEN += 3;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), PHAL_MFDUOX_CMD_BUF,
            PHAL_MFDUOX_CMD_BUF_LEN, pData, dwLength, PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.UpdateRecord information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        (uint8_t) !bIns,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pData,
        dwLength,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ClearRecordFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_MFDUOX_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CLEAR_RECORD;

    /* Add command code. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Add File Number. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bFileNo;

    /* Get TMI Collection Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Perform TMI Collection for command and data. */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_CMDBUFF, PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, NULL, 0,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    /* Exchange Cmd.WriteRecord information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX Transaction Management commands ------------------------------------------------------------------------------------------ */
phStatus_t phalMfDuoX_Sw_CommitTransaction(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppTMC,
    uint16_t * pTMCLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    if((bOption & PHAL_MFDUOX_OPTION_MASK) > PHAL_MFDUOX_OPTION_TRANSACTION_INFO_RETURNED)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_COMMIT_TRANSACTION;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_COMMIT_TRANSACTION;

    /* Add Option to command buffer. */
    if(bOption != PHAL_MFDUOX_OPTION_NOT_EXCHANGED)
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (bOption & PHAL_MFDUOX_OPTION_MASK);

    /* Exchange Cmd.CommitTransaction information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_DATA_TO_READ_UNKNOWN,
        &pResponse,
        &wRspLen));

    /* Update the response parameters. */
    if(bOption == PHAL_MFDUOX_OPTION_TRANSACTION_INFO_RETURNED)
    {
        *ppTMC = pResponse;
        *pTMCLen = 4U;

        *ppResponse = &pResponse[4U];
        *pRspLen = (uint16_t) (wRspLen - 4U);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_AbortTransaction(phalMfDuoX_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_ABORT_TRANSACTION;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_ABORT_TRANSACTION;

    /* Exchange Cmd.AbortTransaction information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CommitReaderID(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pTMRI, uint8_t bTMRILen,
    uint8_t ** ppEncTMRI, uint16_t * pEncTMRILen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t     PH_MEMLOC_REM bOption = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Commit Reader ID command Exchange ----------------------------------------------------------------------------------------------- */
    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_COMMIT_READER_ID;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, PHAL_MFDUOX_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Exchange Cmd.CommitReaderID information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pTMRI,
        bTMRILen,
        &pResponse,
        &wRspLen));

    /* Update the response parameters. */
    *ppEncTMRI = pResponse;
    *pEncTMRILen = wRspLen;

    /* Perform TMICollection ----------------------------------------------------------------------------------------------------------- */

    /* Get TMI Collection status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS,
        &dwTMIStatus));

    /* Perform TMI Collection. */
    if(dwTMIStatus == PH_ON)
    {
        /* Frame the command information. */
        PHAL_MFDUOX_CMD_BUF_LEN = 0;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

        /* Add TMRI to command buffer. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pTMRI, bTMRILen);
        PHAL_MFDUOX_CMD_BUF_LEN += bTMRILen;

        /* Update the response length to use.
         *  In case of Authenticated, EncTMRI information needs to be provided for TMI Collection.
         */
        wRspLen = (uint16_t) ((pDataParams->bAuthState == PHAL_MFDUOX_NOT_AUTHENTICATED) ? 0 : wRspLen);

        /* Set the padding option to use for TMI collection. */
        bOption = (uint8_t) (wRspLen ? PH_TMIUTILS_ZEROPAD_DATABUFF : PH_TMIUTILS_ZEROPAD_CMDBUFF);

        /* Perform TMI COllection. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
            bOption,
            PHAL_MFDUOX_CMD_BUF,
            PHAL_MFDUOX_CMD_BUF_LEN,
            pResponse,
            wRspLen,
            PHAL_MFDUOX_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX Cryptographic support commands ------------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_CryptoRequest(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bAction,
    uint8_t * pInputData, uint16_t wInputLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CRYPTO_REQUEST;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Frame Command Data */
    pCmdData[wCmdDataLen++] = bAction;

    (void) memcpy(&pCmdData[wCmdDataLen], pInputData, wInputLen);
    wCmdDataLen += wInputLen;

    /* Exchange Cmd.CryptoRequest information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CryptoRequestECCSign(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bOperation,
    uint8_t bAlgo, uint8_t bKeyNo, uint8_t bInputSource, uint8_t * pInputData, uint8_t bInputLen, uint8_t ** ppSign,
    uint16_t * pSignLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CRYPTO_REQUEST_ECCSIGN;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Frame command data. */
    pCmdData[wCmdDataLen++] = PHAL_MFDUOX_TARGET_ACTION_ECC_SIGN;
    pCmdData[wCmdDataLen++] = bOperation;

    if((bOperation != PHAL_MFDUOX_TARGET_OPERATION_UPDATE_DATA) &&
        (bOperation != PHAL_MFDUOX_TARGET_OPERATION_FINALIZE_DATA))
    {
        pCmdData[wCmdDataLen++] = bAlgo;
        pCmdData[wCmdDataLen++] = bKeyNo;
    }

    pCmdData[wCmdDataLen++] = bInputSource;
    pCmdData[wCmdDataLen++] = bInputLen;

    (void) memcpy(&pCmdData[wCmdDataLen], pInputData, bInputLen);
    wCmdDataLen += bInputLen;

    /* Exchange Cmd.CryptoRequest_DUOXSign information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        ppSign,
        pSignLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CryptoRequestEcho(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t * pInputData,
    uint8_t bInputLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption(bComOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_CRYPTO_REQUEST_ECHO;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN];

    /* Frame command data. */
    pCmdData[wCmdDataLen++] = PHAL_MFDUOX_TARGET_ACTION_ECC_ECHO;

    (void) memcpy(&pCmdData[wCmdDataLen], pInputData, bInputLen);
    wCmdDataLen += bInputLen;

    /* Exchange Cmd.CryptoRequest_DUOXSign information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX GPIO Management commands ------------------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_ManageGPIO(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bGPIONo, uint8_t bOperation,
    uint8_t * pNFCPauseRspData, uint16_t wNFCPauseRspDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption_PICC = 0;
    uint16_t    PH_MEMLOC_REM wTotalLen = 0;
    uint8_t     PH_MEMLOC_REM bBuffOption_SM = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM bCmdOption = 0;
    uint8_t     PH_MEMLOC_REM bComplete = PH_OFF;

    uint8_t     PH_MEMLOC_REM *pSMBuff = 0;
    uint16_t    PH_MEMLOC_REM wSmBufLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption((uint8_t) wOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_MANAGE_GPIO;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, (uint8_t) wOption,
        &bCmd_ComMode);
    bRsp_ComMode = (uint8_t) wOption;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bGPIONo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOperation;

    /* Compute Total frame length (CmdHdr + CmdData (With or Without MAC)) */
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    wTotalLen = (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wNFCPauseRspDataLen +
        ((bCmd_ComMode == PHAL_MFDUOX_COMMUNICATION_MAC) ? 8U : 0U));
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
#ifndef NXPBUILD__PHAL_MFDUOX_NDA
    wTotalLen = (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wNFCPauseRspDataLen);
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    /* Buffer ManageGPIO command ------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        wTotalLen,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Set buffering options. */
    bBuffOption_SM = PHAL_MFDUOX_RETURN_PLAIN_DATA;
    wBuffOption_PICC = PH_EXCHANGE_BUFFER_CONT;
    bCmdOption = PHAL_MFDUOX_OPTION_NONE;

    /* Generate the Secure Messaging --------------------------------------------------------------------------------------------------- */
    do
    {
        wStatus = phalMfDuoX_Sw_Int_ApplySM(
            pDataParams,
            PH_ON,
            PH_ON,
            bCmd_ComMode,
            PHAL_MFDUOX_CMD_BUF,
            PHAL_MFDUOX_CMD_BUF_LEN,
            pNFCPauseRspData,
            wNFCPauseRspDataLen,
            &pSMBuff,
            &wSmBufLen);

        /* Update SM Buffering options. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
            bBuffOption_SM = (uint8_t) (PH_EXCHANGE_TXCHAINING | PHAL_MFDUOX_RETURN_PLAIN_DATA);

        /* Validate Status */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            break;

        /* End the Loop. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            bComplete = PH_ON;

            /* Re-Apply Buffer and Command Option flags. */
            wBuffOption_PICC = PH_EXCHANGE_BUFFER_LAST;
            bCmdOption = (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_RETURN_PICC_STATUS);
        }

        /* Exchange NFCPauseRspData and MAC (If Applicable) ---------------------------------------------------------------------------- */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
            pDataParams,
            wBuffOption_PICC,
            PHAL_MFDUOX_CHAINING_BIT_INVALID,
            bCmdOption,
            0,
            PH_ON,
            pSMBuff,
            wSmBufLen,
            &pResponse_PICC,
            &wRspLen_PICC,
            &bPiccErrCode));
    } while(bComplete == PH_OFF);

    /* Validate the Status. */
    PH_CHECK_SUCCESS(wStatus);

    /* Remove the Secure Messaging ----------------------------------------------------------------------------------------------------- */
    if(bRsp_ComMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_RemoveSM(
            pDataParams,
            PH_OFF,
            PH_ON,
            PH_ON,
            bRsp_ComMode,
            pResponse_PICC,
            wRspLen_PICC,
            bPiccErrCode,
            ppResponse,
            pRspLen));
    }
    else
    {
        /* In case of SUCCESS and Communication mode as PLAIN, increment the command counter. */
        pDataParams->wCmdCtr++;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ReadGPIO(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_Validate_ComOption((uint8_t) wOption));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_READ_GPIO;

    /* Frame the communication mode to be applied. */
    phalMfDuoX_Int_GetCommMode(pDataParams->bAuthState, (uint8_t) wOption, &bCmd_ComMode);
    bRsp_ComMode = (uint8_t) wOption;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Exchange Cmd.ReadGPIO information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_WriteData(
        pDataParams,
        PH_OFF,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        0,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}





/* MIFARE DUOX ISO7816-4 commands ------------------------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_IsoSelectFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid,
    uint8_t * pDFname, uint8_t bDFnameLen, uint8_t bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatusTmp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM aData[24U];
    uint16_t    PH_MEMLOC_REM wLc = 0;
    uint16_t    PH_MEMLOC_REM wLe = 0;
    uint8_t     PH_MEMLOC_REM aFileId[3U] = { '\0' };
    uint8_t     PH_MEMLOC_REM aPiccDfName[7U] = { 0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00 };
    uint16_t    PH_MEMLOC_REM wVal = 0;
    uint8_t     PH_MEMLOC_REM bInclude_LCLE = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bDFnameLen > 16U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
    if((bOption != PHAL_MFDUOX_FCI_RETURNED) && (bOption != PHAL_MFDUOX_FCI_NOT_RETURNED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    switch(bSelector)
    {
        case PHAL_MFDUOX_SELECTOR_0: /* Select MF, DF or EF, by file identifier */
        case PHAL_MFDUOX_SELECTOR_1: /* Select child DF */
        case PHAL_MFDUOX_SELECTOR_2: /* Select EF under the current DF, by file identifier */
        case PHAL_MFDUOX_SELECTOR_3: /* Select parent DF of the current DF */
                                    /* Selection by EF Id*/
                                    /* Send MSB first to card */
            aFileId[1U] = aData[0U] = pFid[1U];
            aFileId[0U] = aData[1U] = pFid[0U];
            aFileId[2U] = 0x00;
            wLc = 2U;
            break;

        case PHAL_MFDUOX_SELECTOR_4: /* Select by DF name, see Cmd.ISOSelect for VC selection. */
            (void) memcpy(aData, pDFname, bDFnameLen);
            wLc = bDFnameLen;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Update LC LE information for exchange. */
    bInclude_LCLE = (uint8_t) ((bOption == PHAL_MFDUOX_FCI_NOT_RETURNED) ? PHAL_MFDUOX_EXCHANGE_LC_ONLY :
        PHAL_MFDUOX_EXCHANGE_LC_LE_BOTH);

    wStatus = phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        bInclude_LCLE,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_SELECT_FILE,
        bSelector,
        bOption,
        aData,
        wLc,
        wLe,
        ppFCI,
        pFCILen);

    if((wStatus & PH_ERR_MASK) == PHAL_MFDUOX_ERR_DF_7816_GEN_ERROR)
    {
        wStatusTmp = phalMfDuoX_GetConfig(pDataParams, PHAL_MFDUOX_ADDITIONAL_INFO, &wVal);
    }

    if((wStatus == PH_ERR_SUCCESS) || (wVal == PHAL_MFDUOX_ISO7816_ERR_6283))
    {
        /* Reset Authentication should not be targeted for elementary file selection using file ID */
        if(bSelector != 0x02U)
        {
            /* Reset Authentication Status here */
            phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
        }
        /* ISO wrapped mode is on */
        pDataParams->bWrappedMode = PH_ON;

        /* Once the selection Success, update the File Id to master data structure if the selection is done through AID */
        if((bSelector == 0x00U) || (bSelector == 0x01U) || (bSelector == 0x02U))
        {
            (void) memcpy(pDataParams->aAid, aFileId, sizeof(aFileId));
        }
        else if((bSelector == 0x04U))
        {
            /* Update the file ID to all zeros if DF Name is of PICC. */
            if(memcmp(pDFname, aPiccDfName, sizeof(aPiccDfName)) == 0)
            {
                aFileId[0U] = 0x00;
                aFileId[1U] = 0x00;
                aFileId[2U] = 0x00;
            }
            else
            {
                aFileId[0U] = 0xFFU;
                aFileId[1U] = 0xFFU;
                aFileId[2U] = 0xFFU;
            }

            (void) memcpy(pDataParams->aAid, aFileId, sizeof(aFileId));
        }
    }
    else
    {
        /* Nothing to do here. */
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_IsoReadBinary(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid,
    uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    /* Validate Parameters. */
    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_DEFAULT)
    {
        if(bSfid & PHAL_MFDUOX_SFID_ENABLED)
        {
#ifdef RDR_LIB_PARAM_CHECK
            /* Short file id is supplied */
            PHAL_MFDUOX_IS_VALID_FILE_NO(bSfid);
#endif
            /* Short File Identifier from 00 - 1F. */
            bP1 = bSfid;

            /* Offset from 0 - 255. */
            bP2 = bOffset;
        }
        else
        {
            /* Encode Offset from 0 - 32767 */
            bP1 = bSfid;
            bP2 = bOffset;
        }
    }
    else if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        /* Do Nothing. Code is handled internally. */
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LE_ONLY,
        wOption,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_READ_BINARY,
        bP1,
        bP2,
        NULL,
        0,
        dwBytesToRead,
        ppResponse,
        pRspLen);
}

phStatus_t phalMfDuoX_Sw_IsoUpdateBinary(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    if(bSfid & PHAL_MFDUOX_SFID_ENABLED)
    {
#ifdef RDR_LIB_PARAM_CHECK
        /* Short file id is supplied */
        PHAL_MFDUOX_IS_VALID_FILE_NO(bSfid);
#endif
        /* Short File Identifier from 00 - 1F. */
        bP1 = bSfid;

        /* Offset from 0 - 255. */
        bP2 = bOffset;
    }
    else
    {
        /* Encode Offset from 0 - 32767 */
        bP1 = bSfid;
        bP2 = bOffset;
    }

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    wStatus = phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LC_ONLY,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_UPDATE_BINARY,
        bP1,
        bP2,
        pData,
        wDataLen,
        0,
        NULL,
        NULL);

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_IsoReadRecords(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bRecNo, uint8_t bReadAllRecords,
    uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    /* Validate Parameters. */
    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_DEFAULT)
    {
#ifdef RDR_LIB_PARAM_CHECK
        /* Short file id is supplied */
        PHAL_MFDUOX_IS_VALID_FILE_NO(bSfid);
#endif
        /* Record Number. */
        bP1 = bRecNo;

        /* Add SFID and usage of Record number. */
        bP2 = (uint8_t) ((bSfid << 3U) | (0x04U | bReadAllRecords));
    }
    else if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        /* Do Nothing. Code is handled internally. */
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LE_ONLY,
        wOption,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_READ_RECORD,
        bP1,
        bP2,
        NULL,
        0,
        dwBytesToRead,
        ppResponse,
        pRspLen);
}

phStatus_t phalMfDuoX_Sw_IsoAppendRecord(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData,
    uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

#ifdef RDR_LIB_PARAM_CHECK
    /* Short file id is supplied */
    PHAL_MFDUOX_IS_VALID_FILE_NO(bSfid);
#endif

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    /* Current Record. */
    bP1 = 0x00;

    /* Short File Identifier from 00 - 1F. */
    bP2 = (uint8_t) (bSfid << 3U);

    wStatus = phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LC_ONLY,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_APPEND_RECORD,
        bP1,
        bP2,
        pData,
        wDataLen,
        0,
        NULL,
        NULL);

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_IsoGetChallenge(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bExpRsp, uint8_t bExtendedLenApdu,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LE_ONLY,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_GENERIC_CLA,
        PHAL_MFDUOX_CMD_ISO7816_GET_CHALLENGE,
        0x00,
        0x00,
        NULL,
        0,
        bExpRsp,
        ppResponse,
        pRspLen);
}





/* MIFARE DUOX EV Charging command ------------------------------------------------------------------------------------------------------ */
phStatus_t phalMfDuoX_Sw_VdeReadData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bFileNo, uint16_t wBytesToRead,
    uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    /* Validate Parameters. */
    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_DEFAULT)
    {
        bP1 = bFileNo;
    }
    else if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        /* Do Nothing. Code is handled internally. */
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LE_ONLY,
        wOption,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_EV_CHARGING_CLA,
        PHAL_MFDUOX_CMD_VDE_READ_DATA,
        bP1,
        bP2,
        NULL,
        0,
        wBytesToRead,
        ppResponse,
        pRspLen);
}

phStatus_t phalMfDuoX_Sw_VdeWriteData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOperation, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint16_t wDataLen)
{
    /* Validate the parameter */
#ifdef RDR_LIB_PARAM_CHECK
    if(bOperation > PHAL_MFDUOX_EV_OPERATION_LOCK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
    }
#endif

    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LC_LE_BOTH,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_EV_CHARGING_CLA,
        PHAL_MFDUOX_CMD_VDE_WRITE_DATA,
        PHAL_MFDUOX_EV_CHARGING_VDE_WRITE_DATA_P1,
        bOperation,
        pData,
        wDataLen,
        0,
        NULL,
        NULL);
}

phStatus_t phalMfDuoX_Sw_VdeECDSASign(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wBytesToRead, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    /* Validate APDU Format. */
    PHAL_MFDUOX_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalMfDuoX_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_MFDUOX_EXCHANGE_LC_LE_BOTH,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_MFDUOX_ISO7816_EV_CHARGING_CLA,
        PHAL_MFDUOX_CMD_VDE_ECDSA_SIGN,
        PHAL_MFDUOX_EV_CHARGING_VDE_ECDSA_SIGN_P1,
        PHAL_MFDUOX_EV_CHARGING_VDE_ECDSA_SIGN_P2,
        pData,
        wDataLen,
        wBytesToRead,
        ppResponse,
        pRspLen);
}





#ifdef NXPBUILD__PHAL_MFDUOX_NDA
/* MIFARE DUOX Post-DeliveryConfiguration command --------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_AuthenticatePDC(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNoCard, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bUpgradeInfo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM *pRndA = NULL;
    uint8_t     PH_MEMLOC_REM *pRndB = NULL;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Clear buffers */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the pointer. */
    pRndA = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pRndB = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Compute Upgrade Key from ICUpgradeKey. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_Generate_ICUpgradeKey(
        pDataParams,
        wKeyNo,
        wKeyVer,
        bUpgradeInfo));

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_PDC_PART1;

    /* Clear buffers */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame Command PART1 ------------------------------------------------------------------------------------------------------------- */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wKeyNoCard & 0xFF00U) >> 8U);
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (wKeyNoCard & 0x00FFU);
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x01U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bUpgradeInfo;

    /* Exchange Part1 Command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_COMPLETE,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL));

    /* Validate Response Buffer length. */
    if(wRespLen != PH_CRYPTOSYM_AES_BLOCK_SIZE)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Process PART1 Response ---------------------------------------------------------------------------------------------------------- */

    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Decrypt the RndB received from PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pResponse,
        wRespLen,
        pRndB));

    /* Frame Command PART2 ------------------------------------------------------------------------------------------------------------- */

    /* Set the DataParams with command code. */
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_AUTHENTICATE_PDC_PART2;

    /* Clear command buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Add Command Code to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pDataParams->bCmdCode;

    /* Generate RndA */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Seed(pDataParams->pCryptoRngDataParams, pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE));
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, pRndA));

    /* Copy RndA to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRndA, PH_CRYPTOSYM_AES_BLOCK_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Copy RndB to Command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Shift RndB by 8 bits. */
    phalMfDuoX_Int_RotateLeft(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], PH_CRYPTOSYM_AES_BLOCK_SIZE, 1);

    /* Increment command buffer. */
    PHAL_MFDUOX_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Encrypt RndA + RndB' */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t)(PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        &PHAL_MFDUOX_CMD_BUF[1U],
        (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN - 1U),
        &PHAL_MFDUOX_CMD_BUF[1U]));

    /* Exchange Part2 Command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_COMPLETE,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_ON,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL));

    /* Process PART2 Response ---------------------------------------------------------------------------------------------------------- */
    /* Validate Response Length. */
    if(wRespLen != PH_CRYPTOSYM_AES_BLOCK_SIZE)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Decrypt Response to extract Decrypt TI || RndA' || PDCaps || PCDCaps */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pResponse,
        wRespLen,
        pResponse));

    /* Shift RndA by 8 bits. */
    phalMfDuoX_Int_RotateRight(pResponse, PH_CRYPTOSYM_AES_BLOCK_SIZE, 1U);

    /* Validate RndA ------------------------------------------------------------------------------------------------------------------- */
    if(memcmp(pRndA, pResponse, PH_CRYPTOSYM_AES_BLOCK_SIZE) != 0)
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFDUOX);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */





/* MIFARE DUOX Utility functions -------------------------------------------------------------------------------------------------------- */
phStatus_t phalMfDuoX_Sw_GetConfig(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PHAL_MFDUOX_ADDITIONAL_INFO:
            *pValue = pDataParams->wAdditionalInfo;
            break;

        case PHAL_MFDUOX_WRAPPED_MODE:
            *pValue = (uint16_t) pDataParams->bWrappedMode;
            break;

        case PHAL_MFDUOX_SHORT_LENGTH_APDU:
            *pValue = (uint16_t) pDataParams->bShortLenApdu;
            break;

        case PHAL_MFDUOX_AUTH_STATE:
            *pValue = (uint16_t) pDataParams->bAuthState;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_SetConfig(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PHAL_MFDUOX_ADDITIONAL_INFO:
            pDataParams->wAdditionalInfo = wValue;
            break;

        case PHAL_MFDUOX_WRAPPED_MODE:
            pDataParams->bWrappedMode = (uint8_t) wValue;
            break;

        case PHAL_MFDUOX_SHORT_LENGTH_APDU:
            pDataParams->bShortLenApdu = (uint8_t) wValue;
            break;

        case PHAL_MFDUOX_AUTH_STATE:
            pDataParams->bAuthState = (uint8_t) wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDUOX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_ResetAuthentication(phalMfDuoX_Sw_DataParams_t * pDataParams)
{
    phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_ISOGeneralAuthenticateVerify(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pPubBKey, uint16_t wPubBKeyLen,
    uint8_t * pOptsB, uint8_t bOptsBLen, uint8_t * pE_PubBKey, uint16_t wE_PubBKeyLen, uint8_t * pSignature, uint16_t wSignLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;

    uint8_t     PH_MEMLOC_REM aMsgBCont[2U] = { 0xE1U, 0xE1U };

    uint8_t     PH_MEMLOC_REM * pE_PubAKey = NULL;
    uint16_t    PH_MEMLOC_REM wE_PubAKeyLen = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the Pointers */
    pE_PubAKey = PHAL_MFDUOX_PRS_BUF;

    /* Gets the Host Ephemeral Public Key (E.Pub.A) */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_MFDUOX_PUBLIC_KEY_LEN,
        &bCurveID,
        pE_PubAKey,
        &wE_PubAKeyLen));

    /* Remove any Previously loaded keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_InvalidateKey(
        pDataParams->pCryptoDataParamsASym));

    /* Load Device known Public Key (Pub.B) */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKeyDirect(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | bCurveID),
        pPubBKey,
        wPubBKeyLen));

    /* Buffer Constant to buffer for verification. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        aMsgBCont,
        (uint16_t) sizeof(aMsgBCont),
        NULL,
        0));

    /* Buffer Opts.B to buffer for verification. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pOptsB,
        bOptsBLen,
        NULL,
        0));

    /* Buffer Device Ephemeral Public Key (E.Pub.B) */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pE_PubBKey,
        wE_PubBKeyLen,
        NULL,
        0));

    /* Buffer Host Ephemeral Public Key (E.Pub.A) */
    wStatus = phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pE_PubAKey,
        wE_PubAKeyLen,
        pSignature,
        wSignLen);

    /* Reset buffers and DataParams members. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Clear the buffer. */
        (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
        PHAL_MFDUOX_PRS_BUF_LEN = 0;

        /* Update AuthState to Authenticated. */
        pDataParams->bAuthState = PHAL_MFDUOX_ECC_AUTHENTICATED;
        pDataParams->wKeyType = PH_CRYPTOSYM_KEY_TYPE_AES128;

        /* Set TI and Command Counter to zero. */
        (void) memset(pDataParams->aTi, 0x00, sizeof(pDataParams->aTi));
        pDataParams->wCmdCtr = 0;

        /* Invalidate the ASymmetric Crypto context. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoASym_InvalidateKey(pDataParams->pCryptoDataParamsASym));

        /* Clear the local buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;
        PHAL_MFDUOX_CMD_BUF_OFFSET = 0;

        (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM);
        PHAL_MFDUOX_PRS_BUF_LEN = 0;
        PHAL_MFDUOX_PRS_BUF_OFFSET = 0;

        PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_OFF;
        PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_OFF;
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(wStatus, PH_COMP_AL_MFDUOX);
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_GenerateDAMEncKey(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo_PICCDAMEncKey,
    uint16_t wKeyVer_PICCDAMEncKey, uint16_t wKeyNo_AppDAMDefault, uint16_t wKeyVer_AppDAMDefault, uint8_t bAppDAMDefaultKeyVer,
    uint8_t ** ppDAMEncKey, uint16_t * pDAMEncKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType_AppDAMDefault = 0;
    uint16_t    PH_MEMLOC_REM wKeyType_PICCDAMEnc = 0;
    uint16_t    PH_MEMLOC_REM wEncLen = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM bRndLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey_AppDAMDefault = NULL;
    uint8_t     PH_MEMLOC_REM *pKey_PICCDAMEnc = NULL;
    uint8_t     PH_MEMLOC_REM *pIV = NULL;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pKey_AppDAMDefault = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];
    pKey_PICCDAMEnc = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pIV = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Backup the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_BackUpIV(pDataParams->pCryptoDataParamsEnc, &pIV, PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_AppDAMDefault,
        wKeyVer_AppDAMDefault,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey_AppDAMDefault,
        &wKeyType_AppDAMDefault));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType_AppDAMDefault);

    /* Evaluate Key Length. */
    bKeyLen = phKeyStore_GetKeySize(wKeyType_AppDAMDefault);

    /* Get Key out of the key store object for encryption. (PICCDMENC) */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_PICCDAMEncKey,
        wKeyVer_PICCDAMEncKey,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey_PICCDAMEnc,
        &wKeyType_PICCDAMEnc));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType_PICCDAMEnc);

    /* Load PICC DAM Encryption Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey_PICCDAMEnc,
        wKeyType_PICCDAMEnc));

    /* Load Zero Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Evaluate Random number Length. */
    bRndLen = (uint8_t) ((wKeyType_AppDAMDefault == PH_KEYSTORE_KEY_TYPE_AES256) ? 15U : 7U);

    /* Frame Data to be encrypted.*/
    PHAL_MFDUOX_CMD_BUF_LEN = bRndLen;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(
        pDataParams->pCryptoRngDataParams,
        bRndLen,
        PHAL_MFDUOX_CMD_BUF));

    /* Append Application DAM Default key. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pKey_AppDAMDefault, bKeyLen);
    PHAL_MFDUOX_CMD_BUF_LEN += bKeyLen;

    /* Modify Key Length based on KeyType for Application DAM Default Key.
     *  AES128: Key Length should be 24 bytes
     *  AES256: Key Length should be 32 bytes
     */
    if(wKeyType_AppDAMDefault == PH_KEYSTORE_KEY_TYPE_AES128)
        PHAL_MFDUOX_CMD_BUF_LEN += 8U;

    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bAppDAMDefaultKeyVer;

    /* Compute Encryption length. */
    PHAL_MFDUOX_NEAREST_MULTIPLE(PHAL_MFDUOX_CMD_BUF_LEN, wEncLen);

    /* Encrypt bWorkBuffer to obtain DAMEncKey */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        PHAL_MFDUOX_CMD_BUF,
        wEncLen,
        PHAL_MFDUOX_PRS_BUF));

    /* Copy the data to response buffer. */
    *ppDAMEncKey = PHAL_MFDUOX_PRS_BUF;
    *pDAMEncKeyLen = wEncLen;

    /* Set CryptoParams with Session Encryption key. */
    if(pDataParams->bAuthState == PHAL_MFDUOX_EV2_AUTHENTICATED)
    {
        /* Restore the IV. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            pIV,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Load back the session key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pDataParams->aSesAuthENCKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GenerateDAMMAC(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wKeyNo_PICCDAMMAC,
    uint16_t wKeyVer_PICCDAMMAC, uint8_t * pAid, uint8_t * pDamParams, uint8_t bDamParamsLen, uint8_t bKeySettings1,
    uint8_t bKeySettings2, uint8_t bKeySettings3, uint8_t * pKeySetValues, uint8_t bKeySetValuesLen, uint8_t * pISOFileId,
    uint8_t * pISODFName, uint8_t bISODFNameLen, uint8_t * pEncK, uint8_t bEncKLen, uint8_t ** ppDAMMAC, uint16_t * pDAMMACLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;
    uint8_t     PH_MEMLOC_REM *pIV = NULL;

    /* Validate Options */
    PHAL_MFDUOX_VALIDATE_APP_ISO_OPTIONS(bOption);
    PHAL_MFDUOX_VALIDATE_DAMMAC_OPTIONS(bOption);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];
    pIV = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    /* Backup the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_BackUpIV(pDataParams->pCryptoDataParamsMac, &pIV, PH_CRYPTOSYM_AES_BLOCK_SIZE));

    if(!(bOption & PHAL_MFDUOX_GENERATE_DAMMAC_DELETE_APPLICATION))
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_CREATE_DELEGATED_APPLICATION;
    }
    else
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_DELETE_APPLICATION;

    /* Add AID to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pAid, PHAL_MFDUOX_APP_ID_LEN);
    PHAL_MFDUOX_CMD_BUF_LEN += PHAL_MFDUOX_APP_ID_LEN;

    if(!(bOption & PHAL_MFDUOX_GENERATE_DAMMAC_DELETE_APPLICATION))
    {
        /* Add DAMParams to command buffer. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pDamParams, bDamParamsLen);
        PHAL_MFDUOX_CMD_BUF_LEN += bDamParamsLen;

        /* Add Key Settings to command buffer. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings1;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings2;
        if(bKeySettings2 & PHAL_MFDUOX_KEYSETT3_PRESENT)
        {
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bKeySettings3;
            if(bKeySettings3 & PHAL_MFDUOX_KEYSETVALUES_PRESENT)
            {
                (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pKeySetValues, bKeySetValuesLen);
                PHAL_MFDUOX_CMD_BUF_LEN += bKeySetValuesLen;
            }
        }

        /* Add ISO File ID to command buffer. */
        if(bOption & PHAL_MFDUOX_ISO_FILE_ID_AVAILABLE)
        {
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[0U];
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pISOFileId[1U];
        }

        /* Add ISO DFName to command buffer. */
        if(bOption & PHAL_MFDUOX_ISO_DF_NAME_AVAILABLE)
        {
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pISODFName, bISODFNameLen);
            PHAL_MFDUOX_CMD_BUF_LEN += bISODFNameLen;
        }

        /* Add Encryption Key (EncK) to command buffer. */
        (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pEncK, bEncKLen);
        PHAL_MFDUOX_CMD_BUF_LEN += bEncKLen;
    }

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_PICCDAMMAC,
        wKeyVer_PICCDAMMAC,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Load key to CryptoSym for Macing the data information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Load zero Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        sizeof(phalMfDuoX_Sw_ZeroIv)));

    /* Calculate MAC */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_PRS_BUF,
        &bMacLen));

    /* Truncate the Calculated CMAC */
    phalMfDuoX_Sw_Int_TruncateMac(PHAL_MFDUOX_PRS_BUF);

    /* Copy the Truncated CMAC into the return buffer */
    *ppDAMMAC = PHAL_MFDUOX_PRS_BUF;
    *pDAMMACLen = 8U;

    /* Set CryptoParams with Session MAC key. */
    if(pDataParams->bAuthState == PHAL_MFDUOX_EV2_AUTHENTICATED)
    {
        /* Restore the IV. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsMac,
            pIV,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Load back the session key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_GenerateDAMMACSetConfig(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo_PICCDAMMAC,
    uint16_t wKeyVer_PICCDAMMAC, uint8_t * pOld_ISODFName, uint8_t bOld_DFNameLen, uint8_t * pNew_ISODFName, uint8_t bNew_DFNameLen,
    uint8_t ** ppDAMMAC, uint16_t * pDAMMACLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    uint16_t    PH_MEMLOC_REM wKeyType;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    if((bOld_DFNameLen > 16U) || (bNew_DFNameLen > 16U))
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_PICCDAMMAC,
        wKeyVer_PICCDAMMAC,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Creation of command buffer with old DFNameLen and old DFName. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bOld_DFNameLen;
    memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOld_ISODFName, bOld_DFNameLen);
    PHAL_MFDUOX_CMD_BUF_LEN += 16U;

    /* Creation of working buffer with new DFNameLen and new DFName */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bNew_DFNameLen;
    memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pNew_ISODFName, bNew_DFNameLen);
    PHAL_MFDUOX_CMD_BUF_LEN += 16U;

    /* load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Calculate MAC */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_PRS_BUF,
        &bMacLen));

    /* Truncate the Calculated CMAC */
    phalMfDuoX_Sw_Int_TruncateMac(PHAL_MFDUOX_PRS_BUF);

    /* Copy the Truncated CMAC into the return buffer */
    *ppDAMMAC = PHAL_MFDUOX_PRS_BUF;
    *pDAMMACLen = 8;

    if(pDataParams->bAuthState != PHAL_MFDUOX_NOT_AUTHENTICATED)
    {
        /* Load back the session key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CalculateTMV(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wKeyNoTMACKey,
    uint16_t wKeyVerTMACKey, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen,
    uint8_t * pTMI, uint16_t wTMILen, uint8_t ** ppTMV, uint16_t * pTMVLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;

    uint8_t     PH_MEMLOC_REM *pMAC = NULL;

    /* Validate Diversification options and Length. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_OPTIONS(wOption);
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivInputLen);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pMAC = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Generate Session Transaction MAC Key (SesTMMAC). */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateSessionKey_TMAC(
        pDataParams,
        wOption,
        PHAL_MFDUOX_SESSION_MAC,
        wKeyNoTMACKey,
        wKeyVerTMACKey,
        pDivInput,
        bDivInputLen,
        pTMC,
        pUid,
        bUidLen));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Compute Transaction MAC Value. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        pTMI,
        wTMILen,
        pMAC,
        &bMacLen));

    /* Truncate the Calculated CMAC */
    phalMfDuoX_Sw_Int_TruncateMac(pMAC);

    /* Copy MAC to response buffer. */
    *ppTMV = pMAC;
    *pTMVLen = 8U;

    if(pDataParams->bAuthState != PHAL_MFDUOX_NOT_AUTHENTICATED)
    {
        /* Load the session key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_VerifyTransationSignature(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bCurveID, uint8_t * pPubKey,
    uint16_t wPubKey_Len, uint8_t * pTMI, uint16_t wTMILen, uint8_t * pTMC, uint8_t * pTSV, uint16_t wTSVLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Validate Curve ID */
    PHAL_MFDUOX_VALIDATE_CURVE(bCurveID);

    /* Update CurveID based on CryptoASym component. */
    bCurveID = (uint8_t) ((bCurveID == PHAL_MFDUOX_TARGET_CURVE_ID_NIST_P256) ? PH_CRYPTOASYM_CURVE_ID_SECP256R1 :
        PH_CRYPTOASYM_CURVE_ID_BRAINPOOL256R1);

    /* Load Transaction Signature Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKeyDirect(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | bCurveID),
        pPubKey,
        wPubKey_Len));

    /* Add Transaction MAC Input (TMI) for verification. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pTMI,
        wTMILen,
        NULL,
        0));

    /* Add Transaction MAC Counter (TMC) and Signature. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pTMC,
        4U,
        pTSV,
        wTSVLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_DecryptReaderID(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wKeyNo_TMACKey,
    uint16_t wKeyVer_TMACKey, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen,
    uint8_t * pEncTMRI, uint8_t bEncTMRILen, uint8_t ** ppTMRIPrev, uint16_t * pTMRIPrevLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pTMRIPrev = NULL;
    uint8_t     PH_MEMLOC_REM *pIV = NULL;

    /* Validate Diversification options and Length. */
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_OPTIONS(wOption);
    PHAL_MFDUOX_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivInputLen);

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pTMRIPrev = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];
    pIV = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Backup Initialization Vector. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_BackUpIV(pDataParams->pCryptoDataParamsEnc, &pIV,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate Session Transaction MAC Key (SesTMMAC). */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateSessionKey_TMAC(
        pDataParams,
        wOption,
        PHAL_MFDUOX_SESSION_ENC,
        wKeyNo_TMACKey,
        wKeyVer_TMACKey,
        pDivInput,
        bDivInputLen,
        pTMC,
        pUid,
        bUidLen));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt Reader ID. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pEncTMRI,
        bEncTMRILen,
        pTMRIPrev));

    /* Copy MAC to response buffer. */
    *ppTMRIPrev = pTMRIPrev;
    *pTMRIPrevLen = bEncTMRILen;

    if(pDataParams->bAuthState != PHAL_MFDUOX_NOT_AUTHENTICATED)
    {
        /* Restore the IV. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            pIV,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        /* Load the SessionENC key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pDataParams->aSesAuthENCKey,
            pDataParams->wKeyType));

        /* Load the SessionMAC key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_CalculateMACSDM(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bSdmOption, uint16_t wKeyNo_SDMMac,
    uint16_t wKeyVer_SDMMac, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen,
    uint8_t ** ppSDMMAC, uint16_t * pSDMMACLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bSDMMACLen = 0;

    uint8_t     PH_MEMLOC_REM *pSDMMAC = NULL;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pSDMMAC = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Generate and Load SDMFileRead Session MAC keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ComputeSessionKey_SDM(
        pDataParams,
        PHAL_MFDUOX_SESSION_MAC,
        bSdmOption,
        wKeyNo_SDMMac,
        wKeyVer_SDMMac,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Compute SDM Mac value. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        pInData,
        wInDataLen,
        pSDMMAC,
        &bSDMMACLen));

    /* Truncate the Calculated CMAC */
    phalMfDuoX_Sw_Int_TruncateMac(pSDMMAC);

    /* Copy MAC to response buffer. */
    *ppSDMMAC = pSDMMAC;
    *pSDMMACLen = 8U;

    if(pDataParams->bAuthState != PHAL_MFDUOX_NOT_AUTHENTICATED)
    {
        /* Load the SessionMAC key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_VerifySDMSignature(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wPubKeyNo_SDMSig,
    uint16_t wPubKeyPos_SDMSig, uint8_t * pInData, uint16_t wInDataLen, uint8_t * pSignature, uint16_t wSigLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Remove any previously loaded ASymmetric Keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_InvalidateKey(pDataParams->pCryptoDataParamsASym));

    /* Load Transaction Signature Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        wPubKeyNo_SDMSig,
        wPubKeyPos_SDMSig));

    /* Sign Transaction MAC Input. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_DEFAULT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pInData,
        wInDataLen,
        pSignature,
        wSigLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_DecryptSDMENCFileData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bSdmOption,
    uint16_t wKeyNo_SDMEnc, uint16_t wKeyVer_SDMEnc, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr,
    uint8_t * pEncData, uint16_t wEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Generate and Load SDMFileRead Session MAC keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_ComputeSessionKey_SDM(
        pDataParams,
        PHAL_MFDUOX_SESSION_ENC,
        bSdmOption,
        wKeyNo_SDMEnc,
        wKeyVer_SDMEnc,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Load zero Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Frame Initialization vector. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pSDMReadCtr[0U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pSDMReadCtr[1U];
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = pSDMReadCtr[2U];

    /* Get AES block size. */
    PHAL_MFDUOX_NEAREST_MULTIPLE(PHAL_MFDUOX_CMD_BUF_LEN, PHAL_MFDUOX_CMD_BUF_LEN);

    /* Now Encrypt the data Buffer to derive IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_CMD_BUF));

    /* Load computed Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        PHAL_MFDUOX_CMD_BUF,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt SDM Encrypted File Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pEncData,
        wEncDataLen,
        PHAL_MFDUOX_PRS_BUF));

    /* Copy decrypted data to response parameter. */
    *ppPlainData = PHAL_MFDUOX_PRS_BUF;
    *pPlainDataLen = wEncDataLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_DecryptSDMPICCData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pEncData, uint16_t wEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey,
        wKeyType));

    /* Load Zero Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt SDM Encrypted File Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pEncData,
        wEncDataLen,
        PHAL_MFDUOX_PRS_BUF));

    /* Copy decrypted data to response parameter. */
    *ppPlainData = PHAL_MFDUOX_PRS_BUF;
    *pPlainDataLen = wEncDataLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

#endif /* NXPBUILD__PHAL_MFDUOX_SW */
