/*
*         Copyright (c), NXP Semiconductors Bangalore / India
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
*particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X

#include "../phalMfIdentity_Int.h"
#include "phalMfIdentity_Sam_X.h"
#include "phalMfIdentity_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfIdentity_SamAV3_X_Init(phalMfIdentity_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalSamDataParams,
	phTMIUtils_t * pTMIDataParams)
{
    /* data param check */
    if (sizeof(phalMfIdentity_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFIDENTITY);
    PH_ASSERT_NULL_PARAM (pHalSamDataParams, PH_COMP_AL_MFIDENTITY);

    /* init private data */
	pDataParams->wId				= PH_COMP_AL_MFIDENTITY | PHAL_MFIDENTITY_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams	= pHalSamDataParams;
    pDataParams->pTMIDataParams		= pTMIDataParams;
    pDataParams->bAuthMode			= PHAL_MFIDENTITY_NOT_AUTHENTICATED;
    pDataParams->bAuthType          = 0xFF;
    pDataParams->bKeyNo				= 0xFF;
    pDataParams->wAdditionalInfo	= 0x0000;

	pDataParams->bFileOption			= 0x00;
	pDataParams->bSDMOption				= 0x00;
	pDataParams->bUidLength				= 0x00;
	pDataParams->bKeyType				= 0x00;	
	pDataParams->bSDMKeyNo				= 0xFF;	
	pDataParams->bSDMKeyVer				= 0xFF;	
	pDataParams->bRAMKeyNo				= 0xFF;	
	pDataParams->bRAMKeyVer				= 0xFF;

	memset(pDataParams->aAid, 0x00, 3);				/* PRQA S 3200 */
	memset(pDataParams->aUid, 0x00, 3);				/* PRQA S 3200 */
	memset(pDataParams->aSDMReadCtr, 0x00, 3);		/* PRQA S 3200 */
	memset(pDataParams->aSDMENCLen, 0x00, 3);		/* PRQA S 3200 */
	memset(pDataParams->aSDMENCOffset, 0x00, 3);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

/* MIFARE Identity Secure Messaging commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sam_X_AuthenticateEv2(void *pDataParams, uint8_t bAuthType, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput,
	uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Check if First Auth parameter do not contain invalid values. */
	if(bAuthType > PHAL_MFIDENTITY_AUTHFIRST_LRP)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	/* Validate the PCD Input Capability length */
	if(bAuthType < PHAL_MFIDENTITY_AUTHNONFIRST_LRP)
	{
		/* For EV2 mode the length should not exceed 6 bytes. */
		if(bPcdCapsInLen > 6)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
	else
	{
		/* For LRP mode the length should at least be 1 byte. */
		if((bPcdCapsInLen == 0) && (bAuthType == PHAL_MFIDENTITY_AUTHFIRST_LRP))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}

#ifdef RDR_LIB_PARAM_CHECK
	if(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
	{
		/* At PICC level valid key numbes are 1 - 4. */
		if((bKeyNoCard > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST ) || (bKeyNoCard < PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
	else
	{
		/* At App level valid key numbers are 0 - 4. */
		if((bKeyNoCard > PHAL_MFIDENTITY_APP_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivInputLen > 31 ))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif

    /* Validate SAM Key number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/*
	 * Set EV2 Authentication type.
	 * EVx authentication type
	 * Authenticate first
	 * Authenticate non first
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
	bAuthMode = (uint8_t) ((bAuthType & 0x01) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH : PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_NON_FIRST_AUTH);

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) (bAuthMode | ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON :
		PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF));

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_KDF_AV2);
	}

	/* Set the secure messaging. */
	if((bKeyNoCard >= PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_SUPPRESS_SECURE_MESSAGING);
	}

	/* Reset the Authentication state. */
	if(bAuthType & 0x01)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ResetAuthentication(pDataParams));
	}

    wStatus = phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
        bAuthMode,
        PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
		bPcdCapsInLen,
		bPcdCapsInLen ? pPcdCapsIn : NULL,
        pDivInput,
        bDivInputLen,
		pPdCapsOut,
		pPcdCapsOut,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_ON, wStatus, aPiccRetCode));

	/* Update the authentication states. */
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFIDENTITY_AUTHENTICATEEV2;
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = bKeyNoCard;

	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType = (uint8_t) ((bAuthType < PHAL_MFIDENTITY_AUTHNONFIRST_LRP) ?
		PHAL_MFIDENTITY_AUTH_TYPE_EV2 : PHAL_MFIDENTITY_AUTH_TYPE_LRP);

	/* Do not update the auth state if originality keys are used. */
	if((bKeyNoCard >= PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_ResetAuthentication(pDataParams));
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity Memory and Configuration commands. ---------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[28];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif
	
	/* Clear the local buffers. */
	memset(aAppData, 0x00, sizeof(aAppData)); /* PRQA S 3200 */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode)); /* PRQA S 3200 */

	/* Set the Length and Crypto information. */
	aAppData[wAppDataLen++] = 6;

	/* Set Extended Offset in Crypto mode. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

	/* Set Length Presence flag. */
	bLenPresent = PH_ON;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_SET_CONFIG;
	aAppData[wAppDataLen++] = bOption;

    memcpy(&aAppData[wAppDataLen], pData, bDataSize); /* PRQA S 3200 */
	wAppDataLen = (uint16_t) (wAppDataLen + bDataSize);

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;

	/* Set the Command Offset. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - bDataSize));
	}

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Reset the Authentication if configuration is LRP enabling. */
	if((bOption == PHAL_MFIDENTITY_SET_CONFIG_OPTION5) && (pData[4] == 0x02))
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_ResetAuthentication(pDataParams));
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint16_t	PH_MEMLOC_REM wVerLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_GET_VERSION;

	do
	{
		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
			PH_ON,
			PH_OFF,
			0,
			PH_ON,
			0,
			aAppData,
			&wAppDataLen));

		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Updated Finished to end the loop. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			wAppDataLen = 0;
			aAppData[wAppDataLen++] = 0xAF;

			/* Copy the response information to the parameter. */
			memcpy(&pVerInfo[wVerLen], pResponse, wRespLen); /* PRQA S 3200 */
			wVerLen = (uint8_t) (wVerLen + wRespLen);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_SetConfig(pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, (uint8_t * ) &wVerLen));
	*pVerLen = (uint8_t) wVerLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_GetCardUID(void * pDataParams, uint8_t * pUid, uint8_t * pUidLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif

	/* Append expected length to the command frame. */
	aAppData[wAppDataLen++] = 7;
	aAppData[wAppDataLen++] = 0;
	aAppData[wAppDataLen++] = 0;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_GET_CARD_UID;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_ON,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the response to the parameter. */
	memcpy(pUid, pResponse, wRespLen);	/* PRQA S 3200 */
	*pUidLen = (uint8_t) wRespLen;

	/* Do a Set Config of ADDITIONAL_INFO to set  the length of the UID string */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_SetConfig(pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, (uint8_t *) &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity Key Management commands. -------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
	uint8_t		PH_MEMLOC_REM aPiccRetCode[2];

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level ChangeKey cannot be performed. */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
    else
    {
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNoCard > PHAL_MFIDENTITY_APP_KEY_LAST)
		{
			/* Invalid application key specified */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
    }

	if (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif

    /* Check for valid SAM key number and version. */
    if ((wCurrKeyNo > 0x7f) || (wCurrKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

    /* Set the key compilation method. */
	if(wOption == PHAL_MFIDENTITY_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
#ifdef RDR_LIB_PARAM_CHECK
		/* Change for valid diversification options. */
		if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) &&
			((wOption & PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2) != PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}

		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
#endif

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Desfire key number to be changed. */
	bCfg = bKeyNoCard;

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY);
    }

	/* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816);

	/* Exchange the command to PICC. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		bKeyCompMeth,
		bCfg,
		0x00,
		0x00,
		(uint8_t)wCurrKeyNo,
		(uint8_t)wCurrKeyVer,
		(uint8_t)wNewKeyNo,
		(uint8_t)wNewKeyVer,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	wStatus = phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Kill the PICC Authentication. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

		return wStatus;
	}
	else
	{
		/* Reset authentication status only if the key authenticated with is changed. */
		if ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_X_ResetAuthentication(pDataParams));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion, uint8_t * pKeyVerLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
		if((bKeyNo > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST ) || (bKeyNo < PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
    }
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNo > PHAL_MFIDENTITY_APP_KEY_LAST)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
#endif

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_GET_KEY_VERSION;
	aAppData[wAppDataLen++] = bKeyNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pKeyVersion, pResponse, wRespLen);	/* PRQA S 3200 */
	*pKeyVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity File Management commands. ------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_CreateTransactionMacFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint16_t wKeyNo, uint8_t bKeyVer,
	uint8_t bKeyType, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

#ifdef RDR_LIB_PARAM_CHECK
    /* Check for valid file no. and KeyType == AES128 */
	if (bKeyType != PHAL_MFIDENTITY_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    if ((bFileOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
		(bFileOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1) &&
        (bFileOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif

	/* Check for valid SAM keystore number and version. */
    if (wKeyNo > 0x7f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	wStatus = phhalHw_SamAV3_Cmd_DESFire_CreateTMFilePICC(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		(uint8_t) (bDivInputLen ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON : PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF),
		PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816,
		(uint8_t) wKeyNo,
		bKeyVer,
		bFileNo,
		(uint8_t) (bFileOption >> 4),
		pAccessRights,
		bKeyType,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

	/* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_ON, wStatus, aPiccRetCode));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_DELETE_FILE;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto Mode. */
	bCrypto = (uint8_t) ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) ?
		PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC : PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN);

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_GET_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pFSBuffer, pResponse, wRespLen);	/* PRQA S 3200 */
	*pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_GetFileCounters(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pResponse, uint8_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponseTmp = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
	
	/* Validate the parameters */
	if ((bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFIDENTITY_COMMUNICATION_ENC))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Append expected length to the command frame. */
	if (bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC)
	{
		aAppData[wAppDataLen++] = 5;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_GET_FILE_COUNTERS;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bOption,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponseTmp,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pResponse, pResponseTmp, wRespLen);	/* PRQA S 3200 */
	*pRespLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t bTMCLimitLen,
	uint8_t * pTMCLimit)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[16];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;

	/* Frame the Crypto information.
	 * Masking the MSB bit becuase its not support by SAM.
	 * Its actually supported in Software only.
	 */
	bCrypto = (uint8_t) (bOption & 0xFE);

	/* Validate the parameters */
	if ((bCrypto != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        (bCrypto != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bCrypto != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Set the Length and Crypto information. */
	if((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) &&
		(bCrypto == PHAL_MFIDENTITY_COMMUNICATION_ENC))
	{
		/* Start of data. */
		aAppData[wAppDataLen++] = 6;

		/* Set Extended Offset in Crypto mode. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_CHANGE_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;
	aAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
	memcpy(&aAppData[wAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	wAppDataLen += 2;

	/* TMCLimit buffer in command buffer if Bit5 of File Option is SET. */
	if (bFileOption & PHAL_MFIDENTITY_TMCLIMITCONFIG)
    {
		memcpy(&aAppData[wAppDataLen], pTMCLimit, bTMCLimitLen); /* PRQA S 3200 */
        wAppDataLen += bTMCLimitLen;
    }

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_ChangeFileSettingsSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t bSdmOptions,
	uint8_t * pSdmAccessRights, uint8_t * pPICCDataOffset, uint8_t * pSDMMACInputOffset, uint8_t * pSDMENCOffset, uint8_t * pSDMENCLen, uint8_t * pSDMMACOffset,
	uint8_t * pSDMReadCtrOffset, uint8_t * pTMCLimitValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[40];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
	uint8_t		PH_MEMLOC_REM bTMCLimitLen = 0;

	/* Validate the parameters */
	if ((bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Set the Length and Crypto information. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		/* Start of data. */
		aAppData[wAppDataLen++] = 6;

		/* Set Extended Offset in Crypto mode. */
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_CHANGE_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;
	aAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
	if(pAccessRights != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
		wAppDataLen += 2;
	}

	/* Append SDMOption information. */
	aAppData[wAppDataLen++] = bSdmOptions;

	/* Append SDM access rights. */
	if(pSdmAccessRights != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSdmAccessRights, 2); /* PRQA S 3200 */
		wAppDataLen += 2;
	}

	/* Append PICC Data Offset. */
	if(pPICCDataOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pPICCDataOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM MAC Input Offset. */
	if(pSDMMACInputOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMMACInputOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM ENC Offset. */
	if(pSDMENCOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMENCOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM ENC Length. */
	if(pSDMENCLen != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMENCLen, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM MAC Offset. */
	if(pSDMMACOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMMACOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append SDM Read Counter. */
	if(pSDMReadCtrOffset != NULL)
	{
		memcpy(&aAppData[wAppDataLen], pSDMReadCtrOffset, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
	}

	/* Append TMC Limit. */
	if(pTMCLimitValue != NULL)
	{
		/* Evaluate TMCLimitLen. */
		bTMCLimitLen = (uint8_t) ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_LRP) ? 2 :
			(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_EV2) ? 4 : 0);

		memcpy(&aAppData[wAppDataLen], pTMCLimitValue, bTMCLimitLen); /* PRQA S 3200 */
		wAppDataLen += bTMCLimitLen;
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Set the communication option. */
	bCrypto |= bOption;

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity Data Management commands. ------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse,
	uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint32_t	PH_MEMLOC_REM dwLength = 0;
	uint16_t	PH_MEMLOC_REM wRespOffset = 0;

	uint8_t		PH_MEMLOC_REM bIsSDMEnabled = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsSDMENCFileDataEnabled = PH_OFF;
	uint32_t	PH_MEMLOC_REM dwPICCDataLen = 0;
	uint8_t		PH_MEMLOC_REM bSDMMACLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMENCLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMENCOffset = 0;
	uint16_t	PH_MEMLOC_REM wSDMDataLen = 0;
	uint8_t		PH_MEMLOC_REM aPlainData[256];

	uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

	/* Validate the parameter. */
    if (((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Frame the Crypto information. */
	bCrypto = (uint8_t) (bOption & 0xF0);

	/* Frame Presence of length information in the command frame.
	 * The first three bytes specifies number of bytes to be received from PICC.
	 */
	bLenPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

	/* Add Read Length if communication mode is Encrypted. */
	if(bLenPresent)
	{
		memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
		wAppDataOffset = 3;
	}

    aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_READ_DATA_ISO;
    aAppData[wAppDataLen++] = bFileNo;

    memcpy(&aAppData[wAppDataLen], pOffset, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

    memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Compute the length. */
		dwLength = (uint32_t) (pLength[0] | (pLength[1] << 8) | (pLength[2] << 16));

		/* Frame the Option. */
		bTMIOption = (uint8_t)(dwLength ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

		/* Buffer the Command information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0, PHAL_MFIDENTITY_BLOCK_SIZE));
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		((pLength[0] | (pLength[1] << 8) | (pLength[2] << 8)) + 7),
		aAppData,
		&wAppDataLen));

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pRespLen += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Get the status of SDM availability. */
		bIsSDMEnabled = (uint8_t) ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bFileOption & PHAL_MFIDENTITY_FILE_OPTION_SDM_ENABLED) ? 
			PH_ON : PH_OFF);

		/* Decrypt the SDMENCFileData. */
		if(bIsSDMEnabled && ((bOption & 0xF0) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN) && 
			(bFileNo == PHAL_MFIDENTITY_SDM_FILE_ID) && 
			(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED))
		{
			/* Set the SDMPICCDataLen. */
			dwPICCDataLen = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType ? 24 : 16;

			/* Set SDMMAC length. */
			bSDMMACLen = 8;

			/* Get the status of SDMENCFileData */
			bIsSDMENCFileDataEnabled = (uint8_t) ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMOption & PHAL_MFIDENTITY_SDM_ENC_FILE_DATA_PRESENT) ? 
				PH_ON : PH_OFF);

			if(bIsSDMENCFileDataEnabled)
			{
				/* Convert the SDMENCOffset buffer. */
				dwSDMENCOffset = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCOffset[0]       | 
								(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCOffset[1] << 8) | 
								(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCOffset[2] << 16);

				/* Convert the SDMENCLength buffer. */
				dwSDMENCLen = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCLen[0]       | 
							 (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCLen[1] << 8) | 
							 (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCLen[2] << 16);

				/* Update the SDMENCLen. */
				if(*pRespLen < (dwSDMENCLen + (dwPICCDataLen + bSDMMACLen )))
				{
					dwSDMENCLen = *pRespLen - (dwPICCDataLen + bSDMMACLen );
				}

				/* Set the SDMENCFileOffset from the complete response. */
				wRespOffset = (uint16_t) (dwPICCDataLen + dwSDMENCOffset);

				/* Decrypt the SDMENCFileData. */
				PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_DecryptSDMENCFileData(
					pDataParams, 
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyNo, 
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyVer, 
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyNo,
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyVer,
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aUid,
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bUidLength,
					PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMReadCtr,
					&ppResponse[0][wRespOffset],
					(uint16_t) dwSDMENCLen,
					aPlainData));
			}

			/* Compute the Remianing PlainData. */
			wSDMDataLen = (uint16_t) (dwPICCDataLen + dwSDMENCLen + bSDMMACLen);
		}

		/* Frame the Option. */
		bTMIOption = (uint8_t) (dwLength ? PH_TMIUTILS_ZEROPAD_DATABUFF : (PH_TMIUTILS_ZEROPAD_DATABUFF | PH_TMIUTILS_READ_INS));
		bTMIOption = (uint8_t) ((dwSDMENCLen && bIsSDMEnabled && ((bOption & 0xF0) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN) && 
			(bFileNo == PHAL_MFIDENTITY_SDM_FILE_ID)) ? (PH_TMIUTILS_NO_PADDING | (bTMIOption & PH_TMIUTILS_READ_INS)) : bTMIOption);

		/* Collect the data received. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, NULL, 0, &ppResponse[0][dwPICCDataLen], (uint16_t) (dwSDMENCLen ? dwSDMENCOffset : (*pRespLen - wSDMDataLen)), 
			PHAL_MFIDENTITY_BLOCK_SIZE));

		/* Collect the data received. */
		if(dwSDMENCLen && bIsSDMEnabled && ((bOption & 0xF0) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN) && 
			(bFileNo == PHAL_MFIDENTITY_SDM_FILE_ID) && (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED))
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_NO_PADDING, NULL, 0, aPlainData, dwSDMENCLen, PHAL_MFIDENTITY_BLOCK_SIZE));

			/* Collect the data received. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				bTMIOption, NULL, 0, &ppResponse[0][dwPICCDataLen + dwSDMENCOffset + dwSDMENCLen], 
				(uint16_t) (*pRespLen - (wSDMDataLen + dwSDMENCOffset)), PHAL_MFIDENTITY_BLOCK_SIZE));
		}

		/* Reset the TMI buffer Offset. */
		if (!dwLength)
		{
			/* Reset wOffsetInTMI */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFIDENTITY_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

	/* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
	if ((bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
		(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 12;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_WRITE_DATA_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);
	wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			(PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset),
			pData, dwTotLen, PHAL_MFIDENTITY_BLOCK_SIZE));
	}

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFIDENTITY_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFIDENTITY_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
				PH_ON,
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 7),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) && bIsFirstFrame &&
			(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2))
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfIdentity_Sam_X_ReadRecords(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo, uint8_t * pRecCount, uint8_t * pRecSize,
	uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
	uint32_t	PH_MEMLOC_REM dwNumRec = 0;
	uint32_t	PH_MEMLOC_REM dwRecLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;
	uint32_t	PH_MEMLOC_REM dwDataLen = 0;
    uint32_t    PH_MEMLOC_REM dwNumRecCal = 0;
	uint32_t	PH_MEMLOC_REM dwTMIBufOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTMIBufIndex = 0;

	/* Validate the parameter. */
    if (((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Frame the Crypto information. */
	bCrypto = (uint8_t) (bOption & 0xF0);

	/* Frame Presence of length information in the command frame.
	 * The first three bytes specifies number of bytes to be received from PICC.
	 */
	bLenPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

	/* Add Read Length if communication mode is Encrypted. */
	if(bLenPresent)
	{
		memcpy(&aAppData[wAppDataLen], pRecSize, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
		wAppDataOffset = 3;
	}

    aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_READ_RECORDS_ISO;
    aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pRecNo, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pRecCount, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Compute the number of records. */
		dwNumRec = pRecCount[2];
		dwNumRec = dwNumRec << 8 | pRecCount[1];
		dwNumRec = dwNumRec << 8 | pRecCount[0];

		/* Compute the record length. */
		dwRecLen = pRecSize[2];
		dwRecLen = dwRecLen << 8 | pRecSize[1];
		dwRecLen = dwRecLen << 8 | pRecSize[0];

		/* Should should provide atleast RecLen / NumRec to update in TMI collection */
		if(!dwRecLen && !dwNumRec)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}

		/* Frame the Option. */
		bTMIOption = (uint8_t)(dwNumRec ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

		/* Buffer the Command information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0, PHAL_MFIDENTITY_BLOCK_SIZE));
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		(pRecSize[0] | (pRecSize[1] << 8) | (pRecSize[2] << 8)) + 7,
		aAppData,
		&wAppDataLen));

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pRespLen += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		if (!dwNumRec)
		{
			/* Get the TMI buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, &dwTMIBufOffset));

			/* Get the current TMI buffer index. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_BUFFER_INDEX, &dwTMIBufIndex ));

			/* Calculate Response length in case of chaining. */
			dwDataLen = *pRespLen + dwTMIBufIndex -(dwTMIBufOffset + 11);

			/* If user update worng RecSize, we cant calculate recCnt. */
			if(dwDataLen % dwRecLen)
			{
				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
			}

			/* Calculate number of records. */
			dwNumRecCal = dwDataLen / dwRecLen ;

			/* Update the value as TMI buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_VALUE, dwNumRecCal));
		}

		/* Collect the data received. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, *ppResponse, *pRespLen, PHAL_MFIDENTITY_BLOCK_SIZE));

		/* Reset the TMI buffer Offset. */
		if (!dwNumRec)
		{
			/* Reset wOffsetInTMI */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
		}

        if (!dwNumRec)
        {
			/* Reset TMI Buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_WriteRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFIDENTITY_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
	if ((bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
		(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 12;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_WRITE_RECORD_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);
	wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			(PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset),
			pData, dwTotLen, PHAL_MFIDENTITY_BLOCK_SIZE));
	}

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFIDENTITY_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFIDENTITY_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
				PH_ON,
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 7),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFIDENTITY_COMMUNICATION_ENC) && bIsFirstFrame &&
			(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2))
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfIdentity_Sam_X_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_CLEAR_RECORD_FILE;
	aAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_CMDBUFF, aAppData, 2, NULL, 0, PHAL_MFIDENTITY_BLOCK_SIZE));
    }

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity Transaction Management commands. ------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sam_X_CommitTransaction(void * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMV)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if ((bOption & 0x0F) > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_COMMIT_TXN;

	/*
	 * 0x01 and 0x00 are values to support the legacy implementation.
	 *		Here if it 0x00 the Option is not exchanged to the PICC. Its only exchanged for 0x01
	 *
	 * 0x80 and 0x81 are the values to exchange the Option by to PICC.
	 *		If its 0x80, Option byte will be exchanged by the Option information will be zero.
	 *		If its 0x81, Option byte will be exchanged by the Option information will be one.
	 */
	if((bOption & 0x80) || (bOption & 0x0F))
	{
		aAppData[wAppDataLen++] = (bOption & 0x0F);
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	if(bOption)
	{
		memcpy(pTMC, &pResponse[0], 4);	/* PRQA S 3200 */
		memcpy(pTMV, &pResponse[4], 8);	/* PRQA S 3200 */
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_AbortTransaction(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_ABORT_TXN;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_CommitReaderID(void * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aAppData[26];
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wEncTMRILen = 0;
	uint8_t		PH_MEMLOC_REM aPiccErrCode[2];
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Exchange the details to SAM hardware and get the ENC TMRI. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		wStatus = phhalHw_SamAV3_Cmd_TMRI_CommitReaderID(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PHHAL_HW_CMD_SAMAV3_TMRI_ISO_MODE_ISO7816,
			PHHAL_HW_CMD_SAMAV3_COMMIT_READER_ID_PICC_STATE_DESFIRE,
			0,
			&pResponse,
			&wEncTMRILen,
			aPiccErrCode);

		/* Copy the Response to the parameter. */
		memcpy(pEncTMRI, pResponse, wEncTMRILen);	/* PRQA S 3200 */
		wRespLen = wEncTMRILen;

		/* Check TMI Collection Status */
		if (dwTMIStatus && (wStatus == PH_ERR_SUCCESS))
		{
			/* If authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
			 * TMI = TMI || Cmd || TMRICur || EncTMRI || ZeroPadding
			 */

			/* Clear the buffers. */
			wAppDataLen = 0;
			memset(aAppData, 0x00, sizeof(aAppData));	/* PRQA S 3200 */

			/* Frame the command and send it to card. */
			aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_COMMIT_READER_ID;

			/* Frame the TMRI information. */
			memcpy(&aAppData[wAppDataLen], PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams->bUid, 7 );	/* PRQA S 3200 */
			wAppDataLen += 7;

			aAppData[wAppDataLen] = 0x80;

			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_DATABUFF, aAppData, 17, pEncTMRI, 16, PHAL_MFIDENTITY_BLOCK_SIZE));
		}
	}
	else
	{
		/* Clear the buffers. */
		wAppDataLen = 0;
		memset(aAppData, 0x00, sizeof(aAppData));	/* PRQA S 3200 */

		/* Frame the command and send it to card. */
		aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_COMMIT_READER_ID;

		/* Add the TMRI to command buffer. */
		memcpy(&aAppData[wAppDataLen], pTMRI, 16 );	/* PRQA S 3200 */
		wAppDataLen += 16;

		/* Check TMI Collection Status */
		if (dwTMIStatus)
		{
			/* If not authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
			 * TMI = TMI || Cmd || TMRICur || ZeroPadding
			 */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_CMDBUFF, aAppData, wAppDataLen, NULL, 0, PHAL_MFIDENTITY_BLOCK_SIZE));
		}

		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
			PH_ON,
			PH_OFF,
			0,
			PH_ON,
			(uint16_t) (wAppDataLen - 1),
			aAppData,
			&wAppDataLen));

		/* Exchange the data to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen);

		/* Set the PICC error code. */
		aPiccErrCode[0] = pResponse[wRespLen];
		aPiccErrCode[1] = pResponse[wRespLen - 1];

		/* Manipulate the status. */
		if((pResponse[wRespLen] != 0x91) && (pResponse[wRespLen - 1] != 0x00))
		{
			wStatus = PH_ADD_COMPCODE(PHHAL_HW_SAMAV3_ERR_MIFARE_GEN, PH_COMP_HAL);
		}

		/* Decrement the RespLen to remove the status. */
		wRespLen -= 2;
	}

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccErrCode));

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the recieved TMRI */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_SetConfig(pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, (uint8_t *) &wRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity ISO7816-4 commands. ------------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aAppData[25];
	uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aFileId[3] = {'\0'};
	uint16_t    PH_MEMLOC_REM wVal = 0;
	uint8_t		PH_MEMLOC_REM aPiccDfName[7] = {0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00};

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if (bDFnameLen > 16)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	/* If User intends to select through DFName, check for validity of DFName passed */
	if((bSelector ==  PHAL_MFIDENTITY_SELECTOR_4) && (bDFnameLen == 0x00))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	if ((bOption != PHAL_MFIDENTITY_FCI_RETURNED) && (bOption != PHAL_MFIDENTITY_FCI_NOT_RETURNED))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	if((bSelector != PHAL_MFIDENTITY_SELECTOR_4) && (bSelector != PHAL_MFIDENTITY_SELECTOR_3) && (pFid == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif

	/* Frame the command. */
	aAppData[wAppDataLen++] = 0x00;									/* CLA */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_ISO7816_SELECT_FILE; /* INS */
	aAppData[wAppDataLen++] = bSelector;							/* P1 */
	aAppData[wAppDataLen++] = bOption;								/* P2 */

	/* Append LC. */
	if(bSelector != PHAL_MFIDENTITY_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
			aAppData[wAppDataLen++] = 0;
		}
	}

	/* Append the payload and LC. */
	if(bSelector == PHAL_MFIDENTITY_SELECTOR_4)
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = bDFnameLen;

		memcpy(&aAppData[wAppDataLen], pDFname, bDFnameLen); /* PRQA S 3200 */
		wAppDataLen += bDFnameLen;
	}
	else if((bSelector ==  PHAL_MFIDENTITY_SELECTOR_0) || (bSelector == PHAL_MFIDENTITY_SELECTOR_1) ||
		(bSelector == PHAL_MFIDENTITY_SELECTOR_2))
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = 2;

		/* Select MF, DF or EF, by file identifier
		 * Select child DF
		 * Select EF under the current DF, by file identifier
		 * Select parent DF of the current DF
		 */
		aFileId[1] = aAppData[wAppDataLen++] = pFid[1];
		aFileId[0] = aAppData[wAppDataLen++] = pFid[0];
		aFileId[2] = 0;
	}
	else if(bSelector == PHAL_MFIDENTITY_SELECTOR_3)
	{
		/* For Selector as 0x03, there is nothing to be done. */
	}
	else
	{
		aAppData[wAppDataLen++] = 0;
	}

	/* Append LE only if there is an LC. */
	if(bSelector != PHAL_MFIDENTITY_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
		}
		aAppData[wAppDataLen++] = 0;
	}

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen));

	/* Manipulate the status. */
	wStatus = phalMfIdentity_Int_ComputeErrorResponse(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams),
		((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]));

	/* Decrement the RespLen to remove the status. */
	wRespLen -= 2;

	if((wStatus & PH_ERR_MASK) == PHAL_MFIDENTITY_ERR_DF_7816_GEN_ERROR)
	{
		wVal = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
	}

	if((wStatus == PH_ERR_SUCCESS) || (wVal == PHAL_MFIDENTITY_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS))
	{
		/* Reset Authentication should not be targeted for elementary file selection using file ID */
		if(bSelector !=  PHAL_MFIDENTITY_SELECTOR_2)
		{
			/* Reset Authentication Status here */
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfIdentity_Sam_X_ResetAuthentication(pDataParams));
		}

		/* once the selection Success, update the File Id to master data structure if the selection is done through AID */
		if((bSelector ==  PHAL_MFIDENTITY_SELECTOR_0) || (bSelector == PHAL_MFIDENTITY_SELECTOR_1) ||
			(bSelector == PHAL_MFIDENTITY_SELECTOR_2))
		{
			memcpy(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else if((bSelector ==  PHAL_MFIDENTITY_SELECTOR_4))
		{
			/* Update the file ID to all zeros if DF Name is of PICC. */
			if(memcmp( pDFname, aPiccDfName, 7) == 0)
			{
				aFileId[0] = 0x00;
				aFileId[1] = 0x00;
				aFileId[2] = 0x00;
			}
			else
			{
				aFileId[0] = 0xff;
				aFileId[1] = 0xff;
				aFileId[2] = 0xff;
			}

			memcpy(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else
		{
			/* Nothing for Secector 0x03. */
		}
	}
	else
	{
		return wStatus;
	}

	/* Copy the response to the buffer */
	*ppFCI = pResponse;
	*pFCILen = wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity OriginalityCheck commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sam_X_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	}

	/* Add Length information is Communication mode is FULL. */
	if(bCrypto == PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL)
	{
		aAppData[wAppDataLen++] = 56;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		/* Set presence of Length information. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFIDENTITY_CMD_READ_SIG;
	aAppData[wAppDataLen++] = bAddr;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		pSignature,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFIDENTITY_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE DESFire Light Miscellaneous commands. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sam_X_ResetAuthentication(void * pDataParams)
{
    return phalMfIdentity_Sam_X_Int_ResetAuthentication(pDataParams);
}

phStatus_t phalMfIdentity_Sam_X_GetConfig(void * pDataParams, uint16_t wConfig, uint8_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFIDENTITY_ADDITIONAL_INFO:
			pValue[0] = (uint8_t ) (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo & 0x00FF);
			pValue[1] = (uint8_t ) ((PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo & 0xFF00) >> 8);
			break;

		case PHAL_MFIDENTITY_FILE_OPTION:
			*pValue = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bFileOption;
			break;

		case PHAL_MFIDENTITY_SDM_OPTION:
			*pValue = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMOption;
			break;

		case PHAL_MFIDENTITY_KEY_INFO:
			pValue[0] = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyNo;
			pValue[1] = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyVer;
			break;

		case PHAL_MFIDENTITY_SDM_ENC_OFFSET:
			memcpy(pValue, PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCOffset, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_ENC_LENGTH:
			memcpy(pValue, PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCLen, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_READ_CTR:
			memcpy(pValue, PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMReadCtr, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_KEY_TYPE:
			*pValue = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType;
			break;

		case PHAL_MFIDENTITY_RAM_KEY_INFO:
			pValue[0] = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyNo;
			pValue[1] = PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyVer;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint8_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFIDENTITY_ADDITIONAL_INFO:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo = (uint16_t) (pValue[0] | (pValue[1] << 8));
			break;

		case PHAL_MFIDENTITY_FILE_OPTION:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bFileOption = pValue[0];
			break;

		case PHAL_MFIDENTITY_SDM_OPTION:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMOption = pValue[0];
			break;

		case PHAL_MFIDENTITY_KEY_INFO:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyNo = pValue[0];
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bSDMKeyVer = pValue[1];
			break;

		case PHAL_MFIDENTITY_SDM_ENC_OFFSET:
			memcpy(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCOffset, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_ENC_LENGTH:
			memcpy(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMENCLen, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_READ_CTR:
			memcpy(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->aSDMReadCtr, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_KEY_TYPE:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType = pValue[0];
			break;

		case PHAL_MFIDENTITY_RAM_KEY_INFO:
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyNo = pValue[0];
			PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bRAMKeyVer = pValue[1];
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_CalculateTMV(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
	uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI, uint32_t dwTMILen, uint8_t * pTMV)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pMac = NULL;
	uint16_t	PH_MEMLOC_REM wMacLen = 0;

	uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
	uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
	uint16_t	PH_MEMLOC_REM wRemLen = 0;
	uint16_t	PH_MEMLOC_REM wTMIOffset = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Derive Transaction MAC (KSesTMMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ComputeTMACSessionVectors(
		pDataParams,
		PHAL_MFIDENTITY_SAM_X_SESSION_MAC,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pTMC,
		pUid,
		bUidLen));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Perform MAC verification. */
	wRemLen = (uint16_t) dwTMILen;
	wBuffOption = PH_EXCHANGE_TXCHAINING;

	do
	{
		/* Update the finished flag and buffering option. */
		if(wRemLen <= PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH)
		{
			bFinished = PH_ON;
			wBuffOption = PH_EXCHANGE_DEFAULT;
			bExchangeLen = (uint8_t) wRemLen;
		}
		else
		{
			bExchangeLen = PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;
			wRemLen = (uint16_t) (wRemLen - PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH);
		}

		/* Exchange the TMI information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
			&pTMI[wTMIOffset],
			bExchangeLen,
			&pMac,
			&wMacLen);

		/* Validate the response. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}

		/* Update the TMI offset information. */
		wTMIOffset += PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;

	}while(!bFinished);

	/* Copy the Mac to the parameter. */
	memcpy(pTMV, pMac, wMacLen);	/* PRQA S 3200 */

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_X_DecryptReaderID(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
	uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI, uint8_t * pTMRIPrev)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Derive Transaction MAC (KSesTMMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ComputeTMACSessionVectors(
		pDataParams,
		PHAL_MFIDENTITY_SAM_X_SESSION_ENC,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pTMC,
		pUid,
		bUidLen));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Exchange the TMI information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		pEncTMRI,
		16,
		&pResponse,
		&wRespLen));

	/* Copy the decrypted information to the parameter. */
	memcpy(pTMRIPrev, pResponse, wRespLen);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_CalculateMACSDM(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
	uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen,
	uint8_t * pMac)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
	uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
	uint16_t	PH_MEMLOC_REM wRemLen = 0;
	uint16_t	PH_MEMLOC_REM wInputOffset = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ComputeSDMSessionVectors(
		pDataParams,
		PHAL_MFIDENTITY_SAM_X_SESSION_MAC,
		bSdmOption,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pUid,
		bUidLen,
		pSDMReadCtr));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Perform MAC generation. */
	wRemLen = wInDataLen;
	wBuffOption = PH_EXCHANGE_TXCHAINING;

	do
	{
		/* Update the finished flag and buffering option. */
		if(wRemLen <= PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH)
		{
			bFinished = PH_ON;
			wBuffOption = PH_EXCHANGE_DEFAULT;
			bExchangeLen = (uint8_t) wRemLen;
		}
		else
		{
			bExchangeLen = PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;
			wRemLen = (uint16_t) (wRemLen - PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH);
		}

		/* Exchange the TMI information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
			&pInData[wInputOffset],
			bExchangeLen,
			&pResponse,
			&wRespLen);

		/* Validate the response. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}

		/* Update the TMI offset information. */
		wInputOffset += PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;

	}while(!bFinished);

	/* Copy the Mac to the parameter. */
	memcpy(pMac, pResponse, wRespLen);	/* PRQA S 3200 */

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_X_DecryptSDMENCFileData(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo, 
	uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pEncdata, uint16_t wEncDataLen, uint8_t * pPlainData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
	uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
	uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
	uint16_t	PH_MEMLOC_REM wRemLen = 0;
	uint16_t	PH_MEMLOC_REM wEncBuffOffset = 0;
	uint16_t	PH_MEMLOC_REM wPlainBuffOffset = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

	/* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_ComputeSDMSessionVectors(
		pDataParams,
		PHAL_MFIDENTITY_SAM_X_SESSION_ENC,
		(uint8_t) (PHAL_MFIDENTITY_VCUID_PRESENT | PHAL_MFIDENTITY_RDCTR_PRESENT),
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pUid,
		bUidLen,
		pSDMReadCtr));

	/* Set the Option. */
	bOption = (uint8_t) (PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType ? PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_TWO :
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU);

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bOption,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Load the IV. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfIdentity_Sam_X_Int_LoadSDMInitVector(
		pDataParams,
		pSDMReadCtr));

	/* Perform Decryption. */
	wRemLen = wEncDataLen;
	wBuffOption = PH_EXCHANGE_TXCHAINING;

	do
	{
		/* Update the finished flag and buffering option. */
		if(wRemLen <= PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH)
		{
			bFinished = PH_ON;
			wBuffOption = PH_EXCHANGE_DEFAULT;
			bExchangeLen = (uint8_t) wRemLen;
		}
		else
		{
			bExchangeLen = PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;
			wRemLen = (uint16_t) (wRemLen - PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH);
		}

		/* Exchange the Encrypted information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			&pEncdata[wEncBuffOffset],
			bExchangeLen,
			&pResponse,
			&wRespLen);

		/* Validate the response. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}

		/* Update the TMI offset information. */
		wEncBuffOffset += PHAL_MFIDENTITY_SAM_DATA_FRAME_LENGTH;

		/* Copy the decrypted data to the parameter. */
		memcpy(&pPlainData[wPlainBuffOffset], pResponse, wRespLen);	/* PRQA S 3200 */
		wPlainBuffOffset += wRespLen;

	}while(!bFinished);

	return wStatus;
}

phStatus_t phalMfIdentity_Sam_X_DecryptSDMPICCData(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pEncdata, 
	uint16_t wEncDataLen, uint8_t * pPlainData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bEncDataOffset = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wKeyNo,
		(uint8_t) wKeyVer,
		NULL,
		0));

	/* Load the IV and update the Offset in case of LRP type. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR,
			pEncdata,
			8));

		/* Update the Offset. */
		bEncDataOffset = 8;
	}

	/* Perform DecipherDataOffline. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		&pEncdata[bEncDataOffset],
		(uint8_t) (wEncDataLen - bEncDataOffset),
		&pResponse,
		&wRespLen));

	/* Copy the decrypted information to the parameter. */
	memcpy(pPlainData, pResponse, wRespLen);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */
