/*
*         Copyright (c), NXP Semiconductors Bangalore / India
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
*particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X

#include "../phalMfIdentity_Int.h"
#include "phalMfIdentity_Sam_X.h"
#include "phalMfIdentity_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfIdentity_Sam_X_Int_ValidateResponse(void * pDataParams, uint8_t bResetAuth, uint16_t wStatus, uint8_t * pPiccRetCode)
{
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	phStatus_t	PH_MEMLOC_REM wStatus2 = 0;
	uint16_t	PH_MEMLOC_REM wPiccStat = 0;
	uint16_t	PH_MEMLOC_REM wPiccRetCode = 0;

	/* Evaluate the response. */
	if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN) ||
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN))
	{
		/* Evaluate the PICC Status. */
		wPiccRetCode = (uint16_t) ((pPiccRetCode[0] << 8) | pPiccRetCode[1]);
		wPiccStat = (uint16_t) (((wPiccRetCode & 0xFF00) == 0x9100) ? (wPiccRetCode & 0x00FF) : wPiccRetCode);

		/* Validate the PICC Status. */
		wStatus1 = phalMfIdentity_Int_ComputeErrorResponse(pDataParams, wPiccStat);

		/* Reset the Authentication if set. */
		if(bResetAuth)
		{
			PH_CHECK_SUCCESS_FCT(wStatus2, phalMfIdentity_Sam_X_Int_ResetAuthentication(pDataParams));
		}
	}
	else
	{
		if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wStatus1 = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFIDENTITY);
		}

		PH_CHECK_SUCCESS(wStatus);
	}

	return wStatus1;
}

phStatus_t phalMfIdentity_Sam_X_Int_Iso7816Wrap(uint8_t bFirstFrame, uint8_t bLengthPresent, uint8_t bLengthLen, uint8_t bLePresent, uint32_t dwTotDataLen, 
	uint8_t * pPlainData, uint16_t * pDataLen)
{
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
	uint8_t		PH_MEMLOC_REM bLC = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint8_t		PH_MEMLOC_REM bOffset1 = 0;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Read = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Write = PH_OFF;

	/* Extract the command code. */
	if(bFirstFrame)
	{
		bCmdCode = pPlainData[bLengthPresent ? bLengthLen :0];
	}

	/* Set the flag for data operation commands. */
	bIsIsoChainnedCmd_Read = (uint8_t) (((bCmdCode == PHAL_MFIDENTITY_CMD_READ_DATA_ISO) || (bCmdCode == PHAL_MFIDENTITY_CMD_READ_RECORDS_ISO)) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd_Write = (uint8_t) (((bCmdCode == PHAL_MFIDENTITY_CMD_WRITE_DATA_ISO) || (bCmdCode == PHAL_MFIDENTITY_CMD_WRITE_RECORD_ISO)) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd = (uint8_t) (bIsIsoChainnedCmd_Read | bIsIsoChainnedCmd_Write);

	/* Set the LC value */
	bLC = (uint8_t) (bFirstFrame ? (*pDataLen - 1 /* Removing the command code. */) : *pDataLen);

	/* Subtract LC by 3 if Length is present. */
	bLC = (uint8_t) (bLengthPresent ? ( bLC - bLengthLen)  : bLC);

	/* Compute the offset. */
	bOffset = (uint8_t) ((bFirstFrame && bLengthPresent) ? (bLengthLen + 1) : 1);

	/* Compute the offset to be used for moving the current data. */
	bOffset1 = (uint8_t) (bLengthPresent ? (bLengthLen + 5) : 5);

	/* Copy the data to the Pointer. */
	memmove(&pPlainData[bOffset1], &pPlainData[bOffset], bLC);	/* PRQA S 3200 */

	/* Reset the length buffer. */
	*pDataLen = bLengthPresent ? bLengthLen : 0;

	/* Frame the initial ISO7816 header. */
	pPlainData[(*pDataLen)++] = PHAL_MFIDENTITY_WRAPPEDAPDU_CLA;
	pPlainData[(*pDataLen)++] = bCmdCode;
	pPlainData[(*pDataLen)++] = PHAL_MFIDENTITY_WRAPPEDAPDU_P1;
	pPlainData[(*pDataLen)++] = PHAL_MFIDENTITY_WRAPPEDAPDU_P2;

	/* Add LC if there is data. */
	if(bLC)
	{
		pPlainData[(*pDataLen)++] = (uint8_t) (bIsIsoChainnedCmd_Write ? dwTotDataLen : bLC);

		/* Update Frame length based on LC. */
		*pDataLen = *pDataLen + bLC;
	}

	if(bLePresent)
	{
		pPlainData[(*pDataLen)++] = PHAL_MFIDENTITY_WRAPPEDAPDU_LE;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_Int_ResetAuthentication(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset the Authmode and Key number */
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = 0xFF;
    PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = 0xFF;
	PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType = 0;

	/* Kill the PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo, 
	uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM aIV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint8_t		PH_MEMLOC_REM bSvBak = 0;
	uint16_t	PH_MEMLOC_REM wActTMC = 0;
	uint16_t	PH_MEMLOC_REM wSesTMC = 0;
	uint32_t	PH_MEMLOC_REM dwTMC = 0;

	/* Increment the TMC by 1. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_EV2)
	{
		dwTMC = (uint32_t) (pTMC[0] | (pTMC[1] << 8) | (pTMC[2] << 16) | (pTMC[3] << 24));
		dwTMC++;
	}
	else
	{
		wActTMC = (uint16_t) (pTMC[0] | (pTMC[1] << 8));
		wSesTMC = (uint16_t) (pTMC[2] | (pTMC[3] << 8));
		
		wActTMC++;
		wSesTMC++;
		dwTMC = (uint32_t) ((wSesTMC << 16) | wActTMC);
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */
	memset(aIV, 0, 16);		/* PRQA S 3200 */

	/* Compute the session vector. */
	aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFIDENTITY_SAM_X_SESSION_ENC) ? 0xA5 : 0x5A);
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x01;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x80;

	/* Append the TMC information. */
	aSV[bSvLen++] = (uint8_t) (dwTMC & 0xFF);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF00) >> 8);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF0000) >> 16);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF000000) >> 24);

	/* Append the UID infomration. */
	memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
	bSvLen = 16;

	/* Rotate the SV information by 1 for LRP. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFIDENTITY_AUTH_TYPE_LRP)
	{
		bSvBak = aSV[0];
		memcpy(&aSV[0], &aSV[1], 15);	/* PRQA S 3200 */
		aSV[15] = bSvBak;
	}

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_Int_ComputeSDMSessionVectors(void * pDataParams, uint8_t bOption, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, 
	uint16_t wDstKeyNo, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMReadCtr = 0;

	/* Validate the Counter value. */
	dwSDMReadCtr = (uint32_t) (pSDMReadCtr[0] | (pSDMReadCtr[1] << 8) | (pSDMReadCtr[2] << 16) | (pSDMReadCtr[3] << 24));
	if (dwSDMReadCtr == 0xFFFFFF)
	{
		return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFIDENTITY);
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType)
	{
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x01;
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x80;

		aSV[14] = 0x1E;
		aSV[15] = 0xE1;
	}
	else
	{
		aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFIDENTITY_SAM_X_SESSION_ENC) ? 0xC3 : 0x3C);
		aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFIDENTITY_SAM_X_SESSION_ENC) ? 0x3C : 0xC3);
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x01;
		aSV[bSvLen++] = 0x00;
		aSV[bSvLen++] = 0x80;
	}

	/* Append the UID */
	if(bSdmOption & PHAL_MFIDENTITY_VCUID_PRESENT)
	{
		memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
		bSvLen += bUidLen;
	}

	/* Append the SDM ReadCtr information. */
	if(bSdmOption & PHAL_MFIDENTITY_RDCTR_PRESENT)
	{
		aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);
	}

	/* Update the SV length to 16. */
	bSvLen = 16;

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sam_X_Int_LoadSDMInitVector(void * pDataParams, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
	uint8_t		PH_MEMLOC_REM aIV[16];
	uint8_t *	PH_MEMLOC_REM pIV = NULL;
	uint16_t	PH_MEMLOC_REM wIvLen = 0;

	/* Clear the Initialization Vector. */
	memset(aIV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	if(PHAL_MFIDENTITY_RESOLVE_DATAPARAMS(pDataParams)->bKeyType)
	{
		/* IV computation is SDMReadCtr || 0x000000 */

		memcpy(&aIV[wIvLen], pSDMReadCtr, 3);	/* PRQA S 3200 */
		wIvLen += 3;

		/* Update IVLen to allocate three zeros. */
		wIvLen += 3;

		/* Set the Option. */
		bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR;
	}
	else
	{
		/* IV computation is E(KSesSDMFileReadENC; SDMReadCtr || 0x00000000000000000000000000) */
		memcpy(&aIV[wIvLen], pSDMReadCtr, 3);	/* PRQA S 3200 */
		
		/* Set the IV length to 16. */
		wIvLen = 16;

		/* Encrypt the IV. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_DEFAULT,
			aIV,
			(uint8_t) wIvLen,
			&pIV,
			&wIvLen));

		/* Copy the enciphered data to local buffer. */
		memcpy(aIV, pIV, wIvLen);		/* PRQA S 3200 */		
		pIV = NULL;

		/* Set the Option. */
		bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV;
	}

	/* Load the IV. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
			PHAL_MFIDENTITY_RESOLVE_HAL_DATAPARAMS(pDataParams),
			bOption,
			aIV,
			(uint8_t) wIvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */
