/*
*         Copyright (c), NXP Semiconductors Bangalore / India
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
* particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

/** \file
* Generic MIFARE PRIME Lite Application Component of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7230 $
* $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
*
*/

#include <phalMfIdentity.h>
#include <ph_Status.h>
#include <ph_TypeDefs.h>

#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
#include "Sw/phalMfIdentity_Sw.h"
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
#include "Sam_NonX\phalMfIdentity_Sam_NonX.h"
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
#include "Sam_X\phalMfIdentity_Sam_X.h"
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

#ifdef NXPBUILD__PHAL_MFIDENTITY

#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity Secure Messaging commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_AuthenticateEv2(void *pDataParams, uint8_t bFirstAuth, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivLen, uint8_t bLenPcdCapsIn, uint8_t *bPcdCapsIn, uint8_t *bPcdCapsOut, uint8_t *bPdCapsOut)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_AuthenticatEv2");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFirstAuth);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNoCard);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPcdCapsIn);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPcdCapsOut);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPdCapsOut);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFirstAuth_log, &bFirstAuth);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNo_log, &wKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVer_log, &wKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNoCard_log, &bKeyNoCard);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPcdCapsIn_log, bPcdCapsIn, bLenPcdCapsIn);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bDivLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);
	if(bLenPcdCapsIn)  PH_ASSERT_NULL_PARAM(bPcdCapsIn, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_AuthenticateEv2((phalMfIdentity_Sw_DataParams_t *) pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard,
				pDivInput, bDivLen, bLenPcdCapsIn, bPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_AuthenticateEv2(pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput, bDivLen, bPcdCapsIn,
				bLenPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_AuthenticateEv2(pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput, bDivLen, bPcdCapsIn,
				bLenPcdCapsIn, bPcdCapsOut, bPdCapsOut);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(bFirstAuth)
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPcdCapsOut_log, bPcdCapsOut, 6);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bPdCapsOut_log, bPdCapsOut, 6);
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity Memory and Configuration commands. ---------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_SetConfiguration");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDataLen_log, &bDataLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pData_log, pData, bDataLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pData, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_SetConfiguration((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_SetConfiguration(pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_SetConfiguration(pDataParams, bOption, pData, bDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfIdentity_GetVersion(void * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t PH_MEMLOC_REM status;
	uint8_t PH_MEMLOC_REM bVersionLength = 0;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetVersion");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVerInfo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVersionLength);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pVerInfo, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetVersion((phalMfIdentity_Sw_DataParams_t *) pDataParams, pVerInfo, &bVersionLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetVersion(pDataParams, pVerInfo, &bVersionLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetVersion(pDataParams, pVerInfo, &bVersionLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pVerInfo_log, pVerInfo, bVersionLength);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, pVerInfo_log, &bVersionLength);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, pVersionLength_log, &bVersionLength);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_GetCardUID(void * pDataParams, uint8_t * pUid)
{
	phStatus_t PH_MEMLOC_REM status;
	uint8_t PH_MEMLOC_REM bCardUidLength = 0;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetCardUID");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCardUidLength);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetCardUID((phalMfIdentity_Sw_DataParams_t *) pDataParams, pUid, &bCardUidLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetCardUID(pDataParams, pUid, &bCardUidLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetCardUID(pDataParams, pUid, &bCardUidLength);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bCardUidLength);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCardUidLength_log, &bCardUidLength);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity Key Management commands. -------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer,
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
	/**
	* This  function will handle all the three authentication modes: 0x0A, 1A and AA.
	* and all crypto modes i.e., DES, 3DES, 3K3DES, AES
	* The previous authentication status including key number and session key is
	* present in the params  structure.
	* Successful auth. with PICC master key is required if AID = 0x00 else
	* an auth. with the application master key is required.
	*/
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ChangeKey");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOldKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNewKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOldKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNewKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNoCard);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOldKeyVer_log, &wOldKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wNewKeyVer_log, &wNewKeyVer);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOldKeyNo_log, &wOldKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wNewKeyNo_log, &wNewKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNoCard_log, &bKeyNoCard);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivLen_log, &bDivLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bDivLen > 0)PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ChangeKey((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer,
				bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ChangeKey(pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ChangeKey(pDataParams, wOption, wOldKeyNo, wOldKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput, bDivLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
	return status;
}

phStatus_t phalMfIdentity_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo, uint8_t * pKeyVersion, uint8_t * bRxLen)
{
	/**
	* This command can be issued without valid authentication
	*/
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetKeyVersion");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeySetNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pKeyVersion);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bRxLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyNo_log, &bKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeySetNo_log, &bKeySetNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pKeyVersion, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetKeyVersion((phalMfIdentity_Sw_DataParams_t *) pDataParams, bKeyNo, bKeySetNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetKeyVersion(pDataParams, bKeyNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetKeyVersion(pDataParams, bKeyNo, pKeyVersion, bRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pKeyVersion_log, pKeyVersion, *bRxLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bRxLen_log, bRxLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity File Management commands. ------------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_CreateTransactionMacFile(void * pDataParams, uint8_t bFileNo, uint8_t bCommMode, uint8_t * pAccessRights, uint8_t bKeyType,
	uint8_t * bTMKey, uint16_t wKeyNo, uint8_t bTMKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_CreateTransactionMacFile");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCommMode);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bKeyType);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bTMKey);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bTMKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCommMode_log, &bCommMode);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAccessRights_log, pAccessRights, 2);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bKeyType_log, &bKeyType);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNo_log, &wKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bTMKeyVer_log, &bTMKeyVer);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bTMKey_log, bTMKey, 16);
	if(pDivInput != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(bTMKey, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_CreateTransactionMacFile((phalMfIdentity_Sw_DataParams_t *) pDataParams, bFileNo, bCommMode, pAccessRights, bKeyType,
				bTMKey, bTMKeyVer);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_CreateTransactionMacFile(pDataParams, bFileNo, bCommMode, pAccessRights, wKeyNo, bTMKeyVer, bKeyType, pDivInput, bDivInputLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_CreateTransactionMacFile(pDataParams, bFileNo, bCommMode, pAccessRights, wKeyNo, bTMKeyVer, bKeyType, pDivInput, bDivInputLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfIdentity_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_DeleteFile");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_DeleteFile((phalMfIdentity_Sw_DataParams_t *) pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_DeleteFile(pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_DeleteFile(pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * bBufferLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetFileSettings");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFSBuffer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bBufferLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pFSBuffer, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(bBufferLen, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetFileSettings((phalMfIdentity_Sw_DataParams_t *) pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetFileSettings(pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetFileSettings(pDataParams, bFileNo, pFSBuffer, bBufferLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFSBuffer_log, pFSBuffer, (*bBufferLen));
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bBufferLen_log, bBufferLen, 1);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_GetFileCounters(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pFileCounters, uint8_t * pRxLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetFileCounters");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFileCounters);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pFileCounters, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRxLen, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetFileCounters((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetFileCounters(pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetFileCounters(pDataParams, bOption, bFileNo, pFileCounters, pRxLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
	if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFileCounters_log, pFileCounters, (*pRxLen));
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRxLen_log, pRxLen, 1);
	}
#endif
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_ChangeFileSettings(void * pDataParams, uint8_t bCommMode, uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights,
	uint8_t bTmcLimitLen, uint8_t * pTmcLimit)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ChangeFileSettings");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCommMode);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bTmcLimitLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTmcLimit);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCommMode_log, &bCommMode);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileOption_log, &bFileOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bTmcLimitLen_log, &bTmcLimitLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAccessRights_log, pAccessRights, 2);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTmcLimit_log, pTmcLimit, bTmcLimitLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pAccessRights, PH_COMP_AL_MFIDENTITY);
	if(bTmcLimitLen) PH_ASSERT_NULL_PARAM(pTmcLimit, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ChangeFileSettings((phalMfIdentity_Sw_DataParams_t *) pDataParams, bCommMode, bFileNo, bFileOption, pAccessRights, bTmcLimitLen,
				pTmcLimit);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ChangeFileSettings(pDataParams, bCommMode, bFileNo, bFileOption, pAccessRights, bTmcLimitLen,
				pTmcLimit);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ChangeFileSettings(pDataParams, bCommMode, bFileNo, bFileOption, pAccessRights, bTmcLimitLen,
				pTmcLimit);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_ChangeFileSettingsSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights, 
	uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pPICCDataOffset, uint8_t *pSDMMACInputOffset, uint8_t *pSDMENCOffset, uint8_t *pSDMENCLen, 
	uint8_t *pSDMMACOffset, uint8_t *pSDMReadCtrOffset, uint8_t *pTMCLimitValue)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ChangeFileSettingsSDM");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSdmOptions);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSdmAccessRights);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPICCDataOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMMACInputOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMENCOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMENCLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMMACOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMReadCtrOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMCLimitValue);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileOption_log, &bFileOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pAccessRights_log, pAccessRights, 2);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSdmOptions_log, &bSdmOptions);
	if(pSdmAccessRights != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSdmAccessRights_log, pSdmAccessRights, 2);
	if(pPICCDataOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPICCDataOffset_log, pPICCDataOffset, 3);
	if(pSDMReadCtrOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMReadCtrOffset_log, pSDMReadCtrOffset, 3);
	if(pSDMMACInputOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMMACInputOffset_log, pSDMMACInputOffset, 3);
	if(pSDMENCOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMENCOffset_log, pSDMENCOffset, 3);
	if(pSDMENCLen != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMENCLen_log, pSDMENCLen, 3);
	if(pSDMMACOffset != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMMACOffset_log, pSDMMACOffset, 3);
	if(pTMCLimitValue != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMCLimitValue_log, pTMCLimitValue, 4);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ChangeFileSettingsSDM((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, bFileNo, bFileOption, pAccessRights, bSdmOptions,
				pSdmAccessRights, pPICCDataOffset, pSDMMACInputOffset, pSDMENCOffset, pSDMENCLen, pSDMMACOffset, pSDMReadCtrOffset, pTMCLimitValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ChangeFileSettingsSDM(pDataParams, bOption, bFileNo, bFileOption, pAccessRights, bSdmOptions, pSdmAccessRights,
				pPICCDataOffset, pSDMMACInputOffset, pSDMENCOffset, pSDMENCLen, pSDMMACOffset, pSDMReadCtrOffset, pTMCLimitValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ChangeFileSettingsSDM(pDataParams, bOption, bFileNo, bFileOption, pAccessRights, bSdmOptions, pSdmAccessRights,
				pPICCDataOffset, pSDMMACInputOffset, pSDMENCOffset, pSDMENCLen, pSDMMACOffset, pSDMReadCtrOffset, pTMCLimitValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity Data Management commands. ------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_ReadData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, 
	uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ReadData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pLength);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxdata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxdataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOffset_log, pOffset, 3);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pLength_log, pLength, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pLength, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(ppRxdata, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRxdataLen, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ReadData((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, bIns, bFileNo, pOffset, pLength, ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ReadData(pDataParams, bOption, bFileNo, pOffset, pLength, ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ReadData(pDataParams, bOption, bFileNo, pOffset, pLength, ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if((status == PH_ERR_SUCCESS) && ((*pRxdataLen) != 0))
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, ppRxdata_log, *ppRxdata, (*pRxdataLen));
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pRxdataLen_log, pRxdataLen);
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_WriteData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pTxData, 
	uint8_t * pTxDataLen)
{
	phStatus_t  PH_MEMLOC_REM status;
	uint16_t    PH_MEMLOC_REM wDataLen;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_WriteData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxDataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOffset_log, pOffset, 3);

	if(pTxDataLen != NULL)
	{
		wDataLen = (uint16_t) pTxDataLen[1];
		wDataLen = wDataLen << 8;
		wDataLen |= pTxDataLen[0];
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTxData_log, pTxData, wDataLen);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTxDataLen_log, pTxDataLen, 3);
	}
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTxData, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTxDataLen, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_WriteData((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, bIns, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_WriteData(pDataParams, bOption, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_WriteData(pDataParams, bOption, bFileNo, pOffset, pTxData, pTxDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_ReadRecords(void * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pRecNo, uint8_t * pRecCount,
	uint8_t * pRecSize, uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ReadRecords");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCommOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRecNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRecCount);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRecSize);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxdata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxdataLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCommOption_log, &bCommOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRecNo_log, pRecNo, 3);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRecCount_log, pRecCount, 3);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRecSize_log, pRecSize, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRecNo, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRecCount, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(ppRxdata, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRxdataLen, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pRecSize, PH_COMP_AL_MFIDENTITY);

	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ReadRecords((phalMfIdentity_Sw_DataParams_t *) pDataParams, bCommOption, bIns, bFileNo, pRecNo, pRecCount, pRecSize,
				ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ReadRecords(pDataParams, bCommOption, bFileNo, pRecNo, pRecCount, pRecSize,
				ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ReadRecords(pDataParams, bCommOption, bFileNo, pRecNo, pRecCount, pRecSize,
				ppRxdata, pRxdataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if((status == PH_ERR_SUCCESS) && ((*pRxdataLen) != 0))
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, ppRxdata_log, *ppRxdata, (*pRxdataLen));
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pRxdataLen_log, pRxdataLen);
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_WriteRecord(void * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData,
	uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM status;
	uint16_t    PH_MEMLOC_REM wDataLen;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_WriteRecord");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCommOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bIns);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOffset);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCommOption_log, &bCommOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bIns_log, &bIns);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOffset_log, pOffset, 3);

	if(pDataLen != NULL)
	{
		wDataLen = (uint16_t) pDataLen[1];
		wDataLen = wDataLen << 8;
		wDataLen |= pDataLen[0];

		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pData_log, pData, wDataLen);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDataLen_log, pDataLen, 3);
	}
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pOffset, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pData, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pDataLen, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_WriteRecord((phalMfIdentity_Sw_DataParams_t *) pDataParams, bCommOption, bIns, bFileNo, pOffset, pData, pDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_WriteRecord(pDataParams, bCommOption, bFileNo, pOffset, pData, pDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_WriteRecord(pDataParams, bCommOption, bFileNo, pOffset, pData, pDataLen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ClearRecordFile");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bFileNo);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bFileNo_log, &bFileNo);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ClearRecordFile((phalMfIdentity_Sw_DataParams_t *) pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ClearRecordFile(pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ClearRecordFile(pDataParams, bFileNo);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE Identity Transaction Management commands. ------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_CommitTransaction(void * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMV)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_CommitTransaction");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMC);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMV);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bOption_log, &bOption);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bOption & 0x01)
	{
		PH_ASSERT_NULL_PARAM(pTMC, PH_COMP_AL_MFIDENTITY);
		PH_ASSERT_NULL_PARAM(pTMV, PH_COMP_AL_MFIDENTITY);
	}

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_CommitTransaction((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, pTMC, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_CommitTransaction(pDataParams, bOption, pTMC, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_CommitTransaction(pDataParams, bOption, pTMC, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(bOption & 0x01)
	{
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMC_log, pTMC, 4);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMV_log, pTMV, 8);
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_AbortTransaction(void * pDataParams)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_AbortTransaction");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_AbortTransaction((phalMfIdentity_Sw_DataParams_t *) pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_AbortTransaction(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_AbortTransaction(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_CommitReaderID(void * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_CommitReaderID");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMRI);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pEncTMRI);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMRI_log, pTMRI, 16);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTMRI, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_CommitReaderID((phalMfIdentity_Sw_DataParams_t *) pDataParams, pTMRI, pEncTMRI);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_CommitReaderID(pDataParams, pTMRI, pEncTMRI);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_CommitReaderID(pDataParams, pTMRI, pEncTMRI);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	if(*pEncTMRI != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pEncTMRI_log, pEncTMRI, 16);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE Identity ISO7816-4 commands. ------------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t bExtendedLenApdu, uint8_t **ppFCI, uint16_t *pwFCILen)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_IsoSelectFile");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSelector);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pFid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDFname);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDFnameLen);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSelector_log, &bSelector);
	if(pFid != NULL)
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pFid_log, pFid, 2);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDFnameLen_log, &bDFnameLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDFname_log, pDFname, (uint16_t) bDFnameLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_IsoSelectFile((phalMfIdentity_Sw_DataParams_t *) pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen, bExtendedLenApdu,
				ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_IsoSelectFile(pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen, bExtendedLenApdu, ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_IsoSelectFile(pDataParams, bOption, bSelector, pFid, pDFname, bDFnameLen, bExtendedLenApdu, ppFCI, pwFCILen);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE Identity OriginalityCheck commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
	phStatus_t PH_MEMLOC_REM status = 0;
	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ReadSign");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAddr);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSignature);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bAddr_log, &bAddr);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pSignature, PH_COMP_AL_MFIDENTITY);

	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

		return  PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ReadSign((phalMfIdentity_Sw_DataParams_t *) pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ReadSign(pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ReadSign(pDataParams, bAddr, pSignature);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSignature_log, *pSignature, 56);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}




/* MIFARE DESFire Light Miscellaneous commands. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_GetConfig(void * pDataParams, uint16_t wConfig, uint8_t * pValue)
{
	phStatus_t	PH_MEMLOC_REM status;
	uint16_t	PH_MEMLOC_REM wValLen = 0;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetConfig");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pValue, PH_COMP_AL_MFIDENTITY);

	/* Check data parameters */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_GetConfig((phalMfIdentity_Sw_DataParams_t *) pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_GetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_GetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
	if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
	{
		/* Compute the value length. */
		if(wConfig >= PHAL_MFIDENTITY_SDM_ENC_OFFSET)
		{
			wValLen = 3;
		}
		else
		{
			wValLen = 2;
		}

		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue, wValLen);
	}
#endif
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_SetConfig(void * pDataParams, uint16_t wConfig, uint8_t * pValue)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_GetConfig");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pValue_log, &pValue, 3);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Check data parameters */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_SetConfig((phalMfIdentity_Sw_DataParams_t *) pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_SetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_SetConfig(pDataParams, wConfig, pValue);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_ResetAuthentication(void * pDataParams)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_ResetAuthentication");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);

	/* Check data parameters */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_ResetAuthentication((phalMfIdentity_Sw_DataParams_t *) pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_ResetAuthentication(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_ResetAuthentication(pDataParams);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_CalculateTMV(void * pDataParams, uint16_t wOption, uint16_t wKeyNoTMACKey, uint16_t wKeyVerTMACKey,
	uint16_t wRamKeyNo, uint16_t wRamKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, 
	uint8_t bUidLen, uint8_t * pTMI, uint32_t dwTMILen, uint8_t * pTMV)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_CalculateTMV");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNoTMACKey);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVerTMACKey);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMC);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMI);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTMILen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMV);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNoTMACKey_log, &wKeyNoTMACKey);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVerTMACKey_log, &wKeyVerTMACKey);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyNo_log, &wRamKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyVer_log, &wRamKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUidLen_log, &bUidLen);
	PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, wTMILen_log, &dwTMILen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMC_log, pTMC, 4);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bUidLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMI_log, pTMI, (uint16_t) dwTMILen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bDivInputLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTMC, PH_COMP_AL_MFIDENTITY);
	if(bUidLen) PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);
	if(dwTMILen) PH_ASSERT_NULL_PARAM(pTMI, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTMI, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_CalculateTMV((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, pDivInput, bDivInputLen,
				pTMC, pUid, bUidLen, pTMI, dwTMILen, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_CalculateTMV(pDataParams, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, pTMC, pUid, bUidLen,
				pTMI, dwTMILen, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_CalculateTMV(pDataParams, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, pTMC, pUid, bUidLen,
				pTMI, dwTMILen, pTMV);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMV_log, pTMV, 8);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_DecryptReaderID(void * pDataParams, uint16_t wOption, uint16_t wKeyNoTMACKey, uint16_t wKeyVerTMACKey,
	uint16_t wRamKeyNo, uint16_t wRamKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, 
	uint8_t bUidLen, uint8_t * pEncTMRI, uint8_t * pTMRIPrev)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_DecryptReaderID");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNoTMACKey);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVerTMACKey);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMC);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pEncTMRI);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTMRIPrev);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNoTMACKey_log, &wKeyNoTMACKey);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVerTMACKey_log, &wKeyVerTMACKey);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyNo_log, &wRamKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyVer_log, &wRamKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUidLen_log, &bUidLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMC_log, pTMC, 4);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bUidLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pEncTMRI_log, pEncTMRI, 16);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Verify the parameters */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bDivInputLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pTMC, PH_COMP_AL_MFIDENTITY);
	if(bUidLen) PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pEncTMRI, PH_COMP_AL_MFIDENTITY);

	/* Parameter Validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	/* Perform operation on active layer */
	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_DecryptReaderID((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, pDivInput,
				bDivInputLen, pTMC, pUid, bUidLen, pEncTMRI, pTMRIPrev);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_DecryptReaderID(pDataParams, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, pTMC, pUid,
				bUidLen, pEncTMRI, pTMRIPrev);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_DecryptReaderID(pDataParams, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, pTMC, pUid,
				bUidLen, pEncTMRI, pTMRIPrev);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pTMRIPrev_log, pTMRIPrev, 16);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_CalculateMACSDM(void * pDataParams, uint16_t wOption, uint8_t bSdmOption, uint16_t wSDMMacKeyNo, uint16_t wSDMMacKeyVer,
	uint16_t wRamKeyNo, uint16_t wRamKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, 
	uint8_t * pInData, uint16_t wInDataLen, uint8_t * pRespMac)
{

	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_CalculateMACSDM");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSdmOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSDMMacKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSDMMacKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMReadCtr);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pInData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wInDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRespMac);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSdmOption_log, &bSdmOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSDMMacKeyNo_log, &wSDMMacKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSDMMacKeyVer_log, &wSDMMacKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyNo_log, &wRamKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyVer_log, &wRamKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUidLen_log, &bUidLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);

	if(pSDMReadCtr != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMReadCtr_log, pSDMReadCtr, 3);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wInDataLen_log, &wInDataLen);
	if(pInData != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pInData_log, pInData, wInDataLen);
	if(pUid != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bUidLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	if(bDivInputLen) PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pInData, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pSDMReadCtr, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);

	/* parameter validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_CalculateMACSDM((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, bSdmOption, wSDMMacKeyNo, wSDMMacKeyVer, pDivInput,
				bDivInputLen, pUid, bUidLen, pSDMReadCtr, pInData, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_CalculateMACSDM(pDataParams, bSdmOption, wSDMMacKeyNo, wSDMMacKeyVer, wRamKeyNo, wRamKeyVer, pUid, bUidLen, pSDMReadCtr,
				pInData, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_CalculateMACSDM(pDataParams, bSdmOption, wSDMMacKeyNo, wSDMMacKeyVer, wRamKeyNo, wRamKeyVer, pUid, bUidLen, pSDMReadCtr,
				pInData, wInDataLen, pRespMac);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRespMac_log, pRespMac, 8);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

phStatus_t phalMfIdentity_DecryptSDMENCFileData(void * pDataParams, uint16_t wOption, uint16_t wEncKeyNo, uint16_t wEncKeyVer, 
	uint16_t wRamKeyNo, uint16_t wRamKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pUid, uint8_t bUidLen, 
	uint8_t * pSDMReadCtr, uint8_t * pEncdata, uint16_t wEncDataLen, uint8_t * pPlainData)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_DecryptSDMENCFileData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wEncKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wEncKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRamKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUid);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUidLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSDMReadCtr);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pEncdata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wEncDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPlainData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wEncKeyNo_log, &wEncKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wEncKeyVer_log, &wEncKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyNo_log, &wRamKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRamKeyVer_log, &wRamKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUidLen_log, &bUidLen);
	if(pUid != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUid_log, pUid, bUidLen);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	if(pDivInput != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);
	if(pSDMReadCtr != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSDMReadCtr_log, pSDMReadCtr, 3);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wEncDataLen_log, &wEncDataLen);
	if(pEncdata != NULL) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pEncdata_log, pEncdata, wEncDataLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Perform Mandatory parameters Null Check */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pSDMReadCtr, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pEncdata, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pUid, PH_COMP_AL_MFIDENTITY);

	/* Perform Optional parameters NUll Check */
	if(bDivInputLen)
		PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);

	/* parameter validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_DecryptSDMENCFileData((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, wEncKeyNo, wEncKeyVer, 
				pDivInput, bDivInputLen, pUid, bUidLen, pSDMReadCtr, pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_DecryptSDMENCFileData(pDataParams, wEncKeyNo, wEncKeyVer, wRamKeyNo, wRamKeyVer, pUid, bUidLen, 
				pSDMReadCtr, pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_DecryptSDMENCFileData(pDataParams, wEncKeyNo, wEncKeyVer, wRamKeyNo, wRamKeyVer, pUid, bUidLen, 
				pSDMReadCtr, pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPlainData_log, pPlainData, wEncDataLen);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;

}

phStatus_t phalMfIdentity_DecryptSDMPICCData(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pDivInput, 
	uint8_t bDivInputLen, uint8_t * pEncdata, uint16_t wEncDataLen, uint8_t * pPlainData)
{
	phStatus_t PH_MEMLOC_REM status;

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalMfIdentity_DecryptSDMPICCData");
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNo);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVer);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pEncdata);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(wEncDataLen);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPlainData);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wOption_log, &wOption);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNo_log, &wKeyNo);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVer_log, &wKeyVer);
	PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, bDivInputLen);

	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wEncDataLen_log, &wEncDataLen);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pEncdata_log, pEncdata, wEncDataLen);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Perform Mandatory parameters Null Check */
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pEncdata, PH_COMP_AL_MFIDENTITY);

	/* Perform Optional parameters NUll Check */
	if(bDivInputLen)
		PH_ASSERT_NULL_PARAM(pDivInput, PH_COMP_AL_MFIDENTITY);

	/* parameter validation */
	if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_MFIDENTITY)
	{
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	switch(PH_GET_COMPID(pDataParams))
	{
#ifdef NXPBUILD__PHAL_MFIDENTITY_SW
		case PHAL_MFIDENTITY_SW_ID:
			status = phalMfIdentity_Sw_DecryptSDMPICCData((phalMfIdentity_Sw_DataParams_t *) pDataParams, wOption, wKeyNo, wKeyVer, pDivInput, bDivInputLen,
				pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_NONX
		case PHAL_MFIDENTITY_SAMAV3_NONX_ID:
			status = phalMfIdentity_Sam_NonX_DecryptSDMPICCData(pDataParams, wKeyNo, wKeyVer, pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFIDENTITY_SAM_X
		case PHAL_MFIDENTITY_SAMAV3_X_ID:
			status = phalMfIdentity_Sam_X_DecryptSDMPICCData(pDataParams, wKeyNo, wKeyVer, pEncdata, wEncDataLen, pPlainData);
			break;
#endif /* NXPBUILD__PHAL_MFIDENTITY_SAM_X */

		default:
			status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
			break;
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPlainData_log, pPlainData, 16);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
	PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_MFIDENTITY */
