/*
* Copyright 2009 - 2018, 2025 NXP
* NXP Confidential and Proprietary.
* This software is owned or controlled by NXP and may only be used strictly
* in accordance with the applicable license terms. By expressly accepting
* such terms or by downloading, installing, activating and/or otherwise using
* the software, you are agreeing that you have read, and that you agree to
* comply with and are bound by, such license terms. If you do not agree to be
* bound by the applicable license terms, then you may not retain, install,
* activate or otherwise use the software.
*/

/** \file
 * Sam (X-Mode) MIFARE(R) Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 31. July 2009
 *
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalMfc.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFC_SAM_X

#include "phalMfc_Sam_X.h"
#include "phalMfc_Sam_X_Int.h"
#include "../phalMfc_Int.h"
#include <phhalHw_SamAV2_Cmd.h>

#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

#ifdef NXPBUILD__PHAL_MFC_SAMAV2_X
phStatus_t phalMfc_SamAV2_X_Init(phalMfc_SamAV2_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
	phhalHw_SamAV2_DataParams_t * pHalDataParams, phpalMifare_SamAV2_X_DataParams_t * pPalMifareDataParams)
{
    if (sizeof(phalMfc_SamAV2_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFC);
    }

	PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFC);
	PH_ASSERT_NULL_DATA_PARAM (pHalDataParams, PH_COMP_AL_MFC);
	PH_ASSERT_NULL_DATA_PARAM (pPalMifareDataParams, PH_COMP_AL_MFC);

    /* init private data */
    pDataParams->wId                    = PH_COMP_AL_MFC | PHAL_MFC_SAMAV2_X_ID;
    pDataParams->pHalDataParams         = pHalDataParams;
    pDataParams->pPalMifareDataParams   = pPalMifareDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}
#endif /* NXPBUILD__PHAL_MFC_SAMAV2_X */

#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
phStatus_t phalMfc_SamAV3_X_Init(phalMfc_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
	phhalHw_SamAV3_DataParams_t * pHalDataParams, phpalMifare_SamAV3_X_DataParams_t * pPalMifareDataParams)
{
    if (sizeof(phalMfc_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFC);
    }

	PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFC);
	PH_ASSERT_NULL_DATA_PARAM (pHalDataParams, PH_COMP_AL_MFC);
	PH_ASSERT_NULL_DATA_PARAM (pPalMifareDataParams, PH_COMP_AL_MFC);

    /* init private data */
    pDataParams->wId                    = PH_COMP_AL_MFC | PHAL_MFC_SAMAV3_X_ID;
    pDataParams->pHalDataParams         = pHalDataParams;
    pDataParams->pPalMifareDataParams   = pPalMifareDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

phStatus_t phalMfc_Sam_X_Authenticate(void * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint8_t bKeyNo, uint8_t bKeyVer,
	uint8_t * pUid, uint8_t bUidLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

	/* Exchange the information to respective SamX PAL layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_MfcAuthenticateKeyNo(
		PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
		bBlockNo,
		bKeyType,
		bKeyNo,
        bKeyVer,
		&pUid[bUidLen - 4]));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sam_X_Read(void * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_READ(
		pDataParams,
		&bBlockNo,
		1,
		&pResponse,
		&wRespLen);

	/* Validate the response status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfc_Sam_X_ConvertNak(pDataParams, wStatus));

	/* Copy the response data to the parameter. */
	memcpy(pBlockData, pResponse, wRespLen);   /* PRQA S 3200 */

	/* Convert HAL status codes to palMifare status codes. */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sam_X_ReadValue(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t * pAddrData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_READ(
		pDataParams,
		&bBlockNo,
		1,
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfc_Sam_X_ConvertNak(pDataParams, wStatus));

    /* Check format of Value block */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfc_Int_CheckValueBlockFormat(pResponse));

    /* Update the Parameters with the received information. */
    *pAddrData = pResponse[12];
    memcpy(pValue, pResponse, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sam_X_Write(void * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aBlocks[1 /* Block No */ + PHAL_MFC_DATA_BLOCK_LENGTH];
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */
	/* Frame the Blocks information. */
	aBlocks[0] = bBlockNo;
	memcpy(&aBlocks[1], pBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);	/* PRQA S 3200 */

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_WRITE(
		pDataParams,
		aBlocks,
		1 /* Block No */ + PHAL_MFC_DATA_BLOCK_LENGTH,
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	return phalMfc_Sam_X_ConvertNak(pDataParams, wStatus);
}

phStatus_t phalMfc_Sam_X_WriteValue(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t bAddrData)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aBlocks[1 /* Block No */ + PHAL_MFC_DATA_BLOCK_LENGTH];
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

	/* Frame the Blocks information. */
	aBlocks[0] = bBlockNo;

    /* Build a valid value block. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfc_Int_CreateValueBlock(pValue, bAddrData, &aBlocks[1]));

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_WRITE(
		pDataParams,
		aBlocks,
		1 /* Block No */ + PHAL_MFC_DATA_BLOCK_LENGTH,
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	return phalMfc_Sam_X_ConvertNak(pDataParams, wStatus);
}

phStatus_t phalMfc_Sam_X_Increment(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue)
{
	return phalMfc_Int_Value(PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams), PHAL_MFC_CMD_INCREMENT, bBlockNo, pValue);
}

phStatus_t phalMfc_Sam_X_Decrement(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue)
{
	return phalMfc_Int_Value(PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams), PHAL_MFC_CMD_DECREMENT, bBlockNo, pValue);
}

phStatus_t phalMfc_Sam_X_Transfer(void * pDataParams, uint8_t bBlockNo)
{
	return phalMfc_Int_Transfer(PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams), bBlockNo);
}

phStatus_t phalMfc_Sam_X_Restore(void * pDataParams, uint8_t bBlockNo)
{
	uint8_t     PH_MEMLOC_REM aValue[PHAL_MFC_VALUE_BLOCK_LENGTH];

	/* Restore needs four dummy bytes. */
	memset(aValue, 0x00, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

	return phalMfc_Int_Value(PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams), PHAL_MFC_CMD_RESTORE, bBlockNo, aValue);
}

phStatus_t phalMfc_Sam_X_IncrementTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aBlocks[2 /* Src BlockNo + Dst Block No */ + PHAL_MFC_VALUE_BLOCK_LENGTH];
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

	/* Frame the Blocks information. */
	aBlocks[0] = bSrcBlockNo;
	aBlocks[1] = bDstBlockNo;

	/* Add value information to blocks buffer. */
	memcpy(&aBlocks[2], pValue, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_INCREMENT(
		pDataParams,
		aBlocks,
		(2 /* Src BlockNo + Dst Block No */ + PHAL_MFC_VALUE_BLOCK_LENGTH),
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	return phalMfc_Sam_X_ConvertNak(pDataParams, wStatus);
}

phStatus_t phalMfc_Sam_X_DecrementTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aBlocks[2 /* Src BlockNo + Dst Block No */ + PHAL_MFC_VALUE_BLOCK_LENGTH];
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

	/* Frame the Blocks information. */
	aBlocks[0] = bSrcBlockNo;
	aBlocks[1] = bDstBlockNo;

	/* Add value information to blocks buffer. */
	memcpy(&aBlocks[2], pValue, PHAL_MFC_VALUE_BLOCK_LENGTH);  /* PRQA S 3200 */

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_DECREMENT(
		pDataParams,
		aBlocks,
		(2 /* Src BlockNo + Dst Block No */ + PHAL_MFC_VALUE_BLOCK_LENGTH),
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	return phalMfc_Sam_X_ConvertNak(pDataParams, wStatus);
}

phStatus_t phalMfc_Sam_X_RestoreTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aBlocks[2 /* Src BlockNo + Dst Block No */];
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */

	/* Frame the Blocks information. */
	aBlocks[0] = bSrcBlockNo;
	aBlocks[1] = bDstBlockNo;

	/* Exchange the information to respective Sam HAL layer. */
	wStatus = PHHAL_HW_SAM_CMD_MF_RESTORE(
		pDataParams,
		aBlocks,
		2,
		&pResponse,
		&wRespLen);

	/* Convert HAL status codes to palMifare status codes. */
	return phalMfc_Sam_X_ConvertNak(pDataParams, wStatus);
}

phStatus_t phalMfc_Sam_X_PersonalizeUid(void * pDataParams, uint8_t bUidType)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[2];
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
	aCmdBuff[0] = PHAL_MFC_CMD_PERSOUID;
	aCmdBuff[1] = bUidType;

	/* Exchange the information to respective PAL layer. */
	PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
		PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		aCmdBuff,
		2,
		&pResponse,
		&wRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sam_X_GetVersion(void * pDataParams, uint8_t * pVersion)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bCommand;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Update the command variable. */
	bCommand = PHAL_MFC_CMD_GETVERSION;

	/* Exchange the information to respective PAL layer. */
	PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
		PHAL_MFC_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		&bCommand,
		1,
		&pVersion,
		&wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFC);
}

phStatus_t phalMfc_Sam_X_ConvertNak(void * pDataParams, phStatus_t wStatus)
{
	phStatus_t  PH_MEMLOC_REM wStatusTmp = 0;
	uint8_t		PH_MEMLOC_REM bCompID = 0;

	/* Get the component ID. */
	bCompID = PH_GET_COMPID(pDataParams);

	/* Respove the error code with respect to SAM HAL. */
	switch(bCompID)
	{
		case PHAL_MFC_SAMAV2_X_ID:
			/* Convert the component ID's */
			switch(wStatus & PH_ERR_MASK)
			{
				case PHHAL_HW_SAMAV2_ERR_MIFARE_NAK0:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK0, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV2_ERR_MIFARE_NAK1:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK1, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV2_ERR_MIFARE_NAK4:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK4, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV2_ERR_MIFARE_NAK5:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK5, PH_COMP_PAL_MIFARE);
					break;

				case PH_ERR_SUCCESS_INCOMPLETE_BYTE:
					wStatusTmp = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
					break;

				default:
					wStatusTmp = wStatus;
					break;
			}
			break;
#ifdef NXPBUILD__PHAL_MFC_SAMAV3_X
		case PHAL_MFC_SAMAV3_X_ID:
			/* Convert the component ID's */
			switch(wStatus & PH_ERR_MASK)
			{
				case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK0:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK0, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK1:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK1, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK4:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK4, PH_COMP_PAL_MIFARE);
					break;

				case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK5:
					wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK5, PH_COMP_PAL_MIFARE);
					break;

				case PH_ERR_SUCCESS_INCOMPLETE_BYTE:
					wStatusTmp = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
					break;

				default:
					wStatusTmp = wStatus;
					break;
			}
			break;
#endif /* NXPBUILD__PHAL_MFC_SAMAV3_X */
		default:
			wStatusTmp = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFC);
			break;
	}

	return wStatusTmp;
}

#endif /* NXPBUILD__PHAL_MFC_SAM_X */
