/*
 * Copyright 2009 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software MIFARE(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 31. July 2009
 *
 */

#ifndef PHALMFC_SW_H
#define PHALMFC_SW_H

#include <ph_Status.h>
#include <phalMfc.h>

/*
 * Authenticates with the card using MIFARE keys provided.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for Authentication.
 *		bKeyType				: The type of key to be used.
 *									0x0A: PHAL_MFC_KEYA.
 *									0x0B: PHAL_MFC_KEYB.
 *		wKeyNo					: Reference key number to be used from hardware keystore.
 *		wKeyVer					: Reference key version to be used form hardware keystore.
 *		pUid					: The complete UID of the Card.
 *		bUidLen					: The length of the UID.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Authenticate(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNumber,
	uint16_t wKeyVersion, uint8_t * pUid, uint8_t bUidLength);

/*
 * Reads the data from MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for reading the data.
 *
 * Output Parameters:
 *		pData					: The data read from MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Read(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData);

/*
 * Reads the data from from MIFARE PICC. This read data will be in value format.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for reading the value.
 *
 * Output Parameters:
 *		pValue					: The value read from MIFARE PICC.
 *		pAddr					: The address read from MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_ReadValue(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t * pAddrData);

/*
 * Write the data to MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for writing the data.
 *		pData					: The data to be written to MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Write(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData);

/*
 * Write the value to MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for writing the value.
 *		pValue					: The value to be written to MIFARE PICC.
 *		bAddrData				: The address to written to MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_WriteValue(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t bAddrData);

/*
 * Increments the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for incrementing the value.
 *		pValue					: The value to be incremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Increment(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t * pValue);

/*
 * Decrements the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for decrementing the value.
 *		pValue					: The value to be Decremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Decrement(phalMfc_Sw_DataParams_t * pDataParams,  uint8_t bBlockNo, uint8_t * pValue);

/*
 * Transfers the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Transfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo);

/*
 * Restores the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for restoring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_Restore(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bBlockNo);

/*
 * Increments and Transfers the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for incrementing the value.
 *		bDstBlockNo				: The destination block number used for transferring the value.
 *		pValue					: The value to be incremented and transferred.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_IncrementTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo,
	uint8_t * pValue);

/*
 * Decrements and Transfers the value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for Decrementing the value.
 *		bDstBlockNo				: The destination block number used for transferring the value.
 *		pValue					: The value to be decremented and transferred.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_DecrementTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo,
	uint8_t * pValue);

/*
 * Restores the transferred value in a MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for restoring the value.
 *		bDstBlockNo				: The destination block number used for restoring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_RestoreTransfer(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo);

/*
 * Personalizes the UID based om the UID type provided.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bUidType				: One of the below types for personalizing the UID.
 *									0x00: PHAL_MFC_UID_TYPE_UIDF0
 *									0x40: PHAL_MFC_UID_TYPE_UIDF1
 *									0x20: PHAL_MFC_UID_TYPE_UIDF2
 *									0x60: PHAL_MFC_UID_TYPE_UIDF3
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_PersonalizeUid(phalMfc_Sw_DataParams_t * pDataParams, uint8_t bUidType);

/*
 * Perform MIFARE(R) GET VERSION usage command sequence with MIFARE Picc. GET_VERSION command is used to retrieve information
 * about the NTAG family, the product version, storage size and other product data required to identify the specific NTAG-I2C.
 *
 * Output Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		pVersion				: The version returned by PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sw_GetVersion(phalMfc_Sw_DataParams_t * pDataParams, uint8_t * pVersion);

#endif /* PHALMFC_SW_H */
