/*
 * Copyright 2020 - 2021, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#ifdef NXPBUILD__PHAL_MFDFEVX_SAM_X

#include "../phalMfdfEVx_Int.h"
#include "phalMfdfEVx_Sam_X.h"
#include "phalMfdfEVx_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfdfEVx_SamAV3_X_Init(phalMfdfEVx_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_SamAV3_DataParams_t * pHalSamDataParams, phTMIUtils_t * pTMIDataParams, uint8_t * pTmpBuffer,
    uint16_t wTmpBufSize)
{
    /* data param check */
    if(sizeof(phalMfdfEVx_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDFEVX);
    }

    /* Temporary Buffer size check. */
    if(wTmpBufSize < 256U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_MFDFEVX);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFDFEVX);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFDFEVX);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_MFDFEVX | PHAL_MFDFEVX_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pTMIDataParams = pTMIDataParams;
    pDataParams->pTmpBuffer = pTmpBuffer;
    pDataParams->wTmpBufSize = wTmpBufSize;
    pDataParams->bAuthMode = PHAL_MFDFEVX_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFFU;
    pDataParams->bWrappedMode = PH_OFF;
    pDataParams->wAdditionalInfo = 0x0000U;
    pDataParams->bCmdCode = PHAL_MFDFEVX_CMD_INVALID;
    pDataParams->bReturn_FabID = PH_OFF;

    /* Default selected application */
    (void) memset(pDataParams->pAid, 0x00U, 3U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
phStatus_t phalMfdfEVx_Sam_X_Init(phalMfdfEVx_Sam_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    void * pHalSamDataParams, phTMIUtils_t * pTMIDataParams, uint8_t * pTmpBuffer,
    uint16_t wTmpBufSize)
{
    /* data param check */
    if(sizeof(phalMfdfEVx_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDFEVX);
    }

    /* Temporary Buffer size check. */
    if(wTmpBufSize < 256U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_MFDFEVX);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFDFEVX);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFDFEVX);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_MFDFEVX | PHAL_MFDFEVX_SAM_X_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pTMIDataParams = pTMIDataParams;
    pDataParams->pTmpBuffer = pTmpBuffer;
    pDataParams->wTmpBufSize = wTmpBufSize;
    pDataParams->bAuthMode = PHAL_MFDFEVX_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFFU;
    pDataParams->bWrappedMode = PH_OFF;
    pDataParams->wAdditionalInfo = 0x0000U;
    pDataParams->bCmdCode = PHAL_MFDFEVX_CMD_INVALID;
    pDataParams->bReturn_FabID = PH_OFF;

    /* Default selected application */
    (void) memset(pDataParams->pAid, 0x00U, 3U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

/* MIFARE DESFire EVx secure messaging related commands. ------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_Authenticate(void *pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;

    /* Check for CardKeyNo. */
    if((bKeyNoCard & 0x0FU) > 0x0DU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for valid SAM KeyStore number and version. */
    if(IS_VALID_SAM_KEYS(wKeyNo) || (wKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Change for valid diversification options. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
        (wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate diversification input length. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) && (bDivInputLen > 31U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Set Auth mode with diversification enabled. */
    bAuthMode = (uint8_t) (wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION);

    /* Set Diversification flags.
     * For AV1 compatibility mode key diversification methods, TDEA Key, diversified using one encryption round
     */
    if(wOption == PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV1_SINGLE_ENCRYPTION);
    }

    /* Set Diversification flags.
     * AV2 compatibility mode key diversification methods, 3TDEA, AES key
     */
    if(wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV2);
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_AUTHENTICATE);

    /* Unset the current authentication status. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816
        : PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Exchange the information to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(
        pDataParams,
        bAuthMode,
        bISOMode,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        0U,
        NULL,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_AUTHENTICATE);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_AuthenticateISO(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;

    /* Check for CardKeyNo. */
    if((bKeyNoCard & 0x0FU) > 0x0DU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for valid SAM KeyStore number and version. */
    if(IS_VALID_SAM_KEYS(wKeyNo) || (wKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Change for valid diversification options. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
        (wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate diversification input length. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) && (bDivInputLen > 31U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Set Auth mode with diversification enabled. */
    bAuthMode = (uint8_t) (wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION);

    /* Set Diversification flags.
     * For AV1 compatibility mode key diversification methods, TDEA Key, diversified using one encryption round
     */
    if(wOption == PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV1_SINGLE_ENCRYPTION);
    }

    /* Set Diversification flags.
     * AV2 compatibility mode key diversification methods, 3TDEA, AES key
     */
    if(wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV2);
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_AUTHENTICATE_ISO);

    /* Unset the current authentication status. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816
        : PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Exchange the information to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(
        pDataParams,
        bAuthMode,
        bISOMode,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        0U,
        NULL,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_AUTHENTICATEISO);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_AuthenticateAES(void * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };
    uint8_t *   PH_MEMLOC_REM pAppId = NULL;

    /* Check for CardKeyNo. */
    if((bKeyNoCard & 0x0FU) > 0x0DU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for valid SAM KeyStore number and version. */
    if(IS_VALID_SAM_KEYS(wKeyNo) || (wKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Change for valid diversification options. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
        (wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate diversification input length. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) && (bDivInputLen > 31U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Set Auth mode with diversification enabled. */
    bAuthMode = (uint8_t) (wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION);

    /* Set Diversification flags.
     * AV2 compatibility mode key diversification methods, 3TDEA, AES key
     */
    if(wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV2);
    }

    /* Set the secure messaging. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if((bKeyNoCard >= PHAL_MFDFEVX_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFEVX_ORIGINALITY_KEY_LAST) &&
        (memcmp(pAppId, aAppId, 3U) == 0x00U))
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_SUPPRESS_SECURE_MESSAGING);
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_AUTHENTICATE_AES);

    /* Unset the current authentication status. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816
        : PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Exchange the information to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(
        pDataParams,
        bAuthMode,
        bISOMode,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        0U,
        NULL,
        pDivInput,
        bDivInputLen,
        NULL,
        NULL,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Update the authentication states. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_AUTHENTICATEAES);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard);

    /* Do not update the auth state if originality keys are used. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if((bKeyNoCard >= PHAL_MFDFEVX_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFEVX_ORIGINALITY_KEY_LAST) &&
        (memcmp(pAppId, aAppId, 3U) == 0x00U))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_AuthenticateEv2(void *pDataParams, uint8_t bFirstAuth, uint16_t wOption, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCapsIn,
    uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t		PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };
    uint8_t *   PH_MEMLOC_REM pAppId = NULL;

    /* Validate auth type. */
    if(bFirstAuth > 0x01U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for CardKeyNo. */
    if((bKeyNoCard & 0x0FU) > 0x0DU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate SAM Key number and version. */
    if(IS_VALID_SAM_KEYS(wKeyNo) || (wKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate Diversification options. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) &&
        (wOption != (PH_CRYPTOSYM_DIV_MODE_DESFIRE | PH_CRYPTOSYM_DIV_OPTION_2K3DES_FULL)) &&
        (wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Validate diversification input length. */
    if((wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION) && (bDivInputLen > 31U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /*
     * Set EV2 Authentication type.
     * EV2 authentication type
     * Authenticate first
     * Authenticate non first
     * AV2 compatibility mode key diversification methods, 3TDEA, AES key
     */
    bAuthMode = (uint8_t) ((bFirstAuth ? 0x80U : 0xC0U));

    /* Set Auth mode with diversification enabled. */
    bAuthMode = (uint8_t) (bAuthMode | (wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION));

    /* Set Diversification flags.
     * AV2 compatibility mode key diversification methods, 3TDEA, AES key
     */
    if(wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_KDF_AV2);
    }

    /* Set the secure messaging. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if((bKeyNoCard >= PHAL_MFDFEVX_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFEVX_ORIGINALITY_KEY_LAST) &&
        (memcmp(pAppId, aAppId, 3U) == 0x00U))
    {
        bAuthMode = (uint8_t) (bAuthMode | PHAL_MFDFEVX_SAM_X_SUPPRESS_SECURE_MESSAGING);
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, (uint8_t) ((bFirstAuth ? PHAL_MFDFEVX_CMD_AUTHENTICATE_EV2_FIRST :
        PHAL_MFDFEVX_CMD_AUTHENTICATE_EV2_NON_FIRST)));

    /* Unset the current authentication status */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816
        : PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Exchange the information to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(
        pDataParams,
        bAuthMode,
        bISOMode,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        bPcdCapsInLen,
        bPcdCapsInLen ? pPcdCapsIn : NULL,
        pDivInput,
        bDivInputLen,
        pPdCapsOut,
        pPcdCapsOut,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Update the authentication states. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_AUTHENTICATEEV2);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard);

    /* Do not update the auth state if originality keys are used. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if((bKeyNoCard >= PHAL_MFDFEVX_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFEVX_ORIGINALITY_KEY_LAST) &&
        (memcmp(pAppId, aAppId, 3U) == 0x00U))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx Memory and Configuration management commands. -------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_FreeMem(void * pDataParams, uint8_t * pMemInfo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_FREE_MEM);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_FREE_MEM;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pMemInfo, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_FormatPICC(void * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_FORMAT);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_FORMAT;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_NOT_AUTHENTICATED)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the Length and Crypto information. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        /* Start of data. */
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);

        /* Set Extended Offset in Crypto mode. */
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_SET_CONFIG);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_SET_CONFIG;
    pAppData[wAppDataLen++] = bOption;

    (void) memcpy(&pAppData[wAppDataLen], pData, bDataLen);
    wAppDataLen = (uint16_t) (wAppDataLen + bDataLen);

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL;

    /* Set the Command Offset. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = (uint8_t) (bCrypto + (wAppDataLen - bDataLen));
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM bVerLen = 0;
    uint8_t		PH_MEMLOC_REM bFinished = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_VERSION;

    /* Append Return of Option information. */
    if(phalMfdfEVx_Sam_X_Int_GetFabID(pDataParams) == PH_ON)
    {
        pAppData[wAppDataLen++] = phalMfdfEVx_Sam_X_Int_GetFabID(pDataParams);
    }

    do
    {
        /* Wrap the command if required. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
            pDataParams,
            PH_ON,
            PH_OFF,
            0U,
            PH_ON,
            0U,
            pAppData,
            &wAppDataLen));

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCrypto,
            pAppData,
            (uint8_t) wAppDataLen,
            &pResponse,
            &wRespLen);

        /* Evaluate the response. */
        wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

        /* Updated Finished to end the loop. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
        {
            PH_CHECK_SUCCESS(wStatus);
        }
        else
        {
            /* Update Command frame. */
            wAppDataLen = 0U;
            pAppData[wAppDataLen++] = 0xAFU;

            /* Copy the response information to the parameter. */
            (void) memcpy(&pVerInfo[bVerLen], pResponse, wRespLen);
            bVerLen = (uint8_t) (bVerLen + wRespLen);

            /* Update Finish flag. */
            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                bFinished = 1U;
            }
        }
    } while(!bFinished);

    /* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
    *pVerLen = bVerLen;
    phalMfdfEVx_Sam_X_SetConfig(pDataParams, PHAL_MFDFEVX_ADDITIONAL_INFO, bVerLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetCardUID(void * pDataParams, uint8_t bExchangeOption, uint8_t bOption, uint8_t * pUid,
    uint8_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bUidLen = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM bUidOffset = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Check if UID length is provided. */
    bUidLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams) != 4U) &&
        (phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams) != 7U) &&
        (phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams) != 10U) ? 7U :
        phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams));

    /* Append expected length to the command frame. */
    pAppData[wAppDataLen++] = bUidLen;
    pAppData[wAppDataLen++] = 0U;
    pAppData[wAppDataLen++] = 0U;

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_CARD_UID;

    if(bExchangeOption)
    {
        pAppData[wAppDataLen++] = bOption;

        /* Update UID Length if there is NUID available. */
        pAppData[0] = (uint8_t) ((bOption == PHAL_MFDFEVX_GET_CARD_UID_OPTION_NUID_RETURNED) ? (bUidLen + 4U) : bUidLen);
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_ON,
        3U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL;

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Response will be received as
    * 1. 7 byte UID
    * 2. [1 Byte UID Format] + [1 byte UID Length(0x04)] + 4 byte UID
    * 3. [1 Byte UID Format] + [1 byte UID Length(0x0A)] + 10 byte UID
    */
    if(!bExchangeOption)
    {
        if(((wRespLen != PHAL_MFDFEVX_DEFAULT_UID_LENGTH) &&
            (wRespLen != PHAL_MFDFEVX_10B_UID_LENGTH) &&
            (wRespLen != PHAL_MFDFEVX_4B_UID_LENGTH)))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFEVX);
        }
    }

    if(((wRespLen == PHAL_MFDFEVX_10B_UID_LENGTH) || (wRespLen == PHAL_MFDFEVX_4B_UID_LENGTH)) && !bExchangeOption)
    {
        /* In case of 4B/10B UID, strip out first 2 bytes as it contains UID format and UID length */
        wRespLen -= 2U;

        *pUidLen = (uint8_t) wRespLen;

        /* Validate UIDFormat (0x00) for 4byte and 7Byte UID and UIDLength to be equal to real UID */
        if((pUid[0U] != 0x00U) | (pUid[1U] != *pUidLen))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFEVX);
        }
        memcpy(pUid, &pUid[2U], *pUidLen);
    }
    else
    {
        *pUidLen = (uint8_t) wRespLen;

        /* Compute the UIDOffset. */
        bUidOffset = (uint8_t) (((wRespLen == PHAL_MFDFEVX_DEFAULT_UID_LENGTH) ||
            (wRespLen == (PHAL_MFDFEVX_DEFAULT_UID_LENGTH + 4U))) ? 0U : 2U);
    }

    /* Update the UID information to the DataParams. */
    phalMfdfEVx_Sam_X_Int_SetUID(pDataParams, &pUid[bUidOffset], *pUidLen - bUidOffset);

    /* Set the card Length in wAdditionalInfo. This is done to assist C# wrapper as it will not be able
     * to recognize the card UID Length.
     */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_SetConfig(pDataParams, PHAL_MFDFEVX_ADDITIONAL_INFO, wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx Key management commands. ----------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer,
    uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t		PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };
    uint8_t *   PH_MEMLOC_REM pAppId = NULL;

    /* Only if selected Aid is 0x000000. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if(memcmp(pAppId, aAppId, 3U) == 0x00U)
    {
        /* Master key */
        if((bKeyNoCard & 0x3FU) != 0x00U)
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
        else
        {
            /*Do Nothing. This is for PRQA compliance */
        }
    }
    else
    {
        if(bKeyNoCard > 0x0DU)
        {
            /* Invalid application key specified */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
    }

    /* Check for valid SAM key number and version. */
    if(IS_VALID_SAM_KEYS(wOldKeyNo) || (wOldKeyVer > 0xFFU) ||
        IS_VALID_SAM_KEYS(wNewKeyNo) || (wNewKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CHANGE_KEY);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams)
        ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816 : PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Set the key compilation method. */
    if(wOption == PHAL_MFDFEVX_NO_DIVERSIFICATION)
    {
        bKeyCompMeth = 0x00U;
    }
    else
    {
        /* Validate diversification input length. */
        if(bDivInputLen > 31U)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }

        /* Validate option information. */
        if(!(wOption & 0x003EU))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }

        /* Assign the option to local variable. */
        bKeyCompMeth = (uint8_t) wOption;
    }

    /* DESFire key number to be changed. */
    bCfg = (uint8_t) (0x0FU & bKeyNoCard);

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if((bKeyNoCard & 0x3FU) == phalMfdfEVx_Sam_X_Int_GetKeyNo(pDataParams))
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | 0x01U);
    }

    /* Include the key type in the cryptogram for Master Key */
    if(memcmp(pAppId, aAppId, 3U) == 0x00U)
    {
        bCfg = (uint8_t) (bCfg | 0x10U);
    }

    /* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | bISOMode);

    /* Exchange the command to PICC. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ChangeKeyPICC(
        pDataParams,
        bKeyCompMeth,
        bCfg,
        0x00U,
        0x00U,
        (uint8_t) wOldKeyNo,
        (uint8_t) wOldKeyVer,
        (uint8_t) wNewKeyNo,
        (uint8_t) wNewKeyVer,
        pDivInput,
        bDivInputLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Reset the Auth state. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATE)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
        }

        return wStatus;
    }
    else
    {
        /* Reset authentication status only if the key authenticated with is changed. */
        if(((phalMfdfEVx_Sam_X_Int_GetKeyNo(pDataParams) & 0x3FU) == (bKeyNoCard & 0x3FU)))
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ChangeKeyEv2(void * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer,
    uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeySetNo, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t		PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };
    uint8_t *   PH_MEMLOC_REM pAppId = NULL;

    /* Only if selected Aid is 0x000000. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAppId);
    if(memcmp(pAppId, aAppId, 3U) == 0x00U)
    {
            /* PICC Master key */
        if(((bKeyNoCard & 0x3FU) != 0x00U) &&

            /* PICC DAMAuthKey, DAMMACKey, DAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x10U) && ((bKeyNoCard & 0x3FU) != 0x11U) && ((bKeyNoCard & 0x3FU) != 0x12U) &&

            /* PICC NXPDAMAuthKey, NXPDAMMACKey, NXPDAMEncKey */
            ((bKeyNoCard & 0x3FU) != 0x18U) && ((bKeyNoCard & 0x3FU) != 0x19U) && ((bKeyNoCard & 0x3FU) != 0x1AU) &&

            /* PICC VCConfigurationKey, VCProximityKey, VCSelectMACKey, VCSelectENCKey */
            ((bKeyNoCard & 0x3FU) != 0x20U) && ((bKeyNoCard & 0x3FU) != 0x21U) && ((bKeyNoCard & 0x3FU) != 0x22U) &&
            ((bKeyNoCard & 0x3FU) != 0x23U) &&

            /* PICC VCPollingEncKey, VCPollingMACKey */
            ((bKeyNoCard & 0x3FU) != 0x30U) && ((bKeyNoCard & 0x3FU) != 0x31U) &&

            /* MFCKillKey, MFCLicenseMACKey */
            ((bKeyNoCard & 0x3FU) != 0x31U) && ((bKeyNoCard & 0x3FU) != 0x32U))
        {
            /* Invalid card key number supplied */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
        else
        {
            /*Do Nothing. This is for PRQA compliance */
        }
    }
    else
    {
        /* Key numbers between 0D and 21 are not allowed for App level, also key numbers above 23 are not allowed.
           if AID 0x000000 is not selected,At application level, VC keys 0x21, 0x22 and 0x23 can be enabled at application creation,
           Refer reference architecture version 13 */
        if(IS_INVALID_APP_KEY(bKeyNoCard) || IS_INVALID_VC_KEY(bKeyNoCard))
        {
            /* Invalid application key specified */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
    }

    /* Validate KeySet number. */
    if(bKeySetNo > 0x0FU)
    {
        /* Invalid KeySetNo specified */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for valid SAM key number and version. */
    if(IS_VALID_SAM_KEYS(wOldKeyNo) || (wOldKeyVer > 0xFFU) ||
        IS_VALID_SAM_KEYS(wNewKeyNo) || (wNewKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Clear the PICC status code buffer. */
    (void) memset(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 0x00U,
        sizeof(PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CHANGE_KEY_EV2);

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816 :
        PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Set the key compilation method. */
    if(wOption == PHAL_MFDFEVX_NO_DIVERSIFICATION)
    {
        bKeyCompMeth = 0x00U;
    }
    else
    {
        /* Validate diversification input length. */
        if(bDivInputLen > 31U)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }

        /* Validate option information. */
        if(!(wOption & 0x003E))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }

        /* Assign the option to local variable. */
        bKeyCompMeth = (uint8_t) wOption;
    }

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if((bKeyNoCard & 0x3FU) == 0)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | 0x01U);
    }

    /* Set EV2 ChangeKey type. */
    bCfg = (uint8_t) (bCfg | 0x20U);

    /* Include the key type in the cryptogram for Master Key */
    if((memcmp(pAppId, aAppId, 3U) == 0x00U) && ((bKeyNoCard & 0x3FU) == 0x00U))
    {
        bCfg = (uint8_t) (bCfg | 0x10U);
    }

    /* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | bISOMode);

    /* Exchange the command to PICC. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ChangeKeyPICC(
        pDataParams,
        bKeyCompMeth,
        bCfg,
        bKeySetNo,
        (uint8_t) (bKeyNoCard & 0x3FU),
        (uint8_t) wOldKeyNo,
        (uint8_t) wOldKeyVer,
        (uint8_t) wNewKeyNo,
        (uint8_t) wNewKeyVer,
        pDivInput,
        bDivInputLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Reset the Auth state. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATE)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
        }

        return wStatus;
    }
    else
    {
        /* Reset authentication status only if the key authenticated with is changed. */
        if(((phalMfdfEVx_Sam_X_Int_GetKeyNo(pDataParams) & 0x3FU) == (bKeyNoCard & 0x3FU)) &&
            ((bKeySetNo & 0x0FU) == 0U))
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_InitializeKeySet(void * pDataParams, uint8_t bKeySetNo, uint8_t bKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Validate the parameters. */
    if(((bKeySetNo & 0x7FU) > 0x0FU) || (bKeyType > PHAL_MFDFEVX_KEY_TYPE_AES128))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_INITIALIZE_KEY_SET);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_INITIALIZE_KEY_SET;
    pAppData[wAppDataLen++] = bKeySetNo;
    pAppData[wAppDataLen++] = bKeyType;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    if((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEISO) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_FinalizeKeySet(void * pDataParams, uint8_t bKeySetNo, uint8_t bKeyVersion)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Validate the parameters */
    if((bKeySetNo & 0x7FU) > 0x0FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_FINALIZE_KEY_SET);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_FINALIZE_KEY_SET;
    pAppData[wAppDataLen++] = bKeySetNo;
    pAppData[wAppDataLen++] = bKeyVersion;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    if((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEISO) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_RollKeySet(void * pDataParams, uint8_t bKeySetNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Validate parameters */
    if((bKeySetNo & 0x7FU) > 0x0FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ROLL_KEY_SET);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ROLL_KEY_SET;
    pAppData[wAppDataLen++] = bKeySetNo;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    if((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEISO) ||
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Reset the states. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetKeySettings(void * pDataParams, uint8_t * pKeySetting, uint8_t * pKeySettingLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_KEY_SETTINGS);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_KEY_SETTINGS;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pKeySetting, pResponse, wRespLen);
    *pKeySettingLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ChangeKeySettings(void * pDataParams, uint8_t bKeySettings)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CHANGE_KEY_SETTINGS);

    /* Set the Length and Crypto information. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        /* Start of data. */
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 5U : 1U);

        /* Set Extended Offset in Crypto mode. */
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CHANGE_KEY_SETTINGS;
    pAppData[wAppDataLen++] = bKeySettings;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL;

    /* Set the Command Offset. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 1U));
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo, uint8_t * pKeyVersion,
    uint8_t * pKeyVerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    uint8_t *   PH_MEMLOC_REM pAid = NULL;

     /* Only if selected Aid is 0x000000. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAid);
    if((pAid[0U] != 0U) && (pAid[1U] != 0U) && (pAid[2U] != 0U) &&
        ((bKeyNo & 0x0FU) > 0x0DU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_KEY_VERSION);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_KEY_VERSION;
    pAppData[wAppDataLen++] = bKeyNo;

    /* Add KeySet number if set in KeyNo bit 6. */
    if(bKeyNo & PHAL_MFDFEVX_KEYSETVERSIONS)
    {
        pAppData[wAppDataLen++] = bKeySetNo;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pKeyVersion, pResponse, wRespLen);
    *pKeyVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx Application management commands. --------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_CreateApplication(void * pDataParams, uint8_t bOption, uint8_t * pAid, uint8_t bKeySettings1,
    uint8_t bKeySettings2, uint8_t bKeySettings3, uint8_t * pKeySetValues, uint8_t * pISOFileId, uint8_t * pISODFName,
    uint8_t bISODFNameLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Check for valid ISO DFName */
    if((bISODFNameLen > 16U) || (bOption == 0x02U) || (bOption > 0x03U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_APPLN);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_APPLN;

    /* Buffer Application identifier to the command frame. */
    (void) memcpy(&pAppData[wAppDataLen], pAid, 3U);
    wAppDataLen += 3U;

    /* Buffer Key settings information to command frame. */
    pAppData[wAppDataLen++] = bKeySettings1;
    pAppData[wAppDataLen++] = bKeySettings2;

    /* Check if KeySettings 3 to be passed */
    if(bKeySettings2 & PHAL_MFDFEVX_KEYSETT3_PRESENT)
    {
        pAppData[wAppDataLen++] = bKeySettings3;

        /* Buffer key set values if required. */
        if((bKeySettings3 & PHAL_MFDFEVX_KEYSETVALUES_PRESENT) && pKeySetValues != NULL)
        {
            (void) memcpy(&pAppData[wAppDataLen], pKeySetValues, 4U);
            wAppDataLen += 4U;
        }
    }

    /* Buffer ISO FileID to exchange buffer. */
    if((bOption & 0x01U) == 0x01U)
    {
        pAppData[wAppDataLen++] = pISOFileId[0U];
        pAppData[wAppDataLen++] = pISOFileId[1U];
    }

    /* Buffer ISO DFName to exchange buffer. */
    if((bOption & 0x02U) == 0x02U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISODFName, bISODFNameLen);
        wAppDataLen += bISODFNameLen;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_DeleteApplication(void * pDataParams, uint8_t * pAid, uint8_t * pDAMMAC, uint8_t bDAMMAC_Len)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t *   PH_MEMLOC_REM pAID = NULL;

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_DELETE_APPLN);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_DELETE_APPLN;

    /* Buffer Application identifier to the command frame. */
    (void) memcpy(&pAppData[wAppDataLen], pAid, 3U);
    wAppDataLen += 3U;

    /* Append the DAMMAC */
    if(bDAMMAC_Len)
    {
        (void) memcpy(&pAppData[wAppDataLen], pDAMMAC, bDAMMAC_Len);
        wAppDataLen += bDAMMAC_Len;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Reset the states. */
    phalMfdfEVx_Sam_X_Int_GetAID(pDataParams, &pAID);
    if((pAID[0U] == pAid[0U]) && (pAID[1U] == pAid[1U]) && (pAID[2U] == pAid[2U]))
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));
    }

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateDelegatedApplication(void  * pDataParams, uint8_t bOption, uint8_t * pAid, uint8_t * pDamParams,
    uint8_t bKeySettings1, uint8_t bKeySettings2, uint8_t bKeySettings3, uint8_t * pKeySetValues, uint8_t * pISOFileId,
    uint8_t * pISODFName, uint8_t bISODFNameLen, uint8_t * pEncK, uint8_t * pDAMMAC)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Check for valid ISO DFName */
    if((bISODFNameLen > 16U) || (bOption == 0x02U) || (bOption > 0x03U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_DELEGATED_APPLN);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_DELEGATED_APPLN;

    /* Buffer Application identifier to the command frame. */
    (void) memcpy(&pAppData[wAppDataLen], pAid, 3U);
    wAppDataLen += 3U;

    /* Buffer DAM param to the command frame. */
    (void) memcpy(&pAppData[wAppDataLen], pDamParams, 5U);
    wAppDataLen += 5U;

    /* Buffer Key settings information to command frame. */
    pAppData[wAppDataLen++] = bKeySettings1;
    pAppData[wAppDataLen++] = bKeySettings2;

    /* Check if KeySettings 3 to be passed */
    if(bKeySettings2 & PHAL_MFDFEVX_KEYSETT3_PRESENT)
    {
        pAppData[wAppDataLen++] = bKeySettings3;

        /* Buffer key set values if required. */
        if((bKeySettings3 & PHAL_MFDFEVX_KEYSETVALUES_PRESENT) && pKeySetValues != NULL)
        {
            (void) memcpy(&pAppData[wAppDataLen], pKeySetValues, 4U);
            wAppDataLen += 4U;
        }
    }

    /* Buffer ISO FileID to exchange buffer. */
    if((bOption & 0x01U) == 0x01U)
    {
        pAppData[wAppDataLen++] = pISOFileId[0U];
        pAppData[wAppDataLen++] = pISOFileId[1U];
    }

    /* Buffer ISO DFName to exchange buffer. */
    if((bOption & 0x02U) == 0x02U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISODFName, bISODFNameLen);
        wAppDataLen += bISODFNameLen;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_TXCHAINING,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Check for Chaining status. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));
    }

    /* Frame second part of command information. */
    wAppDataLen = 0U;
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;

    (void) memcpy(&pAppData[wAppDataLen], pEncK, 32U);
    wAppDataLen += 32U;

    (void) memcpy(&pAppData[wAppDataLen], pDAMMAC, 8U);
    wAppDataLen += 8U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen));

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_SelectApplication(void * pDataParams, uint8_t bOption, uint8_t * pAppId, uint8_t * pAppId2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_SELECT_APPLN);

    /* Reset the Auth states. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams));

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_SELECT_APPLN;

    /* Add the primary application identifier */
    (void) memcpy(&pAppData[wAppDataLen], pAppId, 3U);
    wAppDataLen += 3U;

    /* Add the secondary application identifier */
    if(bOption == 0x01U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pAppId2, 3U);
        wAppDataLen += 3U;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the AID to the params. */
    phalMfdfEVx_Sam_X_Int_SetAID(pDataParams, pAppId);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetApplicationIDs(void * pDataParams, uint8_t bOption, uint8_t ** ppAidBuff, uint8_t * pNumAIDs)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameter. */
    if(((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_APPLN_IDS);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = (uint8_t) (((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
        ? PHAL_MFDFEVX_CMD_GET_APPLN_IDS : PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME);

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));
    if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
    {
        /* Copy the data to the parameter */
        *ppAidBuff = pResponse;
        *pNumAIDs = (uint8_t) (wRespLen / 3U);
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_GetDFNames(void * pDataParams, uint8_t bOption, uint8_t * pDFBuffer, uint8_t * bSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameter. */
    if(((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_DF_NAMES);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = (uint8_t) (((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
        ? PHAL_MFDFEVX_CMD_GET_DF_NAMES : PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME);

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));
    if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
    {
        /* Copy the data to the parameter */
        (void) memcpy(pDFBuffer, pResponse, wRespLen);
        *bSize = (uint8_t) wRespLen;
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_GetDelegatedInfo(void * pDataParams, uint8_t * pDAMSlot, uint8_t * pDamSlotVer, uint8_t * pQuotaLimit,
    uint8_t * pFreeBlocks, uint8_t * pAid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_DELEGATED_INFO);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_DELEGATED_INFO;

    /* Add DAM Slot number. */
    (void) memcpy(&pAppData[wAppDataLen], pDAMSlot, 2U);
    wAppDataLen += 2U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pDamSlotVer, &pResponse[0U], 1U);
    (void) memcpy(pQuotaLimit, &pResponse[1U], 2U);
    (void) memcpy(pFreeBlocks, &pResponse[3U], 2U);
    (void) memcpy(pAid, &pResponse[5U], 3U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx File management commands. --------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_CreateStdDataFile(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pISOFileId,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pFileSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_STD_DATAFILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_STD_DATAFILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Append ISOFileID is available. */
    if(bOption == 0x01U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISOFileId, 2U);
        wAppDataLen += 2U;
    }

    /* Append communication settings */
    pAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
    (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
    wAppDataLen += 2U;

    /* Append FileSize. */
    (void) memcpy(&pAppData[wAppDataLen], pFileSize, 3U);
    wAppDataLen += 3U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateBackupDataFile(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pISOFileId,
    uint8_t bFileOption, uint8_t *pAccessRights, uint8_t * pFileSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bOption > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_BKUP_DATAFILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_BKUP_DATAFILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Append ISOFileID is available. */
    if(bOption == 0x01U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISOFileId, 2U);
        wAppDataLen += 2U;
    }

    /* Append communication settings */
    pAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
    (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
    wAppDataLen += 2U;

    /* Append FileSize. */
    (void) memcpy(&pAppData[wAppDataLen], pFileSize, 3U);
    wAppDataLen += 3U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateValueFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights,
    uint8_t * pLowerLmit, uint8_t * pUpperLmit, uint8_t * pValue, uint8_t bLimitedCredit)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if((bFileNo & 0x7FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_VALUE_FILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_VALUE_FILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Append communication settings */
    pAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
    (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
    wAppDataLen += 2U;

    /* Append lower limit. */
    (void) memcpy(&pAppData[wAppDataLen], pLowerLmit, 4U);
    wAppDataLen += 4U;

    /* Append upper limit. */
    (void) memcpy(&pAppData[wAppDataLen], pUpperLmit, 4U);
    wAppDataLen += 4U;

    /* Append value. */
    (void) memcpy(&pAppData[wAppDataLen], pValue, 4U);
    wAppDataLen += 4U;

    /* Append LimitedCreditEnabled information */
    pAppData[wAppDataLen++] = bLimitedCredit;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateLinearRecordFile(void * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t * pISOFileId,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bOption > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_LINEAR_RECFILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_LINEAR_RECFILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Append ISOFileID is available. */
    if(bOption == 0x01U)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISOFileId, 2U);
        wAppDataLen += 2U;
    }

    /* Append communication settings */
    pAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
    (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
    wAppDataLen += 2U;

    /* Append record size. */
    (void) memcpy(&pAppData[wAppDataLen], pRecordSize, 3U);
    wAppDataLen += 3U;

    /* Append maximum number of records. */
    (void) memcpy(&pAppData[wAppDataLen], pMaxNoOfRec, 3U);
    wAppDataLen += 3U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateCyclicRecordFile(void * pDataParams, uint8_t bOption, uint8_t  bFileNo, uint8_t * pISOFileId,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pRecordSize, uint8_t * pMaxNoOfRec)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bOption > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_CYCLIC_RECFILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_CYCLIC_RECFILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Append ISOFileID is available. */
    if(bOption == 0x01)
    {
        (void) memcpy(&pAppData[wAppDataLen], pISOFileId, 2U);
        wAppDataLen += 2U;
    }

    /* Append communication settings */
    pAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
    (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
    wAppDataLen += 2U;

    /* Append record size. */
    (void) memcpy(&pAppData[wAppDataLen], pRecordSize, 3U);
    wAppDataLen += 3U;

    /* Append maximum number of records. */
    (void) memcpy(&pAppData[wAppDataLen], pMaxNoOfRec, 3U);
    wAppDataLen += 3U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CreateTransactionMacFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights,
    uint16_t wKeyNo, uint8_t bKeyVer, uint8_t bKeyType, uint8_t * pKey, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bISOMode = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* VAlidate the parameters */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bKeyType != PHAL_MFDFEVX_KEY_TYPE_AES128))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_TRANSTN_MACFILE);

    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_NOT_AUTHENTICATED)
    {
        /* Frame the command information. */
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_TRANSTN_MACFILE;
        pAppData[wAppDataLen++] = bFileNo;
        pAppData[wAppDataLen++] = bFileOption;

        /* Append access rights. */
        (void) memcpy(&pAppData[wAppDataLen], pAccessRights, 2U);
        wAppDataLen += 2U;

        /* Append TMKeyOption. */
        pAppData[wAppDataLen++] = bKeyType;

        /* Append TMKey. */
        (void) memcpy(&pAppData[wAppDataLen], pKey, 16U);
        wAppDataLen += 16U;

        /* Append TMKeyVer. */
        pAppData[wAppDataLen++] = bKeyVer;

        /* Wrap the command if required. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
            pDataParams,
            PH_ON,
            PH_OFF,
            0U,
            PH_ON,
            0U,
            pAppData,
            &wAppDataLen));

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN,
            pAppData,
            (uint8_t) wAppDataLen,
            &pResponse,
            &wRespLen);
    }
    else
    {
        /* Set Native or ISO7816 PICC command format. */
        bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816 :
            PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_CreateTMFilePICC(
            pDataParams,
            (uint8_t) (bDivInputLen > 0),
            bISOMode,
            (uint8_t) wKeyNo,
            bKeyVer,
            bFileNo,
            (uint8_t) (bFileOption >> 4U),
            pAccessRights,
            bKeyType,
            pDivInput,
            bDivInputLen);
    }

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if((bFileNo & 0x7FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_DELETE_FILE);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_DELETE_FILE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetFileIDs(void * pDataParams, uint8_t * pFid, uint8_t * bNumFIDs)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_FILE_IDS);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_FILE_IDS;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));
    if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
    {
        /* Copy the data to the parameter */
        (void) memcpy(pFid, pResponse, wRespLen);
        *bNumFIDs = (uint8_t) wRespLen;
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_GetISOFileIDs(void * pDataParams, uint8_t * pFidBuffer, uint8_t * bNumFIDs)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM bFinished = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_ISO_FILE_IDS);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_ISO_FILE_IDS;

    /* Set the number of FileID variable. */
    *bNumFIDs = 0U;

    do
    {
        /* Wrap the command if required. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
            pDataParams,
            PH_ON,
            PH_OFF,
            0U,
            PH_ON,
            0U,
            pAppData,
            &wAppDataLen));

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCrypto,
            pAppData,
            (uint8_t) wAppDataLen,
            &pResponse,
            &wRespLen);

        /* Evaluate the response. */
        wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
        {
            PH_CHECK_SUCCESS(wStatus);
        }
        else
        {
            /* Update Command frame. */
            wAppDataLen = 0U;
            pAppData[wAppDataLen++] = 0xAFU;

            /* Copy the data to the parameter */
            (void) memcpy(&pFidBuffer[*bNumFIDs * 2U], pResponse, wRespLen);
            *bNumFIDs += (uint8_t) (wRespLen / 2U);

            /* Update Finish flag. */
            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                bFinished = 1U;
            }
        }
    } while(!bFinished);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if((bFileNo & 0x7FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_FILE_SETTINGS);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_FILE_SETTINGS;
    pAppData[wAppDataLen++] = bFileNo;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pFSBuffer, pResponse, wRespLen);
    *pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GetFileCounters(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pResponse,
    uint8_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponseTmp = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;

    /* Validate the parameters */
    if((bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_ENC))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_GET_FILE_COUNTERS);

    /* Append expected length to the command frame. */
    if(bOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
    {
        pAppData[wAppDataLen++] = 5U;
        pAppData[wAppDataLen++] = 0U;
        pAppData[wAppDataLen++] = 0U;

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_FILE_COUNTERS;
    pAppData[wAppDataLen++] = bFileNo;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        3U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bOption,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponseTmp,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    (void) memcpy(pResponse, pResponseTmp, wRespLen);
    *pRespLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pAccessRights, uint8_t bAddInfoLen, uint8_t * pAddInfo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bAddARsLen = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;

    /* Validate the parameters */
    if((bFileNo & 0x3FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_PLAIN >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_ENC >> 4U)) &&
        ((bFileOption & 0x03U) != (PHAL_MFDFEVX_COMMUNICATION_MACD >> 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL;
    if((bOption & 0x30U) != PHAL_MFDFEVX_COMMUNICATION_ENC)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    }

    /* Set the Command Offset. */
    if((bOption & 0x30U) == PHAL_MFDFEVX_COMMUNICATION_ENC)
    {
        if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATEEV2)
            bCrypto |= (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);
        else
        {
            /* Start of data. */
            pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);

            /* Set Extended Offset in Crypto mode. */
            bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;

            /* Set Length Presence flag. */
            bLenPresent = PH_ON;
        }
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CHANGE_FILE_SETTINGS);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CHANGE_FILE_SETTINGS;
    pAppData[wAppDataLen++] = bFileNo;
    pAppData[wAppDataLen++] = bFileOption;
    pAppData[wAppDataLen++] = pAccessRights[0U];
    pAppData[wAppDataLen++] = pAccessRights[1U];

    if(bOption & PHAL_MFDFEVX_EXCHANGE_ADD_INFO_BUFFER_COMPLETE)
    {
        /* SDM buffer in command buffer if Bit6 of File Option is SET.  */
        memcpy(&pAppData[wAppDataLen], pAddInfo, bAddInfoLen);
        wAppDataLen += bAddInfoLen;
    }
    else
    {
        if(bFileOption & PHAL_MFDFEVX_FILE_OPTION_ADDITIONAL_AR_PRESENT)
        {
            /* Compute the Additional ACCESS Rights length. */
            bAddARsLen = (uint8_t) ((bFileOption & PHAL_MFDFEVX_FILE_OPTION_TMCLIMIT_PRESENT)
                ? (bAddInfoLen - 4U) : bAddInfoLen);

            pAppData[wAppDataLen++] = bAddARsLen;
            (void) memcpy(&pAppData[wAppDataLen], pAddInfo, (bAddARsLen * 2U));
            wAppDataLen += (bAddARsLen * 2U);
        }

        /* TMCLimit buffer in command buffer if Bit5 of File Option is SET. */
        if(bFileOption & PHAL_MFDFEVX_FILE_OPTION_TMCLIMIT_PRESENT)
        {
            (void) memcpy(&pAppData[wAppDataLen], &pAddInfo[bAddARsLen], 4U);
            wAppDataLen += 4U;
        }
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx Data management commands. --------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_ReadData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pLength, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint16_t	PH_MEMLOC_REM wOption = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    static uint32_t	PH_MEMLOC_REM dwLength = 0;
    uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameter. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bIns > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        ((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = (uint8_t) (bOption & 0xF0U);

    /* Frame Option parameter. */
    wOption = (uint16_t) (bOption & 0x0FU);
    wOption |= (uint16_t) (bIns ? PHAL_MFDFEVX_SAM_X_ISO_CHAINING : 0U);

    /* Compute the length. */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0U];

    /* Frame the command information based on the option. */
    if((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
        /* Frame additional frame code only for Native chaining. */
        if(!bIns)
        {
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
        }
    }
    else
    {
        /* Frame Presence of length information in the command frame.
         * The first three bytes specifies number of bytes to be received from PICC.
         */
        bLengthPresent = (uint8_t) (((bOption & 0xF0U) == PHAL_MFDFEVX_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

        /* Add Read Length if communication mode is Encrypted. */
        if(bLengthPresent)
        {
            (void) memcpy(&pAppData[wAppDataLen], pLength, 3U);
            wAppDataLen += 3U;
        }

        pAppData[wAppDataLen++] = (uint8_t) ((bIns) ? PHAL_MFDFEVX_CMD_READ_DATA_ISO : PHAL_MFDFEVX_CMD_READ_DATA);
        pAppData[wAppDataLen++] = bFileNo;

        (void) memcpy(&pAppData[wAppDataLen], pOffset, 3U);
        wAppDataLen += 3U;

        (void) memcpy(&pAppData[wAppDataLen], pLength, 3U);
        wAppDataLen += 3U;

        /* Get the TMI Status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

        /* Check TMI Collection Status */
        if(dwTMIStatus)
        {
            /* Frame the Option. */
            bTMIOption = (uint8_t) (dwLength ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

            /* Buffer the Command information to TMI buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                bTMIOption, pAppData, wAppDataLen, NULL, 0U, PHAL_MFDFEVX_BLOCK_SIZE));
        }
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLengthPresent,
        3,
        PH_ON,
        ((pLength[0U] | (pLength[1] << 8U) | (pLength[2U] << 8U)) + 7U),
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        wOption,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        ppResponse,
        pRespLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus && ((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)))
    {
        /* Frame the Option. */
        bTMIOption = (uint8_t) (dwLength ? 0U : PH_TMIUTILS_READ_INS);
        bTMIOption = (uint8_t) ((wStatus == PH_ERR_SUCCESS) ? (bTMIOption | PH_TMIUTILS_ZEROPAD_DATABUFF) : bTMIOption);

        /* Collect the data received. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            bTMIOption, NULL, 0U, *ppResponse, *pRespLen, PHAL_MFDFEVX_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(!dwLength && (wStatus == PH_ERR_SUCCESS))
        {
            /* Reset wOffsetInTMI */
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_SetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0U));
        }
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_WriteData(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint8_t		PH_MEMLOC_REM bOffset = 0;
    uint32_t	PH_MEMLOC_REM dwTotLen = 0;
    uint32_t	PH_MEMLOC_REM dwRemLen = 0;
    uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bIns > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }
    if((bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Compute the maximum length. */
    dwTotLen = (uint32_t) (pDataLen[0U] | (pDataLen[1] << 8U) | (pDataLen[2] << 16U));

    /* Update Offset information in case FULL Mode. */
    if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 12U : 8U);
        pAppData[0U] = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
            (pAppData[0U] + 2U) : pAppData[0U]);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = (uint8_t) (bIns ? PHAL_MFDFEVX_CMD_WRITE_DATA_ISO : PHAL_MFDFEVX_CMD_WRITE_DATA);
    pAppData[wAppDataLen++] = bFileNo;

    memcpy(&pAppData[wAppDataLen], pOffset, 3U);
    wAppDataLen += 3U;

    memcpy(&pAppData[wAppDataLen], pDataLen, 3U);
    wAppDataLen += 3U;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &pAppData[wAppDataOffset],
            (uint16_t) (wAppDataLen - wAppDataOffset), pData, dwTotLen, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Set the buffering flag. */
    wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

    /* Set the lengths. */
    dwRemLen = dwTotLen;

    do
    {
        /* Compute the maximum data to be exchanged.  */
        if(bIsFirstFrame)
        {
            bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 8) /* The command header information is available. */;
            bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 4U) : bDataLen);

            if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
            {
                bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 16U) : (bDataLen - 20U));
            }
        }
        else
        {
            if(bIns)
            {
                bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams)
                    ? (PHAL_MFDFEVX_MAX_WRITE_LEN - 5U) : PHAL_MFDFEVX_MAX_WRITE_LEN));
            }
            else
            {
                bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 1U);
                if(bOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
                {
                    bDataLen = (uint8_t) (((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
                        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)) ?
                        (bDataLen - 11U) : (bDataLen - 3U));
                }

                if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
                {
                    bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2) ? 16U : 32U);
                }
            }
        }

        /* Set the completion flag. */
        if(dwRemLen <= bDataLen)
        {
            bDataLen = (uint8_t) dwRemLen;
            wBufferOption = PH_EXCHANGE_DEFAULT;
            bExchangeComplete = PH_ON;
        }

        /* Copy the data to the buffer. */
        memcpy(&pAppData[wAppDataLen], &pData[bOffset], bDataLen);
        wAppDataLen += bDataLen;

        /* Wrap the command if required. */
        if(bIsFirstFrame || !bIns)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
                pDataParams,
                PH_ON,
                bLenPresent,
                1U,
                PH_OFF,
                (dwTotLen + 7U),
                pAppData,
                &wAppDataLen));
        }

        /* Frame the Crypto information. */
        bCrypto = (uint8_t) (bOption | ((bIns && (dwTotLen > PHAL_MFDFEVX_MAX_WRITE_LEN)) ?
            PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_ISO_CHAINING : PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_DESFIRE_CHAINING));

        /* Set the Offset information for MAC and FULL mode. */
        if(bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN)
        {
            bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : bIns ? 0 : 1);
            bCrypto |= (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams)) ? (bCrypto + 4U) : bCrypto);
            bCrypto = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
                (bCrypto + 2U) : bCrypto);

            /* Reset the Crypto to zero . */
            if(bIns && !bIsFirstFrame)
            {
                bCrypto = 0U;
            }
        }

        /* Set ExtendedOffset for Full mode and ISO Chaining. */
        if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) && bIsFirstFrame &&
            (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
        {
            /* Mask out the previously set offset. */
            bCrypto = (uint8_t) (bCrypto & 0xF0U);

            /* Set the Extended Offset bit. */
            bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
        }

        /* Clear the first frame and Length Presence flag. */
        bIsFirstFrame = PH_OFF;
        bLenPresent = PH_OFF;

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
            pDataParams,
            wBufferOption,
            bCrypto,
            pAppData,
            (uint8_t) wAppDataLen);

        /* Evaluate the response. */
        wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

        /* Return the status if not SUCCESS or SUCCESS_CHAINING. */
        if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            return wStatus;
        }

        /* Update length. */
        bOffset += bDataLen;
        dwRemLen = (uint32_t) (dwRemLen - bDataLen);

        /* Set the Chaining status. */
        wAppDataLen = 0U;

        if(!bIns)
        {
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
        }

    } while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_GetValue(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
    if((bFileNo & 0x7FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if((bCommOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = bCommOption;

    /* Append expected length to the command frame. */
    if(bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
    {
        pAppData[wAppDataLen++] = 4U;
        pAppData[wAppDataLen++] = 0U;
        pAppData[wAppDataLen++] = 0U;

        bLengthPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 3U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_GET_VALUE;
    pAppData[wAppDataLen++] = bFileNo;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_NO_PADDING, &pAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0U,
            PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLengthPresent,
        3U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

     /* Copy the value. */
    (void) memcpy(pValue, pResponse, wRespLen);

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, pValue, wRespLen, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Credit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if((bFileNo & 0x3FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if((bCommOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Update Offset information in case FULL Mode. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREDIT;
    pAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_DATABUFF, &pAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue, 4,
            PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Append data. */
    (void) memcpy(&pAppData[wAppDataLen], pValue, 4U);
    wAppDataLen += 4U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = bCommOption;

    /* Set ExtendedOffset for Full mode and ISO Chaining. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        /* Set the Extended Offset bit. */
        bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
    }
    else
    {
        bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4U /* Data Length */));
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Debit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if((bFileNo & 0x3FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        ((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        ((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Update Offset information in case FULL Mode. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_DEBIT;
    pAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_DATABUFF, &pAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue, 4U,
            PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Append data. */
    (void) memcpy(&pAppData[wAppDataLen], pValue, 4U);
    wAppDataLen += 4U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = bCommOption;

    /* Set ExtendedOffset for Full mode and ISO Chaining. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        /* Set the Extended Offset bit. */
        bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
    }
    else
    {
        bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4U /* Data Length */));
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_LimitedCredit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if((bFileNo & 0x3FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        ((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        ((bCommOption & 0x3FU) != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Update Offset information in case FULL Mode. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 6U : 2U);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_LIMITED_CREDIT;
    pAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_DATABUFF, &pAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue, 4U,
            PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Append data. */
    (void) memcpy(&pAppData[wAppDataLen], pValue, 4);
    wAppDataLen += 4U;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1U,
        PH_OFF,
        0U,
        pAppData,
        &wAppDataLen));

    /* Frame the Crypto information. */
    bCrypto = bCommOption;

    /* Set ExtendedOffset for Full mode and ISO Chaining. */
    if((bCommOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        /* Set the Extended Offset bit. */
        bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
    }
    else
    {
        bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4U /* Data Length */));
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ReadRecords(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pRecNo,
    uint8_t * pRecCount, uint8_t * pRecSize, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint16_t	PH_MEMLOC_REM wOption = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint32_t	PH_MEMLOC_REM dwLength = 0;
    uint32_t	PH_MEMLOC_REM dwNumRec = 0;
    uint32_t	PH_MEMLOC_REM dwRecLen = 0;
    uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameter. */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bIns > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        ((bOption & 0xF0U) != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT) && ((bOption & 0x0FU) != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = (uint8_t) (bOption & 0xF0U);

    /* Frame Option parameter. */
    wOption = (uint16_t) (bOption & 0x0FU);
    wOption |= (uint16_t) (bIns ? PHAL_MFDFEVX_SAM_X_ISO_CHAINING : 0U);

    /* Frame the command information based on the option. */
    if((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
        /* Frame additional frame code only for Native chaining. */
        if(!bIns)
        {
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
        }
    }
    else
    {
        /* Compute the length. */
        dwLength = pRecSize[2];
        dwLength = dwLength << 8U | pRecSize[1U];
        dwLength = dwLength << 8U | pRecSize[0U];

        /* Frame Presence of length information in the command frame.
         * The first three bytes specifies number of bytes to be received from PICC.
         */
        bLengthPresent = (uint8_t) (((bOption & 0xF0U) == PHAL_MFDFEVX_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

        /* Add Read Length if communication mode is Encrypted. */
        if(bLengthPresent)
        {
            (void) memcpy(&pAppData[wAppDataLen], pRecSize, 3U);
            wAppDataLen += 3U;
        }

        pAppData[wAppDataLen++] = (uint8_t) ((bIns) ? PHAL_MFDFEVX_CMD_READ_RECORDS_ISO : PHAL_MFDFEVX_CMD_READ_RECORDS);
        pAppData[wAppDataLen++] = bFileNo;

        (void) memcpy(&pAppData[wAppDataLen], pRecNo, 3U);
        wAppDataLen += 3U;

        (void) memcpy(&pAppData[wAppDataLen], pRecCount, 3U);
        wAppDataLen += 3U;

        /* Get the TMI Status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

        /* Check TMI Collection Status */
        if(dwTMIStatus)
        {
            /* Compute the number of records. */
            dwNumRec = pRecCount[2U];
            dwNumRec = dwNumRec << 8U | pRecCount[1U];
            dwNumRec = dwNumRec << 8U | pRecCount[0U];

            /* Compute the record length. */
            dwRecLen = pRecSize[2U];
            dwRecLen = dwRecLen << 8U | pRecSize[1U];
            dwRecLen = dwRecLen << 8U | pRecSize[0U];

            /* Should provide at least wRecLen / wNumRec to update in TIM collection */
            if(!dwRecLen && !dwNumRec)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
            }

            /* Buffer the Command information to TMI buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF), pAppData, wAppDataLen, NULL, 0U, PHAL_MFDFEVX_BLOCK_SIZE));
        }
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLengthPresent,
        3U,
        PH_ON,
        (pRecSize[0U] | (pRecSize[1U] << 8U) | (pRecSize[2U] << 8U)) + 7U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        wOption,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        ppResponse,
        pRespLen);

    /* Evaluate the response. */
    wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Frame the Option. */
        bTMIOption = (uint8_t) ((wStatus == PH_ERR_SUCCESS) ? PH_TMIUTILS_ZEROPAD_DATABUFF : PH_TMIUTILS_NO_PADDING);

        /* Collect the data received. */
        PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            bTMIOption, NULL, 0U, *ppResponse, *pRespLen, PHAL_MFDFEVX_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(!dwLength && (wStatus == PH_ERR_SUCCESS))
        {
            /* Reset wOffsetInTMI */
            PH_CHECK_SUCCESS_FCT(wStatus1, phTMIUtils_SetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0U));
        }
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_WriteRecord(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint8_t		PH_MEMLOC_REM bOffset = 0;
    uint32_t	PH_MEMLOC_REM dwTotLen = 0;
    uint32_t	PH_MEMLOC_REM dwRemLen = 0;
    uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bIns > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }
    if((bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Compute the maximum length. */
    dwTotLen = (uint32_t) (pDataLen[0U] | (pDataLen[1] << 8U) | (pDataLen[2U] << 16U));

    /* Update Offset information in case FULL Mode. */
    if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 12U : 8U);
        pAppData[0U] = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
            (pAppData[0U] + 2U) : pAppData[0U]);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = (uint8_t) (bIns ? PHAL_MFDFEVX_CMD_WRITE_RECORD_ISO : PHAL_MFDFEVX_CMD_WRITE_RECORD);
    pAppData[wAppDataLen++] = bFileNo;

    memcpy(&pAppData[wAppDataLen], pOffset, 3U);
    wAppDataLen += 3U;

    memcpy(&pAppData[wAppDataLen], pDataLen, 3U);
    wAppDataLen += 3U;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &pAppData[wAppDataOffset],
            (uint16_t) (wAppDataLen - wAppDataOffset), pData, dwTotLen, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Set the buffering flag. */
    wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

    /* Set the lengths. */
    dwRemLen = dwTotLen;

    do
    {
        /* Compute the maximum data to be exchanged.  */
        if(bIsFirstFrame)
        {
            bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 8) /* The command header information is available. */;
            bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 4U) : bDataLen);

            if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
            {
                bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 16U) : (bDataLen - 20U));
            }
        }
        else
        {
            if(bIns)
            {
                bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ?
                    (PHAL_MFDFEVX_MAX_WRITE_LEN - 5U) : PHAL_MFDFEVX_MAX_WRITE_LEN));
            }
            else
            {
                bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 1U);
                if(bOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
                {
                    bDataLen = (uint8_t) (((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
                        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)) ?
                        (bDataLen - 11U) : (bDataLen - 3U));
                }

                if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
                {
                    bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
                        ? 16U : 32U);
                }
            }
        }

        /* Set the completion flag. */
        if(dwRemLen <= bDataLen)
        {
            bDataLen = (uint8_t) dwRemLen;
            wBufferOption = PH_EXCHANGE_DEFAULT;
            bExchangeComplete = PH_ON;
        }

        /* Copy the data to the buffer. */
        memcpy(&pAppData[wAppDataLen], &pData[bOffset], bDataLen);
        wAppDataLen += bDataLen;

        /* Wrap the command if required. */
        if(bIsFirstFrame || !bIns)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
                pDataParams,
                PH_ON,
                bLenPresent,
                1U,
                PH_OFF,
                (dwTotLen + 7U),
                pAppData,
                &wAppDataLen));
        }

        /* Frame the Crypto information. */
        bCrypto = (uint8_t) (bOption | ((bIns && (dwTotLen > PHAL_MFDFEVX_MAX_WRITE_LEN)) ? PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_ISO_CHAINING :
            PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_DESFIRE_CHAINING));

        /* Set the Offset information for MAC and FULL mode. */
        if(bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN)
        {
            bCrypto |= (uint8_t) (bIsFirstFrame ? 8U : bIns ? 0U : 1U);
            bCrypto |= (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams)) ? (bCrypto + 4U) : bCrypto);
            bCrypto = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
                (bCrypto + 2U) : bCrypto);

            /* Reset the Crypto to zero . */
            if(bIns && !bIsFirstFrame)
            {
                bCrypto = 0U;
            }
        }

        /* Set ExtendedOffset for Full mode and ISO Chaining. */
        if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) && bIsFirstFrame &&
            (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
        {
            /* Mask out the previously set offset. */
            bCrypto = (uint8_t) (bCrypto & 0xF0U);

            /* Set the Extended Offset bit. */
            bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
        }

        /* Clear the first frame and Length Presence flag. */
        bIsFirstFrame = PH_OFF;
        bLenPresent = PH_OFF;

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
            pDataParams,
            wBufferOption,
            bCrypto,
            pAppData,
            (uint8_t) wAppDataLen);

        /* Evaluate the response. */
        wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

        /* Return the status if not SUCCESS or SUCCESS_CHAINING. */
        if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            return wStatus;
        }

        /* Update length. */
        bOffset += bDataLen;
        dwRemLen = (uint32_t) (dwRemLen - bDataLen);

        /* Set the Chaining status. */
        wAppDataLen = 0U;

        if(!bIns)
        {
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
        }

    } while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_UpdateRecord(void * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pRecNo,
    uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint8_t		PH_MEMLOC_REM bOffset = 0;
    uint32_t	PH_MEMLOC_REM dwTotLen = 0;
    uint32_t	PH_MEMLOC_REM dwRemLen = 0;
    uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
    if(((bFileNo & 0x7FU) > 0x1FU) || (bIns > 0x01U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }
    if((bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Compute the maximum length. */
    dwTotLen = (uint32_t) (pDataLen[0U] | (pDataLen[1U] << 8U) | (pDataLen[2U] << 16U));

    /* Update Offset information in case FULL Mode. */
    if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        pAppData[wAppDataLen++] = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? 15U : 11U);
        pAppData[0U] = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
            (pAppData[0U] + 2U) : pAppData[0U]);

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;

        /* Set the Application data offset. This should not go for TMI collection. */
        wAppDataOffset = 1U;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = (uint8_t) (bIns ? PHAL_MFDFEVX_CMD_UPDATE_RECORD_ISO : PHAL_MFDFEVX_CMD_UPDATE_RECORD);
    pAppData[wAppDataLen++] = bFileNo;

    memcpy(&pAppData[wAppDataLen], pRecNo, 3U);
    wAppDataLen += 3U;

    memcpy(&pAppData[wAppDataLen], pOffset, 3U);
    wAppDataLen += 3U;

    memcpy(&pAppData[wAppDataLen], pDataLen, 3U);
    wAppDataLen += 3U;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &pAppData[wAppDataOffset],
            (uint16_t) (wAppDataLen - wAppDataOffset), pData, dwTotLen, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Set the buffering flag. */
    wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

    /* Set the lengths. */
    dwRemLen = dwTotLen;

    do
    {
        /* Compute the maximum data to be exchanged.  */
        if(bIsFirstFrame)
        {
            bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 11U) /* The command header information is available. */;
            bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 4U) : bDataLen);

            if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
            {
                bDataLen = (uint8_t) ((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) ? (bDataLen - 16U) : (bDataLen - 20U));
            }
        }
        else
        {
            if(bIns)
            {
                bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ?
                    (PHAL_MFDFEVX_MAX_WRITE_LEN - 5U) : PHAL_MFDFEVX_MAX_WRITE_LEN));
            }
            else
            {
                bDataLen = (uint8_t) (PHAL_MFDFEVX_MAX_WRITE_LEN - 1U);
                if(bOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
                {
                    bDataLen = (uint8_t) (((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEAES) ||
                        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)) ?
                        (bDataLen - 11U) : (bDataLen - 3U));
                }

                if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams))
                {
                    bDataLen = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
                        ? 16U : 32U);
                }
            }
        }

        /* Set the completion flag. */
        if(dwRemLen <= bDataLen)
        {
            bDataLen = (uint8_t) dwRemLen;
            wBufferOption = PH_EXCHANGE_DEFAULT;
            bExchangeComplete = PH_ON;
        }

        /* Copy the data to the buffer. */
        memcpy(&pAppData[wAppDataLen], &pData[bOffset], bDataLen);
        wAppDataLen += bDataLen;

        /* Wrap the command if required. */
        if(bIsFirstFrame || !bIns)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
                pDataParams,
                PH_ON,
                bLenPresent,
                1U,
                PH_OFF,
                (dwTotLen + 10U),
                pAppData,
                &wAppDataLen));
        }

        /* Frame the Crypto information. */
        bCrypto = (uint8_t) (bOption | ((bIns && (dwTotLen > PHAL_MFDFEVX_MAX_WRITE_LEN)) ?
            PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_ISO_CHAINING : PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_DESFIRE_CHAINING));

        /* Set the Offset information for MAC and FULL mode. */
        if(bOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN)
        {
            bCrypto |= (uint8_t) (bIsFirstFrame ? 11 : bIns ? 0 : 1);
            bCrypto |= (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams)) ? (bCrypto + 4U) : bCrypto);
            bCrypto = (uint8_t) ((phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (bIns && (dwTotLen > 0xFEU))) ?
                (bCrypto + 2U) : bCrypto);

            /* Reset the Crypto to zero . */
            if(bIns && !bIsFirstFrame)
            {
                bCrypto = 0U;
            }
        }

        /* Set ExtendedOffset for Full mode and ISO Chaining. */
        if((bOption == PHAL_MFDFEVX_COMMUNICATION_ENC) && bIsFirstFrame &&
            (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
        {
            /* Mask out the previously set offset. */
            bCrypto = (uint8_t) (bCrypto & 0xF0U);

            /* Set the Extended Offset bit. */
            bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;
        }

        /* Clear the first frame and Length Presence flag. */
        bIsFirstFrame = PH_OFF;
        bLenPresent = PH_OFF;

        /* Exchange the information to Sam. */
        wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
            pDataParams,
            wBufferOption,
            bCrypto,
            pAppData,
            (uint8_t) wAppDataLen);

        /* Evaluate the response. */
        wStatus = phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
            PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

        /* Return the status if not SUCCESS or SUCCESS_CHAINING. */
        if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            return wStatus;
        }

        /* Update length. */
        bOffset += bDataLen;
        dwRemLen = (uint32_t) (dwRemLen - bDataLen);

        /* Set the Chaining stratus. */
        wAppDataLen = 0U;

        if(!bIns)
        {
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
        }

    } while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t    PH_MEMLOC_REM wAppDataSize = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
    if((bFileNo & 0x7FU) > 0x1FU)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0U;
    (void) memset(pAppData, 0x00U, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CLEAR_RECORDS_FILE;
    pAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_CMDBUFF, pAppData, 2, NULL, 0, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0U,
        PH_ON,
        0U,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx Transaction management commands. -------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_CommitTransaction(void * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
    if(bOption > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_COMMIT_TXN);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_COMMIT_TXN;

    /* Add Optionif required. */
    if(bOption)
    {
        pAppData[wAppDataLen++] = bOption;
    }

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0,
        PH_ON,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Copy the data to the parameter */
    if(bOption)
    {
        (void) memcpy(pTMC, &pResponse[0], 4);
        (void) memcpy(pTMV, &pResponse[4], 8);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_AbortTransaction(void * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_MAC;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ABORT_TXN);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ABORT_TXN;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0,
        PH_ON,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CommitReaderID(void * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bISOMode = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t	PH_MEMLOC_REM wEncTMRILen = 0;
    uint8_t		PH_MEMLOC_REM aPiccErrCode[2];
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;
    uint8_t *   PH_MEMLOC_REM pSamUid = NULL;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_COMMIT_READER_ID);

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Set Native or ISO7816 PICC command format. */
    bISOMode = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? PHAL_MFDFEVX_SAM_X_ISO_MODE_ISO7816 :
        PHAL_MFDFEVX_SAM_X_ISO_MODE_NATIVE);

    /* Exchange the details to SAM hardware and get the ENC TMRI. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_NOT_AUTHENTICATED)
    {
        /* Exchange the command to Sam hardware .*/
        wStatus = phalMfdfEVx_Sam_X_Int_TMRI_CommitReaderID(
            pDataParams,
            bISOMode,
            0x01U,
            0,
            &pResponse,
            &wEncTMRILen,
            aPiccErrCode);

        /* Copy the Response to the parameter. */
        (void) memcpy(pEncTMRI, pResponse, wEncTMRILen);
        wRespLen = wEncTMRILen;

        /* Check TMI Collection Status */
        if(dwTMIStatus && (wStatus == PH_ERR_SUCCESS))
        {
            /* If authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
             * TMI = TMI || Cmd || TMRICur || EncTMRI || ZeroPadding
             */

            /* Frame the command and send it to card. */
            pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_COMMIT_READER_ID;

            /* Frame the TMRI information. */
            phalMfdfEVx_Sam_X_Int_GetSAMUID(pDataParams, &pSamUid);
            (void) memcpy(&pAppData[wAppDataLen], pSamUid, 7);
            wAppDataLen += 7;

            pAppData[wAppDataLen] = 0x80;

            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                PH_TMIUTILS_ZEROPAD_DATABUFF, pAppData, 17, pEncTMRI, 16, PHAL_MFDFEVX_BLOCK_SIZE));
        }
    }
    else
    {
        /* Frame the command and send it to card. */
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_COMMIT_READER_ID;

        /* Add the TMRI to command buffer. */
        (void) memcpy(&pAppData[wAppDataLen], pTMRI, 16);
        wAppDataLen += 16;

        /* Check TMI Collection Status */
        if(dwTMIStatus)
        {
            /* If not authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
             * TMI = TMI || Cmd || TMRICur || ZeroPadding
             */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
                PH_TMIUTILS_ZEROPAD_CMDBUFF, pAppData, wAppDataLen, NULL, 0, PHAL_MFDFEVX_BLOCK_SIZE));
        }

        /* Wrap the command if required. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
            pDataParams,
            PH_ON,
            PH_OFF,
            0,
            PH_ON,
            (uint16_t) (wAppDataLen - 1),
            pAppData,
            &wAppDataLen));

        /* Exchange the data to SAM. */
        wStatus = phalMfdfEVx_Sam_X_Int_ISO14443_4_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            pAppData,
            (uint8_t) wAppDataLen,
            &pResponse,
            &wRespLen);

        /* Set the PICC error code. */
        aPiccErrCode[0] = pResponse[wRespLen];
        aPiccErrCode[1] = pResponse[wRespLen - 1];

        /* Manipulate the status. */
        if((pResponse[wRespLen] != 0x91) && (pResponse[wRespLen - 1] != 0x00))
        {
            wStatus = PH_ADD_COMPCODE(PHAL_MFDFEVX_SAM_X_ERR_MIFARE_GEN, PH_COMP_HAL);
        }

        /* Decrement the RespLen to remove the status. */
        wRespLen -= 2;
    }

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus, aPiccErrCode));

    /* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the received TMRI */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_SetConfig(pDataParams, PHAL_MFDFEVX_ADDITIONAL_INFO, wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx ISO7816-4 commands. ---------------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid,
    uint8_t * pDFname, uint8_t bDFnameLen, uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM aFileId[3] = { '\0' };
    uint16_t    PH_MEMLOC_REM wVal = 0;
    uint8_t		PH_MEMLOC_REM aPiccDfName[7] = { 0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00 };

    /* Validate the parameters. */
    if((bDFnameLen > 16) || ((bOption != 0x00) && (bOption != 0x0C)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(bSelector > 0x04)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ISO7816_SELECT_FILE);

    /* Frame the command. */
    pAppData[wAppDataLen++] = 0x00;									/* CLA */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ISO7816_SELECT_FILE; /* INS */
    pAppData[wAppDataLen++] = bSelector;							/* P1 */
    pAppData[wAppDataLen++] = bOption;								/* P2 */

    /* Append LC. */
    if(bExtendedLenApdu)
    {
        pAppData[wAppDataLen++] = 0;
        pAppData[wAppDataLen++] = 0;
    }

    /* Append the payload and LC. */
    if(bSelector == 0x04)
    {
        /* Append LC. */
        pAppData[wAppDataLen++] = bDFnameLen;

        (void) memcpy(&pAppData[wAppDataLen], pDFname, bDFnameLen);
        wAppDataLen += bDFnameLen;
    }
    else
    {
        /* Append LC. */
        pAppData[wAppDataLen++] = 2;

        /* Select MF, DF or EF, by file identifier
         * Select child DF
         * Select EF under the current DF, by file identifier
         * Select parent DF of the current DF
         */
        aFileId[1] = pAppData[wAppDataLen++] = pFid[1];
        aFileId[0] = pAppData[wAppDataLen++] = pFid[0];
        aFileId[2] = 0;
    }

    /* Append LE. */
    if(bExtendedLenApdu)
    {
        pAppData[wAppDataLen++] = 0;
    }
    pAppData[wAppDataLen++] = 0;

    /* Exchange the data to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_ISO14443_4_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        pAppData,
        (uint8_t) wAppDataLen,
        &pResponse,
        &wRespLen);

    /* Convert the PICC status and validate it. */
    wStatus = (phStatus_t) ((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]);
    wStatus = phalMfdfEVx_Int_ComputeErrorResponse(pDataParams, wStatus);

    if((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_ERR_DF_7816_GEN_ERROR)
    {
        wVal = phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams);
    }

    if((wStatus == PH_ERR_SUCCESS) || (wVal == PHAL_MFDFEVX_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS))
    {
        /* Reset Authentication should not be targeted for elementary file selection using file ID */
        if(bSelector != 0x02)
        {
            /* Reset Authentication Status here */
            phalMfdfEVx_Sam_X_ResetAuthStatus(pDataParams);
        }

        /* ISO wrapped mode is on */
        phalMfdfEVx_Sam_X_Int_SetWrappedMode(pDataParams, PH_ON);

        /* once the selection Success, update the File Id to master data structure if the selection is done through AID */
        if((bSelector == 0x00) || (bSelector == 0x01) || (bSelector == 0x02))
        {
            phalMfdfEVx_Sam_X_Int_SetAID(pDataParams, aFileId);
        }
        else if((bSelector == 0x04))
        {
            /* Update the file ID to all zeros if DF Name is of PICC. */
            if(memcmp(pDFname, aPiccDfName, 7) == 0)
            {
                aFileId[0] = 0x00;
                aFileId[1] = 0x00;
                aFileId[2] = 0x00;
            }
            else
            {
                aFileId[0] = 0xFFU;
                aFileId[1] = 0xFFU;
                aFileId[2] = 0xFFU;
            }

            phalMfdfEVx_Sam_X_Int_SetAID(pDataParams, aFileId);
        }
        else
        {
            /* Nothing for Selector 0x03. */
        }
    }
    else
    {
        return wStatus;
    }

    /* Copy the response to the buffer */
    *ppFCI = pResponse;
    *pFCILen = wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid,
    uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pBytesRead)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Validate the parameter. */
    if(wOption == PH_EXCHANGE_DEFAULT)
    {
        if(bSfid & 0x80)
        {
            /* Short file id is supplied */
            if((bSfid & 0x7FU) > 0x1F)
            {
                /* Error condition */
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
            }
        }
    }

    if((wOption != PH_EXCHANGE_DEFAULT) && (wOption != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ISO7816_READ_BINARY);

    /* Frame the command information based on the option. */
    if(wOption == PH_EXCHANGE_DEFAULT)
    {
        pAppData[wAppDataLen++] = 0x00;
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ISO7816_READ_BINARY;
        pAppData[wAppDataLen++] = bSfid;
        pAppData[wAppDataLen++] = bOffset;

        if(bExtendedLenApdu)
        {
            pAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x00FF0000) >> 16);
            pAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x0000FF00) >> 8);
        }
        pAppData[wAppDataLen++] = (uint8_t) (dwBytesToRead & 0x000000FF);
    }
    else
    {
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_ISO14443_4_Exchange(
        pDataParams,
        wOption,
        pAppData,
        (uint8_t) wAppDataLen,
        ppResponse,
        pBytesRead);

    /* Update the chaining status code with DESFire component. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDFEVX);
    }

    /* Decrement the Length if its success. */
    if(wStatus == PH_ERR_SUCCESS)
        *pBytesRead -= 2;

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint32_t dwDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;

    /* Validate the parameters */
    if(bSfid & 0x80)
    {
        /* Short file id is supplied */
        if((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ISO7816_UPDATE_BINARY);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = 0x00;
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ISO7816_UPDATE_BINARY;
    pAppData[wAppDataLen++] = bSfid;
    pAppData[wAppDataLen++] = bOffset;

    if(bExtendedLenApdu)
    {
        pAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x00FF0000) >> 16);
        pAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x0000FF00) >> 8);
    }
    pAppData[wAppDataLen++] = (uint8_t) (dwDataLen & 0x000000FF);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    /* Exchange the information to Sam. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen));

    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        bCrypto,
        pData,
        (uint8_t) dwDataLen);


    /* Convert the PICC status and validate it. */
    if((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_DESFIRE_GEN)
    {
        memcpy(&wPiccErrCode, PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 2U);
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_IsoReadRecords(void * pDataParams, uint16_t wOption, uint8_t bRecNo, uint8_t bReadAllFromP1,
    uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pBytesRead)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;

    /* Validate the parameter. */
    if(wOption == PH_EXCHANGE_DEFAULT)
    {
        if(bSfid > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
        }
    }

    if((wOption != PH_EXCHANGE_DEFAULT) && (wOption != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ISO7816_READ_RECORDS);

    /* Frame the command information based on the option. */
    if(wOption == PH_EXCHANGE_DEFAULT)
    {
        pAppData[wAppDataLen++] = 0x00;
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ISO7816_READ_RECORDS;
        pAppData[wAppDataLen++] = bRecNo;
        pAppData[wAppDataLen++] = (uint8_t) ((bSfid <<= 3) | (bReadAllFromP1 ? 0x05 : 0x04));

        if(bExtendedLenApdu)
        {
            pAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x00FF0000) >> 16);
            pAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x0000FF00) >> 8);
        }
        pAppData[wAppDataLen++] = (uint8_t) (dwBytesToRead & 0x000000FF);
    }
    else
    {
        pAppData[wAppDataLen++] = PHAL_MFDFEVX_RESP_ADDITIONAL_FRAME;
    }

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_ISO14443_4_Exchange(
        pDataParams,
        wOption,
        pAppData,
        (uint8_t) wAppDataLen,
        ppResponse,
        pBytesRead);

    /* Update the chaining status code with DESFire component. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDFEVX);
    }

    /* Decrement the Length if its success. */
    if(wStatus == PH_ERR_SUCCESS)
        *pBytesRead -= 2;

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_IsoAppendRecord(void * pDataParams, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData,
    uint32_t dwDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;

    /* Short file id is supplied */
    if((bSfid & 0x7FU) > 0x1F)
    {
        /* Error condition */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_ISO7816_APPEND_RECORD);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = 0x00;
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_ISO7816_APPEND_RECORD;
    pAppData[wAppDataLen++] = 0x00;
    pAppData[wAppDataLen++] = (uint8_t) (bSfid << 3);

    if(bExtendedLenApdu)
    {
        pAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x00FF0000) >> 16);
        pAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x0000FF00) >> 8);
    }
    pAppData[wAppDataLen++] = (uint8_t) (dwDataLen & 0x000000FF);

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;

    /* Exchange the information to Sam. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen));

    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        bCrypto,
        pData,
        (uint8_t) dwDataLen);

    /* Convert the PICC status and validate it. */
    if((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_DESFIRE_GEN)
    {
        memcpy(&wPiccErrCode, PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 2U);
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
    }

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_IsoGetChallenge(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint32_t dwLe,
    uint8_t * pRPICC1)
{
    pRPICC1 = NULL;
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDFEVX_UNUSED_VARIABLE(wKeyNo);
    PHAL_MFDFEVX_UNUSED_VARIABLE(wKeyVer);
    PHAL_MFDFEVX_UNUSED_VARIABLE(dwLe);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pRPICC1);

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_IsoExternalAuthenticate(void * pDataParams, uint8_t * pDataIn, uint8_t bInputLen,
    uint8_t * pDataOut, uint8_t * pOutLen)
{
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataIn);
    PHAL_MFDFEVX_UNUSED_VARIABLE(bInputLen);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataOut);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pOutLen);

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_IsoInternalAuthenticate(void * pDataParams, uint8_t * pDataIn, uint8_t bInputLen,
    uint8_t * pDataOut, uint8_t * pOutLen)
{
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataParams);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataIn);
    PHAL_MFDFEVX_UNUSED_VARIABLE(bInputLen);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pDataOut);
    PHAL_MFDFEVX_UNUSED_VARIABLE(pOutLen);

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);

    /* This command is available only in AV1 Mode */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_IsoAuthenticate(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard,
    uint8_t bIsPICCkey)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bOption = 0;
    uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;

    /* Check for valid card key number. */
    if(bKeyNoCard > 0x0d)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Check for valid SAM KeyStore number and version. */
    if(IS_VALID_SAM_KEYS(wKeyNo) || (wKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_AUTHENTICATE_ISO);

    /* Unset the current authentication status. */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_NOT_AUTHENTICATED);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Frame the Option parameter. */
    bOption = (uint8_t) (bIsPICCkey ? 0x02U : 0x00);

    /* Exchange the information to SAM. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(
        pDataParams,
        bOption,
        0x80U,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        0,
        NULL,
        NULL,
        0,
        NULL,
        NULL,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams));

    /* Convert the PICC status and validate it. */
    if((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_DESFIRE_GEN)
    {
        memcpy(&wPiccErrCode, PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams), 2U);
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Int_ComputeErrorResponse(pDataParams, wPiccErrCode));
    }

    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, PHAL_MFDFEVX_AUTHENTICATEISO);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, bKeyNoCard);
    phalMfdfEVx_Sam_X_Int_SetWrappedMode(pDataParams, PH_ON);

    return wStatus;
}




/* MIFARE DESFire EVx Originality Check functions. ------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM bLengthPresent = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL;
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_NOT_AUTHENTICATED)
    {
        bCrypto = PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_PLAIN;
    }

    /* Add Length information is Communication mode is FULL. */
    if(bCrypto == PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_COMM_MODE_FULL)
    {
        pAppData[wAppDataLen++] = 56;
        pAppData[wAppDataLen++] = 0;
        pAppData[wAppDataLen++] = 0;

        /* Set presence of Length information. */
        bLengthPresent = PH_ON;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_READ_SIG);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_READ_SIG;
    pAppData[wAppDataLen++] = bAddr;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLengthPresent,
        3,
        PH_ON,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_ReadX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen,
        pSignature,
        &wRespLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}




/* MIFARE DESFire EVx MIFARE Classic functions. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_CreateMFCMapping(void * pDataParams, uint8_t bComOption, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pMFCBlockList, uint8_t bMFCBlocksLen, uint8_t bRestoreSource, uint8_t * pMFCLicense, uint8_t bMFCLicenseLen,
    uint8_t * pMFCLicenseMAC)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;

    /* Validate the parameters. */
    if((bComOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bComOption != PHAL_MFDFEVX_COMMUNICATION_ENC))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = bComOption;
    bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_ISO_CHAINING;

    /* Set the Command Offset. */
    if((bComOption == PHAL_MFDFEVX_COMMUNICATION_ENC) &&
        (phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) == PHAL_MFDFEVX_AUTHENTICATEEV2))
    {
        /* Start of data. */
        pAppData[wAppDataLen++] = 0;

        /* Set Extended Offset in Crypto mode. */
        bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;

        /* Set Length Presence flag. */
        bLenPresent = PH_ON;
    }

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_CREATE_MFC_MAPPING);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_CREATE_MFC_MAPPING;
    pAppData[wAppDataLen++] = bFileNo;
    pAppData[wAppDataLen++] = bFileOption;
    pAppData[wAppDataLen++] = bMFCBlocksLen;

    /* Copy the MFCBlockList to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCBlockList, bMFCBlocksLen);
    wAppDataLen += bMFCBlocksLen;

    /* Copy RestoreSource to command buffer. */
    if(bFileOption & 0x04)
    {
        pAppData[wAppDataLen++] = bRestoreSource;
    }

    /* Copy the MFCLicense to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCLicense, bMFCLicenseLen);
    wAppDataLen += bMFCLicenseLen;

    /* Compute the Offset. */
    bOffset = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? (wAppDataLen + 4) : wAppDataLen);
    bOffset = (uint8_t) (bOffset - 1);

    /* Reset the Command Offset. */
    if(bComOption == PHAL_MFDFEVX_COMMUNICATION_ENC)
        pAppData[0] = bOffset;

    /* Copy the MFCLicenseMac to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCLicenseMAC, 8);
    wAppDataLen += 8;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1,
        PH_OFF,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCrypto,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_RestoreTransfer(void * pDataParams, uint8_t bCommOption, uint8_t bTargetFileNo,
    uint8_t bSourceFileNo)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t *	PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
    if((bCommOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_PLAIN_1) &&
        (bCommOption != PHAL_MFDFEVX_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_RESTORE_TRANSFER);

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_RESTORE_TRANSFER;
    pAppData[wAppDataLen++] = bTargetFileNo;
    pAppData[wAppDataLen++] = bSourceFileNo;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        PH_OFF,
        0,
        PH_ON,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        bCommOption,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    /* Get the status of the TMI */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFEVX_RESOLVE_TMI_DATAPARAMS(pDataParams),
            PH_TMIUTILS_ZEROPAD_CMDBUFF, pAppData, 3, NULL, 0, PHAL_MFDFEVX_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_RestrictMFCUpdate(void * pDataParams, uint8_t bOption, uint8_t * pMFCConfig,
    uint8_t bMFCConfigLen, uint8_t * pMFCLicense, uint8_t bMFCLicenseLen, uint8_t * pMFCLicenseMAC)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;
    uint8_t *   PH_MEMLOC_REM pAppData = NULL;
    uint16_t	PH_MEMLOC_REM wAppDataSize = 0;
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t		PH_MEMLOC_REM bLenPresent = 0;

    /* Validate Authentication. */
    if(phalMfdfEVx_Sam_X_Int_GetAuthMode(pDataParams) != PHAL_MFDFEVX_AUTHENTICATEEV2)
        return PH_ADD_COMPCODE(PH_ERR_AUTH_NOT_SUPPORTED, PH_COMP_AL_MFDFEVX);

    /* Get the Global parameters. */
    pAppData = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    wAppDataSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the AppData buffer and length. */
    wAppDataLen = 0;
    (void) memset(pAppData, 0x00, wAppDataSize * sizeof(uint8_t));

    /* Frame the Crypto information. */
    bCrypto = PHAL_MFDFEVX_COMMUNICATION_ENC;
    bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_ISO_CHAINING;
    bCrypto |= PHAL_MFDFEVX_SAM_X_CRYPTO_CONFIG_EXTENDED_OFFSET;

    /* Set Length Presence flag. */
    bLenPresent = PH_ON;

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_RESTRICT_MFC_UPDATE);

    /* Start of data. */
    pAppData[wAppDataLen++] = 0;

    /* Frame the command information. */
    pAppData[wAppDataLen++] = PHAL_MFDFEVX_CMD_RESTRICT_MFC_UPDATE;
    pAppData[wAppDataLen++] = bOption;

    /* Copy the MFCConfig to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCConfig, bMFCConfigLen);
    wAppDataLen += bMFCConfigLen;

    /* Copy the MFCLicense to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCLicense, bMFCLicenseLen);
    wAppDataLen += bMFCLicenseLen;

    /* Compute the Offset. */
    bOffset = (uint8_t) (phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) ? (wAppDataLen + 4) : wAppDataLen);
    bOffset = (uint8_t) (bOffset - 1);

    /* Reset the Command Offset. */
    pAppData[0] = bOffset;

    /* Copy the MFCLicenseMac to command buffer. */
    (void) memcpy(&pAppData[wAppDataLen], pMFCLicenseMAC, 8);
    wAppDataLen += 8;

    /* Wrap the command if required. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_Iso7816Wrap(
        pDataParams,
        PH_ON,
        bLenPresent,
        1,
        PH_OFF,
        0,
        pAppData,
        &wAppDataLen));

    /* Exchange the information to Sam. */
    wStatus = phalMfdfEVx_Sam_X_Int_DESFire_WriteX(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDFEVX_COMMUNICATION_ENC,
        pAppData,
        (uint8_t) wAppDataLen);

    /* Evaluate the response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus,
        PHAL_MFDFEVX_RESOLVE_PICC_ERR_CODE(pDataParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}



/* MIFARE DESFire EVx POST Delivery Configuration function. ---------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_AuthenticatePDC(void * pDataParams, uint8_t bRfu, uint8_t bKeyNoCard, uint16_t wKeyNum,
    uint16_t wKeyVer, uint8_t bUpgradeInfo)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

    PH_UNUSED_VARIABLE(bRfu);

    /* Set the DataParams with command code. */
    phalMfdfEVx_Sam_X_Int_SetCmdCode(pDataParams, PHAL_MFDFEVX_CMD_AUTH_PDC);

    wStatus = phhalHw_SamAV3_Cmd_PDC_Authenticate(
        PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHAL_MFDFEVX_SAM_X_PDC_AUTH_DERIVE_UPGRADE_KEY,
        (uint8_t) wKeyNum,
        (uint8_t) wKeyVer,
        bKeyNoCard,
        &bUpgradeInfo,
        1,
        NULL,
        0,
        &bPiccErrCode);

    /* Return the error code. */
    if((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_MIFARE_GEN)
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ValidateResponse(pDataParams, wStatus, &bPiccErrCode));
    }
    else
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}




/* MIFARE DESFire EVx Miscellaneous functions. ----------------------------------------------------------------------------------------- */
phStatus_t phalMfdfEVx_Sam_X_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PHAL_MFDFEVX_ADDITIONAL_INFO:
            *pValue = phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(pDataParams);
            break;

        case PHAL_MFDFEVX_WRAPPED_MODE:
            *pValue = phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams);
            break;

        case PHAL_MFDFEVX_RETURN_FAB_ID:
            *pValue = phalMfdfEVx_Sam_X_Int_GetFabID(pDataParams);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PHAL_MFDFEVX_ADDITIONAL_INFO:
            phalMfdfEVx_Sam_X_Int_SetAdditionalInfo(pDataParams, wValue);
            break;

        case PHAL_MFDFEVX_WRAPPED_MODE:
            phalMfdfEVx_Sam_X_Int_SetWrappedMode(pDataParams, (uint8_t) wValue);
            break;

        case PHAL_MFDFEVX_RETURN_FAB_ID:
            phalMfdfEVx_Sam_X_Int_SetFabID(pDataParams, (uint8_t) wValue);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ResetAuthStatus(void * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset the AuthMode and Key number */
    phalMfdfEVx_Sam_X_Int_SetAuthMode(pDataParams, 0xFFU);
    phalMfdfEVx_Sam_X_Int_SetKeyNo(pDataParams, 0xFFU);

    /* Kill the PICC Authentication. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_KillAuthentication(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_KILL_AUTHENTICATION_PARTIAL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GenerateDAMEncKey(void * pDataParams, uint16_t wKeyNoDAMEnc, uint16_t wKeyVerDAMEnc,
    uint16_t wKeyNoAppDAMDefault, uint16_t wKeyVerAppDAMDefault, uint8_t bAppDAMDefaultKeyVer, uint8_t * pDAMEncKey)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint16_t	PH_MEMLOC_REM wKeyLen = 0;
    uint8_t	*	PH_MEMLOC_REM pInputBuff = NULL;
    uint8_t		PH_MEMLOC_REM bInputBuffLen = 0;
    uint16_t	PH_MEMLOC_REM bInputBuffSize = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint16_t	PH_MEMLOC_REM wSET = 0;
    uint16_t	PH_MEMLOC_REM wExtSET = 0;
    uint16_t	PH_MEMLOC_REM wKeyType = 0;

    /* Validate the key information. */
    if((IS_VALID_SAM_KEYS(wKeyNoDAMEnc) || (wKeyVerDAMEnc > 0xFFU)) ||
        IS_VALID_SAM_KEYS(wKeyNoAppDAMDefault) ||
        (IS_VALID_SAM_KEYS(wKeyVerAppDAMDefault) || (bAppDAMDefaultKeyVer > 0xFFU)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the KeyInformation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_GetKeyInfo(
        pDataParams,
        (uint8_t) wKeyNoAppDAMDefault,
        &wKeyType,
        &wSET,
        &wExtSET));

    /* Validate the KeyType. */
    if((wKeyType != 0x0030) && (wKeyType != 0x0020) &&
        (wKeyType != 0x0018) && (wKeyType != 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDFEVX);
    }

    /* Check if DumpSecretKey is enabled. */
    if(!(wExtSET & 0x0008))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFEVX);
    }

    /* Get the KeyInformation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_GetKeyInfo(
        pDataParams,
        (uint8_t) wKeyNoDAMEnc,
        &wKeyType,
        &wSET,
        &wExtSET));

    /* Validate the KeyType. */
    if((wKeyType != 0x0030) && (wKeyType != 0x0020) &&
        (wKeyType != 0x0018) && (wKeyType != 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pInputBuff = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    bInputBuffSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the command buffer and length. */
    bInputBuffLen = 0;
    (void) memset(pInputBuff, 0x00, bInputBuffSize * sizeof(uint8_t));

    /* Add the Random number. */
    bInputBuffLen = 7;
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_GetRandom(
        pDataParams,
        bInputBuffLen,
        pInputBuff));

    /* Append the Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DumpSecretKey(
        pDataParams,
        0x00,
        (uint8_t) wKeyNoAppDAMDefault,
        (uint8_t) wKeyVerAppDAMDefault,
        NULL,
        0,
        &pInputBuff[bInputBuffLen],
        &wKeyLen));

    bInputBuffLen += (uint8_t) wKeyLen;

    /* Append the key version. */
    pInputBuff[bInputBuffLen] = bAppDAMDefaultKeyVer;

    /* Set the Input length to 32 bytes default. */
    bInputBuffLen = 32;

    /* Validate the KeyType. */
    if((wKeyType != 0x0030) && (wKeyType != 0x0020) &&
        (wKeyType != 0x0018) && (wKeyType != 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDFEVX);
    }

    /* Perform Offline activation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wKeyNoDAMEnc,
        (uint8_t) wKeyVerDAMEnc,
        NULL,
        0));

    /* Encrypt the Plain data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_EncipherOfflineData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        pInputBuff,
        bInputBuffLen,
        &pResponse,
        &wRespLen));

    /* Copy the response to the parameter. */
    (void) memcpy(pDAMEncKey, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_GenerateDAMMAC(void * pDataParams, uint8_t bOption, uint16_t wKeyNoDAMMAC, uint16_t wKeyVerDAMMAC,
    uint8_t * pAid, uint8_t * pDamParams, uint8_t bKeySettings1, uint8_t bKeySettings2, uint8_t bKeySettings3,
    uint8_t * pKeySetValues, uint8_t * pISOFileId, uint8_t * pISODFName, uint8_t bISODFNameLen,
    uint8_t * pEncK, uint8_t * pDAMMAC)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t	*	PH_MEMLOC_REM pInputBuff = NULL;
    uint8_t		PH_MEMLOC_REM bInputBuffLen = 0;
    uint16_t	PH_MEMLOC_REM bInputBuffSize = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint16_t	PH_MEMLOC_REM wSET = 0;
    uint16_t	PH_MEMLOC_REM wExtSET = 0;
    uint16_t	PH_MEMLOC_REM wKeyType = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wKeyNoDAMMAC) || (wKeyVerDAMMAC > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Get the KeyInformation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_GetKeyInfo(
        pDataParams,
        (uint8_t) wKeyNoDAMMAC,
        &wKeyType,
        &wSET,
        &wExtSET));

    /* Validate the KeyType. */
    if((wKeyType != 0x0030) && (wKeyType != 0x0020) &&
        (wKeyType != 0x0018) && (wKeyType != 0x0000))
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFDFEVX);
    }

    /* Get the Global parameters. */
    pInputBuff = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER(pDataParams);
    bInputBuffSize = PHAL_MFDFEVX_RESOLVE_TEMPORARY_BUFFER_SIZE(pDataParams);

    /* Clear the command buffer and length. */
    bInputBuffLen = 0;
    (void) memset(pInputBuff, 0x00, bInputBuffSize * sizeof(uint8_t));

    /* Frame the Input */
    pInputBuff[bInputBuffLen++] = PHAL_MFDFEVX_CMD_CREATE_DELEGATED_APPLN;

    if((bOption & PHAL_MFDFEVX_GENERATE_DAMMAC_DELETE_APPLICATION) == PHAL_MFDFEVX_GENERATE_DAMMAC_DELETE_APPLICATION)
        pInputBuff[0] = PHAL_MFDFEVX_CMD_DELETE_APPLN;

    /* Append Application Identifier */
    (void) memcpy(&pInputBuff[bInputBuffLen], pAid, 3);
    bInputBuffLen += 3;

    if(!(bOption & PHAL_MFDFEVX_GENERATE_DAMMAC_DELETE_APPLICATION))
    {
        /* Append DAMParams */
        (void) memcpy(&pInputBuff[bInputBuffLen], pDamParams, 5);
        bInputBuffLen += 5;

        /* Append KeySetting Information */
        pInputBuff[bInputBuffLen++] = bKeySettings1;
        pInputBuff[bInputBuffLen++] = bKeySettings2;
        if(bKeySettings2 & PHAL_MFDFEVX_KEYSETT3_PRESENT)
        {
            pInputBuff[bInputBuffLen++] = bKeySettings3;
            if(bKeySettings3 & PHAL_MFDFEVX_KEYSETVALUES_PRESENT && pKeySetValues != NULL)
            {
                (void) memcpy(&pInputBuff[bInputBuffLen], pKeySetValues, 4);
                bInputBuffLen += 4;
            }
        }

        /* Append FileID Information */
        if(bOption & PHAL_MFDFEVX_ISO_FILE_ID_AVAILABLE)
        {
            (void) memcpy(&pInputBuff[bInputBuffLen], pISOFileId, 2);
            bInputBuffLen += 2;
        }

        /* Append DFName Information */
        if(bOption & PHAL_MFDFEVX_ISO_DF_NAME_AVAILABLE)
        {
            (void) memcpy(&pInputBuff[bInputBuffLen], pISODFName, bISODFNameLen);
            bInputBuffLen += bISODFNameLen;
        }

        /* Append EncK Information. */
        (void) memcpy(&pInputBuff[bInputBuffLen], pEncK, 32);
        bInputBuffLen += 32;
    }

    /* Perform Offline activation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wKeyNoDAMMAC,
        (uint8_t) wKeyVerDAMMAC,
        NULL,
        0));

    /* Generate the MAC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_GenerateMAC(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDFEVX_SAM_X_TRUNCATION_MODE_MFP,
        pInputBuff,
        bInputBuffLen,
        &pResponse,
        &wRespLen));

    /* Copy the MAC to parameter. */
    (void) memcpy(pDAMMAC, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_CalculateTMV(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI, uint32_t dwTMILen, uint8_t * pTMV)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t *	PH_MEMLOC_REM pMac = NULL;
    uint16_t	PH_MEMLOC_REM wMacLen = 0;

    uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
    uint32_t	PH_MEMLOC_REM dwRemLen = 0;
    uint16_t	PH_MEMLOC_REM wTMIOffset = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wSrcKeyNo) || (wSrcKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(IS_VALID_SAM_RAM_KEYS(wDstKeyNo) || (wDstKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_SESSION_TMAC_MAC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Perform MAC verification. */
    dwRemLen = (uint16_t) dwTMILen;
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(dwRemLen <= PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) dwRemLen;
        }
        else
        {
            bExchangeLen = PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;
            dwRemLen = (uint16_t) (dwRemLen - PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH);
        }

        /* Exchange the TMI information to SAM. */
        wStatus = phalMfdfEVx_Sam_X_Int_SAM_GenerateMAC(
            pDataParams,
            wBuffOption,
            PHAL_MFDFEVX_SAM_X_TRUNCATION_MODE_MFP,
            &pTMI[wTMIOffset],
            bExchangeLen,
            &pMac,
            &wMacLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the TMI offset information. */
        wTMIOffset += PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;

    } while(!bFinished);

    /* Copy the Mac to the parameter. */
    (void) memcpy(pTMV, pMac, wMacLen);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_DecryptReaderID(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI, uint8_t * pTMRIPrev)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wSrcKeyNo) || (wSrcKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if (IS_VALID_SAM_RAM_KEYS(wDstKeyNo) || (wDstKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_SESSION_TMAC_ENC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Exchange the TMI information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DecipherOfflineData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        pEncTMRI,
        16,
        &pResponse,
        &wRespLen));

    /* Copy the decrypted information to the parameter. */
    (void) memcpy(pTMRIPrev, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_ComputeMFCLicenseMAC(void * pDataParams, uint16_t wOption, uint16_t wMFCLicenseMACKeyNo,
    uint16_t wMFCLicenseMACKeyVer, uint8_t * pInput, uint16_t wInputLen, uint8_t * pDivInput, uint8_t bDivInputLen,
    uint8_t * pMFCLicenseMAC)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    uint8_t		PH_MEMLOC_REM bOption = PH_OFF;
    uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
    uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
    uint16_t	PH_MEMLOC_REM wRemLen = 0;
    uint16_t	PH_MEMLOC_REM wInputOffset = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wMFCLicenseMACKeyNo) || (wMFCLicenseMACKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Frame the Option parameter. */
    bOption = (uint8_t) (wOption != PHAL_MFDFEVX_NO_DIVERSIFICATION);

    /* Perform Offline activation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        bOption,
        (uint8_t) wMFCLicenseMACKeyNo,
        (uint8_t) wMFCLicenseMACKeyVer,
        pDivInput,
        bDivInputLen));

    /* Perform MAC generation. */
    wRemLen = wInputLen;
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(wRemLen <= PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) wRemLen;
        }
        else
        {
            bExchangeLen = PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;
            wRemLen = (uint16_t) (wRemLen - PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH);
        }

        /* Exchange the Input information to SAM. */
        wStatus = phalMfdfEVx_Sam_X_Int_SAM_GenerateMAC(
            pDataParams,
            wBuffOption,
            PHAL_MFDFEVX_SAM_X_TRUNCATION_MODE_MFP,
            &pInput[wInputOffset],
            bExchangeLen,
            &pResponse,
            &wRespLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the TMI offset information. */
        wInputOffset += PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;

    } while(!bFinished);

    /* Copy the Mac to the parameter. */
    (void) memcpy(pMFCLicenseMAC, pResponse, wRespLen);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_CalculateMACSDM(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
    uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData,
    uint16_t wInDataLen, uint8_t * pMac)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
    uint16_t	PH_MEMLOC_REM wRemLen = 0;
    uint16_t	PH_MEMLOC_REM wInputOffset = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wSrcKeyNo) || (wSrcKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(IS_VALID_SAM_KEYS(wDstKeyNo) || (wDstKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ComputeSDMSessionVectors(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_SESSION_MAC,
        bSdmOption,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Perform MAC generation. */
    wRemLen = wInDataLen;
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(wRemLen <= PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) wRemLen;
        }
        else
        {
            bExchangeLen = PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;
            wRemLen = (uint16_t) (wRemLen - PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH);
        }

        /* Exchange the Input information to SAM. */
        wStatus = phalMfdfEVx_Sam_X_Int_SAM_GenerateMAC(
            pDataParams,
            wBuffOption,
            PHAL_MFDFEVX_SAM_X_TRUNCATION_MODE_MFP,
            &pInData[wInputOffset],
            bExchangeLen,
            &pResponse,
            &wRespLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the TMI offset information. */
        wInputOffset += PHAL_MFDFEVX_SAM_DATA_FRAME_LENGTH;

    } while(!bFinished);

    /* Copy the Mac to the parameter. */
    (void) memcpy(pMac, pResponse, wRespLen);

    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_DecryptSDMENCFileData(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
    uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pEncdata,
    uint16_t wEncDataLen, uint8_t * pPlainData)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the key information. */
    if(IS_VALID_SAM_KEYS(wSrcKeyNo) || (wSrcKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    if(IS_VALID_SAM_KEYS(wDstKeyNo) || (wDstKeyVer > 0xFFU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_ComputeSDMSessionVectors(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_SESSION_ENC,
        bSdmOption,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Load the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_LoadSDMInitVector(
        pDataParams,
        pSDMReadCtr));

    /* Exchange the Encrypted information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DecipherOfflineData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        pEncdata,
        (uint8_t) wEncDataLen,
        &pResponse,
        &wRespLen));

    /* Copy the decrypted information to the parameter. */
    (void) memcpy(pPlainData, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_DecryptSDMPICCData(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pEncdata,
    uint16_t wEncDataLen, uint8_t * pPlainData)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bEncDataOffset = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(
        pDataParams,
        PHAL_MFDFEVX_SAM_X_LRP_UPDATE_KEY_RFU,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        NULL,
        0));

    /* Perform DecipherDataOffline. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DecipherOfflineData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        &pEncdata[bEncDataOffset],
        (uint8_t) (wEncDataLen - bEncDataOffset),
        &pResponse,
        &wRespLen));

    /* Copy the decrypted information to the parameter. */
    (void) memcpy(pPlainData, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}
#endif /* NXPBUILD__PHAL_MFDFEVX_SAM_X */
