/*
 * Copyright 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>

#ifdef NXPBUILD__PHAL_MFDFEVX_SAM_X

#include "../phalMfdfEVx_Int.h"
#include "phalMfdfEVx_Sam_X.h"
#include "phalMfdfEVx_Sam_X_Int.h"

#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
#include "..\comps\phhalHw\src\Sam\Commands\02_SecurityConfiguration\phhalHw_Sam_Cmd_SC.h"
#include "..\comps\phhalHw\src\Sam\Commands\03_KeyManagement\phhalHw_Sam_Cmd_KM.h"
#include "..\comps\phhalHw\src\Sam\Commands\05_DataProcessing\phhalHw_Sam_Cmd_DP.h"
#include "..\comps\phhalHw\src\Sam\Commands\08_DESFire\phhalHw_Sam_Cmd_DESFire.h"
#include "..\comps\phhalHw\src\Sam\Commands\12_Common\phhalHw_Sam_Cmd_Common.h"
#include "..\comps\phhalHw\src\Sam\Commands\17_ISO14443_4\phhalHw_Sam_Cmd_ISO14443_4.h"
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

void phalMfdfEVx_Sam_X_Int_SetCmdCode(void * pDataParams, uint8_t bCmdCode)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bCmdCode = bCmdCode;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bCmdCode = bCmdCode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalMfdfEVx_Sam_X_Int_SetAuthMode(void * pDataParams, uint8_t bAuthMode)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bAuthMode = bAuthMode;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bAuthMode = bAuthMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint8_t phalMfdfEVx_Sam_X_Int_GetAuthMode(void * pDataParams)
{
    uint8_t PH_MEMLOC_REM bAuthMode = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            bAuthMode = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bAuthMode;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bAuthMode = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bAuthMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bAuthMode;
}

void phalMfdfEVx_Sam_X_Int_SetKeyNo(void * pDataParams, uint8_t bKeyNo)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bKeyNo = bKeyNo;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bKeyNo = bKeyNo;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint8_t phalMfdfEVx_Sam_X_Int_GetKeyNo(void * pDataParams)
{
    uint8_t PH_MEMLOC_REM bKeyNo = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            bKeyNo = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bKeyNo;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bKeyNo = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bKeyNo;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bKeyNo;
}

void phalMfdfEVx_Sam_X_Int_SetFabID(void * pDataParams, uint8_t bFabID)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bReturn_FabID = bFabID;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bReturn_FabID = bFabID;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint8_t phalMfdfEVx_Sam_X_Int_GetFabID(void * pDataParams)
{
    uint8_t PH_MEMLOC_REM bFabID = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            bFabID = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bReturn_FabID;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bFabID = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bReturn_FabID;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bFabID;
}

void phalMfdfEVx_Sam_X_Int_SetAdditionalInfo(void * pDataParams, uint16_t wValue)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint16_t phalMfdfEVx_Sam_X_Int_GetAdditionalInfo(void * pDataParams)
{
    uint16_t PH_MEMLOC_REM wAddInfo = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wAddInfo = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            wAddInfo = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return wAddInfo;
}

void phalMfdfEVx_Sam_X_Int_SetAID(void * pDataParams, uint8_t * pAID)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_NONX_ID:
            memcpy(((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->pAid, pAID, 3U);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            memcpy(((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->pAid, pAID, 3U);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalMfdfEVx_Sam_X_Int_GetAID(void * pDataParams, uint8_t ** ppAID)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ppAID[0] = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->pAid;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ppAID[0] = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->pAid;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalMfdfEVx_Sam_X_Int_SetUID(void * pDataParams, uint8_t *pUid, uint8_t bUidLen)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            memcpy(((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bUid, pUid, bUidLen);
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bUidLength = bUidLen;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            memcpy(((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bUid, pUid, bUidLen);
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bUidLength = bUidLen;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalMfdfEVx_Sam_X_Int_SetWrappedMode(void * pDataParams, uint8_t bWrappedMode)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bWrappedMode = bWrappedMode;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bWrappedMode = bWrappedMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint8_t phalMfdfEVx_Sam_X_Int_GetWrappedMode(void * pDataParams)
{
    uint8_t PH_MEMLOC_REM bWrappedMode = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            bWrappedMode = ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bWrappedMode = ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bWrappedMode;
}





phStatus_t phalMfdfEVx_Sam_X_Int_ValidateResponse(void * pDataParams, uint16_t wStatus, uint8_t * pPiccRetCode)
{
    uint8_t bISOMode = 0;

    /* Set ISOMode. */
    bISOMode = phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams);

    /* Evaluate the response. */
    if((wStatus == PH_ERR_SUCCESS) ||
        ((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_MIFARE_GEN) ||
        ((wStatus & PH_ERR_MASK) == PHAL_MFDFEVX_SAM_X_ERR_DESFIRE_GEN))
    {
        if(wStatus != PH_ERR_SUCCESS)
        {
            /* Validate if proper wrapped PICC status is available. First byte should be 91. */
            if(bISOMode)
            {
                if(pPiccRetCode[0U] != 0x91U)
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFEVX);
            }

            /* Validate the PICC Status. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Int_ComputeErrorResponse(pDataParams, pPiccRetCode[bISOMode ? 1U : 0U]));
        }
    }
    else
    {
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDFEVX);
        }

        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Int_Iso7816Wrap(void * pDataParams, uint8_t bFirstFrame, uint8_t bLengthPresent, uint8_t bLengthLen,
    uint8_t bLePresent, uint32_t dwTotDataLen, uint8_t * pPlainData, uint16_t * pDataLen)
{
    uint8_t     PH_MEMLOC_REM bCmdCode = 0;
    uint8_t     PH_MEMLOC_REM bLC = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;
    uint8_t     PH_MEMLOC_REM bOffset1 = 0;
    uint8_t     PH_MEMLOC_REM bIsIsoChainnedCmd_Read = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsIsoChainnedCmd_Write = PH_OFF;

    /* Extract the command code. */
    if(bFirstFrame)
    {
        bCmdCode = pPlainData[bLengthPresent ? bLengthLen : 0U];
    }

    /* Set the flag for data operation commands. */
    bIsIsoChainnedCmd_Read = (uint8_t) ((bCmdCode == PHAL_MFDFEVX_CMD_READ_DATA_ISO) || (bCmdCode == PHAL_MFDFEVX_CMD_READ_RECORDS_ISO));
    bIsIsoChainnedCmd_Write = (uint8_t) ((bCmdCode == PHAL_MFDFEVX_CMD_WRITE_DATA_ISO) || (bCmdCode == PHAL_MFDFEVX_CMD_WRITE_RECORD_ISO) ||
        (bCmdCode == PHAL_MFDFEVX_CMD_UPDATE_RECORD_ISO));

    /* Wrap the command if needed. */
    if(phalMfdfEVx_Sam_X_Int_GetWrappedMode(pDataParams) && (*pDataLen != 0U))
    {
        /* Set the LC value */
        bLC = (uint8_t) (bFirstFrame ? (*pDataLen - 1 /* Removing the command code. */) : *pDataLen);

        /* Subtract LC by 3 if Length is present. */
        bLC = (uint8_t) (bLengthPresent ? (bLC - bLengthLen) : bLC);

        /* Compute the offset. */
        bOffset = (uint8_t) ((bFirstFrame && bLengthPresent) ? (bLengthLen + 1U) : 1U);

        /* Compute the offset to be used for moving the current data. */
        bOffset1 = (uint8_t) (bLengthPresent ? (bLengthLen + 5U) : 5U);

        /* Copy the data to the Pointer. */
        (void) memmove(&pPlainData[bOffset1], &pPlainData[bOffset], bLC);

        /* Reset the length buffer. */
        *pDataLen = bLengthPresent ? bLengthLen : 0U;

        /* Frame the initial ISO7816 header. */
        pPlainData[(*pDataLen)++] = PHAL_MFDFEVX_WRAPPEDAPDU_CLA;
        pPlainData[(*pDataLen)++] = bCmdCode;
        pPlainData[(*pDataLen)++] = PHAL_MFDFEVX_WRAPPEDAPDU_P1;
        pPlainData[(*pDataLen)++] = PHAL_MFDFEVX_WRAPPEDAPDU_P2;

        /* Add LC if there is data. */
        if(bLC)
        {
            pPlainData[(*pDataLen)++] = (uint8_t) (bIsIsoChainnedCmd_Write ? dwTotDataLen : bLC);

            /* Update Frame length based on LC. */
            *pDataLen = *pDataLen + bLC;
        }

        if(bLePresent)
        {
            pPlainData[(*pDataLen)++] = PHAL_MFDFEVX_WRAPPEDAPDU_LE;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Int_GetKeyInfo(void * pDataParams, uint8_t bKeyNo, uint16_t * pKeyType, uint16_t * pSET, uint16_t * pExtSET)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wHostMode = 0;
    uint8_t     PH_MEMLOC_REM bIsAV3 = 0;
    uint8_t     PH_MEMLOC_REM bIsRamKey = 0;
    uint8_t     PH_MEMLOC_REM bMode = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;
    uint8_t     PH_MEMLOC_REM aKeyEntry[16];
    uint8_t     PH_MEMLOC_REM bKeyEntryLen = 0;

    /* Get the HostMode. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
        PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHAL_MFDFEVX_SAM_X_CONFIG_HOSTMODE,
        &wHostMode));

    /* Set the flag. */
    bIsAV3 = (uint8_t) (wHostMode >= 0x03U);
    bIsRamKey = (uint8_t) ((bKeyNo >= 0xE0U) && (bKeyNo <= 0xE3U)) ? PH_ON : PH_OFF;

    /* Set the Mode. */
    bMode = 0x01U;
    bMode = (uint8_t) (!bIsAV3 ? 0x00 : bMode);
    bMode = (uint8_t) (bIsRamKey ? 0x00 : bMode);

    /* Get the KeyInformation. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_GetKeyEntry(
        pDataParams,
        bKeyNo,
        bMode,
        aKeyEntry,
        &bKeyEntryLen));

    /* Extract SET configuration. */
    bOffset = (uint8_t) (bIsAV3 ? (bKeyEntryLen - 6U) : (bKeyEntryLen - 3U));
    *pSET = (uint16_t) ((aKeyEntry[bOffset + 1U] << 8U) | aKeyEntry[bOffset]);

    /* Extract ExtSET configuration. */
    bOffset = (uint8_t) (bIsAV3 ? (bKeyEntryLen - 4U) : (bKeyEntryLen - 1U));
    *pExtSET = (uint16_t) ((bIsAV3 ? (aKeyEntry[bOffset + 1] << 8U) : 0U) | aKeyEntry[bOffset]);

    /* Extract the KeyType. */
    *pKeyType = (uint16_t) (*pSET & 0x0078U);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
    uint16_t wDstKeyNo, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSV[16];
    uint8_t     PH_MEMLOC_REM aIV[16];
    uint8_t     PH_MEMLOC_REM bSvLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMC = 0;

    /* Convert TMC to Uint32. */
    dwTMC = (uint32_t) (pTMC[0U] | (pTMC[1U] << 8U) | (pTMC[2U] << 16U | (pTMC[3U] << 24U)));

    /* If TMC is 0xFFFFFFFF, then return error */
    if(dwTMC == 0xFFFFFFFFU)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFPEVX);
    }

    /* Increment the TMC by 1. */
    dwTMC++;

    /* Clear the session vector SV. */
    (void) memset(aSV, 0U, 16U);
    (void) memset(aIV, 0U, 16U);

    /* Compute the session vector. */
    aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFDFEVX_SAM_X_SESSION_TMAC_ENC) ? 0xA5U : 0x5AU);
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x01U;
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x80U;

    /* Append the TMC information. */
    aSV[bSvLen++] = (uint8_t) (dwTMC & 0xFFU);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF00U) >> 8U);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF0000U) >> 16U);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF000000U) >> 24U);

    /* Append the UID information. */
    (void) memcpy(&aSV[bSvLen], pUid, bUidLen);
    bSvLen = 16U;

    /* Exchange the session vector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DeriveKey(
        pDataParams,
        (uint8_t) wSrcKeyNo,
        (uint8_t) wSrcKeyVer,
        (uint8_t) wDstKeyNo,
        aSV,
        16U));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Int_ComputeSDMSessionVectors(void * pDataParams, uint8_t bOption, uint8_t bSdmOption, uint16_t wSrcKeyNo,
    uint16_t wSrcKeyVer, uint16_t wDstKeyNo, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSV[32];
    uint8_t     PH_MEMLOC_REM bSvLen = 0;
    uint32_t    PH_MEMLOC_REM dwSDMReadCtr = 0;

    /* Validate the Counter value. */
    if(pSDMReadCtr != NULL)
    {
        dwSDMReadCtr = (uint32_t) (pSDMReadCtr[0U] | (pSDMReadCtr[1U] << 8U) | (pSDMReadCtr[2U] << 16U) | (pSDMReadCtr[3U] << 24U));
        if(dwSDMReadCtr == 0xFFFFFFU)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFDFEVX);
        }
    }

    /* Clear the session vector SV. */
    (void) memset(aSV, 0, sizeof(aSV));

    /* Frame the default values in session vector. */
    aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFDFEVX_SESSION_ENC) ? 0xC3U : 0x3CU);
    aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFDFEVX_SESSION_ENC) ? 0x3CU : 0xC3U);
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x01U;
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x80U;

    /* Append the UID */
    if(bSdmOption & PHAL_MFDFEVX_VCUID_PRESENT)
    {
        if(pUid != NULL)
        {
            (void) memcpy(&aSV[bSvLen], pUid, bUidLen);
            bSvLen += bUidLen;
        }
        else
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFEVX);
    }

    /* Append the SDM ReadCtr information. */
    if(bSdmOption & PHAL_MFDFEVX_RDCTR_PRESENT)
    {
        if(dwSDMReadCtr != NULL)
        {
            aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFFU);
            aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00U) >> 8U);
            aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000U) >> 16U);
        }
    }

    /* Update the SV length */
    if((bSdmOption & PHAL_MFDFEVX_RDCTR_PRESENT) && (bSvLen > 16U))
        bSvLen = 32U;
    else
        bSvLen = 16U;

    /* Exchange the session vector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_DeriveKey(
        pDataParams,
        (uint8_t) wSrcKeyNo,
        (uint8_t) wSrcKeyVer,
        (uint8_t) wDstKeyNo,
        aSV,
        bSvLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}

phStatus_t phalMfdfEVx_Sam_X_Int_LoadSDMInitVector(void * pDataParams, uint8_t * pSDMReadCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aIV[16];
    uint8_t *   PH_MEMLOC_REM pIV = NULL;
    uint16_t    PH_MEMLOC_REM wIvLen = 0;

    /* Clear the Initialization Vector. */
    (void) memset(aIV, 0, 16U);

    /* IV computation is E(KSesSDMFileReadENC; SDMReadCtr || 0x00000000000000000000000000) */
    (void) memcpy(&aIV[wIvLen], pSDMReadCtr, 3U);

    /* Set the IV length to 16. */
    wIvLen = 16U;

    /* Encrypt the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_EncipherOfflineData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        aIV,
        (uint8_t) wIvLen,
        &pIV,
        &wIvLen));

    /* Copy the enciphered data to local buffer. */
    (void) memcpy(aIV, pIV, wIvLen);
    pIV = NULL;

    /* Load the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfEVx_Sam_X_Int_SAM_LoadInitVector(
        pDataParams,
        0x00,
        aIV,
        (uint8_t) wIvLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFEVX);
}





void phalMfdfEVx_Sam_X_Int_GetSAMUID(void * pDataParams, uint8_t ** ppAID)
{
    /* Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            ppAID[0] = ((phhalHw_Sam_DataParams_t *) (((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->pHalSamDataParams))->bUid;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ppAID[0] = ((phhalHw_SamAV3_DataParams_t *) (((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->pHalSamDataParams))->bUid;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_KillAuthentication(void * pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_KillAuthentication(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_GetRandom(void * pDataParams, uint8_t bExpLen, uint8_t * pRnd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_GetRandom(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bExpLen,
                pRnd);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_GetRandom(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bExpLen,
                pRnd);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_ActivateOfflineKey(void * pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_ActivateOfflineKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bKeyNo,
                bKeyVer,
                pDivInput,
                bDivInputLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bKeyNo,
                bKeyVer,
                pDivInput,
                bDivInputLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_LoadInitVector(void *pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_LoadInitVector(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                pData,
                bDataLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                pData,
                bDataLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_DumpSecretKey(void * pDataParams, uint8_t bDumpMode, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pDivInput,
    uint8_t bDivInputLen, uint8_t * pSecretKey, uint16_t * pSecretKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_DumpSecretKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bDumpMode,
                bKeyNo,
                bKeyVer,
                pDivInput,
                bDivInputLen,
                &pSecretKey,
                pSecretKeyLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_DumpSecretKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bDumpMode,
                bKeyNo,
                bKeyVer,
                pDivInput,
                bDivInputLen,
                pSecretKey,
                &bKeyLen);

            *pSecretKeyLen = bKeyLen;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_GetKeyEntry(void * pDataParams, uint8_t bKeyNo, uint8_t bMode, uint8_t * pKeyEntry, uint8_t * pKeyEntryLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
    uint8_t* PH_MEMLOC_REM pKeyEntry_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wKeyLen = 0;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_GetKeyEntry(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bKeyNo,
                bMode,
                &pKeyEntry_Tmp,
                &wKeyLen);

            memcpy(pKeyEntry, pKeyEntry_Tmp, wKeyLen);
            *pKeyEntryLen = (uint8_t) wKeyLen;
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_GetKeyEntry(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bKeyNo,
                bMode,
                pKeyEntry,
                pKeyEntryLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_DeriveKey(void * pDataParams, uint8_t bSrcKeyNo, uint8_t bSrcKeyVer,
    uint8_t bDstKeyNo, uint8_t * pDeriveIn, uint8_t bDeriveInLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_DeriveKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bSrcKeyNo,
                bSrcKeyVer,
                bDstKeyNo,
                pDeriveIn,
                bDeriveInLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_DeriveKey(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bSrcKeyNo,
                bSrcKeyVer,
                bDstKeyNo,
                pDeriveIn,
                bDeriveInLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_EncipherOfflineData(void * pDataParams, uint16_t wOption, uint8_t * pPlainData,
    uint8_t bPlainDataLen, uint8_t ** ppEncData, uint16_t * pEncDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_EncipherOfflineData(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pPlainData,
                bPlainDataLen,
                ppEncData,
                pEncDataLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pPlainData,
                bPlainDataLen,
                ppEncData,
                pEncDataLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_GenerateMAC(void * pDataParams, uint16_t wOption, uint8_t bNum,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_GenerateMAC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bNum,
                pData,
                bDataLen,
                ppResponse,
                pRespLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bNum,
                pData,
                bDataLen,
                ppResponse,
                pRespLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_SAM_DecipherOfflineData(void *  pDataParams, uint16_t wOption, uint8_t * pEncData,
    uint8_t bEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_DecipherOfflineData(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pEncData,
                bEncDataLen,
                ppPlainData,
                pPlainDataLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pEncData,
                bEncDataLen,
                ppPlainData,
                pPlainDataLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_DESFire_AuthenticatePICC(void * pDataParams, uint8_t bOption, uint8_t bISOMode, uint8_t bDFKeyNo,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t bPCDcap2InLen, uint8_t * pPCDcap2In, uint8_t * pDivInput, uint8_t  bDivInputLen,
    uint8_t * pPDcap2, uint8_t * pPCDcap2, uint8_t * pPiccReturnCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_DESFire_AuthenticatePICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bISOMode,
                bDFKeyNo,
                (uint8_t) bKeyNo,
                (uint8_t) bKeyVer,
                bPCDcap2InLen,
                pPCDcap2In,
                pDivInput,
                bDivInputLen,
                pPDcap2,
                pPCDcap2,
                pPiccReturnCode);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bISOMode,
                bDFKeyNo,
                (uint8_t) bKeyNo,
                (uint8_t) bKeyVer,
                bPCDcap2InLen,
                pPCDcap2In,
                pDivInput,
                bDivInputLen,
                pPDcap2,
                pPCDcap2,
                pPiccReturnCode);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_DESFire_ChangeKeyPICC(void * pDataParams, uint8_t bKeyCompMeth, uint8_t bConfig,
    uint8_t bKeySetNo, uint8_t bDFKeyNo, uint8_t bCurrKeyNo, uint8_t bCurrKeyVer, uint8_t bNewKeyNo, uint8_t bNewKeyVer,
    uint8_t * pDivInput, uint8_t  bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_DESFire_ChangeKeyPICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bKeyCompMeth,
                bConfig,
                bKeySetNo,
                bDFKeyNo,
                bCurrKeyNo,
                bCurrKeyVer,
                bNewKeyNo,
                bNewKeyVer,
                pDivInput,
                bDivInputLen,
                ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->aPiccRetCode);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bKeyCompMeth,
                bConfig,
                bKeySetNo,
                bDFKeyNo,
                bCurrKeyNo,
                bCurrKeyVer,
                bNewKeyNo,
                bNewKeyVer,
                pDivInput,
                bDivInputLen,
                ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->aPiccRetCode);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_DESFire_WriteX(void * pDataParams, uint16_t wOption, uint8_t bCrypto, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_DESFire_WriteX(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bCrypto,
                pData,
                bDataLen,
                ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->aPiccRetCode,
                &((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bPiccRetCodeLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bCrypto,
                pData,
                bDataLen,
                ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->aPiccRetCode,
                &((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bPiccRetCodeLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_DESFire_ReadX(void * pDataParams, uint16_t wOption, uint8_t bCrypto, uint8_t * pAppData,
    uint8_t bAppDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_DESFire_ReadX(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bCrypto,
                pAppData,
                bAppDataLen,
                ppResponse,
                pRespLen,
                ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->aPiccRetCode,
                &((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->bPiccRetCodeLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                bCrypto,
                pAppData,
                bAppDataLen,
                ppResponse,
                pRespLen,
                ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->aPiccRetCode,
                &((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->bPiccRetCodeLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_DESFire_CreateTMFilePICC(void * pDataParams, uint8_t bOption, uint8_t bISOMode,
    uint8_t bKeyNo, uint8_t bKeyVer, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights,
    uint8_t bTMKeyOptions, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_DESFire_CreateTMFilePICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bISOMode,
                bKeyNo,
                bKeyVer,
                bFileNo,
                bFileOption,
                pAccessRights,
                NULL,
                bTMKeyOptions,
                0,
                pDivInput,
                bDivInputLen,
                ((phalMfdfEVx_Sam_X_DataParams_t *) pDataParams)->aPiccRetCode);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_DESFire_CreateTMFilePICC(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bISOMode,
                bKeyNo,
                bKeyVer,
                bFileNo,
                bFileOption,
                pAccessRights,
                bTMKeyOptions,
                pDivInput,
                bDivInputLen,
                ((phalMfdfEVx_SamAV3_X_DataParams_t *) pDataParams)->aPiccRetCode);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_TMRI_CommitReaderID(void * pDataParams, uint8_t bISOMode, uint8_t bState,
    uint16_t wBlockNr, uint8_t ** ppEncTMRI, uint16_t * pEncTMRILen, uint8_t * pStatusCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_TMRI_CommitReaderID(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bISOMode,
                bState,
                wBlockNr,
                ppEncTMRI,
                pEncTMRILen,
                pStatusCode);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_TMRI_CommitReaderID(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bISOMode,
                bState,
                wBlockNr,
                ppEncTMRI,
                pEncTMRILen,
                pStatusCode);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

phStatus_t phalMfdfEVx_Sam_X_Int_ISO14443_4_Exchange(void * pDataParams, uint16_t wOption, uint8_t * pAppDataIn,
    uint8_t bLenAppData, uint8_t ** ppAppDataOut, uint16_t * pAppDataOutLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /*Fetch component ID and invoke appropriate SAM implementation */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_MFDFEVX_INTERNAL
        case PHAL_MFDFEVX_SAM_X_ID:
            wStatus = phhalHw_Sam_Cmd_X_ISO14443_4_Exchange(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pAppDataIn,
                bLenAppData,
                ppAppDataOut,
                pAppDataOutLen);
            break;
#endif /* NXPBUILD__PHAL_MFDFEVX_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_MFDFEVX_SAMAV3_X_ID:
            wStatus = phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
                PHAL_MFDFEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                wOption,
                pAppDataIn,
                bLenAppData,
                ppAppDataOut,
                pAppDataOutLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDFEVX);
            break;
    }
    return wStatus;
}

#endif /* NXPBUILD__PHAL_MFDFEVX_SAM_X */
