/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/**
 * \file
 * DESFire LIGHT application SamAV3 NonX component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phKeyStore.h>
#include <phTMIUtils.h>
#ifdef NXPBUILD__PHAL_MFDFLIGHT_SAM_NONX

#include "../phalMfdfLight_Int.h"
#include "phalMfdfLight_Sam_NonX.h"
#include "phalMfdfLight_Sam_NonX_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfdfLight_SamAV3_NonX_Init(phalMfdfLight_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_SamAV3_DataParams_t * pHalSamDataParams, void * pPalMifareDataParams, phTMIUtils_t * pTMIDataParams)
{
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFDFLIGHT);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFDFLIGHT);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFDFLIGHT);

    /* Data Params size Check. */
    if(sizeof(phalMfdfLight_SamAV3_NonX_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDFLIGHT);
    }

    /* Initialize dataparams structure members. */
    pDataParams->wId = PH_COMP_AL_MFDFLIGHT | PHAL_MFDFLIGHT_SAMAV3_NONX_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pTMIDataParams = pTMIDataParams;
    pDataParams->bKeyNo = 0xFF;
    pDataParams->bAuthMode = PHAL_MFDFLIGHT_NOT_AUTHENTICATED;
    pDataParams->bAuthType = 0xFF;
    pDataParams->wAdditionalInfo = 0x0000;

    memset(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, 0x00, 3);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_AuthenticateEv2(void *pDataParams, uint8_t bAuthType, uint16_t wOption, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCapsIn,
    uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bFirstAuth = 0;
    uint8_t     PH_MEMLOC_REM bAuthMode = 0;

    /* Check if First Auth parameter do not contain invalid values. */
    if(bAuthType > PHAL_MFDFLIGHT_AUTHFIRST_LRP)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Validate the PCD Input Capability length */
    if(bAuthType < PHAL_MFDFLIGHT_AUTHNONFIRST_LRP)
    {
        /* For EV2 mode the length should not exceed 6 bytes. */
        if(bPcdCapsInLen > 6)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }
    else
    {
        /* For LRP mode the length should at least be 1 byte. */
        if((bPcdCapsInLen == 0) && (bAuthType == PHAL_MFDFLIGHT_AUTHFIRST_LRP))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }

    /* Set First or following Auth to wOption parameter. */
    bFirstAuth = (uint8_t) ((bAuthType & 0x01) ? PHAL_MFDFLIGHT_CMD_AUTHENTICATE_EV2_FIRST : PHAL_MFDFLIGHT_CMD_AUTHENTICATE_EV2_NON_FIRST);

    /* Set EV2 or LRP mode. */
    bAuthMode = (uint8_t) ((bAuthType < PHAL_MFDFLIGHT_AUTHNONFIRST_LRP) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2 : PHHAL_HW_CMD_SAMAV3_AUTH_MODE_LRP);

    /* Exchange the commands between Card and SAM hardware to complete Authentication. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_AuthenticatePICC(
        pDataParams,
        bFirstAuth,
        bAuthMode,
        wOption,
        wKeyNo,
        wKeyVer,
        bKeyNoCard,
        pDivInput,
        bDivInputLen,
        pPcdCapsIn,
        bPcdCapsInLen,
        pPcdCapsOut,
        pPdCapsOut));

    /* Set the Authentication type (EV2 or LRP). */
    if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
    {
        PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthType = (uint8_t) ((bAuthType < PHAL_MFDFLIGHT_AUTHNONFIRST_LRP) ?
            PHAL_MFDFLIGHT_AUTH_TYPE_EV2 : PHAL_MFDFLIGHT_AUTH_TYPE_LRP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_SET_CONFIG;
    aCmdBuff[1] = bOption;

    /* Frame the response communication mode. */
    bResp_ComMode = (uint8_t) ((bOption == PHAL_MFDFLIGHT_SET_CONFIG_OPTION5) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.SetConfiguration information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        PHAL_MFDFLIGHT_COMMUNICATION_ENC,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        2,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Reset the Authentication if configuration is LRP enabling. */
    if((bOption == PHAL_MFDFLIGHT_SET_CONFIG_OPTION5) && (pData[4] == 0x02))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_ResetAuthentication(pDataParams));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetVersion(void * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_VERSION;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.GetVersion information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        bComMode,
        aCmdBuff,
        1,
        &pVerInfo,
        &wRespLen));

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, wRespLen));
    *pVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetCardUID(void * pDataParams, uint8_t * pUid, uint8_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM aUid[10];
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_CARD_UID;

    /* Exchange Cmd.GetCardUID information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        PHAL_MFDFLIGHT_COMMUNICATION_ENC,
        PHAL_MFDFLIGHT_COMMUNICATION_ENC,
        aCmdBuff,
        1,
        &pUid,
        &wRespLen));

    /* Copy the complete uid to internal buffer for manipulation. */
    memcpy(aUid, pUid, wRespLen);
    *pUidLen = (uint8_t) wRespLen;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, (uint16_t) (*pUidLen)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer,
    uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Exchange the commands between Card and SAM hardware to Change Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ChangeKeyPICC(
        pDataParams,
        wOption,
        bKeyNoCard,
        wCurrKeyNo,
        wCurrKeyVer,
        wNewKeyNo,
        wNewKeyVer,
        pDivInput,
        bDivInputLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion, uint8_t * pKeyVerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM aAppId[3] = { 0x00, 0x00, 0x00 };

    /* Only if seleted Aid is 0x000000. */
    if(memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00)
    {
        /* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
        if((bKeyNo > PHAL_MFDFLIGHT_ORIGINALITY_KEY_LAST) || (bKeyNo < PHAL_MFDFLIGHT_ORIGINALITY_KEY_FIRST))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }
    else
    {
        /* At App level, 0,1,2,3,4 are valid ones. */
        if(bKeyNo > PHAL_MFDFLIGHT_APP_KEY_LAST)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }
#endif

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_GET_KEY_VERSION;
    aCmdBuff[wCmdLen++] = bKeyNo;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.GetKeyVersion information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        bComMode,
        aCmdBuff,
        wCmdLen,
        &pKeyVersion,
        &wRespLen));

    *pKeyVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_CreateTransactionMacFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pAccessRights, uint16_t wKeyNo, uint8_t bKeyVer, uint8_t bTMKeyOption, uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Exchange the commands between Card and SAM hardware to create Transaction MAC file. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_CreateTMFilePICC(
        pDataParams,
        (uint8_t) (bDivInputLen ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON : PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF),
        bFileNo,
        bFileOption,
        pAccessRights,
        bTMKeyOption,
        (uint8_t) wKeyNo,
        bKeyVer,
        pDivInput,
        bDivInputLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_DELETE_FILE;
    aCmdBuff[wCmdLen++] = bFileNo;

    /* Exchange Cmd.DeleteFile information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetFileIDs(void * pDataParams, uint8_t * pFid, uint8_t * pNumFid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_FILE_IDS;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.GetFileIDs information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        aCmdBuff,
        1,
        &pFid,
        &wRespLen));

    *pNumFid = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetISOFileIDs(void * pDataParams, uint8_t * pFidBuffer, uint8_t * pNumFid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_ISO_FILE_IDS;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.GetISOFileIDs information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        aCmdBuff,
        1,
        &pFidBuffer,
        &wRespLen));

    /* Update the length. */
    *pNumFid = (uint8_t) (wRespLen / 2);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_FILE_SETTINGS;
    aCmdBuff[1] = bFileNo;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.GetFileSettings information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        aCmdBuff,
        2,
        &pFSBuffer,
        &wRespLen));

    /* Update the length. */
    *pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pAccessRights, uint8_t bTMCLimitLen, uint8_t * pTMCLimit)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[10];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;

    /* Frame the Crypto information.
     * Masking the MSB bit becuase its not support by SAM.
     * Its actually supported in Software only.
     */
    bCmd_ComMode = (uint8_t) (bOption & 0xFE);

    /* Validate the parameters */
    if((bCmd_ComMode != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCmd_ComMode != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCmd_ComMode != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_CHANGE_FILE_SETTINGS;
    aCmdBuff[wCmdLen++] = bFileNo;
    aCmdBuff[wCmdLen++] = bFileOption;

    /* Append access rights. */
    memcpy(&aCmdBuff[wCmdLen], pAccessRights, 2);
    wCmdLen += 2;

    /* TMCLimit buffer in command buffer if Bit5 of File Option is SET. */
    if(bFileOption & PHAL_MFDFLIGHT_TMCLIMITCONFIG)
    {
        memcpy(&aCmdBuff[wCmdLen], pTMCLimit, bTMCLimitLen);
        wCmdLen += bTMCLimitLen;
    }

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN : PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.ChangeFileSettings information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        (uint16_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? wCmdLen : 2),
        &aCmdBuff[2],
        (uint16_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? 0 : (wCmdLen - 2)),
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}





phStatus_t phalMfdfLight_Sam_NonX_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pLength, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;

    uint8_t     PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
    if(((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    if((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_READ_DATA_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pOffset, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pLength, 3);
    wCmdLen += 3;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
        &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Compute the length. */
        dwLength = pLength[2];
        dwLength = dwLength << 8 | pLength[1];
        dwLength = dwLength << 8 | pLength[0];

        /* Frame the Option. */
        bTMIOption = (uint8_t) (dwLength ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

        /* Buffer the Command information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            bTMIOption, aCmdBuff, wCmdLen, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) (((bOption & 0xF0) == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Frame the SM to be applied for response. */
    bResp_ComMode = (uint8_t) (bOption & 0xF0);

    /* Exchange Cmd.ReadData information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bCmd_ComMode,
        bResp_ComMode,
        aCmdBuff,
        wCmdLen,
        ppResponse,
        pRespLen));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Frame the Option. */
        bTMIOption = (uint8_t) (dwLength ? PH_TMIUTILS_ZEROPAD_DATABUFF : (PH_TMIUTILS_ZEROPAD_DATABUFF | PH_TMIUTILS_READ_INS));

        /* Collect the data received. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            bTMIOption, NULL, 0, *ppResponse, *pRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(!dwLength)
        {
            /* Reset wOffsetInTMI */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
    if((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_WRITE_DATA_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pOffset, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pDataLen, 3);
    wCmdLen += 3;

    /* Set the lengths. */
    dwDataLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), aCmdBuff, wCmdLen, pData, dwDataLen,
            PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) (bOption & 0xF0);

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.WriteData information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        pData,
        dwDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetValue(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
    if((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_GET_VALUE;
    aCmdBuff[1] = bFileNo;

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) ((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Frame the SM to be applied for response. */
    bResp_ComMode = bCommOption;

    /* Exchange Cmd.GetValue information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bCmd_ComMode,
        bResp_ComMode,
        aCmdBuff,
        2,
        &pValue,
        &wRespLen));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, aCmdBuff, 2, pValue, wRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_Credit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
    if((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_CREDIT;
    aCmdBuff[1] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, aCmdBuff, 2, pValue, 4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = bCommOption;

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.Credit information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        2,
        pValue,
        4,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_Debit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
    if((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_DEBIT;
    aCmdBuff[1] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, aCmdBuff, 2, pValue, 4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = bCommOption;

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.Debit information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        2,
        pValue,
        4,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_LimitedCredit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
    if((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_LIMITED_CREDIT;
    aCmdBuff[1] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, aCmdBuff, 2, pValue, 4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = bCommOption;

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.LimitedCredit information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        2,
        pValue,
        4,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_ReadRecords(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo,
    uint8_t * pRecCount, uint8_t * pRecSize, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwNumRec = 0;
    uint32_t    PH_MEMLOC_REM dwRecLen = 0;

    uint8_t     PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;
    uint32_t    PH_MEMLOC_REM dwNumRecCal = 0;
    uint32_t    PH_MEMLOC_REM dwTMIBufOffset = 0;
    uint32_t    PH_MEMLOC_REM dwTMIBufIndex = 0;

    /* Validate the parameter. */
    if(((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    if((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_READ_RECORDS_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pRecNo, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pRecCount, 3);
    wCmdLen += 3;

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Compute the number of records. */
        dwNumRec = pRecCount[2];
        dwNumRec = dwNumRec << 8 | pRecCount[1];
        dwNumRec = dwNumRec << 8 | pRecCount[0];

        /* Compute the record length. */
        dwRecLen = pRecSize[2];
        dwRecLen = dwRecLen << 8 | pRecSize[1];
        dwRecLen = dwRecLen << 8 | pRecSize[0];

        /* Should should provide atleast RecLen / NumRec to update in TMI collection */
        if(!dwRecLen && !dwNumRec)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }

        /* Frame the Option. */
        bTMIOption = (uint8_t) (dwNumRec ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

        /* Buffer the Command information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            bTMIOption, aCmdBuff, wCmdLen, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) (((bOption & 0xF0) == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Frame the SM to be applied for response. */
    bResp_ComMode = (uint8_t) (bOption & 0xF0);

    /* Exchange Cmd.ReadRecords information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bCmd_ComMode,
        bResp_ComMode,
        aCmdBuff,
        wCmdLen,
        ppResponse,
        pRespLen));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        if(!dwNumRec)
        {
            /* Get the TMI buffer offset. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, &dwTMIBufOffset));

            /* Get the current TMI buffer index. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_BUFFER_INDEX, &dwTMIBufIndex));

            /* Calculate Response length in case of chaining. */
            dwDataLen = *pRespLen + dwTMIBufIndex - (dwTMIBufOffset + 11);

            /* If user update worng RecSize, we cant calculate recCnt. */
            if(dwDataLen % dwRecLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFLIGHT);
            }

            /* Calculate number of records. */
            dwNumRecCal = dwDataLen / dwRecLen;

            /* Update the value as TMI buffer offset. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_VALUE, dwNumRecCal));
        }

        /* Collect the data received. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, *ppResponse, *pRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));

        /* Reset the TMI buffer Offset. */
        if(!dwNumRec)
        {
            /* Reset wOffsetInTMI */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }

        if(!dwNumRec)
        {
            /* Reset TMI Buffer offset. */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_WriteRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
    if((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_WRITE_RECORD_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pOffset, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pDataLen, 3);
    wCmdLen += 3;

    /* Set the lengths. */
    dwDataLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), aCmdBuff, wCmdLen, pData, dwDataLen,
            PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) (bOption & 0xF0);

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.WriteRecord information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        pData,
        dwDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_UpdateRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo,
    uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[11];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bResp_ComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
    if((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_UPDATE_RECORD_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pRecNo, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pOffset, 3);
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pDataLen, 3);
    wCmdLen += 3;

    /* Set the lengths. */
    dwDataLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

    /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), aCmdBuff, wCmdLen, pData, dwDataLen,
            PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the SM to be applied for command. */
    bCmd_ComMode = (uint8_t) (bOption & 0xF0);

    /* Frame the SM to be applied for command. */
    bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) ? PHAL_MFDFLIGHT_COMMUNICATION_PLAIN :
        PHAL_MFDFLIGHT_COMMUNICATION_MACD);

    /* Exchange Cmd.UpdateRecord information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bResp_ComMode,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        pData,
        dwDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_CLEAR_RECORDS_FILE;
    aCmdBuff[1] = bFileNo;

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_CMDBUFF, aCmdBuff, 2, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Frame the Crypto information. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.ClearRecordFile information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        PH_OFF,
        aCmdBuff,
        2,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_CommitTransaction(void * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
    if((bOption & 0x0F) > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command information. */
    aCmdBuff[bCmdLen++] = PHAL_MFDFLIGHT_CMD_COMMIT_TXN;

    /*
     * 0x01 and 0x00 are values to support the legacy implementation.
     *      Here if it 0x00 the Option is not exchanged to the PICC. Its only exchanged for 0x01
     *
     * 0x80 and 0x81 are the values to exchange the Option by to PICC.
     *      If its 0x80, Option byte will be exchanged by the Option information will be zero.
     *      If its 0x81, Option byte will be exchanged by the Option information will be one.
     */
    if((bOption & 0x80) || (bOption & 0x0F))
    {
        aCmdBuff[bCmdLen++] = (bOption & 0x0F);
    }

    /* Frame the Crypto information. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.CommitTransaction information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        PH_OFF,
        aCmdBuff,
        bCmdLen,
        NULL,
        0,
        &pResponse,
        &wRespLen));

    /* Copy the data to the parameter */
    if(bOption)
    {
        memcpy(pTMC, &pResponse[0], 4);
        memcpy(pTMV, &pResponse[4], 8);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_AbortTransaction(void * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_ABORT_TXN;

    /* Frame the Crypto information. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_MACD : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.AbortTransaction information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_WriteData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_MACD,
        PH_OFF,
        aCmdBuff,
        1,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_CommitReaderID(void * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    phStatus_t  PH_MEMLOC_REM wStatus1;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM bPiccRetCode = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[17 + 8 /* MAC */];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResp_Card = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen_Card = 0;
    uint8_t *   PH_MEMLOC_REM pResp_Sam = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen_Sam = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Exchange the details to SAM hardware and get the TMRI and MAC. */
    if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
    {
        wStatus = phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part1(
            PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PHHAL_HW_CMD_SAMAV3_COMMIT_READER_ID_PICC_STATE_DESFIRE,
            0,
            &pResp_Sam,
            &wRespLen_Sam);

        /* Verify if Success chaining response is received from SAM. */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
            return wStatus;
    }
    else
    {
        pResp_Sam = pTMRI;
        wRespLen_Sam = 16;
    }

    /* Clear the buffers. */
    wCmdLen = 0;
    memset(aCmdBuff, 0x00, sizeof(aCmdBuff));

    /* Frame the command and send it to card. */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_COMMIT_READER_ID;

    /* Add the TMRI to command buffer. */
    memcpy(&aCmdBuff[wCmdLen], pResp_Sam, wRespLen_Sam);
    wCmdLen += wRespLen_Sam;

    /* Exchange command information. */
    wStatus = phalMfdfLight_Sam_NonX_Int_CardExchange(
        pDataParams,
        PH_ON,
        PH_EXCHANGE_DEFAULT,
        wCmdLen,
        PH_ON,
        aCmdBuff,
        wCmdLen,
        &pResp_Card,
        &wRespLen_Card,
        &bPiccErrCode);

    /*Exchange the response received from card to SAM hardware */
    if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
    {
        wStatus = phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part2(
            PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
            bPiccErrCode,
            pResp_Card,
            (uint8_t) wRespLen_Card,
            &bPiccRetCode);
    }

    /*
     * Kill PICC Authentication for next SAM call to proceed further
     * This code update is based on information mentioned in MIFARE SAM AV3 known deviations from specification
     * section 5.2, to overcome the issue where if there is no payload for PART-2 exchange.
     */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
            PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));
    }

    /* validate the error code. */
    wStatus = phalMfdfLight_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, bPiccErrCode);

    /* REset Authentication state in case of error. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfLight_Sam_NonX_Int_ResetAuthStatus(pDataParams));

        return wStatus;
    }

    /* Update the ppEncTMRI parameter. */
    wRespLen_Card = (uint16_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        (wRespLen_Card - 8) : wRespLen_Card);
    memcpy(pEncTMRI, pResp_Card, wRespLen_Card);

    /* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the recieved TMRI */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, wRespLen_Card));

   /* Get the TMI Status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* Buffer the Command and Data information to TMI buffer. */
        if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
        {
            /* If authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
             * TMI = TMI || Cmd || TMRICur || EncTMRI || ZeroPadding
             */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_ZEROPAD_DATABUFF, aCmdBuff, 17, pEncTMRI, 16, PHAL_MFDFLIGHT_BLOCK_SIZE));
        }
        else
        {
            /* If not authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
             * TMI = TMI || Cmd || TMRICur || ZeroPadding
             */
            PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
                PH_TMIUTILS_ZEROPAD_CMDBUFF, aCmdBuff, 17, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));

            pEncTMRI = '\0';
        }
    }

    return wStatus;
}




phStatus_t phalMfdfLight_Sam_NonX_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid,
    uint8_t * pDFname, uint8_t bDFnameLen, uint8_t  bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[26];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aFileId[3] = { '\0' };
    uint16_t    PH_MEMLOC_REM wVal = 0;
    uint8_t     PH_MEMLOC_REM aPiccDfName[7] = { 0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00 };
    uint8_t     PH_MEMLOC_REM aPiccStat[2] = { 0x00, 0x00 };

    /* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
    if(bDFnameLen > 16)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
    /* If User intends to select through DFName, check for validity of DFName passed */
    if((bSelector == PHAL_MFDFLIGHT_SELECTOR_4) && (bDFnameLen == 0x00))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
    if((bOption != PHAL_MFDFLIGHT_FCI_RETURNED) && (bOption != PHAL_MFDFLIGHT_FCI_NOT_RETURNED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
    if((bSelector != PHAL_MFDFLIGHT_SELECTOR_4) && (bSelector != PHAL_MFDFLIGHT_SELECTOR_3) && (pFid == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

    /* Frame the command. */
    aCmdBuff[wCmdLen++] = 0x00;
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_SELECT_FILE;
    aCmdBuff[wCmdLen++] = bSelector;
    aCmdBuff[wCmdLen++] = bOption;

    /* Append LC. */
    if(bSelector != PHAL_MFDFLIGHT_SELECTOR_3)
    {
        if(bExtendedLenApdu)
        {
            aCmdBuff[wCmdLen++] = 0;
            aCmdBuff[wCmdLen++] = 0;
        }
    }

    /* Append the payload and LC. */
    if(bSelector == PHAL_MFDFLIGHT_SELECTOR_4)
    {
        /* Append LC. */
        aCmdBuff[wCmdLen++] = bDFnameLen;

        memcpy(&aCmdBuff[wCmdLen], pDFname, bDFnameLen);
        wCmdLen += bDFnameLen;
    }
    else if((bSelector == PHAL_MFDFLIGHT_SELECTOR_0) || (bSelector == PHAL_MFDFLIGHT_SELECTOR_1) ||
        (bSelector == PHAL_MFDFLIGHT_SELECTOR_2))
    {
        /* Append LC. */
        aCmdBuff[wCmdLen++] = 2;

        /* Select MF, DF or EF, by file identifier
         * Select child DF
         * Select EF under the current DF, by file identifier
         * Select parent DF of the current DF
         */
        aFileId[1] = aCmdBuff[wCmdLen++] = pFid[1];
        aFileId[0] = aCmdBuff[wCmdLen++] = pFid[0];
        aFileId[2] = 0;
    }
    else if(bSelector == PHAL_MFDFLIGHT_SELECTOR_3)
    {
        /* For Selector as 0x03, there is nothing to be done. */
    }
    else
    {
        aCmdBuff[wCmdLen++] = 0;
    }

    /* Append LE only if there is an LC. */
    if(bSelector != PHAL_MFDFLIGHT_SELECTOR_3)
    {
        if(bExtendedLenApdu)
        {
            aCmdBuff[wCmdLen++] = 0;
        }
        aCmdBuff[wCmdLen++] = 0;
    }

    /* Exchange Cmd.ISOSelectFile information to PICC. */
    wStatus = phalMfdfLight_Sam_NonX_Int_CardExchange(
        pDataParams,
        PH_OFF,
        PH_EXCHANGE_DEFAULT,
        0,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        &pResponse,
        &wRespLen,
        aPiccStat);

    if((wStatus & PH_ERR_MASK) == PHAL_MFDFLIGHT_ERR_DF_7816_GEN_ERROR)
    {
        wVal = PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
    }

    if((wStatus == PH_ERR_SUCCESS) || (wVal == PHAL_MFDFLIGHT_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS))
    {
        /* Reset Authentication should not be targeted for elementary file selection using file ID */
        if(bSelector != PHAL_MFDFLIGHT_SELECTOR_2)
        {
            /* Reset Authentication Status here */
            PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfLight_Sam_NonX_ResetAuthentication(pDataParams));
        }

        /* once the selection Success, update the File Id to master data structure if the selection is done through AID */
        if((bSelector == PHAL_MFDFLIGHT_SELECTOR_0) || (bSelector == PHAL_MFDFLIGHT_SELECTOR_1) ||
            (bSelector == PHAL_MFDFLIGHT_SELECTOR_2))
        {
            memcpy(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aFileId, sizeof(aFileId));
        }
        else if((bSelector == PHAL_MFDFLIGHT_SELECTOR_4))
        {
            /* Update the file ID to all zeros if DF Name is of PICC. */
            if(memcmp(pDFname, aPiccDfName, 7) == 0)
            {
                aFileId[0] = 0x00;
                aFileId[1] = 0x00;
                aFileId[2] = 0x00;
            }
            else
            {
                aFileId[0] = 0xff;
                aFileId[1] = 0xff;
                aFileId[2] = 0xff;
            }

            memcpy(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aFileId, sizeof(aFileId));
        }
        else
        {
            /* Nothing for Selector 0x03. */
        }
    }
    else
    {
        return wStatus;
    }

    /* Copy the response to the buffer */
    *ppFCI = pResponse;
    *pFCILen = wRespLen;

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_NonX_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid,
    uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint32_t * pBytesRead)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccStat[2] = { 0x00, 0x00 };

#ifdef RDR_LIB_PARAM_CHECK
    /* Validate the parameter. */
    if(bSfid & 0x80)
    {
        /* Short file id is supplied */
        if((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }
#endif
    if((wOption != PH_EXCHANGE_DEFAULT) && (wOption != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Frame the command information based on the option. */
    aCmdBuff[wCmdLen++] = 0x00;
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_READ_BINARY;
    aCmdBuff[wCmdLen++] = bSfid;
    aCmdBuff[wCmdLen++] = bOffset;

    if(bExtendedLenApdu)
    {
        aCmdBuff[wCmdLen++] = (uint8_t) ((dwBytesToRead & 0x00FF0000) >> 16);
        aCmdBuff[wCmdLen++] = (uint8_t) ((dwBytesToRead & 0x0000FF00) >> 8);
    }
    aCmdBuff[wCmdLen++] = (uint8_t) (dwBytesToRead & 0x000000FF);

    /* Exchange Cmd.ISOReadBinary information to PICC. */
    wStatus = phalMfdfLight_Sam_NonX_Int_CardExchange(
        pDataParams,
        PH_OFF,
        wOption,
        0,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        ppResponse,
        &wRespLen,
        aPiccStat);

    *pBytesRead = wRespLen;

    /* Modify the chaining status. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDFLIGHT);
    }

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_NonX_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint32_t dwDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[7];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccStat[2] = { 0x00, 0x00 };

#ifdef RDR_LIB_PARAM_CHECK
    /* Validate the parameter. */
    if(bSfid & 0x80)
    {
        /* Short file id is supplied */
        if((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }
#endif

    /* Frame the command. */
    aCmdBuff[wCmdLen++] = 0x00;                                     /* CLA */
    aCmdBuff[wCmdLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_UPDATE_BINARY; /* INS */
    aCmdBuff[wCmdLen++] = bSfid;                                    /* P1 */
    aCmdBuff[wCmdLen++] = bOffset;

    if(bExtendedLenApdu)
    {
        aCmdBuff[wCmdLen++] = (uint8_t) ((dwDataLen & 0x00FF0000) >> 16);
        aCmdBuff[wCmdLen++] = (uint8_t) ((dwDataLen & 0x0000FF00) >> 8);
    }
    aCmdBuff[wCmdLen++] = (uint8_t) (dwDataLen & 0x000000FF);

    /* Exchange Cmd.ISOSUpdateBinary information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_CardExchange(
        pDataParams,
        PH_OFF,
        PH_EXCHANGE_BUFFER_CONT,
        0,
        PH_OFF,
        aCmdBuff,
        wCmdLen,
        &pResponse,
        &wRespLen,
        aPiccStat));

    wStatus = phalMfdfLight_Sam_NonX_Int_CardExchange(
        pDataParams,
        PH_OFF,
        PH_EXCHANGE_BUFFER_LAST,
        0,
        PH_OFF,
        pData,
        (uint16_t) dwDataLen,
        &pResponse,
        &wRespLen,
        aPiccStat);

    if(wStatus != PH_ERR_SUCCESS)
    {
        /* Reset authentication status */
        PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfLight_Sam_NonX_ResetAuthentication(pDataParams));
    }

    return wStatus;
}




phStatus_t phalMfdfLight_Sam_NonX_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** ppSignature)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFDFLIGHT_CMD_READ_SIG;
    aCmdBuff[1] = bAddr;

    /* Frame the communication mode to be applied. */
    bComMode = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
        PHAL_MFDFLIGHT_COMMUNICATION_ENC : PHAL_MFDFLIGHT_COMMUNICATION_PLAIN);

    /* Exchange Cmd.ReadSign information to Sam and PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ReadData(
        pDataParams,
        bComMode,
        PHAL_MFDFLIGHT_COMMUNICATION_ENC,
        aCmdBuff,
        2,
        ppSignature,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_ResetAuthentication(void * pDataParams)
{
    phalMfdfLight_Sam_NonX_Int_ResetAuthStatus(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch (wConfig)
    {
        case PHAL_MFDFLIGHT_ADDITIONAL_INFO:
            *pValue = PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_NonX_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch (wConfig)
    {
        case PHAL_MFDFLIGHT_ADDITIONAL_INFO:
            PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo = wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_NonX_CalculateTMV(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
    uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI,
    uint32_t dwTMILen, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t *   PH_MEMLOC_REM pMac = NULL;
    uint16_t    PH_MEMLOC_REM wMacLen = 0;

    uint16_t    PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t     PH_MEMLOC_REM bExchangeLen = 0;
    uint16_t    PH_MEMLOC_REM wRemLen = 0;
    uint16_t    PH_MEMLOC_REM wTMIOffset = 0;

    /* Validate the key information. */
    if((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    if(((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHALMFDFLIGHT_SAM_NONX_SESSION_TMAC_MAC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
        PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Perform MAC verification. */
    wRemLen = (uint16_t) dwTMILen;
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(wRemLen <= PHALMFDFLIGHT_SAM_DATA_FRAME_LENGTH)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) wRemLen;
        }
        else
        {
            bExchangeLen = PHALMFDFLIGHT_SAM_DATA_FRAME_LENGTH;
            wRemLen = (uint16_t) (wRemLen - PHALMFDFLIGHT_SAM_DATA_FRAME_LENGTH);
        }

        /* Exchange the TMI information to SAM. */
        wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
            PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
            wBuffOption,
            PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
            &pTMI[wTMIOffset],
            bExchangeLen,
            &pMac,
            &wMacLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the TMI offset information. */
        wTMIOffset += PHALMFDFLIGHT_SAM_DATA_FRAME_LENGTH;

    } while(!bFinished);

    /* Copy the Mac to the parameter. */
    memcpy(pTMV, pMac, wMacLen);

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_NonX_DecryptReaderID(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
    uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI,
    uint8_t * pTMRIPrev)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Validate the key information. */
    if((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    if(((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_NonX_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHALMFDFLIGHT_SAM_NONX_SESSION_TMAC_ENC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
        PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0));

    /* Exchange the TMI information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
        PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_DEFAULT,
        pEncTMRI,
        16,
        &pResponse,
        &wRespLen));

    /* Copy the decrypted information to the parameter. */
    memcpy(pTMRIPrev, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}
#endif /* NXPBUILD__PHAL_MFDFLIGHT_SAM_NONX */
