/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#ifdef NXPBUILD__PHAL_MFDFLIGHT_SAM_X

#include "../phalMfdfLight_Int.h"
#include "phalMfdfLight_Sam_X.h"
#include "phalMfdfLight_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfdfLight_SamAV3_X_Init(phalMfdfLight_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pHalSamDataParams,
	phTMIUtils_t * pTMIDataParams)
{
    /* data param check */
    if (sizeof(phalMfdfLight_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFDFLIGHT);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_AL_MFDFLIGHT);
    PH_ASSERT_NULL_PARAM (pHalSamDataParams, PH_COMP_AL_MFDFLIGHT);

    /* init private data */
	pDataParams->wId				= PH_COMP_AL_MFDFLIGHT | PHAL_MFDFLIGHT_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams	= pHalSamDataParams;
    pDataParams->pTMIDataParams		= pTMIDataParams;
    pDataParams->bAuthMode			= PHAL_MFDFLIGHT_NOT_AUTHENTICATED;
    pDataParams->bAuthType          = 0xFF;
    pDataParams->bKeyNo				= 0xFF;
    pDataParams->wAdditionalInfo	= 0x0000;

    /* Default selected application */
    memset(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, 0x00, 3); /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_AuthenticateEv2(void *pDataParams, uint8_t bAuthType, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput,
	uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bAuthMode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Check if First Auth parameter do not contain invalid values. */
	if(bAuthType > PHAL_MFDFLIGHT_AUTHFIRST_LRP)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}

	/* Validate the PCD Input Capability length */
	if(bAuthType < PHAL_MFDFLIGHT_AUTHNONFIRST_LRP)
	{
		/* For EV2 mode the length should not exceed 6 bytes. */
		if(bPcdCapsInLen > 6)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
	}
	else
	{
		/* For LRP mode the length should at least be 1 byte. */
		if((bPcdCapsInLen == 0) && (bAuthType == PHAL_MFDFLIGHT_AUTHFIRST_LRP))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
	}

#ifdef RDR_LIB_PARAM_CHECK
	if(memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00)
	{
		/* At PICC level valid key numbes are 1 - 4. */
		if((bKeyNoCard > PHAL_MFDFLIGHT_ORIGINALITY_KEY_LAST ) || (bKeyNoCard < PHAL_MFDFLIGHT_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
	}
	else
	{
		/* At App level valid key numbers are 0 - 4. */
		if((bKeyNoCard > PHAL_MFDFLIGHT_APP_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
	}

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFDFLIGHT_NO_DIVERSIFICATION) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFDFLIGHT_NO_DIVERSIFICATION) && (bDivInputLen > 31 ))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
#endif

    /* Validate SAM Key number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

	/*
	 * Set EV2 Authentication type.
	 * EVx authentication type
	 * Authenticate first
	 * Authenticate non first
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
	bAuthMode = (uint8_t) ((bAuthType & 0x01) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH : PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_NON_FIRST_AUTH);

	/* Set Auth mode with diversification enabled. */
	bAuthMode = (uint8_t) (bAuthMode | ((wOption != PHAL_MFDFLIGHT_NO_DIVERSIFICATION) ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON :
		PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF));

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_KDF_AV2);
	}

	/* Set the secure messaging. */
	if((bKeyNoCard >= PHAL_MFDFLIGHT_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFLIGHT_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00))
	{
		bAuthMode = (uint8_t) (bAuthMode | PHHAL_HW_CMD_SAMAV3_SUPPRESS_SECURE_MESSAGING);
	}

	/* Reset the Authentication state. */
	if(bAuthType & 0x01)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ResetAuthentication(pDataParams));
	}

    wStatus = phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
        bAuthMode,
        PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816,
        bKeyNoCard,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
		bPcdCapsInLen,
		bPcdCapsInLen ? pPcdCapsIn : NULL,
        pDivInput,
        bDivInputLen,
		pPdCapsOut,
		pPcdCapsOut,
		aPiccRetCode);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_ON, wStatus, aPiccRetCode));

	/* Update the authentication states. */
	PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFDFLIGHT_AUTHENTICATEEV2;
	PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = bKeyNoCard;

	/* Set the Authentication type (EV2 or LRP). */
	PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthType = (uint8_t) ((bAuthType < PHAL_MFDFLIGHT_AUTHNONFIRST_LRP) ?
		PHAL_MFDFLIGHT_AUTH_TYPE_EV2 : PHAL_MFDFLIGHT_AUTH_TYPE_LRP);

	/* Do not update the auth state if originality keys are used. */
	if((bKeyNoCard >= PHAL_MFDFLIGHT_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFDFLIGHT_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00))
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_ResetAuthentication(pDataParams));
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataSize)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[28];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if (PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFDFLIGHT);
	}
#endif

	/* Clear the local buffers. */
	memset(aAppData, 0x00, sizeof(aAppData)); /* PRQA S 3200 */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode)); /* PRQA S 3200 */

	/* Set the Length and Crypto information. */
	aAppData[wAppDataLen++] = 6;

	/* Set Extended Offset in Crypto mode. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

	/* Set Length Presence flag. */
	bLenPresent = PH_ON;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_SET_CONFIG;
	aAppData[wAppDataLen++] = bOption;

    memcpy(&aAppData[wAppDataLen], pData, bDataSize); /* PRQA S 3200 */
	wAppDataLen = (uint16_t) (wAppDataLen + bDataSize);

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;

	/* Set the Command Offset. */
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - bDataSize));
	}

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Reset the Authentication if configuration is LRP enabling. */
	if((bOption == PHAL_MFDFLIGHT_SET_CONFIG_OPTION5) && (pData[4] == 0x02))
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_ResetAuthentication(pDataParams));
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bVerLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_VERSION;

	do
	{
		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
			PH_ON,
			PH_OFF,
			0,
			PH_ON,
			0,
			aAppData,
			&wAppDataLen));

		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Updated Finished to end the loop. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			wAppDataLen = 0;
			aAppData[wAppDataLen++] = 0xAF;

			/* Copy the response information to the parameter. */
			memcpy(&pVerInfo[bVerLen], pResponse, wRespLen); /* PRQA S 3200 */
			bVerLen = (uint8_t) (bVerLen + wRespLen);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
	phalMfdfLight_Sam_X_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, bVerLen);
	*pVerLen = bVerLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_GetCardUID(void * pDataParams, uint8_t * pUid, uint8_t * pUidLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if (PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFDFLIGHT);
	}
#endif

	/* Append expected length to the command frame. */
	aAppData[wAppDataLen++] = 7;
	aAppData[wAppDataLen++] = 0;
	aAppData[wAppDataLen++] = 0;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_CARD_UID;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_ON,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the response to the parameter. */
	memcpy(pUid, pResponse, wRespLen);	/* PRQA S 3200 */
	*pUidLen = (uint8_t) wRespLen;

	/* Do a Set Config of ADDITIONAL_INFO to set  the length of the UID string */
	phalMfdfLight_Sam_X_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard,
	uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
	uint8_t		PH_MEMLOC_REM aPiccRetCode[2];

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00)
    {
		/* At PICC level ChangeKey cannot be performed. */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
    else
    {
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNoCard > PHAL_MFDFLIGHT_APP_KEY_LAST)
		{
			/* Invalid application key specified */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
    }

	if (PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFDFLIGHT);
	}
#endif

    /* Check for valid SAM key number and version. */
    if ((wCurrKeyNo > 0x7f) || (wCurrKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Clear the PICC status code buffer. */
	memset(aPiccRetCode, 0x00, sizeof(aPiccRetCode));	/* PRQA S 3200 */

    /* Set the key compilation method. */
	if(wOption == PHAL_MFDFLIGHT_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
#ifdef RDR_LIB_PARAM_CHECK
		/* Change for valid diversification options. */
		if ((wOption != PHAL_MFDFLIGHT_NO_DIVERSIFICATION) &&
			((wOption & PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2) != PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}

		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
#endif

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Desfire key number to be changed. */
	bCfg = bKeyNoCard;

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY);
    }

	/* Enable Iso7816 wrapping. */
    bCfg = (uint8_t) (bCfg | PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816);

	/* Exchange the command to PICC. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		bKeyCompMeth,
		bCfg,
		0x00,
		0x00,
		(uint8_t)wCurrKeyNo,
		(uint8_t)wCurrKeyVer,
		(uint8_t)wNewKeyNo,
		(uint8_t)wNewKeyVer,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

    /* Evaluate the response. */
	wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Kill the PICC Authentication. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
			PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

		return wStatus;
	}
	else
	{
		/* Reset authentication status only if the key authenticated with is changed. */
		if ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfLight_Sam_X_ResetAuthentication(pDataParams));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion, uint8_t * pKeyVerLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aAppId, 3) == 0x00)
    {
		/* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
		if((bKeyNo > PHAL_MFDFLIGHT_ORIGINALITY_KEY_LAST ) || (bKeyNo < PHAL_MFDFLIGHT_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
    }
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if(bKeyNo > PHAL_MFDFLIGHT_APP_KEY_LAST)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}
	}
#endif

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_KEY_VERSION;
	aAppData[wAppDataLen++] = bKeyNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pKeyVersion, pResponse, wRespLen);	/* PRQA S 3200 */
	*pKeyVerLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_CreateTransactionMacFile(void * pDataParams, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint16_t wKeyNo, uint8_t bKeyVer,
	uint8_t bKeyType, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aPiccRetCode[2];

#ifdef RDR_LIB_PARAM_CHECK
    /* Check for valid file no. and KeyType == AES128 */
	if (bKeyType != PHAL_MFDFLIGHT_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    if ((bFileOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
		(bFileOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN_1) &&
        (bFileOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bFileOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Check for valid SAM keystore number and version. */
    if (wKeyNo > 0x7f)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	wStatus = phhalHw_SamAV3_Cmd_DESFire_CreateTMFilePICC(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		(uint8_t) (bDivInputLen ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON : PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF),
		PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816,
		(uint8_t) wKeyNo,
		bKeyVer,
		bFileNo,
		(uint8_t) (bFileOption >> 4),
		pAccessRights,
		bKeyType,
		pDivInput,
		bDivInputLen,
		aPiccRetCode);

	/* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_ON, wStatus, aPiccRetCode));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_DeleteFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_DELETE_FILE;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto Mode. */
	bCrypto = (uint8_t) ((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) ?
		PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC : PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN);

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_GetFileIDs(void * pDataParams, uint8_t * pFid, uint8_t * bNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_FILE_IDS;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);
	if((wStatus == PH_ERR_SUCCESS) || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
	{
		/* Copy the data to the parameter */
		memcpy(pFid, pResponse, wRespLen); /* PRQA S 3200 */
		*bNumFIDs = (uint8_t) wRespLen;
	}

	return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_GetISOFileIDs(void * pDataParams, uint8_t * pFidBuffer, uint8_t * bNumFIDs)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[6];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_ISO_FILE_IDS;

	/* Set the number of FileID variable. */
	*bNumFIDs = 0;

	do
	{
		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
			PH_ON,
			PH_OFF,
			0,
			PH_ON,
			0,
			aAppData,
			&wAppDataLen));

		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS(wStatus);
		}
		else
		{
			/* Update Command frame. */
			wAppDataLen = 0;
			aAppData[wAppDataLen++] = 0xAF;

			/* Copy the data to the parameter */
			memcpy(&pFidBuffer[*bNumFIDs * 2], pResponse, wRespLen);	/* PRQA S 3200 */
			*bNumFIDs += (uint8_t) (wRespLen / 2);

			/* Update Finish flag. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
			{
				bFinished = 1;
			}
		}
	}while(!bFinished);

	return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	memcpy(pFSBuffer, pResponse, wRespLen);	/* PRQA S 3200 */
	*pBufferLen = (uint8_t) wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t bTMCLimitLen,
	uint8_t * pTMCLimit)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[16];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;

	/* Frame the Crypto information.
	 * Masking the MSB bit becuase its not support by SAM.
	 * Its actually supported in Software only.
	 */
	bCrypto = (uint8_t) (bOption & 0xFE);

	/* Validate the parameters */
	if ((bCrypto != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCrypto != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCrypto != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Set the Length and Crypto information. */
	if((PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2) &&
		(bCrypto == PHAL_MFDFLIGHT_COMMUNICATION_ENC))
	{
		/* Start of data. */
		aAppData[wAppDataLen++] = 6;

		/* Set Extended Offset in Crypto mode. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_CHANGE_FILE_SETTINGS;
	aAppData[wAppDataLen++] = bFileNo;
	aAppData[wAppDataLen++] = bFileOption;

    /* Append access rights. */
	memcpy(&aAppData[wAppDataLen], pAccessRights, 2); /* PRQA S 3200 */
	wAppDataLen += 2;

	/* TMCLimit buffer in command buffer if Bit5 of File Option is SET. */
	if (bFileOption & PHAL_MFDFLIGHT_TMCLIMITCONFIG)
    {
		memcpy(&aAppData[wAppDataLen], pTMCLimit, bTMCLimitLen); /* PRQA S 3200 */
        wAppDataLen += bTMCLimitLen;
    }

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse,
	uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint32_t	PH_MEMLOC_REM dwLength = 0;

	uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

	/* Validate the parameter. */
    if (((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Frame the Crypto information. */
	bCrypto = (uint8_t) (bOption & 0xF0);

	/* Frame Presence of length information in the command frame.
	* The first three bytes specifies number of bytes to be received from PICC.
	*/
	bLenPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

	/* Add Read Length if communication mode is Encrypted. */
	if(bLenPresent)
	{
		memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
		wAppDataOffset = 3;
	}

    aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_READ_DATA_ISO;
    aAppData[wAppDataLen++] = bFileNo;

    memcpy(&aAppData[wAppDataLen], pOffset, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

    memcpy(&aAppData[wAppDataLen], pLength, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Compute the length. */
		dwLength = pLength[2];
		dwLength = dwLength << 8 | pLength[1];
		dwLength = dwLength << 8 | pLength[0];

		/* Frame the Option. */
		bTMIOption = (uint8_t)(dwLength ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

		/* Buffer the Command information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		((pLength[0] | (pLength[1] << 8) | (pLength[2] << 8)) + 7),
		aAppData,
		&wAppDataLen));

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pRespLen += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Frame the Option. */
		bTMIOption = (uint8_t) (dwLength ? PH_TMIUTILS_ZEROPAD_DATABUFF : (PH_TMIUTILS_ZEROPAD_DATABUFF | PH_TMIUTILS_READ_INS));

		/* Collect the data received. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, NULL, 0, *ppResponse, *pRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));

		/* Reset the TMI buffer Offset. */
		if (!dwLength)
		{
			/* Reset wOffsetInTMI */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFDFLIGHT_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

	/* Validate the parameters */
#ifdef RDR_LIB_PARAM_CHECK
	if ((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 12;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_WRITE_DATA_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);
	wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			(PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset),
			pData, dwTotLen, PHAL_MFDFLIGHT_BLOCK_SIZE));
	}

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
				PH_ON,
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 7),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) && bIsFirstFrame &&
			(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_GetValue(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[11];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters. */
	if ((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Frame the Crypto information. */
	bCrypto = bCommOption;

	/* Append expected length to the command frame. */
	if(bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC)
	{
		aAppData[wAppDataLen++] = 4;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 3;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_GET_VALUE;
	aAppData[wAppDataLen++] = bFileNo;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_NO_PADDING, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	 /* Copy the value. */
    memcpy(pValue, pResponse, wRespLen); /* PRQA S 3200 */

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, pValue, wRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_Credit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
    if ((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Update Offset information in case FULL Mode. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 6;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_CREDIT;
	aAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue,
			4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Append data. */
	memcpy(&aAppData[wAppDataLen], pValue, 4); /* PRQA S 3200 */
	wAppDataLen += 4;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto = bCommOption;

	/* Set ExtendedOffset for Full mode and ISO Chaining. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		/* Set the Extended Offset bit. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
	}
	else
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4 /* Data Length */));
	}

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_Debit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
    if ((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Update Offset information in case FULL Mode. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 6;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_DEBIT;
	aAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue,
			4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Append data. */
	memcpy(&aAppData[wAppDataLen], pValue, 4); /* PRQA S 3200 */
	wAppDataLen += 4;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto = bCommOption;

	/* Set ExtendedOffset for Full mode and ISO Chaining. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		/* Set the Extended Offset bit. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
	}
	else
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4 /* Data Length */));
	}

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_LimitedCredit(void * pDataParams, uint8_t bCommOption, uint8_t bFileNo, uint8_t * pValue)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
    if ((bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Update Offset information in case FULL Mode. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 6;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_LIMITED_CREDIT;
	aAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), pValue,
			4, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

    /* Append data. */
	memcpy(&aAppData[wAppDataLen], pValue, 4); /* PRQA S 3200 */
	wAppDataLen += 4;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		1,
		PH_OFF,
		0,
		aAppData,
		&wAppDataLen));

	/* Frame the Crypto information. */
	bCrypto = bCommOption;

	/* Set ExtendedOffset for Full mode and ISO Chaining. */
	if((bCommOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		/* Set the Extended Offset bit. */
		bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
	}
	else
	{
		bCrypto = (uint8_t) (bCrypto + (wAppDataLen - 4 /* Data Length */));
	}

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_ReadRecords(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo, uint8_t * pRecCount, uint8_t * pRecSize,
	uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
	uint32_t	PH_MEMLOC_REM dwNumRec = 0;
	uint32_t	PH_MEMLOC_REM dwRecLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	uint8_t		PH_MEMLOC_REM bTMIOption = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;
	uint32_t	PH_MEMLOC_REM dwDataLen = 0;
    uint32_t    PH_MEMLOC_REM dwNumRecCal = 0;
	uint32_t	PH_MEMLOC_REM dwTMIBufOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTMIBufIndex = 0;

	/* Validate the parameter. */
    if (((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Frame the Crypto information. */
	bCrypto = (uint8_t) (bOption & 0xF0);

	/* Frame Presence of length information in the command frame.
	* The first three bytes specifies number of bytes to be received from PICC.
	*/
	bLenPresent = (uint8_t) (((bOption & 0xF0) == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? PH_ON : PH_OFF);

	/* Add Read Length if communication mode is Encrypted. */
	if(bLenPresent)
	{
		memcpy(&aAppData[wAppDataLen], pRecSize, 3); /* PRQA S 3200 */
		wAppDataLen += 3;
		wAppDataOffset = 3;
	}

    aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_READ_RECORDS_ISO;
    aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pRecNo, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pRecCount, 3); /* PRQA S 3200 */
    wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Compute the number of records. */
		dwNumRec = pRecCount[2];
		dwNumRec = dwNumRec << 8 | pRecCount[1];
		dwNumRec = dwNumRec << 8 | pRecCount[0];

		/* Compute the record length. */
		dwRecLen = pRecSize[2];
		dwRecLen = dwRecLen << 8 | pRecSize[1];
		dwRecLen = dwRecLen << 8 | pRecSize[0];

		/* Should should provide atleast RecLen / NumRec to update in TMI collection */
		if(!dwRecLen && !dwNumRec)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
		}

		/* Frame the Option. */
		bTMIOption = (uint8_t)(dwNumRec ? PH_TMIUTILS_ZEROPAD_CMDBUFF : (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF));

		/* Buffer the Command information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			bTMIOption, &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset), NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		(pRecSize[0] | (pRecSize[1] << 8) | (pRecSize[2] << 8)) + 7,
		aAppData,
		&wAppDataLen));

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pRespLen += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		if (!dwNumRec)
		{
			/* Get the TMI buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, &dwTMIBufOffset));

			/* Get the current TMI buffer index. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_BUFFER_INDEX, &dwTMIBufIndex ));

			/* Calculate Response length in case of chaining. */
			dwDataLen = *pRespLen + dwTMIBufIndex -(dwTMIBufOffset + 11);

			/* If user update worng RecSize, we cant calculate recCnt. */
			if(dwDataLen % dwRecLen)
			{
				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDFLIGHT);
			}

			/* Calculate number of records. */
			dwNumRecCal = dwDataLen / dwRecLen ;

			/* Update the value as TMI buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_VALUE, dwNumRecCal));
		}

		/* Collect the data received. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_DATABUFF, NULL, 0, *ppResponse, *pRespLen, PHAL_MFDFLIGHT_BLOCK_SIZE));

		/* Reset the TMI buffer Offset. */
		if (!dwNumRec)
		{
			/* Reset wOffsetInTMI */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
		}

        if (!dwNumRec)
        {
			/* Reset TMI Buffer offset. */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH, 0));
        }
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_WriteRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFDFLIGHT_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
    if ((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 12;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_WRITE_RECORD_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);
	wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			(PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset),
			pData, dwTotLen, PHAL_MFDFLIGHT_BLOCK_SIZE));
	}

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 8) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
				PH_ON,
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 7),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 8 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) && bIsFirstFrame &&
			(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_UpdateRecord(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pRecNo, uint8_t * pOffset, uint8_t * pData,
	uint8_t * pDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBufferOption = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[PHAL_MFDFLIGHT_MAX_WRITE_LEN + 20];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t    PH_MEMLOC_REM wAppDataOffset = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint8_t		PH_MEMLOC_REM bIsFirstFrame = PH_ON;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint32_t	PH_MEMLOC_REM dwTotLen = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;
	uint8_t		PH_MEMLOC_REM bExchangeComplete = 0;
    uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameters */
	if ((bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFDFLIGHT_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
#endif

	/* Compute the maximum length. */
	dwTotLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Update Offset information in case FULL Mode. */
	if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) &&
		(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
	{
		aAppData[wAppDataLen++] = 15;

		/* Set Length Presence flag. */
		bLenPresent = PH_ON;

		/* Set the Application data offset. This should not go for TMI collection. */
		wAppDataOffset = 1;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_UPDATE_RECORD_ISO;
	aAppData[wAppDataLen++] = bFileNo;

	memcpy(&aAppData[wAppDataLen], pRecNo, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pOffset, 3);
	wAppDataLen += 3;

	memcpy(&aAppData[wAppDataLen], pDataLen, 3);
	wAppDataLen += 3;

	/* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			(PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF), &aAppData[wAppDataOffset], (uint16_t) (wAppDataLen - wAppDataOffset),
			pData, dwTotLen, PHAL_MFDFLIGHT_BLOCK_SIZE));
	}

	/* Set the buffering flag. */
	wBufferOption = (uint16_t) (PH_EXCHANGE_DEFAULT | PH_EXCHANGE_TXCHAINING);

	/* Set the lengths. */
	dwRemLen = dwTotLen;

	do
	{
		/* Compute the maximum data to be exchanged.  */
		if(bIsFirstFrame)
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 11) /* The command header information is available. */;
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 4) : bDataLen);
			bDataLen = (uint8_t) ((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) ? (bDataLen - 16) : (bDataLen - 20));
		}
		else
		{
			bDataLen = (uint8_t) (PHAL_MFDFLIGHT_MAX_WRITE_LEN - 5);
		}

		/* Set the completion flag. */
		if(dwRemLen <= bDataLen)
		{
			bDataLen = (uint8_t) dwRemLen;
			wBufferOption = PH_EXCHANGE_DEFAULT;
			bExchangeComplete = PH_ON;
		}

		/* Copy the data to the buffer. */
		memcpy(&aAppData[wAppDataLen], &pData[bOffset], bDataLen);
		wAppDataLen += bDataLen;

		/* Wrap the command if required. */
		if(bIsFirstFrame)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
				PH_ON,
				bLenPresent,
				1,
				PH_OFF,
				(dwTotLen + 10),
				aAppData,
				&wAppDataLen));
		}

		/* Frame the Crypto information. */
		bCrypto = (uint8_t) (bOption | PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_ISO_CHAINING);

		/* Set the Offset information for MAC and FULL mode. */
		if(bOption != PHAL_MFDFLIGHT_COMMUNICATION_PLAIN)
		{
			bCrypto |= (uint8_t) (bIsFirstFrame ? 11 : 0);
			bCrypto |= (uint8_t) (bCrypto + 4);

			/* Reset the Crypto to zero . */
			if(!bIsFirstFrame)
			{
				bCrypto = 0;
			}
		}

		/* Set ExtendedOffset for Full mode and ISO Chaining. */
		if((bOption == PHAL_MFDFLIGHT_COMMUNICATION_ENC) && bIsFirstFrame &&
			(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2))
		{
			/* Mask out the previously set offset. */
			bCrypto = (uint8_t) (bCrypto & 0xF0);

			/* Set the Extended Offset bit. */
			bCrypto |= PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET;
		}

		/* Clear the first frame and Length Presence flag. */
		bIsFirstFrame = PH_OFF;
		bLenPresent = PH_OFF;

		/* Enchange the information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBufferOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Evaluate the response. */
		wStatus = phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode);

		/* Return the status if not SUCCESS or SUCCESS_CHAINING. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			return wStatus;
		}

		/* Update length. */
		bOffset += bDataLen;
		dwRemLen = (uint32_t) (dwRemLen - bDataLen);

		/* Set the Chaining satus. */
		wAppDataLen = 0;

	}while(!bExchangeComplete);

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_ClearRecordFile(void * pDataParams, uint8_t bFileNo)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_CLEAR_RECORDS_FILE;
	aAppData[wAppDataLen++] = bFileNo;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if (dwTMIStatus)
	{
		/* Buffer the Command and Data information to TMI buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
			PH_TMIUTILS_ZEROPAD_CMDBUFF, aAppData, 2, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
    }

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_CommitTransaction(void * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMV)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if ((bOption & 0x0F) > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }
#endif

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_COMMIT_TXN;

	/*
	 * 0x01 and 0x00 are values to support the legacy implementation.
	 *		Here if it 0x00 the Option is not exchanged to the PICC. Its only exchanged for 0x01
	 *
	 * 0x80 and 0x81 are the values to exchange the Option by to PICC.
	 *		If its 0x80, Option byte will be exchanged by the Option information will be zero.
	 *		If its 0x81, Option byte will be exchanged by the Option information will be one.
	 */
	if((bOption & 0x80) || (bOption & 0x0F))
	{
		aAppData[wAppDataLen++] = (bOption & 0x0F);
	}

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

	/* Copy the data to the parameter */
	if(bOption)
	{
		memcpy(pTMC, &pResponse[0], 4);	/* PRQA S 3200 */
		memcpy(pTMV, &pResponse[4], 8);	/* PRQA S 3200 */
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_AbortTransaction(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[8];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_MAC;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_ABORT_TXN;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		PH_OFF,
		0,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_CommitReaderID(void * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aAppData[26];
    uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wEncTMRILen = 0;
	uint8_t		PH_MEMLOC_REM aPiccErrCode[2];
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint32_t	PH_MEMLOC_REM dwTMIStatus = 0;

   /* Get the TMI Status. */
	PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Exchange the details to SAM hardware and get the ENC TMRI. */
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_AUTHENTICATEEV2)
	{
		wStatus = phhalHw_SamAV3_Cmd_TMRI_CommitReaderID(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PHHAL_HW_CMD_SAMAV3_TMRI_ISO_MODE_ISO7816,
			PHHAL_HW_CMD_SAMAV3_COMMIT_READER_ID_PICC_STATE_DESFIRE,
			0,
			&pResponse,
			&wEncTMRILen,
			aPiccErrCode);

		/* Copy the Response to the parameter. */
		memcpy(pEncTMRI, pResponse, wEncTMRILen);	/* PRQA S 3200 */
		wRespLen = wEncTMRILen;

		/* Check TMI Collection Status */
		if (dwTMIStatus && (wStatus == PH_ERR_SUCCESS))
		{
			/* If authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
			 * TMI = TMI || Cmd || TMRICur || EncTMRI || ZeroPadding
			 */

			/* Clear the buffers. */
			wAppDataLen = 0;
			memset(aAppData, 0x00, sizeof(aAppData));	/* PRQA S 3200 */

			/* Frame the command and send it to card. */
			aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_COMMIT_READER_ID;

			/* Frame the TMRI information. */
			memcpy(&aAppData[wAppDataLen], PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams->bUid, 7 );	/* PRQA S 3200 */
			wAppDataLen += 7;

			aAppData[wAppDataLen] = 0x80;

			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_DATABUFF, aAppData, 17, pEncTMRI, 16, PHAL_MFDFLIGHT_BLOCK_SIZE));
		}
	}
	else
	{
		/* Clear the buffers. */
		wAppDataLen = 0;
		memset(aAppData, 0x00, sizeof(aAppData));	/* PRQA S 3200 */

		/* Frame the command and send it to card. */
		aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_COMMIT_READER_ID;

		/* Add the TMRI to command buffer. */
		memcpy(&aAppData[wAppDataLen], pTMRI, 16 );	/* PRQA S 3200 */
		wAppDataLen += 16;

		/* Check TMI Collection Status */
		if (dwTMIStatus)
		{
			/* If not authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
			 * TMI = TMI || Cmd || TMRICur || ZeroPadding
			 */
			PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_CMDBUFF, aAppData, wAppDataLen, NULL, 0, PHAL_MFDFLIGHT_BLOCK_SIZE));
		}

		/* Wrap the command if required. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
			PH_ON,
			PH_OFF,
			0,
			PH_ON,
			(uint16_t) (wAppDataLen - 1),
			aAppData,
			&wAppDataLen));

		/* Exchange the data to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_DEFAULT,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen);

		/* Set the PICC error code. */
		aPiccErrCode[0] = pResponse[wRespLen];
		aPiccErrCode[1] = pResponse[wRespLen - 1];

		/* Manipulate the status. */
		if((pResponse[wRespLen] != 0x91) && (pResponse[wRespLen - 1] != 0x00))
		{
			wStatus = PH_ADD_COMPCODE(PHHAL_HW_SAMAV3_ERR_MIFARE_GEN, PH_COMP_HAL);
		}

		/* Decrement the RespLen to remove the status. */
		wRespLen -= 2;
	}

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccErrCode));

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the recieved TMRI */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_SetConfig(pDataParams, PHAL_MFDFLIGHT_ADDITIONAL_INFO, wRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aAppData[25];
	uint16_t	PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aFileId[3] = {'\0'};
	uint16_t    PH_MEMLOC_REM wVal = 0;
	uint8_t		PH_MEMLOC_REM aPiccDfName[7] = {0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00};

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if (bDFnameLen > 16)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
	/* If User intends to select through DFName, check for validity of DFName passed */
	if((bSelector ==  PHAL_MFDFLIGHT_SELECTOR_4) && (bDFnameLen == 0x00))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
	if ((bOption != PHAL_MFDFLIGHT_FCI_RETURNED) && (bOption != PHAL_MFDFLIGHT_FCI_NOT_RETURNED))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
	if((bSelector != PHAL_MFDFLIGHT_SELECTOR_4) && (bSelector != PHAL_MFDFLIGHT_SELECTOR_3) && (pFid == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
	}
#endif

	/* Frame the command. */
	aAppData[wAppDataLen++] = 0x00;
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_SELECT_FILE;
	aAppData[wAppDataLen++] = bSelector;
	aAppData[wAppDataLen++] = bOption;

	/* Append LC. */
	if(bSelector != PHAL_MFDFLIGHT_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
			aAppData[wAppDataLen++] = 0;
		}
	}

	/* Append the payload and LC. */
	if(bSelector == PHAL_MFDFLIGHT_SELECTOR_4)
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = bDFnameLen;

		memcpy(&aAppData[wAppDataLen], pDFname, bDFnameLen); /* PRQA S 3200 */
		wAppDataLen += bDFnameLen;
	}
	else if((bSelector ==  PHAL_MFDFLIGHT_SELECTOR_0) || (bSelector == PHAL_MFDFLIGHT_SELECTOR_1) ||
		(bSelector == PHAL_MFDFLIGHT_SELECTOR_2))
	{
		/* Append LC. */
		aAppData[wAppDataLen++] = 2;

		/* Select MF, DF or EF, by file identifier
		 * Select child DF
		 * Select EF under the current DF, by file identifier
		 * Select parent DF of the current DF
		 */
		aFileId[1] = aAppData[wAppDataLen++] = pFid[1];
		aFileId[0] = aAppData[wAppDataLen++] = pFid[0];
		aFileId[2] = 0;
	}
	else if(bSelector == PHAL_MFDFLIGHT_SELECTOR_3)
	{
		/* For Selector as 0x03, there is nothing to be done. */
	}
	else
	{
		aAppData[wAppDataLen++] = 0;
	}

	/* Append LE only if there is an LC. */
	if(bSelector != PHAL_MFDFLIGHT_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aAppData[wAppDataLen++] = 0;
		}
		aAppData[wAppDataLen++] = 0;
	}

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		aAppData,
		(uint8_t) wAppDataLen,
		&pResponse,
		&wRespLen));

	/* Manipulate the status. */
	wStatus = phalMfdfLight_Int_ComputeErrorResponse(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams),
		((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]));

	/* Decrement the RespLen to remove the status. */
	wRespLen -= 2;

	if((wStatus & PH_ERR_MASK) == PHAL_MFDFLIGHT_ERR_DF_7816_GEN_ERROR)
	{
		wVal = PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
	}

	if((wStatus == PH_ERR_SUCCESS) || (wVal == PHAL_MFDFLIGHT_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS))
	{
		/* Reset Authentication should not be targeted for elementary file selection using file ID */
		if(bSelector !=  PHAL_MFDFLIGHT_SELECTOR_2)
		{
			/* Reset Authentication Status here */
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfdfLight_Sam_X_ResetAuthentication(pDataParams));
		}

		/* once the selection Success, update the File Id to master data structure if the selection is done through AID */
		if((bSelector ==  PHAL_MFDFLIGHT_SELECTOR_0) || (bSelector == PHAL_MFDFLIGHT_SELECTOR_1) ||
			(bSelector == PHAL_MFDFLIGHT_SELECTOR_2))
		{
			memcpy(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else if((bSelector ==  PHAL_MFDFLIGHT_SELECTOR_4))
		{
			/* Update the file ID to all zeros if DF Name is of PICC. */
			if(memcmp( pDFname, aPiccDfName, 7) == 0)
			{
				aFileId[0] = 0x00;
				aFileId[1] = 0x00;
				aFileId[2] = 0x00;
			}
			else
			{
				aFileId[0] = 0xff;
				aFileId[1] = 0xff;
				aFileId[2] = 0xff;
			}

			memcpy(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
		}
		else
		{
			/* Nothing for Secector 0x03. */
		}
	}
	else
	{
		return wStatus;
	}

	/* Copy the response to the buffer */
	*ppFCI = pResponse;
	*pFCILen = wRespLen;

	return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_IsoReadBinary(void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu,
	uint8_t ** ppResponse, uint16_t * pBytesRead)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t		PH_MEMLOC_REM bBufferResp = PH_ON;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Validate the parameter. */
	if (wOption == PH_EXCHANGE_DEFAULT)
	{
		if (bSfid & 0x80)
		{
			/* Short file id is supplied */
			if ((bSfid & 0x7FU) > 0x1F)
			{
				/* Error condition */
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
			}
		}
	}

	if ((wOption != PH_EXCHANGE_DEFAULT) && (wOption != PH_EXCHANGE_RXCHAINING))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;

	/* Frame the command information based on the option. */
	aAppData[wAppDataLen++] = 0x00;
    aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_READ_BINARY;
    aAppData[wAppDataLen++] = bSfid;
    aAppData[wAppDataLen++] = bOffset;

	if(bExtendedLenApdu)
	{
		aAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x00FF0000) >> 16);
		aAppData[wAppDataLen++] = (uint8_t) ((dwBytesToRead & 0x0000FF00) >> 8);
	}
	aAppData[wAppDataLen++] = (uint8_t) (dwBytesToRead & 0x000000FF);

	/* Frame the Buffer Option. */
	wBuffOption = (uint16_t) (wOption | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

	do
	{
		/* Exchange the information to Sam. */
		wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
			wBuffOption,
			bCrypto,
			aAppData,
			(uint8_t) wAppDataLen,
			&pResponse,
			&wRespLen,
			aPiccRetCode,
			&bPiccRetCodeLen);

		/* Update the Buffer Option. */
		wBuffOption = (uint16_t) (PH_EXCHANGE_RXCHAINING | PHHAL_HW_CMD_SAMAV3_ISO_CHAINING);

		/* End the loop. */
		if((wStatus != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}
		else
		{
			/* End the loop if the status is success or PICC has provided a Chainng status. */
			if(wStatus == PH_ERR_SUCCESS)
			{
				bFinished = PH_ON;
			}

			/* Copy the data. */
			*pBytesRead += wRespLen;
			if(bBufferResp)
			{
				*ppResponse = pResponse;
				bBufferResp = PH_OFF;
			}
			else
			{
				memcpy(&ppResponse[0][wOffset], pResponse, wRespLen);
			}
			wOffset += wRespLen;

			/* Update the HAL buffer start position. */
			if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
					PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, wOffset));
			}

			/* Reset the Application Length. */
			wAppDataLen = 0;
		}
	}while(!bFinished);

	/* Update the HAL buffer start position to 0. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PHHAL_HW_CONFIG_RXBUFFER_STARTPOS, 0));

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_IsoUpdateBinary(void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData, uint32_t dwDataLen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
    uint8_t     PH_MEMLOC_REM aAppData[7];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Validate the parameters */
	if (bSfid & 0x80)
    {
        /* Short file id is supplied */
        if ((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
        }
    }

	/* Frame the command information. */
	aAppData[wAppDataLen++] = 0x00;
    aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_ISO7816_UPDATE_BINARY;
    aAppData[wAppDataLen++] = bSfid;
    aAppData[wAppDataLen++] = bOffset;

	if(bExtendedLenApdu)
	{
		aAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x00FF0000) >> 16);
		aAppData[wAppDataLen++] = (uint8_t) ((dwDataLen & 0x0000FF00) >> 8);
	}
	aAppData[wAppDataLen++] = (uint8_t) (dwDataLen & 0x000000FF);

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;

	/* Exchange the information to Sam. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		NULL,
		NULL));

	wStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		bCrypto,
		pData,
		(uint8_t) dwDataLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Convert the PICC status and validate it. */
	if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN)
	{
		wPiccErrCode = (phStatus_t) ((aPiccRetCode[0] << 8) | aPiccRetCode[1]);
		PH_CHECK_SUCCESS_FCT( wStatus, phalMfdfLight_Int_ComputeErrorResponse(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams), wPiccErrCode));
	}

    return wStatus;
}




phStatus_t phalMfdfLight_Sam_X_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCrypto = 0;
	uint8_t     PH_MEMLOC_REM bLenPresent = 0;
    uint8_t     PH_MEMLOC_REM aAppData[12];
	uint16_t    PH_MEMLOC_REM wAppDataLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM aPiccRetCode[2];
	uint8_t		PH_MEMLOC_REM bPiccRetCodeLen = 0;

	/* Frame the Crypto information. */
	bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL;
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFDFLIGHT_NOT_AUTHENTICATED)
	{
		bCrypto = PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_PLAIN;
	}

	/* Add Length information is Communication mode is FULL. */
	if(bCrypto == PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL)
	{
		aAppData[wAppDataLen++] = 56;
		aAppData[wAppDataLen++] = 0;
		aAppData[wAppDataLen++] = 0;

		/* Set presence of Length information. */
		bLenPresent = PH_ON;
	}

	/* Frame the command information. */
	aAppData[wAppDataLen++] = PHAL_MFDFLIGHT_CMD_READ_SIG;
	aAppData[wAppDataLen++] = bAddr;

	/* Wrap the command if required. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_Iso7816Wrap(
		PH_ON,
		bLenPresent,
		3,
		PH_ON,
		0,
		aAppData,
		&wAppDataLen));

	/* Exchange the information to Sam. */
	wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(
		PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->pHalSamDataParams,
		PH_EXCHANGE_DEFAULT,
		bCrypto,
		aAppData,
		(uint8_t) wAppDataLen,
		pSignature,
		&wRespLen,
		aPiccRetCode,
		&bPiccRetCodeLen);

    /* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ValidateResponse(pDataParams, PHAL_MFDFLIGHT_RESET_AUTH_OFF, wStatus, aPiccRetCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_ResetAuthentication(void * pDataParams)
{
    return phalMfdfLight_Sam_X_Int_ResetAuthentication(pDataParams);
}

phStatus_t phalMfdfLight_Sam_X_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFDFLIGHT_ADDITIONAL_INFO:
			*pValue = PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch (wConfig)
    {
		case PHAL_MFDFLIGHT_ADDITIONAL_INFO:
			PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo = wValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}




phStatus_t phalMfdfLight_Sam_X_CalculateTMV(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
	uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI, uint32_t dwTMILen, uint8_t * pTMV)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pMac = NULL;
	uint16_t	PH_MEMLOC_REM wMacLen = 0;

	uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
	uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
	uint16_t	PH_MEMLOC_REM wRemLen = 0;
	uint16_t	PH_MEMLOC_REM wTMIOffset = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Derive Transaction MAC (KSesTMMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ComputeTMACSessionVectors(
		pDataParams,
		PHAL_MFDFLIGHT_SAM_NONX_SESSION_TMAC_MAC,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pTMC,
		pUid,
		bUidLen));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Perform MAC verification. */
	wRemLen = (uint16_t) dwTMILen;
	wBuffOption = PH_EXCHANGE_TXCHAINING;

	do
	{
		/* Update the finished flag and buffering option. */
		if(wRemLen <= PHAL_MFDFLIGHT_SAM_DATA_FRAME_LENGTH)
		{
			bFinished = PH_ON;
			wBuffOption = PH_EXCHANGE_DEFAULT;
			bExchangeLen = (uint8_t) wRemLen;
		}
		else
		{
			bExchangeLen = PHAL_MFDFLIGHT_SAM_DATA_FRAME_LENGTH;
			wRemLen = (uint16_t) (wRemLen - PHAL_MFDFLIGHT_SAM_DATA_FRAME_LENGTH);
		}

		/* Exchange the TMI information to SAM. */
		wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
			PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
			&pTMI[wTMIOffset],
			bExchangeLen,
			&pMac,
			&wMacLen);

		/* Validate the response. */
		if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			bFinished = PH_ON;
		}

		/* Update the TMI offset information. */
		wTMIOffset += PHAL_MFDFLIGHT_SAM_DATA_FRAME_LENGTH;

	}while(!bFinished);

	/* Copy the Mac to the parameter. */
	memcpy(pTMV, pMac, wMacLen);	/* PRQA S 3200 */

	return wStatus;
}

phStatus_t phalMfdfLight_Sam_X_DecryptReaderID(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
	uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI, uint8_t * pTMRIPrev)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDFLIGHT);
    }

	/* Derive Transaction MAC (KSesTMMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfdfLight_Sam_X_Int_ComputeTMACSessionVectors(
		pDataParams,
		PHAL_MFDFLIGHT_SAM_NONX_SESSION_TMAC_ENC,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pTMC,
		pUid,
		bUidLen));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Exchange the TMI information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		pEncTMRI,
		16,
		&pResponse,
		&wRespLen));

	/* Copy the decrypted information to the parameter. */
	memcpy(pTMRIPrev, pResponse, wRespLen);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

#endif /* NXPBUILD__PHAL_MFDFLIGHT_SAM_X */
