/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PHAL_MFDFLIGHT_SAM_X

#include "../phalMfdfLight_Int.h"
#include "phalMfdfLight_Sam_X.h"
#include "phalMfdfLight_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfdfLight_Sam_X_Int_ValidateResponse(void * pDataParams, uint8_t bResetAuth, uint16_t wStatus, uint8_t * pPiccRetCode)
{
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	phStatus_t	PH_MEMLOC_REM wStatus2 = 0;
	uint16_t	PH_MEMLOC_REM wPiccStat = 0;
	uint16_t	PH_MEMLOC_REM wPiccRetCode = 0;

	/* Evaluate the response. */
	if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN) ||
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN))
	{
		/* Evaluate the PICC Status. */
		wPiccRetCode = (uint16_t) ((pPiccRetCode[0] << 8) | pPiccRetCode[1]);
		wPiccStat = (uint16_t) (((wPiccRetCode & 0xFF00) == 0x9100) ? (wPiccRetCode & 0x00FF) : wPiccRetCode);

		/* Validate the PICC Status. */
		wStatus1 = phalMfdfLight_Int_ComputeErrorResponse(pDataParams, wPiccStat);

		/* Reset the Authentication if set. */
		if(bResetAuth)
		{
			PH_CHECK_SUCCESS_FCT(wStatus2, phalMfdfLight_Sam_X_Int_ResetAuthentication(pDataParams));
		}
	}
	else if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
	{
		wStatus1 = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDFLIGHT);
	}
	else
	{
		wStatus1 = wStatus;
	}

	return wStatus1;
}

phStatus_t phalMfdfLight_Sam_X_Int_Iso7816Wrap(uint8_t bFirstFrame, uint8_t bLengthPresent, uint8_t bLengthLen, uint8_t bLePresent, uint32_t dwTotDataLen,
	uint8_t * pPlainData, uint16_t * pDataLen)
{
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
	uint8_t		PH_MEMLOC_REM bLC = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint8_t		PH_MEMLOC_REM bOffset1 = 0;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Read = PH_OFF;
	uint8_t		PH_MEMLOC_REM bIsIsoChainnedCmd_Write = PH_OFF;

	/* Extract the command code. */
	if(bFirstFrame)
	{
		bCmdCode = pPlainData[bLengthPresent ? bLengthLen :0];
	}

	/* Set the flag for data operation commands. */
	bIsIsoChainnedCmd_Read = (uint8_t) (((bCmdCode == PHAL_MFDFLIGHT_CMD_READ_DATA_ISO) || (bCmdCode == PHAL_MFDFLIGHT_CMD_READ_RECORDS_ISO)) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd_Write = (uint8_t) (((bCmdCode == PHAL_MFDFLIGHT_CMD_WRITE_DATA_ISO) || (bCmdCode == PHAL_MFDFLIGHT_CMD_WRITE_RECORD_ISO) ||
		(bCmdCode == PHAL_MFDFLIGHT_CMD_UPDATE_RECORD_ISO)) ? PH_ON : PH_OFF);
	bIsIsoChainnedCmd = (uint8_t) (bIsIsoChainnedCmd_Read | bIsIsoChainnedCmd_Write);

	/* Set the LC value */
	bLC = (uint8_t) (bFirstFrame ? (*pDataLen - 1 /* Removing the command code. */) : *pDataLen);

	/* Subtract LC by 3 if Length is present. */
	bLC = (uint8_t) (bLengthPresent ? ( bLC - bLengthLen)  : bLC);

	/* Compute the offset. */
	bOffset = (uint8_t) ((bFirstFrame && bLengthPresent) ? (bLengthLen + 1) : 1);

	/* Compute the offset to be used for moving the current data. */
	bOffset1 = (uint8_t) (bLengthPresent ? (bLengthLen + 5) : 5);

	/* Copy the data to the Pointer. */
	memmove(&pPlainData[bOffset1], &pPlainData[bOffset], bLC);	/* PRQA S 3200 */

	/* Reset the length buffer. */
	*pDataLen = bLengthPresent ? bLengthLen : 0;

	/* Frame the initial ISO7816 header. */
	pPlainData[(*pDataLen)++] = PHAL_MFDFLIGHT_WRAPPEDAPDU_CLA;
	pPlainData[(*pDataLen)++] = bCmdCode;
	pPlainData[(*pDataLen)++] = PHAL_MFDFLIGHT_WRAPPEDAPDU_P1;
	pPlainData[(*pDataLen)++] = PHAL_MFDFLIGHT_WRAPPEDAPDU_P2;

	/* Add LC if there is data. */
	if(bLC)
	{
		pPlainData[(*pDataLen)++] = (uint8_t) (bIsIsoChainnedCmd_Write ? dwTotDataLen : bLC);

		/* Update Frame length based on LC. */
		*pDataLen = *pDataLen + bLC;
	}

	if(bLePresent)
	{
		pPlainData[(*pDataLen)++] = PHAL_MFDFLIGHT_WRAPPEDAPDU_LE;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_Int_ResetAuthentication(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset the Authmode and Key number */
	PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = 0xFF;
    PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = 0XFF;

	/* Kill the PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

phStatus_t phalMfdfLight_Sam_X_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
	uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM aIV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint8_t		PH_MEMLOC_REM bSvBak = 0;
	uint16_t	PH_MEMLOC_REM wActTMC = 0;
	uint16_t	PH_MEMLOC_REM wSesTMC = 0;
	uint32_t	PH_MEMLOC_REM dwTMC = 0;

	/* Increment the TMC by 1. */
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFDFLIGHT_AUTH_TYPE_EV2)
	{
		dwTMC = (uint32_t) (pTMC[0] | (pTMC[1] << 8) | (pTMC[2] << 16) | (pTMC[3] << 24));
		dwTMC++;
	}
	else
	{
		wActTMC = (uint16_t) (pTMC[0] | (pTMC[1] << 8));
		wSesTMC = (uint16_t) (pTMC[2] | (pTMC[3] << 8));

		wActTMC++;
		wSesTMC++;
		dwTMC = (uint32_t) ((wSesTMC << 16) | wActTMC);
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */
	memset(aIV, 0, 16);		/* PRQA S 3200 */

	/* Compute the session vector. */
	aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFDFLIGHT_SAM_NONX_SESSION_TMAC_ENC) ? 0xA5 : 0x5A);
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x01;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x80;

	/* Append the TMC information. */
	aSV[bSvLen++] = (uint8_t) (dwTMC & 0xFF);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF00) >> 8);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF0000) >> 16);
	aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF000000) >> 24);

	/* Append the UID infomration. */
	memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
	bSvLen = 16;

	/* Rotate the SV information by 1 for LRP. */
	if(PHAL_MFDFLIGHT_RESOLVE_DATAPARAMS(pDataParams)->bAuthType == PHAL_MFDFLIGHT_AUTH_TYPE_LRP)
	{
		bSvBak = aSV[0];
		memcpy(&aSV[0], &aSV[1], 15);	/* PRQA S 3200 */
		aSV[15] = bSvBak;
	}

	/* Load zero IV. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV,
		aIV,
		16));

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFDFLIGHT_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		16));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDFLIGHT);
}

#endif /* NXPBUILD__PHAL_MFDFLIGHT_SAM_X */
