/*
 * Copyright 2016 - 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software MIFARE Plus (R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFP_SAM_NON_X_H
#define PHALMFP_SAM_NON_X_H

#include <ph_Status.h>

/*
 * Performs a Write Perso command. The Write Perso command can be executed using the ISO14443-3 communication protocol
 * (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation).
 *
 * Input Parameters:
 *		pDataParams			: Pointer to this layer's parameter structure.
 *		bLayer4Comm			: ISO14443 protocol to be used.
 *								0x00: PHAL_MFPEV1_ISO14443_L3
 *								0x01: PHAL_MFPEV1_ISO14443_L4
 *		wBlockNr			: Block number to be personalized.
 *		pValue				: The value for the block mentioned in BlockNr parameter.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_WritePerso(void * pDataParams, uint8_t bLayer4Comm, uint16_t wBlockNr, uint8_t * pValue);

/*
 * Performs a Commit Perso command. The Commit Perso command can be executed using the ISO14443-3 communication protocol
 * (after layer 3 activation) or using the ISO14443-4 protocol (after layer 4 activation). This command commits the written
 * data during WritePerso command and switches the SecurityLevel to 1 or 3 based on the option provided.
 *
 * If the Option parameter is 0, only the command code will be exchanges to PICC. This is to maintain the backward
 * compatibility with Mifare Plus PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bLayer4Comm				: ISO14443 protocol to be used.
 *									0x00: PHAL_MFPEV1_ISO14443_L3
 *									0x01: PHAL_MFPEV1_ISO14443_L4
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_CommitPerso(void * pDataParams, uint8_t bLayer4Comm);


/****************************************************************************************************************************************/
/* Mifare Plus Generic command for authentication.																						*/
/****************************************************************************************************************************************/

/*
 * Perform MIFARE Classic Authentication using Sam hardware in NonX mode .
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bBlockNo		: Block number to be used for authentication.
 *		bKeyType		: The type of key to be used.
 *							0x0A: PHHAL_HW_MFC_KEYA
 *							0x0B: PHHAL_HW_MFC_KEYB.
 *		wKeyNo			: Key number to be used in authentication.
 *		wKeyVer			: Key version to be used in authentication.
 *		pUid			: Complete UID of Mifare PICC.
 *		bUidLen			: UID length provided.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_AuthenticateClassicSL2(void * pDataParams, uint8_t bBlockNo, uint8_t bKeyType,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pUid, uint8_t bUidLen);

/*
 * Performs a MIFARE Plus Authentication for Security Level 0. This command performs basic Authenticate First / Non-First
 * command execution and also performs the AuthenticateContinue command internally.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bLayer4Comm				: ISO14443 protocol to be used.
 *									0x00: PHAL_MFPEV1_ISO14443_L3
 *									0x01: PHAL_MFPEV1_ISO14443_L4
 *		bFirstAuth				: Type of authentication to be performed.
 *									0x00: PHAL_MFPEV1_AUTHENTICATE_FIRST
 *									0x01: PHAL_MFPEV1_AUTHENTICATE_NON_FIRST
 *		wBlockNr				: PICC Block number to be used for authentication.
 *		wKeyNumber				: Key number be to used from software or hardware keystore.
 *		wKeyVersion				: Key version be to used from software or hardware keystore.
 *		bLenDivInput			: Length of diversification input used to diversify the key.
 *								  If 0, no diversification is performed.
 *		pDivInput				: Diversification Input used to diversify the key.
 *		bLenPcdCap2				: Length of the input PCD capabilities.
 *		pPcdCap2In				: Buffer containing the Input PCD Capabilities.
 *									If length is non zero, PCDCapabilites should be available.
 *									If length is zero, PCDCapabilites buffer should be NULL.
 *
 * Output Parameters:
 *		pPcdCap2Out				: Buffer containing the Output PCD capabilities. This will be of 6 bytes.
 *		pPdCap2					: Buffer containing the Output PD capabilities. This will be of 6 bytes.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_AuthenticateSL0(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2);

/*
 * Performs a MIFARE Plus Authentication for Security Level 1. This command performs basic Authenticate First / Non-First
 * command execution and also performs the AuthenticateContinue command internally.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bLayer4Comm				: ISO14443 protocol to be used.
 *									0x00: PHAL_MFPEV1_ISO14443_L3
 *									0x01: PHAL_MFPEV1_ISO14443_L4
 *		bFirstAuth				: Type of authentication to be performed.
 *									0x00: PHAL_MFPEV1_AUTHENTICATE_FIRST
 *									0x01: PHAL_MFPEV1_AUTHENTICATE_NON_FIRST
 *		wBlockNr				: PICC Block number to be used for authentication.
 *		wKeyNumber				: Key number be to used from software or hardware keystore.
 *		wKeyVersion				: Key version be to used from software or hardware keystore.
 *		bLenDivInput			: Length of diversification input used to diversify the key.
 *								  If 0, no diversification is performed.
 *		pDivInput				: Diversification Input used to diversify the key.
 *		bLenPcdCap2				: Length of the input PCD capabilities.
 *		pPcdCap2In				: Buffer containing the Input PCD Capabilities.
 *									If length is non zero, PCDCapabilites should be available.
 *									If length is zero, PCDCapabilites buffer should be NULL.
 *
 * Output Parameters:
 *		pPcdCap2Out				: Buffer containing the Output PCD capabilities. This will be of 6 bytes.
 *		pPdCap2					: Buffer containing the Output PD capabilities. This will be of 6 bytes.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_AuthenticateSL1(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2);

/*
 * Performs a MIFARE Plus Authentication for Security Level 2. This command performs basic Authenticate First / Non-First
 * command execution and also performs the AuthenticateContinue command internally.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bLayer4Comm				: ISO14443 protocol to be used.
 *									0x00: PHAL_MFPEV1_ISO14443_L3
 *									0x01: PHAL_MFPEV1_ISO14443_L4
 *		bFirstAuth				: Type of authentication to be performed.
 *									0x00: PHAL_MFPEV1_AUTHENTICATE_FIRST
 *									0x01: PHAL_MFPEV1_AUTHENTICATE_NON_FIRST
 *		wBlockNr				: PICC Block number to be used for authentication.
 *		wKeyNumber				: Key number be to used from software or hardware keystore.
 *		wKeyVersion				: Key version be to used from software or hardware keystore.
 *		bLenDivInput			: Length of diversification input used to diversify the key.
 *								  If 0, no diversification is performed.
 *		pDivInput				: Diversification Input used to diversify the key.
 *		bLenPcdCap2				: Length of the input PCD capabilities.
 *		pPcdCap2In				: Buffer containing the Input PCD Capabilities.
 *									If length is non zero, PCDCapabilites should be available.
 *									If length is zero, PCDCapabilites buffer should be NULL.
 *
 * Output Parameters:
 *		pPcdCap2Out				: Buffer containing the Output PCD capabilities. This will be of 6 bytes.
 *		pPdCap2					: Buffer containing the Output PD capabilities. This will be of 6 bytes.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_AuthenticateSL2(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr, uint16_t wKeyNo,
	uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In, uint8_t * pPcdCap2Out,
	uint8_t * pPdCap2);

/*
 * Performs a MIFARE Plus Authentication for Security Level 3. This command performs basic Authenticate First / Non-First
 * command execution and also performs the AuthenticateContinue command internally.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bFirstAuth				: Type of authentication to be performed.
 *									0x00: PHAL_MFPEV1_AUTHENTICATE_FIRST
 *									0x01: PHAL_MFPEV1_AUTHENTICATE_NON_FIRST
 *		wBlockNr				: PICC Block number to be used for authentication.
 *		wKeyNumber				: Key number be to used from software or hardware keystore.
 *		wKeyVersion				: Key version be to used from software or hardware keystore.
 *		bLenDivInput			: Length of diversification input used to diversify the key.
 *								  If 0, no diversification is performed.
 *		pDivInput				: Diversification Input used to diversify the key.
 *		bLenPcdCap2				: Length of the input PCD capabilities.
 *		pPcdCap2In				: Buffer containing the Input PCD Capabilities.
 *									If length is non zero, PCDCapabilites should be available.
 *									If length is zero, PCDCapabilites buffer should be NULL.
 *
 * Output Parameters:
 *		pPcdCap2Out				: Buffer containing the Output PCD capabilities. This will be of 6 bytes.
 *		pPdCap2					: Buffer containing the Output PD capabilities. This will be of 6 bytes.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_AuthenticateSL3(void * pDataParams, uint8_t bFirstAuth, uint16_t wBlockNr, uint16_t wKeyNo,
	uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2);



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for data operations.																						*/
/****************************************************************************************************************************************/

/*
 * Performs a Multi Block Write command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		wBlockNr				: PICC block number to which the data should be written.
 *		bNumBlocks				: Number of blocks to write (must not be more than 3).
 *		pBlocks					: Block(s) (16 * bNumBlocks bytes).
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_MultiBlockWrite(void * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks);

/*
 * Performs a Multi Block Read command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		wBlockNr				: PICC block number from which the data should be read.
 *		bNumBlocks				: Number of blocks to read (must not be more than 3).
 *
 * Output Parameters:
 *		pBlocks					: Block(s) (16 * bNumBlocks bytes).
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_MultiBlockRead(void * pDataParams, uint8_t bBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks);

/*
 * Performs a Write / Write MACed command. This command writes a 16 byte data to the PICC.
 * The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bEncrypted				: Type of communication to be used. Based on this flag the command code
 *								  will be updated.
 *										0x00: PHAL_MFPEV1_ENCRYPTION_OFF
 *										0x01: PHAL_MFPEV1_ENCRYPTION_ON
 *		bWriteMaced				: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to which the data should be written.
 *		bNumBlocks				: Number of blocks to write.
 *								  15 block for Native communication if configured as Software component.
 *								  13 blocks if configured as Sam NonX component.
 *		pBlocks					: The data to be written. This buffer should have data equal to NumBlocks * 16,
 *								  where 16 is one block size.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Write(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t bNumBlocks,
	uint8_t * pBlocks);

/*
 * Performs a Read / Read MACed command. The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP
 * authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bEncrypted				: Type of communication to be used. Based on this flag the command code
 *								  will be updated.
 *										0x00: PHAL_MFPEV1_ENCRYPTION_OFF
 *										0x01: PHAL_MFPEV1_ENCRYPTION_ON
 *		bReadMaced				: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		bMacOnCmd				: Indicate whether the command should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_COMMAND_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_COMMAND_ON
 *		wBlockNr				: PICC block number from which the data should be read.
 *		bNumBlocks				: Number of blocks to be read.
 *								  15 block for Native communication if configured as Software component.
 *								  13 blocks if configured as Sam NonX component.
 *
 * Output Parameters:
 *		pBlocks					: The data to be written. This buffer should have data equal to NumBlocks * 16,
 *								  where 16 is one block size.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Read(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
	uint8_t bNumBlocks, uint8_t * pBlocks);



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for value operations.																					*/
/****************************************************************************************************************************************/

/*
 * Performs a Write / Write MACed command of a value.
 * The parameter Encrypted, WriteMaced are valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bEncrypted				: Type of communication to be used. Based on this flag the command code
 *								  will be updated.
 *										0x00: PHAL_MFPEV1_ENCRYPTION_OFF
 *										0x01: PHAL_MFPEV1_ENCRYPTION_ON
 *		bWriteMaced				: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to which the value should be written.
 *		pValue					: The value to be written. This buffer should have value equal to 4 bytes.
 *		bAddrData				: The address to be written.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_WriteValue(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t * pValue,
	uint8_t bAddrData);

/*
 * Performs a Read / Read MACed Value command.
 * The parameter Encrypted, ReadMaced and MacOnCmd are valid only for MFP authenticated state and
 * not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bEncrypted				: Type of communication to be used. Based on this flag the command code
 *								  will be updated.
 *										0x00: PHAL_MFPEV1_ENCRYPTION_OFF
 *										0x01: PHAL_MFPEV1_ENCRYPTION_ON
 *		bReadMaced				: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		bMacOnCmd				: Indicate whether the command should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_COMMAND_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_COMMAND_ON
 *		wBlockNr				: PICC block number from which the data should be read.
 *
 * Output Parameters:
 *		pValue					: The value read from the specified block number. The buffer will have 4 bytes
 *								  of value information.
 *		pAddrData				: The address from the read value information.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_ReadValue(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
	uint8_t * pValue, uint8_t * pAddrData);

/*
 * Performs an Increment / Increment MACed command.
 * The parameter IncrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bIncrementMaced			: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to be used for incrementing the value.
 *		pValue					: The value to be incremented. This buffer should have 4 bytes value information.
 *								  The value to be incremented should be LSB first order.
 *								  For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
 *								  0x01, 0x00, 0x00, 0x00.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Increment(void * pDataParams, uint8_t bIncrementMaced, uint16_t wBlockNr, uint8_t * pValue);

/*
 * Performs an Decrement / Decrement MACed command.
 * The parameter DecrementMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bDecrementMaced			: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to be used for decrementing the value.
 *		pValue					: The value to be decremented. This buffer should have 4 bytes value information.
 *								  The value to be decremented should be LSB first order.
 *								  For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
 *								  0x01, 0x00, 0x00, 0x00.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Decrement(void * pDataParams, uint8_t bDecrementMaced, uint16_t wBlockNr, uint8_t * pValue);

/*
 * Performs an Increment Transfer / Increment Transfer MACed command.
 * The parameter IncrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bIncrementTransferMaced	: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wSourceBlockNr			: PICC block number to be used for incrementing the value.
 *		wDestinationBlockNr		: PICC block number to be used for transferring the value.
 *		pValue					: The value to be incremented and transferred. This buffer should have 4 bytes
 *								  value information. The value to be incremented and transferred should be LSB first order.
 *								  For Ex. If the value to be incremented is by 1 times then the pValue buffer will be,
 *								  0x01, 0x00, 0x00, 0x00.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_IncrementTransfer(void * pDataParams, uint8_t bIncrementTransferMaced, uint16_t wSrcBlockNr,
	uint16_t wDstBlockNr, uint8_t * pValue);

/*
 * Performs an Decrement Transfer / Decrement Transfer MACed command.
 * The parameter DecrementTransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bDecrementTransferMaced	: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wSourceBlockNr			: PICC block number to be used for decrementing the value.
 *		wDestinationBlockNr		: PICC block number to be used for transferring the value.
 *		pValue					: The value to be decremented and transferred. This buffer should have 4 bytes
 *								  value information. The value to be Decremented and transferred should be LSB first order.
 *								  For Ex. If the value to be decremented is by 1 times then the pValue buffer will be,
 *								  0x01, 0x00, 0x00, 0x00.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_DecrementTransfer(void * pDataParams, uint8_t bDecrementTransferMaced, uint16_t wSrcBlockNr,
	uint16_t wDstBlockNr, uint8_t * pValue);

/*
 * Performs a Transfer / Transfer MACed command.
 * The parameter TransferMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bTransferMaced			: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Transfer(void * pDataParams, uint8_t bTransferMaced, uint16_t wBlockNr);

/*
 * Performs a Restore / Restore MACed command.
 * The parameter RestoreMaced is valid only for MFP authenticated state and not for MFC authenticate state.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bRestoreMaced			: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Restore(void * pDataParams, uint8_t bRestoreMaced, uint16_t wBlockNr);



/****************************************************************************************************************************************/
/* Mifare Plus Generic command for special commands.																					*/
/****************************************************************************************************************************************/

/*
 * Performs a Reset Auth command.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_ResetAuth(void * pDataParams);

/*
 * Reset the libraries internal secure messaging state.
 *
 * This function must be called before interacting with the PICC to set the libraries internal card-state back to default.\n
 * E.g. when an error occurred or after a reset of the field.
 *
 * Input Parameters:
 *		pDataParams			: Pointer to this layer's parameter structure.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_ResetSecMsgState(void * pDataParams);

/*
 * Performs a Key Change of a MIFARE Plus key. Same as phalMfp_Write, but diversification input can be provided.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bChangeKeyMaced			: Indicate whether the response should be maced. Based on this flag the
 *								  command code will be updated.
 *										0x00: PHAL_MFPEV1_MAC_ON_RESPONSE_OFF
 *										0x01: PHAL_MFPEV1_MAC_ON_RESPONSE_ON
 *		wBlockNr				: PICC block number to which the key should be changed.
 *		wKeyNumber				: Key number to be used from software or hardware keystore.
 *		wKeyVersion				: Key version to be used from software or hardware keystore.
 *		bLenDivInput			: Length of diversification input used to diversify the key.
 *								  If 0, no diversification is performed.
 *		pDivInput				: Diversification Input used to diversify the key.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_ChangeKey(void * pDataParams, uint8_t bChangeKeyMaced, uint16_t wBlockNr, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t bDivInputLen, uint8_t * pDivInput);

/*
 * Performs the whole Proximity Check command chain. This command is supported in Authenticated state only.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		pRndC					: Provided RndC (7 bytes)
 *		bPps1					: Communication Speed byte (PPS1)
 *		bNumSteps				: Number of ProximityCheck cycles.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_ProximityCheck(void * pDataParams, uint8_t * pRndC, uint8_t bPps1, uint8_t bNumSteps);

#endif /* PHALMFP_SAM_NON_X_H */
