/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Mifare Plus application's Sam NonX layer's internal component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFP_SAM_NONX_INT_H
#define PHALMFP_SAM_NONX_INT_H

#include <ph_Status.h>
#include <phalMfp.h>

#include <phhalHw_SamAV2_Cmd.h>
#include "../phalMfp_Int.h"

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */

/* Mifare Plus command codes. */
#define PHAL_MFP_CMD_AUTH_FIRST								0x70	/**< Mfp command code for Authenticate First. */
#define PHAL_MFP_CMD_AUTH_NON_FIRST							0x76	/**< Mfp command code for Authenticate Non First. */
#define PHAL_MFP_CMD_AUTH_CONTINUE							0x72	/**< Mfp command code for Authenticate Continue. */

/** MIFARE Plus SL and Key derivation info states. */
#define PHAL_MFP_AUTHENTICATE_SL1_NO_KDF					0x00	/**< Option to set the key derivation info for SL0 or SL1 layer. */
#define PHAL_MFP_AUTHENTICATE_SL2_KDF						0x08	/**< Option to set the key derivation info for SL2 layer. */
#define PHAL_MFP_AUTHENTICATE_SL3_KDF						0x0C	/**< Option to set the key derivation info for SL3 layer. */

/* Mifare Plus buffer sizes. */
#define PHAL_MFP_CAPABILITY_SIZE							6U		/**< MFP PCD or PD capability's buffer size. */
#define PHAL_MFP_AUTH_CMD_SIZE								39U		/**< MFP Authenticate First command buffer size. */
#define PHAL_MFP_RESP_PD_CHAL_SIZE							16U		/**< MFP Authenticate First response buffer size. */
#define PHAL_MFP_PD_CHAL_SIZE								16U		/**< MFP PD Challange size. */
#define PHAL_MFP_RESP_PCD_CHAL_SIZE							32U		/**< MFP PCD Challange response buffer size. */
#define PHAL_MFP_COMBINED_WRITE_CMD_SIZE					5U		/**< MFP Combined Write command buffer size. */
#define PHAL_MFP_COMBINED_READ_CMD_SIZE						4U		/**< MFP Combined Read command buffer size. */
#define PHAL_MFP_CHANGE_KEY_CMD_SIZE						36U		/**< MFP ChangeKey command buffer size. */

/* Block definitions */
#define PHAL_MFP_DATA_BLOCK_SIZE							16		/* Length of MFP data block. */
#define PHAL_MFP_VALUE_BLOCK_SIZE							4		/* Length of MFP value block. */
#define PHAL_MFP_MAX_WRITE_BLOCK	(PHAL_MFPEV1_DATA_BLOCK_SIZE * 15)	/* Maximum blocks that can be written. */

/* Sam maximum single frame size. */
#define PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE		224U	/**< Maximum number of bytes that can be framed in one single SAM frame.
																 *   => HOST PROTECTION PLAIN = [Cmd + BNr + Ext + RC] + Data
																 *   => HOST PROTECTION MAC   = [Cmd + BNr + Ext + RC] + Data + HOST MAC
																 *   => HOST PROTECTION FULL  = HOST_ENC ( [Cmd + BNr + Ext + RC] + Data ) + HOST_MAC
																 */

#define PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_0          0x8000
#define PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_1          0x8001
#define PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_2          0x8002
#define PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_FIRST      PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_0
#define PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_LAST       PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_2

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
#define PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(DataParams)																						\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV2_ID)      ? ((phalMfp_SamAV2_DataParams_t *) DataParams)->pPalMifareDataParams :		\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV3_NONX_ID) ? ((phalMfp_SamAV3_NonX_DataParams_t *) DataParams)->pPalMifareDataParams :	\
			NULL
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
#ifndef NXPBUILD__PHAL_MFP_SAMAV3_NONX
#define PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(DataParams)																						\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV2_ID)      ? ((phalMfp_SamAV2_DataParams_t *) DataParams)->pPalMifareDataParams :		\
			NULL
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
#define PHHAL_HW_SAM_KILL_AUTHENTICATION(DataParams, Option )																					\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV2_ID) ?																					\
				phhalHw_SamAV2_Cmd_SAM_KillAuthentication(((phalMfp_SamAV2_DataParams_t *) DataParams)->pHalSamDataParams, Option) :			\
																																				\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV3_NONX_ID) ?																			\
				phhalHw_SamAV3_Cmd_SAM_KillAuthentication(((phalMfp_SamAV3_NonX_DataParams_t *) DataParams)->pHalSamDataParams, Option) :	 	\
																																				\
			PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFP)
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
#ifndef NXPBUILD__PHAL_MFP_SAMAV3_NONX
#define PHHAL_HW_SAM_KILL_AUTHENTICATION(DataParams, Option )																					\
			(PH_GET_COMPID(DataParams) == PHAL_MFP_SAMAV2_ID) ?																					\
				phhalHw_SamAV2_Cmd_SAM_KillAuthentication(((phalMfp_SamAV2_DataParams_t *) DataParams)->pHalSamDataParams, Option) :			\
																																				\
			PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFP)
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */

/*
 * Exchanges the command to the card and receives the response also. This interface also verifies for any error
 * and discards the status code from the received data.
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bOption			: Buffering options like First, Last, Cont, Default, etc...
 *		bIsoLayer		: Type of ISO14443 layer protocol to be used while performing authentication.
 *							0x00: ISO14443 L3.
 *							0x01: ISO14443 L4.
 *		pPayload		: The command or payload information to be exchange to card.
 *		wPayloadLen		: The length of payload information.
 *
 * Output Parameters:
 *		ppResponse		: Response received from the card without status code.
 *		pRespLen		: Response length excluding the status code.
 *		pPiccErrCode	: The status code received from the card.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_CardExchange(void * pDataParams, uint16_t wOption, uint8_t bIsoLayer, uint8_t * pPayload,
	uint16_t wPayloadLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode);

/*
 * Perfomrs SAM_AuthenticateMFP. Here the data received from Card will be exchanged o SAM for further processig.
 * Also the data received from
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bLayer4Comm		: ISO14443 protocol to be used.
 *							0x00: PHAL_MFP_ISO14443_L3
 *							0x01: PHAL_MFP_ISO14443_L4
 *		bFirstAuth		: Type of authentication to be performed.
 *							0x01: First Authentication.
 *							0x00: Non First Authentication (Following Authentication).
 *		bKdf			: The security level at which this interface is called. This value will be passed as
 *						  KDF information to HAL layer.
 *							0x00: Security Level 0 KDF.
 *							0x00: Security Level 1 KDF.
 *							0x01: Security Level 2 KDF.
 *							0x03: Security Level 3 KDF.
 *		wBlockNr		: Card's AES block number to be used for authentication.
 *		wKeyNum			: AES key address in hardware key store.
 *		wKeyVer			: AES key version to be used.
 *		pDivInput		: Diversification Input used to diversify the key.
 *		bDivInputLen	: Length of diversification input used to diversify the key.
 *						  If 0, no diversification is performed.
 *		pPcdCap2In		: PCD capibilites 2 information tobe passed.
 *		bPcdCap2InLen	: Length of PCD capibilities 2 to be sent to card.
 *
 * Output Parameters:
 *		pPcdCap2Out		: PCD Capabilities received from card. Consists of 6 bytes of data.
 *					      This PCD capibilities are returned by the card if its First authentication.
 *					      For Non First authentication (Following authentication)this data is not returned.
 *		pPdCap2			: PD Capabilities received from card. Consists of 6 bytes of data.
 *					      This PD capibilities are returned by the card if its First authentication.
 *					      For Non First authentication (Following authentication)this data is not returned.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_AuthenticateMFP(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint8_t bKdf, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCap2In, uint8_t bPcdCap2InLen,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2);

/*
 * Performs a CombinedWrite command with PICC via Sam hardware. This command is valid for Write, ValueWrite, Increment, Decrement,
 * IncrementTrenafer, DecrementTransfer, Transfer and REstore command.
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bCmdCode		: One of the following command codes.
 *							Write				: 0xA0, 0xA1, 0xA2, 0xA3
 *							Increment			: 0xB0, 0xB1
 *							Decrement			: 0xB2, 0xB3
 *							Transfer			: 0xB4, 0xB5
 *							IncrementTransfer	: 0xB6, 0xB7
 *							DecrementTransfer	: 0xB8, 0xB9
 *							Restore				: 0xC2, 0xC3
 *		wSrcBlockNr		: Source block number.
 *		wDstBlockNr		: Destination block number for all Transfer related commands.
 *		pData			: Data to be written to the.
 *		bDataLen		: Length of input data stream.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_CombinedWriteMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wSrcBlockNr, uint16_t wDstBlockNr, uint8_t * pData,
	uint8_t bDataLen );

/*
 * Perform a MFP CombinedRead command. This command is common for Read and ReadValue.
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bCmdCode		: One of the following command codes.
 *							Read				: 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37
 *		wBlockNr		: PICC block number from where the data should be read.
 *		bNumBlocks		: The number of blocks to be read.
 *
 * Output Parameters:
 *		pBlocks			: The data read from the block number specified. The buffer will contain the size calculated form
 *						  bNumBlocks * 16, where 16 is one block size.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_CombinedReadMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks);

/*
 * Performs a Change key command.
 *
 * Input Parameters:
 *		pDataParams		: Pointer to this layer's parameter structure.
 *		bCommand		: PICC command code used to perform the Key Change.
 *		wBlockNr		: Block number available inside the card where the value will be written.
 *		wKeyNum			: AES key address in hardware key store.
 *		wKeyVer			: AES key version to be used.
 *		bDivInputLen	: Length of diversification input used to diversify the key.
 *						  If 0, no diversification is performed.
 *		pDivInput		: Diversification Input used to diversify the key.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_ChangeKeyMFP(void * pDataParams, uint8_t bCommand, uint16_t wBlockNr, uint16_t wKeyNum, uint16_t wKeyVer,
	uint8_t bDivInputLen, uint8_t * pDivInput);

/*
 * Performs the whole Proximity Check command chain. This command is supported in Authenticated state only.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		pRndC					: Provided RndC (7 bytes)
 *		bPps1					: Communication Speed byte (PPS1)
 *		bNumSteps				: Number of ProximityCheck cycles.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_Sam_NonX_Int_ProximityCheck(void * pDataParams, uint8_t * pRndC, uint8_t bPps1, uint8_t bNumSteps);
#endif /* PHALMFP_SAM_NONX_INT_H */
