/*
 * Copyright 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MIFARE Plus EVx application's Sam NonX layer's internal component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFPEVX_SAM_NONX_INT_H
#define PHALMFPEVX_SAM_NONX_INT_H

#include <ph_Status.h>

/* MIFARE Plus EVx command codes. */
#define PHAL_MFPEVX_CMD_AUTH_FIRST                      0x70    /**< MFPEVX command code for Authenticate First. */
#define PHAL_MFPEVX_CMD_AUTH_NON_FIRST                  0x76    /**< MFPEVX command code for Authenticate Non First. */
#define PHAL_MFPEVX_CMD_AUTH_CONTINUE                   0x72    /**< MFPEVX command code for Authenticate Continue. */

/* MIFARE Plus EVx Security Levels to be used while performing authentication. This data will be utilized
 * for updating the P1 information byte of Sam Av3 Hal interface. */
#define PHAL_MFPEVX_SECURITY_LEVEL_0_KDF                0x00    /**< MFPEVX Security Level 0 KDF value to be passed for P1 information byte. */
#define PHAL_MFPEVX_SECURITY_LEVEL_1_KDF                0x00    /**< MFPEVX Security Level 1 KDF value to be passed for P1 information byte. */
#define PHAL_MFPEVX_SECURITY_LEVEL_3_KDF                0x0C    /**< MFPEVX Security Level 3 KDF value to be passed for P1 information byte. */

/* MIFARE Plus EVx buffer sizes. */
#define PHAL_MFPEVX_CAPABILITY_SIZE                     6U      /**< MFPEVX PCD or PD capability's buffer size. */
#define PHAL_MFPEVX_AUTH_CMD_SIZE                       39U     /**< MFPEVX Authenticate First command buffer size. */
#define PHAL_MFPEVX_RESP_PD_CHAL_SIZE                   16U     /**< MFPEVX Authenticate First response buffer size. */
#define PHAL_MFPEVX_PD_CHAL_SIZE                        16U     /**< MFPEVX PD Challenge size. */
#define PHAL_MFPEVX_RESP_PCD_CHAL_SIZE                  32U     /**< MFPEVX PCD Challenge response buffer size. */
#define PHAL_MFPEVX_COMBINED_WRITE_CMD_SIZE             5U      /**< MFPEVX Combined Write command buffer size. */
#define PHAL_MFPEVX_COMBINED_READ_CMD_SIZE              4U      /**< MFPEVX Combined Read command buffer size. */
#define PHAL_MFPEVX_CHANGE_KEY_CMD_SIZE                 36U     /**< MFPEVX ChangeKey command buffer size. */
#define PHAL_MFPEVX_COMMIT_READER_ID_CMD_SIZE           27U     /**< MFPEVX CommitReaderID command buffer size. */

#define PHAL_MFPEVX_IGNORE_PICC_STATUS_CHECK            0x0080  /**< This flag specifies the generic internal wrapping interface not to validate the PICC error codes. */
#define PHAL_MFPEVX_SAM_NONX_SESSION_TMAC_ENC           0x01    /**< Option to perform generate the TMAC session encryption keys. */
#define PHAL_MFPEVX_SAM_NONX_SESSION_TMAC_MAC           0x02    /**< Option to perform generate the TMAC session MAC keys. */

/* Sam maximum single frame size. */
#define PHAL_MFPEVX_SAM_DATA_FRAME_LENGTH               224     /* Maximum data that can be exchanged in case of secure messaging computation by SAM. */
#define PHAL_MFPEVX_SAM_COMBINED_READ_MAX_FRAME_SIZE    224U    /**< Maximum number of bytes that can be framed in one single SAM frame.
                                                                 *   => HOST PROTECTION PLAIN = [Cmd + BNr + Ext + RC] + Data
                                                                 *   => HOST PROTECTION MAC   = [Cmd + BNr + Ext + RC] + Data + HOST MAC
                                                                 *   => HOST PROTECTION FULL  = HOST_ENC ( [Cmd + BNr + Ext + RC] + Data ) + HOST_MAC
                                                                 */

/* Resolves the Hal component to be used for communicating with Sam hardware in NonX mode. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(DataParams)                                                                                                  \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pHalSamDataParams :       \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->pHalSamDataParams :          \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(DataParams)                                                                                                  \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pHalSamDataParams :       \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve PAL MIFARE DataParams. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_PAL_MIFARE_DATAPARAMS(DataParams)                                                                                           \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pPalMifareDataParams :    \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->pPalMifareDataParams :       \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_PAL_MIFARE_DATAPARAMS(DataParams)                                                                                           \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pPalMifareDataParams :    \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve Extended Length APDU. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_EXTENDED_LENGTH_APDU(DataParams)                                                                                            \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bExtendedLenApdu :        \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->bExtendedLenApdu :           \
            NULL

#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_EXTENDED_LENGTH_APDU(DataParams)                                                                                            \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? (( phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bExtendedLenApdu :       \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve Wrapped Mode. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_WRAPPED_MODE(DataParams)                                                                                                    \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bWrappedMode :            \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->bWrappedMode :               \
            NULL

#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_WRAPPED_MODE(DataParams)                                                                                                    \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bExtendedLenApdu :        \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve Auth Mode. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_AUTH_MODE(DataParams)                                                                                                       \
            ((PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) (DataParams))->bAuthMode :            \
            ((PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) (DataParams))->bAuthMode :               \
            NULL))

#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_AUTH_MODE(DataParams)                                                                                                       \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bAuthMode :               \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve SMMode. */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_SM_MODE(DataParams)                                                                                                         \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bSMMode :                 \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->bSMMode :                    \
            NULL

#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_SM_MODE(DataParams)                                                                                                         \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->bSMMode :                 \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

/* Resolve TMI DataParams */
#ifdef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_TMI_DATAPARAMS(DataParams)                                                                                                  \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pTMIDataParams :          \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAM_NONX_ID)    ? ((phalMfpEVx_Sam_NonX_DataParams_t *) DataParams)->pTMIDataParams :             \
            NULL

#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */
#ifndef NXPBUILD__PHAL_MFPEVX_INTERNAL
#define PHAL_MFPEVX_RESOLVE_TMI_DATAPARAMS(DataParams)                                                                                                  \
            (PH_GET_COMPID(DataParams) == PHAL_MFPEVX_SAMAV3_NONX_ID) ? ((phalMfpEVx_SamAV3_NonX_DataParams_t *) DataParams)->pTMIDataParams :          \
            NULL
#endif /* NXPBUILD__PHAL_MFPEVX_INTERNAL */

phStatus_t phalMfpEVx_Sam_NonX_Int_ResetSecMsgState(void * pDataParams);

phStatus_t phalMfpEVx_Sam_NonX_Int_CardExchange(void * pDataParams, uint16_t wOption, uint8_t bIsoLayer, uint8_t bLc, uint8_t * pPayload,
    uint16_t wPayloadLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode);

phStatus_t phalMfpEVx_Sam_NonX_Int_WriteExtMfc(void * pDataParams, uint8_t bCmdCode, uint8_t bBlockNo, uint8_t * pData, uint16_t wDataLen,
    uint8_t * pTMC, uint8_t * pTMV);

phStatus_t phalMfpEVx_Sam_NonX_Int_ReadExtMfc(void * pDataParams, uint8_t bBlockNo, uint8_t * pBlockData);

phStatus_t phalMfpEVx_Sam_NonX_Int_AuthenticateMfc(void * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pUid, uint8_t bUidLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_AuthenticateMFP(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint8_t bKdf, uint16_t wBlockNr,
    uint16_t wKeyNum, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In, uint8_t * pPcdCap2Out,
    uint8_t * pPdCap2);

phStatus_t phalMfpEVx_Sam_NonX_Int_AuthenticateMFP_Ext(void * pDataParams, uint8_t bFirstAuth, uint16_t wBlockNr, uint16_t wKeyNum,
    uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In, uint8_t * pPcdCap2Out,
    uint8_t * pPdCap2);

phStatus_t phalMfpEVx_Sam_NonX_Int_AuthSectorSwitchMFP(void * pDataParams, uint8_t bOption, uint16_t wSSKeyBNr, uint16_t wSSKeyNr, uint16_t wSSKeyVer,
    uint8_t bLenDivInputSSKey, uint8_t * pDivInputSSKey, uint8_t bSecCount, uint16_t *pSectorNos, uint16_t *pKeyNo, uint16_t *pKeyVer,
    uint8_t bLenDivInputSectorKeyBs, uint8_t * pDivInputSectorKeyBs);

phStatus_t phalMfpEVx_Sam_NonX_Int_AuthenticatePDC(void * pDataParams, uint16_t wBlockNr, uint16_t wKeyNum, uint16_t wKeyVer, uint8_t bDivInputLen,
    uint8_t * pDivInput, uint8_t bUpgradeInfo);

phStatus_t phalMfpEVx_Sam_NonX_Int_CombinedWriteMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wSrcBlockNr, uint16_t wDstBlockNr, uint8_t * pData,
    uint8_t bDataLen, uint8_t * pTMC, uint8_t * pTMV);

phStatus_t phalMfpEVx_Sam_NonX_Int_CombinedReadMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks);

phStatus_t phalMfpEVx_Sam_NonX_Int_ChangeKeyMFP(void * pDataParams, uint8_t bCommand, uint16_t wBlockNr, uint16_t wKeyNum, uint16_t wKeyVer,
    uint8_t bDivInputLen, uint8_t * pDivInput);

phStatus_t phalMfpEVx_Sam_NonX_Int_CommitReaderID(void * pDataParams, uint16_t wBlockNr, uint8_t * pEncTMRI);

phStatus_t phalMfpEVx_Sam_NonX_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen);

void phalMfpEVx_Sam_NonX_Int_SetAuthMode(void* pDataParams, uint8_t wValue);

void phalMfpEVx_Sam_NonX_Int_SetSMMode(void* pDataParams, uint8_t wValue);

void phalMfpEVx_Sam_NonX_Int_SetWrappedMode(void* pDataParams, uint8_t wValue);

void phalMfpEVx_Sam_NonX_Int_SetExtendedLenApdu(void* pDataParams, uint8_t wValue);

uint16_t phalMfpEVx_Sam_NonX_Int_GetWrappedMode(void* pDataParams);

uint16_t phalMfpEVx_Sam_NonX_Int_GetAuthMode(void* pDataParams);

uint16_t phalMfpEVx_Sam_NonX_Int_GetExtendedLenApdu(void* pDataParams);

phStatus_t phalMfpEVx_Sam_NonX_Int_KillAuthentication(void* pDataParams, uint8_t bOption);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthenticateMFP_Part1(void* pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t* pPDChal, uint8_t bPDChalLen, uint8_t* pDivInput, uint8_t bDivInputLen, uint8_t** ppPCDChalResp,
    uint16_t* pPCDChalRespLen);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthenticateMFP_Part2(void* pDataParams, uint8_t bPiccErrCode, uint8_t* pPDResp,
    uint8_t bPDRespLen, uint8_t** ppPDCap2, uint8_t** ppPCDCap2, uint8_t* pPiccReturnCode);

phStatus_t phalMfpEVx_Sam_NonX_Int_KillAuthentication(void* pDataParams, uint8_t bOption);

phStatus_t phalMfpEVx_Sam_NonX_Int_DecipherData(void* pDataParams, uint16_t wOption, uint8_t* pEncData, uint8_t bEncDataLen,
    uint8_t* pLength, uint8_t** ppPlainData, uint16_t* pPlainDataLen);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthSectorSwitchMFP_Part1(void* pDataParams, uint8_t bOption,
    uint8_t* pPDChal, uint8_t bPDChalLen, uint16_t wSSKeyBNr, uint8_t bSSKeyNo, uint8_t bSSKeyVer, uint8_t bMSKeyNo,
    uint8_t bMSKeyVer, uint8_t bSectorCount, uint8_t* pKeyBlocks, uint8_t bKeyBlocksLen, uint8_t* pDivInput,
    uint8_t bDivInputLen, uint8_t** ppPCDChalResp, uint16_t* pPCDChalRespLen);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthSectorSwitchMFP_Part2(void* pDataParams, uint8_t bPiccErrCode,
    uint8_t* pPDResp, uint8_t bPDRespLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthenticatePDC_Part1(void* pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t* pPDChal, uint8_t bPDChalLen, uint8_t* pUpgradeInfo, uint8_t bLen, uint8_t* pDivInput,
    uint8_t bDivInputLen, uint8_t** ppPCDChalResp, uint16_t* pPCDChalRespLen);

phStatus_t phalMfpfEVx_Sam_NonX_Int_AuthenticatePDC_Part2(void* pDataParams, uint8_t bPiccErrCode,
    uint8_t* pPDResp, uint8_t bPDRespLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpfEVx_Sam_NonX_Int_CombinedWriteMFP_Routing(void* pDataParams, uint16_t wOption,
    uint8_t* pData, uint8_t bDataLen, uint8_t** ppOutput, uint16_t* pOutputLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpfEVx_Sam_NonX_Int_CombinedReadMFP_Routing(void* pDataParams, uint8_t bLFI, uint16_t wOption, uint8_t* pData,
    uint8_t bDataLen, uint8_t** ppOutput, uint16_t* pOutputLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpfEVx_Sam_NonX_Int_ChangeKeyMFP(void* pDataParams, uint8_t bOption, uint8_t* pData, uint8_t bDataLen,
    uint8_t** ppProtectedData, uint16_t* pProtectedDataLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpEVx_Sam_NonX_Int_CommitReaderID_Part1(void* pDataParams, uint8_t bState, uint16_t wBlockNr,
    uint8_t** ppResponse, uint16_t* pRespLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_CommitReaderID_Part2(void* pDataParams, uint8_t bPiccErrCode,
    uint8_t* pData, uint8_t bDataLen, uint8_t* pPiccReturnCode);

phStatus_t phalMfpEVx_Sam_NonX_Int_LoadInitVector(void* pDataParams, uint8_t bOption, uint8_t* pData, uint8_t bDataLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_DeriveKey(void* pDataParams, uint8_t bSrcKeyNo, uint8_t bSrcKeyVer,
    uint8_t bDstKeyNo, uint8_t* pDeriveIn, uint8_t bDeriveInLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_ActivateOfflineKey(void* pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyV,
    uint8_t* pDivInput, uint8_t bDivInputLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_GenerateMAC(void* pDataParams, uint16_t wOption, uint8_t bNum, uint8_t* pTxData,
    uint8_t bTxDataLen, uint8_t** ppRxData, uint16_t* pRxDataLen);

phStatus_t phalMfpEVx_Sam_NonX_Int_DecipherOfflineData(void* pDataParams, uint16_t wOption, uint8_t* pEncData,
    uint8_t bEncDataLen, uint8_t** ppPlainData, uint16_t* pPlainDataLen);

#endif /* PHALMFPEVX_SAM_NONX_INT_H */
