/*
 * Copyright 2016 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SamX MIFARE(R) Ultralight Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_MFUL_SAM_X

#include <phalMful.h>
#include <ph_RefDefs.h>


#ifdef NXPBUILD__PHHAL_HW_SAMAV2
#include <phhalHw_SamAV2_Cmd.h>
#endif /* NXPBUILD__PHHAL_HW_SAMAV2 */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

#include "phalMful_Sam_X.h"
#include "phalMful_Sam_X_Int.h"
#include "../phalMful_Int.h"

#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_X
phStatus_t phalMful_SamAV2_X_Init(phalMful_SamAV2_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_SamAV2_DataParams_t * pHalSamDataParams, phpalMifare_SamAV2_X_DataParams_t * pPalMifareDataParams)
{
    if(sizeof(phalMful_SamAV2_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFUL);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFUL);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFUL);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFUL);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_MFUL | PHAL_MFUL_SAMAV2_X_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_X */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_X
phStatus_t phalMful_SamAV3_X_Init(phalMful_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_SamAV3_DataParams_t * pHalSamDataParams, phpalMifare_SamAV3_X_DataParams_t * pPalMifareDataParams)
{
    if(sizeof(phalMful_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFUL);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFUL);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFUL);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFUL);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_MFUL | PHAL_MFUL_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */

phStatus_t phalMful_Sam_X_UlcAuthenticate(void * pDataParams, uint8_t bOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pDivInput, uint8_t bDivInputLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bStatusCode = 0;

    wStatus = phalMful_Sam_X_Int_UlcAuthenticate(
        pDataParams,
        bOption,
        wKeyNo,
        wKeyVer,
        pDivInput,
        bDivInputLen,
        &bStatusCode);

    /* Return PICC error code in cases of PICC failure. */
    if(bStatusCode != 0U)
        return PH_ADD_COMPCODE(bStatusCode, PH_COMP_AL_MFUL);

    /* Resolve Sam error. */
    if(wStatus != PH_ERR_SUCCESS)
        return wStatus;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_Read(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Sam_X_Int_MfRead(
        pDataParams,
        &bAddress,
        1U,
        &pResponse,
        &wRespLen));

    (void) memcpy(pData, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_Write(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aData[5];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Frame the data buffer. */
    aData[0U] = bAddress;

    /* Copy the data. */
    (void) memcpy(&aData[1U], pData, 4U);

    /* perform the actual write operation */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Sam_X_Int_MfWrite(
        pDataParams,
        0x0001U /* Buffering is default and writing to Ultralight PICC. */,
        aData,
        5U,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_FastWrite(void * pDataParams, uint8_t * pData)
{
    return phalMful_Int_FastWrite(pDataParams, pData);
}

phStatus_t phalMful_Sam_X_CompatibilityWrite(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aData[17];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Frame the data buffer. */
    aData[0U] = bAddress;

    /* Copy the data. */
    (void) memcpy(&aData[1U], pData, 16U);

    /* perform the actual write operation */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Sam_X_Int_MfWrite(
        pDataParams,
        0x0000U /* Buffering is default and writing to Ultralight PICC. */,
        aData,
        17U,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_IncrCnt(void * pDataParams, uint8_t bCntNum, uint8_t * pCnt)
{
    return phalMful_Int_IncrCnt(pDataParams, bCntNum, pCnt);
}

phStatus_t phalMful_Sam_X_ReadCnt(void * pDataParams, uint8_t bCntNum, uint8_t * pCntValue)
{
    return phalMful_Int_ReadCnt(pDataParams, bCntNum, pCntValue);
}

phStatus_t phalMful_Sam_X_PwdAuth(void * pDataParams, uint8_t bOption, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPwd, uint8_t * pPack)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bPICCStatus = 0;

    /* Validate parameter. */
    if(bOption > PHAL_MFUL_CMD_PWD_AUTH_DIV_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFUL);
    }

#ifndef NXPBUILD__PHAL_MFUL_SAMAV3_X
    /* Actual these parameters are not used when Sam is AV2 mode. When Package manager is build using only SAM AV2 mode
     * these parameter are shown as warning resulting in error in Release mode. So package manager to run smoothly,
     * initializing the unused parameters.
     */
    PH_UNUSED_VARIABLE(bDivInputLen);
    PH_UNUSED_VARIABLE(pDivInput);
    PH_UNUSED_VARIABLE(bKeyNo);
    PH_UNUSED_VARIABLE(bKeyVer);
    PH_UNUSED_VARIABLE(bPICCStatus);
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */

    /* Exchange the  information to Sam hardware. */
    wStatus = phalMful_Sam_X_Int_PwdAuthUL(
        pDataParams,
        bOption,
        bKeyNo,
        bKeyVer,
        pPwd,
        pPack,
        pDivInput,
        bDivInputLen,
        &bPICCStatus);

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_X
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
    {
        return PH_ADD_COMPCODE(bPICCStatus, PH_COMP_AL_MFUL);
    }
    else
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */
    {
        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_GetVersion(void * pDataParams, uint8_t * pVersion)
{
    return phalMful_Int_GetVersion(pDataParams, pVersion);
}

phStatus_t phalMful_Sam_X_FastRead(void * pDataParams, uint8_t  bStartAddr, uint8_t bEndAddr,
    uint8_t ** pData, uint16_t * pNumBytes)
{
    return phalMful_Int_FastRead(pDataParams, bStartAddr, bEndAddr, pData, pNumBytes);
}

phStatus_t phalMful_Sam_X_SectorSelect(void * pDataParams, uint8_t bSecNo)
{
    return phalMful_Int_SectorSelect(pDataParams, bSecNo);
}

phStatus_t phalMful_Sam_X_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature, uint16_t* pDataLen)
{
    return phalMful_Int_ReadSign(pDataParams, bAddr, pSignature, pDataLen);
}

phStatus_t phalMful_Sam_X_ChkTearingEvent(void * pDataParams, uint8_t bCntNum, uint8_t * pValidFlag)
{
    return phalMful_Int_ChkTearingEvent(pDataParams, bCntNum, pValidFlag);
}

phStatus_t phalMful_Sam_X_WriteSign(void * pDataParams, uint8_t bAddress, uint8_t * pSignature)
{
    return phalMful_Int_WriteSign(pDataParams, bAddress, pSignature);
}

phStatus_t phalMful_Sam_X_LockSign(void * pDataParams, uint8_t bLockMode)
{
    return phalMful_Int_LockSign(pDataParams, bLockMode);
}

phStatus_t phalMful_Sam_X_VirtualCardSelect(void * pDataParams, uint8_t * pVCIID, uint8_t bVCIIDLen, uint8_t * pVCTID)
{
    return phalMful_Int_VirtualCardSelect(pDataParams, pVCIID, bVCIIDLen, pVCTID);
}

phStatus_t phalMful_Sam_X_ReadTTStatus(void * pDataParams, uint8_t bAddr, uint8_t * pData)
{
    return phalMful_Int_ReadTTStatus(pDataParams, bAddr, pData);
}

phStatus_t phalMful_Sam_X_GetConfig(void *pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PHAL_MFUL_ADDITIONAL_INFO:
            switch(PH_GET_COMPID(pDataParams))
            {
#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_X
                case PHAL_MFUL_SAMAV2_X_ID:
                    *pValue = ((phalMful_SamAV2_X_DataParams_t *) pDataParams)->bAdditionalInfo;
                    break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_X */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_X
                case PHAL_MFUL_SAMAV3_X_ID:
                    *pValue = ((phalMful_SamAV3_X_DataParams_t *) pDataParams)->bAdditionalInfo;
                    break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */

                default:
                    PH_UNUSED_VARIABLE(pValue);
                    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFUL);
            }
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFUL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PHAL_MFUL_ADDITIONAL_INFO:
            switch(PH_GET_COMPID(pDataParams))
            {
#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_X
                case PHAL_MFUL_SAMAV2_X_ID:
                    ((phalMful_SamAV2_X_DataParams_t *) pDataParams)->bAdditionalInfo = (uint8_t) wValue;
                    break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_X */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_X
                case PHAL_MFUL_SAMAV3_X_ID:
                    ((phalMful_SamAV3_X_DataParams_t *) pDataParams)->bAdditionalInfo = (uint8_t) wValue;
                    break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */

                default:
                    PH_UNUSED_VARIABLE(wValue);
                    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFUL);
            }
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFUL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}
#endif /* NXPBUILD__PHAL_MFUL_SAM_X */
