/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software implementation of NTAG X DNA application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7462 $
 * $Date: 2025-08-29 14:09:06 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 22 Aug 2024
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_NTAGXDNA_SW
#include <string.h>
#include <stdlib.h>
#include <phTools.h>
#include <phKeyStore.h>
#include <phCryptoASym.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>

#include <phalNtagXDna.h>
#include "../phalNtagXDna_Int.h"
#include "phalNtagXDna_Sw.h"
#include "phalNtagXDna_Sw_Int.h"





phStatus_t phalNtagXDna_Sw_Init(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams,
    void * pKeyStoreDataParams, void * pCryptoDataParamsASym, void * pCryptoDataParamsEnc, void * pCryptoDataParamsMac,
    void * pCryptoRngDataParams, uint8_t * pCmdBuf, uint16_t wCmdBufSize, uint8_t * pPrsBuf, uint16_t wPrsBufSize)
{
    /* DataParams Size Validation */
    if(sizeof(phalNtagXDna_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_NTAGXDNA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_DATA_PARAM(pPalMifareDataParams, PH_COMP_AL_NTAGXDNA);

#ifdef NXPBUILD__PH_CRYPTOSYM
    PH_ASSERT_NULL_DATA_PARAM(pKeyStoreDataParams, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsASym, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsEnc, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoDataParamsMac, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_DATA_PARAM(pCryptoRngDataParams, PH_COMP_AL_NTAGXDNA);
#endif /* NXPBUILD__PH_CRYPTOSYM */

    PH_ASSERT_NULL_PARAM(pCmdBuf, PH_COMP_AL_NTAGXDNA);
    PH_ASSERT_NULL_PARAM(pPrsBuf, PH_COMP_AL_NTAGXDNA);

    if(wCmdBufSize < PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_NTAGXDNA);

    if(wPrsBufSize < PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_NTAGXDNA);

    if(wPrsBufSize < wCmdBufSize)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_NTAGXDNA);

    pDataParams->wId = PH_COMP_AL_NTAGXDNA | PHAL_NTAGXDNA_SW_ID;

    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
#ifdef NXPBUILD__PH_CRYPTOSYM
    pDataParams->pCryptoDataParamsASym = pCryptoDataParamsASym;
    pDataParams->pCryptoDataParamsEnc = pCryptoDataParamsEnc;
    pDataParams->pCryptoDataParamsMac = pCryptoDataParamsMac;
    pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifndef NXPBUILD__PH_CRYPTOSYM
    PH_UNUSED_VARIABLE(pCryptoDataParamsASym);
    PH_UNUSED_VARIABLE(pCryptoDataParamsEnc);
    PH_UNUSED_VARIABLE(pCryptoDataParamsMac);
    PH_UNUSED_VARIABLE(pCryptoRngDataParams);
#endif /* NXPBUILD__PH_CRYPTOSYM */

    pDataParams->pCmdBuf = pCmdBuf;
    pDataParams->wCmdBufSize = wCmdBufSize;
    pDataParams->wCmdBufLen = 0;
    pDataParams->wCmdBufOffset = 0;

    pDataParams->pPrsBuf = pPrsBuf;
    pDataParams->wPrsBufSize = wPrsBufSize;
    pDataParams->wPrsBufLen = 0;
    pDataParams->wPrsBufOffset = 0;

    (void) memset(pCmdBuf, 0x00, PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM);
    (void) memset(pPrsBuf, 0x00, PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM);
#ifdef NXPBUILD__PH_CRYPTOSYM
    (void) memset(phalNtagXDna_Sw_ZeroIv, 0x00, sizeof(phalNtagXDna_Sw_ZeroIv));
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));
    (void) memset(pDataParams->aSesNonce, 0x00, sizeof(pDataParams->aSesNonce));
    (void) memset(pDataParams->aTi, 0x00, PHAL_NTAGXDNA_SIZE_TI);
#endif /* NXPBUILD__PH_CRYPTOSYM */
    (void) memset(pDataParams->aAid, 0x00, 3U);

    pDataParams->wCmdCtr = 0;
    pDataParams->wAdditionalInfo = 0x0000;
    pDataParams->bAuthState = PHAL_NTAGXDNA_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFF;
    pDataParams->bPICCDataComplete = PH_OFF;
#ifdef NXPBUILD__PH_CRYPTOSYM
    pDataParams->bHasMACProcessed = PH_OFF;
    pDataParams->bIsENCPending = PH_OFF;
#endif /* NXPBUILD__PH_CRYPTOSYM */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_DeInit(phalNtagXDna_Sw_DataParams_t * pDataParams)
{
    phalNtagXDna_Sw_ResetAuthentication(pDataParams);

    pDataParams->wCmdBufSize = 0;
    pDataParams->wPrsBufSize = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA Authentication commands -------------------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_ISOGeneralAuthenticate(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bDeviceCert_Cached, uint8_t bIsHostResponder,
    uint8_t bCertRepoID, uint8_t bCert_Depth, uint8_t bKeySize, uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint16_t wPub_KeyNo, uint16_t wPub_KeyPos,
    uint8_t * pExpRspLen, uint8_t bExpRspLen, phalNtagXDna_Cert stHostCert, phalNtagXDna_Cert * pDeviceCert)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wIsShortLen = 0;
    uint8_t     PH_MEMLOC_REM bLE_Len = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;

    uint8_t     PH_MEMLOC_REM aLE[2U];

    /* Validate Parameters */
    PHAL_NTAGXDNA_VALIDATE_SESSION_KEY_SIZE(bKeySize);
    PHAL_NTAGXDNA_VALIDATE_CERTIFICATE_DEPTH(bCert_Depth);

    /* Clear LE buffer. */
    (void) memset(aLE, 0x00, sizeof(aLE));

    /* Get the current APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_GetConfig(pDataParams, PHAL_NTAGXDNA_SHORT_LENGTH_APDU, &wIsShortLen));

    /* Update LE Length. */
    if(pExpRspLen != NULL)
    {
        bLE_Len = bExpRspLen;
        (void) memcpy(aLE, pExpRspLen, bLE_Len);
    }
    else
    {
        if(wIsShortLen == PH_ON)
        {
            bLE_Len = 1U;
        }
        else
        {
            bLE_Len = 2U;
        }
    }

    /* Update AuthState to Not Authenticated. */
    pDataParams->bAuthState = PHAL_NTAGXDNA_NOT_AUTHENTICATED;

    /* Get the CurveID from Known Private Key (Priv.A). */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetCurveID(
        pDataParams->pKeyStoreDataParams,
        wPriv_KeyNo,
        wPriv_KeyPos,
        &bCurveID));

    /* Generate Host Ephemeral Key pair. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_GenerateKeyPair(
        pDataParams->pCryptoDataParamsASym,
        bCurveID));

    /* Authentication if Host is initiator */
    if(bIsHostResponder == PH_OFF)
    {
        wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator(
            pDataParams,
            bDeviceCert_Cached,
            bCertRepoID,
            bCert_Depth,
            bKeySize,
            wPriv_KeyNo,
            wPriv_KeyPos,
            wPub_KeyNo,
            wPub_KeyPos,
            aLE,
            bLE_Len,
            stHostCert,
            pDeviceCert);
    }

    /* Authentication if Host is Responder */
    else
    {
        wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder(
            pDataParams,
            bDeviceCert_Cached,
            bCertRepoID,
            bCert_Depth,
            bKeySize,
            wPriv_KeyNo,
            wPriv_KeyPos,
            wPub_KeyNo,
            wPub_KeyPos,
            aLE,
            bLE_Len,
            stHostCert,
            pDeviceCert);
    }

    /* Reset Device Certificate in case of Error. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        pDeviceCert->wEndLeaf_Len = 0;
        pDeviceCert->wParent_Len = 0;
        pDeviceCert->wGrandParent_Len = 0;
    }

    return wStatus;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalNtagXDna_Sw_ISOInternalAuthenticate(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bPrivKeyNo, uint8_t bCurveID,
    uint8_t * pPubBKey, uint16_t wPubBKeyLen, uint8_t * pOptsA, uint8_t bOptsALen, uint8_t * pExpRspLen, uint8_t bExpRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wIsShortLen = 0;
    uint16_t    PH_MEMLOC_REM wLC = 0;
    uint8_t     PH_MEMLOC_REM bLE_Len = 0;
    uint8_t     PH_MEMLOC_REM bRndALen = 0;
    uint8_t     PH_MEMLOC_REM bRndBLen = 0;
    uint8_t     PH_MEMLOC_REM bSigLen = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    uint8_t     PH_MEMLOC_REM aLE[2U];
    uint8_t     PH_MEMLOC_REM aCont[2U] = { 0xF0U, 0xF0U };
    uint8_t     PH_MEMLOC_REM *pRndA = NULL;
    uint8_t     PH_MEMLOC_REM *pRndB = NULL;
    uint8_t     PH_MEMLOC_REM *pSignature = NULL;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Validate Curve ID */
    PHAL_NTAGXDNA_VALIDATE_CURVE(bCurveID);

    /* Clear LE buffer. */
    (void) memset(aLE, 0x00, sizeof(aLE));

    /* Get the current APDU format. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_GetConfig(pDataParams, PHAL_NTAGXDNA_SHORT_LENGTH_APDU, &wIsShortLen));

    /* Compute LC */
    wLC = (uint16_t) (bOptsALen + 2U /* AuthDOHdr */ + 18U /* RndA TLV */);

    /* Update LE Length. */
    if(pExpRspLen != NULL)
    {
        bLE_Len = bExpRspLen;
        (void) memcpy(aLE, pExpRspLen, bLE_Len);
    }
    else
    {
        if(wIsShortLen == PH_ON)
        {
            bLE_Len = 1U;
        }
        else
        {
            bLE_Len = 2U;
        }

        /* Force Extend LE information to be exchanged if greater than 255 bytes. */
        if(wLC > 255U)
        {
            bExpRspLen = 2U;
        }
    }

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Set the pointer. */
    pRndA = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 4U)];
    pRndB = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];
    pSignature = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    /* Frame Cmd.ISOInternal Authenticate command -------------------------------------------------------------------------------------- */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_INTERNAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_P1;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrivKeyNo;

    /* Frame Extended LC. */
    if(bLE_Len > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((wLC & 0xFF00U) >> 8U);
    }
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((wLC & 0x00FFU));

    /* Add Command information for exchange. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Add OptsA information for exchange. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_OFF,
        pOptsA,
        bOptsALen,
        NULL,
        NULL,
        NULL));

    /* Reset Command Buffer */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Generate RndA */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Seed(pDataParams->pCryptoRngDataParams, pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE));
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, pRndA));
    bRndALen = PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Add AuthDOHdr (Authentication Data Object Header) to command buffer. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_AUTH_DO_HDR_TAG;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (2U /* Tag (T) + Length (L) */ + bRndALen /* Length of RndA */);

    /* Add Authentication Data Object for RndA. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_RND_TAG;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bRndALen;

    /* Copy RndA to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pRndA, bRndALen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bRndALen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], aLE, bLE_Len);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bLE_Len;

    /* Exchange command to PICC ---------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        &pResponse,
        &wRspLen,
        NULL));

    /* Validate AuthDOHdr from response. */
    if(pResponse[bOffset] != PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_AUTH_DO_HDR_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
    bOffset = 2U /* AuthDOHdr TL */;

    /* Validate Tag information from RndB TLV */
    if(pResponse[bOffset++] != PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_RND_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);

    /* Extract RndB information. */
    bRndBLen = pResponse[bOffset++];
    (void) memcpy(pRndB, &pResponse[bOffset], bRndBLen);
    bOffset += bRndBLen;

    /* Validate Tag information from Signature TLV */
    if(pResponse[bOffset++] != PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_SIGNATURE_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);

    /* Extract Signature information. */
    bSigLen = pResponse[bOffset++];
    (void) memcpy(pSignature, &pResponse[bOffset], bSigLen);

    /* Load public Key to ASymmetric Crypto component ---------------------------------------------------------------------------------- */
    /* Update CurveID based on CryptoASym component. */
    bCurveID = (uint8_t) ((bCurveID == PHAL_NTAGXDNA_TARGET_CURVE_ID_NIST_P256) ? PH_CRYPTOASYM_CURVE_ID_SECP256R1 :
        PH_CRYPTOASYM_CURVE_ID_BRAINPOOL256R1);

    /* Load Public Key for verification. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKeyDirect(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | bCurveID),
        pPubBKey,
        wPubBKeyLen));

    /* Verify the Signature ------------------------------------------------------------------------------------------------------------ */

    /* Add Constant for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        aCont,
        2U,
        NULL,
        0));

    /* Add Opts.A for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pOptsA,
        bOptsALen,
        NULL,
        0));

    /* Add RndB for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pRndB,
        bRndBLen,
        NULL,
        0));

    /* Add RndA for Hashing and verify the signature. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pRndA,
        bRndALen,
        pSignature,
        bSigLen));

    return wStatus;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_AuthenticateEv2(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bFirstAuth, uint16_t wOption, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen, uint8_t * pPcdCapsIn, uint8_t bLenPcdCapsIn,
    uint8_t * pPcdCapsOut, uint8_t * pPdCapsOut)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;
    uint8_t     PH_MEMLOC_REM *pRndA = NULL;
    uint8_t     PH_MEMLOC_REM *pRndB = NULL;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    /* Validate AuthType flag. */
    if(bFirstAuth > PHAL_NTAGXDNA_AUTH_FIRST)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    /* Validate Diversification options. */
    PHAL_NTAGXDNA_VALIDATE_DIVERSIFICATION_OPTIONS(wOption);

    /* Validate Diversification Length. */
    PHAL_NTAGXDNA_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivLen);
#endif

    /* Clear buffers */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];
    pRndA = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pRndB = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate Keytype */
    PHAL_NTAGXDNA_VALIDATE_KEYTYPE(wKeyType);

    /* Diversify the Key. */
    if((wOption != PHAL_NTAGXDNA_NO_DIVERSIFICATION) && (bDivLen != 0x00))
    {
        /* Key is diversified and put back in bKey */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
            pDataParams->pCryptoDataParamsEnc,
            wOption,
            pKey,
            wKeyType,
            pDivInput,
            bDivLen,
            pKey));
    }

    /* Load the key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey,
        wKeyType));

    /* Frame Command PART1 ------------------------------------------------------------------------------------------------------------- */

    /* Frame Command Code */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (bFirstAuth ? PHAL_NTAGXDNA_CMD_AUTHENTICATE_EV2_FIRST :
        PHAL_NTAGXDNA_CMD_AUTHENTICATE_EV2_NON_FIRST);

    /* Frame Card's key number to use. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyNoCard;

    /* Frame Input PCD Capabilities. */
    if(bFirstAuth)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bLenPcdCapsIn;

        /* Copy PCD Capabilities to Command buffer. */
        if(pPcdCapsIn != NULL)
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pPcdCapsIn, bLenPcdCapsIn);
            PHAL_NTAGXDNA_CMD_BUF_LEN += bLenPcdCapsIn;
        }
    }
    else
    {
        /* No PCD Capabilities for Following Auth. */
    }

    /* Exchange Part1 Command. */
    wStatus = phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_NTAGXDNA_OPTION_COMPLETE,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL);

    /* Check for the Chaining active */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
        return wStatus;
    }

    /* Validate Response Buffer length. */
    if(wRespLen != PH_CRYPTOSYM_AES_BLOCK_SIZE)
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    /* Process PART1 Response ---------------------------------------------------------------------------------------------------------- */
    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        sizeof(phalNtagXDna_Sw_ZeroIv)));

    /* Decrypt the RndB received from PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        pResponse,
        wRespLen,
        pRndB));

    /* Frame Command PART2 ------------------------------------------------------------------------------------------------------------- */
    /* Clear command buffer */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Add Command Code to command buffer. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_RESP_ADDITIONAL_FRAME;

    /* Generate RndA */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Seed(pDataParams->pCryptoRngDataParams, pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE));
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams, PH_CRYPTOSYM_AES_BLOCK_SIZE, pRndA));

    /* Copy RndA to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pRndA, PH_CRYPTOSYM_AES_BLOCK_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Copy RndB to Command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pRndB, PH_CRYPTOSYM_AES_BLOCK_SIZE);

    /* Shift RndB by 8 bits. */
    phalNtagXDna_Int_RotateLeft(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], PH_CRYPTOSYM_AES_BLOCK_SIZE, 1U);

    /* Increment command buffer. */
    PHAL_NTAGXDNA_CMD_BUF_LEN += PH_CRYPTOSYM_AES_BLOCK_SIZE;

    /* Load Iv.*/
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        sizeof(phalNtagXDna_Sw_ZeroIv)));

    /* Encrypt RndA + RndB' */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        &PHAL_NTAGXDNA_CMD_BUF[1U]));

    /* Exchange Part2 Command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_NTAGXDNA_OPTION_COMPLETE,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL));

    /* Process PART2 Response ---------------------------------------------------------------------------------------------------------- */
    /* Validate Response Length. */
    if(wRespLen != (bFirstAuth ? (2U * PH_CRYPTOSYM_AES_BLOCK_SIZE) : PH_CRYPTOSYM_AES_BLOCK_SIZE))
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        sizeof(phalNtagXDna_Sw_ZeroIv)));

    /* Decrypt Response to extract Decrypt TI || RndA' || PDCaps || PCDCaps */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
        pResponse,
        wRespLen,
        pResponse));

    /* Define the offset to use. */
    bOffset = (uint8_t) (bFirstAuth ? 4U : 0);

    /* Shift RndA by 8 bits. */
    phalNtagXDna_Int_RotateRight(&pResponse[bOffset], PH_CRYPTOSYM_AES_BLOCK_SIZE, 1U);

    /* Validate RndA ------------------------------------------------------------------------------------------------------------------- */
    if(memcmp(pRndA, &pResponse[bOffset], PH_CRYPTOSYM_AES_BLOCK_SIZE) != 0)
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);

        /* Authentication failed */
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    /* Extract TI, PCD and PD Capabilities */
    if(bFirstAuth)
    {
        (void) memcpy(pDataParams->aTi, &pResponse[0], PHAL_NTAGXDNA_SIZE_TI);
        (void) memcpy(pPdCapsOut, &pResponse[PHAL_NTAGXDNA_SIZE_TI + PH_CRYPTOSYM_AES_BLOCK_SIZE], PHAL_NTAGXDNA_SIZE_CAPS);
        (void) memcpy(pPcdCapsOut, &pResponse[PHAL_NTAGXDNA_SIZE_TI + PH_CRYPTOSYM_AES_BLOCK_SIZE + PHAL_NTAGXDNA_SIZE_CAPS], PHAL_NTAGXDNA_SIZE_CAPS);
    }
    else
    {
        /* Nothing to do for Following Auth */
    }

    /* Generate the Session Keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GenerateSessionKeys_Sym(pDataParams, wKeyType, pRndA, pRndB));

    /* Session key is generated. */
    if(bFirstAuth)
    {
        pDataParams->wCmdCtr = 0x00;
    }
    else
    {
        /* Nothing to do for Following Auth */
    }

    pDataParams->bAuthState = PHAL_NTAGXDNA_EV2_AUTHENTICATED;
    pDataParams->bKeyNo = bKeyNoCard;
    pDataParams->wKeyType = wKeyType;

    /* Load the Session ENC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pDataParams->aSesAuthENCKey,
        wKeyType));

    /* Load the Session MAC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pDataParams->aSesAuthMACKey,
        wKeyType));

    /* Set the keep Iv ON */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_KEEP_IV,
        PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

    /* Clear the local buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;
    PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF_OFFSET = 0;

    PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_OFF;
    PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalNtagXDna_Sw_ProcessSMApply(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCommMode, uint8_t bOffset, uint8_t bCmdCtrIncr,
    uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_PROCESS_SM;

    /* Add Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_PROCESS_SM_ACTION_APPLY;

    /* Add Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_PROCESS_SM_OPERATION_ONE_SHOT;

    /* Add communication mode. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCommMode;

    /* Add Offset and Data. */
    if((bCommMode == PHAL_NTAGXDNA_COMMUNICATION_MAC) ||
        (bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL))
    {
        /* Add Offset in case of FULL mode */
        if(bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL)
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOffset;
        }

        /* Add Data */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;
    }

    /* Add Command Counter increment */
    else
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCmdCtrIncr;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        PHAL_NTAGXDNA_COMMUNICATION_PLAIN,
        PHAL_NTAGXDNA_COMMUNICATION_PLAIN,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ProcessSMRemove(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCommMode, uint8_t * pData, uint8_t bDataLen,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_PROCESS_SM;

    /* Add Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_PROCESS_SM_ACTION_REMOVE;

    /* Add Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_PROCESS_SM_OPERATION_ONE_SHOT;

    /* Add communication mode. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCommMode;

    /* Add Data */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        PHAL_NTAGXDNA_COMMUNICATION_PLAIN,
        PHAL_NTAGXDNA_COMMUNICATION_PLAIN,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA Memory and Configuration management commands ----------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_FreeMem(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t ** ppMemInfo, uint16_t * pMemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t     PH_MEMLOC_REM bCmdCode = 0;

    /* Set the DataParams with command code. */
    bCmdCode = PHAL_NTAGXDNA_CMD_FREE_MEM;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Command Buffer */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Exchange Cmd.FreeMem information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        &bCmdCode,
        1U,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppMemInfo,
        pMemInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_SetConfiguration(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_SET_CONFIGURATION;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOption;

    /* Exchange Cmd.SetConfiguration information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        PHAL_NTAGXDNA_COMMUNICATION_FULL,
        PHAL_NTAGXDNA_COMMUNICATION_MAC,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pData,
        bDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetConfiguration(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bExchangeOption,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Validate ExchangeOption value. */
    if((bExchangeOption != PH_OFF) && (bExchangeOption != PH_ON))
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);

    /* Clear Command Buffer */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_CONFIGURATION;

    if(bExchangeOption == PH_ON)
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOption;

    /* Exchange Cmd.GetConfiguration information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        PHAL_NTAGXDNA_COMMUNICATION_MAC,
        PHAL_NTAGXDNA_COMMUNICATION_FULL,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalNtagXDna_Sw_ActivateConfiguration(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bConfCount, uint8_t * pConfigList,
    uint8_t bConfigList_Len)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_ACTIVATE_CONFIGURATION;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bConfCount;

    /* Exchange Cmd.SetConfiguration information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pConfigList,
        bConfigList_Len,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetVersion(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppVerInfo,
    uint16_t * pVerInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Clear Command Buffer */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Frame GetVersion command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_VERSION;

    /* Append Option information. */
    if(bOption != PHAL_NTAGXDNA_GET_VERSION_EXCLUDE_FAB_ID)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOption;
    }

    /* Exchange Cmd.GetVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppVerInfo,
        pVerInfoLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_GetCardUID(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t ** ppUid, uint16_t * pUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;
    uint8_t     PH_MEMLOC_REM bCmdCode = 0;

    /* Set the DataParams with command code. */
    bCmdCode = PHAL_NTAGXDNA_CMD_GET_CARD_UID;

    /* Clear Command Buffer */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Exchange Cmd.GetVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bComMode,
        PHAL_NTAGXDNA_COMMUNICATION_FULL,
        &bCmdCode,
        1U,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppUid,
        pUidLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */





/* NTAG X DNA Symmetric Key management commands ---------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_ChangeKey(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo,
    uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pKeyPolicy,
    uint8_t * pDivInput, uint8_t bDivLen)
{
    uint16_t    PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwCRC = 0;
    uint16_t    PH_MEMLOC_REM wOldKeyType;
    uint16_t    PH_MEMLOC_REM wNewKeyType;
    uint16_t    PH_MEMLOC_REM wDivOption = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t     PH_MEMLOC_REM bKeySize = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bResetAuth = PH_OFF;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = PH_OFF;
    uint8_t     PH_MEMLOC_REM bComputeCRC = PH_OFF;

    uint8_t     PH_MEMLOC_REM * pCmdData = NULL;
    uint8_t     PH_MEMLOC_REM * pOldKey = NULL;
    uint8_t     PH_MEMLOC_REM * pNewKey = NULL;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM aAppId[3U] = { 0x00, 0x00, 0x00 };

    /* At PICC level, no changeKey can be performed. */
    if(memcmp(pDataParams->aAid, aAppId, 3U) == 0x00)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    /* Key numbers between 0x00 to 0x04 and 0x10 to 0x17 are allowed if AID 0x000000 is not selected. */
    if(PHAL_NTAGXDNA_IS_INVALID_KEYNO(bKeyNoCard & 0x17U))
    {
        /* Invalid application key specified */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    /* Validate Diversification Length. */
    PHAL_NTAGXDNA_VALIDATE_DIVERSIFICATION_LENGTH(wOption, bDivLen);

    /* Validate Option. */
    if(wOption == 0x0000)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
#endif /* RDR_LIB_PARAM_CHECK */

    /* Clear buffer. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Set the pointer for the keys. */
    pNewKey = &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];
    pOldKey = &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Set buffer to use for Command Data. */
    pCmdData = pNewKey;

    /* Frame ChangeKey command information --------------------------------------------------------------------------------------------- */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CHANGE_KEY;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyNoCard;

    if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyNoCard & 0x17U))
    {
        if(pKeyPolicy == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pKeyPolicy, 2U);
            PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;
        }
    }

    /* Get Old and New Keys ------------------------------------------------------------------------------------------------------------ */
    /* Get the Current Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wCurrKeyNo,
        wCurrKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pOldKey,
        &wOldKeyType));

    /* Verify KeyType. */
    PHAL_NTAGXDNA_VALIDATE_KEYTYPE(wOldKeyType);

    /* Get the New Key information from KeyStore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wNewKeyNo,
        wNewKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pNewKey,
        &wNewKeyType));

    /* Verify KeyType. */
    PHAL_NTAGXDNA_VALIDATE_KEYTYPE(wNewKeyType);

    /* Set the Size of the Key. */
    bKeySize = phKeyStore_GetKeySize(wNewKeyType);

    /* Diversify Keys ------------------------------------------------------------------------------------------------------------------ */
    /* Diversify Current and New Keys. */
    if((wOption != PHAL_NTAGXDNA_NO_DIVERSIFICATION) && (bDivLen != 0x00))
    {
        /* Diversify Old Key */
        if(wOption & PHAL_NTAGXDNA_CHGKEY_DIV_OLD_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_NTAGXDNA_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pOldKey,
                wOldKeyType,
                pDivInput,
                bDivLen,
                pOldKey));
        }

        /* Diversify New Key */
        if(wOption & PHAL_NTAGXDNA_CHGKEY_DIV_NEW_KEY)
        {
            wDivOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
            if(wOption & PHAL_NTAGXDNA_CHGKEY_DIV_METHOD_CMAC)
            {
                wDivOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
            }

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
                pDataParams->pCryptoDataParamsEnc,
                wDivOption,
                pNewKey,
                wNewKeyType,
                pDivInput,
                bDivLen,
                pNewKey));
        }

        /* Reload the IV and key since the diversify function has invalidated the key */
        /* Load the Session key which is valid for this authentication */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pDataParams->aSesAuthENCKey,
            pDataParams->wKeyType));

        /* Enable storing of Initialization vector. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_CONFIG_KEEP_IV,
            PH_CRYPTOSYM_VALUE_KEEP_IV_ON));
    }

    /* Generate Cryptogram ------------------------------------------------------------------------------------------------------------- */
    /* New Key is required in Crypto framing.
    * The new key already part of processing buffer so no need or further processing.
    * Increment the processing buffer include Key size.
    */
    wCmdDataLen += bKeySize;

    /* Copy Version */
    pCmdData[wCmdDataLen++] = (uint8_t) wNewKeyVer;

    /* Append CRC if,
     *  - Targeted key different from authenticated key
     *  - If EE Authentication is performed.
     *  - If not CryptoRequest Keys (0x10 - 0x17)
     */
    bComputeCRC = (uint8_t) ((pDataParams->bKeyNo & 0x3FU) != (bKeyNoCard & 0x3FU));
    bComputeCRC = (uint8_t) ((pDataParams->bAuthState == PHAL_NTAGXDNA_ECC_AUTHENTICATED) ? PH_ON : bComputeCRC);
    bComputeCRC = (uint8_t) (PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyNoCard & 0x17U) ? PH_OFF : bComputeCRC);

    if(bComputeCRC == PH_ON)
    {
        /* Compute CRC. */
        PH_CHECK_SUCCESS_FCT(wStatus, phTools_CalculateCrc32(
            PH_TOOLS_CRC_OPTION_DEFAULT,
            PH_TOOLS_CRC32_PRESET_DF8,
            PH_TOOLS_CRC32_POLY_DF8,
            pCmdData,
            (uint16_t) (wCmdDataLen - 1U),
            &dwCRC));

        /* Add CRC to command buffer. */
        (void) memcpy(&pCmdData[wCmdDataLen], &dwCRC, 4U);
        wCmdDataLen += 4U;

        /* Compute XOR between New Key and Old Key. */
        for(bIndex = 0; bIndex < bKeySize; bIndex++)
        {
            pNewKey[bIndex] = pOldKey[bIndex] ^ pNewKey[bIndex];
        }
    }

    /* Targeted key equal to authenticated key */
    else
    {
        /* Nothing to do here as the cryptogram is already available as part of command buffer. */
    }

    /* Should the Authenticate be reset. */
    bRsp_ComMode = PHAL_NTAGXDNA_COMMUNICATION_MAC;
    if((pDataParams->bKeyNo & 0x3FU) == (bKeyNoCard & 0x3FU))
    {
        bResetAuth = PH_ON;
        bRsp_ComMode = PHAL_NTAGXDNA_COMMUNICATION_PLAIN;
    }

    /* Exchange Cmd.ChangeKey to PICC -------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        PHAL_NTAGXDNA_COMMUNICATION_FULL,
        bRsp_ComMode,
        bResetAuth,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalNtagXDna_Sw_GetKeySettings(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_KEY_SETTINGS;

    /* Add Option information to command buffer. */
    if(bOption != PHAL_NTAGXDNA_KEY_SETTING_EXCLUDE_OPTION)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOption;
    }

    /* Exchange Cmd.GetKeySettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetKeyVersion(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM aAppId[3U] = { 0x00, 0x00, 0x00 };

    /* At PICC level, no changeKey can be performed. */
    if(memcmp(pDataParams->aAid, aAppId, 3U) == 0x00)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    /* Key numbers between 0x00 to 0x04 and 0x10 to 0x17 are allowed if AID 0x000000 is not selected. */
    if(PHAL_NTAGXDNA_IS_INVALID_KEYNO(bKeyNo))
    {
        /* Invalid application key specified */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
#endif /* RDR_LIB_PARAM_CHECK */

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_KEY_VERSION;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyNo;

    /* Exchange Cmd.GetKeyVersion information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA ASymmetric Key management commands --------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_ManageKeyPair(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bKeyNo,
    uint8_t bOption, uint8_t bCurveID, uint8_t * pKeyPolicy, uint8_t bWriteAccess, uint32_t dwKUCLimit,
    uint16_t wPrivKey_No, uint16_t wPrivKey_Pos, uint8_t ** ppResponse, uint16_t *pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdData_Len = 0;

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bComOption));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_MANAGE_KEY_PAIR;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyNo;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOption;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCurveID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pKeyPolicy[0];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pKeyPolicy[1U];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bWriteAccess;

    /* Append the Key Usage Counter Limit. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], (uint8_t *) &dwKUCLimit, 4U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 4U;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN];

    /* Add PrivateKey to command buffer. */
    if(bOption == PHAL_NTAGXDNA_TARGET_ACTION_IMPORT_PRIVATE_KEY)
    {
        /* Get the Private Key from KeyStore. */
        PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKeyASym(
            pDataParams->pKeyStoreDataParams,
            wPrivKey_No,
            wPrivKey_Pos,
            PH_KEYSTORE_KEY_PAIR_PRIVATE,
            &wKeyType,
            &bCurveID,
            pCmdData,
            &wCmdData_Len));
    }

    /* Exchange Cmd.ManageKeyPair information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pCmdData,
        wCmdData_Len,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ManageCARootKey(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bKeyNo,
    uint8_t bCurveID, uint8_t * pAccessRights, uint8_t bWriteAccess, uint16_t wPubKey_No, uint16_t wPubKey_Pos,
    uint8_t * pIssuer, uint8_t bIssuerLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint16_t    PH_MEMLOC_REM wCmdData_Len = 0;

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bComOption));

    /* Check for Issuer pointer. */
    if(bIssuerLen)
        PH_ASSERT_NULL_PARAM(pIssuer, PH_COMP_AL_NTAGXDNA);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_MANAGE_CA_ROOT_KEY;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyNo;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCurveID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pAccessRights[0];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pAccessRights[1U];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bWriteAccess;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x3FU;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x00U;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN];

    /* Add PublicKey to command buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKeyASym(
        pDataParams->pKeyStoreDataParams,
        wPubKey_No,
        wPubKey_Pos,
        PH_KEYSTORE_KEY_PAIR_PUBLIC,
        &wKeyType,
        &bCurveID,
        pCmdData,
        &wCmdData_Len));

    /* Add Issuer to Command buffer. */
    pCmdData[wCmdData_Len++] = bIssuerLen;

    (void) memcpy(&pCmdData[wCmdData_Len], pIssuer, bIssuerLen);
    wCmdData_Len += bIssuerLen;

    /* Exchange Cmd.ManageCARootKey information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bComOption,
        bComOption,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pCmdData,
        wCmdData_Len,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA Certificate management commands ------------------------------------------------------------------------------------------ */
phStatus_t phalNtagXDna_Sw_ManageCertRepo(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bComOption, uint8_t bAction, uint8_t bRepoID,
    uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption((uint8_t) bComOption));

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, (uint8_t) bComOption, &bCmd_ComMode);
    bRsp_ComMode = (uint8_t) bComOption;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_MANAGE_CERT_REPO;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bAction;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bRepoID;

    /* Exchange Cmd.ManageKeyPair information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pData,
        wDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ReadCertRepo(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bRepoID, uint8_t bDataItem,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bExchange_Option = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication and Exchange Options. */
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bExchange_Option = (uint8_t) (bOption & PH_EXCHANGE_MODE_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bRsp_ComMode));

    /* Validate Exchange Options. */
    PHAL_NTAGXDNA_VALIDATE_RX_EXCHANGE_OPTIONS(bExchange_Option);

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;


    /* Frame the command information. */
    if(bExchange_Option != PH_EXCHANGE_RXCHAINING)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_READ_CERT_REPO;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bRepoID;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDataItem;
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.ReadData information to PICC. */
    wStatus = phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        bExchange_Option,
        PH_ON,
        bCmd_ComMode,
        bRsp_ComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppResponse,
        pRspLen);

    /* Validate Status. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        return wStatus;

    return wStatus;
}




/* NTAG X DNA File management commands ------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_CreateStdDataFile(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t * pISOFileId, uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pFileSize)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bOption > PHAL_NTAGXDNA_ISO_FILE_ID_AVAILABLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
    PHAL_NTAGXDNA_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CREATE_STANDARD_DATA_FILE;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Add ISO File ID to command buffer. */
    if(bOption == PHAL_NTAGXDNA_ISO_FILE_ID_AVAILABLE)
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pISOFileId, 2U);
        PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;
    }

    /* Add File Options to command buffer. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;

    /* Add File Size to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pFileSize, 3U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.CreateStdDataFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CreateCounterFile(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t bFileOption,
    uint8_t * pAccessRights, uint32_t dwValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
    PHAL_NTAGXDNA_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CREATE_COUNTER_FILE;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Add File Options to command buffer. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileOption;

    /* Add Access Rights to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pAccessRights, 2U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;

    /* Add File Size to command buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], (uint8_t *) &dwValue, 4U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 4U;

    /* Exchange Cmd.CreateStdDataFile information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bComMode,
        bComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        0,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetFileIDs(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t ** ppFileId, uint16_t * pFileIdLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_FILE_IDS;

    /* Exchange Cmd.GetFileIDs information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppFileId,
        pFileIdLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetISOFileIDs(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t ** ppISOFileId, uint16_t * pISOFileIdLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_ISO_FILE_IDS;

    /* Exchange Cmd.GetISOFileIDs information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppISOFileId,
        pISOFileIdLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetFileSettings(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t ** ppFSBuffer,
    uint16_t * pFSBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, PHAL_NTAGXDNA_COMMUNICATION_INVALID,
        &bComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_FILE_SETTINGS;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.GetFileSettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        bComMode,
        bComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppFSBuffer,
        pFSBufLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_GetFileCounters(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t ** ppFileCounters, uint16_t * pFileCounterLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_GET_FILE_COUNTERS;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.GetFileCounters information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_ON,
        bCmd_ComMode,
        bOption,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN,
        ppFileCounters,
        pFileCounterLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ChangeFileSettings(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint8_t bFileOption, uint8_t * pAccessRights, uint8_t * pAddInfo, uint8_t bAddInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;
    uint8_t *   PH_MEMLOC_REM pCmdData = NULL;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bOption));
    PHAL_NTAGXDNA_VALIDATE_FILE_OPTIONS(bFileOption);
#endif

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CHANGE_FILE_SETTINGS;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Set buffer to use for Command Data. */
    pCmdData = &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN];

    /* Add FileOption to processing buffer. */
    pCmdData[wCmdDataLen++] = bFileOption;

    /* Add Access Rights to processing buffer. */
    (void) memcpy(&pCmdData[wCmdDataLen], pAccessRights, 2U);
    wCmdDataLen += 2U;

    /* Add File Size to processing buffer. */
    (void) memcpy(&pCmdData[wCmdDataLen], pAddInfo, bAddInfoLen);
    wCmdDataLen += bAddInfoLen;

    /* Exchange Cmd.ChangeFileSettings information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bOption,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pCmdData,
        wCmdDataLen,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA Data management commands ------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_ReadData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pLength, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bExchange_Option = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication and Exchange Options. */
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bExchange_Option = (uint8_t) (bOption & PH_EXCHANGE_MODE_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bRsp_ComMode));

    /* Validate Exchange Options. */
    PHAL_NTAGXDNA_VALIDATE_RX_EXCHANGE_OPTIONS(bExchange_Option);

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bCmd_ComMode);

    /* Compute the length. */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0];

    /* Frame the command information. */
    if(bExchange_Option != PH_EXCHANGE_RXCHAINING)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
        PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

        /* Add command code. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_READ_DATA;

        /* Add File Number. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

        /* Add offset. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pOffset, 3U);
        PHAL_NTAGXDNA_CMD_BUF_LEN += 3U;

        /* Add length. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pLength, 3U);
        PHAL_NTAGXDNA_CMD_BUF_LEN += 3U;
    }
    else
    {
        /* Chaining is handled internally. */
    }

    /* Exchange Cmd.ReadData information to PICC. */
    wStatus = phalNtagXDna_Sw_Int_ReadData(
        pDataParams,
        bExchange_Option,
        PH_ON,
        bCmd_ComMode,
        bRsp_ComMode,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        dwLength,
        ppResponse,
        pRspLen);

    /* Validate Status. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        return wStatus;

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_WriteData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset,
    uint8_t * pData, uint8_t * pLength)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwLength = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Frame the communication mode to be verified. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Compute the Total length (Data + [CRL Signature]). */
    dwLength = pLength[2U];
    dwLength = dwLength << 8U | pLength[1U];
    dwLength = dwLength << 8U | pLength[0];

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_WRITE_DATA;

    /* Add File Number. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Add offset. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pOffset, 3U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 3U;

    /* Add length. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pLength, 3U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 3U;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        pData,
        dwLength,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_IncrementCounterFile(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
    uint32_t dwIncrValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

#ifdef RDR_LIB_PARAM_CHECK
    PHAL_NTAGXDNA_IS_VALID_FILE_NO(bFileNo);
#endif

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Frame the communication mode to be verified. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, bOption, &bRsp_ComMode);

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_INCREMENT_COUNTER_FILE;

    /* Add File Number. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bFileNo;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        (uint8_t *) &dwIncrValue,
        4U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA CryptoAPI commands ------------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_CryptoRequestSHA(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bAlgorithm, uint8_t bInputDataSource, uint8_t bResultDst, uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_SHA;

    /* Add SHA Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add Algorithm. */
    if((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) ||
        (bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_ONE_SHOT))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bAlgorithm;
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;
    }

    /* Add Result Destination. */
    if((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE) ||
        (bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_ONE_SHOT))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestRNG(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bNumBytes,
    uint8_t bResultDst, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_RNG;

    /* Add Number of Random bytes required. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bNumBytes;

    /* Add Result Destination. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestECCSign(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bPrivateKeyID, uint8_t bInputDataSource, uint8_t * pData, uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_SIGN;

    /* Add ECCSign Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add ECCSign Algorithm. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x00;

    /* Add PrivateKey ID. */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrivateKeyID;
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestECCVerify(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bCurveID, uint8_t * pHostPubKey, uint8_t bHostPubKeyLen, uint8_t * pSignature, uint8_t bSigLen, uint8_t bInputDataSource,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_VERIFY;

    /* Add ECCVerify Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add Algorithm, Curve IDa and Host Public Key. */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE))
    {
        /* Add ECCVerify Algorithm. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x00;

        /* Add Curve ID */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCurveID;

        /* Add Host Public Key */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pHostPubKey, bHostPubKeyLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bHostPubKeyLen;
    }

    /* Add Signature */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE))
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pSignature, bSigLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bSigLen;
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        &pResponse,
        &wRspLen));

    /* Verify the response for Successful verification */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE))
    {
        if((pResponse[0] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS) &&
            (pResponse[1U] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS))
        {
            wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_SIGNATURE_VERIFICATION_FAILURE, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_CryptoRequestECCDH(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bKeyPairID, uint8_t bSSDestination, uint8_t * pHostPubKey, uint8_t bHostPubKeyLen, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_DH;

    /* Add ECC DH Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add ECC DH KeyPair ID. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyPairID;

    /* Add Shared Secret Destination and Host Public Key. */
    if(bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_TWO_STEP_FIRST)
    {
        /* Add Shared Secret Destination */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bSSDestination;

        /* Add Host Public Key */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pHostPubKey, bHostPubKeyLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bHostPubKeyLen;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestAES(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bPrimitive, uint8_t bKeyID, uint8_t bKeyLen, uint8_t bICVSource, uint8_t * pICV, uint8_t bICVLen,
    uint8_t bInputDataSource, uint8_t bResultDst, uint8_t * pInputData, uint8_t bInputDataLen,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES;

    /* Add AES Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add AES Primitive, KeyId, KeyLen and ICV. */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE))
    {
        /* Add AES Primitive. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrimitive;

        /* Add AES KeyID. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyID;

        /* Add AES Key Len. */
        if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyID) == PH_OFF)
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyLen;
        }

        /* Add ICV Source and ICV. */
        if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CBC_ENCRYPT) ||
            (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CBC_DECRYPT))
        {
            /* Add ICV Source */
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bICVSource;

            /* Add ICV */
            if(bICVSource == PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
            {
                (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pICV, bICVLen);
                PHAL_NTAGXDNA_CMD_BUF_LEN += bICVLen;
            }
        }
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInputData, bInputDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bInputDataLen;
    }

    /* Add Result Destination. */
    if((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_ONE_SHOT))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestAESCMAC(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bPrimitive, uint8_t bKeyID, uint8_t bKeyLen, uint8_t * pCMACSignature, uint8_t bCMACSignLen, uint8_t bInputDataSource,
    uint8_t * pInputData, uint8_t bInputDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_SIGN) ?
        PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_CMAC_SIGN : PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_CMAC_VERIFY);

    /* Add AES Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add AES Primitive, KeyId and KeyLen. */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE))
    {
        /* Add AES Primitive. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrimitive;

        /* Add AES KeyID. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyID;

        /* Add AES Key Len. */
        if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyID) == PH_OFF)
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyLen;
        }
    }

    /* Add CMAC Signature. */
    if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_VERIFY) &&
        ((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE)))
    {
        /* Add CMAC Signature length. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCMACSignLen;

        /* Add CMAC Signature. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pCMACSignature, bCMACSignLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bCMACSignLen;
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInputData, bInputDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bInputDataLen;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    /* Verify the response for Successful verification */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_VERIFY))
    {
        if((ppResponse[0][0] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS) &&
            (ppResponse[0][1U] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS))
        {
            wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_SIGNATURE_VERIFICATION_FAILURE, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
            *pRspLen -= 2U;
        }
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_CryptoRequestAESAEAD(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation,
    uint8_t bPrimitive, uint8_t bKeyID, uint8_t bKeyLen, uint8_t bNonceSource, uint8_t * pNonce, uint8_t bNonceLen, uint16_t wTotal_AAdLen,
    uint16_t wTotal_InputDataLen, uint8_t bAADSource, uint8_t * pAAD, uint8_t bAADLen, uint8_t bInputDataSource, uint8_t * pInputData,
    uint8_t bInputDataLen, uint8_t * pTagData, uint8_t bTagDataLen, uint8_t bResultDst, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_DECRYPT_VERIFY) ||
        (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_DECRYPT_VERIFY))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_AEAD_DECRYPT_VERIFY;
    }
    else
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_AEAD_ENCRYPT_SIGN;
    }

    /* Add AES AEAD Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add Primitive, KeyID, KeyLen Nonce Source, Nonce Length and Nonce */
    if((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) ||
        (bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_ONE_SHOT))
    {
        /* Add AES AEAD Primitive. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrimitive;

        /* Add AES AEAD KeyID. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyID;

        /* Add AES AEAD Key Len. */
        if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyID) == PH_OFF)
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyLen;
        }

        /* Add Nonce Source. */
        if((bPrimitive != PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_ENCRYPT_SIGN_INTERNAL_NONCE) &&
            (bPrimitive != PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_ENCRYPT_SIGN_INTERNAL_NONCE))
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bNonceSource;
        }

        /* Add Nonce Length. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bNonceLen;

        /* Add Nonce. */
        if((bPrimitive != PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_ENCRYPT_SIGN_INTERNAL_NONCE) &&
            (bPrimitive != PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_ENCRYPT_SIGN_INTERNAL_NONCE))
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pNonce, bNonceLen);
            PHAL_NTAGXDNA_CMD_BUF_LEN += bNonceLen;
        }
    }

    /* Add Total AAD length, Total Data Length and Tag Length */
    if(bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT)
    {
        /* Add Total AAD Length. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], (uint8_t *) &wTotal_AAdLen, 2U);
        PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;

        /* Add Total Data Length. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], (uint8_t *) &wTotal_InputDataLen, 2U);
        PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;

        /* Add Tag Length. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bTagDataLen;
    }

    /* Add AAD Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bAADSource;

    /* Add AAD length. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bAADLen;

    /* Add AAD. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pAAD, bAADLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bAADLen;

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataLen;

    /* Add Input Data */
    if(bInputDataSource == PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInputData, bInputDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bInputDataLen;
    }

    /* Add Tag Length and Tag Data. */
    if((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_ONE_SHOT) ||
        ((bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE) &&
        ((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_DECRYPT_VERIFY) ||
        (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_DECRYPT_VERIFY))))
    {
        /* Add Tag Length */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bTagDataLen;

        /* Add Tag Data */
        if(pTagData != NULL)
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pTagData, bTagDataLen);
            PHAL_NTAGXDNA_CMD_BUF_LEN += bTagDataLen;
        }
    }

    /* Add Result Destination. */
    if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_DECRYPT_VERIFY) ||
        (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_DECRYPT_VERIFY))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    /* Verify the response for Successful verification */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        ((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_CCM_DECRYPT_VERIFY) ||
         (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_GCM_DECRYPT_VERIFY)))
    {
        if((ppResponse[0][*pRspLen - 2U] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS) &&
           (ppResponse[0][*pRspLen - 1U] != PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS))
        {
            wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_SIGNATURE_VERIFICATION_FAILURE, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
            *pRspLen -= 2U;
        }
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return PH_ADD_COMPCODE(wStatus, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestWriteInternalBuffer(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bDestination,
    uint8_t * pData, uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_WRITE_INTERNAL_BUFFER;

    /* Add Destination */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDestination;

    /* Add Data Length */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDataLen;

    /* Add Data */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestHMAC(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation, uint8_t bPrimitive,
    uint8_t bDigestAlgo, uint8_t bKeyID, uint8_t bKeyLen, uint8_t * pHMac, uint8_t bHMacLen, uint8_t bInputDataSource, uint8_t * pInputData,
    uint8_t bInputDataLen, uint8_t bResultDst, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_HMAC;

    /* Add HMAC Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add HMAC Primitive. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPrimitive;

    /* Add HMAC Digest Algorithm, KeyId and KeyLen. */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_FINALIZE))
    {
        /* Add HMAC Digest Algorithm. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDigestAlgo;

        /* Add HMAC KeyID. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyID;

        /* Add HMAC Key Len. */
        if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyID) == PH_OFF)
        {
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyLen;
        }
    }

    /* Add HMAC Signature. */
    if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_VERIFY) &&
        ((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE)))
    {
        /* Add HMAC Signature. */
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pHMac, bHMacLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bHMacLen;
    }

    /* Add Input Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataSource;

    /* Add Input Data length. */
    if(bInputDataSource != PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInputDataLen;
    }

    /* Add Data */
    else
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInputData, bInputDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bInputDataLen;
    }

    /* Add Result Destination. */
    if((bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_SIGN) &&
        ((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE)))
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;
    }

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    /* Verify the response for Successful verification */
    if((bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_INIT) &&
        (bOperation != PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_UPDATE) &&
        (bPrimitive == PHAL_NTAGXDNA_CRYPTOAPI_PRIMITIVE_VERIFY))
    {
        if((ppResponse[0][0] != (uint8_t) ((PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS & 0xFF00) >> 8U)) &&
            (ppResponse[0][1U] != (uint8_t) (PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS & 0x00FF)))
        {
            wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_SIGNATURE_VERIFICATION_FAILURE, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_CryptoRequestHKDF(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bOperation, uint8_t bDigestAlgo,
    uint8_t bKeyID, uint8_t bKeyLen, uint8_t bSaltSource, uint8_t * pSaltData, uint8_t bSaltDataLen, uint8_t bInfoSource, uint8_t * pInfoData,
    uint8_t bInfoDataLen, uint8_t bResultDst, uint8_t bResultLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_HKDF;

    /* Add HKDF Operation to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Add HKDF Digest Algorithm. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bDigestAlgo;

    /* Add HKDF KeyID. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyID;

    /* Add HKDF Key Len. */
    if(PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(bKeyID) == PH_OFF)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeyLen;
    }

    /* Add Salt Source, Salt Data and Salt Length. */
    if(bOperation == PHAL_NTAGXDNA_CRYPTOAPI_OPERATION_EXTRACT_EXPAND)
    {
        /* Add Salt Source. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bSaltSource;

        /* Add Salt Length. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bSaltDataLen;

        /* Add Salt Data. */
        if(bSaltSource == PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pSaltData, bSaltDataLen);
            PHAL_NTAGXDNA_CMD_BUF_LEN += bSaltDataLen;
        }
    }

    /* Add Info Data Source. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInfoSource;

    /* Add Info Data length. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bInfoDataLen;

    /* Add Info Data */
    if(bInfoSource == PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER)
    {
        (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInfoData, bInfoDataLen);
        PHAL_NTAGXDNA_CMD_BUF_LEN += bInfoDataLen;
    }

    /* Add Result Destination. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultDst;

    /* Add Result Length. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bResultLen;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_CryptoRequestECHO(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData,
    uint8_t bDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Extract Communication option */
    bCmd_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);
    bRsp_ComMode = (uint8_t) (bOption & PHAL_NTAGXDNA_COMM_OPTIONS_MASK);

    /* Validate Communication Options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption(bCmd_ComMode));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Add command code. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST;

    /* Add Crypto Action to perform. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECHO;

    /* Add Data */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pData, bDataLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bDataLen;

    /* Exchange Cmd.WriteData information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        1U,
        &PHAL_NTAGXDNA_CMD_BUF[1U],
        (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - 1U),
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA GPIO Management commands ------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_ManageGPIO(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bGPIONo, uint8_t bOperation,
    uint8_t * pNFCPauseRspData, uint16_t wNFCPauseRspDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption_PICC = 0;
    uint16_t    PH_MEMLOC_REM wTotalLen = 0;
    uint8_t     PH_MEMLOC_REM bBuffOption_SM = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM bCmdOption = 0;
    uint8_t     PH_MEMLOC_REM bComplete = PH_OFF;

    uint8_t     PH_MEMLOC_REM *pSMBuff = 0;
    uint16_t    PH_MEMLOC_REM wSmBufLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;

    /* To resolve warning */
    PH_UNUSED_VARIABLE(bBuffOption_SM);

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption((uint8_t) wOption));

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, (uint8_t) wOption,
        &bCmd_ComMode);
    bRsp_ComMode = (uint8_t) wOption;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_MANAGE_GPIO;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bGPIONo;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bOperation;

    /* Compute Total frame length (CmdHdr + CmdData (With or Without MAC)) */
#ifdef NXPBUILD__PH_CRYPTOSYM
    wTotalLen = (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN + wNFCPauseRspDataLen +
        ((bCmd_ComMode == PHAL_NTAGXDNA_COMMUNICATION_MAC) ? 8U : 0U));
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifndef NXPBUILD__PH_CRYPTOSYM
    wTotalLen = (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN + wNFCPauseRspDataLen);
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Buffer ManageGPIO command ------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_NTAGXDNA_OPTION_NONE,
        wTotalLen,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Set buffering options. */
    bBuffOption_SM = PHAL_NTAGXDNA_RETURN_PLAIN_DATA;
    wBuffOption_PICC = PH_EXCHANGE_BUFFER_CONT;
    bCmdOption = PHAL_NTAGXDNA_OPTION_NONE;

    /* Generate the Secure Messaging --------------------------------------------------------------------------------------------------- */
    do
    {
        wStatus = phalNtagXDna_Sw_Int_ApplySM(
            pDataParams,
            PH_ON,
            PH_ON,
            bCmd_ComMode,
            PHAL_NTAGXDNA_CMD_BUF,
            PHAL_NTAGXDNA_CMD_BUF_LEN,
            pNFCPauseRspData,
            wNFCPauseRspDataLen,
            &pSMBuff,
            &wSmBufLen);

        /* Update SM Buffering options. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
            bBuffOption_SM = (uint8_t) (PH_EXCHANGE_TXCHAINING | PHAL_NTAGXDNA_RETURN_PLAIN_DATA);

        /* Validate Status */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            break;

        /* End the Loop. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            bComplete = PH_ON;

            /* Re-Apply Buffer and Command Option flags. */
            wBuffOption_PICC = PH_EXCHANGE_BUFFER_LAST;
            bCmdOption = (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_RETURN_PICC_STATUS);
        }

        /* Exchange NFCPauseRspData and MAC (If Applicable) ---------------------------------------------------------------------------- */
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
            pDataParams,
            wBuffOption_PICC,
            bCmdOption,
            0,
            PH_ON,
            pSMBuff,
            wSmBufLen,
            &pResponse_PICC,
            &wRspLen_PICC,
            &bPiccErrCode));
    } while(bComplete == PH_OFF);

    /* Validate the Status. */
    PH_CHECK_SUCCESS(wStatus);

    /* Remove the Secure Messaging ----------------------------------------------------------------------------------------------------- */
    if(bRsp_ComMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_RemoveSM(
            pDataParams,
            PH_ON,
            PH_ON,
            PH_ON,
            bRsp_ComMode,
            pResponse_PICC,
            wRspLen_PICC,
            bPiccErrCode,
            ppResponse,
            pRspLen));
    }
    else
    {
        /* In case of SUCCESS and Communication mode as PLAIN, increment the command counter. */
        pDataParams->wCmdCtr++;

        /* Copy response data to parameter */
        if(pRspLen != NULL)
        {
            *ppResponse = pResponse_PICC;
            *pRspLen = wRspLen_PICC;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ReadGPIO(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmd_ComMode = 0;
    uint8_t     PH_MEMLOC_REM bRsp_ComMode = 0;

    /* Validate Communication options. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_Validate_ComOption((uint8_t) wOption));

    /* Frame the communication mode to be applied. */
    phalNtagXDna_Int_GetCommMode(pDataParams->bAuthState, (uint8_t) wOption, &bCmd_ComMode);
    bRsp_ComMode = (uint8_t) wOption;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame the command information. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_READ_GPIO;

    /* Exchange Cmd.ReadGPIO information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_WriteData(
        pDataParams,
        bCmd_ComMode,
        bRsp_ComMode,
        PH_OFF,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        0,
        ppResponse,
        pRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}





/* NTAG X DNA ISO7816-4 commands ------------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_IsoSelectFile(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid,
    uint8_t * pDFname, uint8_t bDFnameLen, uint8_t bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM aData[24];
    uint16_t    PH_MEMLOC_REM wLc = 0;
    uint16_t    PH_MEMLOC_REM wLe = 0;
    uint8_t     PH_MEMLOC_REM aFileId[3] = { '\0' };
    uint8_t     PH_MEMLOC_REM aPiccDfName[7] = { 0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00 };
    uint8_t     PH_MEMLOC_REM bInclude_LCLE = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if(bDFnameLen > 16U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
    if((bSelector == PHAL_NTAGXDNA_SELECTOR_4) && (bDFnameLen == 0x00U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
    if((bOption != PHAL_NTAGXDNA_FCI_RETURNED) && (bOption != PHAL_NTAGXDNA_FCI_NOT_RETURNED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
    if((bSelector != PHAL_NTAGXDNA_SELECTOR_4) && (bSelector != PHAL_NTAGXDNA_SELECTOR_3) && (pFid == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }
#endif

    /* Validate APDU Format. */
    PHAL_NTAGXDNA_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    switch(bSelector)
    {
        case PHAL_NTAGXDNA_SELECTOR_0: /* Select MF, DF or EF, by file identifier */
        case PHAL_NTAGXDNA_SELECTOR_1: /* Select child DF */
        case PHAL_NTAGXDNA_SELECTOR_2: /* Select EF under the current DF, by file identifier */
        case PHAL_NTAGXDNA_SELECTOR_3: /* Select parent DF of the current DF */
                                       /* Selection by EF Id*/
                                       /* Send MSB first to card */
            if(pFid != NULL)
            {
                aFileId[1U] = aData[0] = pFid[1U];
                aFileId[0] = aData[1U] = pFid[0];
                aFileId[2U] = 0x00;
                wLc = 2U;
            }
            break;

        case PHAL_NTAGXDNA_SELECTOR_4: /* Select by DF name. */
            (void) memcpy(aData, pDFname, bDFnameLen);
            wLc = bDFnameLen;
            break;

        default:
            break;
            /*return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);*/
    }

    /* Update LC LE information for exchange. */
    bInclude_LCLE = (uint8_t) ((bOption == PHAL_NTAGXDNA_FCI_NOT_RETURNED) ? PHAL_NTAGXDNA_EXCHANGE_LC_ONLY :
        PHAL_NTAGXDNA_EXCHANGE_LC_LE_BOTH);

    /* Exclude LC and LE if Selector == 0x03 */
    bInclude_LCLE = (uint8_t) ((bSelector == PHAL_NTAGXDNA_SELECTOR_3) ? PHAL_NTAGXDNA_EXCLUDE_LC_LE :
        bInclude_LCLE);

    wStatus = phalNtagXDna_Sw_Int_Send7816Apdu(
        pDataParams,
        bInclude_LCLE,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_NTAGXDNA_ISO7816_GENERIC_CLA,
        PHAL_NTAGXDNA_CMD_ISO7816_SELECT_FILE,
        bSelector,
        bOption,
        aData,
        wLc,
        wLe,
        ppFCI,
        pFCILen);

    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Reset Authentication should not be targeted for elementary file selection using file ID */
        if(bSelector != 0x02U)
        {
            /* Reset Authentication Status here */
            phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
        }

        /* Once the selection Success, update the File Id to master data structure if the selection is done through AID */
        if((bSelector == PHAL_NTAGXDNA_SELECTOR_0) || (bSelector == PHAL_NTAGXDNA_SELECTOR_1) ||
            (bSelector == PHAL_NTAGXDNA_SELECTOR_2))
        {
            (void) memcpy(pDataParams->aAid, aFileId, sizeof(aFileId));
        }
        else if(bSelector == PHAL_NTAGXDNA_SELECTOR_4)
        {
            /* Update the file ID to all zeros if DF Name is of PICC. */
            if(memcmp(pDFname, aPiccDfName, sizeof(aPiccDfName)) == 0)
            {
                aFileId[0] = 0x00;
                aFileId[1U] = 0x00;
                aFileId[2U] = 0x00;
            }
            else
            {
                aFileId[0] = 0xFFU;
                aFileId[1U] = 0xFFU;
                aFileId[2U] = 0xFFU;
            }

            (void) memcpy(pDataParams->aAid, aFileId, sizeof(aFileId));
        }
    }
    else
    {
        /* Nothing to do here. */
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_IsoReadBinary(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid,
    uint32_t dwBytesToRead, uint8_t bExtendedLenApdu, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    /* Validate Parameters. */
    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_DEFAULT)
    {
        if(bSfid & PHAL_NTAGXDNA_SFID_ENABLED)
        {
#ifdef RDR_LIB_PARAM_CHECK
            /* Short file id is supplied */
            PHAL_NTAGXDNA_IS_VALID_FILE_NO(bSfid);
#endif
            /* Short File Identifier from 00 - 1F. */
            bP1 = bSfid;

            /* Offset from 0 - 255. */
            bP2 = bOffset;
        }
        else
        {
            /* Encode Offset from 0 - 32767 */
            bP1 = bSfid;
            bP2 = bOffset;
        }
    }
    else if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        /* Do Nothing. Code is handled internally. */
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    /* Validate APDU Format. */
    PHAL_NTAGXDNA_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    return phalNtagXDna_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_NTAGXDNA_EXCHANGE_LE_ONLY,
        wOption,
        bExtendedLenApdu,
        PHAL_NTAGXDNA_ISO7816_GENERIC_CLA,
        PHAL_NTAGXDNA_CMD_ISO7816_READ_BINARY,
        bP1,
        bP2,
        NULL,
        0,
        dwBytesToRead,
        ppResponse,
        pRspLen);
}

phStatus_t phalNtagXDna_Sw_IsoUpdateBinary(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu,
    uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;

    if(bSfid & PHAL_NTAGXDNA_SFID_ENABLED)
    {
#ifdef RDR_LIB_PARAM_CHECK
        /* Short file id is supplied */
        PHAL_NTAGXDNA_IS_VALID_FILE_NO(bSfid);
#endif
        /* Short File Identifier from 00 - 1F. */
        bP1 = bSfid;

        /* Offset from 0 - 255. */
        bP2 = bOffset;
    }
    else
    {
        /* Encode Offset from 0 - 32767 */
        bP1 = bSfid;
        bP2 = bOffset;
    }

    /* Validate APDU Format. */
    PHAL_NTAGXDNA_VALIDATE_APDU_FORMAT(bExtendedLenApdu);

    wStatus = phalNtagXDna_Sw_Int_Send7816Apdu(
        pDataParams,
        PHAL_NTAGXDNA_EXCHANGE_LC_ONLY,
        PH_EXCHANGE_DEFAULT,
        bExtendedLenApdu,
        PHAL_NTAGXDNA_ISO7816_GENERIC_CLA,
        PHAL_NTAGXDNA_CMD_ISO7816_UPDATE_BINARY,
        bP1,
        bP2,
        pData,
        wDataLen,
        0,
        NULL,
        NULL);

    return wStatus;
}





/* NTAG X DNA Utility functions -------------------------------------------------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_GetConfig(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PHAL_NTAGXDNA_ADDITIONAL_INFO:
            *pValue = pDataParams->wAdditionalInfo;
            break;

        case PHAL_NTAGXDNA_SHORT_LENGTH_APDU:
            *pValue = (uint16_t) pDataParams->bShortLenApdu;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_SetConfig(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PHAL_NTAGXDNA_ADDITIONAL_INFO:
            pDataParams->wAdditionalInfo = wValue;
            break;

        case PHAL_NTAGXDNA_SHORT_LENGTH_APDU:
            pDataParams->bShortLenApdu = (uint8_t) wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_ResetAuthentication(phalNtagXDna_Sw_DataParams_t * pDataParams)
{
    phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_CalculateMACSDM(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bSdmOption, uint16_t wKeyNo_SDMMac,
    uint16_t wKeyVer_SDMMac, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen,
    uint8_t ** ppSDMMAC, uint16_t * pSDMMACLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bSDMMACLen = 0;

    uint8_t     PH_MEMLOC_REM *pSDMMAC = NULL;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pSDMMAC = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 1U)];

    /* Generate and Load SDMFileRead Session MAC keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ComputeSessionKey_SDM(
        pDataParams,
        PHAL_NTAGXDNA_SESSION_MAC,
        bSdmOption,
        wKeyNo_SDMMac,
        wKeyVer_SDMMac,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Compute SDM Mac value. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        pInData,
        wInDataLen,
        pSDMMAC,
        &bSDMMACLen));

    /* Truncate the Calculated CMAC */
    phalNtagXDna_Sw_Int_TruncateMac(pSDMMAC);

    /* Copy MAC to response buffer. */
    *ppSDMMAC = pSDMMAC;
    *pSDMMACLen = 8U;

    if(pDataParams->bAuthState != PHAL_NTAGXDNA_NOT_AUTHENTICATED)
    {
        /* Load the SessionMAC key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pDataParams->aSesAuthMACKey,
            pDataParams->wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_VerifySDMSignature(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wPubKeyNo_SDMSig,
    uint16_t wPubKeyPos_SDMSig, uint8_t * pInData, uint16_t wInDataLen, uint8_t * pSignature, uint16_t wSigLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Remove any previously loaded ASymmetric Keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_InvalidateKey(pDataParams->pCryptoDataParamsASym));

    /* Load Transaction Signature Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        wPubKeyNo_SDMSig,
        wPubKeyPos_SDMSig));

    /* Sign Transaction MAC Input. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_DEFAULT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pInData,
        wInDataLen,
        pSignature,
        wSigLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_DecryptSDMENCFileData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bSdmOption,
    uint16_t wKeyNo_SDMEnc, uint16_t wKeyVer_SDMEnc, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr,
    uint8_t * pEncData, uint16_t wEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Generate and Load SDMFileRead Session MAC keys. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ComputeSessionKey_SDM(
        pDataParams,
        PHAL_NTAGXDNA_SESSION_ENC,
        bSdmOption,
        wKeyNo_SDMEnc,
        wKeyVer_SDMEnc,
        pUid,
        bUidLen,
        pSDMReadCtr));

    /* Load zero Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Initialization vector. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pSDMReadCtr[0];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pSDMReadCtr[1U];
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = pSDMReadCtr[2U];

    /* Get AES block size. */
    PHAL_NTAGXDNA_NEAREST_MULTIPLE(PHAL_NTAGXDNA_CMD_BUF_LEN, PHAL_NTAGXDNA_CMD_BUF_LEN);

    /* Now Encrypt the data Buffer to derive IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CBC,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        PHAL_NTAGXDNA_CMD_BUF));

    /* Load computed Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        PHAL_NTAGXDNA_CMD_BUF,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt SDM Encrypted File Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pEncData,
        wEncDataLen,
        PHAL_NTAGXDNA_PRS_BUF));

    /* Copy decrypted data to response parameter. */
    *ppPlainData = PHAL_NTAGXDNA_PRS_BUF;
    *pPlainDataLen = wEncDataLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_DecryptSDMPICCData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pEncData, uint16_t wEncDataLen, uint8_t ** ppPlainData, uint16_t * pPlainDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Set the pointer. */
    pKey = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_NTAGXDNA_VALIDATE_KEYTYPE(wKeyType);

    /* Load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey,
        wKeyType));

    /* Load Zero Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt SDM Encrypted File Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT),
        pEncData,
        wEncDataLen,
        PHAL_NTAGXDNA_PRS_BUF));

    /* Copy decrypted data to response parameter. */
    *ppPlainData = PHAL_NTAGXDNA_PRS_BUF;
    *pPlainDataLen = wEncDataLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_NTAGXDNA_SW */
