/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic P40 CmdLib Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_RefDefs.h>
#include <phhalHw.h>
#include <phKeyStore.h>

#ifdef NXPBUILD__PHAL_P40CMDPRIV_SW

#include <phalP40CmdPriv.h>
#include "phalP40CmdPriv_Sw.h"
#include "../phalP40Cmdpriv_Int.h"

/***************************************************************************************/
/* Private function prototype   */
static phStatus_t phhalP40CmdPriv_Sw_SendRawMessage(
    phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
    uint8_t * pMsg,
    uint8_t bMsgLen,
    uint8_t * pOptData,
    uint16_t bOptDataLen,
    uint8_t ** ppRxBuffer,
    uint16_t * pwRxBufferLen
    );

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_Init(
                                  phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
                                  uint16_t wDataParamSize,
                                  void * pHalDataParams,
                                  void * pKeyStoreDataParams)
{
    if (wDataParamSize != sizeof(phalP40CmdPriv_Sw_DataParams_t))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_P40CMDPRIV);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    pDataParams->wId                    = PH_COMP_AL_P40CMDPRIV | PHAL_P40CMDPRIV_SW_ID;
    pDataParams->pHal                   = pHalDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_GetVersionP(
    phalP40CmdPriv_Sw_DataParams_t * pDataParams,
    uint8_t * pVersionData
    )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal;
    uint16_t       PH_MEMLOC_REM wCntr;
    uint8_t *       PH_MEMLOC_REM pRxBuffer ;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bTxCmd = PHAL_P40CMDPRIV_GETVERSION_P;


    PH_ASSERT_NULL (pVersionData);

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        &bTxCmd,
        1,
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out and step over status code */
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    if (bRetVal == PHAL_P40CMDPRIV_INT_ERR_OK)
    {
        /* check buffer bounds  BEWARE of compiler padding in struct*/
        if (wRxBufferLen != PHAL_P40CMDPRIV_VERSIONINFO_SIZE)
        {
            return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
        }

        /* read out version response info*/
        for (wCntr = 0; wCntr < wRxBufferLen; wCntr++)
        {
            ((uint8_t*)pVersionData)[wCntr] = pRxBuffer[wCntr];
        }
    }

    return PH_ADD_COMPCODE( phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_WriteMemoryP(
    phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
    uint16_t wStartAddr,
    uint8_t bUseRandom,
    uint8_t bDataLength,
    uint8_t * pData)
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal  = PHAL_P40CMDPRIV_ERR_RESP_LENGTH;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bHeader[5];

    /* build header*/
    bHeader[0] = PHAL_P40CMDPRIV_WRITEMEMORY_P;
    bHeader[1] = (uint8_t) (wStartAddr & 0xff);
    bHeader[2] = (uint8_t) (wStartAddr >> 8);
    bHeader[3] = bUseRandom;
    bHeader[4] = bDataLength;

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        (uint8_t)5,
        pData,
        bDataLength == 0 ? (uint16_t)256 : (uint16_t)bDataLength,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    return PH_ADD_COMPCODE( phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_ReadMemoryP(
    phalP40CmdPriv_Sw_DataParams_t * pDataParams,
    uint16_t wStartAddr,
    uint8_t bDataLength,
    uint8_t * pData,
    uint16_t *  pReadLength
    )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal  = PHAL_P40CMDPRIV_ERR_RESP_LENGTH;
    uint16_t       PH_MEMLOC_REM wCntr;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bHeader[4];

    PH_ASSERT_NULL (pData);

    /* build header*/
    bHeader[0] = PHAL_P40CMDPRIV_READMEMORY_P;
    bHeader[1] = (uint8_t) (wStartAddr & 0xff);
    bHeader[2] = (uint8_t) (wStartAddr >> 8);
    bHeader[3] = bDataLength;

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        4,
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code */
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    if (bRetVal == PHAL_P40CMDPRIV_INT_ERR_OK)
    {
        /* set read length*/
        *pReadLength = wRxBufferLen;

        if ( (( bDataLength == 0 ) && (wRxBufferLen != 256)) ||
            (( bDataLength > 0 ) && (wRxBufferLen > bDataLength)) )
        {
            return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
        }

        /* read out memory block */
        for (wCntr = 0; wCntr < wRxBufferLen; wCntr++)
        {
            pData[wCntr] =  pRxBuffer[wCntr];
        }
    }

    return PH_ADD_COMPCODE(phalP40CmdPriv_Int_TranslateError(bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_CalculateChecksumP(
    phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
    uint16_t wStartAddr,
    uint16_t wLength,
    uint8_t bCksumType,
    uint16_t wDestAddress,
    uint8_t * pCrc
    )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal = 0;
    uint16_t       PH_MEMLOC_REM wCntr;
    uint8_t         PH_MEMLOC_REM bExpectedLength;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bHeader[8];
    uint8_t         PH_MEMLOC_REM bHeaderLength;

    PH_ASSERT_NULL (pCrc);

    switch (bCksumType)
    {
    case PHAL_P40CMDPRIV_CKSUM_CRC8:
        bExpectedLength = 1;
        break;
    case PHAL_P40CMDPRIV_CKSUM_CRC16:
        bExpectedLength = 2;
        break;
    case PHAL_P40CMDPRIV_CKSUM_CRC32:
        bExpectedLength = 4;
        break;
    default :
        /* return error ????? */
        bExpectedLength = 0;
        break;
    }

    /* build header*/
    bHeader[0] = PHAL_P40CMDPRIV_CALCULATECKSM_P;
    /* LSB format for words*/
    bHeader[1] = (uint8_t) (wStartAddr & 0xff);
    bHeader[2] = (uint8_t) (wStartAddr >> 8);
    bHeader[3] = (uint8_t) (wLength & 0xff);
    bHeader[4] = (uint8_t) (wLength >> 8);
    bHeader[5] = bCksumType;
    bHeaderLength = 6;

    if ( wDestAddress != 0)
    {
        /* LSB */
        bHeader[6] = (uint8_t) (wDestAddress & 0xff);
        bHeader[7] = (uint8_t) (wDestAddress >> 8);
        bHeaderLength += 2;
    }

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        bHeaderLength,
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    if (bRetVal != PHAL_P40CMDPRIV_INT_ERR_OK)
    {

        if (bExpectedLength != wRxBufferLen)
        {
            return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
        }

        /* read out memory block */
        for (wCntr = 0; wCntr < wRxBufferLen; wCntr++)
        {
            pCrc[wCntr] =  pRxBuffer[wCntr];
        }
    }
    return PH_ADD_COMPCODE(phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_ExecuteP (
                                       phalP40CmdPriv_Sw_DataParams_t * pDataParams,
                                       uint16_t wExeAddr,
                                       uint8_t bMagic,
                                       uint8_t bParamLength,
                                       uint8_t * pParamBuffer,
                                       uint16_t bRespSize,
                                       uint8_t * pRespBuffer,
                                       uint16_t * pRespLength  )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal = PHAL_P40CMDPRIV_INT_ERR_UNKNOWN;
    uint16_t       PH_MEMLOC_REM wCntr;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bHeader[4];
    uint8_t         PH_MEMLOC_REM bTmpLength;

    if (bRespSize > 0)
    {
        PH_ASSERT_NULL (pRespBuffer);
    }

    /* init length value  if one exists */
    if (pRespLength != NULL)
    {
        *pRespLength = 0;
    }

    /* build header*/
    bHeader[0] = PHAL_P40CMDPRIV_EXECUTE_P;
    /* LSB format for words*/
    bHeader[1] = (uint8_t) (wExeAddr & 0xff);
    bHeader[2] = (uint8_t) (wExeAddr >> 8);
    bHeader[3] = bMagic;

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        4,
        pParamBuffer,
        bParamLength,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    if (bRetVal == PHAL_P40CMDPRIV_INT_ERR_OK)
    {
        /*read length if there is one*/
        if (wRxBufferLen > 0)
        {
            bTmpLength = *pRxBuffer;
            pRxBuffer++;
            wRxBufferLen--;

            if (bTmpLength > 0)
            {
                /* large enough buffer and lengths match */
                if ( (bTmpLength > bRespSize) ||
                    (bTmpLength != wRxBufferLen) )
                {
                    return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
                }
                /* write return value length if one exists*/
                /* What to do when user supplies no length storage*/
                if (pRespLength != NULL)
                {
                    *pRespLength = bTmpLength;
                }

                /* read out memory block */
                for (wCntr = 0; wCntr < bTmpLength; wCntr++)
                {
                    pRespBuffer[wCntr] =  pRxBuffer[wCntr];
                }
            }
        }
    }

    return PH_ADD_COMPCODE( phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_AuthenticateP (
    phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
    uint8_t bKeyId,
    uint16_t wKeyNumber,
    uint16_t wKeyVersion)
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal = PHAL_P40CMDPRIV_INT_ERR_UNKNOWN;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    uint8_t         PH_MEMLOC_REM bMsg[PH_KEYSTORE_KEY_TYPE_AES128_SIZE +2];
    uint16_t       PH_MEMLOC_REM bKeystoreKeyType;

    /* Bail out if we haven't got a keystore */
    if (pDataParams->pKeyStoreDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);
    }

    /* prepend header*/
    bMsg[0] = PHAL_P40CMDPRIV_AUTHENTICATE_P;
    bMsg[1] = bKeyId;

    /* retrieve Key from keystore */
    PH_CHECK_SUCCESS_FCT(status, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNumber,
        wKeyVersion,
        (uint8_t)PH_KEYSTORE_KEY_TYPE_AES128_SIZE,
        &bMsg[2],
        &bKeystoreKeyType));

    /* check key type */
    if (bKeystoreKeyType != PH_KEYSTORE_KEY_TYPE_AES128)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bMsg,
        sizeof(bMsg),
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE( PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }

    return PH_ADD_COMPCODE( phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_FastTestP(
                                       phalP40CmdPriv_Sw_DataParams_t * pDataParams,
                                       uint8_t bTestCode
                                       )
{
    phStatus_t       PH_MEMLOC_REM status;
    uint8_t             PH_MEMLOC_REM bRetVal = PHAL_P40CMDPRIV_INT_ERR_UNKNOWN;
    uint8_t *           PH_MEMLOC_REM pRxBuffer;
    uint16_t           PH_MEMLOC_REM wRxBufferLen;
    uint8_t             PH_MEMLOC_REM bHeader[2];

    /* build command header and transmit*/
    bHeader[0] = PHAL_P40CMDPRIV_FASTTEST_P;
    bHeader[1] = bTestCode;

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        2,
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }
    return PH_ADD_COMPCODE(phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}


/***************************************************************************************/
phStatus_t phalP40CmdPriv_Sw_SwitchLifeCycleP (
    phalP40CmdPriv_Sw_DataParams_t * pDataParams,
    uint8_t bLifeCycle
    )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t         PH_MEMLOC_REM bRetVal = PHAL_P40CMDPRIV_INT_ERR_UNKNOWN;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bHeader[2];

    /* build command header and transmit*/
    bHeader[0] = PHAL_P40CMDPRIV_SWITCH_LIFE_CYCLE_P;      /* PHAL_P40CMDPRIV_FASTTEST_P; */
    bHeader[1] = bLifeCycle;

    PH_CHECK_SUCCESS_FCT(status,
        phhalP40CmdPriv_Sw_SendRawMessage(
        pDataParams,
        bHeader,
        2,
        NULL,
        0,
        &pRxBuffer,
        &wRxBufferLen));

    /* read out  and step over status code*/
    if (wRxBufferLen > 0)
    {
        bRetVal = *pRxBuffer;
        pRxBuffer++;
        wRxBufferLen--;
    }
    else
    {
        return PH_ADD_COMPCODE(PHAL_P40CMDPRIV_ERR_RESP_LENGTH, PH_COMP_AL_P40CMDPRIV);
    }
    return PH_ADD_COMPCODE(phalP40CmdPriv_Int_TranslateError( bRetVal), PH_COMP_AL_P40CMDPRIV);
}

/***************************************************************************************/
/* Private function  Low level transfer used by all command functions   */
static phStatus_t phhalP40CmdPriv_Sw_SendRawMessage(
    phalP40CmdPriv_Sw_DataParams_t *  pDataParams,
    uint8_t * pMsg,
    uint8_t bMsgLen,
    uint8_t * pOptData,
    uint16_t bOptDataLen,
    uint8_t ** ppRxBuffer,
    uint16_t * pwRxBufferLen
    )
{
    phStatus_t   PH_MEMLOC_REM statusTmp ;
    phStatus_t   PH_MEMLOC_REM status;
    uint16_t   PH_MEMLOC_REM wTxCrc;
    uint16_t   PH_MEMLOC_REM wRxCrc;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_TXCRC,
        &wTxCrc));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_RXCRC,
        &wRxCrc));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_TXCRC,
        PH_ON));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_RXCRC,
        PH_ON));

    status = phhalHw_Exchange(
        pDataParams->pHal,
        (uint16_t)((pOptData != NULL) ? PH_EXCHANGE_BUFFERED_BIT : PH_EXCHANGE_DEFAULT),
        pMsg, bMsgLen,
        ppRxBuffer, pwRxBufferLen);

    if ((status & PH_ERR_MASK)==PH_ERR_SUCCESS)
    {
        if (pOptData != NULL)
        {
            status = phhalHw_Exchange(
                pDataParams->pHal,
                PH_EXCHANGE_LEAVE_BUFFER_BIT,
                pOptData,
                bOptDataLen,
                ppRxBuffer, pwRxBufferLen);
        }
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_TXCRC,
        wTxCrc));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pDataParams->pHal,
        PHHAL_HW_CONFIG_RXCRC,
        wRxCrc));

    return status;
}

#endif  /* NXPBUILD__PHAL_P40CMDPRIV_SW */
