/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic P40 CmdLib Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_RefDefs.h>
#include <phhalHw.h>

#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phKeyStore.h>


#ifdef NXPBUILD__PHAL_P40CMDPUB_SW

#include <phalP40CmdPub.h>
#include "phalP40CmdPub_Sw.h"
#include "phalP40CmdPub_Sw_Int.h"
#include "../phalP40CmdPub_Int.h"

#include <phpalMifare.h>

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_Init(
                  phalP40CmdPub_Sw_DataParams_t * pDataParams,
                  uint16_t wDataParamSize,
                  void *    pPalMifareDataParams,
                  void *    pKeyStoreDataParams,
                  void *    pHalDataParams,
                  void *    pCryptoDataParamsEnc,
                  void *    pCryptoRngDataParams
                  )
{
    if (wDataParamSize != sizeof(phalP40CmdPub_Sw_DataParams_t))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_P40CMDPUB);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pPalMifareDataParams);
    PH_ASSERT_NULL (pKeyStoreDataParams);
    PH_ASSERT_NULL (pHalDataParams);
    PH_ASSERT_NULL (pCryptoDataParamsEnc);
    PH_ASSERT_NULL (pCryptoRngDataParams);

    pDataParams->wId  = PH_COMP_AL_P40CMDPUB | PHAL_P40CMDPUB_SW_ID;
    pDataParams->pPalMifare = pPalMifareDataParams;
    pDataParams->pKeyStore = pKeyStoreDataParams;
    pDataParams->pHal = pHalDataParams;
    pDataParams->pCryptoEnc = pCryptoDataParamsEnc;
    pDataParams->pCryptoRng = pCryptoRngDataParams;


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_GetVersion (
               phalP40CmdPub_Sw_DataParams_t * pDataParams,
               uint8_t  bLifeCycleId,
               uint8_t* pVersionData
               )
{
    phStatus_t   PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    PH_ASSERT_NULL (pVersionData);

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    bLifeCycleId,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_GETVERSION_INS,
                                    0,	0,	0,	PHAL_P40CMDPUB_VERSIONINFO_SIZE,
                                    NULL, 0,
                                    &pRxBuffer, &wRxBufferLen );

    /* standard error check*/
    status = phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status);

    if (status == PH_ERR_SUCCESS)
    {
        /*Check length and copy info to struct*/
        if (wRxBufferLen ==  PHAL_P40CMDPUB_VERSIONINFO_SIZE)
        {
            memcpy(pVersionData, pRxBuffer, PHAL_P40CMDPUB_VERSIONINFO_SIZE); /* PRQA S 3200 */
        }
        else
        {
            status = PHAL_P40CMDPUB_ERR_RESP_LENGTH;
        }
    }
    return PH_ADD_COMPCODE(status,PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_DownloadCode(
                phalP40CmdPub_Sw_DataParams_t *  pDataParams,
                uint16_t wStartAddr,
                uint8_t* pData,
                uint16_t wDataLength
                )
{
    return phalP40CmdPub_Sw_Int_Download (
                                                        pDataParams,
                                                        PHAL_P40CMDPUB_DOWNLOADCODE_INS,
                                                        wStartAddr, pData, wDataLength);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_DownloadConst(
                phalP40CmdPub_Sw_DataParams_t *  pDataParams,
                uint16_t wStartAddr,
                uint8_t* pData,
                uint16_t wDataLength
               )
{
    return phalP40CmdPub_Sw_Int_Download (
                                                        pDataParams,
                                                        PHAL_P40CMDPUB_DOWNLOADCONST_INS,
                                                        wStartAddr, pData, wDataLength);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_DownloadEE(
                phalP40CmdPub_Sw_DataParams_t *  pDataParams,
                uint16_t wStartAddr,
                uint8_t* pData,
                uint16_t wDataLength
                )
{
    return phalP40CmdPub_Sw_Int_Download (
                                                        pDataParams,
                                                        PHAL_P40CMDPUB_DOWNLOADEE_INS,
                                                        wStartAddr, pData, wDataLength);
}


/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_DownloadPatch(
                phalP40CmdPub_Sw_DataParams_t * pDataParams,
                uint8_t bPatchNr,
                uint16_t wPatchAddress,
                uint8_t * pPatch,
                uint16_t wPatchLength
                )
{
    phStatus_t   PH_MEMLOC_REM status = PH_ERR_SUCCESS;

    uint8_t *   PH_MEMLOC_REM pRxBuffer;
    uint16_t   PH_MEMLOC_REM wRxBufferLen;
    uint8_t     PH_MEMLOC_REM bHeader[2]; /* needed to put address separately*/

    bHeader[0] = (uint8_t)(wPatchAddress >> 8);
    bHeader[1] = (uint8_t)(wPatchAddress);

    PH_CHECK_SUCCESS_FCT(status, phalP40CmdPub_Sw_Int_Exchange(
                                                                                pDataParams,
                                                                                PH_EXCHANGE_BUFFER_FIRST,
                                                                                PH_OFF,
                                                                                PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                                                                PHAL_P40CMDPUB_DOWNLOADPATCH_INS,
                                                                                bPatchNr, 0,
                                                                                wPatchLength + 2, 0,
                                                                                bHeader, sizeof(bHeader),
                                                                                NULL, NULL));

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_BUFFER_LAST,
                                    PH_OFF,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,	/* CLA. */
                                    PHAL_P40CMDPUB_DOWNLOADPATCH_INS,  /* INS */
                                    bPatchNr, 0,            /* P1. *//* P2 */
                                    wPatchLength + 2, 0,   /* Lc  Le */
                                    pPatch, wPatchLength, /**< additional data. *//**< additional data length. */
                                    &pRxBuffer, &wRxBufferLen);

    /* standard error check*/
    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_DownloadFlash(
                phalP40CmdPub_Sw_DataParams_t *  pDataParams,
                uint16_t wStartAddr,
                uint8_t* pData,
                uint16_t wDataLength
                )
{
    return phalP40CmdPub_Sw_Int_Download (
                                                        pDataParams,
                                                        PHAL_P40CMDPUB_DOWNLOADFLASH_INS,
                                                        wStartAddr, pData, wDataLength);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_LockApplication (
                phalP40CmdPub_Sw_DataParams_t *	pDataParams
                )
{
    phStatus_t   PH_MEMLOC_REM status       = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

 status =  phalP40CmdPub_Sw_Int_Exchange(
                    pDataParams,
                    PH_EXCHANGE_DEFAULT,
                    PH_OFF,
                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                    PHAL_P40CMDPUB_LOCKAPPLICATION_INS,
                    0, 0, 0, 0,
                    NULL, 0,
                    &pRxBuffer, &wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_FormatApplication (
                phalP40CmdPub_Sw_DataParams_t *	pDataParams,
                uint8_t	bTargetApp
                )
{
    phStatus_t   PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t *      PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                    pDataParams,
                    PH_EXCHANGE_DEFAULT,
                    PH_OFF,
                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                    PHAL_P40CMDPUB_FORMATAPPLICATION_INS,
                    bTargetApp, 0, 0, 0,
                    NULL, 0,
                    &pRxBuffer, &wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_SwitchLifeCycle (
            phalP40CmdPub_Sw_DataParams_t * pDataParams,
           uint8_t  bLifeCycleId,
           uint8_t bLcType
            )
{
    phStatus_t   PH_MEMLOC_REM status        = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    bLifeCycleId,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_SWITCHLIFECYCLE_INS,
                                    bLcType, 0, 0, 0,
                                    NULL, 0,
                                    &pRxBuffer,&wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_ExecuteTrapdoor (
            phalP40CmdPub_Sw_DataParams_t *	pDataParams,
            uint8_t  bLifeCycleId
            )
{
    phStatus_t   PH_MEMLOC_REM status       = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                                    pDataParams,
                                                    PH_EXCHANGE_DEFAULT,
                                                    bLifeCycleId,
                                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                                    PHAL_P40CMDPUB_EXECUTETRAPDOOR_INS,
                                                    0, 0, 0, 0,
                                                    NULL, 0,
                                                    &pRxBuffer, &wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_SetConfiguration(
                    phalP40CmdPub_Sw_DataParams_t * pDataParams,
                    uint8_t bDisableCmdTestFuse,
                    uint8_t bDisableTPTestFuse,
                    uint8_t bUseExtendedConfig,
                    uint16_t wApp1CodeSize,
                    uint16_t wApp1ConstSize,
                    uint16_t wApp2CodeSize,
                    uint16_t wApp2ConstSize,
                    uint16_t wCommBufferSize,
                    uint8_t bNumKeys,
                    uint8_t bDefaultApp
           )
{
    phStatus_t   PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM bData[PHAL_P40CMDPUB_CONFIGURATION_SIZE];
    uint16_t       PH_MEMLOC_REM wDataSize = 0;

    if(bUseExtendedConfig)
    {
        bData[0] = 12;
        bData[1] = (uint8_t)(wApp1CodeSize>>8);
        bData[2] = (uint8_t)(wApp1CodeSize);
        bData[3] = (uint8_t)(wApp1ConstSize>>8);
        bData[4] = (uint8_t)(wApp1ConstSize);
        bData[5] = (uint8_t)(wApp2CodeSize>>8);
        bData[6] = (uint8_t)(wApp2CodeSize);
        bData[7] = (uint8_t)(wApp2ConstSize>>8);
        bData[8] = (uint8_t)(wApp2ConstSize);
        bData[9] = (uint8_t)(wCommBufferSize>>8);
        bData[10] = (uint8_t)(wCommBufferSize);
        bData[11] = bNumKeys;
        bData[12] = bDefaultApp;

        wDataSize = PHAL_P40CMDPUB_CONFIGURATION_SIZE;
    }

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    PH_OFF,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_SETCONFIGURATION_INS,
                                    bDisableCmdTestFuse,bDisableTPTestFuse,
                                    wDataSize, 0,
                                    bData, wDataSize,
                                    &pRxBuffer, &wRxBufferLen );

    /* standard error check*/
    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_GetConfiguration(
                   phalP40CmdPub_Sw_DataParams_t * pDataParams,
                   uint8_t * pDisableCmdTestFuse,
                   uint8_t * pDisableTPTestFuse,
                   uint16_t * pApp1CodeSize,
                   uint16_t * pApp1ConstSize,
                   uint16_t * pApp2CodeSize,
                   uint16_t * pApp2ConstSize,
                   uint16_t * pCommBufferSize,
                   uint8_t * pNumKeys,
                   uint8_t * pDefaultApp
                   )
{
    phStatus_t   PH_MEMLOC_REM status       = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    PH_OFF,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_GETCONFIGURATION_INS,
                                    0, 0,
                                    0, PHAL_P40CMDPUB_CONFIGURATION_SIZE + 2,
                                    NULL, 0,
                                    &pRxBuffer, &wRxBufferLen );
    /* quick error check*/
    status = phalP40CmdPub_Sw_Int_CheckReturnResult ( pDataParams, status);

    if (status == PH_ERR_SUCCESS)
    {
        /*Check length and copy info to struct*/
        if (wRxBufferLen ==  PHAL_P40CMDPUB_CONFIGURATION_SIZE)
        {
            if (pDisableCmdTestFuse)
                *pDisableCmdTestFuse = pRxBuffer[0];
            if (pDisableTPTestFuse)
                  *pDisableTPTestFuse = pRxBuffer[1];
            if(pApp1CodeSize)
                *pApp1CodeSize = (uint16_t)((((uint16_t)pRxBuffer[2]) << 8) | (pRxBuffer[3]));
            if(pApp1ConstSize)
                *pApp1ConstSize = (uint16_t)((((uint16_t)pRxBuffer[4]) << 8) | (pRxBuffer[5]));
            if(pApp2CodeSize)
                *pApp2CodeSize = (uint16_t)((((uint16_t)pRxBuffer[6]) << 8) | (pRxBuffer[7]));
            if(pApp2ConstSize)
                *pApp2ConstSize = (uint16_t)((((uint16_t)pRxBuffer[8]) << 8) | (pRxBuffer[9]));
            if(pCommBufferSize)
                *pCommBufferSize = (uint16_t)((((uint16_t)pRxBuffer[10]) << 8) | (pRxBuffer[11]));
            if(pNumKeys)
                *pNumKeys = pRxBuffer[12];
            if(pDefaultApp)
                *pDefaultApp = pRxBuffer[13];
        }
        else
        {
            status = PHAL_P40CMDPUB_ERR_RESP_LENGTH;
        }
    }
    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_FastTest(
                    phalP40CmdPub_Sw_DataParams_t * pDataParams,
                    uint8_t bLifeCycleId,
                    uint8_t bFastTestType
                    )
{
    phStatus_t   PH_MEMLOC_REM status        = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    bLifeCycleId,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_FASTTEST_INS,
                                    bFastTestType, 0, 0, 0,
                                    NULL, 0,
                                    &pRxBuffer,&wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_VerifyDownload (
                                    phalP40CmdPub_Sw_DataParams_t *	pDataParams,
                                    uint16_t wSHA5
                                    )
{
    phStatus_t   PH_MEMLOC_REM status        = PH_ERR_SUCCESS;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    PH_EXCHANGE_DEFAULT,
                                    PH_OFF,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_VERIFYDOWNLOAD_INS,
                                    (uint8_t) (wSHA5>>8), (uint8_t)wSHA5, 0, 0,
                                    NULL, 0,
                                    &pRxBuffer, &wRxBufferLen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_GetChallenge(
                                    phalP40CmdPub_Sw_DataParams_t * pDataParams,
                                    uint8_t  bLifeCycleId,
                                    uint8_t bRequestLen,
                                    uint8_t* pChallenge
                                    )
{
    phStatus_t   PH_MEMLOC_REM status;


    uint16_t    PH_MEMLOC_REM wRxlen = 0;
    uint8_t     PH_MEMLOC_REM *pRecv;


    /*  check Lifecycle range */
    if( bLifeCycleId > PHAL_P40CMDPUB_LIFECYCLE_RELEASE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_P40CMDPUB);
    }

    /* Reset authentication status */
    phalP40CmdPub_Sw_Int_ResetAuthStatus(pDataParams);

    status =  phalP40CmdPub_Sw_Int_Exchange(
                                    pDataParams,
                                    bLifeCycleId,
                                    PH_EXCHANGE_DEFAULT,
                                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                                    PHAL_P40CMDPUB_GETCHALLENGE_INS,
                                    0, 0, 0, bRequestLen,
                                    NULL, 0,
                                    &pRecv,&wRxlen );

    status =phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status);

    if ( wRxlen == bRequestLen)
    {
        memcpy(pChallenge, pRecv, wRxlen); /* PRQA S 3200 */
    }
    else
    {
        status = PHAL_P40CMDPUB_ERR_RESP_LENGTH;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_ExternalAuthenticate(
                                        phalP40CmdPub_Sw_DataParams_t * pDataParams,
                                        uint8_t  bLifeCycleId,
                                        uint8_t bCryptoMethod,
                                        uint8_t bKeyId,
                                        uint8_t* pEdata,
                                        uint8_t bEdataLen
                                         )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint16_t    PH_MEMLOC_REM wRxlen = 0;
    uint8_t     PH_MEMLOC_REM *pRecv;
/*
    uint8_t     PH_MEMLOC_REM bIvLen;
    uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_3K3DES_KEY_SIZE];
    uint16_t    PH_MEMLOC_REM wKeyType;
*/

/*  check Lifecycle range */
    if( bLifeCycleId > PHAL_P40CMDPUB_LIFECYCLE_RELEASE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_P40CMDPUB);
    }

    /* First get the key from key store */
    /*
    PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
        pDataParams->pKeyStore,
        wKeyNumber,
        wKeyVersion,
        sizeof(bKey),
        bKey,
        &wKeyType
        ));

        bIvLen = PH_CRYPTOSYM_AES_BLOCK_SIZE;
*/
    /* Load Key */
        /*
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoEnc,
        bKey,
        wKeyType
        ));
        */
    /* Load Iv */
    /*
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
        pDataParams->pCryptoEnc,
        pDataParams->bIv,
        bIvLen));
*/
    /* Encrypt RPCD1 + RPICC1 */
    /*
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
        pDataParams->pCryptoEnc,
        bCryptoMethod | PH_EXCHANGE_BUFFER_CONT,
        pEdata,
        bEdataLen,
        pEdata
        ));
*/
    /* Store IV */
/*    memcpy(pDataParams->bIv, pEdata, bIvLen);  PRQA S 3200 */

    /* Send the APDU */
    status = phalP40CmdPub_Sw_Int_Exchange(
                    pDataParams,
                    bLifeCycleId,
                    PH_EXCHANGE_DEFAULT,
                    PHAL_P40CMDPUB_INT_CLA_DEFAULT,
                    PHAL_P40CMDPUB_EXTAUTHENTICATE_INS,
                    bCryptoMethod, bKeyId, bEdataLen, 0,
                    pEdata, bEdataLen,
                    &pRecv, &wRxlen );

    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_InternalAuthenticate(
                     phalP40CmdPub_Sw_DataParams_t * pDataParams,
                     uint8_t  bLifeCycleId,
                     uint8_t bCryptoMethod,
                     uint8_t bKeyId,

                     uint8_t* pEdata,
                     uint8_t bEdataLen
                     )
{
    phStatus_t   PH_MEMLOC_REM status        = PH_ERR_SUCCESS;
    if(bLifeCycleId || bCryptoMethod || bKeyId || pEdata || bEdataLen);
    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}


/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_MaintainanceReqa(
                                        phalP40CmdPub_Sw_DataParams_t * pDataParams
                                        )
{
    phStatus_t   PH_MEMLOC_REM status;
    uint8_t *       PH_MEMLOC_REM pRxBuffer;
    uint16_t       PH_MEMLOC_REM wRxBufferLen;
    uint8_t         PH_MEMLOC_REM cmd[1];

    cmd[0] = PHAL_P40CMDPUB_MAINTAINANCE_INS;

    status = phpalMifare_ExchangeL3(
                                                                pDataParams->pPalMifare,
                                                                PH_EXCHANGE_DEFAULT,
                                                                cmd,  1,
                                                                &pRxBuffer, &wRxBufferLen);

    return PH_ADD_COMPCODE(status, PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************/
phStatus_t phalP40CmdPub_Sw_Authenticate (
            phalP40CmdPub_Sw_DataParams_t * pDataParams,
            uint8_t bLifeCycleId,
            uint8_t bKeyId,
            uint16_t  wKeyNumber,
            uint16_t wKeyVersion
            )
{
    phStatus_t   PH_MEMLOC_REM statusTmp;

    uint16_t    PH_MEMLOC_REM wKeyType;
    uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_3K3DES_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bIvLen;


    uint8_t     PH_MEMLOC_REM bRpcd1[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bRpicc1[PH_CRYPTOSYM_AES128_KEY_SIZE];

    uint8_t     PH_MEMLOC_REM bWorkBuffer[PH_CRYPTOSYM_AES128_KEY_SIZE * 2];
    uint8_t     PH_MEMLOC_REM bRndLen;
    uint8_t     PH_MEMLOC_REM bCryptoMethod = PH_CRYPTOSYM_CIPHER_MODE_CBC;

    /*  check Lifecycle range */
    if( bLifeCycleId > PHAL_P40CMDPUB_LIFECYCLE_RELEASE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_P40CMDPUB);
    }


    if (bKeyId > PHAL_P40CMDPUB_AUTHKEY_ORIGINALITY)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_P40CMDPUB);
    }

    /* First get the key from key store */
    PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
        pDataParams->pKeyStore,
        wKeyNumber,
        wKeyVersion,
        sizeof(bKey),
        bKey,
        &wKeyType
        ));

    switch (wKeyType)
    {
    case PH_KEYSTORE_KEY_TYPE_2K3DES:
    case PH_KEYSTORE_KEY_TYPE_DES:
        bRndLen = PH_CRYPTOSYM_2K3DES_KEY_SIZE;
        bIvLen = PH_CRYPTOSYM_DES_BLOCK_SIZE;
        break;

    case PH_KEYSTORE_KEY_TYPE_3K3DES:
        bRndLen = PH_CRYPTOSYM_2K3DES_KEY_SIZE;
        bIvLen = PH_CRYPTOSYM_DES_BLOCK_SIZE;
        break;

    case PH_KEYSTORE_KEY_TYPE_AES128:
        bRndLen = PH_CRYPTOSYM_AES128_KEY_SIZE;
        bIvLen = PH_CRYPTOSYM_AES_BLOCK_SIZE;
        break;

    default:
        /* Invalid key type */
        return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_P40CMDPUB);
    }

	PH_CHECK_SUCCESS_FCT( statusTmp, phalP40CmdPub_Sw_GetChallenge(
		pDataParams,
		bLifeCycleId,
		bRndLen,
		bRpicc1));


    /* Generate PCD1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoRng_Seed(pDataParams->pCryptoRng, bRpicc1, bRndLen));
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoRng_Rnd(pDataParams->pCryptoRng, bRndLen, bRpcd1));

    /* move Rpcd and Rpicc into work buffer */
    memcpy(&bWorkBuffer[0], bRpcd1, bRndLen); /* PRQA S 3200 */
    memcpy(&bWorkBuffer[bRndLen], bRpicc1, bRndLen); /* PRQA S 3200 */

    /* Load Key */
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoEnc,
        bKey,
        wKeyType
        ));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
        pDataParams->pCryptoEnc,
        pDataParams->bIv,
        bIvLen));

    /* Encrypt RPCD1 + RPICC1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
        pDataParams->pCryptoEnc,
        bCryptoMethod | PH_EXCHANGE_BUFFER_CONT,
        bWorkBuffer,
        bRndLen * 2,
        bWorkBuffer
        ));

    /* Store IV */
    memcpy(pDataParams->bIv, bWorkBuffer, bIvLen); /* PRQA S 3200 */

	PH_CHECK_SUCCESS_FCT( statusTmp, phalP40CmdPub_Sw_ExternalAuthenticate(
		pDataParams,
		bLifeCycleId,
		bCryptoMethod,
		bKeyId,
		bWorkBuffer,
		bRndLen * 2));

    /* Generate PCD2
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoRng_Seed(pDataParams->pCryptoRng, bRpcd1, bRndLen));
    PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoRng_Rnd(pDataParams->pCryptoRng, bRndLen, bRpcd2));


    memcpy(&bWorkBuffer[0], bRpcd2, bRndLen);
    bIndex = bIndex + bRndLen;
 */
/*
    PH_CHECK_SUCCESS_FCT(statusTmp, phalP40CmdPub_Sw_InternalAuthenticate(
        pDataParams,
        bWorkBuffer,
        bIndex,
        bWorkBuffer,
        &bIndex
        ));
*/
    /* Verify bRpcd2. Store bRpicc2. Generate session key */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_P40CMDPUB);
}

#endif  /* NXPBUILD__PHAL_P40CMDPUB_SW */
