/*
 * Copyright 2013, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal P40 CmdPub_Sw Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phKeyStore.h>
#include <phCryptoSym.h>

#ifdef NXPBUILD__PHAL_P40CMDPUB_SW

#include "phalP40CmdPub_Sw_Int.h"
#include "../phalP40CmdPub_Int.h"
#include "phalP40CmdPub.h"

#include <phhalHw.h>
#include <phpalMifare.h>

/***************************************************************************************************************************/
void phalP40CmdPub_Sw_Int_Reset_Current_Cmd (
    phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd
    )
{
    pCurrentCmd->wLc = 0;
    pCurrentCmd->wLe = 0;
    pCurrentCmd->bSw1 = 0;
    pCurrentCmd->bSw2 = 0;
    pCurrentCmd->bFlags = PHPALCMDPUB_APDUFLAG_DEFAULT;

}

/***************************************************************************************************************************/
uint8_t phalP40CmdPub_Sw_Int_EncodeLc (
                                       phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,
                                       uint8_t * pDest
                                       )
{
    uint8_t PH_MEMLOC_REM retval = 0;

    PH_ASSERT_NULL (pCurrentCmd);

    /* set value according to flags*/
    if (pCurrentCmd->wLc > 0)
    {
        /* write short. The size < 256 has already been checked*/
        if ( (pCurrentCmd->bFlags &	PHPALCMDPUB_APDUFLAG_USE_EXTENDED) == 0)
        {
            *pDest = (uint8_t)pCurrentCmd->wLc;
            retval = 1;
        }
        else   /* write as extended MSB first*/
        {
            *(pDest++)= 0x00;
            *(pDest++) = (uint8_t)(pCurrentCmd->wLc >> 8);
            *pDest = (uint8_t) pCurrentCmd->wLc;
            retval = 3;
        }
        pCurrentCmd->bFlags |= PHPALCMDPUB_APDUFLAG_LC_PRESENT;
    }
    else  /* Lc will be not included */
    {
        pCurrentCmd->bFlags &= (uint8_t)~(uint8_t)PHPALCMDPUB_APDUFLAG_LC_PRESENT;
    }

    return retval;
}

/***************************************************************************************************************************/
uint8_t phalP40CmdPub_Sw_Int_EncodeLe (
                                       phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,
                                       uint8_t pData[3]
)
{
    uint8_t PH_MEMLOC_REM idxPtr = 0;
    uint8_t PH_MEMLOC_REM retval = 0;

    PH_ASSERT_NULL (pCurrentCmd);

    if  ((pCurrentCmd->wLe == 0) &&
        ( (pCurrentCmd->bFlags & PHPALCMDPUB_APDUFLAG_MAX_LE)== 0 ))
    {  /* Le is absent*/
        return retval;
    }
    /* write in extended length*/
    if  (( pCurrentCmd->bFlags & PHPALCMDPUB_APDUFLAG_USE_EXTENDED) == PHPALCMDPUB_APDUFLAG_USE_EXTENDED)
    {
        if  (  (pCurrentCmd->bFlags & PHPALCMDPUB_APDUFLAG_LC_PRESENT) == 0)
        {
            /* prepend a zero only if the Lc field is absent.  */
            pData[idxPtr++] = 0x00;
            retval ++;
        }
        /* Now write in the Le as 2 bytes (16 bit truncated) MSB first  */
        pData[idxPtr++] = (uint8_t) (pCurrentCmd->wLe >> 8);
        pData[idxPtr++] = (uint8_t) (pCurrentCmd->wLe & 0xFF);
        retval += 2;
    }
    else
    { /* short */
        pData[0] = (uint8_t) pCurrentCmd->wLe; /* value 256 will truncate to a zero byte*/
        retval++;
    }
    return retval;
}

/***************************************************************************************************************************/
phStatus_t phalP40CmdPub_Sw_Int_CheckForExtendedLength (
    phalP40CmdPub_Sw_CurrentApdu_t * pCurrentCmd,
    uint16_t wOptions,
    uint16_t wLc,
    uint16_t wLe
    )
{
    PH_ASSERT_NULL (pCurrentCmd);

    if  (  (wLc > 255) ||
           (wLe > 256) ||
            (( wOptions & PHALP40CMDPUB_I7816P4_EXCHANGE_LE_64K) == PHALP40CMDPUB_I7816P4_EXCHANGE_LE_64K) )
    {
        pCurrentCmd->bFlags |= PHPALCMDPUB_APDUFLAG_USE_EXTENDED;
    }

    /* copy Lc as is*/
    pCurrentCmd->wLc = wLc;

    if  ( (wLe == 0) &&
          ((wOptions & PHALP40CMDPUB_I7816P4_EXCHANGE_LE_MAX)) ==  PHALP40CMDPUB_I7816P4_EXCHANGE_LE_MAX)
    {
         pCurrentCmd->bFlags |= PHPALCMDPUB_APDUFLAG_MAX_LE;
    }

    pCurrentCmd->wLe = wLe;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************************************************/
phStatus_t phalP40CmdPub_Sw_Int_Exchange(
    phalP40CmdPub_Sw_DataParams_t * pDataParams,
    uint16_t   wOption,
    uint8_t      bUseL3,
    uint8_t     bCla,
    uint8_t     bIns,
    uint8_t     bP1,
    uint8_t     bP2,
    uint16_t   wLc,
    uint16_t   wLe,
    uint8_t *   pTxBuffer,
    uint16_t   wTxLength,
    uint8_t ** ppRxBuffer,
    uint16_t * pRxLength
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;

    /* header buffer must be large enought to hold max 3 Lc bytes*/
    /* also reuse to write Le bytes*/
    uint8_t     PH_MEMLOC_REM bHdrBuffer[7];
    uint16_t    PH_MEMLOC_REM wHdrBufferLen;

    if (wTxLength > 0)
    {
        PH_ASSERT_NULL (pTxBuffer);
    }
    /* push in any command data we have*/
    if ( wOption & PH_EXCHANGE_RXCHAINING)
    {
        PH_CHECK_SUCCESS_FCT( statusTmp,  phalP40CmdPub_Sw_Int_ExchangeRaw(
            pDataParams,
            wOption,
            bUseL3,
            NULL, NULL,
            NULL, NULL));
    }

    /* first pass so initialize current command info, do checks and send header*/
    if ( ! (wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) )
    {
        phalP40CmdPub_Sw_Int_Reset_Current_Cmd (&pDataParams->sCurrentApdu);

        phalP40CmdPub_Sw_Int_CheckForExtendedLength ( &pDataParams->sCurrentApdu,
            wOption, wLc, wLe);

        bHdrBuffer[0] = bCla;
        bHdrBuffer[1] = bIns;
        bHdrBuffer[2] = bP1;
        bHdrBuffer[3] = bP2;
        wHdrBufferLen = 4;

        wHdrBufferLen = phalP40CmdPub_Sw_Int_EncodeLc( &pDataParams->sCurrentApdu,
            &bHdrBuffer[wHdrBufferLen] ) + wHdrBufferLen;

        /* Perform header preload */
        PH_CHECK_SUCCESS_FCT( statusTmp,  phalP40CmdPub_Sw_Int_ExchangeRaw(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            bUseL3,
            bHdrBuffer, wHdrBufferLen,
            NULL, NULL));
    }

    /* push in any command data we have*/
    if ( wTxLength > 0)
    {
        PH_CHECK_SUCCESS_FCT( statusTmp,  phalP40CmdPub_Sw_Int_ExchangeRaw(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            bUseL3,
            pTxBuffer, wTxLength,
            NULL, NULL));
    }
    /* time to transmit, so append Le if needed and go */
    if ( ! (wOption & PH_EXCHANGE_BUFFERED_BIT) )
    {
        wHdrBufferLen = phalP40CmdPub_Sw_Int_EncodeLe( &pDataParams->sCurrentApdu,
            bHdrBuffer );

        /* Perform final exchange */
        statusTmp =  phalP40CmdPub_Sw_Int_ExchangeRaw(
            pDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            bUseL3,
            bHdrBuffer, wHdrBufferLen,
            ppRxBuffer, pRxLength );

        /* If rx chaining, we do not have the status bytes yet
        just return what we got to do intermediate processing.
        User must call command again with rx_chaining set*/
        if (statusTmp != PH_ERR_SUCCESS_CHAINING)
        {
            /* should receive at least SW1 and SW2 */
            if (*pRxLength < 2)
            {
                return  PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_P40CMDPUB);
            }

            /* copy SW1 and SW2 and remove from return stream*/
            pDataParams->sCurrentApdu.bSw1 = (*ppRxBuffer)[*pRxLength -2];
            pDataParams->sCurrentApdu.bSw2 = (*ppRxBuffer)[*pRxLength -1];
            *pRxLength -=2;
        }

        return PH_ADD_COMPCODE(statusTmp, PH_COMP_AL_P40CMDPUB);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_P40CMDPUB);
}
/***************************************************************************************************************************/
phStatus_t phalP40CmdPub_Sw_Int_ExchangeRaw(
    phalP40CmdPub_Sw_DataParams_t * pDataParams,
    uint16_t wOption,
    uint8_t   bUseL3,
    uint8_t * pTxBuffer,
    uint16_t wTxLength,
    uint8_t ** ppRxBuffer,
    uint16_t * pRxLength
    )
{
    phStatus_t	PH_MEMLOC_REM statusTmp;

    if (bUseL3 == PH_OFF)
    {
        statusTmp = phpalMifare_ExchangeL4(
            pDataParams->pPalMifare,
            wOption,
            pTxBuffer,  wTxLength,
            ppRxBuffer, pRxLength);
    }
    else
    {
        statusTmp =  phpalMifare_ExchangeL3(
            pDataParams->pPalMifare,
            wOption,
            pTxBuffer, wTxLength,
            ppRxBuffer, pRxLength);
    }

    return  PH_ADD_COMPCODE(statusTmp, PH_COMP_AL_P40CMDPUB);

}

/***************************************************************/
phStatus_t phalP40CmdPub_Sw_Int_CheckReturnResult (
    phalP40CmdPub_Sw_DataParams_t *  pDataParams,
    phStatus_t   status)
{
    uint16_t  PH_MEMLOC_REM wRetVal  = PHAL_P40CMDPUB_ERR_UNKNOWN;
    /* quick error check*/
    if (status == PH_ERR_SUCCESS)
    {
        /* get raw SW1 and SW2 info*/
        wRetVal = (uint16_t)(((uint16_t)phalP40CmdPub_Sw_Int_LastSW1Result(pDataParams) << 8) | phalP40CmdPub_Sw_Int_LastSW2Result(pDataParams));

        status = phalP40CmdPub_Int_TranslateError( wRetVal);
    }

    return status;
}


/***************************************************************************************************************************/
uint8_t phalP40CmdPub_Sw_Int_LastSW1Result( phalP40CmdPub_Sw_DataParams_t * pDataParams )
{
    return pDataParams->sCurrentApdu.bSw1;
}

/***************************************************************************************************************************/
uint8_t phalP40CmdPub_Sw_Int_LastSW2Result( phalP40CmdPub_Sw_DataParams_t * pDataParams )
{
    return pDataParams->sCurrentApdu.bSw2;
}


/*********************************************
* Internal download function implementation
**********************************************/
phStatus_t phalP40CmdPub_Sw_Int_Download (
    phalP40CmdPub_Sw_DataParams_t *  pDataParams,
    uint8_t bCmd,
    uint16_t wStartAddr,
    uint8_t* pData,
    uint16_t wDataLength
    )
{
    phStatus_t   PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    uint8_t *   PH_MEMLOC_REM pRxBuffer;
    uint16_t   PH_MEMLOC_REM wRxBufferLen;

    status =  phalP40CmdPub_Sw_Int_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PH_OFF,
        PHAL_P40CMDPUB_INT_CLA_DEFAULT,
        bCmd,
        (uint8_t)(wStartAddr >> 8),  (uint8_t)wStartAddr,
        wDataLength, 0,
        pData, wDataLength,
        &pRxBuffer, &wRxBufferLen );

    /* standard error check*/
    return PH_ADD_COMPCODE(phalP40CmdPub_Sw_Int_CheckReturnResult(pDataParams, status), PH_COMP_AL_P40CMDPUB);
}

/***************************************************************************************************************************/
void phalP40CmdPub_Sw_Int_ResetAuthStatus(phalP40CmdPub_Sw_DataParams_t * pDataParams)
{
    memset(pDataParams->bSessionKey, 0x00, sizeof(pDataParams->bSessionKey)); /* PRQA S 3200 */
/*    pDataParams->bKeyNo = 0xFF;*/
    memset(pDataParams->bIv, 0x00, sizeof(pDataParams->bIv)); /* PRQA S 3200 */
/*    pDataParams->bAuthMode = PHAL_MFDF_NOT_AUTHENTICATED;
      pDataParams->bCryptoMethod = 0xFF;
      pDataParams->bWrappedMode = 0x00;
*/
}


#endif  /* NXPBUILD__PHAL_P40CMDPUB_SW */
