/*
 * Copyright 2016 - 2018, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Sam X Mode Virtual Card Architecture(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalVca.h>
#include <phpalMifare.h>
#include <phKeyStore.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_VCA_SAM_X

#include "../phalVca_Int.h"
#include "phalVca_Sam_X.h"
#include <phhalHw_SamAV2_Cmd.h>

#ifdef NXPBUILD__PHAL_VCA_INTERNAL
#include "../comps/phhalHw/src/Sam/Commands/07_VirtualCard/phhalHw_Sam_Cmd_VC.h"
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

phStatus_t phalVca_SamAV2_X_Init(phalVca_SamAV2_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV2_DataParams_t * pSamHal,
    phpalMifare_SamAV2_X_DataParams_t * pPalMifareDataParams, phalVca_SamAV2_X_IidTable_t * pIidTable, phalVca_SamAV2_X_KeysTable_t * pKeysTable,
    uint16_t wNumIidKeysTableEntries, phalVca_SamAV2_X_CardTableEntry_t * pCardTableStorage, uint16_t wNumCardTableStorageEntries)
{
    if(sizeof(phalVca_SamAV2_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pIidTable, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pKeysTable, PH_COMP_AL_VCA);
    if(wNumCardTableStorageEntries)
        PH_ASSERT_NULL_PARAM(pCardTableStorage, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAMAV2_X_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;
    pDataParams->pIidTable = pIidTable;
    pDataParams->pKeysTable = pKeysTable;
    pDataParams->wNumIidKeysTableEntries = wNumIidKeysTableEntries;
    pDataParams->wCurrentIidKeysTablePos = 0U;
    pDataParams->wCurrentCardTablePos = 0U;
    pDataParams->pCardTable = pCardTableStorage;
    pDataParams->wNumCardTableEntries = wNumCardTableStorageEntries;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
phStatus_t phalVca_SamAV3_X_Init(phalVca_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pSamHal,
    phpalMifare_SamAV3_X_DataParams_t * pPalMifareDataParams)
{
    if(sizeof(phalVca_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAMAV3_X_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */

#ifdef NXPBUILD__PHAL_VCA_INTERNAL
phStatus_t phalVca_Sam_X_Init(phalVca_Sam_X_DataParams_t* pDataParams, uint16_t wSizeOfDataParams, phhalHw_Sam_DataParams_t* pSamHal,
    phpalMifare_Sam_X_DataParams_t* pPalMifareDataParams)
{
    if(sizeof(phalVca_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAM_X_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

phStatus_t phalVca_Sam_X_StartCardSelection(phalVca_SamAV2_X_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bIndex;

    /* Send the complete stream to the SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(pDataParams->pSamHal, PH_EXCHANGE_BUFFER_FIRST,
        NULL, 0U, NULL, 0U, NULL, NULL, NULL));

    pDataParams->wCurrentCardTablePos = 0U;
    pDataParams->wCurrentIidKeysTablePos = 0U;

    for(bIndex = 0U; bIndex < pDataParams->wNumCardTableEntries; ++bIndex)
    {
        pDataParams->pCardTable[bIndex].bNumKeyDuos = 0U;
        pDataParams->pCardTable[bIndex].wIidIndex = 0xFFU;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_FinalizeCardSelection(phalVca_SamAV2_X_DataParams_t * pDataParams, uint16_t* pNumValidIids)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wResponseLength;
    uint16_t    PH_MEMLOC_REM wIndex;
    uint16_t    PH_MEMLOC_REM wIidIndex;
    uint8_t     PH_MEMLOC_REM bValidIids = 0;
    uint8_t     PH_MEMLOC_REM bDummyValidIids = 0;
    uint8_t     PH_MEMLOC_REM bDummyIidIndex = 0;
    uint8_t     PH_MEMLOC_REM bPiccReturnCode;

    /* Send the complete stream to the SAM */
    wStatus = phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(pDataParams->pSamHal, PH_EXCHANGE_BUFFER_LAST,
        NULL, 0U, &pDataParams->pPcdCaps[1], pDataParams->pPcdCaps[0], &pResponse, &wResponseLength,
        &bPiccReturnCode);

    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) &&
        (wStatus != PH_ADD_COMPCODE(PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN, PH_COMP_HAL)))
    {
        return wStatus;
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(1U, bPiccReturnCode));

    if(wResponseLength != pDataParams->wCurrentCardTablePos * 11U)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_AL_VCA);
    }

    wIidIndex = 0U;

    /* Now we need to store the card data */
    for(wIndex = 0U; wIndex < pDataParams->wCurrentCardTablePos; ++wIndex)
    {
        /* We always copy the data into the card table */
        (void) memcpy(pDataParams->pCardTable[wIndex].pCardData, &pResponse[1], 10);

        /* Was it a valid entry? */
        if(pResponse[0U] != 0xFFU)
        {
            /* Copy the found IID index */
            ++bValidIids;
            pDataParams->pCardTable[wIndex].wIidIndex = wIidIndex + pResponse[0U];
        }
        else
        {
            /* Invalidate the found IID index */
            ++bDummyValidIids;
            pDataParams->pCardTable[wIndex].wIidIndex = bDummyIidIndex + pResponse[0U];
        }
        /* Update the IidIndex */
        wIidIndex = wIidIndex + pDataParams->pCardTable[wIndex].bNumKeyDuos;
        pResponse += (10U /* Card Data */ + 1U /* Success index */);
    }

    *pNumValidIids = bValidIids;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_SelectVc(phalVca_SamAV2_X_DataParams_t * pDataParams, uint16_t wValidIidIndex, uint16_t wKeyNo,
    uint16_t wKeyVer)
{
    /* local variables */
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bPiccReturnCode;

    status = phalVca_Sam_X_ResolveValidIndex(
        pDataParams,
        wValidIidIndex,
        &wValidIidIndex);

    /* for the case of an overflow we always send the first VCIID number to the SAM */
    if((status  & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER)
    {
        wValidIidIndex = 0U;
    }
    else
    {
        PH_CHECK_SUCCESS(status);
    }

    /* Prepare Buffer */
    status = phhalHw_SamAV2_Cmd_MfpSelectVirtualCard(pDataParams->pSamHal, (uint8_t) wKeyNo, (uint8_t) wKeyVer,
        pDataParams->pIidTable[pDataParams->pCardTable[wValidIidIndex].wIidIndex].pIid, NULL, 0U, &bPiccReturnCode);

    if(((status & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((status & PH_ERR_MASK) != PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN))
    {
        return status;
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(1U, bPiccReturnCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_DeselectVc(phalVca_SamAV2_X_DataParams_t * pDataParams)
{
    /* local variables */
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM aCommand[1 /* command code */];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRxLength;

    /* command code */
    aCommand[0U] = PHAL_VCA_CMD_DVC;

    /* command exchange */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCommand,
        1U,
        &pResponse,
        &wRxLength));

    /* check response */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(wRxLength, pResponse[0]));

    /* check response length */
    if(wRxLength != 1U /* STATUS */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_VcSupport(phalVca_SamAV2_X_DataParams_t * pDataParams, uint8_t * pIid, uint16_t wKeyEncNumber, uint16_t wKeyEncVersion,
    uint16_t wKeyMacNumber, uint16_t wKeyMacVersion)
{
    /* Check available space in key duos list */
    if(pDataParams->wCurrentIidKeysTablePos >= pDataParams->wNumIidKeysTableEntries)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    /* Add keys to key list */
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[0] = (uint8_t) wKeyEncNumber;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[1] = (uint8_t) wKeyEncVersion;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[2] = (uint8_t) wKeyMacNumber;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[3] = (uint8_t) wKeyMacVersion;

    (void) memcpy(pDataParams->pIidTable[pDataParams->wCurrentIidKeysTablePos].pIid, pIid, 16U);
    ++pDataParams->wCurrentIidKeysTablePos;
    ++pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_VcSupportLast(phalVca_SamAV2_X_DataParams_t * pDataParams, uint8_t * pIid, uint8_t bLenCap, uint8_t * pPcdCapabilities,
    uint16_t wKeyEncNumber, uint16_t wKeyEncVersion, uint16_t wKeyMacNumber, uint16_t wKeyMacVersion)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t *   PH_MEMLOC_REM pKeyDuosPointer = 0;
    uint8_t *   PH_MEMLOC_REM pIidPointer = 0;

    /* Copy PCD Caps */
    (void) memset(pDataParams->pPcdCaps, 0x00, 4);

    if(bLenCap)
    {
        if(bLenCap > 3U)
        {
            bLenCap = 3U;
        }
        pDataParams->pPcdCaps[0] = bLenCap;
        (void) memcpy(&pDataParams->pPcdCaps[1], pPcdCapabilities, bLenCap);
    }

    /* Check available space in IID list */
    if(pDataParams->wCurrentIidKeysTablePos >= pDataParams->wNumIidKeysTableEntries)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    /* Check available space in card table list */
    if(pDataParams->wCurrentCardTablePos >= pDataParams->wNumCardTableEntries)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    /* Add keys to key list */
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[0] = (uint8_t) wKeyEncNumber;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[1] = (uint8_t) wKeyEncVersion;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[2] = (uint8_t) wKeyMacNumber;
    pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos].pKeys[3] = (uint8_t) wKeyMacVersion;
    (void) memcpy(pDataParams->pIidTable[pDataParams->wCurrentIidKeysTablePos].pIid, pIid, 16U);
    ++pDataParams->wCurrentIidKeysTablePos;
    ++pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;

    /* Safety check */
    if(pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos > pDataParams->wCurrentIidKeysTablePos)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_AL_VCA);
    }

    /* Add Num key Duos */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(pDataParams->pSamHal, PH_EXCHANGE_BUFFER_CONT,
        &pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos, 1U, NULL, 0U, NULL, NULL, NULL));

    pKeyDuosPointer = pDataParams->pKeysTable[pDataParams->wCurrentIidKeysTablePos - pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos].pKeys;
    pIidPointer = pDataParams->pIidTable[pDataParams->wCurrentIidKeysTablePos - pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos].pIid;

    for(bIndex = 0U; bIndex < pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos; ++bIndex)
    {
        /* Add key Duos */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(pDataParams->pSamHal, PH_EXCHANGE_BUFFER_CONT,
            pKeyDuosPointer, 4U, NULL, 0U, NULL, NULL, NULL));

        pKeyDuosPointer = pKeyDuosPointer + 4U;

        /* Add Iid */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(pDataParams->pSamHal, PH_EXCHANGE_BUFFER_CONT,
            pIidPointer, 16U, NULL, 0U, NULL, NULL, NULL));

        pIidPointer = pIidPointer + 16U;
    }

    pDataParams->wCurrentCardTablePos++;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_GetIidInfo(phalVca_SamAV2_X_DataParams_t * pDataParams, uint16_t wValidIidIndex, uint16_t * pIidIndex,
    uint8_t * pVcUidSize, uint8_t * pVcUid, uint8_t * pInfo, uint8_t * pPdCapabilities)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_X_ResolveValidIndex(
        pDataParams,
        wValidIidIndex,
        &wValidIidIndex));

    *pIidIndex = pDataParams->pCardTable[wValidIidIndex].wIidIndex;
    *pInfo = pDataParams->pCardTable[wValidIidIndex].pCardData[0];
    pPdCapabilities[0U] = pDataParams->pCardTable[wValidIidIndex].pCardData[1];
    pPdCapabilities[1U] = pDataParams->pCardTable[wValidIidIndex].pCardData[2];

    if(*pInfo & 0x80U)
    {
        *pVcUidSize = 4U;
        (void) memcpy(pVcUid, &pDataParams->pCardTable[wValidIidIndex].pCardData[3U], 4U);
    }
    else
    {
        *pVcUidSize = 7U;
        (void) memcpy(pVcUid, &pDataParams->pCardTable[wValidIidIndex].pCardData[3U], 7U);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
phStatus_t phalVca_Sam_X_IsoSelect(void * pDataParams, uint8_t bOption, uint8_t bDFnameLen,
    uint8_t * pDFname, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t bEncKeyNo, uint8_t bEncKeyVer,
    uint8_t bMacKeyNo, uint8_t bMacKeyVer, uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint16_t    PH_MEMLOC_REM wPiccRetCode = 0;

    /* Let HAL do the rest */
    wStatus = phalVca_Sam_X_VCA_Select(
        pDataParams,
        bOption,
        bEncKeyNo,
        bEncKeyVer,
        bMacKeyNo,
        bMacKeyVer,
        pDFname,
        bDFnameLen,
        pDivInput,
        bDivInputLen,
        &pResponse,
        pRespLen,
        &wPiccRetCode);

    /* Validate PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, wPiccRetCode));
    }
    else
    {
        /* Check for SUCCESS. */
        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */


phStatus_t phalVca_Sam_X_ProximityCheck(phalVca_SamAV2_X_DataParams_t * pDataParams, uint8_t bGenerateRndC, uint8_t * pRndC,
    uint8_t bPps1, uint8_t bNumSteps, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pUsedRndC)
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bReturnCode;

    /* Satisfy Compiler */
    if(bPps1 || pUsedRndC || bGenerateRndC || pRndC);

    /* Recalculate bNumSteps to X-config */
    switch(bNumSteps)
    {
        case 1:
            bNumSteps = 0U;
            break;
        case 2:
            bNumSteps = 4U;
            break;
        case 3:
            bNumSteps = 3U;
            break;
        case 4:
        case 5: /* Note - not correct translation */
            bNumSteps = 2U;
            break;
        case 6: /* Note - not correct translation */
        case 7:
            bNumSteps = 1U;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    status = phhalHw_SamAV2_Cmd_MfpProximityCheck(
        pDataParams->pSamHal,
        PHHAL_HW_SAMAV2_CMD_MFP_PROXIMITY_CHECK_DEFAULT,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        bNumSteps,
        NULL,
        0U,
        &bReturnCode);

    if((status != PH_ERR_SUCCESS) && (status != (PH_COMP_HAL | PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)))
    {
        /* Remap crypto error */
        if(status == (PH_COMP_HAL | PHHAL_HW_SAMAV2_ERR_CRYPTO))
        {
            return PH_ADD_COMPCODE(PHAL_VCA_ERR_AUTH, PH_COMP_AL_VCA);
        }

        return status;
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(1, bReturnCode));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
phStatus_t phalVca_Sam_X_ProximityCheckNew(void *pDataParams, uint8_t bOption, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bNumOfRand, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pOption,
    uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bOption_Tmp = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;
    uint8_t *   PH_MEMLOC_REM pResponse_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the Option */
    bOption_Tmp = bOption;
    bOption_Tmp |= (uint8_t) (phalVca_Sam_X_GetWrappedMode(pDataParams) ? 0x00U /* PHHAL_HW_SAMAV3_CMD_PC_NATIVE_FORMAT or PHHAL_HW_SAM_CMD_PC_NATIVE_FORMAT*/ :
        0x04U /* HAL_HW_SAMAV3_CMD_PC_WRAPPED_FORMAT or HAL_HW_SAM_CMD_PC_WRAPPED_FORMAT*/);
    bOption_Tmp |= (uint8_t) (bDivInputLen ? 0x01U /* PHHAL_HW_SAMAV3_CMD_PROXIMITY_CHECK_DIV_ON or PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON*/ :
        PHHAL_HW_SAMAV3_CMD_PROXIMITY_CHECK_DIV_OFF);

    wStatus = phalVca_Sam_X_VCA_ProximityCheck(
        pDataParams,
        bOption_Tmp,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        bNumOfRand,
        pDivInput,
        bDivInputLen,
        &pResponse_Tmp,
        &wRespLen);

    /* Validate PICC error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, pResponse_Tmp[0U]));
    }
    else
    {
        /* Copy the response to the parameters. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Response starts from position 1. */
            bOffset = 1U;

            /* Copy the Option. */
            *pOption = pResponse_Tmp[bOffset++];

            /* Save Published Response Time from response data. */
            pPubRespTime[0U] = pResponse_Tmp[bOffset++];
            pPubRespTime[1U] = pResponse_Tmp[bOffset++];

            /* Save PPS from response data */
            if(*pOption & 0x01U)
            {
                *pResponse = pResponse_Tmp[bOffset];
                *pRespLen = 1U;
            }

            /* Save ActBitRate from response data */
            if(*pOption & 0x02U)
            {
                memcpy(pResponse, &pResponse_Tmp[bOffset], (wRespLen - bOffset));
                *pRespLen = (uint8_t) (wRespLen - bOffset);
            }
        }
        else
        {
            /* Do Nothing */
        }
    }

    return wStatus;
}




phStatus_t phalVca_Sam_X_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
    case PHAL_VCA_ADDITIONAL_INFO:
        phalVca_Sam_X_SetAdditionalInfo(pDataParams, wValue);
        break;

    case PHAL_VCA_WRAPPED_MODE:
        phalVca_Sam_X_SetWrappedMode(pDataParams, (uint8_t)wValue);
        break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_GetConfig(phalVca_SamAV3_X_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PHAL_VCA_ADDITIONAL_INFO:
            *pValue = pDataParams->wAdditionalInfo;
            break;

        case PHAL_VCA_WRAPPED_MODE:
            *pValue = pDataParams->bWrappedMode;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_X_ResolveValidIndex(phalVca_SamAV2_X_DataParams_t * pDataParams, uint16_t wIidIndex, uint16_t * pValidIndex)
{
    uint8_t     PH_MEMLOC_REM bCurrentValidIndex = 0;
    uint8_t     PH_MEMLOC_REM bDummyValidIndex = 0;
    uint8_t     PH_MEMLOC_REM bDummyFoundIndex;
    uint8_t     PH_MEMLOC_REM bIndex;
    uint16_t    PH_MEMLOC_REM bFoundIndex = pDataParams->wCurrentCardTablePos;

    for(bIndex = 0; bIndex < pDataParams->wCurrentCardTablePos; ++bIndex)
    {
        if(pDataParams->pCardTable[bIndex].wIidIndex == 0xFFU)
        {
            if(bCurrentValidIndex == wIidIndex)
            {
                bDummyFoundIndex = bIndex;
            }
            else
            {
                bDummyFoundIndex = bIndex;
            }
            bDummyValidIndex++;
        }
        else
        {
            if(bCurrentValidIndex == wIidIndex)
            {
                bFoundIndex = bIndex;
            }
            else
            {
                bDummyFoundIndex = bIndex;
            }
            bCurrentValidIndex++;
        }
    }

    /* Check if the index is valid */
    if(bFoundIndex >= pDataParams->wCurrentCardTablePos)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    *pValidIndex = bFoundIndex;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */

phStatus_t phalVca_Sam_X_VCA_Select(void* pDataParams, uint8_t bOption, uint8_t bEncKeyNo, uint8_t bEncKeyVer, uint8_t bMacKeyNo,
    uint8_t bMacKeyVer, uint8_t* pIID, uint8_t bIIDLen, uint8_t* pDivInput, uint8_t bDivInputLen, uint8_t** ppResponse,
    uint16_t* pRespLen, uint16_t* pPiccRetCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
    case PHAL_VCA_SAM_X_ID:
        wStatus = phhalHw_Sam_Cmd_VCA_Select(
            PHAL_VCA_RESOLVE_HAL_DATAPARAMS_X(pDataParams),
            bOption,
            bEncKeyNo,
            bEncKeyVer,
            bMacKeyNo,
            bMacKeyVer,
            pIID,
            bIIDLen,
            pDivInput,
            bDivInputLen,
            ppResponse,
            pRespLen,
            pPiccRetCode);
        break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
    case PHAL_VCA_SAMAV3_X_ID:
        wStatus = phhalHw_SamAV3_Cmd_VCA_Select(
            PHAL_VCA_RESOLVE_HAL_DATAPARAMS_X(pDataParams),
            bOption,
            bEncKeyNo,
            bEncKeyVer,
            bMacKeyNo,
            bMacKeyVer,
            pIID,
            bIIDLen,
            pDivInput,
            bDivInputLen,
            ppResponse,
            pRespLen,
            pPiccRetCode);
        break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

    default:
        wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
        break;
    }

    return wStatus;
}

phStatus_t phalVca_Sam_X_VCA_ProximityCheck(void* pDataParams, uint8_t bOption, uint8_t bKeyNo,
    uint8_t bKeyVer, uint8_t bNumOfRand, uint8_t* pDivInput, uint8_t bDivInputLen, uint8_t** ppResponse, uint16_t* pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
    case PHAL_VCA_SAM_X_ID:
        wStatus = phhalHw_Sam_Cmd_VCA_ProximityCheck(
            PHAL_VCA_RESOLVE_HAL_DATAPARAMS_X(pDataParams),
            bOption,
            bKeyNo,
            bKeyVer,
            bNumOfRand,
            pDivInput,
            bDivInputLen,
            ppResponse,
            pRespLen);
        break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
    case PHAL_VCA_SAMAV3_X_ID:
        wStatus = phhalHw_SamAV3_Cmd_VCA_ProximityCheck(
            PHAL_VCA_RESOLVE_HAL_DATAPARAMS_X(pDataParams),
            bOption,
            bKeyNo,
            bKeyVer,
            bNumOfRand,
            pDivInput,
            bDivInputLen,
            ppResponse,
            pRespLen);
        break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

    default:
        wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
        break;
    }

    return wStatus;
}

void phalVca_Sam_X_SetAdditionalInfo(void* pDataParams, uint16_t wValue)
{
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
    case PHAL_VCA_SAM_X_ID:
        ((phalVca_Sam_X_DataParams_t*)pDataParams)->wAdditionalInfo = wValue;
        break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
    default:
        ((phalVca_SamAV3_X_DataParams_t*)pDataParams)->wAdditionalInfo = wValue;
        break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint16_t phalVca_Sam_X_GetAdditionalInfo(void* pDataParams)
{
    uint16_t PH_MEMLOC_REM wValue = 0;

    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
            wValue = ((phalVca_Sam_X_DataParams_t*) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            wValue = ((phalVca_SamAV3_X_DataParams_t*) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return wValue;
}

void phalVca_Sam_X_SetWrappedMode(void* pDataParams, uint16_t wValue)
{
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
    case PHAL_VCA_SAM_X_ID:
        ((phalVca_Sam_X_DataParams_t*)pDataParams)->wAdditionalInfo = wValue;
        break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
    default:
        ((phalVca_SamAV3_X_DataParams_t*)pDataParams)->wAdditionalInfo = wValue;
        break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

uint8_t phalVca_Sam_X_GetWrappedMode(void* pDataParams)
{
    uint8_t PH_MEMLOC_REM bValue = 0;

    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
            bValue = ((phalVca_Sam_X_DataParams_t*) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bValue = ((phalVca_SamAV3_X_DataParams_t*) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bValue;
}
#endif /* NXPBUILD__PHAL_VCA_SAM_X */
