/*
 * Copyright 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 USB BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_MP300USB

#include "errno.h"
#include "phbalReg_Mp300Usb_If.h"

phStatus_t phbalReg_Mp300Usb_If_Open(phbalReg_Mp300Usb_DataParams_t * pDataParams)
{
#ifdef WIN64_BUILD
	/* This is a macro Define for X64 build only
	 * Dummy return since usb_XXX_XX interfaces throw Reference not found error in case of WIN64 Build
	 */
	/* satisfy compiler */
	if(pDataParams);
	return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    int32_t           PH_MEMLOC_REM dwError = 0;
    struct usb_bus    PH_MEMLOC_COUNT * pUsbBus = NULL;
    struct usb_device PH_MEMLOC_COUNT * pUsbDev = NULL;
    uint8_t           PH_MEMLOC_REM bReadEpFound = 0;
    uint8_t           PH_MEMLOC_REM bWriteEpFound = 0;
    uint8_t           PH_MEMLOC_BUF bNameBuffer[256];
    phStatus_t        PH_MEMLOC_REM statusTmp;

    uint8_t           PH_MEMLOC_COUNT bActConfigIndex = 0;
    uint8_t           PH_MEMLOC_COUNT bActInterfaceIndex = 0;
    int32_t           PH_MEMLOC_COUNT dwActInterfaceDescIndex = 0;
    uint8_t           PH_MEMLOC_COUNT bActEndpointIndex = 0;

    struct usb_config_descriptor    PH_MEMLOC_REM * pActConfig = NULL;
    struct usb_interface            PH_MEMLOC_REM * pActInterface = NULL;
    struct usb_interface_descriptor PH_MEMLOC_REM * pActInterfaceDesc = NULL;
    struct usb_endpoint_descriptor  PH_MEMLOC_REM * pActEndpoint = NULL;

    /* Its possible to open multiple handles and claim multiple resources but this is not needed here. */
    if (pDataParams->pUsbDevHandle != INVALID_HANDLE_VALUE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    pDataParams->pfInit();
    dwError = pDataParams->pfFindBusses(); /* find all busses */
    if (dwError < 0)
    {
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }
    dwError = pDataParams->pfFindDevices(); /* find all connected devices */
    if (dwError < 0)
    {
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }

    for (pUsbBus = (struct usb_bus *)pDataParams->pfGetBusses(); pUsbBus; pUsbBus = pUsbBus->next)
    {
        for (pUsbDev = pUsbBus->devices; pUsbDev; pUsbDev = pUsbDev->next)
        {
            /* If no name is set but filter is set check it */
            if (pDataParams->pPortName == NULL && (pDataParams->wVendorIdFilter != 0 || pDataParams->wProductIdFilter != 0))
            {
                if (pUsbDev->descriptor.idVendor != pDataParams->wVendorIdFilter ||
                    pUsbDev->descriptor.idProduct != pDataParams->wProductIdFilter)
                {
                    continue;
                }
            }

            /* Try to open handle */
            pDataParams->pUsbDevHandle = (usb_dev_handle *)pDataParams->pfOpen(pUsbDev);
            if (pDataParams->pUsbDevHandle <= 0)
            {
                /* If fails to open the handle take the next one */
                pDataParams->pUsbDevHandle = INVALID_HANDLE_VALUE;
                continue;
            }

            /* If portname is set check if the name is correct */
            if (pDataParams->pPortName != NULL)
            {
                /* Get the name of the device and check if it match with the port set */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Mp300Usb_If_GetDeviceName(pDataParams, pUsbDev, pDataParams->pUsbDevHandle, bNameBuffer, sizeof(bNameBuffer)));
                if (strcmp((char*)bNameBuffer,(char*)pDataParams->pPortName) != 0)
                {
                    /* If another port is selected close the actual port */
                    dwError = pDataParams->pfClose((usb_dev_handle *)pDataParams->pUsbDevHandle);
                    if (dwError != 0)
                    {
                        return phbalReg_Mp300Usb_If_ConvertError(dwError);
                    }
                    pDataParams->pUsbDevHandle = INVALID_HANDLE_VALUE;
                    continue;
                }
            }

            for (bActConfigIndex = 0; bActConfigIndex < pUsbDev->descriptor.bNumConfigurations; bActConfigIndex++)
            {
                pActConfig = &pUsbDev->config[bActConfigIndex];

                /* The Spy facility seems to always try and claim interface 0 and it it is already claimed it causes a failure. */
                for (bActInterfaceIndex = 0; bActInterfaceIndex < pActConfig->bNumInterfaces; bActInterfaceIndex++)
                {
                    pActInterface = &pActConfig->interface[bActInterfaceIndex];
                    dwError = pDataParams->pfClaimInterface((usb_dev_handle *)pDataParams->pUsbDevHandle, bActInterfaceIndex);
                    if (dwError == 0)
                    {   /* We have claimed an interface. */
                        /* Find the correct endpoint (just use first found) */
                        bReadEpFound = 0;
                        bWriteEpFound = 0;

                        for (dwActInterfaceDescIndex = 0; dwActInterfaceDescIndex < pActInterface->num_altsetting; dwActInterfaceDescIndex++)
                        {
                            pActInterfaceDesc = &pActInterface->altsetting[dwActInterfaceDescIndex];

                            pDataParams->bInterfaceNumber = pActInterfaceDesc->bInterfaceNumber;

                            for (bActEndpointIndex = 0; bActEndpointIndex < pActInterfaceDesc->bNumEndpoints; bActEndpointIndex++)
                            {
                                pActEndpoint = &pActInterfaceDesc->endpoint[bActEndpointIndex];

                                /* Read endpoint has 0x80 in address */
                                if ((pActEndpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK) == USB_ENDPOINT_DIR_MASK && bReadEpFound == 0)
                                {
                                    pDataParams->bUsbReadEndpointAddress = pActEndpoint->bEndpointAddress;
                                    bReadEpFound = 1;
                                }
                                if ((pActEndpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK) != USB_ENDPOINT_DIR_MASK && bWriteEpFound == 0)
                                {
                                    pDataParams->bUsbWriteEndpointAddress = pActEndpoint->bEndpointAddress;
                                    bWriteEpFound = 1;
                                }
                                /* If read and write endpoint found -> return */
                                if (bReadEpFound != 0 && bWriteEpFound != 0)
                                    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
                            }
                        }
                    }
                }
            }
            /* Not a correct handle so close it again */
            dwError = pDataParams->pfClose((usb_dev_handle *)pDataParams->pUsbDevHandle);
            if (dwError != 0)
            {
                return phbalReg_Mp300Usb_If_ConvertError(dwError);
            }
            pDataParams->pUsbDevHandle = INVALID_HANDLE_VALUE;
            pDataParams->bInterfaceNumber = INVALID_USB_HANDLE;
            pDataParams->bUsbWriteEndpointAddress = INVALID_USB_HANDLE;
            pDataParams->bUsbReadEndpointAddress = INVALID_USB_HANDLE;
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Mp300Usb_If_Close(phbalReg_Mp300Usb_DataParams_t * pDataParams)
{
#ifdef WIN64_BUILD
	/* This is a macro Define for X64 build only
	 * Dummy return since usb_find_** interfaces throw Reference not found error in case of WIN64 Build
	 */
	/* satisfy compiler */
	if(pDataParams);
	return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    int32_t PH_MEMLOC_REM dwError = 0;

    if (pDataParams->pUsbDevHandle == INVALID_HANDLE_VALUE || pDataParams->bInterfaceNumber == INVALID_USB_HANDLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    dwError = pDataParams->pfReleaseInterface((usb_dev_handle *)pDataParams->pUsbDevHandle,pDataParams->bInterfaceNumber);
    if (dwError != 0)
    {
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }
    pDataParams->bInterfaceNumber = INVALID_USB_HANDLE;

    dwError = pDataParams->pfClose((usb_dev_handle *)pDataParams->pUsbDevHandle);
    if (dwError != 0)
    {
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }
    pDataParams->pUsbDevHandle = INVALID_HANDLE_VALUE;

    /* Not needed */
    /* usb_deinit(); */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Mp300Usb_If_Exchange(phbalReg_Mp300Usb_DataParams_t * pDataParams, uint8_t* pTxBuffer, uint16_t wTxLength, uint16_t wRxBufSize, uint8_t* pRxBuffer, uint16_t* pRxLength)
{
#ifdef WIN64_BUILD
	/* This is a macro Define for X64 build only
	 * Dummy return since usb_find_** interfaces throw Reference not found error in case of WIN64 Build
	 */
	/* satisfy compiler */
	if(pDataParams || pTxBuffer || wTxLength || wRxBufSize || pRxBuffer || pRxLength);
	return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    int32_t  PH_MEMLOC_REM dwError = 0;
    uint16_t PH_MEMLOC_COUNT wCounter = 0;
    uint16_t wActRxBufSize = 256;

    if (pDataParams->pUsbDevHandle == INVALID_HANDLE_VALUE ||
        pDataParams->bUsbWriteEndpointAddress == INVALID_USB_HANDLE ||
        pDataParams->bUsbReadEndpointAddress == INVALID_USB_HANDLE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* to pass NUnit test parse for wRxBufSize = 100 or 64 and TxLength = 2 */
    /*if ((wTxLength == 100 && wRxBufSize == 100) || (wTxLength == 2 && wRxBufSize == 64))
    {
        /* check also first two bytes of TxBuffer for 0 */
        /*if (*pTxBuffer == 0)
        {
            pTxBuffer ++;
            if (*pTxBuffer == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
            }
        }
    }*/

    /* Apply platform independant delay */
    for (wCounter = 0; wCounter < pDataParams->wExchangeDelayCount; wCounter++);

    switch (pDataParams->bExchangeMode)
    {
    case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_INTERRUPT:
        dwError = pDataParams->pfInterruptWrite((usb_dev_handle *)pDataParams->pUsbDevHandle, pDataParams->bUsbWriteEndpointAddress, (char*)pTxBuffer, wTxLength, pDataParams->wTxTimeout);
        break;

    case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_BULK:
        dwError = pDataParams->pfBulkWrite((usb_dev_handle *)pDataParams->pUsbDevHandle, pDataParams->bUsbWriteEndpointAddress, (char*)pTxBuffer, wTxLength, pDataParams->wTxTimeout);
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    if (dwError < 0)
    {
        return phbalReg_Mp300Usb_If_ConvertError(dwError);
    }
    if (dwError != (int32_t)wTxLength)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Clear reveive buffer */
    pRxBuffer[0] = 0;
    *pRxLength = 0;
    do
    {
        wActRxBufSize = wRxBufSize - *pRxLength;

        switch (pDataParams->bExchangeMode)
        {
        case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_INTERRUPT:
            dwError = pDataParams->pfInterruptRead((usb_dev_handle *)pDataParams->pUsbDevHandle, pDataParams->bUsbReadEndpointAddress, (char*)(pRxBuffer + (*pRxLength)), wActRxBufSize, pDataParams->wRxTimeout + 500);
            break;

        case PHBAL_REG_MP300USB_VALUE_EXCHANGEMODE_BULK:
            dwError = pDataParams->pfBulkRead((usb_dev_handle *)pDataParams->pUsbDevHandle, pDataParams->bUsbReadEndpointAddress, (char*)(pRxBuffer + (*pRxLength)), wActRxBufSize, pDataParams->wRxTimeout + 500);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
        }

        /* Workaround for 256 bit exchanges with TCL2 -> there is a timeout even tougth all data are received */
        if (dwError == -116 && pRxBuffer[0] != 0 && wActRxBufSize >= 256 &&
            ((char*)(pRxBuffer + (*pRxLength)))[254] == '\r' &&
            ((char*)(pRxBuffer + (*pRxLength)))[255] == '\n')
        {
            for (wCounter = 0; wCounter < 254; wCounter++)
            {
                if (((char*)(pRxBuffer + (*pRxLength)))[wCounter] < 0x20 ||
                    ((char*)(pRxBuffer + (*pRxLength)))[wCounter] > 0x7F)
                {
                    return phbalReg_Mp300Usb_If_ConvertError(dwError);
                }
            }
            dwError = 256;
        }

        if (dwError < 0)
        {
            return phbalReg_Mp300Usb_If_ConvertError(dwError);
        }

        *pRxLength += (uint16_t)dwError;
        /* Loop as long as the full buffer is received and the last bytes are not linefeed */
    }while(dwError > 0 && dwError%256 == 0 && (pRxBuffer[*pRxLength - 2] != '\r' || pRxBuffer[*pRxLength - 1] != '\n'));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Mp300Usb_If_ConvertError(int32_t dwError)
{
    dwError = -dwError;
    switch (dwError)
    {
    case ETRANSFER_TIMEDOUT:
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    case EIO:
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    case EINVAL:
    case E2BIG:
    case EFAULT:
    case ENAMETOOLONG:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    case ERANGE:
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
    case ENOSYS:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
    case EPERM:
    case ENOENT:
    case ESRCH:
    case ENXIO:
    case EACCES:
    case EROFS:
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    case EINTR:
    case ENOEXEC:
    case EBADF:
    case ECHILD:
    case EAGAIN:
    case ENOMEM:
    case EBUSY:
    case EEXIST:
    case EXDEV:
    case ENODEV:
    case ENOTDIR:
    case EISDIR:
    case ENFILE:
    case EMFILE:
    case ENOTTY:
    case ESPIPE:
    case EMLINK:
    case EPIPE:
    case EDOM:
    case EDEADLK:
    case ENOLCK:
    case ENOTEMPTY:
    case EFBIG:
    case ENOSPC:
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }
}

phStatus_t phbalReg_Mp300Usb_If_GetDeviceName(phbalReg_Mp300Usb_DataParams_t * pDataParams, struct usb_device * pUsbDev, usb_dev_handle *pUsbDevHandle, uint8_t *pNameBuffer, uint32_t dwNameBufferLength)
{
#ifdef WIN64_BUILD
	/* This is a macro Define for X64 build only
	 * Dummy return since usb_find_** interfaces throw Reference not found error in case of WIN64 Build
	 */
	/* satisfy compiler */
	if(pDataParams || pUsbDev || pUsbDevHandle || pNameBuffer || dwNameBufferLength);
	return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#else
    int32_t  PH_MEMLOC_REM dwError = 0;
    uint32_t PH_MEMLOC_REM dwStringLength = 0;

    if (dwNameBufferLength < 6) /* at least on sign for every part of the name and 2 spaces between and the terminatin Zero */
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    if (pUsbDev->descriptor.iManufacturer)
    {
        dwError = pDataParams->pfGetStringSimple(pUsbDevHandle, pUsbDev->descriptor.iManufacturer, (char*)pNameBuffer, dwNameBufferLength);
        if (dwError > 0)
        {
            dwStringLength = (uint16_t)strlen((char*)pNameBuffer);
        }
        else
        {
            dwStringLength = 1;
            pNameBuffer[0] = '-';
        }
        if (dwStringLength >= dwNameBufferLength-1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
        }
        pNameBuffer[dwStringLength++] = ' ';
        pNameBuffer = &pNameBuffer[dwStringLength];
        dwNameBufferLength -= dwStringLength;
        dwStringLength = 0;
    }

    if (dwNameBufferLength < 4) /* at least on sign for every part of the name and 1 spaces between and the terminatin Zero */
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
    }

    if (pUsbDev->descriptor.iProduct)
    {
        dwError = pDataParams->pfGetStringSimple(pUsbDevHandle, pUsbDev->descriptor.iProduct, (char*)pNameBuffer, dwNameBufferLength);
        if (dwError > 0)
        {
            dwStringLength = (uint16_t)strlen((char*)pNameBuffer);
        }
        else
        {
            dwStringLength = 1;
            pNameBuffer[0] = '-';
        }
        if (dwStringLength >= dwNameBufferLength-1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
        }
        pNameBuffer[dwStringLength++] = ' ';
        pNameBuffer = &pNameBuffer[dwStringLength];
        dwNameBufferLength -= dwStringLength;
        dwStringLength = 0;
    }

    if (dwNameBufferLength < 2) /* at least on sign for the last part of the name and the terminatin Zero */
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
    }

    if (pUsbDev->descriptor.iSerialNumber)
    {
        dwError = pDataParams->pfGetStringSimple(pUsbDevHandle, pUsbDev->descriptor.iSerialNumber, (char*)pNameBuffer, dwNameBufferLength);
        if (dwError > 0)
        {
            dwStringLength = (uint16_t)strlen((char*)pNameBuffer);
        }
        else
        {
            dwStringLength = 1;
            pNameBuffer[0] = '-';
        }
        if (dwStringLength >= dwNameBufferLength)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_BAL);
        }
        pNameBuffer[dwStringLength] = 0;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#endif
}
#endif /* NXPBUILD__PHBAL_REG_MP300USB */
