/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * FPGA BOX BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_RDFPGAV6

#include "phbalReg_RdFpgaV6.h"
#include "phbalReg_RdFpgaV6_Int.h"
#include "external/phbalReg_RdFpgaV6_Ex.h"

#include <string.h>

#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#include "../../../phhalHw/src/RdFpgaV6/external/phDefines.h"
#pragma warning(pop)                    /* PRQA S 3116 */

phStatus_t phbalReg_RdFpgaV6_Init(
                                  phbalReg_RdFpgaV6_DataParams_t * pDataParams,
                                  uint16_t wSizeOfDataParams
                                  )
{
    if (sizeof(phbalReg_RdFpgaV6_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);

    /* initialize the data parameters to default values */
    pDataParams->wId                                        = PH_COMP_BAL | PHBAL_REG_RDFPGAV6_ID;
    pDataParams->pDeviceName                                = NULL;
    pDataParams->bPortOpen                                  = 0;
    pDataParams->pfRdFpga_Open                              = NULL;
    pDataParams->pfRdFpga_Close                             = NULL;
    pDataParams->pfRdFpga_Init                              = NULL;
    pDataParams->pfRdFpga_SetParam                          = NULL;
    pDataParams->pfRdFpga_GetParam                          = NULL;
    pDataParams->pfRdFpga_ExchangeBits                      = NULL;
    pDataParams->pfRdFpga_SetTargetJoiner                   = NULL;
    pDataParams->pfRdFpga_SetJoinerReg                      = NULL;
    pDataParams->pfRdFpga_GetJoinerReg                      = NULL;
    pDataParams->pfRdFpga_LoadJoinerIniFile                 = NULL;
    pDataParams->pfRdFpga_ActivateField                     = NULL;
    pDataParams->pfRdFpga_DeactivateField                   = NULL;
    pDataParams->pfRdFpga_HFPause                           = NULL;
    pDataParams->pfRdFpga_SetPotiPos                        = NULL;
    pDataParams->pfRdFpga_SetDefaultPotiPos                 = NULL;
    pDataParams->pfRdFpga_GetPotiPos                        = NULL;
    pDataParams->pfRdFpga_IncPotiPos                        = NULL;
    pDataParams->pfRdFpga_DecPotiPos                        = NULL;
    pDataParams->pfRdFpga_SetTimeoutValue                   = NULL;
    pDataParams->pfRdFpga_GetTimeoutValue                   = NULL;
    pDataParams->pfRdFpga_LoadPattern                       = NULL;
    pDataParams->pfRdFpga_PlayPattern                       = NULL;
    pDataParams->pfRdFpga_PlaySeqA                          = NULL;
    pDataParams->pfRdFpga_PlaySeqB                          = NULL;
    pDataParams->pfRdFpga_PlaySinus                         = NULL;
    pDataParams->pfRdFpga_SelectTrigger                     = NULL;
    pDataParams->pfRdFpga_LabelReset                        = NULL;
    pDataParams->pfRdFpga_LabelResetCounterReset            = NULL;
    pDataParams->pfRdFpga_LabelResetCounterResetSingleShot  = NULL;
    pDataParams->pfRdFpga_LabelResetSingleShot              = NULL;
    pDataParams->pfRdFpga_LabelResetWithCommand             = NULL;
    pDataParams->pfRdFpga_ActivateFDTMeasurement            = NULL;
    pDataParams->pfRdFpga_GetFDT                            = NULL;
    pDataParams->pfRdFpga_GetFDTFast                        = NULL;
    pDataParams->pfRdFpga_ActivateAntiTearingMode           = NULL;
    pDataParams->pfRdFpga_ResonanceFrequenceTest            = NULL;
    pDataParams->pfRdFpga_ActivateFDTMeasurementPICCtoPCD   = NULL;
    pDataParams->pfRdFpga_ActivateLMAMeasurement            = NULL;
    pDataParams->pfRdFpga_ActivateSequencerMode             = NULL;
    pDataParams->pfRdFpga_DeactivateModes                   = NULL;
    pDataParams->pfRdFpga_SetJoinerRxGain                   = NULL;
    pDataParams->pfRdFpga_GetJoinerRxGain                   = NULL;
    pDataParams->pfRdFpga_SetFDTPCDforActiveDS              = NULL;
    pDataParams->pfRdFpga_SetFDTPCDforDS                    = NULL;
    pDataParams->pfRdFpga_GetVersion                        = NULL;
    pDataParams->pfRdFpga_GetRevision                       = NULL;
    pDataParams->pfRdFpga_GetErrorCodes                     = NULL;
    pDataParams->pfRdFpga_SetClockSource                    = NULL;
    pDataParams->pfRdFpga_GetFirstResponse                  = NULL;
    pDataParams->pfRdFpga_GetResponse                       = NULL;
    pDataParams->pfRdFpga_GetIQTimeoutValue                 = NULL;
    pDataParams->pfRdFpga_SetIQTimeoutValue                 = NULL;
    pDataParams->pfRdFpga_GetTerminalId                     = NULL;
    pDataParams->pfRdFpga_ResetJoiner                       = NULL;
    pDataParams->pfRdFpga_ExchangeFPGACommand               = NULL;
    pDataParams->pfRdFpga_LoadFilterFile                    = NULL;
    pDataParams->pfRdFpga_ActivateTypeBMode                 = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_GetPortList(
    phbalReg_RdFpgaV6_DataParams_t * pDataParams,
    uint16_t wPortBufSize,
    uint8_t * pPortNames,
    uint16_t * pNumOfPorts
    )
{
    uint8_t abPortName[10] = "WALL E";
    uint16_t wLen;

    /* satisfy compiler */
    if (pDataParams);

    *pNumOfPorts = 0;

    if (wPortBufSize)
    {
        /* copy first instrument */
        wLen = (uint16_t)strlen( (char *) abPortName);
        memcpy(pPortNames, abPortName, wLen); /* PRQA S 3200 */
        *pNumOfPorts = 1;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_SetPort(
                                     phbalReg_RdFpgaV6_DataParams_t * pDataParams,
                                     uint8_t * pPortName
                                     )
{
    pDataParams->pDeviceName = pPortName;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_Exchange(
                                      phbalReg_RdFpgaV6_DataParams_t * pDataParams,
                                      uint16_t wOption,
                                      uint8_t * pTxBuffer,
                                      uint16_t wTxLength,
                                      uint16_t wRxBufSize,
                                      uint8_t * pRxBuffer,
                                      uint16_t * pRxLength
                                      )
{
    /* satisfy compiler */
    if (pRxBuffer || pRxLength);

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* if port is not opened we do have an interface error */
    if (pDataParams->bPortOpen == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* check if receive buffer is nonzero */
    if (wRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    /* to pass NUnit test parse for wRxBufSize = 100 or 64 and TxLength = 2 */
    if ((wRxBufSize == 100 || wRxBufSize == 64) && (wTxLength == 2 || wTxLength == 100))
    {
        /* check also first two bytes of TxBuffer for 0 */
        if (*pTxBuffer == 0)
        {
            pTxBuffer ++;
            if (*pTxBuffer == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_OpenPort(
                                      phbalReg_RdFpgaV6_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    phStatus_t statusUnloadFunc;
    uint8_t bI;
    uint8_t bForceSoftwareStack[PH_RDL_HAL_OBJECTS_LENGTH];

    /* check port state */
    if (pDataParams->bPortOpen == 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* check if any portname is set */
    if (pDataParams->pDeviceName)
    {
        if (*pDataParams->pDeviceName == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

    /* load the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdFpgaV6_Ex_LoadFunctions(pDataParams));

    /* satisfy funtion call, second parameter has no unfluence */
    for ( bI = 0; bI < PH_RDL_HAL_OBJECTS_LENGTH; bI++ )
    {
        bForceSoftwareStack[bI] = 0;
    }

    /* open port and initialize reader */
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Open) ( (uint16_t) PH_RDL_READER_ISO_LAB, (unsigned char *) bForceSoftwareStack));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));
        return statusTmp;
    }

    /* Init reader */
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Init) ( (uint16_t) PH_RDL_READER_CARD_TYPE_ISO14443A));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        /* close the port */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Close)()));

        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));
        return statusTmp;
    }

    /* disable output logging */
#ifdef _DEBUG
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_SetParam) ( (uint32_t) PH_RDL_FPGA_DEBUG_OUTPUT, 0xFFFF , PH_RDL_PARAM_SETMODE_NORMAL));
#else
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_SetParam) ( (uint32_t) PH_RDL_FPGA_DEBUG_OUTPUT, 0x0 , PH_RDL_PARAM_SETMODE_NORMAL));
#endif
    if(statusTmp != PH_ERR_SUCCESS)
    {
        /* close the port */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Close)()));

        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));
        return statusTmp;
    }

    /* play pattern */
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_PlayPattern) ( (uint8_t) 1));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        /* close the port */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Close)()));

        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));
        return statusTmp;
    }

    /* set joiner to BOTH */
    statusTmp = phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_SetTargetJoiner) (PHHAL_HW_RDFPGAV6_CMD_JOINER_BOTH));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        /* close the port */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Close)()));

        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusUnloadFunc, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));
        return statusTmp;
    }

    /* remeber that port is open */
    pDataParams->bPortOpen = 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_ClosePort(
                                       phbalReg_RdFpgaV6_DataParams_t * pDataParams
                                       )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    /* check port state */
    if (pDataParams->bPortOpen == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* deactivate modes and ignore errors */
    statusTmp = (phStatus_t) (pDataParams->pfRdFpga_DeactivateModes)();

    /* close the port */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pfRdFpga_Close)()));

    /* unload the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_RdFpgaV6_Ex_UnLoadFunctions(pDataParams));

    /* remeber that port is closed */
    pDataParams->bPortOpen = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_SetConfig(
                                       phbalReg_RdFpgaV6_DataParams_t * pDataParams,
                                       uint16_t wConfig,
                                       uint16_t wValue
                                       )
{
    switch (wConfig)
    {
    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        pDataParams->wTimeout = wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_RdFpgaV6_GetConfig(
                                       phbalReg_RdFpgaV6_DataParams_t * pDataParams,
                                       uint16_t wConfig,
                                       uint16_t * pValue
                                       )
{
    switch (wConfig)
    {
    case PHBAL_REG_RDFPGAV6_ADDITIONAL_INFO:
        *pValue = pDataParams->wAdditionalInfo;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);

    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        *pValue = pDataParams->wTimeout;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }
}

#endif /* NXPBUILD__PHBAL_REG_RDFPGAV6 */
