/** @file 
 *  @brief API header file for scr01_dll.dll
 */

#ifdef __cplusplus
extern "C"
{
#endif

#ifndef _EXPORT_
#define _EXPORT_ __declspec(dllexport)
#endif


#define SCR01_OK                              0x0000 /**< SUCCESS, Error free operation		              */

// define the general error codes

#define SCR01_E_INITIALIZE                    0x0001 /**< error during call of scr01.Initialize()		  */
#define SCR01_E_DEVICEINFO                    0x0002 /**< error during call of scr01.GetDeviceInfo		  */
#define SCR01_E_LOADFPGA                      0x0003 /**< error during call of scr01.LoadFPGA()			  */
#define SCR01_E_ALREADYINITIALIZED            0x0004 /**<												  */
#define SCR01_E_WRITEREGISTER                 0x0005 /**< error during call of scr01.scbWriteRegister()   */
#define SCR01_E_NOTINITIALIZED				  0x0006 /**<												  */
#define SCR01_E_READREGISTER                  0x0007 /**< error during call of scr01.scbReadRegister()    */
#define SCR01_E_DISPLAYWRONGPOSITION          0x0008 /**< wrong start address for write display           */
#define SCR01_E_DISPLAYWRONGLENGTH            0x0009 /**< wrong length for write display                  */
#define SCR01_E_WRITEINITDISPLAY              0x000a /**<                                                 */
#define SCR01_E_READEEPROMFILE                0x000b /**<                                                 */
#define SCR01_E_CORRUPTEEPROMFILE             0x000c /**<                                                 */
#define SCR01_E_WRONGEEPROMSIGNATURE          0x000d /**<                                                 */
#define SCR01_E_WRONGEEPROMCRC                0x000e /**<                                                 */
#define SCR01_E_DISPLAYCONTROL                0x000f /**<                                                 */
#define SCR01_E_SETSWITCH                     0x0010 /**<                                                 */
#define SCR01_E_CLEARSWITCH                   0x0011 /**<                                                 */
#define SCR01_E_SETTP1DIR                     0x0012 /**<                                                 */
#define SCR01_E_SETTP2DIR                     0x0013 /**<                                                 */
#define SCR01_E_SETTP1OUT                     0x0014 /**<                                                 */
#define SCR01_E_SETTP2OUT                     0x0015 /**<                                                 */
#define SCR01_E_CONTROLVOLTAGES               0x0016 /**<                                                 */
#define SCR01_E_CONTROLIOPORTS                0x0017 /**<                                                 */
#define SCR01_E_CONTROLCLOCKS                 0x0018 /**<                                                 */
#define SCR01_E_INITUART                      0x0019 /**<                                                 */
#define SCR01_E_SETBAUDRATE                   0x001A /**<                                                 */
#define SCR01_E_SETCWT                        0x001B /**<                                                 */
#define SCR01_E_SETBWT                        0x001C /**<                                                 */
#define SCR01_E_GETCOLDATR                    0x001D /**<                                                 */
#define SCR01_E_READBLOCK                     0x001E /**< error during call of scr01.scbReadBlock()       */
#define SCR01_E_WRITEBLOCK                    0x001F /**< error during call of scr01.scbWriteBlock()      */
#define SCR01_E_WRITEBUFFER2MEMORY            0x0020 /**<                                                 */
#define SCR01_E_RECEIVESTREAM                 0x0021 /**<                                                 */
#define SCR01_E_CLOSE                         0x0023 /**<                                                 */
#define SCR01_E_READFX2MEM                    0x0024 /**<                                                 */
#define SCR01_E_NOTSUPPORTEDHWVERSION         0x0025 /**<                                                 */
#define SCR01_E_SERIALEECRCERROR              0x0026 /**<                                                 */
#define SCR01_E_FATALERROR                    0x0027 /**<                                                 */
#define SCR01_E_PARAMERROR                    0x0028 /**<                                                 */
#define SCR01_E_GENERALHCIERROR               0x0029 /**<                                                 */
#define SCR01_E_HCPNOFRAMERECEIVED            0x002A /**<                                                 */
#define SCR01_E_SETANTITEAR                   0x002B /**<                                                 */
#define SCR01_E_RESETFPGA                     0x002C /**<                                                 */
#define SCR01_E_NOREADERFOUND                 0x002D /**<                                        		  */
#define SCR01_E_SERIALNOTFOUND                0x002E /**<                                        		  */
#define SCR01_E_OPENLOGFILEFAILED             0x002F /**<                                        		  */
#define SCR01_E_FEATURENOTSUPPORTED           0x0030 /**<                                        		  */
#define SCR01_E_TRIGIDXNOTSUPP                0x0031 /**<                                        		  */
#define SCR01_E_LOGFILENOTOPEN                0x0032 /**<                                        		  */
#define SCR01_E_USERLOGTOOLONG                0x0033 /**<                                        		  */
#define SCR01_E_SHAREDMEMORY                  0x0034 /**<                                        		  */
#define SCR01_E_SCR2DLLERROR                  0x0035 /**<                                        		  */       
#define SCR01_E_SCR2_WRAPPER_ERROR            0x0036 /**<                                        		  */       
#define SCR01_E_TRIGGER_PORT_NOT_AVAILABLE    0x0037 /**<                                        		  */       
    



// ************************** general functions ********************************



#define SCR01_GENERAL_PARAM_CLOSE_KEEP_POWER    1       

_EXPORT_ unsigned int scr01_general_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );





_EXPORT_ unsigned int scr01_enumerate_readers ( unsigned int  * reader_serials, 
										        unsigned int  * reader_count  );



/**
 * @brief Select the SCR01 reader instance by serial number
 * @param serial The serial number of requested SCR01 reader (without preceding zeroes)
 * @return SCR01_OK or error code
 *
 * This function selects a SCR01 instance. This function has to be called
 * before calling any scr01_initialize* function. If scr01_select_by_serial_number
 * is not used, then the SCR01 is used anyway, if only one instance is found, otherwise
 * it's not defined, which reader is used by scr01_initialize*. Therefore, if more than
 * one SCR01 is connected to the host, then use the function scr01_select_by_serial_number
 * to select the right SCR01 instance.
 */

_EXPORT_ unsigned int scr01_select_by_serial_number (unsigned int serial);   


/**
 * @brief Initialize the SCR01 reader hardware
 * @return SCR01_OK or error code
 *
 * This function initializes the SCR01 reader hardware, the default settings
 * for the VCC (3.0V) and clock (5.0 MHz) are used.
 */

_EXPORT_ unsigned int scr01_initialize    ();   

/**
 * @brief Initialize the SCR01 reader hardware
 * @param voltage VCC voltage level
 * @return SCR01_OK or error code
 *
 * This function initializes the SCR01 reader hardware, the default setting
 * for the clock (5.0 MHz) is used. The parameter "voltage" controls the VCC value.
 */

_EXPORT_ unsigned int scr01_initialize_v  (double voltage);

/**
 * @brief Initialize the SCR01 reader hardware
 * @param voltage VCC voltage level in Volt
 * @param frequency Clock frequency in MHz
 * @return SCR01_OK or error code
 *
 * This function initializes the SCR01 reader hardware. The parameter "voltage" 
 * controls the VCC value. The parameter "frequency" controls the clock
 * frequency.
 */

_EXPORT_ unsigned int scr01_initialize_vf (double voltage, double frequency);

_EXPORT_ unsigned int scr_enumerate_readers ( unsigned int  * reader_serial, 
										      unsigned int  * reader_count  );

_EXPORT_ unsigned int scr01_get_readers   (unsigned int  * reader_number, 
										   unsigned int  * reader_length,
										   unsigned char * reader_name);

_EXPORT_ unsigned int scr01_get_serial (unsigned int * serial);

_EXPORT_ unsigned int scr01_get_api_version (char * version_string);

_EXPORT_ unsigned int scr01_get_locked_devices (unsigned int * device_serials, unsigned char * device_count);
_EXPORT_ unsigned int scr01_unlock_device (unsigned int device_serial);


_EXPORT_ unsigned int scr01_set_user_display (char * user_string, unsigned char pos);

_EXPORT_ unsigned int scr01_close         ();

_EXPORT_ unsigned int scr01_start_logging (const char * filename);
_EXPORT_ unsigned int scr01_stop_logging  ();
_EXPORT_ unsigned int scr01_set_log_verbose_level (unsigned int verbose_level);

_EXPORT_ unsigned int scr01_add_user_log  (const char * log_entry);
_EXPORT_ unsigned int scr01_get_apdu_log_length (unsigned int * log_len);
_EXPORT_ unsigned int scr01_get_apdu_log_string (char * log_string);


_EXPORT_ void         scr01_hal_sleep     (unsigned int mseconds);

_EXPORT_ unsigned int scr01_set_trigger1  (unsigned int index);
_EXPORT_ unsigned int scr01_set_trigger2  (unsigned int index);

_EXPORT_ unsigned int scr01_set_icc_trigger  (double icc_value);
_EXPORT_ unsigned int scr01_set_vtp_trigger  (double vtp_value);




#define SCR01_TRIG_COMM_IF_I2C              0
#define SCR01_TRIG_COMM_IF_SPI              1
// to be extended ...

#define SCR01_TRIG_MODE_DISABLED            0
#define SCR01_TRIG_MODE_HIGH_LEVEL          1
// to be extended ...

_EXPORT_ unsigned int scr01_set_interface_trigger1  (unsigned int comm_interface, unsigned int trigger_mode);
_EXPORT_ unsigned int scr01_set_interface_trigger2  (unsigned int comm_interface, unsigned int trigger_mode);

_EXPORT_ unsigned int scr01_set_interface_trigger_timeout  (unsigned int timeout_ms);





// ************************** TCP/IP functions ********************************

#define SCR01_SOCKET_E_GENERAL_ERROR          0x9A01 /**<                                        		  */       
#define SCR01_SOCKET_E_NO_OPEN_SOCKET         0x9A02 /**<                                        		  */     
#define SCR01_SOCKET_E_NOT_AVAILABLE          0x9A03 /**<                                        		  */
#define SCR01_SOCKET_E_WRONG_PREAMBLE         0x9A04 /**<                                        		  */
#define SCR01_SOCKET_E_ACK_ERROR              0x9A05 /**<                                        		  */
#define SCR01_SOCKET_E_LEN_ERROR              0x9A06 /**<                                        		  */
#define SCR01_SOCKET_E_CMD_NOT_SUPPORTED      0x9A07 /**<                                        		  */
#define SCR01_SOCKET_E_EMPTY_FRAME            0x9A08 /**<                                        		  */
#define SCR01_SOCKET_E_TIMEOUT                0x9A09 /**<                                        		  */

_EXPORT_ unsigned int scr01_connect_iss    (const char * server_name, unsigned int server_name_length,
                                            const char * server_port, unsigned int server_port_length);   

_EXPORT_ unsigned int scr01_disconnect_iss ( );   




//////////////////////////////////////////////////////////////////////////////////
//
// Set IO configuration:
//

#define SECIF_NONE                  0	/**< Default, SWP is second interface      */
#define SECIF_DWP_IO3IO4            1	/**< DWP via IO3 and IO4                   */
#define SECIF_SCSPI_CLKIO1IO2IO3    2	/**< SCSPI                                 */
#define SECIF_SCI2C_IO2IO3          3	/**< I2C over IO2/IO3                      */
#define SECIF_S2C_IO2IO3            4	/**< Reserved, don't use it                */
#define SECIF_SCI2C_CLKIO1          5	/**< I2C over CLK/IO1                      */
#define SECIF_SPI_CLKIO1IO2IO3IO4   6	/**< T1 over SPI                           */
#define SECIF_DWP_TP1TP2            7	/**< DWP via TP1 and TP2                   */
#define SECIF_RAWI2C_IO3IO4         8	/**< RAW I2C via IO3 and IO4               */
#define SECIF_RAWI2C_CLKIO1         9	/**< RAW I2C via CLK and IO1               */
#define SECIF_T1I2C_IO2IO3         12	/**< T1 over I2C via IO2/IO3               */
#define SECIF_T1I2C_IO3IO4         13	/**< T1 over I2C via IO3/IO4               */
#define SECIF_SPI_CLKIO1IO2TP1TP2  14	/**< T1 over SPI, VCC=1.8V only !          */
#define SECIF_RAWI2C_IO2IO3        15	/**< RAW I2C via IO2 and IO3               */
#define SECIF_SPI_TP1TP2SWPIO2     16	/**< T1 over SPI, VCC=1.8V only !          */
#define SECIF_SPI_IO2IO3CLKIO4     17	/**< T1 over SPI                           */

#define RST_MODE_ISO7816            0	/**< Default, controlled by UART           */
#define RST_MODE_HIGH               1	/**< Driven active high                    */
#define RST_MODE_LOW                2	/**< Driven active low	                   */

#define CLK_MODE_ISO7816            0	/**< Default, controlled by UART	       */
#define CLK_MODE_HIGH               1	/**< Driven active high	                   */
#define CLK_MODE_LOW                2	/**< Driven active low	                   */
#define CLK_MODE_NONE               3	/**< Not driven, tristate	               */

/**
 * @brief Set a certain IO configuration
 * @param second_if_mode Selects the second IO interface
 *   - Select one of the following :
 *       - #SECIF_NONE             
 *       - #SECIF_DWP_IO3IO4         
 *       - #SECIF_SPI_CLKIO1IO2IO3   
 *       - #SECIF_I2C_IO2IO3         
 *       - #SECIF_S2C_IO2IO3         
 *       - #SECIF_I2C_CLKIO1         
 *       - #SECIF_SPI_CLKIO1IO2IO3IO4
 *       - #SECIF_DWP_TP1TP2         
 *       - #SECIF_RAWI2C_IO3IO4         
 *       - #SECIF_RAWI2C_CLKIO1         
 *       - #SECIF_T1I2C_IO2IO3         
 *       - #SECIF_T1I2C_IO3IO4         
 *       - #SECIF_T1I2C_TP1TP2         
 * @param rst_mode Mode of RST pin
 *   - Select one of the following :
 *       - #RST_MODE_ISO7816       
 *       - #RST_MODE_HIGH            
 *       - #RST_MODE_LOW             
 * @param clk_mode Mode of CLK pin
 *   - Select one of the following :
 *       - #CLK_MODE_ISO7816      
 *       - #CLK_MODE_HIGH           
 *       - #CLK_MODE_LOW            
 *       - #CLK_MODE_NONE           
 * @return 
 * - #SCR01_OK 
 * - Other error code
 *
 * This function selects the second interface (in addition to the always selected
 * ISO7816-UART) and controls the behaviour of the RST and CLK pins.
 */

_EXPORT_ unsigned int scr01_set_io_configuration ( unsigned char second_if_mode,
												   unsigned char rst_mode,
												   unsigned char clk_mode );




//////////////////////////////////////////////////////////////////////////////////
//
// Set VCC voltage
//

#define SCR01_E_WRONG_VCC_VALUE                 0xE001	///< VCC parameter is out of range (0.0 ... 6.0 V)
 

_EXPORT_ unsigned int scr01_set_vcc (double voltage); 

_EXPORT_ unsigned int scr01_get_vcc (double * voltage);

_EXPORT_ unsigned int scr01_poweron_reset (); 


//////////////////////////////////////////////////////////////////////////////////
//
// Control trapdoor
//

#define SCR01_TRAPDOOR_OFF					    0
#define SCR01_TRAPDOOR_ON					    1

#define SCR01_TRAPDOOR_TP1						0
#define SCR01_TRAPDOOR_TP2						1

#define SCR01_TRAPDOOR_PERMANENT				0
#define SCR01_TRAPDOOR_CONDITIONAL				1

_EXPORT_ unsigned int scr01_set_trapdoor  (unsigned char trapdoor_value,	// SCR01_TRAPDOOR_ON, SCR01_TRAPDOOR_OFF
										   unsigned char trapdoor_select,	// SCR01_TRAPDOOR_TP1, SCR01_TRAPDOOR_TP2 
										   unsigned char trapdoor_mode );	// SCR01_TRAPDOOR_PERMANENT, SCR01_TRAPDOOR_CONDITIONAL



//////////////////////////////////////////////////////////////////////////////////
//
// Measurements
//

typedef enum _SCR01_MEASURE {

	SCR01_MEASURE_NONE=0,
    SCR01_MEASURE_VTP1,
    SCR01_MEASURE_VTP2

} SCR01_MEASURE;


_EXPORT_ unsigned int scr01_ctrl_measurement( SCR01_MEASURE measure_select);

_EXPORT_ unsigned int scr01_get_vtp (double * voltage);   

_EXPORT_ unsigned int scr01_get_icc (double * current);   






//////////////////////////////////////////////////////////////////////////////////
//
// Control IO ports
//

#define SCR01_E_PORT_NOT_ACCESSIBLE             0xE003				// port is not accessible

#define SCR01_IOMODE_OUT0						0					// output mode, drive level 0
#define SCR01_IOMODE_OUT1						1					// output mode, drive level 1
#define SCR01_IOMODE_IN 						2					// input mode , tristate (not available for CLK and RST pins on SCR01)

#define SCR01_IOSEL_IO1							1					// select IO1
#define SCR01_IOSEL_IO2							2					// select IO2
#define SCR01_IOSEL_IO3							3					// select IO3
#define SCR01_IOSEL_IO4							4					// select IO4
#define SCR01_IOSEL_CLK							5					// select CLK 
#define SCR01_IOSEL_RST							6					// select RST 
#define SCR01_IOSEL_EXT0					    7					// select extension port 0 
#define SCR01_IOSEL_EXT1					    8					// select extension port 1 
#define SCR01_IOSEL_EXT2					    9					// select extension port 2 
#define SCR01_IOSEL_EXT3					    10					// select extension port 3 (different pin number on SCR2 !)
#define SCR01_IOSEL_EXT4					    11					// select extension port 4 (different pin number on SCR2 !) 
#define SCR01_IOSEL_EXT5					    12					// select extension port 5 
#define SCR01_IOSEL_TP1 					    13					// select TP1 
#define SCR01_IOSEL_TP2 					    14					// select TP2 

_EXPORT_ unsigned int scr01_set_io_port (unsigned char iosel,       // SCR01_IOSEL_IO1, SCR01_IOSEL_IO2, ... 
                                         unsigned char iomode);	  	// SCR01_IOMODE_OUT0, SCR01_IOMODE_OUT1, SCR01_IOMODE_IN
										
_EXPORT_ unsigned int scr01_get_io_port (unsigned char   iosel,     // SCR01_IOSEL_IO1, SCR01_IOSEL_IO2, ... 
                                         unsigned char * ioval);	// pointer to variable
										




//////////////////////////////////////////////////////////////////////////////////
//
// Control UART clock frequency
//

#define SCR01_E_WRONG_UARTFREQUENCY_VALUE       0xE002	// out of range (0.0 ... 20.0 MHz)

_EXPORT_ unsigned int scr01_set_uart_frequency (double frequency); 





//////////////////////////////////////////////////////////////////////////////////
//
// Control UART IO pin
//

#define SCR01_UART_IO_NONE                0
#define SCR01_UART_IO_IO1                 1
#define SCR01_UART_IO_IO2                 2
#define SCR01_UART_IO_IO3                 3
#define SCR01_UART_IO_IO4                 4

_EXPORT_ unsigned int scr01_set_uart_io_pin (unsigned char io_pin_sel); 





//////////////////////////////////////////////////////////////////////////////////
//
// ISO7816 low level functions
//

#define SCR01_CHM_DISABLED      0
#define SCR01_CHM_P6X           1
#define SCR01_CHM_P71           2

_EXPORT_ unsigned int scr01_iso7816_set_chm (unsigned char chm_mode); 


_EXPORT_ unsigned int scr01_get_cold_atr (unsigned char * atr, unsigned int * atrlength);
_EXPORT_ unsigned int scr01_get_warm_atr (unsigned char * atr, unsigned int * atrlength);

_EXPORT_ unsigned int scr01_get_warm_atr_delayed (unsigned int    delay_ms,
                                                  unsigned int    clk_count,
                                                  unsigned char * atr, 
                                                  unsigned int  * atrlength);


#define SCR01_RAW_EXCHANGE_TIMEOUT              0xE081
#define SCR01_RAW_EXCHANGE_PARITY_ERROR         0xE082
#define SCR01_RAW_EXCHANGE_RECEIVE_ERROR        0xE083
#define SCR01_RAW_EXCHANGE_TEARERROR            0xE085

_EXPORT_ unsigned int scr01_raw_send               (unsigned char * databuffer, unsigned int   datalength);
_EXPORT_ unsigned int scr01_raw_receive            (unsigned char * databuffer, unsigned int * datalength, double timeout_us);

_EXPORT_ unsigned int scr01_raw_exchange           (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr01_raw_exchange_poweroff  (unsigned char * databuffer, unsigned int * datalength, double timeout_us, unsigned int power_off_time_us);
_EXPORT_ unsigned int scr01_raw_exchange_warmreset (unsigned char * databuffer, unsigned int * datalength, double timeout_us, unsigned int warm_reset_time_us);

_EXPORT_ unsigned int scr01_raw_exchange_t0        (unsigned char * databuffer, unsigned int * datalength, double timeout_us);


#define SCR01_RAW_PULSE_ERROR                   0xE086

_EXPORT_ unsigned int scr01_raw_set_resetpulse_length (unsigned int   clk_count);
_EXPORT_ unsigned int scr01_raw_get_resetpulse_length (unsigned int * clk_count);
_EXPORT_ unsigned int scr01_raw_resetpulse ();


#define SCR01_RAW_NOT_SUPPORTED_FIDI            0xE084

_EXPORT_ unsigned int scr01_raw_setbaudrate (unsigned char FI, unsigned char DI); 


//////////////////////////////////////////////////////////////////////////////////
//
// FTB P73
//

#define SCR01_FTB_INIT_FAILED                   0xE086
#define SCR01_FTB_EXCHANGE_TIMEOUT              0xE087
#define SCR01_FTB_EXCHANGE_PARITYERROR          0xE088
#define SCR01_FTB_EXCHANGE_RECEIVEERROR         0xE089

_EXPORT_ unsigned int scr01_ftb_init        (unsigned int ftb_password, unsigned char ftb_option);
_EXPORT_ unsigned int scr01_ftb_sequence    (unsigned int ftb_password, unsigned char ftb_option);
_EXPORT_ unsigned int scr01_ftb_exchange    (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr01_ftb_send        (unsigned char * databuffer, unsigned int   datalength);
_EXPORT_ unsigned int scr01_ftb_set_seq_start_time  (unsigned int sequence_start_time_us);


// FTB parameter selection
#define SCR01_FTB_SEQ_START_TIME_US            0
#define SCR01_FTB_MODE                         1

// FTB parameter values
#define SCR01_FTB_MODE_STANDARD                0
#define SCR01_FTB_MODE_EOS_GOANNA              1
#define SCR01_FTB_MODE_EOS_FPGA                2

_EXPORT_ unsigned int scr01_ftb_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );




//////////////////////////////////////////////////////////////////////////////////
//
// ISO7816 higher level functions

//
#define ISO7816_MSG_OK                          0x0000


#define ISO7816_MSG_PROTOCOL_NOT_SUPPORTED		0x8001
#define ISO7816_MSG_ATR_CORRUPTED				0x8002
#define ISO7816_MSG_PPS_ERROR					0x8004
#define ISO7816_MSG_TA1_VALUE_NOT_SUPPORTED		0x8005
#define ISO7816_MSG_IFSD_ERROR					0x8006
#define ISO7816_MSG_TIMEOUT  					0x8007
#define ISO7816_MSG_PARITY_ERROR                0x8008
#define ISO7816_MSG_IFS_ERROR					0x8009
#define ISO7816_MSG_STOPBIT_ERROR				0x800a
#define ISO7816_MSG_T0_PARTLY   				0x800b
#define ISO7816_MSG_TA2_ERROR   				0x800c
#define ISO7816_MSG_TA2_MISMATCH                0x800d
#define ISO7816_MSG_TA2_BIT5SET                 0x800e
#define ISO7816_MSG_ATR_TCK_MISSING             0x800f
#define ISO7816_MSG_ATR_TCK_WRONG               0x8010
#define ISO7816_MSG_ATR_HIST_BYTES_ERROR        0x8011
#define ISO7816_MSG_ATR_WRONG_TS                0x8012
#define ISO7816_MSG_ATR_CODING_ERROR            0x8013
#define ISO7816_MSG_ATR_WRONG_LENGTH            0x8014
#define ISO7816_MSG_TEAR_ERROR                  0x8015

#define ISO7816_FORCE_T0_PROTOCOL               0x0000
#define ISO7816_FORCE_T1_PROTOCOL               0x0001
#define ISO7816_AUTO_PROTOCOL                   0x0002

_EXPORT_ unsigned int scr01_iso7816_activate      (unsigned char protocol); // behaves as scr01_iso7816_activate_cold
_EXPORT_ unsigned int scr01_iso7816_activate_cold (unsigned char protocol);
_EXPORT_ unsigned int scr01_iso7816_activate_warm (unsigned char protocol);

_EXPORT_ unsigned int scr01_iso7816_deactivate ();

#define ISO7816_MSG_NOATR_ERROR                 0x8081

_EXPORT_ unsigned int scr01_iso7816_getatr   (unsigned char * atr_ptr, unsigned int * atr_length);


#define ISO7816_MSG_APDU_TIMEOUT				0x8101

#define ISO7816_MSG_APDU_T0_FRAME_TO_SHORT      0x8111
#define ISO7816_MSG_APDU_T0_NO_DATA_AVAILABLE   0x8112
#define ISO7816_MSG_APDU_T0_FRAME_LENGTH_ERROR  0x8113
#define ISO7816_MSG_APDU_T0_FATAL_ERROR         0x8114
#define ISO7816_MSG_APDU_T0_UNEXPECTED_SW1      0x8115
#define ISO7816_MSG_APDU_T0_WRONG_CLA_BYTE      0x8116
#define ISO7816_MSG_APDU_T0_WRONG_INS_BYTE      0x8117
#define ISO7816_MSG_APDU_T0_NOT_SUPPORTED_CASE  0x8118

#define ISO7816_MSG_APDU_T1_FATAL_ERROR         0x8121
#define ISO7816_MSG_APDU_T1_FRAME_LENGTH_ERROR  0x8122
#define ISO7816_MSG_APDU_T1_CRC_NOT_SUPPORTED   0x8123
#define ISO7816_MSG_APDU_T1_LRC_ERROR           0x8124
#define ISO7816_MSG_APDU_T1_TOO_MANY_R_FRAMES   0x8125


_EXPORT_ unsigned int scr01_iso7816_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_iso7816_exchange_poweroff  (unsigned char * databuffer, unsigned int * length, unsigned int power_off_time_us);
_EXPORT_ unsigned int scr01_iso7816_exchange_warmreset (unsigned char * databuffer, unsigned int * length, unsigned int warmreset_time_us);

_EXPORT_ unsigned int scr01_iso7816_generic_sframe_exchange (unsigned char command, unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_iso7816_get_execution_time_us (unsigned int * execution_time);

_EXPORT_ unsigned int scr01_iso7816_modify_timeout ( unsigned int timeoutvalue_us );


#define ISO7816_PROTOCOL_GETRESPONSE_CASE4          1
#define ISO7816_PROTOCOL_ATR_WAIT_CLOCKS            2
#define ISO7816_PROTOCOL_CLOCKSTOP_COUNT1           3
#define ISO7816_PROTOCOL_CLOCKSTOP_COUNT2           4
#define ISO7816_PROTOCOL_T1_DISABLE_ERROR_HANDLING  5
#define ISO7816_PROTOCOL_COLD_RESET_TIME            6

_EXPORT_ unsigned int scr01_iso7816_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );


#define ISO7816_PROTOCOL_DEFAULT_SETTINGS               0x00000000
#define ISO7816_PROTOCOL_DISABLE_GETRESPONSE_CASE4      0x00000001

_EXPORT_ unsigned int scr01_iso7816_modify_protocol ( unsigned int protocol_control );


#define SCR01_ISO7816_CLOCK_AUTO                 0      // default, clock controlled by UART
#define SCR01_ISO7816_CLOCK_FORCE_0              1      // force CLK to 0
#define SCR01_ISO7816_CLOCK_FORCE_1              2      // force CLK to 1

_EXPORT_ unsigned int scr01_iso7816_clock_mode (unsigned int clock_mode);




//////////////////////////////////////////////////////////////////////////////////
//
//  SWP
//
//





_EXPORT_ unsigned int scr01_swp_set_timeout ( unsigned int timeout_ms );

_EXPORT_ unsigned int scr01_swp_set_T_S1_HIGH    ( unsigned int   delay_us );
_EXPORT_ unsigned int scr01_swp_get_T_S1_HIGH    ( unsigned int * delay_us );

_EXPORT_ unsigned int scr01_swp_set_T_S2_ACT_RES ( unsigned int   delay_us );
_EXPORT_ unsigned int scr01_swp_get_T_S2_ACT_RES ( unsigned int * delay_us );


#define SCR01_SWPMODE_SWP                0x01 
#define SCR01_SWPMODE_DWP_IO34           0x02 
#define SCR01_SWPMODE_DWP_TP12           0x03 

#define SCR01_SWP_ACTIVATION_MODE_BUS    0
#define SCR01_SWP_ACTIVATION_MODE_ACT    1
#define SCR01_SWP_ACTIVATION_MODE_HCI    2

#define SCR01_SWP_POWER_CYCLE_DISABLED   0
#define SCR01_SWP_POWER_CYCLE_ENABLED    1

#define SCR01_SWP_POWER_MODE_ON          0
#define SCR01_SWP_POWER_MODE_OFF         1


_EXPORT_ unsigned int scr01_swp_activate   ( double frequency, unsigned char activationmode, unsigned char powercycle );
_EXPORT_ unsigned int scr01_swp_deactivate ( unsigned char powermode );

#define SCR01_SWP_CLEAR_CONTEXT          0
#define SCR01_SWP_KEEP_CONTEXT           1

#define SCR01_SWP_M_NO_FRAME_RECEIVED    0x80b0

_EXPORT_ unsigned int scr01_swp_reactivate ( unsigned char activationmode, unsigned char clearcontext );

_EXPORT_ unsigned int scr01_swp_get_actsynctime ( unsigned int * actsynctime_us );


#define SCR01_SWP_E_NOATR_AVAILABLE      0x80A1
#define SCR01_SWP_E_NOATS_AVAILABLE      0x80A2
#define SCR01_SWP_E_ATR_CORRUPTED        0x80A3

_EXPORT_ unsigned int scr01_swp_getatr     (unsigned char * atr_ptr, unsigned int * atr_length);
_EXPORT_ unsigned int scr01_swp_getats     (unsigned char * ats_ptr, unsigned int * ats_length);

_EXPORT_ unsigned int scr01_swp_get_sessionid  (unsigned char * session_id);


struct _Type_A_card_RF_gate_registry {

    unsigned char mode;
    unsigned char uid_reg[10];
    unsigned char uid_len;
    unsigned char sak;
    unsigned char atqa[2];
             char application_data[32];
    unsigned char application_data_len;
    unsigned char fwi_sfgi;
    unsigned char cid_support;
    unsigned char clt_support;
    unsigned char datarate_max[3];

};

_EXPORT_ unsigned int scr01_swp_get_type_a_card_rf_registry (struct _Type_A_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_set_type_a_card_rf_registry (struct _Type_A_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_clear_type_a_card_rf_registry ();



struct _Type_B_card_RF_gate_registry {

    unsigned char mode;
    unsigned char pupi_reg[10];
    unsigned char pupi_len;
    unsigned char afi;
    unsigned char atqb[4];
    unsigned char higher_layer_response[32];
    unsigned char higher_layer_response_len;
    unsigned char datarate_max[3];

};

_EXPORT_ unsigned int scr01_swp_get_type_b_card_rf_registry (struct _Type_B_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_set_type_b_card_rf_registry (struct _Type_B_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_clear_type_b_card_rf_registry ();




struct _Type_F_card_RF_gate_registry {

	unsigned char mode;
	unsigned char speed_cap;
	unsigned char clt_support;

};

_EXPORT_ unsigned int scr01_swp_get_type_f_card_rf_registry (struct _Type_F_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_set_type_f_card_rf_registry (struct _Type_F_card_RF_gate_registry * registry);
_EXPORT_ unsigned int scr01_swp_clear_type_f_card_rf_registry ();




#define SCR01_SWP_GATE_CARD_EMULATION          0    // for historical reasons, means Type A
#define SCR01_SWP_GATE_CARD_EMULATION_TYPE_A   0
#define SCR01_SWP_GATE_WIRED_MODE              1
#define SCR01_SWP_GATE_CARD_EMULATION_TYPE_B   2
#define SCR01_SWP_GATE_CARD_EMULATION_TYPE_F   3
#define SCR01_SWP_GATE_APDU                    4


#define SCR01_SWP_EVENT_FIELD_ON                    1
#define SCR01_SWP_EVENT_CARD_DEACTIVATED            2
#define SCR01_SWP_EVENT_CARD_ACTIVATED              3
#define SCR01_SWP_EVENT_FIELD_OFF                   4
#define SCR01_SWP_EVENT_WTX                         5
#define SCR01_SWP_EVENT_R_APDU                      6
#define SCR01_SWP_EVENT_ATR                         7
#define SCR01_SWP_EVENT_RF_TECHNOLOGIES_DETECTED    8
#define SCR01_SWP_EVENT_START_EXPRESS_MODE          9
#define SCR01_SWP_EVENT_SOFT_RESET                  10
#define SCR01_SWP_EVENT_SOFT_RESET_RESP             11
#define SCR01_SWP_EVENT_SLOW_EMM_ACT_NTF            12

#define SCR01_SWP_E_EVENT_NOT_SUPPORTED        0x9182

_EXPORT_ unsigned int scr01_swp_send_event   (unsigned char gate, unsigned char event_type);

_EXPORT_ unsigned int scr01_swp_send_event_2 (unsigned char pipe_id, unsigned char event_type, unsigned char emm_active);



#define SCR01_SWP_E_EVENT_DATA_TOO_LONG        0x9189
#define SCR01_SWP_E_EVENT_DATA_TOO_SHORT       0x918A

_EXPORT_ unsigned int scr01_swp_send_event_with_parameter   (unsigned char   gate, 
                                                             unsigned char   event_type, 
                                                             unsigned char * event_parameter,
                                                             unsigned int    event_parameter_length);

_EXPORT_ unsigned int scr01_swp_send_event_with_parameter_2 (unsigned char   pipe_id, 
                                                             unsigned char   event_type, 
                                                             unsigned char * event_parameter,
                                                             unsigned int    event_parameter_length, 
															 unsigned char            emm_active);




#define max_event_data_count 200

struct _Event_Data {

    unsigned char pipe_id;
    unsigned char gate_id;
    unsigned char event_id;

    unsigned int  event_data_length;
    unsigned char event_data[4096];

    unsigned int  event_raw_length;
    unsigned char event_raw[4100];
    
};


#define SCR01_SWP_E_TOO_MANY_EVENTS            0x9183
#define SCR01_SWP_E_WRONG_EVENT_INDEX          0x9184

_EXPORT_ unsigned int scr01_swp_get_event_count (unsigned int * event_count);
_EXPORT_ unsigned int scr01_swp_get_event_data  (unsigned int   event_index, struct _Event_Data * event_data);
_EXPORT_ unsigned int scr01_swp_clear_events    ();


#define SCR01_SWP_E_NO_EVENT_AVAILABLE         0x9187

_EXPORT_ unsigned int scr01_swp_get_event_count_2 (unsigned int * event_count);
_EXPORT_ unsigned int scr01_swp_get_event_data_2  (struct _Event_Data * event_data);



_EXPORT_ unsigned int scr01_swp_set_rf_error_indicator (unsigned char rf_error_indicator );

#define SCR01_SWP_E_HCP_ERROR_0		           0x9185
#define SCR01_SWP_E_HCP_ERROR_1		           0x9186

_EXPORT_ unsigned int scr01_swp_apdu_exchange          (unsigned char * databuffer, unsigned int * length, unsigned char gate);
_EXPORT_ unsigned int scr01_swp_apdu_exchange_poweroff (unsigned char * databuffer, unsigned int * length, unsigned char gate, unsigned int power_off_time_us);


_EXPORT_ unsigned int scr01_swp_native_exchange ( unsigned char * databuffer, unsigned int * length, unsigned char gate );


#define SCR01_SWP_E_NO_TIME_AVAILABLE          0x9181

_EXPORT_ unsigned int scr01_swp_get_execution_time_us (unsigned int * execution_time);
_EXPORT_ unsigned int scr01_swp_get_apdu_time_us (unsigned int * apdu_time);



// general handling
#define SCR01_SWP_E_PIPE_COULD_NOT_CREATE      0x9101
#define SCR01_SWP_E_PIPE_WRONG_PIPE_ID         0x9102
#define SCR01_SWP_E_GATE_NOT_SUPPORTED         0x9103
#define SCR01_SWP_E_PIPE_COULD_NOT_DELETE      0x9104
#define SCR01_SWP_E_PIPE_COULD_NOT_OPEN        0x9105
#define SCR01_SWP_E_PIPE_COULD_NOT_CLOSE       0x9106
#define SCR01_SWP_E_ANY_E_NOK                  0x9107
#define SCR01_SWP_E_NO_ATR_AVAILABLE           0x9108
#define SCR01_SWP_E_GENERAL_ERROR              0x9109
#define SCR01_SWP_E_UNEXPECTED_FRAME           0x910A
#define SCR01_SWP_E_FRAME_LENGTH_ERROR         0x910B
#define SCR01_SWP_E_EVENT_DATA_TOO_LARGE       0x910C
#define SCR01_SWP_E_SWPRECVLENERROR            0x910D 
#define SCR01_SWP_E_SWPRECVCRCERROR            0x910E 
#define SCR01_SWP_E_SWPACT0FAILED              0x910F 
#define SCR01_SWP_E_SWPSENDSTREAM              0x9110 
#define SCR01_SWP_E_SWPGATENOTOPEN             0x9111 
#define SCR01_SWP_E_SWPPIPENOTOPEN             0x9112 
#define SCR01_SWP_M_SWPRECVNOFRAME             0x9113 
#define SCR01_SWP_M_SWPRECVOVERFLOW            0x9114
#define SCR01_SWP_E_MAXRETRYREACHED            0x9115
#define SCR01_SWP_E_SWPNOACKNOWLEDGE           0x9116
#define SCR01_SWP_E_FELICA_CRC_ERROR           0x9117
#define SCR01_SWP_E_CLT_GOTO_INIT              0x9118
#define SCR01_SWP_E_CLT_GOTO_HALT              0x9119
#define SCR01_SWP_E_CLT_ADMIN_FIELD            0x911A
#define SCR01_SWP_E_CLT_WRONG_ALIGNMENT        0x911B
#define SCR01_SWP_E_FELICA_NO_DATA_FIELD       0x911C

_EXPORT_ unsigned int scr01_swp_adm_notify_pipe_created (unsigned char gate_id, unsigned char pipe_id);
_EXPORT_ unsigned int scr01_swp_adm_notify_pipe_deleted (unsigned char pipe_id);
_EXPORT_ unsigned int scr01_swp_any_open_pipe           (unsigned char pipe_id);
_EXPORT_ unsigned int scr01_swp_any_close_pipe          (unsigned char pipe_id);





// loop back gate
#define SCR01_SWP_E_LOOPBACK_COULD_NOT_OPEN    0x9001
#define SCR01_SWP_E_LOOPBACK_WRONG_PIPE_ID     0x9002
#define SCR01_SWP_E_LOOPBACK_PIPE_NOT_OPEN     0x9003
#define SCR01_SWP_E_LOOPBACK_PIPE_ALREADY_OPEN 0x9004

_EXPORT_ unsigned int scr01_swp_open_loop_back          (unsigned char pipe_id);
_EXPORT_ unsigned int scr01_swp_close_loop_back         ();
_EXPORT_ unsigned int scr01_swp_exchange_loop_back      (unsigned char * databuffer, unsigned int * length);


// for debugging on lower layers

_EXPORT_ unsigned int scr01_swp_get_hci_seqnumbers (unsigned char * NR, unsigned char * NS);
_EXPORT_ unsigned int scr01_swp_set_hci_seqnumbers (unsigned char   NR, unsigned char   NS);


#define SCR01_SWP_E_RAW_FRAME_TOO_LONG         0x9201
#define SCR01_SWP_E_RAW_FRAME_TOO_SHORT        0x9202

_EXPORT_ unsigned int scr01_swp_send_raw_frame     (unsigned char * databuffer, unsigned int   length);
_EXPORT_ unsigned int scr01_swp_read_raw_frame     (unsigned char * databuffer, unsigned int * length);


#define SCR01_SWP_AUTO_ACK_ENABLED    0
#define SCR01_SWP_AUTO_ACK_DISABLED   1

_EXPORT_ unsigned int scr01_swp_control_auto_ack   (unsigned char auto_ack_mode);



#define SCR01_SWP_RECV_ACK_IGNORED    0
#define SCR01_SWP_RECV_ACK_DELIVERED  1

_EXPORT_ unsigned int scr01_swp_control_recv_ack   (unsigned char recv_ack_mode);






//////////////////////////////////////////////////////////////////////////////////
//
// SPI5, "5 port SPI", "T1 over SPI"
//

#define SCR01_SPI5_E_FRAMELENGTHERROR				0x9301
#define SCR01_SPI5_E_TIMEOUT						0x9302
#define SCR01_SPI5_E_GENERALERROR					0x9303
#define SCR01_SPI5_E_EDCERROR						0x9304
#define SCR01_SPI5_E_FATALERROR 			        0x9305
#define SCR01_SPI5_E_TOO_MANY_R_FRAMES              0x9306
#define SCR01_SPI5_E_NO_TIME_AVAILABLE              0x9307
#define SCR01_SPI5_E_TOO_MANY_REPEATS               0x9308 
#define SCR01_SPI5_E_NADERROR                       0x9309
#define SCR01_SPI5_E_SFRAMEERROR                    0x930A
#define SCR01_SPI5_E_ANTITEAR						0x930B
#define SCR01_SPI5_E_RECEIVESTREAM                  0x930C
#define SCR01_SPI5_E_PROTOCOLERROR                  0x930D
#define SCR01_SPI5_E_CRCERROR                       0x930E
#define SCR01_SPI5_E_LRCERROR                       0x930F
#define SCR01_SPI5_E_NOTSUPPORTED                   0x9310
#define SCR01_SPI5_E_OS_REMOVED                     0x9311


#define SCR01_SPI5_PROTOCOLMODE_NORMAL_5A           2           // NAD(TX)=0x5A, continuos polling
#define SCR01_SPI5_PROTOCOLMODE_P73_5A              3           // NAD(TX)=0x5A, interval polling (every 1 msec)
#define SCR01_SPI5_PROTOCOLMODE_READY_HIGH_5A       4           // NAD(TX)=0x5A, wait on ready line high
#define SCR01_SPI5_PROTOCOLMODE_READY_LOW_5A        5           // NAD(TX)=0x5A, wait on ready line low

_EXPORT_ unsigned int scr01_spi5_set_protocol_mode (unsigned char mode);


#define SCR01_SPI5_LOGIF_SPI                        0
#define SCR01_SPI5_LOGIF_VIRTUALISO                 1

_EXPORT_ unsigned int scr01_spi5_set_logicalinterface (unsigned char logicalinterface);

_EXPORT_ unsigned int scr01_spi5_set_ifsc_ifsd (unsigned int ifs);

_EXPORT_ unsigned int scr01_spi5_reset_protocol ();

#define SCR01_SPI5_BLOCKNUMBER_0    0
#define SCR01_SPI5_BLOCKNUMBER_1    1
#define SCR01_SPI5_BLOCKNUMBER_AUTO 255

_EXPORT_ unsigned int scr01_spi5_apdu_exchange (unsigned char * databuffer, unsigned int * length, unsigned char blocknumber);

_EXPORT_ unsigned int scr01_spi5_apdu_exchange_poweroff (unsigned char * databuffer, unsigned int * length, unsigned char blocknumber, unsigned int power_off_time_us);

_EXPORT_ unsigned int scr01_spi5_raw_exchange  (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_spi5_raw_send  (unsigned char * databuffer, unsigned int length);
_EXPORT_ unsigned int scr01_spi5_raw_read  (unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr01_spi5_set_bitrate (double bitrate);

_EXPORT_ unsigned int scr01_spi5_set_start_guard_time (double guard_clocks);
_EXPORT_ unsigned int scr01_spi5_set_end_guard_time (double guard_clocks);

_EXPORT_ unsigned int scr01_spi5_get_execution_time_us (unsigned int * execution_time);

_EXPORT_ unsigned int scr01_spi5_get_apdu_time_us (unsigned int * apdu_time);


#define SCR01_SPI5_SFRAME_INTERFACERESET            0
#define SCR01_SPI5_SFRAME_ENDOFSESSION              1
#define SCR01_SPI5_SFRAME_GETATR                    2
#define SCR01_SPI5_SFRAME_GETPOWERSTATE             3
#define SCR01_SPI5_SFRAME_GETPROTOCOLVERSION        4

_EXPORT_ unsigned int scr01_spi5_sframe_exchange (unsigned char sframetype, unsigned char * infbuffer, unsigned int * inflength);

_EXPORT_ unsigned int scr01_spi5_generic_sframe_exchange (unsigned char command, unsigned char * infbuffer, unsigned int * inflength);


#define SCR01_SPI5_PROTOCOL_T1_DISABLE_ERROR_HANDLING  1
#define SCR01_SPI5_PROTOCOL_NAD_LRC                    2
#define SCR01_SPI5_PROTOCOL_POLLING_TIME_MS            3
#define SCR01_SPI5_PROTOCOL_USE_CRC                    4
#define SCR01_SPI5_PROTOCOL_SEND_NAD_VALUE             5
#define SCR01_SPI5_PROTOCOL_RECV_NAD_VALUE             6
#define SCR01_SPI5_PROTOCOL_GP_MODE                    7
#define SCR01_GPSPI_PROTOCOL_IFSC                      8
#define SCR01_SPI5_PROTOCOL_BWT_US                     9
#define SCR01_GPSPI_PROTOCOL_POLLING_TIME_US           10

_EXPORT_ unsigned int scr01_spi5_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );

_EXPORT_ unsigned int scr01_spi5_get_parameter ( unsigned int parameter_select, unsigned int * parameter_value );




/////////////////////////////////////////////////////////////////
//
// T1I2C, "T1 over I2C"
//


#define SCR01_T1I2C_E_FRAMELENGTHERROR				0x9701
#define SCR01_T1I2C_E_TIMEOUT						0x9702
#define SCR01_T1I2C_E_GENERALERROR					0x9703
#define SCR01_T1I2C_E_LRCERROR						0x9704
#define SCR01_T1I2C_E_FATALERROR 			        0x9705
#define SCR01_T1I2C_E_TOO_MANY_R_FRAMES             0x9706
#define SCR01_T1I2C_E_NO_TIME_AVAILABLE             0x9707
#define SCR01_T1I2C_E_TOO_MANY_REPEATS              0x9708 
#define SCR01_T1I2C_E_CLOCKSTRETCHING               0x9709 
#define SCR01_T1I2C_E_NACKERROR                     0x970A 
#define SCR01_T1I2C_E_SFRAMEERROR                   0x970B 
#define SCR01_T1I2C_E_HSACKERROR                    0x970C 
#define SCR01_T1I2C_E_NADERROR                      0x970D
#define SCR01_T1I2C_E_OVERFLOWERROR                 0x970E
#define SCR01_T1I2C_E_CRCERROR						0x970F
#define SCR01_T1I2C_E_TRIGGER_TIMEOUT               0x9710
#define SCR01_T1I2C_E_ANTITEAR                      0x9711
#define SCR01_T1I2C_E_PROTOCOLERROR                 0x9712
#define SCR01_T1I2C_E_NOTSUPPORTED                  0x9713
#define SCR01_T1I2C_E_SEND_TIMEOUT                  0x9714
#define SCR01_T1I2C_E_OS_REMOVED                    0x9715


#define SCR01_T1I2C_LOGIF_SE                        0
#define SCR01_T1I2C_LOGIF_UICC                      1

_EXPORT_ unsigned int scr01_t1i2c_set_logicalinterface (unsigned char logicalinterface);

_EXPORT_ unsigned int scr01_t1i2c_set_bitrate (double bitrate);

_EXPORT_ unsigned int scr01_t1i2c_set_txtimeout_us (unsigned int txtimeout_us);

_EXPORT_ unsigned int scr01_t1i2c_set_slave_address (unsigned char slave_address);

_EXPORT_ unsigned int scr01_t1i2c_set_hs_master_code (unsigned char hs_master_code);

_EXPORT_ unsigned int scr01_t1i2c_set_ifs (unsigned int ifs);

_EXPORT_ unsigned int scr01_t1i2c_soft_reset (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_t1i2c_get_atr (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_t1i2c_end_of_session (unsigned char * databuffer, unsigned int * length);

#define SCR01_T1I2C_SFRAME_SOFT_RESET            0
#define SCR01_T1I2C_SFRAME_END_OF_SESSION        1

#define SCR01_GPI2C_SFRAME_RELEASE               8
#define SCR01_GPI2C_SFRAME_SWR                   9
#define SCR01_GPI2C_SFRAME_CIP                   10

_EXPORT_ unsigned int scr01_t1i2c_sframe_exchange (unsigned char sframe_select, unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_t1i2c_generic_sframe_exchange (unsigned char command, unsigned char * databuffer, unsigned int * length);


#define SCR01_T1I2C_BLOCKNUMBER_0    0
#define SCR01_T1I2C_BLOCKNUMBER_1    1
#define SCR01_T1I2C_BLOCKNUMBER_AUTO 255

_EXPORT_ unsigned int scr01_t1i2c_apdu_exchange (unsigned char * databuffer, unsigned int * length, unsigned char blocknumber);

_EXPORT_ unsigned int scr01_t1i2c_apdu_exchange_poweroff (unsigned char * databuffer, unsigned int * length, unsigned char blocknumber, unsigned int power_off_time_us);

_EXPORT_ unsigned int scr01_t1i2c_reset_protocol ();

_EXPORT_ unsigned int scr01_t1i2c_get_execution_time_us (unsigned int * execution_time);

_EXPORT_ unsigned int scr01_t1i2c_get_apdu_time_us (unsigned int * apdu_time);


#define SCR01_T1I2C_PROTOCOL_NAD_LRC                    1
#define SCR01_T1I2C_PROTOCOL_DIS_HS_STRETCH             2
#define SCR01_T1I2C_PROTOCOL_USE_CRC                    3
#define SCR01_T1I2C_PROTOCOL_T1_DISABLE_ERROR_HANDLING  4
#define SCR01_T1I2C_PROTOCOL_DIS_HS_MODE                5
#define SCR01_T1I2C_PROTOCOL_DIS_ALL_STRETCH            6
#define SCR01_T1I2C_PROTOCOL_SEND_NAD_VALUE             7
#define SCR01_T1I2C_PROTOCOL_RECV_NAD_VALUE             8
#define SCR01_T1I2C_PROTOCOL_GP_MODE                    9
#define SCR01_T1I2C_PROTOCOL_POLLING_TIME_MS           10
#define SCR01_GPI2C_PROTOCOL_POLLING_TIME_US           11
#define SCR01_GPI2C_PROTOCOL_IFSC                      12
#define SCR01_T1I2C_PROTOCOL_BWT_US                    13
#define SCR01_T1I2C_PROTOCOL_FORCE_HS_MODE             14
#define SCR01_T1I2C_PROTOCOL_SEND_INDELAY              15
#define SCR01_T1I2C_PROTOCOL_RECV_INDELAY              16
#define SCR01_T1I2C_PROTOCOL_RECV_STOP_DELAY           17

// further parameters, usable on SCR2 only, the meaning of these parameters can be found in SCR2 manual
#define SCR_T1I2C_BITRATE                              31
#define SCR_T1I2C_LOGICALINTERFACE                     32
#define SCR_T1I2C_SLAVE_ADDRESS                        33
#define SCR_T1I2C_TXRETRYTIME                          34
#define SCR_T1I2C_HSMASTERCODE                         35
#define SCR_T1I2C_SEND_SEQ_NR                          36
#define SCR_T1I2C_RECV_SEQ_NR                          37
#define SCR_T1I2C_TEARING_TIMEOUT                      38
#define SCR_T1I2C_TEARING_MODE                         39


_EXPORT_ unsigned int scr01_t1i2c_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );

_EXPORT_ unsigned int scr01_t1i2c_get_parameter ( unsigned int parameter_select, unsigned int * parameter_value );








//////////////////////////////////////////////////////////////////////////////////
//
// Dual Interface routines
//


// Parameter 1 - dual_if_mode

#define DUALIF_NONE 			        0	// "UART only", default setting

#define DUALIF_SWP_SPI5			        1	// "SWP" and "T1 over SPI" 
#define DUALIF_DWP_TP1TP2_SPI5	        2	// "DWP via TP1/TP2" and "T1 over SPI" 
#define DUALIF_DWP_IO3IO4_SPI5	        3	// "DWP via IO3/IO4" and "T1 over SPI" 

#define DUALIF_SWP_UART 		        4	// "SWP" and "ISO7816" 
#define DUALIF_DWP_TP1TP2_UART 	        5	// "DWP via TP1/TP2" and "ISO7816" 
#define DUALIF_DWP_IO3IO4_UART	        6	// "DWP via IO3/IO4" and "ISO7816" 

#define DUALIF_SPI5_T1I2C_IO3IO4        7	// "T1I2C via IO3/IO4" and "T1 over SPI", VCC=1.8V only !!!

  // SCR2 only :
#define DUALIF_T1I2C_IO2IO3_I2CSLAVE   20	// "T1I2C  via IO2/IO3" and "I2CSLAVE via IO5/IO6"
#define DUALIF_RAWI2C_IO2IO3_I2CSLAVE  21	// "RAWI2C via IO2/IO3" and "I2CSLAVE via IO5/IO6"


// further combinations to be added later on ...

// Parameter 2 and 3 as for scr01_set_io_configuration, see above

_EXPORT_ unsigned int scr01_set_dualio_configuration ( unsigned char dual_if_mode,
												       unsigned char rst_mode,
												       unsigned char clk_mode );


#define SCR01_E_WRONG_INTERFACES_SELECTED           0x9801

_EXPORT_ unsigned int scr01_iso7816_swp_apdu_exchange (unsigned char * databuffer_iso7816, 
													   unsigned int  * length_iso7816,
													   unsigned int  * retvalue_iso7816,
													   unsigned int    startdelay_iso7816,

													   unsigned char * databuffer_swp, 
													   unsigned int  * length_swp,
													   unsigned char   gate_swp,
													   unsigned int  * retvalue_swp,
													   unsigned int    startdelay_swp
													  );

_EXPORT_ unsigned int scr01_spi5_swp_apdu_exchange (unsigned char * databuffer_spi5, 
													unsigned int  * length_spi5,
													unsigned int  * retvalue_spi5,
													unsigned int    startdelay_spi5,

													unsigned char * databuffer_swp, 
													unsigned int  * length_swp,
													unsigned char   gate_swp,
													unsigned int  * retvalue_swp,
													unsigned int    startdelay_swp
													);


_EXPORT_ unsigned int scr01_spi5_t1i2c_apdu_exchange (unsigned char * databuffer_spi5, 
												      unsigned int  * length_spi5,
												      unsigned int  * retvalue_spi5,
												      unsigned int    startdelay_spi5,
                                                      
												      unsigned char * databuffer_t1i2c, 
												      unsigned int  * length_t1i2c,
												      unsigned int  * retvalue_t1i2c,
												      unsigned int    startdelay_t1i2c
												      );









//////////////////////////////////////////////////////////////////////////////////
//
// RAW I2C, can be used to control I2C MEMORY (for instance)
//


#define SCR01_RAWI2C_E_FRAMELENGTHERROR				0x9601
#define SCR01_RAWI2C_E_FRAMESENDERROR				0x9602
#define SCR01_RAWI2C_E_NOT_ACKNOWLEDGED				0x9603
#define SCR01_RAWI2C_E_FRAMEREADERROR               0x9604
#define SCR01_RAWI2C_E_WRONGCONTROLBYTE             0x9605
#define SCR01_RAWI2C_E_WRONGTIMEOUTVALUE            0x9606
#define SCR01_RAWI2C_E_ACKHSERROR   				0x9607
#define SCR01_RAWI2C_E_TRIGGER_TIMEOUT              0x9608
#define SCR01_RAWI2C_E_INSERTION_ABORTED            0x9609
#define SCR01_RAWI2C_E_STRETCH_TIMEOUT              0x960a
#define SCR01_RAWI2C_E_BITRATE_ERROR                0x960b


_EXPORT_ unsigned int scr01_rawi2c_set_bitrate (double bitrate);

_EXPORT_ unsigned int scr01_rawi2c_set_controlbyte (unsigned char controlbyte);

_EXPORT_ unsigned int scr01_rawi2c_set_hs_master_code (unsigned char hs_master_code);

_EXPORT_ unsigned int scr01_rawi2c_send_data (unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr01_rawi2c_send_chm (unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr01_rawi2c_read_data (unsigned char * controlbuffer , unsigned int control_length,
                                              unsigned char * readdatabuffer, unsigned int readdata_length);

_EXPORT_ unsigned int scr01_rawi2c_exchange_frame (unsigned char * commanddatabuffer,     unsigned int commandlength,
                                                   unsigned char * responsecontrolbuffer, unsigned int responsecontrollength,
                                                   unsigned char * responsedatabuffer,    unsigned int responsedatalength,
                                                   unsigned int    wait_time_ms);

_EXPORT_ unsigned int scr01_rawi2c_read_data2 (unsigned char * readdatabuffer, unsigned int readdata_length);

_EXPORT_ unsigned int scr01_rawi2c_exchange2_frame (unsigned char * commanddatabuffer,     unsigned int commandlength,
                                                    unsigned char * responsedatabuffer,    unsigned int responsedatalength,
                                                    unsigned int    wait_time_ms);



_EXPORT_ unsigned int scr01_rawi2c_exchange_data (unsigned char * commanddatabuffer,     unsigned int commandlength,
                                                  unsigned char * responsedatabuffer,    unsigned int responsedatalength,
                                                  unsigned int    wait_time_us);



#define SCR01_RAWI2C_PROTOCOL_DIS_HS_STRETCH       1
#define SCR01_RAWI2C_PROTOCOL_ERROR_MODE           2
#define SCR01_RAWI2C_PROTOCOL_ERROR_START          3
#define SCR01_RAWI2C_PROTOCOL_ERROR_LENGTH         4
#define SCR01_RAWI2C_PROTOCOL_ERROR_PATTERN        5
#define SCR01_RAWI2C_PROTOCOL_IGNORE_NACK          6
#define SCR01_RAWI2C_PROTOCOL_DIS_HS_MODE          7
#define SCR01_RAWI2C_PROTOCOL_DIS_ALL_STRETCH      8
#define SCR01_RAWI2C_PROTOCOL_FORCE_HS_MODE        9
#define SCR01_RAWI2C_PROTOCOL_SEND_INDELAY        10
#define SCR01_RAWI2C_PROTOCOL_RECV_INDELAY        11
#define SCR01_RAWI2C_PROTOCOL_RECV_STOP_DELAY     12

// for SCR2 only :
#define SCR01_RAWI2C_PROTOCOL_OVERRUN_SEND_COUNT  21
#define SCR01_RAWI2C_PROTOCOL_OVERRUN_RECV_COUNT  22
#define SCR01_RAWI2C_PROTOCOL_OVERRUN_IGNORE_NACK 23
#define SCR01_RAWI2C_PROTOCOL_SUPPRESS_START      30
#define SCR01_RAWI2C_PROTOCOL_SUPPRESS_STOP       31



#define SCR01_RAWI2C_ERROR_MODE_REPLACE_CONTINUE  0
#define SCR01_RAWI2C_ERROR_MODE_REPLACE_STOP      1
#define SCR01_RAWI2C_ERROR_MODE_INSERT_CONTINUE   2
#define SCR01_RAWI2C_ERROR_MODE_INSERT_STOP       3

// the following parameters apply for SCR2 only
#define SCR01_RAWI2C_PROTOCOL_ERROR_SDADELAY   32
#define SCR01_RAWI2C_PROTOCOL_ERROR_SDADEFAULT 33
#define SCR01_RAWI2C_ERROR_SDADEFAULT_LOW         0
#define SCR01_RAWI2C_ERROR_SDADEFAULT_HIGH        1
#define SCR01_RAWI2C_ERROR_SDADEFAULT_PULL        2

_EXPORT_ unsigned int scr01_rawi2c_set_parameter ( unsigned int parameter_select, unsigned int parameter_value );



// For SCR2 only !

typedef enum _SCR_RAWI2C_RESET_MODES {

    SCR_RAWI2C_RESET_NONE=0,
    SCR_RAWI2C_RESET_COLD=1,
    SCR_RAWI2C_RESET_WARM=2

} SCR_RAWI2C_RESET_MODES;


_EXPORT_ unsigned int scr_rawi2c_observe_interface (SCR_RAWI2C_RESET_MODES reset_mode, unsigned int observe_time_us,
                                                    unsigned int * scl_edges, unsigned int * sda_edges,
                                                    double * mean_current_mA, double * max_current_mA);







// ############################################### FOR INTERNAL USE ONLY, DON'T USE IT ###############################################


#define SCR01_SCSPI_E_FRAMELENGTHERROR				0x9401
#define SCR01_SCSPI_E_GENERALERROR					0x9402
#define SCR01_SCSPI_E_WRONGPCB						0x9403
#define SCR01_SCSPI_E_ATR_WRONG_TAG                 0x9404
#define SCR01_SCSPI_E_ATR_CODING_ERROR              0x9405
#define SCR01_SCSPI_E_NO_ATR                        0x9406
#define SCR01_SCSPI_E_PE_PARAM_ERROR                0x9407
#define SCR01_SCSPI_E_PARAMETER_EXCHANGE            0x9408
#define SCR01_SCSPI_E_PE_WRONG_PCB                  0x9409
#define SCR01_SCSPI_E_SEND_ERROR                    0x940A
#define SCR01_SCSPI_E_GENERAL_STATE_ERROR           0x940B
#define SCR01_SCSPI_E_RECV_SEQUENCE_ERROR           0x940C
#define SCR01_SCSPI_E_PROTOCOL_EXCEPTION            0x940D
#define SCR01_SCSPI_NO_ADDR_ACK                     0x940E
#define SCR01_SCSPI_RECV_TIMEOUT                    0x940F
#define SCR01_SCSPI_E_APDU_TIMEOUT				    0x9410

#define SCR01_SCSPI_POWER_CYCLE_DISABLED   0
#define SCR01_SCSPI_POWER_CYCLE_ENABLED    1

_EXPORT_ unsigned int scr01_scspi_activate (double bitrate, unsigned char powercycle );

_EXPORT_ unsigned int scr01_scspi_apdu_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_scspi_get_execution_time_us (unsigned int * execution_time_us);





#define SCR01_SCI2C_E_FRAMELENGTHERROR				0x9501
#define SCR01_SCI2C_E_GENERALERROR					0x9502
#define SCR01_SCI2C_E_WRONGPCB						0x9503
#define SCR01_SCI2C_E_ATR_WRONG_TAG                 0x9504
#define SCR01_SCI2C_E_ATR_CODING_ERROR              0x9505
#define SCR01_SCI2C_E_NO_ATR                        0x9506
#define SCR01_SCI2C_E_PE_PARAM_ERROR                0x9507
#define SCR01_SCI2C_E_PARAMETER_EXCHANGE            0x9508
#define SCR01_SCI2C_E_PE_WRONG_PCB                  0x9509
#define SCR01_SCI2C_E_SEND_ERROR                    0x950A
#define SCR01_SCI2C_E_GENERAL_STATE_ERROR           0x950B
#define SCR01_SCI2C_E_RECV_SEQUENCE_ERROR           0x950C
#define SCR01_SCI2C_E_PROTOCOL_EXCEPTION            0x950D
#define SCR01_SCI2C_NO_ADDR_ACK                     0x950E
#define SCR01_SCI2C_RECV_TIMEOUT                    0x950F
#define SCR01_SCI2C_E_APDU_TIMEOUT				    0x9510

#define SCR01_SCI2C_POWER_CYCLE_DISABLED   0
#define SCR01_SCI2C_POWER_CYCLE_ENABLED    1

_EXPORT_ unsigned int scr01_sci2c_activate (double bitrate, unsigned char powercycle );

_EXPORT_ unsigned int scr01_sci2c_apdu_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr01_sci2c_get_execution_time_us (unsigned int * execution_time_us);







// ####################################### FOR USE WITH SCR2 HARDWARE ONLY, DON'T USE IT WITH SCR01 ########################################



//////////////////////////////////////////////////////////////////////////////////
//
// Bit banging functionality (SCR2 only)
//


// define the BITBANG error codes

typedef enum _SCR_E_BITBANG {

    SCR_E_BITBANG_GENERAL_ERROR = 0x9900,
    SCR_E_BITBANG_TIMEOUT_ERROR,
	SCR_E_BITBANG_VECTOR_MEM_FULL,
	SCR_E_BITBANG_NO_VECTORS_DEFINED,
	SCR_E_BITBANG_NO_RESULTS_AVAILABLE,
	SCR_E_BITBANG_WRONG_VECTOR_NUMBER,
	SCR_E_BITBANG_SEQUENCE_OVERFLOW,
	SCR_E_BITBANG_STOP_ERROR

} SCR_E_BITBANG;


// define the BITBANG parameters

typedef enum _SCR_BITBANG_PARAMS {

    SCR_BITBANG_IOVECTOR_COUNT      = 0,		// read-only
    SCR_BITBANG_IOVECTOR_STOP_POS   = 1   		// read-only

} SCR_BITBANG_PARAMS;


// define the BITBANG vector types

typedef enum _SCR_BITBANG_VECTOR_TYPES {

    SCR_BITBANG_DURATION_VECTOR      = 0,
    SCR_BITBANG_CONDITIONAL_VECTOR   = 1

} SCR_BITBANG_VECTOR_TYPES;


typedef enum _SCR_BITBANG_PORTVALS {

	// for all vector types
    SCR_BITBANG_PORT_RELEASED     = 0,

	// for SCR_BITBANG_DURATION_VECTOR
    SCR_BITBANG_PORT_SET_OUT0     = 1,
    SCR_BITBANG_PORT_SET_OUT1     = 2,
    SCR_BITBANG_PORT_SET_IN       = 3,

	// for SCR_BITBANG_CONDITIONAL_VECTOR
    SCR_BITBANG_PORT_EXPECT_0     = 5,
    SCR_BITBANG_PORT_EXPECT_1     = 6

} SCR_BITBANG_PORTVALS;


typedef struct _SCR_BITBANG_IOVECTOR {

    SCR_BITBANG_VECTOR_TYPES vector_type;

	unsigned int vector_time_value_ns;  // 20 ... 1.000.000 (nsecs)

	unsigned char	vcc_control_enabled;	    // true          / false
	double  vcc_control_value;		    // 0 ... 6.0 (V) / don't care

    SCR_BITBANG_PORTVALS CLK;
    SCR_BITBANG_PORTVALS RST;
    SCR_BITBANG_PORTVALS IO1;
    SCR_BITBANG_PORTVALS IO2;
    SCR_BITBANG_PORTVALS IO3;
    SCR_BITBANG_PORTVALS IO4;
    SCR_BITBANG_PORTVALS IO5;
    SCR_BITBANG_PORTVALS IO6;
    SCR_BITBANG_PORTVALS SWP;
    SCR_BITBANG_PORTVALS TP1;
    SCR_BITBANG_PORTVALS TP2;
    SCR_BITBANG_PORTVALS EXT0;
    SCR_BITBANG_PORTVALS EXT1;
    SCR_BITBANG_PORTVALS EXT2;

} SCR_BITBANG_IOVECTOR;


typedef struct _SCR_BITBANG_IOSTATES {

	unsigned int vector_duration_ns;  

	unsigned int CLK_VAL;
	unsigned int RST_VAL;
	unsigned int IO1_VAL;
	unsigned int IO2_VAL;
	unsigned int IO3_VAL;
	unsigned int IO4_VAL;
	unsigned int IO5_VAL;
	unsigned int IO6_VAL;
	unsigned int SWP_VAL;
	unsigned int TP1_VAL;
	unsigned int TP2_VAL;
	unsigned int EXT0_VAL;
	unsigned int EXT1_VAL;
	unsigned int EXT2_VAL;

} SCR_BITBANG_IOSTATES;


typedef enum _SCR_BITBANG_CMD_CODES {

    SCR_BITBANG_CMD_SET_R0      = 0x10,
    SCR_BITBANG_CMD_SET_R1      = 0x11,
    SCR_BITBANG_CMD_SET_R2      = 0x12,
    SCR_BITBANG_CMD_SET_R3      = 0x13,

	SCR_BITBANG_CMD_DJNZ_R0     = 0x20,
	SCR_BITBANG_CMD_DJNZ_R1     = 0x21,
	SCR_BITBANG_CMD_DJNZ_R2     = 0x22,
	SCR_BITBANG_CMD_DJNZ_R3     = 0x23,

	SCR_BITBANG_CMD_STOP_LOW    = 0x30,
	SCR_BITBANG_CMD_STOP_HIGH   = 0x31


} SCR_BITBANG_CMD_CODES;


typedef enum _SCR_BITBANG_PORT_NAMES { 
	
    SCR_BITBANG_PORT_CLK=0,
    SCR_BITBANG_PORT_RST,
	SCR_BITBANG_PORT_IO1,
    SCR_BITBANG_PORT_IO2,
    SCR_BITBANG_PORT_IO3,
    SCR_BITBANG_PORT_IO4,
    SCR_BITBANG_PORT_IO5,
    SCR_BITBANG_PORT_IO6,
    SCR_BITBANG_PORT_SWP,
    SCR_BITBANG_PORT_TP1,
    SCR_BITBANG_PORT_TP2,
    SCR_BITBANG_PORT_EXT0,
    SCR_BITBANG_PORT_EXT1,
    SCR_BITBANG_PORT_EXT2

} SCR_BITBANG_PORT_NAMES;



_EXPORT_ unsigned int scr_bitbang_set_parameter (SCR_BITBANG_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr_bitbang_get_parameter (SCR_BITBANG_PARAMS param_select, unsigned int * param_value); 


_EXPORT_ unsigned int scr_bitbang_clear_vectors (); 

_EXPORT_ unsigned int scr_bitbang_add_io_vector (SCR_BITBANG_IOVECTOR * bitbang_iovector); 
_EXPORT_ unsigned int scr_bitbang_add_cmd_vector (SCR_BITBANG_CMD_CODES command, unsigned int parameter); 

_EXPORT_ unsigned int scr_bitbang_get_iostates (unsigned int vector_number, SCR_BITBANG_IOSTATES * bitbang_iostates); 

_EXPORT_ unsigned int scr_bitbang_start_sequence (); 

_EXPORT_ unsigned int scr_bitbang_dump_sequence (char * vcd_file_name); 







//////////////////////////////////////////////////////////////////////////////////
//
// I2CSLAVE functionality (SCR2 only)
//

// define the I2CSLAVE error codes

typedef enum _SCR_E_I2CSLAVE {

    SCR_E_I2CSLAVE_GENERAL_ERROR = 0x9A00

} SCR_E_I2CSLAVE;


// define the I2CSLAVE parameters

typedef enum _SCR_I2CSLAVE_PARAMS {

    SCR_I2CSLAVE_SLAVE_ENABLE=0,
    SCR_I2CSLAVE_SLAVE_ADDRESS,
	SCR_I2CSLAVE_SCL_FILTER_LENGTH,
	SCR_I2CSLAVE_SDA_FILTER_LENGTH,
	SCR_I2CSLAVE_MEM_PROG_TIME,
	SCR_I2CSLAVE_MEM_PAGE_SIZE,
	SCR_I2CSLAVE_MEM_BUSY_MODE

} SCR_I2CSLAVE_PARAMS;


_EXPORT_ unsigned int scr_i2cslave_set_parameter (SCR_I2CSLAVE_PARAMS param_select, unsigned int   param_value);
_EXPORT_ unsigned int scr_i2cslave_get_parameter (SCR_I2CSLAVE_PARAMS param_select, unsigned int * param_value);

_EXPORT_ unsigned int scr_i2cslave_set_memory (unsigned char * mem_buffer, unsigned int mem_address, unsigned int mem_length);
_EXPORT_ unsigned int scr_i2cslave_get_memory (unsigned char * mem_buffer, unsigned int mem_address, unsigned int mem_length);







#ifdef __cplusplus
}
#endif

