#ifdef __cplusplus
extern "C"
{
#endif

#ifndef _EXPORT_
#define _EXPORT_ __declspec(dllexport)
#endif


// define the general error codes

#define SCR2_OK  0x0000 

typedef enum _SCR2_E_GENERAL {

    SCR2_E_INITIALIZE = 0x8000,                     
    SCR2_E_ENUMERATE,  
    SCR2_E_SHAREDMEMORY,
    SCR2_E_SERIAL_ERROR,
    SCR2_E_READER_NOT_FOUND,    
    SCR2_E_SERIAL_NOT_FOUND,
    SCR2_E_NOT_INITIALIZED,                         
    SCR2_E_ALREADY_INITIALIZED,                         
    SCR2_E_READ_REGISTER,                           
    SCR2_E_WRITE_REGISTER,                          
    SCR2_E_READ_MEMORY,                             
    SCR2_E_WRITE_MEMORY,                            
    SCR2_E_SET_TIMEOUT,                             
    SCR2_E_TSENS_NOT_ENABLED,                       
    SCR2_E_TSENS_ERROR,				                
    SCR2_E_PARAM_ERROR,				                
    SCR2_E_CONTROL_IO_PORTS, 		                
    SCR2_E_FATALERROR,
    SCR2_E_EXCEPTION,
    SCR2_E_TRACEERROR,                              
    SCR2_E_TRACEINCONSISTENT,                       
    SCR2_E_TRACEFILE,                               
    SCR2_E_LOGGERISACTIVE,                          
    SCR2_E_LOGGERNOTSTARTED,                        
    SCR2_E_LOGGERNOTSTOPPED,                        
    SCR2_E_LOGGEROVERFLOW,                          
    SCR2_E_GENERALHCIERROR,                         
    SCR2_E_CONTROL_TRAPDOOR,                        
    SCR2_E_FEATURENOTSUPPORTED,                     
    SCR2_E_PORTNOTAVAILABLE,                        
    SCR2_E_COMMIFNOTENABLED,                        
    SCR2_E_COMMIFNOTSELECTED,                       
    SCR2_E_DUALIFNOTSELECTED,                       
    SCR2_E_WRONG_VTP_VALUE,                         
    SCR2_E_COMMIF1NOTAVAILABLE,                     
    SCR2_E_COMMIF2NOTAVAILABLE,                     
    SCR2_E_GTKWFILE,                                
    SCR2_E_SUPPORT_DLL_NOT_FOUND,                   
    SCR2_E_TRIM_WRITE,                                 
    SCR2_E_MEASUREMENT_DISABLED,                      
    SCR2_E_DISPLAYSTRING_TOO_LONG,                  
    SCR2_E_WRONG_VCC_VALUE,                         
    SCR2_E_CLOCK1_ERROR,                             
    SCR2_E_CLOCK2_ERROR,                             
    SCR2_E_ABORTED_BY_USER,                         
    SCR2_E_MEMORY_ALLOCATION_FAILED,
    SCR2_E_LOGFILE_ERROR,
    SCR2_E_LOGFILE_NOT_OPEN,
    SCR2_E_TRIGGERPORTNOTAVAILABLE,      
    SCR2_E_IDENT_CRC_ERROR,
    SCR2_E_IDENT_NOT_FOUND,
    SCR2_E_EXTI2C_NOT_ENABLED,
    SCR2_E_EXTI2C_NOT_ACKNOWLEDGED,
    SCR2_E_EXTI2C_FRAMESENDERROR,
    SCR2_E_EXTI2C_FRAMEREADERROR

} SCR2_E_GENERAL;

_EXPORT_ void scr2_get_error_string (unsigned int error_code, char * error_string); 

_EXPORT_ unsigned int scr2_enumerate (unsigned char * instancecount, int * serial_numbers);
_EXPORT_ unsigned int scr2_initialize (int serial_number);   
_EXPORT_ unsigned int scr2_close ();

_EXPORT_ unsigned int scr2_get_locked_devices (unsigned int * device_serials, unsigned char * device_count);
_EXPORT_ unsigned int scr2_unlock_device (unsigned int device_serial);

_EXPORT_ unsigned int scr2_get_hardware_info (char * serial, unsigned int * hw_version, unsigned int * hw_config);   
_EXPORT_ unsigned int scr2_get_dll_version (char * dllversion);   

_EXPORT_ unsigned int scr2_set_user_display (char * user_string);

_EXPORT_ void         scr2_hal_sleep(unsigned int mseconds);

_EXPORT_ unsigned int scr2_hw_extension_info ( char * hw_extension_info );


// define the general (global) parameters

typedef enum _SCR2_GENERAL_PARAMS {

    SCR2_GENERAL_PARAM_CLOSE_KEEP_POWER = 0

} SCR2_GENERAL_PARAMS;


_EXPORT_ unsigned int scr2_general_set_parameter (SCR2_GENERAL_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_general_get_parameter (SCR2_GENERAL_PARAMS param_select, unsigned int * param_value); 



//////////////////////////////////////////////////////////////////////////////////
//
// Log file support
//


_EXPORT_ unsigned int scr2_open_log_file (const char * logfilename);

_EXPORT_ unsigned int scr2_close_log_file ();

_EXPORT_ unsigned int scr2_add_user_log  (const char * log_entry);

_EXPORT_ unsigned int scr2_get_apdu_log_length (unsigned int * log_len);

_EXPORT_ unsigned int scr2_get_apdu_log_string (char * log_string);

_EXPORT_ unsigned int scr2_set_log_verbose_level (unsigned int verbose_level);



//////////////////////////////////////////////////////////////////////////////////
//
// Control VCC voltage
//


_EXPORT_ unsigned int scr2_set_vcc (float voltage); 

_EXPORT_ unsigned int scr2_get_vcc (float * voltage); 

_EXPORT_ unsigned int scr2_poweroff_reset (); 





//////////////////////////////////////////////////////////////////////////////////
//
// Set IO configuration 
//

typedef enum _SCR2_COMMIF { 

	SCR2_COMMIF_NONE=0,	                /**< No interface                          */
	SCR2_COMMIF_ISO7816_IO1,            /**< ISO7816 via CLK, RST, IO1             */
	SCR2_COMMIF_SWP,                    /**< SWP via SWPIO                         */
	SCR2_COMMIF_DWP_IO3IO4,             /**< DWP via IO3 and IO4                   */
	SCR2_COMMIF_DWP_TP1TP2,             /**< DWP via TP1 and TP2                   */
	SCR2_COMMIF_SCSPI_CLKIO1IO2IO3,     /**< SCSPI                                 */
	SCR2_COMMIF_SCI2C_IO2IO3,           /**< I2C over IO2/IO3                      */
	SCR2_COMMIF_RF,                     /**< RF via antenna                        */
	SCR2_COMMIF_SCI2C_CLKIO1,           /**< I2C over CLK/IO1                      */
	SCR2_COMMIF_T1SPI_CLKIO1IO2IO3IO4,  /**< T1 over SPI                           */
	SCR2_COMMIF_RAWI2C_IO3IO4,          /**< RAW I2C via IO3 and IO4               */
	SCR2_COMMIF_RAWI2C_CLKIO1,          /**< RAW I2C via CLK and IO1               */
	SCR2_COMMIF_FTB,                    /**< FTB via CLK, RST, IO1, IO2, IO3       */
	SCR2_COMMIF_T1I2C_IO2IO3,           /**< T1 over I2C via IO2 and IO3           */
	SCR2_COMMIF_T1I2C_IO3IO4,           /**< T1 over I2C via IO3 and IO4           */
	SCR2_COMMIF_T1I2C_IO5IO6,           /**< T1 over I2C via IO5 and IO6           */
	SCR2_COMMIF_T1SPI_CLKIO1IO2TP1TP2,  /**< T1 over SPI                           */
    SCR2_COMMIF_SIM,                    /**< Rear SIM interface                    */
	SCR2_COMMIF_RAWI2C_IO2IO3,          /**< RAW I2C via IO2 and IO3               */
	SCR2_COMMIF_I2CSLAVE_IO5IO6,        /**< I2C SLAVE on IO5 and IO6              */
	SCR2_COMMIF_T1SPI_IO2IO3CLKIO4,     /**< T1 over SPI                           */
	SCR2_COMMIF_FTB2                    /**< FTB via CLK, RST, IO1, IO3, IO4       */
        
} SCR2_COMMIF;


_EXPORT_ unsigned int scr2_set_commif_configuration ( SCR2_COMMIF   commif_1,
                                                      SCR2_COMMIF   commif_2
													);






//////////////////////////////////////////////////////////////////////////////////
//
// Control IO ports
//


typedef enum _IO_PORT_NAMES { 
	
	SCR2_PORT_IO1=0,
    SCR2_PORT_IO2,
    SCR2_PORT_IO3,
    SCR2_PORT_IO4,
    SCR2_PORT_IO5,
    SCR2_PORT_IO6,
    SCR2_PORT_CLK,
    SCR2_PORT_RST,
    SCR2_PORT_SWP,
    SCR2_PORT_TP1,
    SCR2_PORT_TP2

} IO_PORT_NAMES;
                      
typedef enum _IO_PORT_MODES { 
	
	SCR2_PORTMODE_RELEASED=0,    // release port
    SCR2_PORTMODE_OUT0,          // output mode, drive level 0
    SCR2_PORTMODE_OUT1,          // output mode, drive level 1
    SCR2_PORTMODE_PULLUP,        // input mode , pull up 
    SCR2_PORTMODE_TRISTATE,      // input mode , tristate
    SCR2_PORTMODE_ANALOG,        // output mode, driven by DAC directly
    SCR2_PORTMODE_PULSE          // output mode, pulse (sequence)

} IO_PORT_MODES;



_EXPORT_ unsigned int scr2_control_io_port (IO_PORT_NAMES iosel,        // SCR2_PORT_IO1, SCR2_PORT_IO2, ...
                                            IO_PORT_MODES iomode);	  	// SCR2_PORTMODE_NONE, SCR2_PORTMODE_OUT0, SCR2_PORTMODE_OUT1, ...
										
_EXPORT_ unsigned int scr2_read_io_port (IO_PORT_NAMES   iosel,         // SCR2_PORT_IO1, SCR2_PORT_IO2, ...
                                         unsigned char * ioval);	    // pointer to variable
										


// define the IO PORT parameters

typedef enum _SCR2_IOPORT_PARAMS {

    SCR2_IOPORT_PULSE_HIGH_NSEC   = 0,		
    SCR2_IOPORT_PULSE_LOW_NSEC,  		
    SCR2_IOPORT_PULSE_COUNT,
    SCR2_IOPORT_PULSE_DEFAULT,
    SCR2_IOPORT_PULSE_ENABLE

} SCR2_IOPORT_PARAMS;


_EXPORT_ unsigned int scr2_ioport_set_parameter (SCR2_IOPORT_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_ioport_get_parameter (SCR2_IOPORT_PARAMS param_select, unsigned int * param_value); 



//////////////////////////////////////////////////////////////////////////////////
//
// Control Extension ports
//


typedef enum _SCR2_EXT_PORTS {
    SCR2_EXT_PORT0=0,
    SCR2_EXT_PORT1,
    SCR2_EXT_PORT2,
    SCR2_EXT_PORT3,
    SCR2_EXT_PORT4,
    SCR2_EXT_PORT5
} SCR2_EXT_PORTS;

typedef enum _SCR2_EXT_PORT_MODES {
    SCR2_EXT_TRISTATE=0,
    SCR2_EXT_OUT0,
    SCR2_EXT_OUT1
} SCR2_EXT_PORT_MODES;

_EXPORT_ unsigned int scr2_ctrl_ext_port ( SCR2_EXT_PORTS ext_select, SCR2_EXT_PORT_MODES ext_mode );
_EXPORT_ unsigned int scr2_read_ext_port ( SCR2_EXT_PORTS ext_select, unsigned char * readvalue );




//////////////////////////////////////////////////////////////////////////////////
//
// Control I2C interface on extension ports
//

typedef enum _SCR2_EXTI2C_MODES {
    SCR2_EXTI2C_DISABLED=0,
    SCR2_EXTI2C_ENABLED
} SCR2_EXTI2C_MODES;

_EXPORT_ unsigned int scr2_exti2c_enable ( SCR2_EXTI2C_MODES exti2c_enable );

_EXPORT_ unsigned int scr2_exti2c_send_data (unsigned char slave_address, unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr2_exti2c_read_data (unsigned char slave_address, unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr2_exti2c_read_data_comb (unsigned char   slave_address,
                                                  unsigned char * sendbuffer, unsigned int sendlength,
                                                  unsigned char * readbuffer, unsigned int readlength);






//////////////////////////////////////////////////////////////////////////////////
//
// Control analog and trapdoor
//

#define SCR2_TRAPDOOR_OFF					        0
#define SCR2_TRAPDOOR_ON					        1

#define SCR2_TRAPDOOR_TP1						    0
#define SCR2_TRAPDOOR_TP2						    1

#define SCR2_TRAPDOOR_PERMANENT					    0
#define SCR2_TRAPDOOR_CONDITIONAL				    1

_EXPORT_ unsigned int scr2_set_trapdoor  ( unsigned char trapdoor_value,	// SCR2_TRAPDOOR_ON, SCR2_TRAPDOOR_OFF
										   unsigned char trapdoor_select,	// SCR2_TRAPDOOR_TP1, SCR2_TRAPDOOR_TP2 
										   unsigned char trapdoor_mode );	// SCR2_TRAPDOOR_PERMANENT, SCR2_TRAPDOOR_CONDITIONAL


_EXPORT_ unsigned int scr2_set_vtp1 (float voltage);

_EXPORT_ unsigned int scr2_set_vtp2 (float voltage);


typedef enum _SCR2_MEASURE {

	SCR2_MEASURE_NONE=0,
    SCR2_MEASURE_VTP1,
    SCR2_MEASURE_VTP2,
    SCR2_MEASURE_ITP1,
    SCR2_MEASURE_ITP2

} SCR2_MEASURE;

_EXPORT_ unsigned int scr2_ctrl_measurement( SCR2_MEASURE measure_select);

_EXPORT_ unsigned int scr2_get_vtp (double * voltage);   

_EXPORT_ unsigned int scr2_get_icc (double * current);   

_EXPORT_ unsigned int scr2_get_itp (double * current);   






//////////////////////////////////////////////////////////////////////////////////
//
// Trace (data logger) functionality
//

#define SCR2_TRACE_TIMEBASE_20NS		            0
#define SCR2_TRACE_TIMEBASE_40NS		            1
#define SCR2_TRACE_TIMEBASE_100NS		            2

#define SCR2_TRACE_START_UNCOND			            0
#define SCR2_TRACE_START_COND			            1

#define SCR2_TRACE_OVERFLOW_STOP    	            0
#define SCR2_TRACE_OVERFLOW_CONTINUE	            1

_EXPORT_ unsigned int scr2_trace_start_logger (unsigned int time_base, unsigned int start_mode, unsigned int overflow_mode);

_EXPORT_ unsigned int scr2_trace_stop_logger (unsigned char * overflow_detected);

_EXPORT_ unsigned int scr2_trace_write_vcd_file (char * filename, unsigned int * trace_length_us, unsigned int max_time_us, unsigned char skip_analog); 

_EXPORT_ unsigned int scr2_trace_write_vcd_file_average (char * filename, unsigned int * trace_length_us, unsigned int max_time_us, unsigned int average_sample_count); 



//////////////////////////////////////////////////////////////////////////////////
//
// Test pin functionality
//


typedef enum _SCR2_TRIGGERS {

    SCR2_TRIG_INPUT=0,
    SCR2_TRIG_STATIC_1,
    SCR2_TRIG_CLOCK_1MHZ,
    SCR2_TRIG_ICC_LEVEL,
    SCR2_TRIG_VTP_LEVEL,
    SCR2_TRIG_T1I2C_SEND,
    SCR2_TRIG_T1I2C_RECV,
    SCR2_TRIG_T1I2C_EXEC,
    SCR2_TRIG_T1SPI_SEND,
    SCR2_TRIG_T1SPI_RECV,
    SCR2_TRIG_T1SPI_EXEC,
    SCR2_TRIG_T1I2C_SCL_OUT,
    SCR2_TRIG_T1I2C_SCL_DRIVE,
    SCR2_TRIG_T1I2C_SDA_OUT,
    SCR2_TRIG_T1I2C_SDA_DRIVE,
    SCR2_TRIG_T1I2C_ERROR_INSERTION_ACTIVE,
    SCR2_TRIG_T1I2C_ERROR_INSERTION_SCL_OUT,
    SCR2_TRIG_T1I2C_ERROR_INSERTION_SDA_OUT,
    SCR2_TRIG_RF_TEST_SIGOUT,
    SCR2_TRIG_RF_TEST_AUX1,
    SCR2_TRIG_RF_TEST_AUX2,
    SCR2_TRIG_RF_TEST_IRQ,
    SCR2_TRIG_RF_TEST_FIELD_ENABLED,
    SCR2_TRIG_RAWI2C_SCL_OUT,
    SCR2_TRIG_RAWI2C_SCL_DRIVE,
    SCR2_TRIG_RAWI2C_SDA_OUT,
    SCR2_TRIG_RAWI2C_SDA_DRIVE,
    SCR2_TRIG_RAWI2C_ERROR_INSERTION_ACTIVE,
    SCR2_TRIG_RAWI2C_ERROR_INSERTION_SCL_OUT,
    SCR2_TRIG_RAWI2C_ERROR_INSERTION_SDA_OUT

} SCR2_TRIGGERS;


_EXPORT_ unsigned int scr2_trigger_set_port1 ( SCR2_TRIGGERS trigger_select );

_EXPORT_ unsigned int scr2_trigger_set_port2 ( SCR2_TRIGGERS trigger_select );

_EXPORT_ unsigned int scr2_trigger_set_iccvalue ( double iccvalue );

_EXPORT_ unsigned int scr2_trigger_set_vtpvalue ( double vtpvalue );



typedef enum _SCR2_TRIGGER_HW_INTERFACES {

    SCR2_TRIG_HW_RAWI2C = 0,
    SCR2_TRIG_HW_T1I2C,
    SCR2_TRIG_HW_SCI2C,
    SCR2_TRIG_HW_T1SPI

} SCR2_TRIGGER_HW_INTERFACES;

typedef enum _SCR2_TRIGGER_HW_MODES {

    SCR2_TRIG_MODE_DISABLED = 0,
    SCR2_TRIG_MODE_HIGH_LEVEL

} SCR2_TRIGGER_HW_MODES;

_EXPORT_ unsigned int scr2_trigger_set_hw_interface_wait1  (SCR2_TRIGGER_HW_INTERFACES hw_interface, SCR2_TRIGGER_HW_MODES trigger_mode);
_EXPORT_ unsigned int scr2_trigger_set_hw_interface_wait2  (SCR2_TRIGGER_HW_INTERFACES hw_interface, SCR2_TRIGGER_HW_MODES trigger_mode);

_EXPORT_ unsigned int scr2_trigger_set_timeout  (unsigned int timeout_ms);




//////////////////////////////////////////////////////////////////////////////////
//
// Bit banging functionality
//


// define the BITBANG error codes

typedef enum _SCR2_E_BITBANG {

    SCR2_E_BITBANG_GENERAL_ERROR = 0x9100,
    SCR2_E_BITBANG_TIMEOUT_ERROR,
	SCR2_E_BITBANG_VECTOR_MEM_FULL,
	SCR2_E_BITBANG_NO_VECTORS_DEFINED,
	SCR2_E_BITBANG_NO_RESULTS_AVAILABLE,
	SCR2_E_BITBANG_WRONG_VECTOR_NUMBER,
	SCR2_E_BITBANG_SEQUENCE_OVERFLOW,
	SCR2_E_BITBANG_STOP_ERROR

} SCR2_E_BITBANG;


// define the BITBANG parameters

typedef enum _SCR2_BITBANG_PARAMS {

    SCR2_BITBANG_IOVECTOR_COUNT      = 0,		// read-only
    SCR2_BITBANG_IOVECTOR_STOP_POS   = 1   		// read-only

} SCR2_BITBANG_PARAMS;


_EXPORT_ unsigned int scr2_bitbang_set_parameter (SCR2_BITBANG_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_bitbang_get_parameter (SCR2_BITBANG_PARAMS param_select, unsigned int * param_value); 



// define the BITBANG vector types

typedef enum _SCR2_BITBANG_IOVECTOR_TYPES {

    SCR2_BITBANG_DURATION_VECTOR      = 0,
    SCR2_BITBANG_CONDITIONAL_VECTOR   = 1

} SCR2_BITBANG_IOVECTOR_TYPES;


typedef enum _SCR2_BITBANG_PORTVALS {

	// for all vector types
    SCR2_BITBANG_PORT_RELEASED     = 0,

	// for SCR2_BITBANG_DURATION_VECTOR
    SCR2_BITBANG_PORT_SET_OUT0     = 1,
    SCR2_BITBANG_PORT_SET_OUT1     = 2,
    SCR2_BITBANG_PORT_SET_IN       = 3,

	// for SCR2_BITBANG_CONDITIONAL_VECTOR
    SCR2_BITBANG_PORT_EXPECT_0     = 5,
    SCR2_BITBANG_PORT_EXPECT_1     = 6

} SCR2_BITBANG_PORTVALS;


typedef struct _SCR2_BITBANG_IOVECTOR {

    SCR2_BITBANG_IOVECTOR_TYPES vector_type;

	unsigned int vector_time_value_ns;  // 20 ... 1.000.000 (nsecs)

	unsigned char	vcc_control_enabled;	    // true          / false
	double  vcc_control_value;		    // 0 ... 6.0 (V) / don't care

    SCR2_BITBANG_PORTVALS CLK;
    SCR2_BITBANG_PORTVALS RST;
    SCR2_BITBANG_PORTVALS IO1;
    SCR2_BITBANG_PORTVALS IO2;
    SCR2_BITBANG_PORTVALS IO3;
    SCR2_BITBANG_PORTVALS IO4;
    SCR2_BITBANG_PORTVALS IO5;
    SCR2_BITBANG_PORTVALS IO6;
    SCR2_BITBANG_PORTVALS SWP;
    SCR2_BITBANG_PORTVALS TP1;
    SCR2_BITBANG_PORTVALS TP2;
    SCR2_BITBANG_PORTVALS EXT0;
    SCR2_BITBANG_PORTVALS EXT1;
    SCR2_BITBANG_PORTVALS EXT2;

} SCR2_BITBANG_IOVECTOR;


typedef enum _SCR2_BITBANG_CMD_CODES {

    SCR2_BITBANG_CMD_SET_R0      = 0x10,
    SCR2_BITBANG_CMD_SET_R1      = 0x11,
    SCR2_BITBANG_CMD_SET_R2      = 0x12,
    SCR2_BITBANG_CMD_SET_R3      = 0x13,

	SCR2_BITBANG_CMD_DJNZ_R0     = 0x20,
	SCR2_BITBANG_CMD_DJNZ_R1     = 0x21,
	SCR2_BITBANG_CMD_DJNZ_R2     = 0x22,
	SCR2_BITBANG_CMD_DJNZ_R3     = 0x23,

	SCR2_BITBANG_CMD_STOP_LOW    = 0x30,
	SCR2_BITBANG_CMD_STOP_HIGH   = 0x31

} SCR2_BITBANG_CMD_CODES;



typedef struct _SCR2_BITBANG_IOSTATES {

	unsigned int vector_duration_ns;  

	unsigned int CLK_VAL;
	unsigned int RST_VAL;
	unsigned int IO1_VAL;
	unsigned int IO2_VAL;
	unsigned int IO3_VAL;
	unsigned int IO4_VAL;
	unsigned int IO5_VAL;
	unsigned int IO6_VAL;
	unsigned int SWP_VAL;
	unsigned int TP1_VAL;
	unsigned int TP2_VAL;
	unsigned int EXT0_VAL;
	unsigned int EXT1_VAL;
	unsigned int EXT2_VAL;

} SCR2_BITBANG_IOSTATES;



typedef enum _SCR2_BITBANG_PORT_NAMES { 
	
    SCR2_BITBANG_PORT_CLK=0,
    SCR2_BITBANG_PORT_RST,
	SCR2_BITBANG_PORT_IO1,
    SCR2_BITBANG_PORT_IO2,
    SCR2_BITBANG_PORT_IO3,
    SCR2_BITBANG_PORT_IO4,
    SCR2_BITBANG_PORT_IO5,
    SCR2_BITBANG_PORT_IO6,
    SCR2_BITBANG_PORT_SWP,
    SCR2_BITBANG_PORT_TP1,
    SCR2_BITBANG_PORT_TP2,
    SCR2_BITBANG_PORT_EXT0,
    SCR2_BITBANG_PORT_EXT1,
    SCR2_BITBANG_PORT_EXT2

} SCR2_BITBANG_PORT_NAMES;



_EXPORT_ unsigned int scr2_bitbang_clear_vectors  (); 
_EXPORT_ unsigned int scr2_bitbang_add_io_vector  (SCR2_BITBANG_IOVECTOR  * bitbang_io_vector); 
_EXPORT_ unsigned int scr2_bitbang_add_cmd_vector (SCR2_BITBANG_CMD_CODES command, unsigned int parameter); 

_EXPORT_ unsigned int scr2_bitbang_get_iostates (unsigned int vector_number, SCR2_BITBANG_IOSTATES * bitbang_iostates); 

_EXPORT_ unsigned int scr2_bitbang_start_sequence (); 

_EXPORT_ unsigned int scr2_bitbang_dump_sequence (char * vcd_file_name); 






//////////////////////////////////////////////////////////////////////////////////
//
// Pattern Generator (PATGEN) functionality (EXPERIMENTAL, don't use yet)
//


// define the PATGEN error codes

typedef enum _SCR2_E_PATGEN {

    SCR2_E_PATGEN_GENERAL_ERROR   = 0x9400,
    SCR2_E_PATGEN_UNDERFLOW_ERROR, 
    SCR2_E_PATGEN_SYNC_ERROR, 
    SCR2_E_PATGEN_VECTOR_ERROR,
    SCR2_E_PATGEN_EXPECT_ERROR,
    SCR2_E_PATGEN_FILE_ERROR,
    SCR2_E_PATGEN_ATF_ERROR1,
    SCR2_E_PATGEN_ATF_ERROR2,
    SCR2_E_PATGEN_ATF_ERROR3,
    SCR2_E_PATGEN_STF_ERROR,
    SCR2_E_PATGEN_STF_FILESIZE,
    SCR2_E_PATGEN_TRIGGER_ERROR

} SCR2_E_PATGEN;


// define the PATGEN parameters

typedef enum _SCR2_PATGEN_PARAMS {

    SCR2_PATGEN_VECTOR_COUNT      = 0,		// read-only
    SCR2_PATGEN_CYCLE_COUNT,                // read-only
    SCR2_PATGEN_TEST_MODE,
    SCR2_PATGEN_TEST_CHKSUM,       		    // read-only
    SCR2_PATGEN_FREQSEL,
    SCR2_PATGEN_SIGNATURE                   // read-only

} SCR2_PATGEN_PARAMS;


_EXPORT_ unsigned int scr2_patgen_set_parameter (SCR2_PATGEN_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_patgen_get_parameter (SCR2_PATGEN_PARAMS param_select, unsigned int * param_value); 

_EXPORT_ unsigned int scr2_patgen_translate_atf (const char * atffilename);


// define the stf trigger parameters

typedef enum _SCR2_PATGEN_TRIGGER {

    SCR2_PATGEN_TRIGGER_OFF    = 0,	
    SCR2_PATGEN_TRIGGER_TRIG1,              
    SCR2_PATGEN_TRIGGER_TRIG2

} SCR2_PATGEN_TRIGGER;


_EXPORT_ unsigned int scr2_patgen_execute_stf (const char * stffilename, SCR2_PATGEN_TRIGGER trigger_mode, unsigned int trigger_cycle);


_EXPORT_ unsigned int scr2_patgen_release_ports ();




//////////////////////////////////////////////////////////////////////////////////
//
// TDA interface (trigger-delay-action)
//

// define the TDA error codes

typedef enum _SCR2_E_TDA {

    SCR2_E_TDA_GENERAL = 0x9D00,
    SCR2_E_TDA_UNUSED,
    SCR2_E_TDA_CONSISTENCY_ERROR,
    SCR2_E_TDA_RESOURCE_ERROR

} SCR2_E_TDA;


typedef enum _SCR2_TDA_TRIGGER_SOURCES {

	SCR2_TDA_TRIGGER_SOURCE_NO_TRIGGER        = 0,

	SCR2_TDA_TRIGGER_SOURCE_COMM_STARTED      = 1,

    SCR2_TDA_TRIGGER_SOURCE_TRIG1_RISING      = 4,
    SCR2_TDA_TRIGGER_SOURCE_TRIG1_FALLING     = 5,
    SCR2_TDA_TRIGGER_SOURCE_TRIG2_RISING      = 6,
    SCR2_TDA_TRIGGER_SOURCE_TRIG2_FALLING     = 7,

    SCR2_TDA_TRIGGER_SOURCE_TDA1_ACTION       = 8,
    SCR2_TDA_TRIGGER_SOURCE_TDA2_ACTION       = 9,
    SCR2_TDA_TRIGGER_SOURCE_TDA3_ACTION       = 10,
    SCR2_TDA_TRIGGER_SOURCE_TDA4_ACTION       = 11

} SCR2_TDA_TRIGGER_SOURCES;


typedef enum _SCR2_TDA_TRIGGER_ACTIONS {

	SCR2_TDA_TRIGGER_ACTION_POWER_OFF         = 1,
	SCR2_TDA_TRIGGER_ACTION_POWER_ON          = 2,
	SCR2_TDA_TRIGGER_ACTION_COMMUNICATION     = 3

    // planned :
	// SCR2_TDA_TRIGGER_ACTION_START_BITBANG  = 4


} SCR2_TDA_TRIGGER_ACTIONS;


_EXPORT_ unsigned int scr2_tda_clear_triggers  (); 

_EXPORT_ unsigned int scr2_tda_set_trigger  ( unsigned int             trigger_select,
										      SCR2_TDA_TRIGGER_SOURCES trigger_source,
											  unsigned int             trigger_delay,
									          SCR2_TDA_TRIGGER_ACTIONS trigger_action); 

typedef enum _SCR2_TDA_ABORT_CONDITIONS {

    SCR2_TDA_ABORT_DISABLED                   = 0,

    SCR2_TDA_ABORT_RAWI2C_RX_ADDR_ACK         = 1,
    SCR2_TDA_ABORT_RAWI2C_TX_ADDR_ACK         = 2,
    SCR2_TDA_ABORT_T1I2C_RX_ADDR_ACK          = 3,
    SCR2_TDA_ABORT_T1I2C_TX_ADDR_ACK          = 4

} SCR2_TDA_ABORT_CONDITIONS;


_EXPORT_ unsigned int scr2_tda_start  (unsigned int timeout_us, SCR2_TDA_ABORT_CONDITIONS abort_conditions); 

_EXPORT_ unsigned int scr2_tda_stop  (); 



typedef enum _SCR2_TDA_TRIGGER_STATUS {

	SCR2_TDA_TRIGGER_STATUS_TDA1_ACTIVE       = 0x00000001,
	SCR2_TDA_TRIGGER_STATUS_TDA2_ACTIVE       = 0x00000002,
	SCR2_TDA_TRIGGER_STATUS_TDA3_ACTIVE       = 0x00000004,
	SCR2_TDA_TRIGGER_STATUS_TDA4_ACTIVE       = 0x00000008,

    SCR2_TDA_TRIGGER_STATUS_TDA1_TRIGGERED    = 0x00000100,
	SCR2_TDA_TRIGGER_STATUS_TDA2_TRIGGERED    = 0x00000200,
	SCR2_TDA_TRIGGER_STATUS_TDA3_TRIGGERED    = 0x00000400,
	SCR2_TDA_TRIGGER_STATUS_TDA4_TRIGGERED    = 0x00000800,

    SCR2_TDA_TRIGGER_STATUS_TDA1_DONE         = 0x00010000,
	SCR2_TDA_TRIGGER_STATUS_TDA2_DONE         = 0x00020000,
	SCR2_TDA_TRIGGER_STATUS_TDA3_DONE         = 0x00040000,
	SCR2_TDA_TRIGGER_STATUS_TDA4_DONE         = 0x00080000,

  	SCR2_TDA_TRIGGER_STATUS_TIMEOUT           = 0x01000000,
  	SCR2_TDA_TRIGGER_STATUS_ABORTED           = 0x02000000,
  	/*SCR2_TDA_TRIGGER_STATUS_FINISHED          = 0x80000000*/


} SCR2_TDA_TRIGGER_STATUS;

_EXPORT_ unsigned int scr2_tda_get_status  ( unsigned int * tda_status); 



//////////////////////////////////////////////////////////////////////////////////
//
// FTB P73
//

// define the FTB error codes

typedef enum _SCR2_E_FTB {

    SCR2_E_FTB_INIT_FAILED = 0x8400,
    SCR2_E_FTB_EXCHANGE_TIMEOUT,
    SCR2_E_FTB_EXCHANGE_PARITYERROR,
    SCR2_E_FTB_EXCHANGE_RECEIVEERROR

} SCR2_E_FTB;


// define the FTB parameters

typedef enum _SCR2_FTB_PARAMS {

    SCR2_FTB_UART_FREQUENCY=0,
    SCR2_FTB_SEQ_START_TIME,
    SCR2_FTB_I2C_BITRATE,
    SCR2_FTB_I2C_SADDR,
    SCR2_FTB_MODE

} SCR2_FTB_PARAMS;


// SCR2_FTB_MODE values

#define SCR2_FTB_MODE_STANDARD                0
#define SCR2_FTB_MODE_EOS_GOANNA              1
#define SCR2_FTB_MODE_EOS_FPGA                2


_EXPORT_ unsigned int scr2_ftb_set_parameter (SCR2_FTB_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_ftb_get_parameter (SCR2_FTB_PARAMS param_select, unsigned int * param_value); 



_EXPORT_ unsigned int scr2_ftb_init          (unsigned int ftb_password, unsigned char ftb_option);
_EXPORT_ unsigned int scr2_ftb_sequence      (unsigned int ftb_password, unsigned char ftb_option);

_EXPORT_ unsigned int scr2_ftb_uart_exchange (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr2_ftb_uart_send     (unsigned char * databuffer, unsigned int   datalength);

_EXPORT_ unsigned int scr2_ftb_i2c_exchange  (unsigned char * databuffer, unsigned int commandlength, unsigned int responselength, double timeout_us);
_EXPORT_ unsigned int scr2_ftb_i2c_send      (unsigned char * databuffer, unsigned int commandlength);







//////////////////////////////////////////////////////////////////////////////////
//
// ISO7816 functions
//

// define the ISO7816 specific error codes

typedef enum _SCR2_E_ISO7816 {

    SCR2_E_ISO7816_PROTOCOL_NOT_SUPPORTED = 0x8100,		
    SCR2_E_ISO7816_ATR_CORRUPTED,
    SCR2_E_ISO7816_PROTOCOL_NOT_ACTIVE,
    SCR2_E_ISO7816_PPS_ERROR,					
    SCR2_E_ISO7816_TA1_VALUE_NOT_SUPPORTED,		
    SCR2_E_ISO7816_TIMEOUT,  					
    SCR2_E_ISO7816_PARITY_ERROR,                 
    SCR2_E_ISO7816_IFS_ERROR,					
    SCR2_E_ISO7816_STOPBIT_ERROR,				
    SCR2_E_ISO7816_T0_PARTLY,   				    
    SCR2_E_ISO7816_TA2_ERROR,   				    
    SCR2_E_ISO7816_TA2_MISMATCH,                 
    SCR2_E_ISO7816_TA2_BIT5SET,                  
    SCR2_E_ISO7816_NOATR_ERROR, 
    SCR2_E_ISO7816_ATR_TCK_MISSING,              
    SCR2_E_ISO7816_ATR_TCK_WRONG,                
    SCR2_E_ISO7816_ATR_HIST_BYTES_ERROR,         
    SCR2_E_ISO7816_ATR_WRONG_TS,                 
    SCR2_E_ISO7816_ATR_CODING_ERROR,             
    SCR2_E_ISO7816_ATR_WRONG_LENGTH,             
    SCR2_E_ISO7816_APDU_TIMEOUT,				    
    SCR2_E_ISO7816_APDU_T0_FRAME_TO_SHORT,       
    SCR2_E_ISO7816_APDU_T0_NO_DATA_AVAILABLE,    
    SCR2_E_ISO7816_APDU_T0_FRAME_LENGTH_ERROR,   
    SCR2_E_ISO7816_APDU_T0_FATAL_ERROR,          
    SCR2_E_ISO7816_APDU_T0_UNEXPECTED_SW1,       
    SCR2_E_ISO7816_APDU_T0_WRONG_CLA_BYTE,       
    SCR2_E_ISO7816_APDU_T0_WRONG_INS_BYTE,       
    SCR2_E_ISO7816_APDU_T0_NOT_SUPPORTED_CASE,   
    SCR2_E_ISO7816_APDU_T1_FATAL_ERROR,          
    SCR2_E_ISO7816_APDU_T1_FRAME_LENGTH_ERROR,   
    SCR2_E_ISO7816_APDU_T1_CRC_NOT_SUPPORTED,    
    SCR2_E_ISO7816_APDU_T1_LRC_ERROR,            
    SCR2_E_ISO7816_APDU_T1_TOO_MANY_R_FRAMES, 
    SCR2_E_ISO7816_WRONG_FREQUENCY,
    SCR2_E_RAW_NOT_SUPPORTED_FIDI,
    SCR2_E_RAW_EXCHANGE_TIMEOUT,
    SCR2_E_RAW_EXCHANGE_PARITY_ERROR,
    SCR2_E_RAW_EXCHANGE_RECEIVE_ERROR,
    SCR2_E_RAW_EXCHANGE_TEARERROR,
    SCR2_E_ISO7816_TEAR_TIMEOUT,
    SCR2_E_RAW_SEND_ERROR,
    SCR2_E_RAW_PULSE_ERROR

} SCR2_E_ISO7816;


typedef enum _SCR2_ISO7816_PARAMS {

    SCR2_ISO7816_FREQ=0,
    SCR2_ISO7816_CHM_VALUE,
    SCR2_ISO7816_CHM_TYPE,
    SCR2_ISO7816_FIDI,
    SCR2_ISO7816_F,                     // read-only
    SCR2_ISO7816_D,                     // read-only
    SCR2_ISO7816_CLKSETU,               // read-only
    SCR2_ISO7816_CONVENTION,            // read-only
    SCR2_ISO7816_SEND_SEQ_NR,
    SCR2_ISO7816_RECV_SEQ_NR,
    SCR2_ISO7816_PROTOCOL,
    SCR2_ISO7816_ENABLE_TWEAKS,
    SCR2_ISO7816_DISABLE_GET_RESPONSE,
    SCR2_ISO7816_FORCE_CLOCK_MODE,
    SCR2_ISO7816_ATR_WAIT_COUNT,
    SCR2_ISO7816_CLOCK_STOP_COUNT1,
    SCR2_ISO7816_CLOCK_STOP_COUNT2,
    SCR2_ISO7816_DISABLE_T1_ERROR_HANDLING,
    SCR2_ISO7816_COLD_RESET_TIME,
    SCR2_ISO7816_POWEROFFTIME,
    SCR2_ISO7816_RESETPULSELENGTH,
    SCR2_ISO7816_TESTMODE,
    SCR2_ISO7816_TIME_EXECUTION,        // read-only
    SCR2_ISO7816_TIME_COMM_SEND,        // read-only
    SCR2_ISO7816_TIME_COMM_RECV,        // read-only
    SCR2_ISO7816_TIME_HOST,             // read-only
    SCR2_ISO7816_TEARING_TIMEOUT,
    SCR2_ISO7816_TEARING_MODE,
    SCR2_ISO7816_ATR_TA1,               // read-only
    SCR2_ISO7816_ATR_FIRST_PROTOCOL,    // read-only
    SCR2_ISO7816_ATR_SECOND_PROTOCOL,   // read-only
    SCR2_ISO7816_ATR_CWI,               // read-only
    SCR2_ISO7816_ATR_BWI,               // read-only
    SCR2_ISO7816_ATR_WI,                // read-only                   
    SCR2_ISO7816_ATR_N,                 // read-only
    SCR2_ISO7816_ATR_X,                 // read-only
    SCR2_ISO7816_ATR_IFSC,              // read-only
    SCR2_ISO7816_SKIP_PPS,
    SCR2_ISO7816_PREFERRED_PROTOCOL,
    SCR2_ISO7816_PPS0_VALUE,
    SCR2_ISO7816_PPS0_MODE,
    SCR2_ISO7816_PPS1_VALUE,
    SCR2_ISO7816_PPS1_MODE,
    SCR2_ISO7816_PPS2_VALUE,
    SCR2_ISO7816_PPS2_MODE,
    SCR2_ISO7816_PPS3_VALUE,
    SCR2_ISO7816_PPS3_MODE,
    SCR2_ISO7816_PCK_VALUE,
    SCR2_ISO7816_PCK_MODE,
    SCR2_ISO7816_SKIP_IFS,
    SCR2_ISO7816_IFSD_MODE,
    SCR2_ISO7816_IFSD_VALUE,
    SCR2_ISO7816_CWT_MODE,
    SCR2_ISO7816_CWT_VALUE,
    SCR2_ISO7816_BWT_MODE,
    SCR2_ISO7816_BWT_VALUE,
    SCR2_ISO7816_WWT_MODE,
    SCR2_ISO7816_WWT_VALUE,
    SCR2_ISO7816_N_MODE,
    SCR2_ISO7816_N_VALUE,
    SCR2_ISO7816_X_MODE,
    SCR2_ISO7816_X_VALUE,
    SCR2_ISO7816_IFSC_MODE,
    SCR2_ISO7816_IFSC_VALUE

} SCR2_ISO7816_PARAMS;

// constants for tearing mode :

#define SCR2_ISO7816_TEARING_DISABLED    0x0000
#define SCR2_ISO7816_TEARING_ENABLED     0x0001

#define SCR2_ISO7816_TEARING_COLD        0x0010
#define SCR2_ISO7816_TEARING_WARM        0x0000

#define SCR2_ISO7816_TEARING_STOP        0x0100
#define SCR2_ISO7816_TEARING_CONT        0x0000

#define SCR2_ISO7816_TEARING_EXECONLY    0x0000
#define SCR2_ISO7816_TEARING_EXECCOMM    0x1000

#define SCR2_ISO7816_CHM_P6X             0
#define SCR2_ISO7816_CHM_P40             1
#define SCR2_ISO7816_CHM_P71             2

#define SCR2_ISO7816_CLOCK_AUTO          0      // default, clock controlled by UART
#define SCR2_ISO7816_CLOCK_FORCE_0       1      // force CLK to 0
#define SCR2_ISO7816_CLOCK_FORCE_1       2      // force CLK to 1

_EXPORT_ unsigned int scr2_iso7816_set_parameter (SCR2_ISO7816_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_iso7816_get_parameter (SCR2_ISO7816_PARAMS param_select, unsigned int * param_value); 


#define SCR2_ISO7816_COLD_ACTIVATION                0
#define SCR2_ISO7816_WARM_ACTIVATION                1

_EXPORT_ unsigned int scr2_iso7816_raw_activate    (unsigned char power_mode);
_EXPORT_ unsigned int scr2_iso7816_raw_deactivate  ();
_EXPORT_ unsigned int scr2_iso7816_raw_exchange    (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr2_iso7816_raw_exchange_t0 (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr2_iso7816_raw_send        (unsigned char * databuffer, unsigned int   datalength);
_EXPORT_ unsigned int scr2_iso7816_raw_receive     (unsigned char * databuffer, unsigned int * datalength, double timeout_us);
_EXPORT_ unsigned int scr2_iso7816_raw_resetpulse  ();

_EXPORT_ unsigned int scr2_iso7816_get_atr (unsigned char * atr_ptr, unsigned int * atr_length);

#define SCR2_ISO7816_FORCE_T0_PROTOCOL              0
#define SCR2_ISO7816_FORCE_T1_PROTOCOL              1
#define SCR2_ISO7816_AUTO_PROTOCOL                  2

_EXPORT_ unsigned int scr2_iso7816_protocol_activate (unsigned char protocol, unsigned char power_mode); 
_EXPORT_ unsigned int scr2_iso7816_protocol_pps      (unsigned char protocol);
_EXPORT_ unsigned int scr2_iso7816_protocol_ifs      ();

_EXPORT_ unsigned int scr2_iso7816_protocol_deactivate ();
_EXPORT_ unsigned int scr2_iso7816_protocol_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr2_iso7816_sframe_exchange (unsigned char command, unsigned char * databuffer, unsigned int * length);




//////////////////////////////////////////////////////////////////////////////////
//
// T1SPI, "T1 over SPI"
//
//

// define the T1SPI error codes

typedef enum _SCR2_E_T1SPI {

    SCR2_E_T1SPI_FRAMELENGTHERROR = 0x8200,
    SCR2_E_T1SPI_TIMEOUT,
    SCR2_E_T1SPI_GENERALERROR,
    SCR2_E_T1SPI_EDCERROR,
    SCR2_E_T1SPI_FATALERROR,
    SCR2_E_T1SPI_TOO_MANY_R_FRAMES,
    SCR2_E_T1SPI_TOO_MANY_REPEATS,
    SCR2_E_T1SPI_RECEIVESTREAM,
    SCR2_E_T1SPI_NO_TIME_AVAILABLE,
    SCR2_E_T1SPI_TEARINGEVENT,
    SCR2_E_T1SPI_NADERROR,
    SCR2_E_T1SPI_SFRAMEERROR,
    SCR2_E_T1SPI_PROTOCOLERROR,
    SCR2_E_T1SPI_LRCERROR,
    SCR2_E_T1SPI_CRCERROR,
    SCR2_E_T1SPI_NOTSUPPORTED,
    SCR2_E_T1SPI_OS_REMOVED,
    SCR2_E_T1SPI_TRIGGER_TIMEOUT,
    SCR2_E_T1SPI_POWER_OFF_ERROR

} SCR2_E_T1SPI;

typedef enum _SCR2_T1SPI_PARAMS {

    SCR2_T1SPI_BITRATE=0,
    SCR2_T1SPI_SEND_SEQ_NR,
    SCR2_T1SPI_RECV_SEQ_NR,
    SCR2_T1SPI_BWT,
    SCR2_T1SPI_IFS,
    SCR2_T1SPI_PROTOCOLMODE,
    SCR2_T1SPI_LOGICALINTERFACE,
    SCR2_T1SPI_CS_START_DELAY,
    SCR2_T1SPI_CS_STOP_DELAY,
    SCR2_T1SPI_TIME_EXECUTION,              // read-only
    SCR2_T1SPI_TIME_COMM_SEND,              // read-only
    SCR2_T1SPI_TIME_COMM_RECV,              // read-only
    SCR2_T1SPI_TIME_HOST,                   // read-only
    SCR2_T1SPI_TEARING_TIMEOUT,
    SCR2_T1SPI_TEARING_MODE,
    SCR2_T1SPI_T1_DISABLE_ERROR_HANDLING,
    SCR2_T1SPI_POLLING_TIME_MS,
    SCR2_GPSPI_POLLING_TIME_US,
    SCR2_T1SPI_NAD_LRC,
    SCR2_T1SPI_USE_CRC,
    SCR2_T1SPI_GP_MODE,
    SCR2_T1SPI_SEND_NAD_VALUE,
    SCR2_T1SPI_RECV_NAD_VALUE,
    SCR2_T1SPI_GP_IFSC

} SCR2_T1SPI_PARAMS;


// constants for tearing mode :

#define SCR2_T1SPI_TEARING_DISABLED                  0x0000
#define SCR2_T1SPI_TEARING_ENABLED                   0x0001

#define SCR2_T1SPI_TEARING_COLD                      0x0010
#define SCR2_T1SPI_TEARING_WARM                      0x0000

#define SCR2_T1SPI_TEARING_STOP                      0x0100
#define SCR2_T1SPI_TEARING_CONT                      0x0000

#define SCR2_T1SPI_TEARING_EXECONLY                  0x0000
#define SCR2_T1SPI_TEARING_EXECCOMM                  0x1000

// Protocol modes :

#define SCR2_T1SPI_PROTOCOLMODE_NORMAL_5A            2
#define SCR2_T1SPI_PROTOCOLMODE_P73_5A               3
#define SCR2_T1SPI_PROTOCOLMODE_READY_HIGH_5A        4
#define SCR2_T1SPI_PROTOCOLMODE_READY_LOW_5A         5

#define SCR2_T1SPI_LOGIF_SPI                         0
#define SCR2_T1SPI_LOGIF_VIRTUALISO                  1

_EXPORT_ unsigned int scr2_t1spi_set_parameter (SCR2_T1SPI_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_t1spi_get_parameter (SCR2_T1SPI_PARAMS param_select, unsigned int * param_value); 



#define SCR2_T1SPI_POWER_CYCLE_DISABLED              0
#define SCR2_T1SPI_POWER_CYCLE_ENABLED               1
#define SCR2_T1SPI_POWER_MODE_ON                     0
#define SCR2_T1SPI_POWER_MODE_OFF                    1

_EXPORT_ unsigned int scr2_t1spi_protocol_activate   ( unsigned char powercycle );
_EXPORT_ unsigned int scr2_t1spi_protocol_deactivate ( unsigned char powermode );

_EXPORT_ unsigned int scr2_t1spi_set_ifs (unsigned int ifs);

_EXPORT_ unsigned int scr2_t1spi_protocol_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr2_t1spi_raw_exchange      (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr2_t1spi_raw_send          (unsigned char * databuffer, unsigned int   length);
_EXPORT_ unsigned int scr2_t1spi_raw_read          (unsigned char * databuffer, unsigned int   length);


#define SCR2_T1SPI_SFRAME_INTERFACERESET            0
#define SCR2_T1SPI_SFRAME_ENDOFSESSION              1
#define SCR2_T1SPI_SFRAME_GETATR                    2
#define SCR2_T1SPI_SFRAME_GETPOWERSTATE             3
#define SCR2_T1SPI_SFRAME_GETPROTOCOLVERSION        4

_EXPORT_ unsigned int scr2_t1spi_sframe_exchange (unsigned char sframetype, unsigned char * infbuffer, unsigned int * inflength);

_EXPORT_ unsigned int scr2_t1spi_generic_sframe_exchange (unsigned char command, unsigned char * infbuffer, unsigned int * inflength);








//////////////////////////////////////////////////////////////////////////////////
//
// T1I2C, "T1 over I2C"
//
//

// define the T1I2C error codes

typedef enum _SCR2_E_T1I2C {

    SCR2_E_T1I2C_FRAMELENGTHERROR = 0x8700,
    SCR2_E_T1I2C_TIMEOUT,
    SCR2_E_T1I2C_NACKERROR,
    SCR2_E_T1I2C_CLOCKSTRETCHING,
    SCR2_E_T1I2C_GENERALERROR,
    SCR2_E_T1I2C_LRCERROR,
    SCR2_E_T1I2C_FATALERROR,
    SCR2_E_T1I2C_TOO_MANY_R_FRAMES,
    SCR2_E_T1I2C_TOO_MANY_REPEATS,
    SCR2_E_T1I2C_RECEIVESTREAM,
    SCR2_E_T1I2C_SFRAMEERROR,
    SCR2_E_T1I2C_NO_TIME_AVAILABLE,
    SCR2_E_T1I2C_TEARINGEVENT,
    SCR2_E_T1I2C_HSACKERROR,
    SCR2_E_T1I2C_NADERROR,
    SCR2_E_T1I2C_OVERFLOWERROR,
    SCR2_E_T1I2C_TRIGGER_TIMEOUT,
    SCR2_E_T1I2C_PROTOCOLERROR,
    SCR2_E_T1I2C_CRCERROR,
    SCR2_E_T1I2C_CIPCONSISTENCYERROR,
    SCR2_E_T1I2C_NOTSUPPORTED,
    SCR2_E_T1I2C_OS_REMOVED,
    SCR2_E_T1I2C_POWER_OFF_ERROR

} SCR2_E_T1I2C;

typedef enum _SCR2_T1I2C_PARAMS {

    SCR2_T1I2C_BITRATE=0,
    SCR2_T1I2C_LOGICALINTERFACE,
    SCR2_T1I2C_SLAVE_ADDRESS,
    SCR2_T1I2C_NAD_LRC,
    SCR2_T1I2C_USE_CRC,
    SCR2_T1I2C_GP_MODE,
    SCR2_T1I2C_DIS_HS_STRETCH,
    SCR2_T1I2C_DIS_ALL_STRETCH,
    SCR2_T1I2C_SEND_NAD_VALUE,
    SCR2_T1I2C_RECV_NAD_VALUE,
    SCR2_T1I2C_TXRETRYTIME,
    SCR2_T1I2C_POLLING_TIME_MS,
    SCR2_GPI2C_POLLING_TIME_US,
    SCR2_GPI2C_IFSC,
    SCR2_T1I2C_HSMASTERCODE,
    SCR2_T1I2C_SEND_SEQ_NR,
    SCR2_T1I2C_RECV_SEQ_NR,
    SCR2_T1I2C_BWT,
    SCR2_T1I2C_TIME_EXECUTION,              // read-only
    SCR2_T1I2C_TIME_COMM_SEND,              // read-only
    SCR2_T1I2C_TIME_COMM_RECV,              // read-only
    SCR2_T1I2C_TIME_HOST,                   // read-only
    SCR2_T1I2C_TEARING_TIMEOUT,
    SCR2_T1I2C_TEARING_MODE,
    SCR2_T1I2C_T1_DISABLE_ERROR_HANDLING,
    SCR2_T1I2C_DIS_HS_MODE,
    SCR2_T1I2C_FORCE_HS_MODE,
    SCR2_T1I2C_SEND_INDELAY,
    SCR2_T1I2C_RECV_INDELAY,
    SCR2_T1I2C_RECV_STOP_DELAY

} SCR2_T1I2C_PARAMS;


// constants for tearing mode :

#define SCR2_T1I2C_TEARING_DISABLED                  0x0000
#define SCR2_T1I2C_TEARING_ENABLED                   0x0001

#define SCR2_T1I2C_TEARING_STOP                      0x0100
#define SCR2_T1I2C_TEARING_CONT                      0x0000

#define SCR2_T1I2C_TEARING_EXECONLY                  0x0000
#define SCR2_T1I2C_TEARING_EXECCOMM                  0x1000

#define SCR2_T1I2C_LOGIF_SE                          0
#define SCR2_T1I2C_LOGIF_UICC                        1

_EXPORT_ unsigned int scr2_t1i2c_set_parameter (SCR2_T1I2C_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_t1i2c_get_parameter (SCR2_T1I2C_PARAMS param_select, unsigned int * param_value); 



#define SCR2_T1I2C_POWER_CYCLE_DISABLED              0
#define SCR2_T1I2C_POWER_CYCLE_ENABLED               1
#define SCR2_T1I2C_POWER_MODE_ON                     0
#define SCR2_T1I2C_POWER_MODE_OFF                    1

_EXPORT_ unsigned int scr2_t1i2c_protocol_activate   ( unsigned char powercycle );
_EXPORT_ unsigned int scr2_t1i2c_protocol_deactivate ( unsigned char powermode );

_EXPORT_ unsigned int scr2_t1i2c_protocol_exchange (unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr2_t1i2c_sframe_exchange (unsigned char command, unsigned char * databuffer, unsigned int * length);

_EXPORT_ unsigned int scr2_t1i2c_reset_protocol ();

_EXPORT_ unsigned int scr2_t1i2c_set_ifs (unsigned int ifs);
_EXPORT_ unsigned int scr2_t1i2c_end_of_session (unsigned char * databuffer, unsigned int * length);
_EXPORT_ unsigned int scr2_t1i2c_soft_reset (unsigned char * databuffer, unsigned int * length);
_EXPORT_ unsigned int scr2_t1i2c_get_atr (unsigned char * databuffer, unsigned int * length);



//////////////////////////////////////////////////////////////////////////////////
//
// I2CSLAVE
//
//

// define the I2CSLAVE error codes

typedef enum _SCR2_E_I2CSLAVE {

    SCR2_E_I2CSLAVE_GENERAL_ERROR = 0x9000

} SCR2_E_I2CSLAVE;


// define the I2CSLAVE parameters

typedef enum _SCR2_I2CSLAVE_PARAMS {

    SCR2_I2CSLAVE_SLAVE_ENABLE=0,
    SCR2_I2CSLAVE_SLAVE_ADDRESS,
	SCR2_I2CSLAVE_SCL_FILTER_LENGTH,
	SCR2_I2CSLAVE_SDA_FILTER_LENGTH,
	SCR2_I2CSLAVE_MEM_PROG_TIME,
	SCR2_I2CSLAVE_MEM_PAGE_SIZE,
	SCR2_I2CSLAVE_MEM_BUSY_MODE

} SCR2_I2CSLAVE_PARAMS;


_EXPORT_ unsigned int scr2_i2cslave_set_parameter (SCR2_I2CSLAVE_PARAMS param_select, unsigned int   param_value);
_EXPORT_ unsigned int scr2_i2cslave_get_parameter (SCR2_I2CSLAVE_PARAMS param_select, unsigned int * param_value);

_EXPORT_ unsigned int scr2_i2cslave_set_memory (unsigned char * mem_buffer, unsigned int mem_address, unsigned int mem_length);
_EXPORT_ unsigned int scr2_i2cslave_get_memory (unsigned char * mem_buffer, unsigned int mem_address, unsigned int mem_length);



//////////////////////////////////////////////////////////////////////////////////
//
// SCSPI, "Smart Card SPI"
//
//

// define the SCSPI error codes

typedef enum _SCR2_E_SCSPI {

    SCR2_E_SCSPI_RECV_TIMEOUT = 0x8A00,
    SCR2_E_SCSPI_FRAMELENGTHERROR,
    SCR2_E_SCSPI_NO_ADDR_ACK,
    SCR2_E_SCSPI_PARAMETER_EXCHANGE,
    SCR2_E_SCSPI_GENERALERROR,
    SCR2_E_SCSPI_SEND_ERROR,
    SCR2_E_SCSPI_APDU_TIMEOUT,
    SCR2_E_SCSPI_ATR_CODING_ERROR,
    SCR2_E_SCSPI_GENERAL_STATE_ERROR,
    SCR2_E_SCSPI_PROTOCOL_EXCEPTION,
    SCR2_E_SCSPI_WRONG_PCB,
    SCR2_E_SCSPI_RECV_SEQUENCE_ERROR,
    SCR2_E_SCSPI_ATR_WRONG_TAG,
    SCR2_E_SCSPI_ATR_CORRUPT_TAG,
    SCR2_E_SCSPI_NO_ATR,
    SCR2_E_SCSPI_PE_PARAM_ERROR,
    SCR2_E_SCSPI_PE_WRONG_PCB

} SCR2_E_SCSPI;

typedef enum _SCR2_SCSPI_PARAMS {

    SCR2_SCSPI_BITRATE=0,
    SCR2_SCSPI_SUB_FRAME_SIZE,         
    SCR2_SCSPI_SEND_COUNTER,
    SCR2_SCSPI_RECV_COUNTER,
    SCR2_SCSPI_CDB_MS,
    SCR2_SCSPI_CDB_SM,
    SCR2_SCSPI_COMM_SPEED,              // read-only
    SCR2_SCSPI_SDPV,                    // read-only
    SCR2_SCSPI_LRC_SUPPORTED,           // read-only
    SCR2_SCSPI_FWI,                     // read-only
    SCR2_SCSPI_HLDO,                    // read-only

    SCR2_SCSPI_TIME_EXECUTION,          // read-only
    SCR2_SCSPI_TIME_COMM_SEND,          // read-only
    SCR2_SCSPI_TIME_COMM_RECV,          // read-only
    SCR2_SCSPI_TIME_HOST,               // read-only
    SCR2_SCSPI_TEARING_TIMEOUT,
    SCR2_SCSPI_TEARING_MODE

} SCR2_SCSPI_PARAMS;



// constants for SCSPI tearing mode :

#define SCR2_SCSPI_TEARING_DISABLED                  0x0000
#define SCR2_SCSPI_TEARING_ENABLED                   0x0001

#define SCR2_SCSPI_TEARING_COLD                      0x0010
#define SCR2_SCSPI_TEARING_WARM                      0x0000

#define SCR2_SCSPI_TEARING_STOP                      0x0100
#define SCR2_SCSPI_TEARING_CONT                      0x0000

#define SCR2_SCSPI_TEARING_EXECONLY                  0x0000
#define SCR2_SCSPI_TEARING_EXECCOMM                  0x1000


_EXPORT_ unsigned int scr2_scspi_set_parameter (SCR2_SCSPI_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_scspi_get_parameter (SCR2_SCSPI_PARAMS param_select, unsigned int * param_value); 

#define SCR2_SCSPI_POWER_CYCLE_DISABLED               0
#define SCR2_SCSPI_POWER_CYCLE_ENABLED                1

_EXPORT_ unsigned int scr2_scspi_protocol_activate   ( unsigned char powercycle );

#define SCR2_SCSPI_POWER_MODE_ON                      0
#define SCR2_SCSPI_POWER_MODE_OFF                     1

_EXPORT_ unsigned int scr2_scspi_protocol_deactivate ( unsigned char powermode );

_EXPORT_ unsigned int scr2_scspi_protocol_exchange (unsigned char * databuffer, unsigned int * length);







//////////////////////////////////////////////////////////////////////////////////
//
// SCI2C, "Smart Card I2C"
//
//

// define the SCI2C error codes

typedef enum _SCR2_E_SCI2C {

    SCR2_E_SCI2C_RECV_TIMEOUT = 0x8F00,
    SCR2_E_SCI2C_FRAMELENGTHERROR,
    SCR2_E_SCI2C_NO_ADDR_ACK,
    SCR2_E_SCI2C_PARAMETER_EXCHANGE,
    SCR2_E_SCI2C_GENERALERROR,
    SCR2_E_SCI2C_SEND_ERROR,
    SCR2_E_SCI2C_APDU_TIMEOUT,
    SCR2_E_SCI2C_ATR_CODING_ERROR,
    SCR2_E_SCI2C_GENERAL_STATE_ERROR,
    SCR2_E_SCI2C_PROTOCOL_EXCEPTION,
    SCR2_E_SCI2C_WRONG_PCB,
    SCR2_E_SCI2C_RECV_SEQUENCE_ERROR,
    SCR2_E_SCI2C_ATR_WRONG_TAG,
    SCR2_E_SCI2C_ATR_CORRUPT_TAG,
    SCR2_E_SCI2C_NO_ATR,
    SCR2_E_SCI2C_PE_PARAM_ERROR,
    SCR2_E_SCI2C_PE_WRONG_PCB,
    SCR2_E_SCI2C_TEAR_TIMEOUT,
    SCR2_E_SCI2C_POWER_OFF_ERROR

} SCR2_E_SCI2C;

typedef enum _SCR2_SCI2C_PARAMS {

    SCR2_SCI2C_BITRATE=0,
    SCR2_SCI2C_SLAVEADDRESS,
    SCR2_SCI2C_SEND_COUNTER,
    SCR2_SCI2C_RECV_COUNTER,
    SCR2_SCI2C_CDB_MS,
    SCR2_SCI2C_CDB_SM,
    SCR2_SCI2C_COMM_SPEED,              // read-only
    SCR2_SCI2C_SDPV,                    // read-only
    SCR2_SCI2C_LRC_SUPPORTED,           // read-only
    SCR2_SCI2C_FWI,                     // read-only
    SCR2_SCI2C_HLDO,                    // read-only

    SCR2_SCI2C_TIME_EXECUTION,          // read-only
    SCR2_SCI2C_TIME_COMM_SEND,          // read-only
    SCR2_SCI2C_TIME_COMM_RECV,          // read-only
    SCR2_SCI2C_TIME_HOST,               // read-only
    SCR2_SCI2C_TEARING_TIMEOUT,
    SCR2_SCI2C_TEARING_MODE

} SCR2_SCI2C_PARAMS;




// constants for SCI2C tearing mode :

#define SCR2_SCI2C_TEARING_DISABLED                  0x0000
#define SCR2_SCI2C_TEARING_ENABLED                   0x0001

#define SCR2_SCI2C_TEARING_COLD                      0x0010
#define SCR2_SCI2C_TEARING_WARM                      0x0000

#define SCR2_SCI2C_TEARING_STOP                      0x0100
#define SCR2_SCI2C_TEARING_CONT                      0x0000

#define SCR2_SCI2C_TEARING_EXECONLY                  0x0000
#define SCR2_SCI2C_TEARING_EXECCOMM                  0x1000


_EXPORT_ unsigned int scr2_sci2c_set_parameter (SCR2_SCI2C_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_sci2c_get_parameter (SCR2_SCI2C_PARAMS param_select, unsigned int * param_value); 

#define SCR2_SCI2C_POWER_CYCLE_DISABLED               0
#define SCR2_SCI2C_POWER_CYCLE_ENABLED                1

_EXPORT_ unsigned int scr2_sci2c_protocol_activate   ( unsigned char powercycle );

#define SCR2_SCI2C_POWER_MODE_ON                      0
#define SCR2_SCI2C_POWER_MODE_OFF                     1

_EXPORT_ unsigned int scr2_sci2c_protocol_deactivate ( unsigned char powermode );

_EXPORT_ unsigned int scr2_sci2c_protocol_exchange (unsigned char * databuffer, unsigned int * length);




//////////////////////////////////////////////////////////////////////////////////
//
//  SWP
//
//

// define the RF error codes

typedef enum _SCR2_E_RF {

    SCR2_E_RF_GENERAL_ERROR = 0x8E00,
    SCR2_E_RF_NO_FIELD_PRESENT,
    SCR2_E_RF_TIMEOUT,
    SCR2_E_RF_FRAME_LENGTH_ERROR,
    SCR2_E_RF_CRC_ERROR,
    SCR2_E_RF_RECEIVE_ERROR,
    SCR2_E_RF_TEARINGEVENT,
    SCR2_E_RF_ATQA_ERROR,
    SCR2_E_RF_UID_ERROR,
    SCR2_E_RF_ANTICOL_ERROR,
    SCR2_E_RF_BAUDRATE_NOT_SUPPORTED,
    SCR2_E_RF_PPS_ERROR,
    SCR2_E_RF_DESELECT_ERROR,
    SCR2_E_RF_ATS_CONSISTENCY_ERROR,
    SCR2_E_RF_ATQB_ERROR,
    SCR2_E_RF_TOO_MANY_R_FRAMES,
    SCR2_E_RF_TOO_MANY_REPEATS,
    SCR2_E_RF_PROTOCOL_ERROR,
    SCR2_E_RF_PROTOCOL_NOT_SUPPORTED,
    SCR2_E_RF_ATTRIB_ERROR,
    SCR2_E_RF_MIFARE_AUTHENT_ERROR,
    SCR2_E_RF_MIFARE_READ_ERROR,
    SCR2_E_RF_MIFARE_WRITE_ERROR,
    SCR2_E_RF_MIFARE_INCREMENT_ERROR,
    SCR2_E_RF_MIFARE_DECREMENT_ERROR,
    SCR2_E_RF_MIFARE_TRANSFER_ERROR,
    SCR2_E_RF_MIFARE_RESTORE_ERROR,
    SCR2_E_RF_FELICA_RESPONSE_CODE_ERROR,
    SCR2_E_RF_FELICA_KEY_NOT_FOUND,
    SCR2_E_RF_FELICA_INVALID_SYSTEM_CODE,
    SCR2_E_RF_FELICA_AUTHENTICATE1_ERROR,
    SCR2_E_RF_FELICA_AUTHENTICATE2_ERROR,
	SCR2_E_RF_DESFIRE_UNEXPECTED_RESPONSE,
    SCR2_E_RF_DESFIRE_WRONG_SW1,
    SCR2_E_RF_DESFIRE_WRONG_SW2,
    SCR2_E_RF_DESFIRE_MAC_ERROR,
	SCR2_E_RF_DESFIRE_CRC_ERROR,
	SCR2_E_RF_DESFIRE_NO_AUTHENTICATION,
	SCR2_E_RF_DESFIRE_RESPONSE_BUFFER,
	SCR2_E_RF_DESFIRE_INTEGRITY_ERROR,
	SCR2_E_RF_DESFIRE_PERMISSION_DENIED,
	SCR2_E_RF_DESFIRE_PARAMETER_ERROR,
	SCR2_E_RF_DESFIRE_LENGTH_ERROR

} SCR2_E_RF;



typedef enum _SCR2_RF_PARAMS {

    SCR2_RF_FIELDOFF_TIME=0,
    SCR2_RF_FIELDGUARD_TIME,
    SCR2_RF_LAST_BITS_RECV,
    SCR2_RF_COLLISION_INFO,

    SCR2_RF_TIME_EXECUTION,          // read-only
    SCR2_RF_TIME_COMM_SEND,          // read-only
    SCR2_RF_TIME_COMM_RECV,          // read-only

    SCR2_RF_TEARING_TIMEOUT,
    SCR2_RF_TEARING_MODE,

    SCR2_RF_PROTOCOL,
    SCR2_RF_TX_BAUDRATE,
    SCR2_RF_RX_BAUDRATE,

    SCR2_RF_FSDI,
    SCR2_RF_FSCI,
    SCR2_RF_FWI,
    SCR2_RF_SFGI,

    SCR2_RF_CID_SUPPORTED,
    SCR2_RF_CID,
    SCR2_RF_CID_ENABLED,

    SCR2_RF_NAD_SUPPORTED,
    SCR2_RF_NAD,
    SCR2_RF_NAD_ENABLED,

    SCR2_RF_BLOCK_NUMBER,
    SCR2_RF_DISABLE_PROTOCOL_ERROR_HANDLING,


    SCR2_RF_TYPE_A_COLLMODE,

    SCR2_RF_TYPE_B_AFI,
    SCR2_RF_TYPE_B_PARAM,

	SCR2_RF_DESFIRE_ENABLE_MFC

} SCR2_RF_PARAMS;



// constants for tearing mode :

#define SCR2_RF_TEARING_DISABLED                  0x0000
#define SCR2_RF_TEARING_ENABLED                   0x0001

#define SCR2_RF_TEARING_EXECONLY                  0x0000
#define SCR2_RF_TEARING_EXECCOMM                  0x1000


_EXPORT_ unsigned int scr2_rf_set_parameter (SCR2_RF_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_rf_get_parameter (SCR2_RF_PARAMS param_select, unsigned int * param_value); 





_EXPORT_ unsigned int scr2_rf_field_reset ();
_EXPORT_ unsigned int scr2_rf_field_on ();
_EXPORT_ unsigned int scr2_rf_field_off ();



typedef enum _SCR2_RF_BAUDRATES {

    SCR2_RF_BR_106 = 0,
    SCR2_RF_BR_212 = 1,
    SCR2_RF_BR_424 = 2,
    SCR2_RF_BR_848 = 3

} SCR2_RF_BAUDRATES;

typedef enum _SCR2_RF_PROTOCOLS {

    SCR2_RF_PROTOCOL_TYPE_A = 0,
    SCR2_RF_PROTOCOL_TYPE_B = 1,
    SCR2_RF_PROTOCOL_TYPE_F = 2

} SCR2_RF_PROTOCOLS;

_EXPORT_ unsigned int scr2_rf_set_protocol ( SCR2_RF_PROTOCOLS rf_protocol,
                                             SCR2_RF_BAUDRATES tx_baudrate,
                                             SCR2_RF_BAUDRATES rx_baudrate  );



typedef enum _SCR2_RF_OPTIONS {

    SCR2_RF_CRC_OFF=0,
    SCR2_RF_CRC_ON,
    SCR2_RF_FIELD_RESET_DISABLED,
    SCR2_RF_FIELD_RESET_ENABLED,
    SCR2_RF_TYPEA_ACTIVATE_L3,
    SCR2_RF_TYPEA_ACTIVATE_L4,
    SCR2_RF_MIFARE_KEYA,
    SCR2_RF_MIFARE_KEYB,
    SCR2_RF_REQA,
    SCR2_RF_WUPA,
    SCR2_RF_REQB,
    SCR2_RF_WUPB

} SCR2_RF_OPTIONS;


_EXPORT_ unsigned int scr2_rf_exchange_frame ( unsigned char * databuffer, unsigned int  * length,
                                               unsigned char   txlastbits, unsigned char   rxalign,
                                               unsigned int    timeout_us,
                                               SCR2_RF_OPTIONS crc_on,     SCR2_RF_OPTIONS field_reset);


typedef struct _SCR2_RF_TYPE_A_DATA {

    unsigned char SAK;
    unsigned char ATQA[2];
    unsigned char UID[10];
    unsigned char UIDLEN;
    unsigned char ATS[256];
    unsigned char ATSLEN;

} SCR2_RF_TYPE_A_DATA;


_EXPORT_ unsigned int scr2_rf_activate_type_a ( SCR2_RF_BAUDRATES tx_baudrate, SCR2_RF_BAUDRATES rx_baudrate, SCR2_RF_OPTIONS activation_layer, SCR2_RF_OPTIONS req_wup_select );
_EXPORT_ unsigned int scr2_rf_get_type_a_data ( SCR2_RF_TYPE_A_DATA * type_a_data );


typedef struct _SCR2_RF_TYPE_B_DATA {

    unsigned char PUPI[4];
    
    // TBD

} SCR2_RF_TYPE_B_DATA;

_EXPORT_ unsigned int scr2_rf_activate_type_b ( SCR2_RF_BAUDRATES tx_baudrate, SCR2_RF_BAUDRATES rx_baudrate, SCR2_RF_OPTIONS req_wup_select );
_EXPORT_ unsigned int scr2_rf_get_type_b_data ( SCR2_RF_TYPE_B_DATA * type_b_data );


_EXPORT_ unsigned int scr2_rf_protocol_exchange (unsigned char * databuffer, unsigned int * length);
_EXPORT_ unsigned int scr2_rf_protocol_deselect ();




//////////////////////////////////////////////////////////////////////////////////
//
//  MIFARE
//
//


_EXPORT_ unsigned int scr2_rf_mifare_authent   (unsigned char block, unsigned char * key, unsigned char * uid, SCR2_RF_OPTIONS key_select);
_EXPORT_ unsigned int scr2_rf_mifare_disable_crypto ();
_EXPORT_ unsigned int scr2_rf_mifare_read      (unsigned char block, unsigned char * blockdata);
_EXPORT_ unsigned int scr2_rf_mifare_write     (unsigned char block, unsigned char * blockdata);
_EXPORT_ unsigned int scr2_rf_mifare_increment (unsigned char block, unsigned int value);
_EXPORT_ unsigned int scr2_rf_mifare_decrement (unsigned char block, unsigned int value);
_EXPORT_ unsigned int scr2_rf_mifare_transfer  (unsigned char block);
_EXPORT_ unsigned int scr2_rf_mifare_restore   (unsigned char block);





/////////////////////////////////////////////////////////////////////////////////
//
//  FeliCa
//
//

typedef struct _SCR2_RF_TYPE_F_DATA {

    unsigned short systemcode;
    unsigned char  IDm[8];
    unsigned char  PMm[8];

} SCR2_RF_TYPE_F_DATA;


_EXPORT_ unsigned int scr2_rf_felica_add_des_key ( unsigned short keyid, unsigned char * keyvalue);

_EXPORT_ unsigned int scr2_rf_felica_polling ( unsigned short systemcode, unsigned char timeslot);

_EXPORT_ unsigned int scr2_rf_felica_authentication ( unsigned short * area_codes, unsigned int area_code_count,
                                                      unsigned short * service_codes, unsigned int service_code_count);




//////////////////////////////////////////////////////////////////////////////////
//
//  Desfire
//
//


typedef enum _SCR2_RF_DESIFRE_OPTIONS {

    SCR2_RF_DESFIRE_AUTHENT_D40=0,
	SCR2_RF_DESFIRE_AUTHENT_ISO,
	SCR2_RF_DESFIRE_AUTHENT_AES,
	SCR2_RF_DESFIRE_AUTHENT_EV2,

    SCR2_RF_DESFIRE_COMMMODE_PLAIN,
    SCR2_RF_DESFIRE_COMMMODE_MAC,
    SCR2_RF_DESFIRE_COMMMODE_MAC_OPT_A,
    SCR2_RF_DESFIRE_COMMMODE_MAC_OPT_B,
    SCR2_RF_DESFIRE_COMMMODE_FULL,

    SCR2_RF_DESFIRE_KEYTYPE_2TDEA,
    SCR2_RF_DESFIRE_KEYTYPE_3TDEA,
    SCR2_RF_DESFIRE_KEYTYPE_AES

} SCR2_RF_DESIFRE_OPTIONS;


_EXPORT_ unsigned int scr2_rf_desfire_select ( unsigned char * AID );

_EXPORT_ unsigned int scr2_rf_desfire_authenticate ( SCR2_RF_DESIFRE_OPTIONS authent_mode, unsigned char authent_key_number, unsigned char * authent_key, unsigned int authent_key_len );

_EXPORT_ unsigned int scr2_rf_desfire_pdc ( unsigned char * pdc_key, unsigned char pdc_option );

_EXPORT_ unsigned int scr2_rf_desfire_exchange ( SCR2_RF_DESIFRE_OPTIONS comm_mode,
												 unsigned char   command_code,
												 unsigned char * command_header,
												 unsigned int    command_header_len,
												 unsigned char * command_data,
												 unsigned int    command_data_len,
												 unsigned char * response_code,
												 unsigned char * response_data,
												 unsigned int  * response_data_len,
												 unsigned char            force_extlen);

_EXPORT_ unsigned int scr2_rf_desfire_get_version ( unsigned char * VERSION, unsigned int * VERSION_len );

_EXPORT_ unsigned int scr2_rf_desfire_format ();

_EXPORT_ unsigned int scr2_rf_desfire_create_application ( unsigned char * AID, SCR2_RF_DESIFRE_OPTIONS key_type );
_EXPORT_ unsigned int scr2_rf_desfire_delete_application ( unsigned char * AID );

_EXPORT_ unsigned int scr2_rf_desfire_change_key ( unsigned char key_number, unsigned char key_version, unsigned char * key, unsigned int key_length );

_EXPORT_ unsigned int scr2_rf_desfire_get_card_uid ( unsigned char * UID, unsigned int * UID_len );

_EXPORT_ unsigned int scr2_rf_desfire_create_standarddata_file ( SCR2_RF_DESIFRE_OPTIONS file_comm_mode, unsigned int file_number, unsigned int file_size, unsigned int file_access_rights );
_EXPORT_ unsigned int scr2_rf_desfire_create_backupdata_file   ( SCR2_RF_DESIFRE_OPTIONS file_comm_mode, unsigned int file_number, unsigned int file_size, unsigned int file_access_rights );

_EXPORT_ unsigned int scr2_rf_desfire_delete_file ( unsigned int file_number );

_EXPORT_ unsigned int scr2_rf_desfire_write_file  ( unsigned int file_number, unsigned char * file_data, unsigned int file_length, unsigned int file_offset, SCR2_RF_DESIFRE_OPTIONS file_commmode );
_EXPORT_ unsigned int scr2_rf_desfire_read_file   ( unsigned int file_number, unsigned char * file_data, unsigned int file_length, unsigned int file_offset, SCR2_RF_DESIFRE_OPTIONS file_commmode );

_EXPORT_ unsigned int scr2_rf_desfire_commit_transaction ( );


_EXPORT_ unsigned int scr2_rf_desfire_cmac_EV2 ( unsigned char * databuffer, unsigned int data_length, unsigned char * key, unsigned char * mac);







//////////////////////////////////////////////////////////////////////////////////
//
//  SWP
//
//

// define the SWP error codes

typedef enum _SCR2_E_SWP {

    SCR2_E_SWP_NO_FRAME_RECEIVED = 0x8300,               
    SCR2_E_SWP_NOATR_AVAILABLE,                  
    SCR2_E_SWP_NOATS_AVAILABLE,                 
    SCR2_E_SWP_ATR_CORRUPTED,                    
    SCR2_E_SWP_NO_EVENT_AVAILABLE,               
    SCR2_E_SWP_HCP_ERROR_0,		                
    SCR2_E_SWP_HCP_ERROR_1,		                
    SCR2_E_SWP_NO_TIME_AVAILABLE,                
    SCR2_E_SWP_GATE_NOT_SUPPORTED,               
    SCR2_E_SWP_NO_ATR_AVAILABLE,                 
    SCR2_E_SWP_GENERAL_ERROR,                    
    SCR2_E_SWP_UNEXPECTED_FRAME,                 
    SCR2_E_SWP_FRAME_LENGTH_ERROR,               
    SCR2_E_SWP_EVENT_DATA_TOO_LARGE,             
    SCR2_E_SWP_SWPRECVLENERROR,                  
    SCR2_E_SWP_SWPRECVCRCERROR,                  
    SCR2_E_SWP_SWPACT0FAILED,                    
    SCR2_E_SWP_SWPSENDSTREAM,                    
    SCR2_E_SWP_SWPGATENOTOPEN,                   
    SCR2_E_SWP_SWPPIPENOTOPEN,                   
    SCR2_E_SWP_SWPRECVOVERFLOW,                  
    SCR2_E_SWP_MAXRETRYREACHED,                  
    SCR2_E_SWP_PIPE_ALREADY_OPEN,          
    SCR2_E_SWP_RAW_FRAME_TOO_LONG,               
    SCR2_E_SWP_RAW_FRAME_TOO_SHORT,              
    SCR2_E_SWP_TOO_MANY_EVENTS,                  
    SCR2_E_SWP_WRONG_EVENT_INDEX,
    SCR2_E_SWP_REGISTRYNOTDEFINED,
    SCR2_E_SWP_ANY_E_NOT_CONNECTED,
    SCR2_E_SWP_ANY_E_CMD_PAR_UNKNOWN,
    SCR2_E_SWP_ANY_E_NOK,    
    SCR2_E_SWP_ADM_E_NO_PIPES_AVAILABLE,
    SCR2_E_SWP_ANY_E_REG_PAR_UNKNOWN,
    SCR2_E_SWP_ANY_E_PIPE_NOT_OPENED,
    SCR2_E_SWP_ANY_E_CMD_NOT_SUPPORTED,
    SCR2_E_SWP_ANY_E_INHIBITED,
    SCR2_E_SWP_ANY_E_TIMEOUT,
    SCR2_E_SWP_ANY_E_REG_ACCESS_DENIED,
    SCR2_E_SWP_ANY_E_PIPE_ACCESS_DENIED,
    SCR2_E_SWP_UNKNOWNRESPONSE,
    SCR2_E_SWP_NOACKNOWLEDGE,
    SCR2_E_SWP_CLT_SESSION_NOT_OPEN,
    SCR2_E_SWP_CLT_SESSION_ERROR,
    SCR2_E_SWP_CLT_WRONG_ALIGNMENT,
    SCR2_E_SWP_CLT_GOTO_INIT,
    SCR2_E_SWP_CLT_GOTO_HALT,
    SCR2_E_SWP_CLT_ADMIN_FIELD

} SCR2_E_SWP;


typedef enum _SCR2_SWP_PARAMS {

    SCR2_SWP_BITRATE=0,
    SCR2_SWP_BUS_VOLTAGE,
    SCR2_SWP_TIMEOUT,
    SCR2_SWP_T_S1_HIGH,
    SCR2_SWP_T_S2_ACT_RES,
    SCR2_SWP_EN_RND_SESSIONID,
	SCR2_SWP_GATE,
    SCR2_SWP_HCI_NR,
    SCR2_SWP_HCI_NS,
    SCR2_SWP_DISABLE_AUTOACK,
    SCR2_SWP_IGNORE_RECV_ACK,
    SCR2_SWP_RF_ERROR_INDICATOR,
    SCR2_SWP_TIME_EXECUTION,    // read-only
    SCR2_SWP_TIME_COMM_SEND,    // read-only
    SCR2_SWP_TIME_COMM_RECV,    // read-only
    SCR2_SWP_TIME_HOST,         // read-only
    SCR2_SWP_TIME_ACTSYNC,      // read-only
    SCR2_SWP_TEARING_TIMEOUT,
    SCR2_SWP_TEARING_MODE

} SCR2_SWP_PARAMS;


typedef enum _SCR2_SWP_GATES {

    SCR2_SWP_GATE_LINK_MANAGEMENT = 0,
    SCR2_SWP_GATE_IDENTITY_MANAGEMENT,
    SCR2_SWP_GATE_ADMINISTRATION,
    SCR2_SWP_GATE_LOOP_BACK,
    SCR2_SWP_GATE_TYPE_A_CARD_RF,
    SCR2_SWP_GATE_TYPE_B_CARD_RF,
    SCR2_SWP_GATE_TYPE_B2_CARD_RF,
    SCR2_SWP_GATE_TYPE_F_CARD_RF,
    SCR2_SWP_GATE_TYPE_A_READER_RF,
    SCR2_SWP_GATE_TYPE_B_READER_RF,
    SCR2_SWP_GATE_TYPE_A_APPLICATION,
    SCR2_SWP_GATE_TYPE_B_APPLICATION,
    SCR2_SWP_GATE_CONNECTIVITY,
    SCR2_SWP_GATE_APDU_READER,
    SCR2_SWP_GATE_APDU

} SCR2_SWP_GATES;


// constants for tearing mode :

#define SCR2_SWP_TEARING_DISABLED    0x0000
#define SCR2_SWP_TEARING_ENABLED     0x0001

#define SCR2_SWP_TEARING_STOP        0x0100
#define SCR2_SWP_TEARING_CONT        0x0000

#define SCR2_SWP_TEARING_EXECONLY    0x0000
#define SCR2_SWP_TEARING_EXECCOMM    0x1000


_EXPORT_ unsigned int scr2_swp_set_parameter (SCR2_SWP_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_swp_get_parameter (SCR2_SWP_PARAMS param_select, unsigned int * param_value); 


#define SCR2_SWPMODE_SWP                            1 
#define SCR2_SWPMODE_DWP_IO34                       2 
#define SCR2_SWPMODE_DWP_TP12                       3 

#define SCR2_SWP_ACTIVATION_MODE_BUS                0
#define SCR2_SWP_ACTIVATION_MODE_ACT                1
#define SCR2_SWP_ACTIVATION_MODE_HCI                2

#define SCR2_SWP_POWER_CYCLE_DISABLED               0
#define SCR2_SWP_POWER_CYCLE_ENABLED                1

#define SCR2_SWP_POWER_MODE_ON                      0
#define SCR2_SWP_POWER_MODE_OFF                     1


_EXPORT_ unsigned int scr2_swp_protocol_activate   ( unsigned char activationmode, unsigned char powercycle );
_EXPORT_ unsigned int scr2_swp_protocol_deactivate ( unsigned char powermode );

#define SCR2_SWP_CLEAR_CONTEXT                      0
#define SCR2_SWP_KEEP_CONTEXT                       1

_EXPORT_ unsigned int scr2_swp_protocol_reactivate ( unsigned char activationmode, unsigned char clearcontext );

_EXPORT_ unsigned int scr2_swp_protocol_exchange   ( unsigned char * databuffer, unsigned int * length );


_EXPORT_ unsigned int scr2_swp_native_exchange     ( SCR2_SWP_GATES gate, unsigned char * databuffer, unsigned int * length );




#define SCR2_SWP_GATE_PIPE_DELETED                  0
#define SCR2_SWP_GATE_PIPE_CREATED                  1
#define SCR2_SWP_GATE_PIPE_OPENED                   2

_EXPORT_ unsigned int scr2_swp_get_gate_status (SCR2_SWP_GATES gate, unsigned int * gate_status, unsigned char * pipe_id);



_EXPORT_ unsigned int scr2_swp_getatr     (unsigned char * atr_ptr, unsigned int * atr_length);
_EXPORT_ unsigned int scr2_swp_getats     (unsigned char * ats_ptr, unsigned int * ats_length);



_EXPORT_ unsigned int scr2_swp_send_raw_frame (unsigned char * databuffer, unsigned int   length);
_EXPORT_ unsigned int scr2_swp_read_raw_frame (unsigned char * databuffer, unsigned int * length);




typedef struct __SCR2_SWP_Administration_gate_registry {

	unsigned char SESSION_IDENTITY[8];
    unsigned char MAX_PIPE;
    unsigned char WHITELIST[10];
    unsigned char WHITELIST_len;
    unsigned char HOST_LIST[10];
    unsigned char HOST_LIST_len;
    unsigned char HOST_ID;
    unsigned char HOST_TYPE[2];
    unsigned char HOST_TYPE_LIST[20];
    unsigned char HOST_TYPE_LIST_len;
    unsigned char MANAGING_HOST_ID;
    unsigned char MH_CAPABILITY;
    unsigned char MH_AVAILABILITY_STATE;

} _SCR2_SWP_Administration_gate_registry;


typedef struct __SCR2_SWP_Connectivity_gate_registry {

	unsigned char UI_STATE;

} _SCR2_SWP_Connectivity_gate_registry;


typedef struct __SCR2_SWP_Link_Management_gate_registry {

	unsigned char REC_ERROR[2];

} _SCR2_SWP_Link_Management_gate_registry;

typedef struct __SCR2_SWP_Identity_Management_gate_registry {

	unsigned char VERSION_SW[3];
	unsigned char VERSION_HARD[3];
	         char VENDOR_NAME[21];      // UTF8 string
    unsigned char MODEL_ID;
    unsigned char HCI_VERSION;
    unsigned char MAX_CURRENT;
    unsigned char GATES_LIST[20];
    unsigned char GATES_LIST_len;

} _SCR2_SWP_Identity_Management_gate_registry;



typedef struct __SCR2_SWP_Type_A_card_RF_gate_registry {

	unsigned char mode;
	unsigned char uid_reg[10];
	unsigned char uid_len;
	unsigned char sak;
	unsigned char atqa[2];
	         char application_data[32];
    unsigned char application_data_len;
	unsigned char fwi_sfgi;
	unsigned char cid_support;
	unsigned char clt_support;
	unsigned char datarate_max[3];

} _SCR2_SWP_Type_A_card_RF_gate_registry;


typedef struct __SCR2_SWP_Type_B_card_RF_gate_registry {

	unsigned char mode;
	unsigned char pupi_reg[10];
	unsigned char pupi_len;
	unsigned char afi;
	unsigned char atqb[4];
	unsigned char higher_layer_response[32];
    unsigned char higher_layer_response_len;
	unsigned char datarate_max[3];

} _SCR2_SWP_Type_B_card_RF_gate_registry;


typedef struct __SCR2_SWP_Type_B2_card_RF_gate_registry {

	unsigned char mode;
	unsigned char pat_in[32];
    unsigned char pat_in_len;
	unsigned char dat_out[32];
    unsigned char dat_out_len;

} _SCR2_SWP_Type_B2_card_RF_gate_registry;


typedef struct __SCR2_SWP_Type_F_card_RF_gate_registry {

	unsigned char mode;
	unsigned char speed_cap;
	unsigned char clt_support;

} _SCR2_SWP_Type_F_card_RF_gate_registry;


typedef struct __SCR2_SWP_APDU_reader_gate_registry {

	unsigned char ATR[32];
    unsigned char ATR_len;

} _SCR2_SWP_APDU_reader_gate_registry;


typedef struct __SCR2_SWP_APDU_gate_registry {

	unsigned int MAX_C_APDU_SIZE;
    unsigned int MAX_WAIT_TIME;

} _SCR2_SWP_APDU_gate_registry;


typedef struct __SCR2_SWP_Type_A_reader_RF_gate_registry {

	unsigned char uid[10];
	unsigned char uid_len;

	unsigned char atqa[2];
	         char application_data[32];
    unsigned char application_data_len;
	unsigned char sak;
	unsigned char fwi_sfgi;
	unsigned char datarate_max;
    unsigned char operating_status;
    unsigned char status_event_en;

} _SCR2_SWP_Type_A_reader_RF_gate_registry;



typedef struct __SCR2_SWP_Type_B_reader_RF_gate_registry {

	unsigned char pupi[10];
	unsigned char pupi_len;

	unsigned char application_data[32];
	unsigned char application_data_len;
	
	unsigned char afi;
	
	unsigned char higher_layer_response[32];
	unsigned char higher_layer_response_len;

	unsigned char higher_layer_data[32];
	unsigned char higher_layer_data_len;

    unsigned char operating_status;
    unsigned char status_event_en;

} _SCR2_SWP_Type_B_reader_RF_gate_registry;





_EXPORT_ unsigned int scr2_swp_get_registry   (SCR2_SWP_GATES gate, void * registry);
_EXPORT_ unsigned int scr2_swp_set_registry   (SCR2_SWP_GATES gate, void * registry);
_EXPORT_ unsigned int scr2_swp_clear_registry (SCR2_SWP_GATES gate);

_EXPORT_ unsigned int scr2_swp_print_registry (SCR2_SWP_GATES gate);



// HCP Event handling 

_EXPORT_ unsigned int scr2_swp_send_event ( unsigned char   pipe_id,
                                            unsigned char   event_id, 
                                            unsigned char * event_parameter,
                                            unsigned int    event_parameter_length);

typedef struct __SCR2_SWP_Event_Data {

    unsigned char pipe_id;
    unsigned char gate_id;
    unsigned char event_id;

    unsigned int  event_data_length;
    unsigned char event_data[4096];

    unsigned int  event_raw_length;
    unsigned char event_raw[4100];
    
} _SCR2_SWP_Event_Data;


_EXPORT_ unsigned int scr2_swp_get_event_count (unsigned int * event_count);
_EXPORT_ unsigned int scr2_swp_get_event_data  (_SCR2_SWP_Event_Data * event_data);
_EXPORT_ unsigned int scr2_swp_clear_events    ();



// HCP Command handling

typedef enum _SCR2_SWP_HCI_COMMANDS {

    CMD_ANY_SET_PARAMETER=0x01,
    CMD_ANY_GET_PARAMETER=0x02,
    CMD_ANY_OPEN_PIPE=0x03,
    CMD_ANY_CLOSE_PIPE=0x04,
    CMD_ADM_CREATE_PIPE=0x10,
    CMD_ADM_DELETE_PIPE=0x11,
    CMD_ADM_NOTIFY_PIPE_CREATED=0x12,
    CMD_ADM_NOTIFY_PIPE_DELETED=0x13,
    CMD_ADM_CLEAR_ALL_PIPE=0x14,
    CMD_ADM_NOTIFY_ALL_PIPE_CLEARED=0x15

} SCR2_SWP_HCI_COMMANDS;


_EXPORT_ unsigned int scr2_swp_send_command ( unsigned char   pipe_id, 
                                              unsigned char   command, 
                                              unsigned char * parameter,
                                              unsigned int  * parameter_length);





//////////////////////////////////////////////////////////////////////////////////
//
// SWP MIFARE
//

// define the MIFARE error codes

typedef enum _SCR2_E_MIFARE {

    SCR2_E_MIFARE_PARITY_ERROR = 0x8800,
    SCR2_E_MIFARE_FRAME_LENGTH_ERROR,
    SCR2_E_MIFARE_NOT_SUPPORTED_UID,
    SCR2_E_MIFARE_GATE_NOT_OPEN,
    SCR2_E_MIFARE_AUTHENT_ERROR,
    SCR2_E_MIFARE_CRC_ERROR,
    SCR2_E_MIFARE_UNEXPECTED_RESPONSE,
    SCR2_E_MIFARE_ACK_ERROR

} SCR2_E_MIFARE;


typedef enum _MIFARE_KEY_SELECT {
    
    MIFARE_KEY_A = 0,
    MIFARE_KEY_B

} MIFARE_KEY_SELECT;


_EXPORT_ unsigned int scr2_swp_open_clt_mifare ();

_EXPORT_ unsigned int scr2_swp_mifare_authent_block (unsigned char blocknumber, MIFARE_KEY_SELECT keyAB, unsigned char * mifare_key);

_EXPORT_ unsigned int scr2_swp_mifare_read_block (unsigned char blocknumber, unsigned char * blockdata);

_EXPORT_ unsigned int scr2_swp_mifare_write_block (unsigned char blocknumber, unsigned char * blockdata);

_EXPORT_ unsigned int scr2_swp_mifare_write_value (unsigned char blocknumber, unsigned int value, unsigned char address);

_EXPORT_ unsigned int scr2_swp_mifare_increment (unsigned char blocknumber, unsigned int value);

_EXPORT_ unsigned int scr2_swp_mifare_decrement (unsigned char blocknumber, unsigned int value);

_EXPORT_ unsigned int scr2_swp_mifare_restore (unsigned char blocknumber);

_EXPORT_ unsigned int scr2_swp_mifare_transfer (unsigned char blocknumber);

_EXPORT_ unsigned int scr2_swp_mifare_halt ();




//////////////////////////////////////////////////////////////////////////////////
//
// SWP FeliCa
//

// define the FeliCa error codes

typedef enum _SCR2_E_FELICA {

    SCR2_E_FELICA_CRC_ERROR = 0x8900,
    SCR2_E_FELICA_FRAME_LENGTH_ERROR,
    SCR2_E_FELICA_GATE_NOT_OPEN,
    SCR2_E_FELICA_NO_DATA_FIELD

} SCR2_E_FELICA;


_EXPORT_ unsigned int scr2_swp_open_clt_felica ();

_EXPORT_ unsigned int scr2_swp_felica_exchange_frame (unsigned char * databuffer,
                                                      unsigned int  * datalength,
                                                      unsigned int    timeout_us);

_EXPORT_ unsigned int scr2_swp_felica_polling (unsigned int    system_code, 
                                               unsigned char   request_code,
                                               unsigned char   time_slot,
                                               unsigned char * response,
                                               unsigned int  * response_length);

_EXPORT_ unsigned int scr2_swp_felica_exchange_native (unsigned char * databuffer,
                                                       unsigned int  * datalength);





//////////////////////////////////////////////////////////////////////////////////
//
// RAW I2C, can be used to control I2C MEMORY (for instance)
//

// define the RAWI2C error codes

typedef enum _SCR2_E_RAWI2C {

    SCR2_E_RAWI2C_FRAMELENGTHERROR = 0x8500,			
    SCR2_E_RAWI2C_FRAMESENDERROR,			
    SCR2_E_RAWI2C_NOT_ACKNOWLEDGED,	
    SCR2_E_RAWI2C_ACKHSERROR,
    SCR2_E_RAWI2C_FRAMEREADERROR,            
    SCR2_E_RAWI2C_WRONGCONTROLBYTE,          
    SCR2_E_RAWI2C_WRONGTIMEOUTVALUE,
    SCR2_E_RAWI2C_TRIGGER_TIMEOUT,
    SCR2_E_RAWI2C_INSERTION_ABORTED,
    SCR2_E_RAWI2C_STRETCH_TIMEOUT,
    SCR2_E_RAWI2C_NO_INTERFACE_SELECTED,
    SCR2_E_RAWI2C_BITRATE_ERROR,
    SCR2_E_RAWI2C_POWER_OFF_ERROR
 
} SCR2_E_RAWI2C;

typedef enum _SCR2_RAWI2C_PARAMS {

    SCR2_RAWI2C_BITRATE=0,
    SCR2_RAWI2C_SLAVEADDRESS,
    SCR2_RAWI2C_HSMASTERCODE,
    SCR2_RAWI2C_DIS_HS_STRETCH,
    SCR2_RAWI2C_DIS_ALL_STRETCH,
    SCR2_RAWI2C_IGNORE_NACK,
    SCR2_RAWI2C_ERROR_MODE,
    SCR2_RAWI2C_ERROR_START,
    SCR2_RAWI2C_ERROR_LENGTH,
    SCR2_RAWI2C_ERROR_PATTERN,
    SCR2_RAWI2C_ERROR_SDADELAY,
    SCR2_RAWI2C_ERROR_SDADEFAULT,
    SCR2_RAWI2C_DIS_HS_MODE,
    SCR2_RAWI2C_FORCE_HS_MODE,
    SCR2_RAWI2C_OVERRUN_SEND_COUNT,
    SCR2_RAWI2C_OVERRUN_RECV_COUNT,
    SCR2_RAWI2C_OVERRUN_IGNORE_NACK,
    SCR2_RAWI2C_SUPPRESS_START,
    SCR2_RAWI2C_SUPPRESS_STOP,
    SCR2_RAWI2C_SEND_INDELAY,
    SCR2_RAWI2C_RECV_INDELAY,
    SCR2_RAWI2C_RECV_STOP_DELAY

} SCR2_RAWI2C_PARAMS;

typedef enum _SCR2_RAWI2C_ERROR_MODES {

    SCR2_RAWI2C_ERROR_MODE_REPLACE_CONTINUE=0,
    SCR2_RAWI2C_ERROR_MODE_REPLACE_STOP=1,
    SCR2_RAWI2C_ERROR_MODE_INSERT_CONTINUE=2,
    SCR2_RAWI2C_ERROR_MODE_INSERT_STOP=3,
    SCR2_RAWI2C_ERROR_SDADEFAULT_LOW=4,
    SCR2_RAWI2C_ERROR_SDADEFAULT_HIGH=5,
    SCR2_RAWI2C_ERROR_SDADEFAULT_PULL=6

} SCR2_RAWI2C_ERROR_MODES;


_EXPORT_ unsigned int scr2_rawi2c_set_parameter (SCR2_RAWI2C_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_rawi2c_get_parameter (SCR2_RAWI2C_PARAMS param_select, unsigned int * param_value); 

_EXPORT_ unsigned int scr2_rawi2c_send_data (unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr2_rawi2c_send_chm (unsigned char * databuffer, unsigned int length);

_EXPORT_ unsigned int scr2_rawi2c_read_data (unsigned char * readdatabuffer, unsigned int readdata_length);

_EXPORT_ unsigned int scr2_rawi2c_read_data_comb (unsigned char * controlbuffer , unsigned int control_length,
                                                  unsigned char * readdatabuffer, unsigned int readdata_length);

_EXPORT_ unsigned int scr2_rawi2c_exchange_data (unsigned char * commanddatabuffer,     unsigned int commandlength,
                                                 unsigned char * responsedatabuffer,    unsigned int responsedatalength,
                                                 unsigned int    wait_time_us);

_EXPORT_ unsigned int scr2_rawi2c_exchange_data_comb (unsigned char * commanddatabuffer,     unsigned int commandlength,
                                                      unsigned char * responsecontrolbuffer, unsigned int responsecontrollength,
                                                      unsigned char * responsedatabuffer,    unsigned int responsedatalength,
                                                      unsigned int    wait_time_us);


typedef enum _SCR2_RAWI2C_RESET_MODES {

    SCR2_RAWI2C_RESET_NONE=0,
    SCR2_RAWI2C_RESET_COLD=1,
    SCR2_RAWI2C_RESET_WARM=2

} SCR2_RAWI2C_RESET_MODES;


_EXPORT_ unsigned int scr2_rawi2c_observe_interface (SCR2_RAWI2C_RESET_MODES reset_mode, unsigned int observe_time_us,
                                                     unsigned int * scl_edges, unsigned int * sda_edges,
                                                     double * mean_current_mA, double * max_current_mA);










//////////////////////////////////////////////////////////////////////////////////
//
// Rear SIM
//


// re-use parameters and error codes from ISO7816 interface


_EXPORT_ unsigned int scr2_sim_set_parameter (SCR2_ISO7816_PARAMS param_select, unsigned int   param_value); 
_EXPORT_ unsigned int scr2_sim_get_parameter (SCR2_ISO7816_PARAMS param_select, unsigned int * param_value); 

_EXPORT_ unsigned int scr2_sim_get_atr (unsigned char * atr_ptr, unsigned int * atr_length);

_EXPORT_ unsigned int scr2_sim_protocol_activate (unsigned char protocol, unsigned char power_mode); 
_EXPORT_ unsigned int scr2_sim_protocol_pps      (unsigned char protocol);
_EXPORT_ unsigned int scr2_sim_protocol_ifs      ();

_EXPORT_ unsigned int scr2_sim_protocol_deactivate ();
_EXPORT_ unsigned int scr2_sim_protocol_exchange (unsigned char * databuffer, unsigned int * length);



//////////////////////////////////////////////////////////////////////////////////
//
// P71
//

// define the P71 error codes

typedef enum _SCR2_E_P71 {

    SCR2_E_P71_SAM_COMMERROR = 0x8600,
    SCR2_E_P71_SAM_SYNTAXERROR,

    SCR2_E_KEYSIM_NOT_OPEN,
    SCR2_E_PCC_DECRYPTION_FAILED,
    SCR2_E_PCC_MAC_FAILED,
    SCR2_E_PCC_PADDING_FAILED

} SCR2_E_P71;


_EXPORT_ unsigned int scr2_p71_sam_open (unsigned char * password);


_EXPORT_ unsigned int scr2_p71_sam_get_key_info (unsigned char * selected_key, unsigned char * key_info);


_EXPORT_ unsigned int scr2_p71_sam_select_key_set (unsigned char key_set);


_EXPORT_ unsigned int scr2_p71_sam_set_key_set_v1 (         char * key_info,
                                                   unsigned char * enc_key,
                                                   unsigned char * mac_key,
                                                   unsigned char * auth_key );

_EXPORT_ unsigned int scr2_p71_sam_set_key_set_v2 (         char * key_info,
                                                   unsigned char * enc_key,
                                                   unsigned char * mac_key,
                                                   unsigned char * gvmac,
                                                   unsigned char * auth00cmd,
                                                   unsigned char * authffcmd );

_EXPORT_ unsigned int scr2_p71_sam_set_key_set_v3 (         char * key_info,
                                                   unsigned char * enc_key,
                                                   unsigned char * mac_key,
                                                   unsigned char * gvmac,
                                                   unsigned char * auth00cmd,
                                                   unsigned char * authffcmd );

_EXPORT_ unsigned int scr2_p71_sam_set_key_set_v4 (         char * key_info,
                                                   unsigned char * enc_key,
                                                   unsigned char * mac_key,
                                                   unsigned char * gvmac,
                                                   unsigned char * auth00cmd,
                                                   unsigned char * authffcmd );


_EXPORT_ unsigned int scr2_p71_sam_translate_command (unsigned char * command, unsigned int * length,
                                                      unsigned int  * result                          ) ;


_EXPORT_ unsigned int scr2_p71_sam_translate_response (unsigned char * response, unsigned int * length,
                                                       unsigned int  * result                           ) ;


_EXPORT_ unsigned int scr2_p71_sam_clear_cache ();






//////////////////////////////////////////////////////////////////////////////////
//
// Dual interface - APDU
//


_EXPORT_ unsigned int scr2_dualif_apdu_exchange ( unsigned char * databuffer_commif1, 
												  unsigned int  * length_commif1,
												  unsigned int  * retvalue_commif1,
												  unsigned int    startdelay_commif1,

												  unsigned char * databuffer_commif2, 
												  unsigned int  * length_commif2,
												  unsigned int  * retvalue_commif2,
												  unsigned int    startdelay_commif2
												);



//////////////////////////////////////////////////////////////////////////////////
//
// Dual interface - RAW
//


typedef struct _SCR2_DUALIF_RAWI2C_PARAMETERS {

    unsigned char * commanddatabuffer;
    unsigned int    commandlength;
    unsigned char * responsedatabuffer;
    unsigned int    responsedatalength;
    unsigned int    wait_time_us;

} SCR2_DUALIF_RAWI2C_PARAMETERS;

typedef struct _SCR2_DUALIF_RF_PARAMETERS {

    unsigned char * databuffer;
    unsigned int  * length;
    unsigned char   txlastbits;
    unsigned char   rxalign;
    unsigned int    timeout_us;
    SCR2_RF_OPTIONS crc_on;

} SCR2_DUALIF_RF_PARAMETERS;



_EXPORT_ unsigned int scr2_dualif_raw_exchange  ( void          * parameters_commif1,
												  unsigned int  * retvalue_commif1,
												  unsigned int    startdelay_commif1,

												  void          * parameters_commif2,
												  unsigned int  * retvalue_commif2,
												  unsigned int    startdelay_commif2
												);







//############################# FELICA SAM ###############################


typedef enum _SCR2_E_SAM_FELICA {

    SCR2_E_SAM_FELICA_SYNTAXERROR = 0x8c00,
    SCR2_E_SAM_FELICA_FRAMELENGTH,
    SCR2_E_SAM_FELICA_WRONGSW,
    SCR2_E_SAM_FELICA_UNEXPECTEDPACKET,
    SCR2_E_SAM_FELICA_WRONGDISPATCHER,
    SCR2_E_SAM_FELICA_AUTH1FAILED,
    SCR2_E_SAM_FELICA_AUTH2FAILED,
    SCR2_E_SAM_FELICA_NOTAUTHENTICATED,
    SCR2_E_SAM_FELICA_WRONGSEQNUMBER,
    SCR2_E_SAM_FELICA_WRONGSEQID,
    SCR2_E_SAM_FELICA_WRONGMAC,
    SCR2_E_SAM_FELICA_UNEXPECTEDRESPONSE,
    SCR2_E_SAM_FELICA_WRONGDATATYPE

} SCR2_E_SAM_FELICA;


typedef enum _SCR2_SAM_FELICA_MODES {

    SCR2_RCS500_NORMAL_MODE = 0,
    SCR2_RCS500_ADMIN_MODE

} SCR2_SAM_FELICA_MODES;

_EXPORT_ unsigned int scr2_sam_felica_exchange_packet_plain (unsigned char * packet_data,
                                                             unsigned int  * packet_length);

_EXPORT_ unsigned int scr2_sam_felica_exchange_packet_aes_crypted (unsigned char   dispatcher,
                                                                   unsigned char   command_code, 
                                                                   unsigned char   sub_command_code,
                                                                   unsigned char * command_data,
                                                                   unsigned int  * command_length);

_EXPORT_ unsigned int scr2_sam_felica_exchange_packet_des_crypted (unsigned char   dispatcher,
                                                                   unsigned char   command_code, 
                                                                   unsigned char * command_data,
                                                                   unsigned int  * command_length);

_EXPORT_ unsigned int scr2_sam_felica_authent_des (SCR2_SAM_FELICA_MODES   authent_mode,
                                                   unsigned char         * authent_key);

_EXPORT_ unsigned int scr2_sam_felica_authent_aes (SCR2_SAM_FELICA_MODES   authent_mode,
                                                   unsigned char         * authent_key);


_EXPORT_ unsigned int scr2_sam_felica_change_communication_mode (unsigned char transmission_rate,
                                                                 unsigned char data_type);

_EXPORT_ unsigned int scr2_sam_felica_change_packet_type ();

_EXPORT_ unsigned int scr2_sam_felica_get_last_error (unsigned int * error_code);

_EXPORT_ unsigned int scr2_sam_felica_get_rwsam_key_versions (unsigned int * key_versions);

_EXPORT_ unsigned int scr2_sam_felica_get_felica_key_info (unsigned char key_index,
                                                           unsigned char * key_info_data,
                                                           unsigned int  * key_info_length);

typedef struct __SCR2_felica_key_registration_package {

    unsigned char   package_type;
    unsigned char   key_type;
    unsigned char   system_code[2]; 
    unsigned char   area_service_code[2];
    unsigned char   key_version[2]; 
    unsigned char   key_value[8];

} _SCR2_felica_key_registration_package;


_EXPORT_ unsigned int scr2_sam_felica_register_felica_key (unsigned int    system_code,
                                                           unsigned int    area_service_code,
                                                           unsigned int    key_version,
                                                           unsigned char * key_value,
                                                           unsigned char * user_package_key);


#ifdef __cplusplus
}
#endif

