/*
 * Copyright 2013, 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * RS232 (Windows) BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SERIALWIN

#include "phbalReg_SerialWin.h"
#include "phbalReg_SerialWin_Int.h"

#ifndef ONESTOPBIT
#define ONESTOPBIT 1
#endif

#ifndef TWOSTOPBITS
#define TWOSTOPBITS 2
#endif

phStatus_t phbalReg_SerialWin_Init(
                                   phbalReg_SerialWin_DataParams_t * pDataParams,
                                   uint16_t wSizeOfDataParams,
                                   uint16_t wEnumBufSize,
                                   uint8_t * pEnumBuffer
                                   )
{
    if (sizeof(phbalReg_SerialWin_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    if (wEnumBufSize > 0) PH_ASSERT_NULL (pEnumBuffer);
    /* set enumeration buffer to NULL if no size is given */
    if (wEnumBufSize == 0)
    {
        pEnumBuffer = NULL;
    }

    /* initialize the data parameters to default values */
    pDataParams->wId                        = PH_COMP_BAL | PHBAL_REG_SERIALWIN_ID;
    pDataParams->pComHandle                 = NULL;
    pDataParams->dwBitRate                  = 9600;
    pDataParams->wTimeout                   = PHBAL_REG_SERIALWIN_RX_TIMEOUT_VAL;
    pDataParams->wAdditionalRxDelayBytes    = 0;
    pDataParams->wTimeoutMultiplier         = 0;
    pDataParams->pComPortString             = NULL;
    pDataParams->wEnumBufSize               = wEnumBufSize;
    pDataParams->pEnumBuffer                = pEnumBuffer;
    pDataParams->bStopBits                  = ONESTOPBIT;
    pDataParams->bDtr                       = PHBAL_REG_SERIALWIN_VALUE_DTR_LOW;
    pDataParams->bRts                       = PHBAL_REG_SERIALWIN_VALUE_RTS_LOW;
    pDataParams->bParity                    = PHBAL_REG_SERIALWIN_VALUE_PARITY_NONE;
    pDataParams->bFI                        = 1;
    pDataParams->bDI                        = 1;

#ifdef _WIN32
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wSizeOfDataParams || wEnumBufSize || pEnumBuffer);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_GetPortList(
    phbalReg_SerialWin_DataParams_t * pDataParams,
    uint16_t wPortBufSize,
    uint8_t * pPortNames,
    uint16_t * pNumOfPorts
    )
{
#ifdef _WIN32
    HKEY hKey;
    LONG eRetCode;
    DWORD eCnt;
    DWORD eDataLen;
    DWORD eValueLen;

    /* GetPortList requires an enumeration buffer */
    if (pDataParams->pEnumBuffer == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    *pNumOfPorts = 0;

    /* open a registry key to query the serial com ports */
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "HARDWARE\\DEVICEMAP\\SERIALCOMM", 0, KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS) /* PRQA S 274 */
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
    }

    eCnt = 0;
    do
    {
        eValueLen = pDataParams->wEnumBufSize;
        if (eValueLen > 0)
        {
            pDataParams->pEnumBuffer[0] = '\0';
        }
        eDataLen = wPortBufSize;

        /* query the values of the given registry key */
        eRetCode = RegEnumValue(
            hKey,
            eCnt++,
            (LPSTR)pDataParams->pEnumBuffer,
            &eValueLen,
            NULL,
            NULL,
            pPortNames,
            &eDataLen);

        /* check for buffer overflow */
        if (eRetCode == ERROR_MORE_DATA)
        {
            eRetCode = RegCloseKey(hKey);
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
        }

        /* valid entry found */
        if (eRetCode == ERROR_SUCCESS)
        {
            pPortNames += eDataLen;
			*pPortNames = '\0';
			pPortNames++;
            wPortBufSize = (uint16_t)wPortBufSize - (uint16_t) (eDataLen - 1);
            ++(*pNumOfPorts);
        }
    } while (eRetCode != ERROR_NO_MORE_ITEMS);

    /* close the handle */
    if (RegCloseKey(hKey) != ERROR_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wPortBufSize || pPortNames || pNumOfPorts);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_SetPort(
                                      phbalReg_SerialWin_DataParams_t * pDataParams,
                                      uint8_t * pPortName
                                      )
{
#ifdef _WIN32

    pDataParams->pComPortString = (int8_t*)pPortName;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || pPortName);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_OpenPort(
                                       phbalReg_SerialWin_DataParams_t * pDataParams
                                       )
{
#ifdef _WIN32
    phStatus_t PH_MEMLOC_REM status;

    /* Check if port is already open */
    if (pDataParams->pComHandle != NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* Open port */
    pDataParams->pComHandle = CreateFile(
        (LPCSTR)pDataParams->pComPortString,    /* Serial port file name. */
        GENERIC_READ | GENERIC_WRITE,           /* Access (read-write) mode. */
        0,                                      /* Share mode. */
        NULL,                                   /* Pointer to the security attribute. */
        OPEN_EXISTING,                          /* How to open the serial port. */
        0,                                      /* Port attributes. */
        NULL);                                  /* Handle to port with attribute to copy. */

    /* Error Checking */
    if ((pDataParams->pComHandle == INVALID_HANDLE_VALUE) || (pDataParams->pComHandle == NULL))
    {
        pDataParams->pComHandle = NULL;
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Apply default settings to the port */
    status = phbalReg_SerialWin_Int_SetMode(pDataParams->pComHandle, pDataParams->dwBitRate, pDataParams->bStopBits, pDataParams->bParity, pDataParams->bDtr, pDataParams->bRts );
    if (status != PH_ERR_SUCCESS)
    {
        CloseHandle(pDataParams->pComHandle);  /* PRQA S 3200 */
        pDataParams->pComHandle = NULL;
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Set the communication timeout */
    status = phbalReg_SerialWin_Int_SetTimeout(pDataParams);
    if (status != PH_ERR_SUCCESS)
    {
        CloseHandle(pDataParams->pComHandle);  /* PRQA S 3200 */
        pDataParams->pComHandle = NULL;
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_ClosePort(
                                        phbalReg_SerialWin_DataParams_t * pDataParams
                                        )
{
#ifdef _WIN32
    BOOL status;

    if (pDataParams->pComHandle == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    status = CloseHandle(pDataParams->pComHandle);
    pDataParams->pComHandle = NULL;

    /* Close the handle if it exists */
    if (status == FALSE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_Exchange(
                                       phbalReg_SerialWin_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint16_t wRxBufSize,
                                       uint8_t * pRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
#ifdef _WIN32
    BOOL        PH_MEMLOC_REM status;
	DWORD		PH_MEMLOC_REM dwBytesRead;
    DWORD		PH_MEMLOC_REM dwBytesWritten;
    DWORD       PH_MEMLOC_REM dwErrors;
    COMSTAT     PH_MEMLOC_BUF abComstat;

    /* Check options */
    if ((wOption & PH_EXCHANGE_BUFFER_MASK) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Write the transmit data */
    if (pTxBuffer != NULL || wTxLength != 0)
    {
        status = WriteFile(pDataParams->pComHandle, pTxBuffer, wTxLength, &dwBytesWritten, NULL);

        /* Error check */
        if((status == FALSE) || (dwBytesWritten != (uint32_t) wTxLength))
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

    if (pRxBuffer != NULL || wRxBufSize != 0)
    {
        /* Read the received data */
        status = ReadFile(pDataParams->pComHandle, pRxBuffer, wRxBufSize, &dwBytesRead, NULL);
    }
    else
    {
        status = TRUE;
        dwBytesRead = 0;
    }

    /* General error check */
    if (status == FALSE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    if((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) != PHBAL_REG_SERIALWIN_SUPRESS_CHECKS)
    {
        /* Retrieve Comm-Status */
        status = ClearCommError(pDataParams->pComHandle, &dwErrors, &abComstat);

        /* General error check */
        if(status == FALSE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* Return received length */
        *pRxLength = (uint16_t)dwBytesRead;

        /* Buffer overflow check */
        if(abComstat.cbInQue > 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
        }

        /* Check for timeout error */
        if(dwBytesRead == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
        }
    }

    /* Return received length */
    *pRxLength = (uint16_t) dwBytesRead;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || pTxBuffer || wTxLength || wRxBufSize || pRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_SetConfig(
                                        phbalReg_SerialWin_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t wValue
                                        )
{
#ifdef _WIN32
    phStatus_t PH_MEMLOC_REM statusTmp;

    switch (wConfig)
    {
        /* set a new FI and bit rate */
    case PHBAL_REG_SERIALWIN_CONFIG_FI:
        pDataParams->bFI = (uint8_t) (wValue & 0x00FF);
        statusTmp = phbalReg_SerialWin_Int_SetBitRateFromFiDi(pDataParams);

        /* Adjust additional Timeout */
        if (pDataParams->pComHandle != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
        }
        break;

        /* set a new DI and bit rate */
    case PHBAL_REG_SERIALWIN_CONFIG_DI:
        pDataParams->bDI = (uint8_t) (wValue & 0x00FF);
        statusTmp = phbalReg_SerialWin_Int_SetBitRateFromFiDi(pDataParams);

        /* Adjust additional Timeout */
        if (pDataParams->pComHandle != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
        }
        break;

        /* set a new bit rate */
    case PHBAL_REG_SERIALWIN_CONFIG_BITRATE:
        {
            if((wValue < PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600) || (wValue > PHBAL_REG_SERIALWIN_VALUE_BITRATE_460800))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }

            pDataParams->dwBitRate = wValue;
            /* set bit rate */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetBitRate(pDataParams->pComHandle, pDataParams->dwBitRate));

            /* Adjust additional Timeout */
            if (pDataParams->pComHandle != NULL)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
            }
            break;
        }

        /* set new stop bits */
    case PHBAL_REG_SERIALWIN_CONFIG_STOPBITS:
        {
            switch (wValue)
            {
            case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE:
                pDataParams->bStopBits = ONESTOPBIT;
                break;

            case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE5:
                pDataParams->bStopBits = ONE5STOPBITS;
                break;

            case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO:
                pDataParams->bStopBits = TWOSTOPBITS;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetMode(pDataParams->pComHandle, pDataParams->dwBitRate, pDataParams->bStopBits, pDataParams->bParity, pDataParams->bDtr, pDataParams->bRts));
            break;
        }

        /* set the communication timeout */
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        {
            pDataParams->wTimeout = wValue;
            if (pDataParams->pComHandle != NULL)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
            }
            break;
        }

    case PHBAL_REG_SERIALWIN_CONFIG_DTR:
        {
            DWORD func;

            pDataParams->bDtr = (uint8_t)wValue;
            switch(pDataParams->bDtr)
            {
                case PHBAL_REG_SERIALWIN_VALUE_DTR_LOW:
                    func = CLRDTR;
                    break;
                case PHBAL_REG_SERIALWIN_VALUE_DTR_HIGH:
                    func = SETDTR;
                    break;
                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
                    break;
            }

            if (EscapeCommFunction(pDataParams->pComHandle, func) == FALSE)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            PurgeComm( pDataParams->pComHandle, PURGE_TXCLEAR | PURGE_RXCLEAR );
        }
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_RTS:
        {
            DWORD func;

            pDataParams->bRts = (uint8_t)wValue;
            switch(pDataParams->bRts)
            {
                case PHBAL_REG_SERIALWIN_VALUE_RTS_LOW:
                    func = CLRRTS;
                    break;
                case PHBAL_REG_SERIALWIN_VALUE_RTS_HIGH:
                    func = SETRTS;
                    break;
                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }

            if (EscapeCommFunction(pDataParams->pComHandle, func) == FALSE)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            PurgeComm( pDataParams->pComHandle, PURGE_TXCLEAR | PURGE_RXCLEAR );
        }
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_PARITY:
        pDataParams->bParity = (uint8_t)wValue;
        statusTmp = phbalReg_SerialWin_Int_SetMode(pDataParams->pComHandle, pDataParams->dwBitRate, pDataParams->bStopBits, pDataParams->bParity, pDataParams->bDtr, pDataParams->bRts);
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_TIMEOUT_MULTIPLIER_MS:
        pDataParams->wTimeoutMultiplier = wValue;
        if (pDataParams->pComHandle != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
        }
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_ADDITIONAL_RXDELAY_BYTES:
        pDataParams->wAdditionalRxDelayBytes = wValue;
        if (pDataParams->pComHandle != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SerialWin_Int_SetTimeout(pDataParams));
        }
        break;

        /* configuration parameter not available */
    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wConfig || wValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_GetConfig(
                                        phbalReg_SerialWin_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t * pValue
                                        )
{
#ifdef _WIN32
    switch (wConfig)
    {
    case PHBAL_REG_SERIALWIN_CONFIG_FI:
        *pValue = pDataParams->bFI;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_DI:
        *pValue = pDataParams->bDI;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_BITRATE:
        {
            *pValue = (uint16_t)pDataParams->dwBitRate;
            break;
        }

    case PHBAL_REG_SERIALWIN_CONFIG_STOPBITS:
        {
            switch (pDataParams->bStopBits)
            {
            case ONESTOPBIT:
                *pValue = PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE;
                break;

            case ONE5STOPBITS:
                *pValue = PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE5;
                break;

            case TWOSTOPBITS:
                *pValue = PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
            }
            break;
        }
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        *pValue = pDataParams->wTimeout;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_DTR:
        *pValue = pDataParams->bDtr;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_RTS:
        *pValue = pDataParams->bRts;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_PARITY:
        *pValue = pDataParams->bParity;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_TIMEOUT_MULTIPLIER_MS:
        *pValue = pDataParams->wTimeoutMultiplier;
        break;

    case PHBAL_REG_SERIALWIN_CONFIG_ADDITIONAL_RXDELAY_BYTES:
        *pValue = pDataParams->wAdditionalRxDelayBytes;
        break;

        /* configuration parameter not available */
    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wConfig || pValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

#endif /* NXPBUILD__PHBAL_REG_SERIALWIN */
