/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * BAL SocketWin Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SOCKETWIN

#include "phbalReg_SocketWin.h"
#include "phbalReg_SocketWin_Int.h"

TySocketWinCallback socketwinCallback = NULL;

phStatus_t phbalReg_SocketWin_Init(
                                   phbalReg_SocketWin_DataParams_t * pDataParams,
                                   uint16_t wSizeOfDataParams,
                                   uint8_t bMode,
                                   uint16_t wTxBufSize,
                                   uint8_t * pTxBuffer,
                                   uint16_t wWsaBufferSize,
                                   uint8_t * pWsaBuffer
                                   )
{
#ifdef _WIN32
    if (sizeof(phbalReg_SocketWin_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    /* Verify exchange buffer */
    if (wTxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    /* Verify WSA buffer */
    if (wWsaBufferSize < sizeof(WSADATA))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pWsaBuffer);

    /* Check operating mode */
    switch (bMode)
    {
    case PHBAL_REG_SOCKETWIN_MODE_RAW:
    case PHBAL_REG_SOCKETWIN_MODE_RS232:
    case PHBAL_REG_SOCKETWIN_MODE_I2C:
    case PHBAL_REG_SOCKETWIN_MODE_SPI:
    case PHBAL_REG_SOCKETWIN_MODE_HEX:
    case PHBAL_REG_SOCKETWIN_MODE_TELNET:
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* initialize the data parameters */
    pDataParams->wId                = PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID;
    pDataParams->pSocketHostName    = (uint8_t*)PHBAL_REG_SOCKETWIN_HOST_DEFAULT;
    pDataParams->pSocketPortName    = (uint8_t*)PHBAL_REG_SOCKETWIN_PORT_DEFAULT;
    pDataParams->pUserName          = (uint8_t*)PHBAL_REG_SOCKETWIN_USER_DEFAULT;
    pDataParams->pTxBuffer          = pTxBuffer;
    pDataParams->wTxBufSize         = wTxBufSize;
    pDataParams->pWsaData           = pWsaBuffer;
    pDataParams->dwServerSocket     = INVALID_SOCKET;
    pDataParams->bMode              = bMode;


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wSizeOfDataParams || wTxBufSize || pTxBuffer || wWsaBufferSize || pWsaBuffer);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_GetPortList(
    phbalReg_SocketWin_DataParams_t * pDataParams,
    uint16_t wPortBufSize,
    uint8_t * pPortNames,
    uint16_t * pNumOfPorts
    )
{
#ifdef _WIN32
    uint16_t wLen;

    /* satisfy compiler */
    if (pDataParams);

    if (wPortBufSize < (uint16_t)strlen(PHBAL_REG_SOCKETWIN_HOST_DEFAULT) + strlen(PHBAL_REG_SOCKETWIN_PORT_DEFAULT) + 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }
    wLen = (uint16_t)strlen(PHBAL_REG_SOCKETWIN_HOST_DEFAULT) + 1;
    memcpy(pPortNames, PHBAL_REG_SOCKETWIN_HOST_DEFAULT, wLen); /* PRQA S 3200 */
    pPortNames += wLen;

    wLen = (uint16_t)strlen(PHBAL_REG_SOCKETWIN_PORT_DEFAULT) + 1;
    memcpy(pPortNames, PHBAL_REG_SOCKETWIN_PORT_DEFAULT, wLen); /* PRQA S 3200 */

    *pNumOfPorts = 2;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wPortBufSize || pPortNames || pNumOfPorts);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_SetPort(
                                      phbalReg_SocketWin_DataParams_t * pDataParams,
                                      uint8_t * pPortName
                                      )
{
#ifdef _WIN32
    if (strlen((char*)pPortName) > 0)
    {
        pDataParams->pSocketHostName = pPortName;
    }
    else
    {
        pDataParams->pSocketHostName = (uint8_t*)PHBAL_REG_SOCKETWIN_HOST_DEFAULT;
    }
    pPortName += strlen((char*)pPortName) + 1;
    if (strlen((char*)pPortName) > 0)
    {
        pDataParams->pSocketPortName = pPortName;
    }
    else
    {
        pDataParams->pSocketPortName = (uint8_t*)PHBAL_REG_SOCKETWIN_PORT_DEFAULT;
    }
    pPortName += strlen((char*)pPortName) + 1;
    if (strlen((char*)pPortName) > 0)
    {
        pDataParams->pUserName = pPortName;
    }
    else
    {
        pDataParams->pUserName = (uint8_t*)PHBAL_REG_SOCKETWIN_USER_DEFAULT;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || pPortName);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_OpenPort(
                                       phbalReg_SocketWin_DataParams_t * pDataParams
                                       )
{
#ifdef _WIN32
    int32_t iResult;
    uint8_t bMode;
    uint8_t bCommand[18];
    phStatus_t status;
    phStatus_t statusTmp;
    uint16_t wValue = 0;
    uint16_t wRespLen;
    uint8_t pRxBuffer[128];
    struct addrinfo *result = NULL;
    struct addrinfo *ptr = NULL;
    struct addrinfo hints;

    if (pDataParams->dwServerSocket != INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* Initialize Winsock */
    iResult = WSAStartup(MAKEWORD(2, 2), (WSADATA *)pDataParams->pWsaData);
    if (iResult != 0)
    {
        WSACleanup(); /* PRQA S 3200 */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    ZeroMemory( &hints, sizeof(hints) ); /* PRQA S 3200 */
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = IPPROTO_TCP;

    /* Resolve the server address and port */
    iResult = getaddrinfo((char*)pDataParams->pSocketHostName, (char*)pDataParams->pSocketPortName, &hints, &result);
    if ( iResult != 0 )
    {
        WSACleanup(); /* PRQA S 3200 */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Attempt to connect to an address until one succeeds */
    for (ptr = result; ptr != NULL ; ptr = ptr->ai_next)
    {
        /* Create a SOCKET for connecting to server */
        pDataParams->dwServerSocket = (uint32_t)socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol);
        if (pDataParams->dwServerSocket == INVALID_SOCKET)
        {
            freeaddrinfo(result);
            WSACleanup(); /* PRQA S 3200 */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* Connect to server. */
        iResult = connect( pDataParams->dwServerSocket, ptr->ai_addr, (int32_t)ptr->ai_addrlen);
        if (iResult == SOCKET_ERROR)
        {
            iResult = closesocket(pDataParams->dwServerSocket);
            pDataParams->dwServerSocket = INVALID_SOCKET;
            if (iResult != 0)
            {
                break;
            }
        }
        else
        {
            /* On telnet mode remove welcome message */
            if(pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_TELNET)
            {
                /* read current IO timeout */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
                /* set IO timeout to 1ms  to speed up timeout error */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 1000));

                /* Recv output after successful login */
                iResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)pRxBuffer, sizeof(pRxBuffer), 0);

                /* restore IO timeout use statusTmp to not overwrite response from send */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

                if (iResult == 0 || iResult == SOCKET_ERROR)
                {
                    iResult = closesocket(pDataParams->dwServerSocket);
                    pDataParams->dwServerSocket = INVALID_SOCKET;
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }
            }
            break;
        }
    }

    freeaddrinfo(result);

    if (pDataParams->dwServerSocket == INVALID_SOCKET)
    {
        WSACleanup(); /* PRQA S 3200 */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* if there is a username set, authenticate with the simulation server */
    if (strlen((char *) pDataParams->pUserName) > 0)
    {
        bMode = pDataParams->bMode;
        pDataParams->bMode = PHBAL_REG_SOCKETWIN_MODE_RAW;

        sprintf((char*)bCommand, "user%s\n", pDataParams->pUserName); /* PRQA S 3200 */

        status = phbalReg_SocketWin_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCommand,
            (uint16_t)strlen((char*)bCommand),
            (uint16_t) sizeof(bCommand),
            bCommand,
            &wRespLen);

        pDataParams->bMode = bMode;

        PH_CHECK_SUCCESS(status);

        /* Check for ACK string */
        if (memcmp(bCommand, PHBAL_REG_SOCKETWIN_ACK_RX, strlen(PHBAL_REG_SOCKETWIN_ACK_RX)) != 0)
        {
            status = phbalReg_SocketWin_ClosePort(pDataParams);
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_ClosePort(
                                        phbalReg_SocketWin_DataParams_t * pDataParams
                                        )
{
#ifdef _WIN32
    LONG lResult;
    int32_t iResult1, iResult2;

    if (pDataParams->dwServerSocket == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* cleanup */
    lResult = shutdown(pDataParams->dwServerSocket, SD_BOTH);
    iResult1 = closesocket(pDataParams->dwServerSocket);
    pDataParams->dwServerSocket = INVALID_SOCKET;
    iResult2 = WSACleanup(); /* PRQA S 3200 */
    if (lResult != S_OK || iResult1 == SOCKET_ERROR || iResult2 == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_Exchange(
                                       phbalReg_SocketWin_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint16_t wRxBufSize,
                                       uint8_t * pRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
#ifdef _WIN32
    int32_t		PH_MEMLOC_REM iResult;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bProtBuffer[5];
    uint8_t *   PH_MEMLOC_REM pTmpBuffer;
    uint16_t    PH_MEMLOC_REM wTmpBufferLen;
    uint16_t    PH_MEMLOC_REM wRxLengthTmp;
    uint8_t     PH_MEMLOC_REM bEchoBuffer[100];
    int32_t     PH_MEMLOC_REM iTmpResult;

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Parameter check */
    if (wTxLength == 0 && pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_HEX)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Use dummy storage for RxLength if not given */
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLengthTmp;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* Set TxBuffer */
    pTmpBuffer = pDataParams->pTxBuffer;
    wTmpBufferLen = pDataParams->wTxBufSize;

    /* Tx-Coding for different protocols */
    switch (pDataParams->bMode)
    {
        /* I2C mode */
    case PHBAL_REG_SOCKETWIN_MODE_I2C:
        /* Read operation */
        if (wTxLength == 1)
        {
            /* Perform write operation with zero data upfront */
            bProtBuffer[0] = 0x42;
            /* shift address and clear RD/NWR bit to indicate write operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1);
            /* append register address */
            bProtBuffer[2] = pTxBuffer[0];

            /* Convert to string */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_HexToString(
                bProtBuffer,
                3,
                wTmpBufferLen,
                pTmpBuffer));

            /* Send Write */
            wTxLength = (uint16_t)strlen((char*)pTmpBuffer);
            iResult = send((SOCKET)pDataParams->dwServerSocket, (char*)pTmpBuffer, (int)wTxLength, 0);
            if ((uint16_t)iResult != wTxLength)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Receive response */
            iResult = recv(
                (SOCKET)pDataParams->dwServerSocket,
                (char*)pTmpBuffer,
                (int)wTmpBufferLen,
                0);

            /* Interface error check */
            if (iResult == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Timeout error check */
            if (iResult == SOCKET_ERROR)
            {
                return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
            }

            /* Retrieve received length */
            *pRxLength = (uint16_t)iResult;

            /* Length check */
            if (*pRxLength < 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
            pTmpBuffer[*pRxLength - 2] = 0;

            /* Convert to HEX format */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_StringToHex(pTmpBuffer, pRxLength));

            /* Length check */
            if (*pRxLength != 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* ACK check */
            if (pTmpBuffer[1] != 0x02)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* reset TxLength */
            wTxLength = 0;

            bProtBuffer[0] = (uint8_t)(0x80 | (uint8_t)(wRxBufSize & 0x7F));
            /* shift address and set RD/NWR bit to indicate read operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1) | 0x01;
        }
        /* Write operation */
        else
        {
            bProtBuffer[0] = (uint8_t)(0x40 | (uint8_t)((wTxLength + 1) & 0x3F));
            /* shift address and clear RD/NWR bit to indicate write operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1);
        }

        /* Convert to string */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_HexToString(
            bProtBuffer,
            2,
            wTmpBufferLen,
            pTmpBuffer));

        /* Adjust TxBuffer pointer */
        pTmpBuffer[4] = '\0';
        wTmpBufferLen = wTmpBufferLen - (uint16_t)strlen((char*)pTmpBuffer);
        pTmpBuffer += strlen((char*)pTmpBuffer);
        break;

        /* SPI mode */
    case PHBAL_REG_SOCKETWIN_MODE_SPI:
        bProtBuffer[0] = (uint8_t)(wTxLength << 3 >> 8);
        bProtBuffer[1] = (uint8_t)(wTxLength << 3);

        /* Convert to string */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_HexToString(
            bProtBuffer,
            2,
            wTmpBufferLen,
            pTmpBuffer));

        /* Adjust TxBuffer pointer */
        pTmpBuffer[4] = '\0';
        wTmpBufferLen = wTmpBufferLen - (uint16_t)strlen((char*)pTmpBuffer);
        pTmpBuffer += strlen((char*)pTmpBuffer);
        break;

        /* nothing to do for other protocols */
    default:
        break;
    }

    /* Convert from HEX format if neccessary */
    if (pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_RAW && pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_HEX &&
        pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_TELNET)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_HexToString(
            pTxBuffer,
            wTxLength,
            wTmpBufferLen,
            pTmpBuffer));

        pTxBuffer = pDataParams->pTxBuffer;
        wTxLength = (uint16_t)strlen((char*)pDataParams->pTxBuffer);
    }
    else if(pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_HEX)
    {
        /* String check */
        if (wTxLength != (uint16_t)strlen((char*)pTxBuffer))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
    }

	if(socketwinCallback != NULL)
	{
		/* Log the Data Sent to ISS */
		socketwinCallback(pTxBuffer, wTxLength, SEND_DATA);
	}
	if (wTxLength != 0)
    {
        /* Check if pTxBuffer contains \n and cut if true */
        if(pTxBuffer[wTxLength - 1] == '\n' && pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_TELNET)
            wTxLength -= 1;

        /* Send TxBuffer */
        iResult = send((SOCKET)pDataParams->dwServerSocket, (char*)pTxBuffer, (int)wTxLength, 0);
        if ((uint16_t)iResult != wTxLength)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        if(pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_TELNET)
        {
            /* Send \r\n */
            iResult = send((SOCKET)pDataParams->dwServerSocket, "\r\n", 2, 0);

            if ((uint16_t)iResult != 2)
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);

            pTmpBuffer = pTxBuffer;
            /* Recv Echo */
            while (wTxLength > sizeof(bEchoBuffer))
            {
                iResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)bEchoBuffer, sizeof(bEchoBuffer), 0);
                /* Ceck if recv length and data is ok */
                if (iResult != sizeof(bEchoBuffer) || strncmp((char*)bEchoBuffer, (char*)pTmpBuffer, sizeof(bEchoBuffer)))
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                wTxLength -= sizeof(bEchoBuffer);
                pTmpBuffer += sizeof(bEchoBuffer);
            }
            iResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)bEchoBuffer, wTxLength, 0);
            /* Ceck if recv length is ok */
            if(iResult != wTxLength || strncmp((char*)bEchoBuffer, (char*)pTmpBuffer, wTxLength))
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /*Recv \r\n */
            iResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)bEchoBuffer, 2, 0);
            if(iResult != 2 || bEchoBuffer[0] != '\r' || bEchoBuffer[1] != '\n')
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
        }
    }

    /* Read data from socket */
    if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_RAW || pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_HEX || pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_TELNET)
    {
        iResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)pRxBuffer, (int)wRxBufSize, 0);

        /* Interface error check */
        if (iResult == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* Timeout error check */
        if (iResult == SOCKET_ERROR)
        {
            return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
        }

        /* Retrieve received length */
        *pRxLength = (uint16_t)iResult;

        if(pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_TELNET)
        {
            if (pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_HEX)
            {
                /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
                pRxBuffer[*pRxLength - 1] = 0;

				if(socketwinCallback != NULL)
				{
					/* Log the Data received from ISS */
					socketwinCallback(pRxBuffer, *pRxLength, RECEIVE_DATA);
				}
            }
        }
        else
        {
            /* 3 case:
            * 1. received in \r\n
            * 2. received without \r\n
            * 3. received with just \r */

            /* case 3: */
            if (*pRxLength >= 1 && pRxBuffer[*pRxLength - 1] == '\r')
            {
                /* Just wait for \n */
                iTmpResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)bEchoBuffer, 1, 0);
                if (iTmpResult != 1 || bEchoBuffer[0] != '\n')
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }
                *pRxLength = *pRxLength - 1;
            }
            /* case 1: */
            else if (*pRxLength >= 2 && pRxBuffer[*pRxLength - 2] == '\r' && pRxBuffer[*pRxLength - 1] == '\n')
            {
                *pRxLength = *pRxLength - 2;
            }
            /* case 2: */
            else
            {
                /* wait for \r\n */
                iTmpResult = recv((SOCKET)pDataParams->dwServerSocket, (char*)bEchoBuffer, 2, 0);
                if (iTmpResult != 2 || bEchoBuffer[0] != '\r' || bEchoBuffer[1] != '\n')
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }
            }
        }
    }
    else
    {
        /* Reception loop */
        wTmpBufferLen = 0;
        do
        {
            iResult = recv(
                (SOCKET)pDataParams->dwServerSocket,
                (char*)&pDataParams->pTxBuffer[wTmpBufferLen],
                (int)(pDataParams->wTxBufSize - wTmpBufferLen),
                0);

            /* Interface error check */
            if (iResult == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Timeout error check */
            if (iResult == SOCKET_ERROR)
            {
                return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
            }

            wTmpBufferLen = wTmpBufferLen + (uint16_t)iResult;
        }
        while ((iResult >= 2) && (pDataParams->pTxBuffer[wTmpBufferLen - 2] != 0x0D) && (pDataParams->pTxBuffer[wTmpBufferLen - 1] != 0x0A));

        /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
        pDataParams->pTxBuffer[wTmpBufferLen - 2] = 0;
        --wTmpBufferLen;

        /* Convert to HEX format if neccessary */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SocketWin_StringToHex(pDataParams->pTxBuffer, &wTmpBufferLen));

        /* Set initial RxBuffer pointer */
        pTmpBuffer = pDataParams->pTxBuffer;

        /* Rx-Coding for different protocols */
        switch (pDataParams->bMode)
        {
            /* I2C mode */
        case PHBAL_REG_SOCKETWIN_MODE_I2C:
            /* Read operation */
            if (bProtBuffer[0] & 0x80) /* PRQA S 3353 */
            {
                /* ACK check */
                if (pTmpBuffer[0] != 0x01)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Advance to data */
                ++pTmpBuffer;
                --wTmpBufferLen;
            }
            /* Write operation */
            else
            {
                /* Length check */
                if (wTmpBufferLen != 2)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Ignore length byte */
                ++pTmpBuffer;
                --wTmpBufferLen;

                /* ACK check */
                if (pTmpBuffer[0] != (bProtBuffer[0] & 0x3F))
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Advance to data */
                ++pTmpBuffer;
                --wTmpBufferLen;
            }
            break;

            /* SPI mode */
        case PHBAL_REG_SOCKETWIN_MODE_SPI:

            /* Length check */
            if (wTmpBufferLen < 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Length word check */
            if ((pTmpBuffer[0] != bProtBuffer[0]) ||
                (pTmpBuffer[1] != bProtBuffer[1]))
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Advance to data */
            pTmpBuffer += 2;
            wTmpBufferLen = wTmpBufferLen - 2;
            break;

            /* nothing to do for other protocols */
        default:
            break;
        }

        /* Copy resulting HEX data to receive buffer */
        if (wTmpBufferLen > 0)
        {
            if (wTmpBufferLen <= wRxBufSize)
            {
                memcpy(pRxBuffer, pTmpBuffer, wTmpBufferLen);  /* PRQA S 3200 */
                *pRxLength = wTmpBufferLen;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || pTxBuffer || wTxLength || wRxBufSize || pRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_SetConfig(
                                        phbalReg_SocketWin_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t wValue
                                        )
{
#ifdef _WIN32
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM bMode;
    uint8_t     PH_MEMLOC_REM bCommand[20];
    uint16_t    PH_MEMLOC_REM wRespLen;
    uint32_t    PH_MEMLOC_REM dwOptVal;
    int32_t     PH_MEMLOC_REM dOptLen;

    switch (wConfig)
    {
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        /* set new timout */
        dOptLen = sizeof(dwOptVal);
        dwOptVal = (uint32_t)wValue;
        dOptLen = setsockopt(pDataParams->dwServerSocket, SOL_SOCKET, SO_SNDTIMEO, (char*)&dwOptVal, (int)dOptLen);
        if (dOptLen != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        /* set new timout */
        dOptLen = sizeof(dwOptVal);
        dwOptVal = (uint32_t)wValue;
        dOptLen = setsockopt(pDataParams->dwServerSocket, SOL_SOCKET, SO_RCVTIMEO, (char*)&dwOptVal, (int)dOptLen);
        if (dOptLen != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_MODE:

        /* Check operating mode */
        switch (wValue)
        {
        case PHBAL_REG_SOCKETWIN_MODE_RAW:
        case PHBAL_REG_SOCKETWIN_MODE_RS232:
        case PHBAL_REG_SOCKETWIN_MODE_I2C:
        case PHBAL_REG_SOCKETWIN_MODE_SPI:
        case PHBAL_REG_SOCKETWIN_MODE_HEX:
            pDataParams->bMode = (uint8_t)wValue;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_BITRATE:

        /* This is not valid for RAW mode */
        if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_RAW || pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_HEX)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
        /* Check if the value is valid for RS232 mode */
        else if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_RS232)
        {
            if (wValue > PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_1228800)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* Check if the value is valid for I2C mode */
        else if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_I2C)
        {
            if ((wValue < PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_10K) ||
                (wValue > PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_400K) &&
                (
                (wValue < PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2CL_1M) ||
                (wValue > PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2CL_5M)
                ))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* Check if the value is valid for SPI mode */
        else if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_SPI)
        {
            if ((wValue < PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_10K) ||
                (wValue > PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_8M))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* This should never happen */
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        /* save configuration */
        pDataParams->bConfigValue = (uint8_t)wValue;

        /* backup current mode */
        bMode = pDataParams->bMode;

        /* set mode to RAW */
        pDataParams->bMode = PHBAL_REG_SOCKETWIN_MODE_RAW;

        /* build config command */
        sprintf((char*)bCommand, "config%02X\n", pDataParams->bConfigValue);  /* PRQA S 3200 */

        /* send config command */
        status = phbalReg_SocketWin_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCommand,
            (uint16_t)strlen((char*)bCommand),
            sizeof(bCommand),
            bCommand,
            &wRespLen);

        /* restore previous mode */
        pDataParams->bMode = bMode;

        /* Check success */
        PH_CHECK_SUCCESS(status);
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_I2C_SLAVE_ADDR:

        pDataParams->bI2cSlaveAddr = (uint8_t)wValue;
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_SETCARD:

        /* backup current mode */
        bMode = pDataParams->bMode;

        /* set mode to RAW */
        pDataParams->bMode = PHBAL_REG_SOCKETWIN_MODE_RAW;

        /* build setcard command */
        sprintf((char*)bCommand, "setcard%02X%02X\n", (uint8_t)(wValue >> 8), (uint8_t)(wValue));  /* PRQA S 3200 */

        /* send config command */
        status = phbalReg_SocketWin_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCommand,
            (uint16_t)strlen((char*)bCommand),
            sizeof(bCommand),
            bCommand,
            &wRespLen);

        /* restore previous mode */
        pDataParams->bMode = bMode;

        /* Check success */
        PH_CHECK_SUCCESS(status);
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_RESET:

        if (wValue)
        {
            /* backup current mode */
            bMode = pDataParams->bMode;

            /* set mode to RAW */
            pDataParams->bMode = PHBAL_REG_SOCKETWIN_MODE_RAW;

            /* build reset command */
            sprintf((char*)bCommand, "reset\n");  /* PRQA S 3200 */

            /* send reset command */
            status = phbalReg_SocketWin_Exchange(
                pDataParams,
                PH_EXCHANGE_DEFAULT,
                bCommand,
                (uint16_t)strlen((char*)bCommand),
                sizeof(bCommand),
                bCommand,
                &wRespLen);

            /* Check success */
            PH_CHECK_SUCCESS(status);

            if (memcmp(bCommand, "ACK", 3))
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wConfig || wValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_GetConfig(
                                        phbalReg_SocketWin_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t * pValue
                                        )
{
#ifdef _WIN32
    uint32_t    PH_MEMLOC_REM dwOptVal;
    int32_t     PH_MEMLOC_REM dOptLen;

    switch (wConfig)
    {
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        /* retrieve timout */
        dOptLen = sizeof(dwOptVal);
        dOptLen = getsockopt(pDataParams->dwServerSocket, SOL_SOCKET, SO_SNDTIMEO, (char*)&dwOptVal, (int*)&dOptLen);
        if (dOptLen != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* return timeout */
        if (dwOptVal > 0xFFFF)
        {
            *pValue = 0xFFFF;
        }
        else
        {
            *pValue = (uint16_t)dwOptVal;
        }
        break;

    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
        /* retrieve timout */
        dOptLen = sizeof(dwOptVal);
        dOptLen = getsockopt(pDataParams->dwServerSocket, SOL_SOCKET, SO_RCVTIMEO, (char*)&dwOptVal, (int*)&dOptLen);
        if (dOptLen != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* return timeout */
        if (dwOptVal > 0xFFFF)
        {
            *pValue = 0xFFFF;
        }
        else
        {
            *pValue = (uint16_t)dwOptVal;
        }
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_MODE:
        *pValue = (uint16_t)pDataParams->bMode;
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_BITRATE:
        *pValue = (uint16_t)pDataParams->bConfigValue;
        break;

    case PHBAL_REG_SOCKETWIN_CONFIG_I2C_SLAVE_ADDR:
        *pValue = (uint16_t)pDataParams->bI2cSlaveAddr;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams || wConfig || pValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SocketWin_RegisterCallback(phbalReg_SocketWin_DataParams_t * pDataParams, TySocketWinCallback callback)
{
	pDataParams =  NULL; /* to Avoid compiler warnings */
	socketwinCallback = callback;

	 return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_SOCKETWIN */
