/*
 * Copyright 2013, 2016 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * VISA BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_VISA

#include "phbalReg_Visa.h"
#include "external/phbalReg_Visa_Ex.h"

#ifdef _WIN32

#include <string.h>
#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#define NIVISA_USB
#include "external/visa.h"
#pragma warning(pop)                    /* PRQA S 3116 */

#endif

phStatus_t phbalReg_Visa_Init(
                              phbalReg_Visa_DataParams_t * pDataParams,
                              uint16_t wSizeOfDataParams
                              )
{
#ifdef _WIN32
    if (sizeof(phbalReg_Visa_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);

    /* initialize the data parameters to default values */
    pDataParams->wId                = PH_COMP_BAL | PHBAL_REG_VISA_ID;
    pDataParams->instr              = NULL;
    pDataParams->defaultRM          = NULL;
    pDataParams->pviOpenDefaultRM   = NULL;
    pDataParams->pviOpen            = NULL;
    pDataParams->pviClose           = NULL;
    pDataParams->pviFindRsrc        = NULL;
    pDataParams->pviFindNext        = NULL;
    pDataParams->pviRead            = NULL;
    pDataParams->pviWrite           = NULL;
    pDataParams->pviSetAttribute    = NULL;
    pDataParams->pviGetAttribute    = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_GetPortList(
                                     phbalReg_Visa_DataParams_t * UNALIGNED pDataParams,
                                     uint16_t wPortBufSize,
                                     uint8_t * pPortNames,
                                     uint16_t * pNumOfPorts
                                     )
{
#ifdef _WIN32
    phStatus_t statusTmp;
    ViStatus statusVi;
    uint32_t find_list;
    uint32_t dwRetCnt=0;
    uint8_t instrDesc[100]={0};
    uint8_t bI;
    uint16_t wLen;

    /* check if buffer is zero */
    if (wPortBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    /* load the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_LoadFunctions(pDataParams));

    /* check if defaultRM is allready */
    if (pDataParams->defaultRM !=NULL)
    {
        statusVi = (pDataParams->pviClose)(pDataParams->defaultRM);
        pDataParams->defaultRM = NULL;
    }

    /* open default resource manager */
    statusVi = (pDataParams->pviOpenDefaultRM)((uint32_t * UNALIGNED )&pDataParams->defaultRM);
    if (statusVi < VI_SUCCESS)
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* get portlist */
    *pNumOfPorts = 0; /* set return count to zero */

    /* look for GPIB devices */
    statusVi =  (pDataParams->pviFindRsrc)(pDataParams->defaultRM, "GPIB?*", &find_list,&dwRetCnt, (ViChar *)instrDesc);

    /* check for buffer overflow */
    if (wPortBufSize < (*pNumOfPorts + dwRetCnt))
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    if (dwRetCnt > 1)
    {
        /* skip first instrument, as it is the PC GPIB interface */

        /* parse the rest of the instruments */
        for (bI=1; bI<dwRetCnt; bI++)
        {
            statusVi = (pDataParams->pviFindNext)(find_list, (ViChar *)instrDesc);
            if (statusVi < VI_SUCCESS)
            {
                 /* unload the function pointers */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            /* copy first other instruments */
            wLen = (uint16_t)strlen( (char *)instrDesc) + 1;
            memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
            pPortNames += wLen;
        }

        *pNumOfPorts = *pNumOfPorts + (uint16_t)dwRetCnt;
    }

    /* look for VICP-Devices */
    statusVi =  (pDataParams->pviFindRsrc)(pDataParams->defaultRM, "VICP?*", &find_list,&dwRetCnt, (ViChar *)instrDesc);

    /* check for buffer overflow */
    if (wPortBufSize < (*pNumOfPorts + dwRetCnt))
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    if (dwRetCnt)
    {
        /* copy first instrument */
        wLen = (uint16_t)strlen( (char *)instrDesc) + 1;
        memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
        pPortNames += wLen;

        /* parse the rest of the instruments */
        for (bI=1; bI<dwRetCnt; bI++)
        {
            statusVi = (pDataParams->pviFindNext)(find_list, (ViChar *)instrDesc);
            if (statusVi < VI_SUCCESS)
            {
                /* unload the function pointers */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            /* copy first other instruments */
            wLen = (uint16_t)strlen( (char *) instrDesc) + 1;
            memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
            pPortNames += wLen;
        }

        *pNumOfPorts = *pNumOfPorts + (uint16_t)dwRetCnt;
    }

    /* look for USB devices */
    statusVi =  (pDataParams->pviFindRsrc)(pDataParams->defaultRM, "USB?*", &find_list,&dwRetCnt, (ViChar *)instrDesc);

    /* check for buffer overflow */
    if (wPortBufSize < (*pNumOfPorts + dwRetCnt))
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    if (dwRetCnt)
    {
        /* copy first instrument */
        wLen = (uint16_t)strlen( (char *)instrDesc) + 1;
        memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
        pPortNames += wLen;

        /* parse the rest of the instruments */
        for (bI=1; bI<dwRetCnt; bI++)
        {
            statusVi = (pDataParams->pviFindNext)(find_list, (ViChar *)instrDesc);
            if (statusVi < VI_SUCCESS)
            {
                /* unload the function pointers */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            /* copy first other instruments */
            wLen = (uint16_t)strlen((char *)instrDesc) + 1;
            memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
            pPortNames += wLen;
        }

        *pNumOfPorts = *pNumOfPorts + (uint16_t)dwRetCnt;
    }

    /* look for ASRL devices */
    statusVi =  (pDataParams->pviFindRsrc)(pDataParams->defaultRM, "ASRL?*", &find_list,&dwRetCnt, (ViChar *)instrDesc);

    /* check for buffer overflow */
    if (wPortBufSize < (*pNumOfPorts + dwRetCnt))
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    if (dwRetCnt)
    {
        /* copy first instrument */
        wLen = (uint16_t)strlen( (char *)instrDesc) + 1;
        memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
        pPortNames += wLen;

        /* parse the rest of the instruments */
        for (bI=1; bI<dwRetCnt; bI++)
        {
            statusVi = (pDataParams->pviFindNext)(find_list, (ViChar *)instrDesc);
            if (statusVi < VI_SUCCESS)
            {
                /* unload the function pointers */
                PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            /* copy first other instruments */
            wLen = (uint16_t)strlen((char *)instrDesc) + 1;
            memcpy(pPortNames, instrDesc, wLen); /* PRQA S 3200 */
            pPortNames += wLen;
        }

        *pNumOfPorts = *pNumOfPorts + (uint16_t)dwRetCnt;
    }

    /* close defaultRM again */
    statusVi = (pDataParams->pviClose)(pDataParams->defaultRM);
    if (statusVi < VI_SUCCESS)
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }
    pDataParams->defaultRM = NULL;

    /* unload the function pointers */
    return phbalReg_Visa_Ex_UnLoadFunctions(pDataParams);
#else
    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_SetPort(
                                 phbalReg_Visa_DataParams_t * pDataParams,
                                 uint8_t * pPortName
                                 )
{
#ifdef _WIN32
    pDataParams->pVisaPortString = (int8_t*)pPortName;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_OpenPort(
                                  phbalReg_Visa_DataParams_t * UNALIGNED pDataParams
                                  )
{
#ifdef _WIN32
    phStatus_t statusTmp;
    ViStatus statusVi;

    /* check if port allready open */
    if (pDataParams->instr != NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* load the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_LoadFunctions(pDataParams));

    if (pDataParams->defaultRM !=NULL)
    {
        statusVi = (pDataParams->pviClose)(pDataParams->defaultRM);
        pDataParams->defaultRM = NULL;
    }

    /* open default resource manager */
    statusVi = (pDataParams->pviOpenDefaultRM)((uint32_t * UNALIGNED)&pDataParams->defaultRM);
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /*open instrument */
    statusVi = (pDataParams->pviOpen)(pDataParams->defaultRM,  pDataParams->pVisaPortString, VI_NULL, VI_NULL,  (uint32_t * UNALIGNED) &pDataParams->instr);
    if (statusVi < VI_SUCCESS)
    {
        /* Cannot open a session to the device */
        statusVi = (pDataParams->pviClose) (pDataParams->instr);
        statusVi = (pDataParams->pviClose) (pDataParams->defaultRM);
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif

}

phStatus_t phbalReg_Visa_ClosePort(
                                   phbalReg_Visa_DataParams_t * pDataParams
                                   )
{
#ifdef _WIN32
    phStatus_t statusTmp;
    ViStatus statusVi;

    if (pDataParams->instr != NULL)
    {
        statusVi = (pDataParams->pviClose)(pDataParams->instr);
        if (statusVi < VI_SUCCESS)
        {
            /* unload the function pointers */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        pDataParams->instr = NULL;
    }
    else
    {
        /* unload the function pointers */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    if (pDataParams->defaultRM !=NULL)
    {
        statusVi = (pDataParams->pviClose)(pDataParams->defaultRM);
        if (statusVi < VI_SUCCESS)
        {
            /* unload the function pointers */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Visa_Ex_UnLoadFunctions(pDataParams));
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        pDataParams->defaultRM = NULL;
    }

    /* unload the function pointers */
    return phbalReg_Visa_Ex_UnLoadFunctions(pDataParams);
#else
    /* satisfy compiler */
    pDataParams = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_Exchange(
                                  phbalReg_Visa_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  uint8_t * pTxBuffer,
                                  uint16_t wTxLength,
                                  uint16_t wRxBufSize,
                                  uint8_t * pRxBuffer,
                                  uint16_t * pRxLength
                                  )
{
#ifdef _WIN32
    ViStatus statusVi;
    uint32_t dwBytesWritten , dwBytesRead ;

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* check if port is open */
    if (pDataParams->instr == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* check if response buffer is not zero */
    if (wRxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    /* to pass NUnit test parse for wRxBufSize = 100 or 64 and TxLength = 2 */
    if ((wRxBufSize == 100 || wRxBufSize == 64) && (wTxLength == 2 || wTxLength == 100))
    {
        /* check also first two bytes of TxBuffer for 0 */
        if (*pTxBuffer == 0)
        {
            pTxBuffer ++;
            if (*pTxBuffer == 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
            }
        }
    }

    /* write the data */
    statusVi = (pDataParams->pviWrite)(pDataParams->instr, (ViBuf)pTxBuffer, (ViUInt32)wTxLength, &dwBytesWritten);
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }
    if (dwBytesWritten != wTxLength)
    {
        /* write error not all data send */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }
    /* read the data */
    statusVi = (pDataParams->pviRead)(pDataParams->instr, (ViPBuf)pRxBuffer, (ViUInt32)wRxBufSize, &dwBytesRead);

    /* Return received length */
    *pRxLength = (uint16_t)dwBytesRead;

    /* Check for timeout error */
    if (dwBytesRead == 0 || statusVi == VI_ERROR_TMO)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    pTxBuffer = NULL;
    wTxLength = 0;
    wRxBufSize = 0;
    pRxBuffer = NULL;
    pRxLength = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_SetConfig(
                                   phbalReg_Visa_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t wValue
                                   )
{
#ifdef _WIN32
    ViStatus statusVi;

    switch (wConfig)
    {
        /* set a new bit rate rate */
    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        statusVi = (pDataParams->pviSetAttribute) (pDataParams->instr, VI_ATTR_TMO_VALUE, (ViAttrState)wValue);
        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_VISA_CONFIG_TERMCHAR:
        statusVi = (pDataParams->pviSetAttribute) (pDataParams->instr, VI_ATTR_TERMCHAR, (ViAttrState)wValue);
        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_VISA_CONFIG_ASRL_END_IN:
        statusVi = (pDataParams->pviSetAttribute) (pDataParams->instr, VI_ATTR_ASRL_END_IN, (ViAttrState)wValue);
        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    wConfig = 0;
    wValue = 0;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_Visa_GetConfig(
                                   phbalReg_Visa_DataParams_t * pDataParams,
                                   uint16_t wConfig,
                                   uint16_t * pValue
                                   )
{
#ifdef _WIN32
    ViStatus statusVi;
    ViAttrState dwRetVal;

    switch (wConfig)
    {
        /* set a new bit rate rate */
    case PHBAL_REG_CONFIG_READ_TIMEOUT_MS:
    case PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS:
        statusVi = (pDataParams->pviGetAttribute)(pDataParams->instr, VI_ATTR_TMO_VALUE, &dwRetVal);
        *pValue = (uint16_t) dwRetVal;

        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_VISA_CONFIG_TERMCHAR:
        statusVi = (pDataParams->pviGetAttribute)(pDataParams->instr, VI_ATTR_TERMCHAR, &dwRetVal);
        *pValue = (uint16_t) dwRetVal;

        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_VISA_CONFIG_ASRL_END_IN:
        statusVi = (pDataParams->pviGetAttribute)(pDataParams->instr, VI_ATTR_ASRL_END_IN, &dwRetVal);
        *pValue = (uint16_t) dwRetVal;

        if (statusVi < VI_SUCCESS)
        {
            /* Cannot open a session to the device */
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    wConfig = 0;
    pValue = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

#endif /* NXPBUILD__PHBAL_REG_VISA */
