/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Amplifier AR75A250 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_AMP_AR75A250

#include <phdlAmp.h>
#include "phdlAmp_AR75A250.h"
#include "phdlAmp_AR75A250_Int.h"
#include "../phdlAmp_Int.h"

phStatus_t phdlAmp_AR75A250_Init(
                                 phdlAmp_AR75A250_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pBalRegDataParams
                                 )
{
    if (sizeof(phdlAmp_AR75A250_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_AMP);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_AMP | PHDL_AMP_AR75A250_ID;
    pDataParams->pBalRegDataParams		= pBalRegDataParams;
    pDataParams->wCurrentGain			= 0;
    pDataParams->bWait                  = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_AR75A250_SetPower(
                                     phdlAmp_AR75A250_DataParams_t * pDataParams,
                                     uint8_t bPowerOn
                                     )
{
    phStatus_t statusTmp;

    if (bPowerOn)
    {
        /* switch amplifier on with command "P1\n" */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_Int_Send(pDataParams,(uint8_t *) "P1\n",3));
    }
    else
    {
        /* switch amplifier off with command "P0\x0A"; */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_Int_Send(pDataParams,(uint8_t *) "P0\x0A",3));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_AR75A250_SetGain(
                                    phdlAmp_AR75A250_DataParams_t * pDataParams,
                                    uint16_t wGainValue
                                    )
{
    phStatus_t statusTmp;
    uint8_t aCmd[6] = "G0000";
    uint16_t wWaitTime;

    if (wGainValue > PHDL_AMP_AR75A250_MAX_GAIN)
    {
        wGainValue = PHDL_AMP_AR75A250_MAX_GAIN;
    }

    /* calculate time amplifier needs to change settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_AR75A250_Int_CalcWaitTime(pDataParams, wGainValue, &wWaitTime));

    /* the amplifier wants as termination character 0x0A:  <LF> ( without <CR> ) */
    aCmd[0] = 'G';
    aCmd[4] = (uint8_t)(wGainValue %10) + '0';
    wGainValue /= 10;
    aCmd[3] = (uint8_t)(wGainValue %10) + '0';
    wGainValue /= 10;
    aCmd[2] = (uint8_t)(wGainValue %10) + '0';
    wGainValue /= 10;
    aCmd[1] = (uint8_t)(wGainValue %10) + '0';
    aCmd[5] = '\x0A';

    if(pDataParams->bWait == PH_ON)
    {
        /* send command and wait for the settings to be applied */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_Int_SendWait(pDataParams,(uint8_t *)aCmd, 6, wWaitTime));
    }
    else
    {
        /* send command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_Int_Send(pDataParams,(uint8_t *)aCmd, 6));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_AR75A250_SetConfig(
                                      phdlAmp_AR75A250_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t wValue
                                      )
{
    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        /* check parameter */
        if ( wValue != PH_ON && wValue != PH_OFF )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_AMP);
        }
        /* assign value */
        pDataParams->bWait = (uint8_t) wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_AR75A250_GetConfig(
                                      phdlAmp_AR75A250_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t * pwValue
                                      )
{
    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        *pwValue = (uint16_t) pDataParams->bWait;
        break;

    case PHDL_AMP_CONFIG_MIN_GAIN:
        *pwValue = PHDL_AMP_AR75A250_MIN_GAIN;
        break;

    case PHDL_AMP_CONFIG_MAX_GAIN:
        *pwValue = PHDL_AMP_AR75A250_MAX_GAIN;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

#endif /* NXPBUILD__PHDL_AMP_AR75A250 */
