/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Amplifier BBA100 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_AMP_BBA100

#include <phdlAmp.h>
#include "phdlAmp_BBA100.h"
#include "phdlAmp_BBA100_Int.h"
#include "../phdlAmp_Int.h"

#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */

phStatus_t phdlAmp_BBA100_Int_CalcWaitTime(
    phdlAmp_BBA100_DataParams_t * pDataParams,
    uint16_t wNewGain,
    uint16_t * wWaitTime
    )
{
    uint16_t wGainDiff;

    /* calculate difference */
    if (pDataParams->wCurrentGain > wNewGain)
    {
        wGainDiff = pDataParams->wCurrentGain - wNewGain;
    }
    else
    {
        wGainDiff = wNewGain - pDataParams->wCurrentGain;
    }
    /* copy new value */
    pDataParams->wCurrentGain = wNewGain;

    if (wGainDiff > PHDL_AMP_BBA100_MAX_GAIN/2)
    {
        *wWaitTime = 1000;
    }
    else if (wGainDiff > PHDL_AMP_BBA100_MAX_GAIN/4)
    {
        *wWaitTime = 500;
    }
    else if (wGainDiff > PHDL_AMP_BBA100_MAX_GAIN/8)
    {
        *wWaitTime = 250;
    }
    else
    {
        *wWaitTime = 50;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_BBA100_Int_SendWaitFinished(
    phdlAmp_BBA100_DataParams_t * pDataParams,
    uint8_t * pCommand
    )
{
    phStatus_t statusTmp, statusTimeout;
    uint32_t dwValue=0;
    uint8_t pTmpBuffer[100];
    uint16_t wTmpLen=0;
    uint16_t wOldTimeoutValue=0;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wOldTimeoutValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 0));

    /* send data */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        1,
        pTmpBuffer,
        &wTmpLen);

    /* A timeout was expected so success */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
    }

    /* restore IO timeout use statusTimeout to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTimeout, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wOldTimeoutValue));

    /* And send the wait finished command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA100_Int_ExchangeGetUInt(pDataParams, (uint8_t *)"*OPC?\n", &dwValue, 1));
    /* *OPC? should always return 1 */
    if (dwValue != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }
    /* Error status is already checked with GetUInt function */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_BBA100_Int_ExchangeGetUInt(
    phdlAmp_BBA100_DataParams_t * pDataParams,
    uint8_t * pCommand,
    uint32_t * dwResponse,
    uint8_t numberDigits
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxLen;

    if (numberDigits > sizeof(bRx) - 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* send data */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        numberDigits+1,
        bRx,
        &wRxLen);

    /* Check if there was only numberDigits received */
    if (bRx[numberDigits] != '\n')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    PH_CHECK_SUCCESS(statusTmp);

    if (sscanf((const char *)bRx, "%d\n", dwResponse) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
    }

    /* Check if there was an error */
    return phdlAmp_BBA100_Int_CheckError(pDataParams);
}

phStatus_t phdlAmp_BBA100_Int_ExchangeGetResp(
    phdlAmp_BBA100_DataParams_t * pDataParams,
    uint8_t * pCommand,
    uint8_t * pResponse,
    uint16_t wRespSize
    )
{
    phStatus_t statusTmp;
    uint16_t wRxLen = 0;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        wRespSize,
        pResponse,
        &wRxLen);

    if (((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT) && wRxLen > 0)
    {
        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
    }

    PH_CHECK_SUCCESS(statusTmp);

    /* Check if there was an error */
    return phdlAmp_BBA100_Int_CheckError(pDataParams);
}

phStatus_t phdlAmp_BBA100_Int_CheckError(
    phdlAmp_BBA100_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint32_t errorCnt;
    uint8_t errorBuffer[100];
    uint32_t errorCode;
    uint16_t wRxLen;

    /* get error cnt */
    /* Assume that the error cnt just have one digit */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t *)("SYSTem:ERRor:COUNt?\n"),
        (uint16_t)strlen("SYSTem:ERRor:COUNt?\n"),
        2, /* one digit error cnt and \n */
        errorBuffer,
        &wRxLen);

    /* Check if there was only one digit error cnt */
    if (errorBuffer[1] != '\n')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    PH_CHECK_SUCCESS(statusTmp);
    if (sscanf((const char *)errorBuffer, "%d\n", &errorCnt) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
    }

    errorCode = 0;
    while (errorCnt != 0)
    {
        /* send data */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t *)("SYSTem:ERRor?\n"),
            (uint16_t)strlen("SYSTem:ERRor?\n"),
            sizeof(errorBuffer),
            errorBuffer,
            &wRxLen);

        /* If there was an timeout or success and data are available */
        if ((((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT) || ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)) && wRxLen > 0)
        {
            statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
            if (sscanf((const char *)errorBuffer, "%d,\n", &errorCode) != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }
        }
        errorCnt--;
    }
    if (errorCode != 0)
    {
        /* -201 Instrument cannot carry out operation because it is in local mode Switch to remote mode */
        if (errorCode == (uint32_t)(-201))
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_AMP);
        }
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

#endif /* NXPBUILD__PHDL_AMP_BBA100 */
