/*
 * Copyright 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Amplifier BBA150 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_AMP_BBA150
#include <string.h>
#include <phdlAmp.h>
#include "phdlAmp_BBA150.h"
#include "phdlAmp_BBA150_Int.h"
#include "../phdlAmp_Int.h"

#include <stdio.h>                      /* PRQA S 5124 */
phStatus_t phdlAmp_BBA150_Init(
                                 phdlAmp_BBA150_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pBalRegDataParams
                                 )
{
    if (sizeof(phdlAmp_BBA150_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_AMP);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_AMP | PHDL_AMP_BBA150_ID;
    pDataParams->pBalRegDataParams		= pBalRegDataParams;
    pDataParams->wCurrentGain			= PHDL_AMP_BBA150_MIN_GAIN;
    pDataParams->bWait                  = PH_ON;
    pDataParams->bSelectedPath          = 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

#pragma warning(disable:4996)           /* PRQA S 3116 */
phStatus_t phdlAmp_BBA150_SetPower(
                                     phdlAmp_BBA150_DataParams_t * pDataParams,
                                     uint8_t bPowerOn
                                     )
{
    phStatus_t statusTmp;
    uint32_t dwValue;
    uint8_t bRFOutputState;
    uint8_t bRetryCnt = 0;
    uint8_t aCmdBuffer[50];
    uint8_t aRespBuffer[50];

    if (bPowerOn)
    {
        /* Wait till the owner is correct set */
        bRetryCnt = 0;
        dwValue = PH_OFF;
        do
        {
            if (bRetryCnt != 0)
            {
                phTools_Sleep(1000);
            }
            /* get actual power state */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_ExchangeGetResp(pDataParams, (uint8_t *)"SYSTem:LOCK:OWNer?\r\n", aRespBuffer, sizeof(aRespBuffer)));
            if (strlen((const char *)aRespBuffer) < 3)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }
#ifndef WIN64_BUILD
            if (strnicmp((const char *)aRespBuffer, "LOC", 3) != 0)
            {
                dwValue = PH_ON;
            }
#endif
#pragma warning(default:4996)           /* PRQA S 3116 */
            bRetryCnt++;
        }
        while (bRetryCnt < 5 && dwValue != PH_ON);

        if (dwValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_AMP);
        }

        /* Get the current output state */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_ExchangeGetUInt(pDataParams, (uint8_t *)"RF:OUTPut:STATe?\r\n", &dwValue, 1));
        if (dwValue > PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }
        bRFOutputState = (uint8_t)dwValue;

        if (bRFOutputState == PH_OFF) /* If output is switched off */
        {
            /* get actual selected path */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_ExchangeGetResp(pDataParams, (uint8_t *)"RF:BAND:PATH?\r\n", aRespBuffer, sizeof(aRespBuffer)));
            if (sscanf((const char *)aRespBuffer, "%d\r\n", &dwValue) != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }

            /* If correct path is not selected -> select it */
            if (dwValue != pDataParams->bSelectedPath)
            {
                if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "RF:BAND:PATH %d\r\n", pDataParams->bSelectedPath) < 0)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
                }

                /* Select Path */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_SendWaitFinished(pDataParams, aCmdBuffer));

                /* Wait until path is set correct */
                bRetryCnt = 0;
                do
                {
                    if (bRetryCnt != 0)
                    {
                        phTools_Sleep(1000);
                    }
                    /* get actual selected path */
                    phdlAmp_BBA150_Int_ExchangeGetResp(pDataParams, (uint8_t *)"RF:BAND:PATH?\r\n", aRespBuffer, sizeof(aRespBuffer));
                    if (sscanf((const char *)aRespBuffer, "%d\r\n", &dwValue) != 1)
                    {
                        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
                    }
                    bRetryCnt++;
                }
                while (bRetryCnt < 10 && dwValue != pDataParams->bSelectedPath);

                if (dwValue != pDataParams->bSelectedPath)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
                }
            }

            /* set gain to current gain */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_SetGain(pDataParams, pDataParams->wCurrentGain));

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_SendWaitFinished(pDataParams, (uint8_t *)"RF:OUTPut:STATe 1\r\n"));       /* Switch On */

            /* Wait until state is set correct */
            bRetryCnt = 0;
            do
            {
                if (bRetryCnt != 0)
                {
                    phTools_Sleep(1000);
                }
                /* get actual output state */
                phdlAmp_BBA150_Int_ExchangeGetUInt(pDataParams, (uint8_t *)"RF:OUTPut:STATe?\r\n", &dwValue, 1);
                bRetryCnt++;
            }
            while (bRetryCnt < 10 && dwValue != PH_ON);

            if (dwValue != PH_ON)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
            }
            bRFOutputState = (uint8_t)dwValue;
        }
    }
    else
    {
        /* Get the current output state */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_ExchangeGetUInt(pDataParams, (uint8_t *)"RF:OUTPut:STATe?\r\n", &dwValue, 1));
        if (dwValue > PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }
        bRFOutputState = (uint8_t)dwValue;

        if (bRFOutputState == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_SendWaitFinished(pDataParams, (uint8_t *)"RF:OUTPut:STATe 0\r\n"));       /* Switch On */

            /* Wait until state is set correct */
            bRetryCnt = 0;
            do
            {
                if (bRetryCnt != 0)
                {
                    phTools_Sleep(1000);
                }
                /* get actual output state */
                phdlAmp_BBA150_Int_ExchangeGetUInt(pDataParams, (uint8_t *)"RF:OUTPut:STATe?\r\n", &dwValue, 1);
                bRetryCnt++;
            }
            while (bRetryCnt < 10 && dwValue != PH_OFF);

            if (dwValue != PH_OFF)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
            }
            bRFOutputState = (uint8_t)dwValue;
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_BBA150_SetGain(
                                    phdlAmp_BBA150_DataParams_t * pDataParams,
                                    uint16_t wGainValue
                                    )
{
    phStatus_t statusTmp;
    int16_t wCalcGain;
    uint8_t aCmdBuffer[50];
    uint8_t aRespBuffer[50];
    uint8_t * pRespBuffer = aRespBuffer;
    uint16_t wWaitTime;
    float32_t fGainValue;
    float32_t fGainActual;
    uint16_t wGainActual;

    if (wGainValue > PHDL_AMP_BBA150_MAX_GAIN)
    {
        wGainValue = PHDL_AMP_BBA150_MAX_GAIN;
    }

    wCalcGain = (1500 - PHDL_AMP_BBA150_MAX_GAIN + wGainValue);
    fGainValue = wCalcGain / 100.0f;

    if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "CONTrol%d:AMODe:FGAin %.2f\r\n", pDataParams->bSelectedPath, fGainValue) < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* Select Gain */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_SendWaitFinished(pDataParams, aCmdBuffer));

    /* calculate time amplifier needs to change settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_CalcWaitTime(pDataParams, wGainValue, &wWaitTime));

    if(pDataParams->bWait == PH_ON)
    {
        phTools_Sleep(wWaitTime);
        do
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_BBA150_Int_ExchangeGetResp(pDataParams, (uint8_t *)"CONTrol1:AMODe:FGAin?\r\n", pRespBuffer, sizeof(aRespBuffer)));
            if (strlen((const char *)pRespBuffer) < 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }
            if (sscanf((const char *)pRespBuffer, "%f", &fGainActual) != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }
            if (fGainActual > 6553.0)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_AMP);
            }
            /* Calc gain value and round */
            wGainActual = (uint16_t)(0.5 + (fGainActual*100 + PHDL_AMP_BBA150_MAX_GAIN - 1500));
        } while(wGainActual != wGainValue);
    }

    pDataParams->wCurrentGain = wGainValue;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_BBA150_SetConfig(
                                      phdlAmp_BBA150_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t wValue
                                      )
{
    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        /* check parameter */
        if ( wValue != PH_ON && wValue != PH_OFF )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_AMP);
        }
        /* assign value */
        pDataParams->bWait = (uint8_t) wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_BBA150_GetConfig(
                                      phdlAmp_BBA150_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t * pwValue
                                      )
{
    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        *pwValue = (uint16_t) pDataParams->bWait;
        break;

    case PHDL_AMP_CONFIG_MIN_GAIN:
        *pwValue = PHDL_AMP_BBA150_MIN_GAIN;
        break;

    case PHDL_AMP_CONFIG_MAX_GAIN:
        *pwValue = PHDL_AMP_BBA150_MAX_GAIN;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

#endif /* NXPBUILD__PHDL_AMP_BBA150 */
