/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Amplifier MPRFA Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_AMP_MPRFA

#include <phdlAmp.h>
#include "phdlAmp_MPRFA.h"
#include "phdlAmp_MPRFA_Int.h"
#include "../phdlAmp_Int.h"

#include <stdio.h>                      /* PRQA S 5124 */

phStatus_t phdlAmp_MPRFA_Init(
                                 phdlAmp_MPRFA_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 void * pBalRegDataParams
                                 )
{
    if (sizeof(phdlAmp_MPRFA_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_AMP);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_AMP | PHDL_AMP_MPRFA_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->wCurrentGain           = PHDL_AMP_MPRFA_MIN_GAIN_V2;
    pDataParams->wMinGain               = PHDL_AMP_MPRFA_NOT_DEF_GAIN;
    pDataParams->bWait                  = PH_ON;
    pDataParams->bOn                    = PH_OFF;
    pDataParams->bVersionString[0]      = 0;
    pDataParams->bSerialString[0]      = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_SetPower(
                                     phdlAmp_MPRFA_DataParams_t * pDataParams,
                                     uint8_t bPowerOn
                                     )
{
    phStatus_t statusTmp;
    uint16_t wValue;
    uint8_t aCmdBuffer[50];
    uint8_t aRespBuffer[50];
    uint32_t dwResponse = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetVersion(pDataParams));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetSerial(pDataParams));

    /* Get Status */
    if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "10\r\n") < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_ExchangeGetResp(pDataParams, aCmdBuffer, aRespBuffer, sizeof(aRespBuffer), NULL));
    if (sscanf((char *)&aRespBuffer[3], (char *)"%08d", &dwResponse) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* If status is not remote for version 2 -> error */
    if ((dwResponse & PHDL_AMP_MPRFA_LOCAL_BIT) == PHDL_AMP_MPRFA_LOCAL_BIT && pDataParams->bVersionString[8] == '2')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* if ON set the gain to the desired value */
    if (bPowerOn == PH_ON)
    {
        pDataParams->bOn = PH_ON;
        statusTmp = phdlAmp_MPRFA_SetGain(pDataParams, pDataParams->wCurrentGain);
        if (statusTmp != PH_ERR_SUCCESS)
        {
            pDataParams->bOn = PH_OFF;
        }
        return statusTmp;
    }
    else
    {
        /* else set gain to zero  and release remote access*/
        pDataParams->bOn = PH_ON; /* Just in case */
        wValue = pDataParams->wCurrentGain;
        statusTmp = phdlAmp_MPRFA_SetGain(pDataParams, 0);
        pDataParams->wCurrentGain = wValue;
        if (statusTmp == PH_ERR_SUCCESS)
        {
            pDataParams->bOn = PH_OFF;
        }
        PH_CHECK_SUCCESS(statusTmp);

        /* Release remote */
        if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "18\r\n") < 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_ExchangeGetResp(pDataParams, aCmdBuffer, aRespBuffer, sizeof(aRespBuffer), NULL));
        if (memcmp("18 00000000", aRespBuffer, 11) != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }
        return statusTmp;
    }
}

phStatus_t phdlAmp_MPRFA_SetGain(
                                    phdlAmp_MPRFA_DataParams_t * pDataParams,
                                    uint16_t wGainValue
                                    )
{
    phStatus_t statusTmp;
    uint8_t aCmdBuffer[50];
    uint8_t aExpectedRespBuffer[50];
    int32_t wExpectedLength = 0;
    uint8_t aRespBuffer[50];
    uint16_t wWaitTime;
    uint16_t wGainFull = 0;
    uint16_t wGainPart = 0;
    uint16_t wMaxGain = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_GetConfig(pDataParams, PHDL_AMP_CONFIG_MAX_GAIN, &wMaxGain));

    if (wGainValue > wMaxGain)
    {
        wGainValue = wMaxGain;
    }

    if (pDataParams->bOn != PH_ON)
    {
        /* If the amplifier not is switched on just save the gain value */
        pDataParams->wCurrentGain = wGainValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
    }

    wGainFull = wGainValue / 10;
    wGainPart = wGainValue % 10;

    if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "00 %02d.%d\r\n", wGainFull, wGainPart) < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }
    wExpectedLength = sprintf_s((char *)aExpectedRespBuffer, sizeof(aExpectedRespBuffer), "00 %08d", wGainFull);
    if (wExpectedLength < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* Select Gain */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_ExchangeGetResp(pDataParams, aCmdBuffer, aRespBuffer, sizeof(aRespBuffer), NULL));

    /* Check response */
    if (memcmp(aRespBuffer, aExpectedRespBuffer, wExpectedLength) != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    /* calculate time amplifier needs to change settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_CalcWaitTime(pDataParams, wGainValue, wMaxGain, &wWaitTime));

    if(pDataParams->bWait == PH_ON)
    {
        phTools_Sleep(wWaitTime);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_SetConfig(
                                      phdlAmp_MPRFA_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t wValue
                                      )
{
    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        /* check parameter */
        if ( wValue != PH_ON && wValue != PH_OFF )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_AMP);
        }
        /* assign value */
        pDataParams->bWait = (uint8_t) wValue;
        break;

    case PHDL_AMP_CONFIG_MIN_GAIN:
        /* check parameter, the maximum value of min gain is 10% */
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_AMP);
        }
        pDataParams->wMinGain = wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_GetConfig(
                                      phdlAmp_MPRFA_DataParams_t * pDataParams,
                                      uint16_t wIdentifier,
                                      uint16_t * pwValue
                                      )
{
    phStatus_t statusTmp;

    switch(wIdentifier)
    {
    case PHDL_AMP_CONFIG_WAIT:
        *pwValue = (uint16_t) pDataParams->bWait;
        break;

    case PHDL_AMP_CONFIG_MIN_GAIN:
        /* If the min gain is not set -> return the defaul values */
        if (pDataParams->wMinGain == PHDL_AMP_MPRFA_NOT_DEF_GAIN)
        {
            if (pDataParams->bVersionString[0] == 0x00)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetVersion(pDataParams));
            }
            if (pDataParams->bVersionString[8] == '2')
            {
                *pwValue = PHDL_AMP_MPRFA_MIN_GAIN_V2;
            }
            else
            {
                *pwValue = PHDL_AMP_MPRFA_MIN_GAIN_V3;
            }
        }
        else
        {
            *pwValue = pDataParams->wMinGain;
        }
        break;

    case PHDL_AMP_CONFIG_MAX_GAIN:
        if (pDataParams->bVersionString[0] == 0x00)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetVersion(pDataParams));
        }
        if (pDataParams->bVersionString[8] == '2')
        {
            *pwValue = PHDL_AMP_MPRFA_MAX_GAIN_V2;
        }
        else
        {
            *pwValue = PHDL_AMP_MPRFA_MAX_GAIN_V3;
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_AMP);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

#endif /* NXPBUILD__PHDL_AMP_MPRFA */
