/*
 * Copyright 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Amplifier MPRFA Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_AMP_MPRFA

#include <phdlAmp.h>
#include <phbalReg.h>
#include "phdlAmp_MPRFA.h"
#include "phdlAmp_MPRFA_Int.h"
#include "../phdlAmp_Int.h"

#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <stdio.h>              /* PRQA S 5124 */
#include <string>
#include <map>
#include <vector>
#include <algorithm>
#include <iostream>
#include <windows.h>
#include <cfgmgr32.h> // for MAX_DEVICE_ID_LEN
#pragma comment(lib, "Setupapi.lib")
#include <Setupapi.h>
#pragma warning(pop)            /* PRQA S 3116 */

phStatus_t phdlAmp_MPRFA_Int_CalcWaitTime(
    phdlAmp_MPRFA_DataParams_t * pDataParams,
    uint16_t wNewGain,
    uint16_t wMaxGain,
    uint16_t * wWaitTime
    )
{
    uint16_t wGainDiff;

    /* calculate difference */
    if (pDataParams->wCurrentGain > wNewGain)
    {
        wGainDiff = pDataParams->wCurrentGain - wNewGain;
    }
    else
    {
        wGainDiff = wNewGain - pDataParams->wCurrentGain;
    }
    /* copy new value */
    pDataParams->wCurrentGain = wNewGain;

    if (wGainDiff > wMaxGain/2)
    {
        *wWaitTime = 1000;
    }
    else if (wGainDiff > wMaxGain/4)
    {
        *wWaitTime = 500;
    }
    else if (wGainDiff > wMaxGain/8)
    {
        *wWaitTime = 250;
    }
    else
    {
        *wWaitTime = 50;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_Int_GetVersion(
    phdlAmp_MPRFA_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t aCmdBuffer[50];
    uint8_t aRespBuffer[50];
    uint16_t wRespLen = 0;

    /* Get the version of the amplifier */
    if (sprintf_s((char *)aCmdBuffer, sizeof(aCmdBuffer), "HELO\r\n") < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_ExchangeGetResp(pDataParams, aCmdBuffer, aRespBuffer, sizeof(aRespBuffer), &wRespLen));
    if (memcmp("HELO 0000 MPRFA", aRespBuffer, 15) != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }
    if (aRespBuffer[wRespLen - 1] == '\n' && aRespBuffer[wRespLen - 2] == '\r')
    {
        aRespBuffer[wRespLen - 2] = 0;
    }
    else if(aRespBuffer[wRespLen - 1] == '\r')
    {
        aRespBuffer[wRespLen - 1] = 0;
    }

    strncpy((char *)pDataParams->bVersionString, (const char *)(aRespBuffer + 10), PHDL_AMP_MPRFA_VERSION_SIZE);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}



phStatus_t phdlAmp_MPRFA_Int_GetSerial(
    phdlAmp_MPRFA_DataParams_t * pDataParams
    )
{
    HDEVINFO di;
    SP_DEVINFO_DATA did;
    std::vector<GUID> allClassGuids;
    int32_t dwIdx = 0;
    uint32_t dwGuidIdx = 0;
    TCHAR devInstanceId[MAX_DEVICE_ID_LEN];
    uint8_t aBuffer[1000];
    uint8_t aExpectedFriendlyName[100];
    phStatus_t statusTmp;
    int8_t *pPortName = NULL;
    ////HardwareInfo hd;

    /* SetupDiGetClassDevs() does not work if the class GUID is NULL and the
    // 4th parameter is different from DIGCF_ALLCLASSES. Therefore we collect all
    // the class GUIDs in the first step, and then collect all the connected devices
    // by feeding the collected GUIDs to SetupDiGetClassDevs() in the 2nd step

    // Step 1 */
    {
        di = SetupDiGetClassDevs(NULL,
                                 NULL,
                                 NULL,
                                 DIGCF_ALLCLASSES);

        if (di == INVALID_HANDLE_VALUE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }

        dwIdx = 0;
        do
        {
            SP_DEVINFO_DATA did1;
            did1.cbSize = sizeof(SP_DEVINFO_DATA);
            if (SetupDiEnumDeviceInfo(di, dwIdx, &did1) == FALSE)
            {
                if (::GetLastError() == ERROR_NO_MORE_ITEMS)
                {
                    break;
                }
                else
                {
                    /*error, but loop might continue? */
                }
            }
            if (std::find(allClassGuids.begin(), allClassGuids.end(), did1.ClassGuid) ==
                                                                            allClassGuids.end())
            {
                allClassGuids.push_back(did1.ClassGuid);
            }
            dwIdx++;
        } while(dwIdx != 0);

        if (SetupDiDestroyDeviceInfoList(di) == FALSE)
        {
            /*error, but should be ignored? */
        }
    }

    /* Step 2 */
    for (dwGuidIdx=0; dwGuidIdx<allClassGuids.size(); dwGuidIdx++)
    {
        HDEVINFO di1 = SetupDiGetClassDevs(&allClassGuids[dwGuidIdx],
                                          NULL,
                                          NULL,
                                          DIGCF_PRESENT);
        if (di1 == INVALID_HANDLE_VALUE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
        }

        dwIdx = 0;
        do
        {
            did.cbSize = sizeof(SP_DEVINFO_DATA);
            if (SetupDiEnumDeviceInfo(di1, dwIdx, &did) == FALSE)
            {
                if (::GetLastError() == ERROR_NO_MORE_ITEMS)
                {
                    break;
                }
                else
                {
                    /*error, but loop might continue? */
                }
            }
            memset(devInstanceId, 0, MAX_DEVICE_ID_LEN);
            if (SetupDiGetDeviceInstanceId(di1, &did, devInstanceId, MAX_DEVICE_ID_LEN, NULL) == FALSE)
            {
                /*error, but loop might continue? */
            }

            dwIdx++;


            /* Only evaluate MPRFA IDs */
            if (strncmp(devInstanceId, "USB\\VID_4242&PID_000A\\RFA", strlen("USB\\VID_4242&PID_000A\\RFA")) != 0)
            {
                continue;
            }

            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetHardwareInfoProperty(&di1, &did, SPDRP_DEVICEDESC, aBuffer, sizeof(aBuffer)));
            if (strcmp((char *)aBuffer, "MPRFA") != 0)
            {
                continue;
            }
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetHardwareInfoProperty(&di1, &did, SPDRP_MFG, aBuffer, sizeof(aBuffer)));
            if (strcmp((char *)aBuffer, "Micropross") != 0)
            {
                continue;
            }
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_MPRFA_Int_GetHardwareInfoProperty(&di1, &did, SPDRP_FRIENDLYNAME, aBuffer, sizeof(aBuffer)));

            /* Find start of com port string */
            pPortName = ((phbalReg_SerialWin_DataParams_t *)pDataParams->pBalRegDataParams)->pComPortString;
			/* [CAST] Basically the same type, got casted because of ph_TypeDefs change*/
            for (; strlen((char*)pPortName) >=4; pPortName++)
            {
                if (_strnicmp((char*)pPortName, "COM", 3) == 0)
                {
                    break;
                }
            }

            sprintf((char *)aExpectedFriendlyName, "MPRFA (%s)", pPortName);

            if (strcmp((char *)aBuffer, (char *)aExpectedFriendlyName) != 0)
            {
                continue;
            }

            strcpy((char *)pDataParams->bSerialString, devInstanceId + strlen("USB\\VID_4242&PID_000A\\"));
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
            /*FillHardwareInfo(di1, did, hd);

            result.insert(std::map<std::wstring, HardwareInfo>::value_type(devInstanceId, hd));*/
        } while(dwIdx != 0);

        if (SetupDiDestroyDeviceInfoList(di) == FALSE)
        {
            //error, but should be ignored?
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_Int_GetHardwareInfoProperty(
    void *di, void *did,
    uint32_t dwProperty,
    uint8_t *pData,
    uint16_t wDataSize
    )
{
    DWORD reqSize = 0;

    if (!SetupDiGetDeviceRegistryProperty(*((HDEVINFO *)di), (SP_DEVINFO_DATA *)did, dwProperty, NULL, NULL, 0, &reqSize))
    {
        //error, but loop might continue?
        ::GetLastError();
    }
    if (reqSize > wDataSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_AMP);
    }
    if (!SetupDiGetDeviceRegistryProperty(*((HDEVINFO *)di), (SP_DEVINFO_DATA *)did, dwProperty, NULL, pData, reqSize, NULL))
    {
        // device does not have this property set
        memset(pData, 0, wDataSize);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlAmp_MPRFA_Int_ExchangeGetResp(
    phdlAmp_MPRFA_DataParams_t * pDataParams,
    uint8_t * pCommand,
    uint8_t * pResponse,
    uint16_t wRespSize,
    uint16_t * pwRxLen
    )
{
    phStatus_t statusTmp;
    uint16_t wRxLen = 0;

    /* send data */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pCommand,
        (uint16_t)strlen((char *)pCommand),
        wRespSize,
        pResponse,
        &wRxLen);

    PH_CHECK_SUCCESS(statusTmp);

    if (memcmp(pCommand, pResponse, 2) != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_AMP);
    }

    if (pwRxLen != NULL)
    {
        *pwRxLen = wRxLen;
    }

    return statusTmp;
}

#endif /* NXPBUILD__PHDL_AMP_MPRFA */
