/*
 * Copyright 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */


/** \file
 * ISO3 FilterCalibration Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_NxpBuild.h>
#include <phbalReg.h>
#include <phTools.h>
#include <phhalHw.h>
#include <stdlib.h>

#ifdef NXPBUILD__PHDL_ISO3_FILTERCALIBRATION

#include "phdlISO3.h"
#include "phdlISO3_FilterCalibration_Cmd.h"
#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_reg.h"

#include <phhalHw.h>
#include <phdlOsci.h>
#include <phhalHw_ISO3_cmd.h>
#include <phdlOsci_ISO3SoftScope_Cmd.h>

phStatus_t phdlISO3_FilterCalibration_Cmd_Calibrate(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t bAntennaType,
    uint64_t qwCalibrationTimeStamp
    )
{
    phStatus_t statusTmp;
    IIRFilterCoeffs_t filter_coefficients;
    uint64_t qwServerTimeStamp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* Enable remote buffering */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_SetConfig(
        pDataParams,
        PHDL_ISO3_FILTERCALIBRATION_CONFIG_USE_REMOTE_BUFFERS,
        PH_ON));

    /* Buffer Maximum Length Sequence on server */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_GetMaximumLengthSequence(
        pDataParams,
        NULL,
        0,
        NULL))

    /* Buffer System Response on server */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_GetSystemResponse(
        pDataParams,
        NULL,
        0,
        NULL))

    /* Disable remote buffering */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_SetConfig(
        pDataParams,
        PHDL_ISO3_FILTERCALIBRATION_CONFIG_USE_REMOTE_BUFFERS,
        PH_OFF));

    /* Calculate filter coefficients on the server */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_CalculateFilterCoefficients(
        pDataParams,
        bAntennaType,
        qwCalibrationTimeStamp,
        PHDL_ISO3_FILTERCALIBRATION_REMOTE_BUFFER_MLS,
        PHDL_ISO3_FILTERCALIBRATION_REMOTE_BUFFER_SYS_RESP));

    /* Get calculated filter coefficients */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_GetFilterCoefficients(
        pDataParams,
        bAntennaType,
        &filter_coefficients,
        &qwServerTimeStamp));

    if(qwCalibrationTimeStamp != qwServerTimeStamp)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_DL_ISO3);
    }

    /* Apply filter coefficients */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_SetFilterCoefficients(
        pDataParams,
        bAntennaType,
        filter_coefficients,
        qwServerTimeStamp));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    phhalHw_ISO3_MlsWhiteNoiseCtrlReg_t dwWhiteNoiseCtrl;
    uint8_t agc_freeze[] = {0x01, 0x00, 0x00, 0x00};
    uint8_t agc_unfreeze[] = {0x00, 0x00, 0x00, 0x00};

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    /* Freeze AGC control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
        agc_freeze,
        sizeof(uint32_t)));

    dwWhiteNoiseCtrl.mlsSeed = 0x4111;
    dwWhiteNoiseCtrl.switchTx = PH_ON;
    dwWhiteNoiseCtrl.enableMls = PH_ON;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_MLS_WHITE_NOISE_CTRL,
        (uint8_t*)&dwWhiteNoiseCtrl,  /** value: 0xc0001111 */
        sizeof(uint32_t)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalISO3DataParams, PHHAL_HW_TIME_MILLISECONDS, 2000));

    /* after 2000ms stop playing MLS */
    dwWhiteNoiseCtrl.enableMls = PH_OFF;
    dwWhiteNoiseCtrl.switchTx = PH_OFF;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_MLS_WHITE_NOISE_CTRL,
        (uint8_t*)&dwWhiteNoiseCtrl,  /** value: 0x0 */
        sizeof(uint32_t)));

    /* Unfreeze AGC control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_FREEZE_GAIN_CTRL,
        agc_unfreeze,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_GetMaximumLengthSequence(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t* pbMlsBuffer,
    uint32_t dwMlsBufferSizeBytes,
    uint32_t* pdwMlsBufferValidBytes)
{
    phStatus_t statusTmp;
    phdlOsci_ISO3SoftScope_DataParams_t* pDlOsciDataParams;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    int64_t qwPrevBigBufferSizeNumSamples;
    uint16_t wUseRemoteBuffer;
    uint32_t dwCounter;

    /* Error checks */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);
    PH_ASSERT_NULL(pDataParams->pDlScopeDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_GetConfig(
        pDataParams,
        PHDL_ISO3_FILTERCALIBRATION_CONFIG_USE_REMOTE_BUFFERS,
        &wUseRemoteBuffer));

    if(!wUseRemoteBuffer &&
        (pbMlsBuffer == NULL || dwMlsBufferSizeBytes == 0 || pdwMlsBufferValidBytes == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_ISO3);
    }

    pDlOsciDataParams = (phdlOsci_ISO3SoftScope_DataParams_t*) pDataParams->pDlScopeDataParams;
    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    /* Configure scope */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_TRIGGER_SRC,
        PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_MLS));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        &qwPrevBigBufferSizeNumSamples));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        PHDL_ISO3_FILTERCALIBRATION_MAXIMUM_LENGTH_SEQUENCE_NUM_SAMPLES));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_DATA_SRC,
        PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_DAC));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED,
        PH_OFF));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_SAMPLE_SIZE,
        (dwMlsBufferSizeBytes/PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)));

    /* Play maximum length sequence */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(
        pDataParams));

    if(wUseRemoteBuffer)
    {
        /* Store samples into remote buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Cmd_StoreBigBufferDataInRemoteBuffer(
            pDlOsciDataParams,
            PHDL_ISO3_FILTERCALIBRATION_REMOTE_BUFFER_MLS,
            PHDL_ISO3_FILTERCALIBRATION_MAXIMUM_LENGTH_SEQUENCE_NUM_SAMPLES *
            PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE));
    }
    else
    {
        /* Get samples (without waveform header) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetWaveForm(
            pDlOsciDataParams,
            PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER,
            dwMlsBufferSizeBytes,
            pdwMlsBufferValidBytes,
            pbMlsBuffer,
            NULL));

    }

    /* Restore configs */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED,
        PH_ON));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        qwPrevBigBufferSizeNumSamples));

    if(!wUseRemoteBuffer)
    {
        for(dwCounter = 0; dwCounter <
            (dwMlsBufferSizeBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE); dwCounter++)
        {
            if(((uint16_t*)pbMlsBuffer)[dwCounter] != 0x7fff &&
                ((uint16_t*)pbMlsBuffer)[dwCounter] != 0x8001)
            {
                PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_DL_ISO3);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_GetSystemResponse(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t* pbSysResBuffer,
    uint32_t dwSysResBufferSizeBytes,
    uint32_t * pdwSysResBufferValidBytes)
{
    phStatus_t statusTmp;
    phdlOsci_ISO3SoftScope_DataParams_t * pDlOsciDataParams;
    phhalHw_ISO3_DataParams_t * pHalISO3DataParams;
    int64_t qwPrevBigBufferSizeNumSamples;
    uint16_t wUseRemoteBuffer;

    /* Error Checks */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);
    PH_ASSERT_NULL(pDataParams->pDlScopeDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_GetConfig(
        pDataParams,
        PHDL_ISO3_FILTERCALIBRATION_CONFIG_USE_REMOTE_BUFFERS,
        &wUseRemoteBuffer));

    if(!wUseRemoteBuffer &&
        (pbSysResBuffer == NULL || dwSysResBufferSizeBytes == 0 || pdwSysResBufferValidBytes == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_ISO3);
    }

    pDlOsciDataParams = (phdlOsci_ISO3SoftScope_DataParams_t*) pDataParams->pDlScopeDataParams;
    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    /* Configure scope */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_TRIGGER_SRC,
        PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_MLS));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        &qwPrevBigBufferSizeNumSamples));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        PHDL_ISO3_FILTERCALIBRATION_MAXIMUM_LENGTH_SEQUENCE_NUM_SAMPLES));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_DATA_SRC,
        PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_RX));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED,
        PH_OFF));

    /* Play sequence */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(
        pDataParams));

    if(wUseRemoteBuffer)
    {
        /* Store samples into remote buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Cmd_StoreBigBufferDataInRemoteBuffer(
            pDlOsciDataParams,
            PHDL_ISO3_FILTERCALIBRATION_REMOTE_BUFFER_SYS_RESP,
            PHDL_ISO3_FILTERCALIBRATION_MAXIMUM_LENGTH_SEQUENCE_NUM_SAMPLES *
            PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE));
    }
    else
    {
        /* Get samples (without waveform header) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetWaveForm(
            pDlOsciDataParams,
            PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER,
            dwSysResBufferSizeBytes,
            pdwSysResBufferValidBytes,
            pbSysResBuffer,
            NULL));
    }

    /* Restore configs */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED,
        PH_ON));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_SetConfig64(
        pDlOsciDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
        qwPrevBigBufferSizeNumSamples));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_GetFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t bAntenna,
    IIRFilterCoeffs_t* pCoeffs,
    uint64_t* pqwTimeStamp)
{
    phStatus_t statusTmp;
    uint32_t dwAntenna;
    uint8_t pInBuffer[sizeof(IIRFilterCoeffs_t) + sizeof(uint64_t)];
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDataParams->pHalDataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;
    dwAntenna = (uint32_t)bAntenna;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_SERVER_CMD_GET_COEFF,
        (uint8_t*)&dwAntenna,
        sizeof(dwAntenna)));

    /**  recv package: [ iir filter coeffs | time stamp ]  */

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_RecvServerResponse(
        pHalISO3DataParams,
        pInBuffer,
        sizeof(pInBuffer)));

    memcpy(pCoeffs, pInBuffer, sizeof(IIRFilterCoeffs_t));
    memcpy(pqwTimeStamp, pInBuffer + sizeof(IIRFilterCoeffs_t), sizeof(uint64_t));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_SetFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t bAntenna,
    IIRFilterCoeffs_t coeffs,
    uint64_t qwTimeStamp)
{
    phStatus_t statusTmp;
    uint32_t dwAntenna;
    uint8_t pOutBuffer[(sizeof(uint32_t) + sizeof(IIRFilterCoeffs_t)) + sizeof(uint64_t)];
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDataParams->pHalDataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    /** send package: [ 32-bit antenna type | iir filter coeffs | 64-bit time stamp ]  */

    dwAntenna = (uint32_t)bAntenna;
    memcpy(pOutBuffer, &dwAntenna, sizeof(uint32_t));
    memcpy(pOutBuffer + sizeof(uint32_t), &coeffs, sizeof(IIRFilterCoeffs_t));
    memcpy(pOutBuffer + sizeof(uint32_t) + sizeof(IIRFilterCoeffs_t), &qwTimeStamp,
        sizeof(uint64_t));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_SERVER_CMD_SET_COEFF,
        pOutBuffer,
        sizeof(pOutBuffer)));

    /* write sec 1 */
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_1A1,
        (uint8_t*)&coeffs.sec1.a1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_1A2,
        (uint8_t*)&coeffs.sec1.a2, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_1B0,
        (uint8_t*)&coeffs.sec1.b0, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_1B1,
        (uint8_t*)&coeffs.sec1.b1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_1B2,
        (uint8_t*)&coeffs.sec1.b2, sizeof(uint32_t));

    /* write sec 2 */
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_2A1,
        (uint8_t*)&coeffs.sec2.a1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_2A2,
        (uint8_t*)&coeffs.sec2.a2, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_2B0,
        (uint8_t*)&coeffs.sec2.b0, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_2B1,
        (uint8_t*)&coeffs.sec2.b1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_2B2,
        (uint8_t*)&coeffs.sec2.b2, sizeof(uint32_t));

    /* write sec 3 */
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_3A1,
        (uint8_t*)&coeffs.sec3.a1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_3A2,
        (uint8_t*)&coeffs.sec3.a2, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_3B0,
        (uint8_t*)&coeffs.sec3.b0, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_3B1,
        (uint8_t*)&coeffs.sec3.b1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_3B2,
        (uint8_t*)&coeffs.sec3.b2, sizeof(uint32_t));

    /* write sec 4 */
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_4A1,
        (uint8_t*)&coeffs.sec4.a1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_4A2,
        (uint8_t*)&coeffs.sec4.a2, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_4B0,
        (uint8_t*)&coeffs.sec4.b0, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_4B1,
        (uint8_t*)&coeffs.sec4.b1, sizeof(uint32_t));
    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_4B2,
        (uint8_t*)&coeffs.sec4.b2, sizeof(uint32_t));

    phhalHw_ISO3_Cmd_WriteFpgaRegister(pHalISO3DataParams, PHHAL_HW_ISO3_REG_IIRFILTER_GAIN,
        (uint8_t*)&coeffs.gain, sizeof(uint32_t));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_ISO3);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_StoreInRemoteBuffer(
                        phdlISO3_FilterCalibration_DataParams_t * pDataParams,
                        uint32_t dwRemoteBufferID,
                        uint8_t * pbData,
                        uint32_t dwDataNumBytes
                        )
{
    uint8_t* pbCmdParams;
    phStatus_t status;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDataParams->pHalDataParams);
    PH_ASSERT_NULL_PARAM(pbData, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(dwDataNumBytes, PH_COMP_HAL);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    if(dwDataNumBytes > UINT_MAX - sizeof(dwRemoteBufferID))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    pbCmdParams = (uint8_t*)malloc(dwDataNumBytes + sizeof(dwRemoteBufferID));
    memcpy(pbCmdParams, &dwRemoteBufferID, sizeof(dwRemoteBufferID));
    memcpy(pbCmdParams + sizeof(dwRemoteBufferID), pbData, dwDataNumBytes);

    status = phhalHw_ISO3_Cmd_SendServerCommand(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_SERVER_CMD_STORE_FROM_HOST,
        pbCmdParams,
        dwDataNumBytes + sizeof(dwRemoteBufferID));

    free(pbCmdParams);
    PH_CHECK_SUCCESS(status);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phdlISO3_FilterCalibration_Cmd_CalculateFilterCoefficients(
    phdlISO3_FilterCalibration_DataParams_t * pDataParams,
    uint8_t bAntennaType,
    uint64_t qwCalibrationTimeStamp,
    uint32_t dwBufferIDforMLS,
    uint32_t dwBufferIDforSysRes
    )
{
    phStatus_t statusTmp;
    uint32_t pdwParams[5];
    uint32_t dwServerSuccess = PH_ERR_SUCCESS;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDataParams->pHalDataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalDataParams;

    /* Let the server calculate filter coefficients and update time stamp. Afterwards
     * remote buffer are cleared and memory is freed.
     * The arguments are
     * [ Antenna type ][ Time stamp ][ Remote Buffer ID for MLS ][ Remote Buffer ID for sys resp ]
     */
    *((uint32_t*)(pdwParams + 0)) = bAntennaType;                                        /* 32-bit antenna type */
    *((uint64_t*)(pdwParams + 1)) = qwCalibrationTimeStamp;                              /* 64-bit time stamp */
    *((uint32_t*)(pdwParams + 2));
    *((uint32_t*)(pdwParams + 3)) = dwBufferIDforMLS;       /* 32-bit remote Buffer ID for MLS */
    *((uint32_t*)(pdwParams + 4)) = dwBufferIDforSysRes;  /* 32-bit remote Buffer ID for sig resp */

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_SERVER_CMD_CALIBRATE,
        (uint8_t*) pdwParams,
        sizeof(pdwParams)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_RecvServerResponse(
        pHalISO3DataParams,
        (uint8_t*)&dwServerSuccess,
        sizeof(dwServerSuccess)));

    /* Handle error types */
    switch (dwServerSuccess)
    {
    case PH_ERR_SUCCESS:
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_DL_ISO3);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}



#endif /* NXPBUILD__PHDL_ISO3_FILTERCALIBRATION */
