/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * AR75A250 Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_AR75A250

#include <phdlMstAmpOsc.h>
#include "phdlMstAmpOsc_AR75A250.h"
#include "phdlMstAmpOsc_AR75A250_Int.h"

#include <phdlAmp.h>
#include <phdlOsci.h>
#include <math.h>

phStatus_t phdlMstAmpOsc_AR75A250_Init(
                                       phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams,
                                       uint16_t wSizeOfDataParams,
                                       void * pDlAmpDataParams,
                                       void * pDlOsciDataParams,
                                       uint8_t bChannel
                                       )
{
    if (sizeof(phdlMstAmpOsc_AR75A250_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_MSTAMPOSC);
    }

    /* check channel parameter */
    if (bChannel < 1 || bChannel > 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pDlAmpDataParams);
    PH_ASSERT_NULL (pDlOsciDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_MSTAMPOSC | PHDL_MSTAMPOSC_AR75A250_ID;
    pDataParams->pDlAmpDataParams   =	pDlAmpDataParams;
    pDataParams->pDlOsciDataParams   =	pDlOsciDataParams;
    pDataParams->wMinFieldStrengthAmp = 0;
    pDataParams->wMaxFieldStrengthAmp = 0;
    pDataParams->bChannel = bChannel;
    pDataParams->bPrecision = 2;
    pDataParams->bCheckPrecision = PH_OFF;
    pDataParams->bCal = PH_OFF;
    pDataParams->wMinGainVal = 0;
    pDataParams->wMaxGainVal = 4095;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_AR75A250_InitMstAmpOsc(
    phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Init Osci */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_InitOsci(pDataParams->pDlOsciDataParams));

    /* read the maximum and minimum amplifier settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDataParams->pDlAmpDataParams, PHDL_AMP_CONFIG_MIN_GAIN, &pDataParams->wMinGainVal));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_GetConfig(pDataParams->pDlAmpDataParams, PHDL_AMP_CONFIG_MAX_GAIN, &pDataParams->wMaxGainVal));

    /* set amp gain to minimum */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal));

    /* switch on amplifier */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_ON));


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_AR75A250_SetConfig(
    phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t wValue
    )
{
    switch (wIdentifier)
    {
    case   PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* check value */
        if (wValue > 100)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }

        /* store value */
        pDataParams->bPrecision = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        /* store value */
        pDataParams->bCheckPrecision = (uint8_t)wValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_AR75A250_GetConfig(
    phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t * pwValue
    )
{
    switch (wIdentifier)
    {
    case PHDL_MSTAMPOSC_CONFIG_PRECISION:
        /* return precision */
        *pwValue = (uint16_t)pDataParams->bPrecision;
        break;

    case PHDL_MSTAMPOSC_CONFIG_CHECK_PRECISION:
        /* return check precision */
        *pwValue = (uint16_t)pDataParams->bCheckPrecision;
        break;
            case PHDL_MSTAMPOSC_CONFIG_MIN_FIELD:
        /* return min field */
        *pwValue = pDataParams->wMinFieldStrengthAmp;
        break;

    case PHDL_MSTAMPOSC_CONFIG_MAX_FIELD:
        /* return max field */
        *pwValue = pDataParams->wMaxFieldStrengthAmp;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_AR75A250_Cal(
                                      phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams
                                      )
{
    phStatus_t statusTmp;
    phStatus_t statusAmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    phStatus_t statusOsci;
    uint16_t wTemp;


    /* set gain value to Max */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMaxGainVal));

    /* read field strength */
    statusOsci = phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wTemp);
    if ( (statusOsci & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        statusTmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        return statusOsci;
    }
    pDataParams->wMaxFieldStrengthAmp = wTemp;

    /* set gain value to Min */
    statusAmp = phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinGainVal);
    if ( (statusAmp & PH_ERR_MASK) != PH_ERR_SUCCESS) /* In case of an error on setting gain to min: Switch of Amplifier */
    {
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        return statusAmp;
    }

    /* read field strength */
    statusOsci = phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wTemp);
    if ( (statusOsci & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        statusTmp = phdlAmp_SetPower(pDataParams->pDlAmpDataParams, PH_OFF);
        return statusOsci;
    }
    pDataParams->wMinFieldStrengthAmp = wTemp;

    /* set calibrated flag */
    pDataParams->bCal = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}


phStatus_t phdlMstAmpOsc_AR75A250_SetFieldStrength(
    phdlMstAmpOsc_AR75A250_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp;
    uint16_t wGainValue;
    uint16_t wFieldStrengthMeas;
    uint16_t wPrecsionsLimitLower = wFieldStrength - (uint16_t)((float32_t)wFieldStrength * (float32_t)pDataParams->bPrecision / 100 );
    uint16_t wPrecsionsLimitUpper = wFieldStrength + (uint16_t)((float32_t)wFieldStrength * (float32_t)pDataParams->bPrecision / 100 );
    uint8_t bCounter = 0;

    /* check if calibrated */
    if (pDataParams->bCal == PH_OFF)
    {
        /*return error code*/
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_NOT_CAL, PH_COMP_DL_MSTAMPOSC);
    }

    /* check field strength limits */
    if (wFieldStrength < pDataParams->wMinFieldStrengthAmp)
    {
        /* set amplification to minimum if too low */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, 0));

        /* return error code */
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_LOW, PH_COMP_DL_MSTAMPOSC);
    }
    if (wFieldStrength > pDataParams->wMaxFieldStrengthAmp)
    {
        /* set amplification to minimum if too low */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, 0));

        /* return error code */
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }
    do
    {
        /* calculate gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_AR75A250_Int_CalcGainVal(pDataParams, wFieldStrength, &wGainValue));

        /* set gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams,wGainValue));

        /* check precision */
        if (pDataParams->bCheckPrecision == PH_ON)
        {
            /* read field strength */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, &wFieldStrengthMeas));

            /* check if within the precision limits */
            if (wFieldStrengthMeas < wPrecsionsLimitLower || wFieldStrengthMeas > wPrecsionsLimitUpper)
            {
                /* if we tried to recalibrate 3 times we assume it is not possible */
                if (bCounter >= PHDL_MSTAMPOSC_NUM_OF_RECAL)
                {
                    /* set gain to min */
                    PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, 0));

                    /* allready tried to recalibrate, precision not achieveable */
                    return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
                }
                /* try to recalibrate, assume thermal drift */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_AR75A250_Cal(pDataParams));
                /* increase counter */
                bCounter++;
            }
            else
            {
                /* everthing is fine, we can leave */
                break;
            }
        }
    }
    while( pDataParams->bCheckPrecision == PH_ON );

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_AR75A250 */
