/*
 * Copyright 2019 - 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 Setup Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISO3

#include <phdlMstAmpOsc.h>
#include "phdlMstAmpOsc_ISO3.h"
#include "phdlMstAmpOsc_ISO3_Int.h"

#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_int.h"

#include <phhalHw_ISO3_Cmd.h>

phStatus_t phdlMstAmpOsc_ISO3_Init(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wSizeOfDataParams,
    void * pHalDataParams
    )
{
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pHalDataParams);

    if(sizeof(phdlMstAmpOsc_ISO3_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_MSTAMPOSC);
    }

    /* Check HAL data parameters */
    if (PH_GET_COMPCODE(pHalDataParams) != PH_COMP_HAL ||
        PH_GET_COMPID(pHalDataParams) != PHHAL_HW_ISO3_ID)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_ISO3);
    }

    pDataParams->wId                      = PH_COMP_DL_MSTAMPOSC | PHDL_MSTAMPOSC_ISO3_ID;
    pDataParams->pHalDataParams           = (void*) pHalDataParams;
    pDataParams->wMaxFieldStrength        = (uint16_t)8500;
    pDataParams->wMaxFieldStrengthConfigured = (uint16_t)8500;
    pDataParams->wMinFieldStrength        = (uint16_t) 0;
    pDataParams->dwDacGain                = (uint32_t) PHDL_MSTAMPOSC_ISO3_DAC_MIN_GAIN;
    pDataParams->bCurentAntennaType       = (uint8_t) PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1;
    pDataParams->dwFieldStrengthRatioA1   = (uint32_t) 0x0000DD54;
    pDataParams->dwFieldStrengthRatioA2   = (uint32_t) 0x0000DD54;
    pDataParams->dwFieldStrengthRatioA3   = (uint32_t) 0x0000DD54;

    pDataParams->bRxThresholdMode = PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_DISABLED;
    pDataParams->wMinRxThresholdFS = 750;
    pDataParams->wMaxRxThresholdFS = 7500;
    pDataParams->wMinRxThreshold = 800;
    pDataParams->wMaxRxThreshold = 512;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISO3_InitMstAmpOsc(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint32_t pTxBuffer[2];

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* set default attenuator values */
    phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_0DB , PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_0DB);
    phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_6DB , PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_6DB);
    phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_12DB , PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_12DB);
    phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_18DB , PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_18DB);
    phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_24DB , PHDL_MSTAMPOSC_ISO3_STEP_ATT_DEFAULT_24DB);

    /* get fieldstrength ratio vom box */
    /* antenna type 1 */
    pTxBuffer[0] = PHHAL_HW_ISO3_SERVER_CONF_FIELD_STRENGTH;
    pTxBuffer[1] = PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1;

    phhalHw_ISO3_Cmd_SendServerCommand(
        pHalDataParams,
        PHHAL_HW_ISO3_SERVER_CMD_GET_CONFIG,
        (uint8_t*)&pTxBuffer,
        sizeof(pTxBuffer));

    phhalHw_ISO3_Cmd_RecvServerResponse(
        pHalDataParams,
        (uint8_t*)&(pDataParams->dwFieldStrengthRatioA1),
        sizeof(pDataParams->dwFieldStrengthRatioA1));

    /* antenna type 2 */
    pTxBuffer[1] = PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2;

    phhalHw_ISO3_Cmd_SendServerCommand(
        pHalDataParams,
        PHHAL_HW_ISO3_SERVER_CMD_GET_CONFIG,
        (uint8_t*)&pTxBuffer,
        sizeof(pTxBuffer));

    phhalHw_ISO3_Cmd_RecvServerResponse(
        pHalDataParams,
        (uint8_t*)&(pDataParams->dwFieldStrengthRatioA2),
        sizeof(pDataParams->dwFieldStrengthRatioA2));

    /* antenna type 3 */
    pTxBuffer[1] = PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3;

    phhalHw_ISO3_Cmd_SendServerCommand(
        pHalDataParams,
        PHHAL_HW_ISO3_SERVER_CMD_GET_CONFIG,
        (uint8_t*)&pTxBuffer,
        sizeof(pTxBuffer));

    phhalHw_ISO3_Cmd_RecvServerResponse(
        pHalDataParams,
        (uint8_t*)&(pDataParams->dwFieldStrengthRatioA3),
        sizeof(pDataParams->dwFieldStrengthRatioA3));

    /* Make sure to set step attenuators to 0dB */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT, PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT, PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}


phStatus_t phdlMstAmpOsc_ISO3_GetConfig(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t * pwValue
    )
{
    phStatus_t statusTmp;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch (wIdentifier)
    {
    case PHDL_MSTAMPOSC_CONFIG_MAX_FIELD:
        *pwValue = (uint16_t) pDataParams->wMaxFieldStrength;
        break;
    case PHDL_MSTAMPOSC_CONFIG_MIN_FIELD:
        *pwValue = (uint16_t) pDataParams->wMinFieldStrength;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_0DB:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_6DB:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_12DB:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_18DB:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_24DB:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT:
        *pwValue = (uint16_t) pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT];
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_CUR_FIELD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_GetFieldStrength(pDataParams, pwValue,
            pDataParams->bCurentAntennaType));
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_CUR_ANTENNA:
        *pwValue = (uint16_t) pDataParams->bCurentAntennaType;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MODE:
        *pwValue = (uint16_t)pDataParams->bRxThresholdMode;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN_FS:
        *pwValue = pDataParams->wMinRxThresholdFS;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX_FS:
        *pwValue = pDataParams->wMaxRxThresholdFS;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN:
        *pwValue = pDataParams->wMinRxThreshold;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX:
        *pwValue = pDataParams->wMaxRxThreshold;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISO3_SetConfig(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wIdentifier,
    uint16_t wValue
    )
{
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    phStatus_t statusTmp;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    switch (wIdentifier)
    {
    case PHDL_MSTAMPOSC_CONFIG_MAX_FIELD:
        /* check value not more than 30 A/m allowed*/
        if (wValue > 30000)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMaxFieldStrength = wValue;
        pDataParams->wMaxFieldStrengthConfigured = wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_0DB:
        pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB] = wValue;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_6DB:
        pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB]  = wValue;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_12DB:
        pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB] = wValue;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_18DB:
        pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB] = wValue;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_24DB:
        pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB] = wValue;
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT:
        if (pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT] != wValue)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_UpdateStepAttenuators(pHalDataParams, pDataParams->wStepAttConfigs[wValue]));
            pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT] = wValue;
        }
        break;
    case PHDL_MSTAMPOSC_ISO3_CONFIG_CUR_ANTENNA:
        pDataParams->bCurentAntennaType = (uint8_t) wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MODE:
        /* check value */
        if (wValue > PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_DOWN)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bRxThresholdMode = (uint8_t)wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN_FS:
        pDataParams->wMinRxThresholdFS = wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX_FS:
        pDataParams->wMaxRxThresholdFS = wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MIN:
        /* check value */
        if (wValue < 0 || wValue > 65535)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMinRxThreshold = wValue;
        break;

    case PHDL_MSTAMPOSC_ISO3_CONFIG_RX_THRESHOLD_TARGET_MAX:
        /* check value */
        if (wValue < 0 || wValue > 65535)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->wMaxRxThreshold = wValue;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISO3_SetFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrengthMilliAPM
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint16_t fieldState;
    phStatus_t status;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t next;
    uint32_t steering;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t tmp;
    int32_t maxRetry;
    uint8_t attenuationShouldOnlyDecrease = 0; /* This is used to avoid condition where increasing attenuation understeers and decreasing oversteers.
                                               Meaning that there is no good attenuation seting that keeps DAC at full dynamic range.
                                               Understeering is always better choice than oversteering, so set field algorithm should favour understeering setting.
                                               */
    uint32_t dwMaxAgcReference = 0;
    uint32_t dwMaxAgcReferenceConfigured = 0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    status = PH_ERR_SUCCESS;
    next.amplify_deviation = 0;
    next.agc_reference = 0;
    next.rfu = 0;
    tmp.agc_reference = 0;
    tmp.amplify_deviation = 0;
    tmp.rfu = 0;

    /* check if field is switched on */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
        pHalDataParams,
        PHHAL_HW_ISO3_CONFIG_FIELD_STATE,
        &fieldState));

    if(fieldState == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

    /* check field strength limits */
    if (wFieldStrengthMilliAPM > pDataParams->wMaxFieldStrength)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(pDataParams));
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }

    /* Enable automatic gain control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pHalDataParams, PHHAL_HW_ISO3_CONFIG_DISABLE_AGC, PH_OFF));

    /* calculate value for next field strength */
    if(pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        next.agc_reference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA1 * (float32_t)wFieldStrengthMilliAPM / 1000.0f);
        dwMaxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA1 * (float32_t)pDataParams->wMaxFieldStrength / 1000.0f);
        dwMaxAgcReferenceConfigured = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA1 * (float32_t)pDataParams->wMaxFieldStrengthConfigured / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        next.agc_reference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA2 * (float32_t)wFieldStrengthMilliAPM / 1000.0f);
        dwMaxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA2 * (float32_t)pDataParams->wMaxFieldStrength / 1000.0f);
        dwMaxAgcReferenceConfigured = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA2 * (float32_t)pDataParams->wMaxFieldStrengthConfigured / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
    {
        next.agc_reference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA3 * (float32_t)wFieldStrengthMilliAPM / 1000.0f);
        dwMaxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA3 * (float32_t)pDataParams->wMaxFieldStrength / 1000.0f);
        dwMaxAgcReferenceConfigured = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA3 * (float32_t)pDataParams->wMaxFieldStrengthConfigured / 1000.0f);
    }
    else
    {
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_INVALID_ANTENNA_TYPE, PH_COMP_DL_MSTAMPOSC);
    }

    /* Sets new field strength */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
        (uint8_t*)(&next),
        sizeof(uint32_t)));

    /* Wait until field strength settles.*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t) 10)); /* If AGC is slowed down this value migth be to low */

    /* Control TX attenuation to utilize full dynamic range of DAC converter. */
    for (maxRetry = 6; maxRetry > 0; maxRetry--)
    {
        /* Check AGC gain value for steering */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_AvgAgcGain(pDataParams, &steering))

        if (status == PH_ERR_SUCCESS && steering >= (uint32_t)((float32_t)PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL * 0.95f)) /* AGC is oversteered, trying to decrease attenuation on TX path */
        {
            if (pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT] == PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB)
            {
                /* Can't decrease TX attenuation anymore.
                Field strength can't be increased.
                Automatic Gain Control is oversteered.
                Setting very low field strength in order not to burn the antenna.*/
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(pDataParams));
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
            }

            /* If remove dB attenuation will have a peak higher than Hmax then first set the fs to Hmax/2 */
            if (wFieldStrengthMilliAPM * 2 > pDataParams->wMaxFieldStrengthConfigured)
            {
                /* step 1: set fs to half of the max fs */
                tmp.agc_reference = dwMaxAgcReferenceConfigured / 2;
                tmp.amplify_deviation = next.amplify_deviation;
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                    pHalDataParams,
                    PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
                    (uint8_t*) &(tmp),
                    sizeof(uint32_t)));

                /* step 2: wait until field strenght is set */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t)10)); /* If AGC is slowed down this value migth be to low */
            }

            /* step 3: decrease attenuation by 6 dB (i.e. current field strength * 2) */
            attenuationShouldOnlyDecrease = 1;
            switch(pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT])
                {
                case PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB:
                    status = phdlMstAmpOsc_ISO3_SetConfig(
                        pDataParams,
                        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                        PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB);
                    break;
                case PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB:
                    status = phdlMstAmpOsc_ISO3_SetConfig(
                        pDataParams,
                        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                        PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB);
                    break;
                case PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB:
                    status = phdlMstAmpOsc_ISO3_SetConfig(
                        pDataParams,
                        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                        PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB);
                    break;
                case PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB:
                    status = phdlMstAmpOsc_ISO3_SetConfig(
                        pDataParams,
                        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                        PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB);
                    break;
                default:
                    status = PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
                }

            if (wFieldStrengthMilliAPM * 2 > pDataParams->wMaxFieldStrengthConfigured)
            {
                /* step 4: wait until step attenuators are set */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t)10)); /* If AGC is slowed down this value migth be to low */

                /* step 5: Restore field strenght */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                    pHalDataParams,
                    PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
                    (uint8_t*) &(next),
                    sizeof(uint32_t)));
            }
        }
        else if ((status == PH_ERR_SUCCESS && steering <= PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL / 2) && (attenuationShouldOnlyDecrease == 0)) /* AGC is understeered. Increasing attenuation on TX path */
        {
            /* step 1: double the current field strength to avoid card reset */
            tmp.agc_reference = 2 * next.agc_reference;
            /* Limit agc reference to max used fieldstrength */
            if (tmp.agc_reference > dwMaxAgcReference)
                tmp.agc_reference = dwMaxAgcReference;
            tmp.amplify_deviation = next.amplify_deviation;
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                pHalDataParams,
                PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
                (uint8_t*) &(tmp),
                sizeof(uint32_t)));

            /* step 2: wait until field strenght is set */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t) 10)); /* If AGC is slowed down this value migth be to low */

            /* step 3: increase attenuation by 6 dB (i.e. current field strength / 2) */
            switch(pDataParams->wStepAttConfigs[PHDL_MSTAMPOSC_ISO3_STEP_ATT_CURRENT])
            {
            case PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB:
                status = phdlMstAmpOsc_ISO3_SetConfig(
                    pDataParams,
                    PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                    PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB);
                break;
            case PHDL_MSTAMPOSC_ISO3_STEP_ATT_6DB:
                status = phdlMstAmpOsc_ISO3_SetConfig(
                    pDataParams,
                    PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                    PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB);
                break;
            case PHDL_MSTAMPOSC_ISO3_STEP_ATT_12DB:
                status = phdlMstAmpOsc_ISO3_SetConfig(
                    pDataParams,
                    PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                    PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB);
                break;
            case PHDL_MSTAMPOSC_ISO3_STEP_ATT_18DB:
                status = phdlMstAmpOsc_ISO3_SetConfig(
                    pDataParams,
                    PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
                    PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB);
                break;
            case PHDL_MSTAMPOSC_ISO3_STEP_ATT_24DB:
                /* Field strenght is very low. Attenuation can't be set to higher value. DAC is understeered and quantization noise can appear.*/
                status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
                maxRetry = 0; /* Finish the loop */
                break;
            default:
                status = PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
            }

            /* step 4: wait until step attenuators are set */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t) 10)); /* If AGC is slowed down this value migth be to low */

            /* step 5: Restore field strenght */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
                pHalDataParams,
                PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
                (uint8_t*) &(next),
                sizeof(uint32_t)));
        }
        else /* Field strength is set or there was an error*/
        {
            break;
        }

        /* Wait until field strength settles.*/
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(pHalDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t) 10)); /* If AGC is slowed down this value migth be to low */
    }

    /* Adjust the reader reception sensitivity */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_AdjustReaderRxThreshold(pDataParams, wFieldStrengthMilliAPM));

    return status;
}

phStatus_t phdlMstAmpOsc_ISO3_Cal(
  phdlMstAmpOsc_ISO3_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint16_t maxField;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* Enable automatic gain control */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
        pHalDataParams,
        PHHAL_HW_ISO3_CONFIG_DISABLE_AGC,
        PH_OFF));

    /* find the maximum value using linear search */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_FindMaxFieldStrength(pDataParams, pDataParams->wMaxFieldStrengthConfigured, (uint16_t*)&maxField));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_BinSearchMaxFieldStrength(pDataParams, (uint16_t*)&maxField));*/

    pDataParams->wMaxFieldStrength = maxField;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISO3 */
