/*
 * Copyright 2019, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 Setup Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_ISO3

#include <phdlMstAmpOsc.h>
#include <phhalHw_ISO3_Cmd.h>
#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_int.h"

#include "phdlMstAmpOsc_ISO3.h"
#include "phdlMstAmpOsc_ISO3_Int.h"

phStatus_t phdlMstAmpOsc_ISO3_Int_BinSearchMaxFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrengthLimitMilliAPM,
    uint16_t * pwMaxFieldStrengthMilliAPM
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint16_t previousDSA;
    uint16_t fieldState;
    uint32_t steering;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltageLow;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltageHigh;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltageMiddle;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltageZero;
    uint8_t bAgcStable;
    uint32_t maxAgcReference;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);
    PH_ASSERT_NULL(pwMaxFieldStrengthMilliAPM);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* check if field is switched on */
    phhalHw_GetConfig(
        pHalDataParams,
        PHHAL_HW_ISO3_CONFIG_FIELD_STATE,
        &fieldState);

    if(fieldState == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Calculate limit for fieldstrength check */
    if(pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA1 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA2 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA3 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else
    {
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_INVALID_ANTENNA_TYPE, PH_COMP_DL_MSTAMPOSC);
    }

    /* remember previous settings */
    phdlMstAmpOsc_ISO3_GetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT, &previousDSA);

    voltageZero.amplify_deviation = 0;
    voltageZero.agc_reference = 0;
    voltageZero.rfu = 0;

    /* Set target to zero before remove attenuator */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
        (uint8_t*)(&voltageZero),
        sizeof(uint32_t)));

    /* set DSA value to 0dB */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(
        pDataParams,
        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
        PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
        pHalDataParams,
        PHHAL_HW_TIME_MILLISECONDS,
        (uint16_t)10));

    /* binary search to find maximum gain below steering = 0x1ffff */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
        (uint8_t*)(&steering),
        sizeof(uint32_t)));

    voltageLow.amplify_deviation = 0;
    voltageLow.agc_reference = PHDL_MSTAMPOSC_ISO3_DAC_MIN_GAIN;
    voltageLow.rfu = 0;
    voltageHigh.amplify_deviation = 0;
    voltageHigh.agc_reference = maxAgcReference;
    voltageHigh.rfu = 0;
    voltageMiddle.amplify_deviation = 0;
    voltageMiddle.agc_reference = PHDL_MSTAMPOSC_ISO3_DAC_MIN_GAIN + 1;
    voltageMiddle.rfu = 0;

    bAgcStable = 0;

    while (bAgcStable == 0 || steering <= (PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL - 0xff))
    {
        /* additional stopping criteria. voltage does not change any more */
        if (voltageMiddle.agc_reference != (voltageLow.agc_reference + voltageHigh.agc_reference) / 2)
        {
            voltageMiddle.agc_reference = (voltageLow.agc_reference + voltageHigh.agc_reference) / 2;
        }
        else
        {
            break;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
            (uint8_t*)(&voltageMiddle),
            sizeof(uint32_t)));

        /* wait some time */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
            pHalDataParams,
            PHHAL_HW_TIME_MILLISECONDS,
            (uint16_t) 1));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
            (uint8_t*)(&steering),
            sizeof(uint32_t)));

        /* check if AGC is in a stable state */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_Int_AgcStabilityCheck(
            pDataParams,
            &bAgcStable));

        if (bAgcStable == 0 || steering == PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL)
        {
            /* still too high */
            voltageHigh.agc_reference = voltageMiddle.agc_reference;
        }
        else
        {
            /* too small */
            voltageLow.agc_reference = voltageMiddle.agc_reference;
        }
    }

    /* calculate value for next field strength */
    if(pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageMiddle.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA1 * 1000.0);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageMiddle.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA2 * 1000.0);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageMiddle.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA3 * 1000.0);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* restore previous settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(
        pDataParams,
        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
        (uint8_t) previousDSA));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phdlMstAmpOsc_ISO3_Int_FindMaxFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrengthLimitMilliAPM,
    uint16_t * pwMaxFieldStrengthMilliAPM
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint16_t previousDSA;
    uint16_t fieldState;
    uint32_t steering;
    uint32_t steeringPrevious;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltageNext;
    uint32_t stepSize;
    uint8_t bAgcStable;
    uint32_t maxAgcReference;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);
    PH_ASSERT_NULL(pwMaxFieldStrengthMilliAPM);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* check if field is switched on */
    phhalHw_GetConfig(
        pHalDataParams,
        PHHAL_HW_ISO3_CONFIG_FIELD_STATE,
        &fieldState);

    if(fieldState == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }

    /* Calculate limit for fieldstrength check */
    if(pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA1 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA2 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
    {
        maxAgcReference = (uint32_t)((float32_t)pDataParams->dwFieldStrengthRatioA3 * (float32_t)wFieldStrengthLimitMilliAPM / 1000.0f);
    }
    else
    {
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_INVALID_ANTENNA_TYPE, PH_COMP_DL_MSTAMPOSC);
    }

    /* remember previous settings */
    phdlMstAmpOsc_ISO3_GetConfig(pDataParams, PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT, &previousDSA);

    voltageNext.amplify_deviation = 0;
    voltageNext.agc_reference = 0;
    voltageNext.rfu = 0;

    /* Set target to zero before remove attenuator */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
        (uint8_t*)(&voltageNext),
        sizeof(uint32_t)));

    /* set DSA value to 0 damping */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(
        pDataParams,
        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
        PHDL_MSTAMPOSC_ISO3_STEP_ATT_0DB));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
        pHalDataParams,
        PHHAL_HW_TIME_MILLISECONDS,
        (uint16_t)10));

    /* linear search to find maximum gain below steering = 0x1ffff */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
        (uint8_t*)(&steering),
        sizeof(uint32_t)));

    stepSize = 0x1000;

    while (steering <= PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL - (2 * stepSize))
    {
        steeringPrevious = steering;
        voltageNext.agc_reference += stepSize;
        if (voltageNext.agc_reference > maxAgcReference)
        {
            voltageNext.agc_reference = maxAgcReference;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
            (uint8_t*)(&voltageNext),
            sizeof(uint32_t)));

        /* maybe wait some time? */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Wait(
            pHalDataParams,
            PHHAL_HW_TIME_MILLISECONDS,
            (uint16_t) 1));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
            (uint8_t*)(&steering),
            sizeof(uint32_t)));

        phdlMstAmpOsc_ISO3_Int_AgcStabilityCheck(pDataParams, &bAgcStable);
        if (bAgcStable == 0)
        {
            voltageNext.agc_reference -= stepSize * 2; /* agc is unstable */
            break;
        }

        /* Limit reached */
        if (voltageNext.agc_reference == maxAgcReference)
        {
            break;
        }

        /*
        //steering += 0x100;
        //if (steering < steeringPrevious) /* this can happen, if the CAL coil signal is no sine wave (due to clipping)
        //{
        //    /* decrease the current field strength and take that value as the maximum
        //    voltageNext.agc_reference -= stepSize * 2;
        //    break;
        //}*/

        /* adaptive linear increase of stepsize */
        if (steering < (PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL * 0.5))
        {
            stepSize = 0x8000;
            continue;
        }
        else if(steering < (PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL * 0.75))
        {
            stepSize = 0x2000;
            continue;
        }
        else if (steering < (PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL * 0.90))
        {
            stepSize = 0x1000;
            continue;
        }
        else
        {
            stepSize = 0x400;
        }
    }

    /* calculate value for next field strength */
    if(pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageNext.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA1 * 1000.0);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageNext.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA2 * 1000.0);
    }
    else if (pDataParams->bCurentAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
    {
        *pwMaxFieldStrengthMilliAPM = (uint16_t)((float32_t)voltageNext.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA3 * 1000.0);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* restore previous settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_ISO3_SetConfig(
        pDataParams,
        PHDL_MSTAMPOSC_ISO3_CONFIG_STEP_ATT_CURRENT,
        (uint8_t)previousDSA));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}


phStatus_t phdlMstAmpOsc_ISO3_Int_GetFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t * pdwFieldStrengthMilliAPM,
    uint8_t bAntennaType
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t voltage;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    /* check, if damping needs to be increased */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
        pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_CAL_COIL_VOLT_ADDR,
        (uint8_t*)(&voltage),
        sizeof(uint32_t)));

    /* calculate value for next field strength */
    if(bAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_1)
    {
        *pdwFieldStrengthMilliAPM = (uint16_t)((float32_t) voltage.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA1  * 1000.0f);
    }
  else if(bAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
  else if(bAntennaType == PHDL_MSTAMPOSC_ISO3_ANTENNA_TYPE_3)
  {
      *pdwFieldStrengthMilliAPM = (uint16_t)((float32_t) voltage.agc_reference / (float32_t)pDataParams->dwFieldStrengthRatioA3  * 1000.0f);
  }
  else
  {
    return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_INVALID_ANTENNA_TYPE, PH_COMP_DL_MSTAMPOSC);
  }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phdlMstAmpOsc_ISO3_Int_AgcStabilityCheck(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint8_t * pAgcIsStable
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    int16_t wValCount;
    int16_t wNumValues;
    uint16_t wRegVals[128];
    int32_t dwMean;
    int32_t dwVar;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;
    wNumValues = 128;

    memset(wRegVals, 0, sizeof(wRegVals));

    /* build a command queue that contains multiple read-register commands */
    for (wValCount = 0; wValCount < wNumValues; wValCount++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_BuildRegisterRead(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
            sizeof(uint32_t)));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Int_DirectExchange(
        pHalDataParams));

    dwMean = 0;
    for (wValCount = 0; wValCount < wNumValues; wValCount++)
    {
        wRegVals[wValCount] = *((uint32_t*)(pHalDataParams->pRxBuffer) + wValCount) & 0xffff;
        dwMean += wRegVals[wValCount];
    }
    dwMean /= (uint32_t)wNumValues;

    dwVar = 0;
    for (wValCount = 0; wValCount < wNumValues; wValCount++)
    {
        dwVar += (wRegVals[wValCount] - dwMean) * (wRegVals[wValCount] - dwMean);
    }
    dwVar /= (uint32_t)wNumValues;

    if (dwVar > 0x1000) /* or other threshold */
    {
        *pAgcIsStable = 0; /* not stable */
    }
    else
    {
        *pAgcIsStable = 1; /* AGC is stable */
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phdlMstAmpOsc_ISO3_Int_AvgAgcGain(phdlMstAmpOsc_ISO3_DataParams_t * pDataParams, uint32_t * pAvgAgcGain)
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalDataParams;
    uint32_t gain;
    uint8_t maxAvgSamples = 10;
    uint8_t counter = 0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;

    *pAvgAgcGain = 0;
    for (counter = 0; counter < maxAvgSamples; counter++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR,
            (uint8_t*)(&gain),
            sizeof(uint32_t)));

        *pAvgAgcGain = *pAvgAgcGain + gain;
    }
    *pAvgAgcGain = *pAvgAgcGain / maxAvgSamples;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_ISO3_Int_AdjustReaderRxThreshold(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    float64_t fGainExpected = 0.0;
    uint16_t wGainExpected = 0;
    phhalHw_ISO3_DataParams_t* pHalDataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalDataParams);

    pHalDataParams = (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams;


    /* Adjust the RxGain Target acording to the FS */
    if (pDataParams->bRxThresholdMode != PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_DISABLED)
    {
        switch (pDataParams->bRxThresholdMode)
        {
        case PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_CONSTANT:
            wGainExpected = pDataParams->wMinRxThreshold;
            break;

        case PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR:
        case PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT:
        case PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_UP:
        case PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_DOWN:
            /* Interpolation does not work if FS for min and max is the same */
            if (pDataParams->wMinRxThresholdFS == pDataParams->wMaxRxThresholdFS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
            }

            /* Linear interpolation */
            fGainExpected = (double)pDataParams->wMinRxThreshold +
                ((double)(wFieldStrength - pDataParams->wMinRxThresholdFS)*
                ((double)(pDataParams->wMaxRxThreshold - pDataParams->wMinRxThreshold)/(double)(pDataParams->wMaxRxThresholdFS - pDataParams->wMinRxThresholdFS)));

            /* Limit depending on mode */
            if ((pDataParams->bRxThresholdMode == PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT || pDataParams->bRxThresholdMode == PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_DOWN) &&
                wFieldStrength < pDataParams->wMinRxThresholdFS)
            {
                fGainExpected = pDataParams->wMinRxThreshold;
            }
            if ((pDataParams->bRxThresholdMode == PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT || pDataParams->bRxThresholdMode == PHDL_MSTAMPOSC_ISO3_RX_THRESHOLD_TARGET_MODE_LINEAR_LIMIT_UP) &&
                wFieldStrength > pDataParams->wMaxRxThresholdFS)
            {
                fGainExpected = pDataParams->wMaxRxThreshold;
            }

            /* Limit also for the max and min possible value */
            if (fGainExpected < 0.0)
            {
                fGainExpected = 0.0;
            }
            if (fGainExpected > 65535.0)
            {
                fGainExpected = 65535.0;
            }

            wGainExpected = (uint16_t)(0.5 + fGainExpected);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
        }

        if (wGainExpected < 0)
        {
            wGainExpected = 0;
        }

        if (wGainExpected > 65535)
        {
            wGainExpected = 65535;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pHalDataParams, PHHAL_HW_ISO3_CONFIG_RX_NOISE_POWER_TH, wGainExpected));
    }

    return statusTmp;
}

phStatus_t phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(phdlMstAmpOsc_ISO3_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    phdlMstAmpOsc_ISO3_GainCtrlReg_t zero;

    zero.amplify_deviation = 0;
    zero.agc_reference = 0;
    zero.rfu = 0;

    /* Setting very low field strength in order not to burn the antenna. */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        (phhalHw_ISO3_DataParams_t*)pDataParams->pHalDataParams,
        PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR,
        (uint8_t*)(&zero),
        sizeof(uint32_t)));

    return statusTmp;
}

#endif /* NXPBUILD__PHDL_MSTAMPOSC_ISO3 */
