/*
 * Copyright 2017, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Oscilloscope Agilent DSO7052A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A

#include <phdlOsci.h>
#include "phdlOsci_DSO7052A.h"
#include "phdlOsci_DSO7052A_Int.h"
#include "phdlOsci_DSO7052A_Cmd.h"
#include "../phdlOsci_Int.h"

#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#pragma warning(disable:4200)           /* PRQA S 3116 */
#pragma warning(disable:4201)           /* PRQA S 3116 */
#include <stdio.h>                      /* PRQA S 5124 */
#pragma warning(pop)                    /* PRQA S 3116 */

phStatus_t phdlOsci_DSO7052A_Init(
                                phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                uint16_t wSizeOfDataParams,
                                phbalReg_Visa_DataParams_t * pBalRegDataParams
                                )
{
    if (sizeof(phdlOsci_DSO7052A_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                 = PH_COMP_DL_OSCI | PHDL_OSCI_DSO7052A_ID;
    pDataParams->pBalRegDataParams   = pBalRegDataParams;
    /* default value for cetecom antenna eg. RMS = FieldStrength * 0.318(318/1000) */
    pDataParams->wFieldStrengthMultiplier = 318;

    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;

    pDataParams->wAverageFact = 5;
    pDataParams->wRangeMin = 60;
    pDataParams->wRangeMax = 72;
    pDataParams->bMeasurementOptions = PHDL_OSCI_MEASUREMENT_ADJUST_TIME | PHDL_OSCI_MEASUREMENT_ADJUST_RANGE;

    pDataParams->bSelectedChannel = PHDL_OSCI_CHANNEL_1;

    pDataParams->bAcOrDcMeasurementCh1 = PHDL_OSCI_DSO7052A_AC;
    pDataParams->bAcOrDcMeasurementCh2 = PHDL_OSCI_DSO7052A_AC;
    pDataParams->bNumberOfChannels = 4;

/* FIXXME CH3 + 4??? */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_SetConfig64(
                                     phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t qwValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t abBuffer[50];
    int64_t qwCurrentConfigValue = 0;

    switch (wIdentifier)
    {
        /*FIXXME zusammenfassen */
        /*
        bit
        1 On Off
        1 Coupling AC/DC
        1 Impd 1M50
        1 BW Limit
        1 Fine
        1 Invert
        Probe
        2    Units Volt/Amps/Watts
        1    Probe
            Skew
        ~16 Adjust (offset)
        ~8 Scale
        */
    case PHDL_OSCI_CONFIG_TRIGGER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, (uint16_t)qwValue));
        pDataParams->wCurrentTriggerMode = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConfigOsci(pDataParams, (uint8_t)qwValue));
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        pDataParams->wFieldStrengthMultiplier = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wAverageFact = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        if (qwValue < 5 ||
            qwValue > 70 ||
            qwValue >= pDataParams->wRangeMax)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMin = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        if (qwValue < 10 ||
            qwValue > 75 ||
            qwValue <= pDataParams->wRangeMin)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMax = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        if ((qwValue & ~PHDL_OSCI_MEASUREMENT_MASK) != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bMeasurementOptions = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_DSO7052A_CONFIG_SAVE_CSV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SaveImage(pDataParams, (uint16_t)qwValue));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        if (qwValue < PHDL_OSCI_CHANNEL_1 ||
            qwValue > PHDL_OSCI_CHANNEL_4 ||
            qwValue - PHDL_OSCI_CHANNEL_1 >= pDataParams->bNumberOfChannels)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bSelectedChannel = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"BWLimit", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        if (qwValue != PHDL_OSCI_COUPLING_AC && qwValue != PHDL_OSCI_COUPLING_DC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"COUPling", (uint8_t *)((qwValue == PHDL_OSCI_COUPLING_AC) ? "AC" : "DC")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"DISPlay", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        if (qwValue != PHDL_OSCI_IMPEDANCE_FIFTY && qwValue != PHDL_OSCI_IMPEDANCE_ONE_MEGA)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"IMPedance", (uint8_t *)((qwValue == PHDL_OSCI_IMPEDANCE_FIFTY) ? "FIFTy" : "ONEMeg")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"INVert", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lld", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"PROBe", abBuffer));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
        if (qwValue != PHDL_OSCI_UNIT_AMPERE && qwValue != PHDL_OSCI_UNIT_VOLT)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"UNITs", (uint8_t *)((qwValue == PHDL_OSCI_UNIT_AMPERE) ? "AMPere" : "VOLT")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VERNier", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        if (qwValue != PHDL_OSCI_TRIGGER_TYPE_EDGE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams,
            (uint8_t *)"MODE", (uint8_t *)"EDGE"));
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        if (qwValue > PHDL_OSCI_CHANNEL_DIGITAL15)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        switch(qwValue)
        {
        case PHDL_OSCI_CHANNEL_1: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"CHANnel1")); break;
        case PHDL_OSCI_CHANNEL_2: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"CHANnel2")); break;
        case PHDL_OSCI_CHANNEL_3: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"CHANnel3")); break;
        case PHDL_OSCI_CHANNEL_4: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"CHANnel4")); break;
        case PHDL_OSCI_CHANNEL_EXTERNAL: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"EXTernal")); break;
        case PHDL_OSCI_CHANNEL_LINE: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"LINE")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL0: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital0")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL1: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital1")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL2: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital2")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL3: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital3")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL4: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital4")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL5: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital5")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL6: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital6")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL7: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital7")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL8: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital8")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL9: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital9")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL10: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital10")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL11: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital11")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL12: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital12")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL13: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital13")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL14: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital14")); break;
        case PHDL_OSCI_CHANNEL_DIGITAL15: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SOURce", (uint8_t *)"DIGital15")); break;
        default:
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
            }
        }
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        if (qwValue != PHDL_OSCI_EDGE_SLOPE_POSITIVE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_NEGATIVE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_ALTERNATE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_EITHER)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        switch(qwValue)
        {
        case PHDL_OSCI_EDGE_SLOPE_POSITIVE: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SLOPe", (uint8_t *)"POSitive")); break;
        case PHDL_OSCI_EDGE_SLOPE_NEGATIVE: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SLOPe", (uint8_t *)"NEGative")); break;
        case PHDL_OSCI_EDGE_SLOPE_ALTERNATE: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SLOPe", (uint8_t *)"ALTernate")); break;
        case PHDL_OSCI_EDGE_SLOPE_EITHER: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams, (uint8_t *)"SLOPe", (uint8_t *)"EITHer")); break;
        default:
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
            }
        }
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        if (qwValue != PHDL_OSCI_TIMEBASE_MODE_MAIN &&
            qwValue != PHDL_OSCI_TIMEBASE_MODE_ROLL &&
            qwValue != PHDL_OSCI_TIMEBASE_MODE_WINDOW &&
            qwValue != PHDL_OSCI_TIMEBASE_MODE_XY)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        switch(qwValue)
        {
        case PHDL_OSCI_TIMEBASE_MODE_MAIN: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams, (uint8_t *)"MODE", (uint8_t *)"MAIN")); break;
        case PHDL_OSCI_TIMEBASE_MODE_ROLL: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams, (uint8_t *)"MODE", (uint8_t *)"ROLL")); break;
        case PHDL_OSCI_TIMEBASE_MODE_WINDOW: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams, (uint8_t *)"MODE", (uint8_t *)"WINDow")); break;
        case PHDL_OSCI_TIMEBASE_MODE_XY: PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams, (uint8_t *)"MODE", (uint8_t *)"XY")); break;
        default:
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
            }
        }
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams,
            (uint8_t *)"VERNier", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldmV", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"OFFSet", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldmV", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"SCALe", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldmV", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"RANGE", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldmV", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerConfig(pDataParams,
            (uint8_t *)"LEVel", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldnS", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams,
            (uint8_t *)"POSition", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldnS", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams,
            (uint8_t *)"SCALe", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldnS", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTimebaseConfig(pDataParams,
            (uint8_t *)"RANGe", abBuffer));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_GetConfig64(
                                     phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t * pqwValue
                                     )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint8_t abBuffer[50];
    int32_t qwTmpValue = 0;
    uint16_t wTmpValue = 0;
    uint16_t wRxlen;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
        *pqwValue = pDataParams->wCurrentTriggerMode;
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        *pqwValue = pDataParams->wFieldStrengthMultiplier;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        *pqwValue = pDataParams->wAverageFact;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        *pqwValue = pDataParams->wRangeMin;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        *pqwValue = pDataParams->wRangeMax;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        *pqwValue = pDataParams->bMeasurementOptions;
        break;

    case PHDL_OSCI_DSO7052A_CONFIG_VOLT_AVERAGE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Cmd_GetAverage(pDataParams, 2, &wTmpValue));
        *pqwValue = wTmpValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        *pqwValue = pDataParams->bSelectedChannel;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"BWLimit", abBuffer, 50));
        if (sscanf((char *)abBuffer, "%d", &qwTmpValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        if (qwTmpValue == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (qwTmpValue == 1)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"COUPling", abBuffer, 50));
        if (strncmp((char *)abBuffer, "AC", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_COUPLING_AC;
        }
        else if (strncmp((char *)abBuffer, "DC", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_COUPLING_DC;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"DISPlay", abBuffer, 50));
        if (sscanf((char *)abBuffer, "%d", &qwTmpValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        if (qwTmpValue == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (qwTmpValue == 1)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"IMPedance", abBuffer, 50));
        if (strncmp((char *)abBuffer, "ONEM", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_IMPEDANCE_ONE_MEGA;
        }
        else if (strncmp((char *)abBuffer, "FIFT", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_IMPEDANCE_FIFTY;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"INVert", abBuffer, 50));
        if (sscanf((char *)abBuffer, "%d", &qwTmpValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        if (qwTmpValue == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (qwTmpValue == 1)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"PROBe", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1, pqwValue));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"UNITs", abBuffer, 50));
        if (strncmp((char *)abBuffer, "VOLT", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_UNIT_VOLT;
        }
        else if (strncmp((char *)abBuffer, "AMP", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_UNIT_AMPERE;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VERNier", abBuffer, 50));
        if (sscanf((char *)abBuffer, "%d", &qwTmpValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        if (qwTmpValue == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (qwTmpValue == 1)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTriggerConfig(pDataParams,
            (uint8_t *)"MODE", abBuffer, 50));
        if (strncmp((char *)abBuffer, "EDGE", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_EDGE;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTriggerConfig(pDataParams,
            (uint8_t *)"SOURce", abBuffer, 50));
        if (strncmp((char *)abBuffer, "CHAN1", 5) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_1;
        }
        else if (strncmp((char *)abBuffer, "CHAN2", 5) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_2;
        }
        else if (strncmp((char *)abBuffer, "CHAN3", 5) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_3;
        }
        else if (strncmp((char *)abBuffer, "CHAN4", 5) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_4;
        }
        else if (strncmp((char *)abBuffer, "EXT", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_EXTERNAL;
        }
        else if (strncmp((char *)abBuffer, "LINE", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_CHANNEL_LINE;
        }
        else if (strncmp((char *)abBuffer, "DIG", 3) == 0)
        {
            if (sscanf((char *)abBuffer, "DIG%d", &qwTmpValue) != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
            }
            *pqwValue = PHDL_OSCI_CHANNEL_DIGITAL0 + (uint16_t)qwTmpValue;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTriggerConfig(pDataParams,
            (uint8_t *)"SLOPe", abBuffer, 50));
        if (strncmp((char *)abBuffer, "NEG", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_NEGATIVE;
        }
        else if (strncmp((char *)abBuffer, "POS", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_POSITIVE;
        }
        else if (strncmp((char *)abBuffer, "EITH", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_EITHER;
        }
        else if (strncmp((char *)abBuffer, "ALT", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_ALTERNATE;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTimebaseConfig(pDataParams,
            (uint8_t *)"MODE", abBuffer, 50));
        if (strncmp((char *)abBuffer, "MAIN", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_TIMEBASE_MODE_MAIN;
        }
        else if (strncmp((char *)abBuffer, "WIND", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_TIMEBASE_MODE_WINDOW;
        }
        else if (strncmp((char *)abBuffer, "XY", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_TIMEBASE_MODE_XY;
        }
        else if (strncmp((char *)abBuffer, "ROLL", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_TIMEBASE_MODE_ROLL;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTimebaseConfig(pDataParams,
            (uint8_t *)"VERNier", abBuffer, 50));
        if (sscanf((char *)abBuffer, "%d", &qwTmpValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        if (qwTmpValue == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (qwTmpValue == 1)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"OFFSet", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"SCALe", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"RANGE", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTriggerConfig(pDataParams,
            (uint8_t *)"LEVel", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTimebaseConfig(pDataParams,
            (uint8_t *)"POSition", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000000000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTimebaseConfig(pDataParams,
            (uint8_t *)"SCALe", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000000000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetTimebaseConfig(pDataParams,
            (uint8_t *)"RANGe", abBuffer, 50));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1000000000, pqwValue));
        break;

    case PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE:
        dwPrintLength = sprintf((char *)abCmd, ":ACQuire:SRAT?\n");
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , sizeof(abBuffer), abBuffer, &wRxlen));
        PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertStringToInt64(
            abBuffer, 1, pqwValue));
        break;

    case PHDL_OSCI_CONFIG_NUMBER_CHANNELS:
        *pqwValue = pDataParams->bNumberOfChannels;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }
/* FIXXME other parameter? */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_SetConfig(
                                     phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t wValue
                                     )
{
    return phdlOsci_DSO7052A_SetConfig64(pDataParams, wIdentifier, wValue);
}

phStatus_t phdlOsci_DSO7052A_GetConfig(
                                     phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t * pwValue
                                     )
{
    phStatus_t statusTmp = 0;
    int64_t qwTempValue = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetConfig64(pDataParams, wIdentifier, &qwTempValue));
    if (qwTempValue < 0 || qwTempValue > 65535)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *pwValue = (uint16_t)qwTempValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_GetFieldStrength(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint16_t * pwFieldStrength
    )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }

        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
        {
            /* get in correct range */
            statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }

        /* Only perform command if previous was success */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* get the VRMS of channel */
            statusTmp = phdlOsci_DSO7052A_Int_GetRMS(pDataParams , bChannel, &fTemp);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    fTemp = (float32_t)((fTemp * (float32_t) PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR * (float32_t) PHDL_OSCI_FIELD_STRENGTH_DIVISOR )/ (float32_t) pDataParams->wFieldStrengthMultiplier);
    *pwFieldStrength = (uint16_t)fTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_GetAmpl(
                                   phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwAmplitude
                                   )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }

        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
        {
            /* get in correct range */
            statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }

        /* Only perform command if previous was success */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* get the VRMS of channel */
            statusTmp = phdlOsci_DSO7052A_Int_GetAmpl(pDataParams , bChannel, &fTemp);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pwAmplitude = (uint16_t) (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_GetPeakPeak(
                                   phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfPkPk
                                   )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    float64_t fTemp = 0.0;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }

        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
        {
            /* get in correct range */
            statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }

        /* Only perform command if previous was success */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* get the VRMS of channel */
            statusTmp = phdlOsci_DSO7052A_Int_GetPeakPeak(pDataParams , bChannel, &fTemp);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pdfPkPk = (float64_t) (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_GetRMS(
                                   phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwRMS
                                   )
{
    phStatus_t statusTmp;
    float64_t dfValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_GetRMSDouble(pDataParams, bChannel, &dfValue));

    *pwRMS = (uint16_t)(dfValue + 0.5);

    return statusTmp;
}

phStatus_t phdlOsci_DSO7052A_GetRMSDouble(
                                   phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfRMS
                                   )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }

        statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
        {
            /* get in correct range */
            statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }

        /* Only perform command if previous was success */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* get the VRMS of channel */
            statusTmp = phdlOsci_DSO7052A_Int_GetRMS(pDataParams , bChannel, &fTemp);
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
            {
                continue;
            }
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pdfRMS = (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_SetCorrRange(
                                        phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                        uint8_t  bChannel
                                        )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS, statusRestore;
    uint8_t bFSAdjustCnt = 0;
    uint16_t wOrigMinRange, wOrigMaxRange;
    uint16_t wTriggerMode;

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    wOrigMinRange = pDataParams->wRangeMin;
    wOrigMaxRange = pDataParams->wRangeMax;
    wTriggerMode = pDataParams->wCurrentTriggerMode;
    /* check if mode is not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        /* set mode to auto to be able to get in correct range in any case */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_AUTO));
    }

    /* We try to adjust the osci range 5 times if we have some measurment problems */
    for (bFSAdjustCnt = 0; bFSAdjustCnt < 5; bFSAdjustCnt++)
    {
        /* If we have at least two retries that fails -> we adjust the scale on the scope */
        if (bFSAdjustCnt > 1)
        {
            pDataParams->wRangeMax = (uint16_t)max(25, (int32_t)wOrigMaxRange - ((int32_t)bFSAdjustCnt-1) * 10);
            pDataParams->wRangeMin = (uint16_t)max(20, (int32_t)wOrigMinRange - ((int32_t)bFSAdjustCnt-1) * 10);
        }
        /* get in correct range */
        statusTmp = phdlOsci_DSO7052A_Int_SetCorrRange(pDataParams, bChannel);
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
        {
            continue;
        }
        break;
    }

    /* Restore scaling */
    pDataParams->wRangeMin = wOrigMinRange;
    pDataParams->wRangeMax = wOrigMaxRange;
    /* Restore triggermode if it was not set to auto */
    if (wTriggerMode != PHDL_OSCI_TRIGGER_AUTO)
    {
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, wTriggerMode));
    }

    /* return the status */
    return statusTmp;
}

phStatus_t phdlOsci_DSO7052A_GetWaveForm(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       uint32_t dwWaveFormBufferSize,
                                       uint32_t * pdwWaveFormLength,
                                       uint8_t * pWaveFormBuffer,
                                       uint8_t * pbHeaderOffset
                                       )
{
    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    *pbHeaderOffset = 11;

    /* get waveform */
    return phdlOsci_DSO7052A_Int_GetWaveForm(pDataParams, bChannel, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
}

phStatus_t phdlOsci_DSO7052A_InitOsci(
                                    phdlOsci_DSO7052A_DataParams_t * pDataParams
                                    )
{
    phStatus_t statusTmp;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_IdentifyOscilloscope(pDataParams));

    /* clear all status bits */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"*CLS\n", 5, 200));

    /* check if clear worked */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* configure global */
    return phdlOsci_DSO7052A_Int_ConfigGlobal(pDataParams);
}

#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */
