/*
 * Copyright 2017 - 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Oscilloscope Agilent DSO7052A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A

#include "phdlOsci_DSO7052A.h"
#include "phdlOsci_DSO7052A_Int.h"
#include "../phdlOsci_Int.h"
#include <stdio.h>              /* PRQA S 5124 */
#include <math.h>

/* for visa functionality displaced function exchange large */
#include "../../../phbalReg/src/Visa/external/visa.h"
#include "../../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"

phStatus_t phdlOsci_DSO7052A_Int_ChkLastCmd(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[50];
    uint16_t wRxlen;
    int16_t wCmdCode;

    /* read command errror register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)":SYSTem:ERRor?\n", 15, 50, (uint8_t *)bRx, &wRxlen));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi16((int8_t*)bRx, &wCmdCode));

    /* analyse response */

    if (wCmdCode == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }
    else if ((wCmdCode <= -400) && (wCmdCode >= -440))
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_UNKNOWN_COMMAND, PH_COMP_DL_OSCI);
    }
    else if (wCmdCode == -222)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }
    else if ((wCmdCode <= -220) && (wCmdCode >= -223))
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_UNKNOWN_COMMAND, PH_COMP_DL_OSCI);
    }
    else if ((wCmdCode <= -100) && (wCmdCode >= -183))
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_STRING_ERROR, PH_COMP_DL_OSCI);
    }
    else if (wCmdCode == PHDL_OSCI_ERR_READ_WAVEFORM_BUFFER_TOO_SMALL)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_READ_WAVEFORM_BUFFER_TOO_SMALL, PH_COMP_DL_OSCI);
    }
    else if (wCmdCode == -256)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STOR_FILE_NOT_FOUND, PH_COMP_DL_OSCI);
    }
    else if (wCmdCode == -257)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_FILENAME_ILLEGAL, PH_COMP_DL_OSCI);
    }
    else if (wCmdCode == -250)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_NO_MASS_STORAGE_PRESENT, PH_COMP_DL_OSCI);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_DSO7052A_Int_WaitExe(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams
                                       )
{
    uint8_t bRx[10];
    uint16_t wRxlen;

    /* read operation complete, osci only responds if the last operation is completed */
    return phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)"*OPC?\n", 6, 10, (uint8_t *)bRx, &wRxlen);
}

phStatus_t phdlOsci_DSO7052A_Int_WaitArmed(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;
    int16_t wStatus;

    do
    {
        /* read if the trigger is armed */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)"AER?\n", 5, 10, (uint8_t *)bRx, &wRxlen));
        /* convert response */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi16((int8_t*)bRx, &wStatus));
    } while(wStatus == 0);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ClearArmed(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen;
    /* read if the trigger is armed */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)"AER?\n", 5, 10, (uint8_t *)bRx, &wRxlen));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ChkCmd(
                                      phdlOsci_DSO7052A_DataParams_t * pDataParams
                                      )
{
    phStatus_t statusTmp;

    /* wait for command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_WaitExe(pDataParams));

    /* check if last command was interpreted correct */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

/* FilenameParameters: LSB TypeA 0, TypeB 1; 2-3rd Bit bitrate: 0->106, 1->212, 2->424, 3->848 */
phStatus_t phdlOsci_DSO7052A_Int_SaveImage(
         phdlOsci_DSO7052A_DataParams_t * pDataParams,
         uint16_t wFilenameParameters
         )
{
    phStatus_t statusTmp;
    uint16_t wFieldStrength;
    uint8_t bBitrate;

    uint8_t abCmdForm[20] = ":SAVE:WAV:FORM CSV\n";
    uint8_t abCmdPath[20] = ":SAVE:PWD \"/usb0/\"\n";
    uint8_t abCmdSave[30] = ":SAVE:WAV \"106_A_0000mA.csv\"\n";

    if (wFilenameParameters & (0x01))
    {
        abCmdSave[15] = 'B';
    }

    bBitrate = 0x00;
    bBitrate = (wFilenameParameters >> 1) & 0x03;

    switch (bBitrate)
    {
        case 0x00:
            abCmdSave[11] = '1';
            abCmdSave[12] = '0';
            abCmdSave[13] = '6';
            break;
        case 0x01:
            abCmdSave[11] = '2';
            abCmdSave[12] = '1';
            abCmdSave[13] = '2';
            break;
        case 0x02:
            abCmdSave[11] = '4';
            abCmdSave[12] = '2';
            abCmdSave[13] = '4';
            break;
        case 0x03:
            abCmdSave[11] = '8';
            abCmdSave[12] = '4';
            abCmdSave[13] = '8';
            break;
        default:
            break;
    }

    wFieldStrength = 0x0000;
    wFieldStrength = (wFilenameParameters >> 3);

    abCmdSave[20] = (uint8_t)(wFieldStrength % 10) + '0';
    wFieldStrength /= 10;
    abCmdSave[19] = (uint8_t)(wFieldStrength % 10) + '0';
    wFieldStrength/= 10;
    abCmdSave[18] = (uint8_t)(wFieldStrength % 10) + '0';
    wFieldStrength /= 10;
    abCmdSave[17] = (uint8_t)(wFieldStrength % 10) + '0';

    /* Set binary format */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, abCmdForm, 20, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set path for saving (/usbo/) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, abCmdPath, 20, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Save image according to settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, abCmdSave, 29, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_SetCoupling(
     phdlOsci_DSO7052A_DataParams_t * pDataParams,
     uint8_t bChannel,
     uint8_t bACDC
    )
{
    phStatus_t statusTmp;
    uint8_t abCmd[16] = ":CHAN1:COUP AC\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[5] = (uint8_t)'0' + bChannel;
    }

    if (bACDC == PHDL_OSCI_DSO7052A_DC)
    {
        abCmd[12] = (uint8_t)'D';
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, abCmd, 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_IdentifyOscilloscope(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[200];
    uint16_t wRxlen;
    uint8_t bTemp;

    uint8_t abCmd[19] = "*IDN?\n";

    /*  */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 6 , 200, (uint8_t *)bRx, &wRxlen));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"DSO6054A", 8, &bTemp));
    if (bTemp)
    {
        pDataParams->bNumberOfChannels = 4;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"DSO7052A", 8, &bTemp));
    if (bTemp)
    {
        pDataParams->bNumberOfChannels = 2;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"DSO7054B", 8, &bTemp));
    if (bTemp)
    {
        pDataParams->bNumberOfChannels = 4;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_SetChannelConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t *pParam,
    uint8_t *pValue
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];

    dwPrintLength = sprintf((char *)abCmd, ":CHAN%d:%s %s\n", bChannel, pParam, pValue);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetChannelConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t *pParam,
    uint8_t *pValue,
    uint16_t wValueSize
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint16_t wRxlen;

    dwPrintLength = sprintf((char *)abCmd, ":CHAN%d:%s?\n", bChannel, pParam);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , wValueSize, (uint8_t *)pValue, &wRxlen));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_SetTriggerConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pParam,
    uint8_t *pValue
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];

    dwPrintLength = sprintf((char *)abCmd, ":TRIG:%s %s\n", pParam, pValue);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetTriggerConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pParam,
    uint8_t *pValue,
    uint16_t wValueSize
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint16_t wRxlen;

    dwPrintLength = sprintf((char *)abCmd, ":TRIG:%s?\n", pParam);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , wValueSize, (uint8_t *)pValue, &wRxlen));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_SetTimebaseConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pParam,
    uint8_t *pValue
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];

    dwPrintLength = sprintf((char *)abCmd, ":TIMebase:%s %s\n", pParam, pValue);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetTimebaseConfig(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pParam,
    uint8_t *pValue,
    uint16_t wValueSize
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint16_t wRxlen;

    dwPrintLength = sprintf((char *)abCmd, ":TIMebase:%s?\n", pParam);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , wValueSize, (uint8_t *)pValue, &wRxlen));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_IsScopeRunning(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t *pIsRunning
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint16_t wRxlen;
    uint8_t bRx[200];
    uint64_t qwTmpValue;

    dwPrintLength = sprintf((char *)abCmd, ":OPERegister:CONDition?\n");
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , sizeof(bRx), bRx, &wRxlen));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams));

    if (sscanf((char *)bRx, "%d", (int32_t *) &qwTmpValue) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
    if ((qwTmpValue & 0x0008) == 0x0008)
    {
        *pIsRunning = PH_ON;
    }
    else
    {
        *pIsRunning = PH_OFF;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ChannelOnOff(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bActive,
    uint8_t bChannel
    )
{
    phStatus_t statusTmp;

    uint8_t abCmd[19] = ":CHAN1:DISP 0\n";

    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

    abCmd[5] = (uint8_t)'0' + bChannel;

    if (bActive == PH_ON)
    {
        abCmd[12] = '1';
    }
    else
    {
        abCmd[12] = '0';
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 14));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetAmpl(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       float32_t * pfAmplitude
                                       )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[20];
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[19] = ":MEAS:VAMP? CHAN1\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[16] = (uint8_t)'0' + bChannel;
    }

    /* reset value for amplitude */
    *pfAmplitude = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 18 , 20, (uint8_t *)bRx, &wRxlen));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx), &fTemp2));
        fTemp += fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-6)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pfAmplitude = fTemp / pDataParams->wAverageFact;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetPeakPeak(
                                       phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       float64_t * pdfPkPk
                                       )
{
    phStatus_t statusTmp;
    float64_t fTemp, fTemp2;
    uint8_t bRx[20];
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[19] = ":MEAS:VPP? CHAN1\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[15] = (uint8_t)'0' + bChannel;
    }

    /* reset value for amplitude */
    *pdfPkPk = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 17 , 20, (uint8_t *)bRx, &wRxlen));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(((int8_t *)bRx), &fTemp2));
        fTemp += fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-6)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pdfPkPk = fTemp / pDataParams->wAverageFact;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetAverage(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t  bChannel,
    float32_t * pfAverage
    )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[20];
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[18] = ":MEAS:VAV? CHAN1\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[15] = (uint8_t)'0' + bChannel;
    }

    /* reset value for average */
    *pfAverage = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 17 , 20, (uint8_t *)bRx, &wRxlen));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx), &fTemp2));
        fTemp += fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-6)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }
    /* calculate average */
    *pfAverage = fTemp / (float32_t) pDataParams->wAverageFact;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetRMS(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t  bChannel,
    float32_t * pfRMS
    )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[20];
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[19] = ":MEAS:VRMS? CHAN1\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[16] = (uint8_t)'0' + bChannel;
    }

    /* reset value for amplitude */
    *pfRMS = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 18 , 20, (uint8_t *)bRx, &wRxlen));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx), &fTemp2));
        fTemp += fTemp2 * fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-12)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pfRMS = (float32_t)sqrt(fTemp / (float32_t) pDataParams->wAverageFact);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_OverUndershoot(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t  bChannel,
    uint8_t * pBIsOverUnderShoot
    )
{
    phStatus_t statusTmp;
    uint32_t dwDiv;
    float32_t fTemp;
    uint8_t bRx[40];
    uint16_t wRxlen;

    uint8_t abCmd[18] = ":MEAS:VPP? CHAN1\n";
    abCmd[15] = (uint8_t)'0' + bChannel;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetDiv(pDataParams, bChannel, &dwDiv));

    if (
        ((pDataParams->bAcOrDcMeasurementCh1 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 1)) ||
        ((pDataParams->bAcOrDcMeasurementCh2 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 2)) ||
        ((pDataParams->bAcOrDcMeasurementCh3 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 3)) ||
        ((pDataParams->bAcOrDcMeasurementCh4 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 4))
        )
    {
        phdlOsci_DSO7052A_Int_GetRMS(pDataParams, bChannel, &fTemp);
        fTemp = fTemp * 2000;
        if (fTemp > (dwDiv*7.5)) /* 7.5 to ensure that there is no possible overflow */
        {
            /* overshoot */
            *pBIsOverUnderShoot = 1;
        }
        else if (fTemp < dwDiv)
        {
            /* undershoot */
            *pBIsOverUnderShoot = 2;
        }
        else
        {
            /* everything ok */
            *pBIsOverUnderShoot = 0;
        }

    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 17 , 40, (uint8_t *)bRx, &wRxlen));

        /* convert response */
        statusTmp =  phdlOsci_Int_Atof((int8_t *)bRx, &fTemp);
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW)
        {
            *pBIsOverUnderShoot = 1;
        }
        else
        {
            PH_CHECK_SUCCESS(statusTmp);

            fTemp = fTemp * 1000;

            if (fTemp > (dwDiv*7.5))
            {
                /* overshoot */
                *pBIsOverUnderShoot = 1;
            }
            else if (fTemp < dwDiv)
            {
                /* undershoot */
                *pBIsOverUnderShoot = 2;
            }
            else
            {
                /* everything ok */
                *pBIsOverUnderShoot = 0;
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_CorrDiv(
                                       uint16_t wProbe,
                                       uint32_t  dwOldDiv,
                                       uint32_t * dwNewDiv
                                       )
{
    if (dwOldDiv < 20)
    {
        /* set to smallest DIV */
        *dwNewDiv = 20;
    }
    else if (dwOldDiv <= (uint32_t)(5000 * wProbe))
    {
        *dwNewDiv = dwOldDiv;
    }
    else
    {
        /* divisor beyond 5 volts per division not available */
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ConvertDivInCmd(
    uint32_t  dwDiv,
    uint8_t  bChannel,
    uint8_t * pCmdLength,
    uint8_t * pCmd
    )
{

    /* prepare common section of command */
    pCmd[0] = ':';
    pCmd[1] = 'C';
    pCmd[2] = 'H';
    pCmd[3] = 'A';
    pCmd[4] = 'N';
    pCmd[5] = (uint8_t)'0' + bChannel;
    pCmd[6] = ':';
    pCmd[7] = 'S';
    pCmd[8] = 'C';
    pCmd[9] = 'A';
    pCmd[10] = 'L';
    pCmd[11] = ' ';

    if (dwDiv < 10) /* Div >= 0.001V and <0.010V */
    {
        /* build command */
        pCmd[12] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit*/
        pCmd[13] = 'm';
        pCmd[14] = 'V';
        pCmd[15] = '\n';
        *pCmdLength = 16;
    }
    else if (dwDiv < 100) /* Div >= 0.01V and <0.10V */
    {
        /* build command */
        pCmd[13] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[12] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit*/
        pCmd[14] = 'm';
        pCmd[15] = 'V';
        pCmd[16] = '\n';
        *pCmdLength = 17;
    }
    else if (dwDiv < 1000) /* Div >= 0.1V and <1.0V */
    {
        /* build command */
        pCmd[14] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[13] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[12] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit*/
        pCmd[15] = 'm';
        pCmd[16] = 'V';
        pCmd[17] = '\n';
        *pCmdLength = 18;
    }
    else if (dwDiv < 10000) /* Div >= 1.0V and <10.0V */
    {
        /* build command */
        dwDiv /=10;
        pCmd[15] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[14] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[13] = '.';
        pCmd[12] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        /* add unit */
        pCmd[16] = ' ';
        pCmd[17] = 'V';
        pCmd[18] = '\n';
        *pCmdLength = 19;
    }
    else if (dwDiv < 100000) /* Div >= 10.0V and <100.0V */
    {
        /* build command */
        dwDiv /=10;
        pCmd[16] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[15] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[14] = '.';
        pCmd[13] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[12] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        /* add unit */
        pCmd[17] = ' ';
        pCmd[18] = 'V';
        pCmd[19] = '\n';
        *pCmdLength = 20;
    }
    else /* divisor bigger than or exact 100 is not useable */
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_COULD_NOT_CONVERT_DIVISOR_INTO_CMD, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetDiv(
                                      phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                      uint8_t  bChannel,
                                      uint32_t * dwDiv
                                      )
{
    phStatus_t statusTmp;
    float32_t fTemp;
    uint8_t bRx[40];
    uint16_t wRxlen;
    uint8_t abCmd[14] = ":CHAN1:SCAL?\n";

    /* check channel number */
    if (bChannel != 1)
    {
        abCmd[5] = (uint8_t)'0' + bChannel;
    }

    /* read out divisions */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, 13 , 40, (uint8_t *)bRx, &wRxlen));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof((int8_t *)bRx, &fTemp));

    /* return divisions */
    fTemp *=1000;
    *dwDiv = (uint32_t) (fTemp + 0.5); /* Round */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetBestDiv(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint16_t wProbe,
    uint32_t dwOldDiv,
    float64_t dfPkPK,
    uint32_t * dwNewDiv
    )
{
    phStatus_t statusTmp;
    float32_t lower_limit, upper_limit;

    /* correct Divisor (to ensure non zero) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_CorrDiv(wProbe, dwOldDiv, &dwOldDiv));

    /* calc limits */
    lower_limit = (float32_t)pDataParams->wRangeMin / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)dwOldDiv;
    upper_limit = (float32_t)pDataParams->wRangeMax / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)dwOldDiv;

    /* check if in pkpk in range */
    if (dfPkPK >= lower_limit && dfPkPK <= upper_limit)
    {
        /* amplitude is ok, return old DIV */
        *dwNewDiv = dwOldDiv;
    }
    else
    {
        /* calculate optimum DIV */
        dwOldDiv = (uint32_t)(dfPkPK * 2 * (float32_t)PHDL_OSCI_RANGE_DIVISOR / (float32_t)(pDataParams->wRangeMax + pDataParams->wRangeMin));

        /* correct Divisor and copy it to new one */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_CorrDiv(wProbe, dwOldDiv, dwNewDiv));

        /* Recalulate limit and round up if still not in limit */
        upper_limit = (float32_t)pDataParams->wRangeMax / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)*dwNewDiv;
        if (dfPkPK > upper_limit)
        {
            dwOldDiv++;
            /* correct Divisor and copy it to new one */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_CorrDiv(wProbe, dwOldDiv, dwNewDiv));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_SetCorrRange(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t  bChannel
)
{
    phStatus_t statusTmp;
    uint8_t bOvershoot, bCounter = 0;
    uint32_t dwTempDiv;
    uint32_t dwNewDiv;
    float32_t fRMS;
    float64_t dfPkPK;
    uint8_t  bCmdLength;
    uint8_t	abCmd[20];
    uint16_t wProbe;
    uint8_t  bChannelTmp;

    bChannelTmp = pDataParams->bSelectedChannel;
    pDataParams->bSelectedChannel = bChannel;
    statusTmp = phdlOsci_DSO7052A_GetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_PROBE, &wProbe);
    pDataParams->bSelectedChannel = bChannelTmp;
    PH_CHECK_SUCCESS(statusTmp);

    /* correct osci range */

    /* get current DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetDiv(pDataParams, bChannel, &dwTempDiv));

    /* check status of channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_OverUndershoot(pDataParams, bChannel, &bOvershoot));

    /* handle undershoot */
    bCounter = 0;
    while (bOvershoot == 2 && dwTempDiv > 20)
    {
        dwTempDiv /= 5;
        if (dwTempDiv < 20) /* Minimum resulution on scope possible */
        {
            dwTempDiv = 20;
        }

        /* generate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertDivInCmd(dwTempDiv, bChannel, &bCmdLength, (uint8_t *)abCmd));

        /* send command and wait 200 ms */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 200));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        /* check for overshoot */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_OverUndershoot(pDataParams, bChannel, &bOvershoot));

        /* in case  of getting stuck exit with error */
        if (bCounter > 20)
        {
            return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
        }
        bCounter++;
    }

    /* handle overshoot */
    bCounter = 0;
    while (bOvershoot == 1 && dwTempDiv < (uint32_t)(5000 * wProbe))
    {
        /* if overshoot get current DIV */
        /* PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetDiv(pDataParams, bChannel, &dwTempDiv)); */
        /* double DIV to be fast */
        dwTempDiv *=5;
        if (dwTempDiv >(uint32_t)(5000 * wProbe)) /* Maxmum resulution on scope possible */
        {
            dwTempDiv = 5000 * wProbe;
        }

        /* generate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertDivInCmd(dwTempDiv, bChannel, &bCmdLength, (uint8_t *)abCmd));

        /* send command and wait 200 ms */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 200));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        /* check for overshoot */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_OverUndershoot(pDataParams, bChannel, &bOvershoot));

        /* in case  of getting stuck exit with error */
        if (bCounter >20)
        {
            return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
        }
        bCounter++;
    }

    /* now signal is in measurable range */

    /* get current pkpk */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetPeakPeak(pDataParams, bChannel, &dfPkPK));

    if (
        ((pDataParams->bAcOrDcMeasurementCh1 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 1)) ||
        ((pDataParams->bAcOrDcMeasurementCh2 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 2)) ||
        ((pDataParams->bAcOrDcMeasurementCh3 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 3)) ||
        ((pDataParams->bAcOrDcMeasurementCh4 == PHDL_OSCI_DSO7052A_DC)&&(bChannel == 4))
        )
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetRMS(pDataParams, bChannel, &fRMS));
        dfPkPK = fRMS*2;
    }
    dfPkPK = dfPkPK * (float32_t)PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR;

    /* get current DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetDiv(pDataParams, bChannel, &dwTempDiv));

    /* get the best divisor */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetBestDiv(pDataParams, wProbe, dwTempDiv, dfPkPK, &dwNewDiv));

    /* if the new divisor is different we have to act */
    if (dwNewDiv != dwTempDiv)
    {
        /* generate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertDivInCmd(dwNewDiv, bChannel, &bCmdLength, (uint8_t *)abCmd));

        /* send command and wait 200 ms */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 200));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        /* check for overshoot */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_OverUndershoot(pDataParams, bChannel, &bOvershoot));

        /* remove eventually still existent overshoot */
        bCounter = 0;

        if (bOvershoot == 1)
        {
            /* get current DIV */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_GetDiv(pDataParams, bChannel, &dwTempDiv));

            do
            {
                /* check for overshoot at the begining to ensure that the divisions */
                /* are increased by 20% after the overshoot is removed as well */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_OverUndershoot(pDataParams, bChannel, &bOvershoot));

                /* increase Div by 20% */
                dwTempDiv = (uint32_t)((float32_t)dwTempDiv * 1.2) ;

                /* generate command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConvertDivInCmd(dwTempDiv, bChannel, &bCmdLength, (uint8_t *)abCmd));

                /* send command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *) abCmd, bCmdLength, 200));

                /* check command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

                /* in case  of getting stuck exit with error */
                if (bCounter > 50)
                {
                    return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
                }
                bCounter++;
            } while (bOvershoot == 1);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_GetWaveForm(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t  bSource,
    uint32_t dwWaveFormBufferSize,
    uint32_t * pdwWaveFormLength,
    uint8_t * pWaveFormBuffer,
    uint8_t * pbHeaderOffset
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[100];
    uint16_t wRxlen;
    uint32_t dwMaxWavePoints = 0;
    int32_t dwPreamblePoints = 0;
    uint8_t bScopeIsRunning;

    /* visa cmd strings */
    /* Conffig commands */
    uint8_t abCmdStop[7] =                  ":STOP\n";
    uint8_t abCmdSetAcquireTypeNormal[16] = ":ACQ:TYPE NORM\n";
    uint8_t abCmdSetTimeModeMain[16] =      ":TIM:MODE MAIN\n";
    uint8_t abCmdSetWaveSource[17] =        ":WAV:SOUR CHAN1\n";
    uint8_t abCmdSetWavePointsModeMax[20] = ":WAV:POIN:MODE MAX\n";
    uint8_t abCmdSetWaveFormatWord[16] =    ":WAV:FORM WORD\n";
    uint8_t abCmdSetWaveByteMSBFirst[15] =  ":WAV:BYT MSBF\n";
    uint8_t abCmdSetWavePointsMax[15] =     ":WAV:POIN MAX\n";
    uint8_t abCmdRun[6] =                   ":RUN\n";

    /* Getter commands */
    uint8_t abCmdGetWavePointsMax[27] =     ":WAVeform:POINts? MAXimum\n";
    uint8_t abCmdGetWavePreamble[11] =      ":WAV:PRE?\n";
    uint8_t abCmdGetWaveData[12] =          ":WAV:DATA?\n";

    /* preamble conversion variables */
    uint16_t bStartIdxs[PHDLOSCI_DSO7052A_PREAMBLE_ITEM_COUNT];
    uint8_t bPreambleCounter = 0;
    uint16_t wRxBufIdx = 0;

    /* Data header variables */
    uint8_t bDataHeaderLength = 0;
    uint8_t bDataHeaderTmp = 0;
    uint32_t dwDataSizeBytes = 0;

    /* set channel number */
    abCmdSetWaveSource[14] = (uint8_t)('0' + bSource);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_IsScopeRunning(pDataParams, &bScopeIsRunning));

    if (bScopeIsRunning == PH_ON)
    {
        /* Stop the acquisition */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdStop, (uint16_t)sizeof(abCmdStop)-1, 200));
    }

    /* configure waveform setup */
    /* configure Acquiretype */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetAcquireTypeNormal, (uint16_t)sizeof(abCmdSetAcquireTypeNormal)-1, 200));

    /* set MAIN as Timebase Mode*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetTimeModeMain, (uint16_t)sizeof(abCmdSetTimeModeMain)-1, 200));

    /* set the source channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetWaveSource, (uint16_t)sizeof(abCmdSetWaveSource)-1, 200));

    /* set MAX as Points Mode*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetWavePointsModeMax, (uint16_t)sizeof(abCmdSetWavePointsModeMax)-1, 200));

    /* set WORD Format for Data*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetWaveFormatWord, (uint16_t)sizeof(abCmdSetWaveFormatWord)-1, 200));

    /* set MSB first */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetWaveByteMSBFirst, (uint16_t)sizeof(abCmdSetWaveByteMSBFirst)-1, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* set points to acquire to MAX */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdSetWavePointsMax, (uint16_t)sizeof(abCmdSetWavePointsMax)-1, 200));

    /* query to determine the maximum number of points that can be retrieved at the current settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmdGetWavePointsMax, (uint16_t)sizeof(abCmdGetWavePointsMax)-1, (uint16_t)sizeof(bRx), (uint8_t *) bRx, &wRxlen));
    if (wRxlen > 0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui32((int8_t *) bRx, (uint32_t *)&dwMaxWavePoints));
        if (dwMaxWavePoints < 1)
        {
            return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_DATA_ERROR, PH_COMP_DL_OSCI);
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_COULD_NOT_DETERMINE_MAX_POINTS, PH_COMP_DL_OSCI);
    }

    /* Check if data will fit into buffer */
    if (dwWaveFormBufferSize - PHDLOSCI_DSO7052A_PREAMBLE_LENGTH - PHDLOSCI_DSO7052A_MAX_DATA_HEADER_LENGTH - 1 /* '\n' */ < dwMaxWavePoints * 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
    }

    /* get data preamble */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmdGetWavePreamble, (uint16_t)sizeof(abCmdGetWavePreamble)-1, (uint16_t)sizeof(bRx), (uint8_t *) bRx, &wRxlen));
    /* Get all starting indexes in preample string */
    wRxBufIdx = 0;
    bPreambleCounter = 0;
    bStartIdxs[bPreambleCounter++] = 0; /* First Index is always 0! */
    do
    {
        if (bRx[wRxBufIdx++] == ',')
        {
            bStartIdxs[bPreambleCounter++] = wRxBufIdx;
        }
    } while (bPreambleCounter < PHDLOSCI_DSO7052A_PREAMBLE_ITEM_COUNT && wRxBufIdx < wRxlen);
    /* Check if all preample index was found */
    if (bPreambleCounter != PHDLOSCI_DSO7052A_PREAMBLE_ITEM_COUNT)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_READ_WAVEFORM_HEADER_PARSE_INVALID, PH_COMP_DL_OSCI);
    }
    /* Extract number of points */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32((int8_t *) &bRx[bStartIdxs[2]], (int32_t *) &dwPreamblePoints));

    /* Check that size in preample match max size */
    if (dwMaxWavePoints != (uint32_t)dwPreamblePoints)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_DL_OSCI);
    }

    /* initiate file transfer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_BalReg_VisaExchangeLarge(pDataParams, (uint8_t *)abCmdGetWaveData, (uint16_t)sizeof(abCmdGetWaveData)-1,
        dwWaveFormBufferSize - PHDLOSCI_DSO7052A_PREAMBLE_LENGTH, &pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH],
        pdwWaveFormLength));

    if (bScopeIsRunning == PH_ON)
    {
        /* reset into Run mode */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)abCmdRun, (uint16_t)sizeof(abCmdRun)-1, 200));
    }

    /* Parse the large exchange response */
    /* <binary block length bytes>,<binary data>
        For example, to transmit 1000 bytes of data, the syntax would be:
        #800001000<1000 bytes of data><NL>
        8 is the number of digits that follow
        00001000 is the number of bytes to be transmitted
        <1000 bytes of data> is the actual data
    */
    if (pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH] != '#')
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_READ_WAVEFORM_HEADER_PARSE_INVALID, PH_COMP_DL_OSCI);
    }
    if (pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 1] < '0' || pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 1] > '0' + PHDLOSCI_DSO7052A_MAX_DATA_HEADER_LENGTH - PHDLOSCI_DSO7052A_MIN_DATA_HEADER_LENGTH)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_READ_WAVEFORM_HEADER_PARSE_INVALID, PH_COMP_DL_OSCI);
    }
    bDataHeaderLength = pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 1] - '0';
    /* Backup data in result string */
    bDataHeaderTmp = pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 2 + bDataHeaderLength];
    /* Set a zero for parsing */
    pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 2 + bDataHeaderLength] = 0;
    /* Extract number of bytes */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui32((int8_t *) &pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 2], (uint32_t *) &dwDataSizeBytes));
    /* restore data */
    pWaveFormBuffer[PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 2 + bDataHeaderLength] = bDataHeaderTmp;
    if (dwMaxWavePoints * 2 != dwDataSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_DL_OSCI);
    }

    /* Convert Preamble */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi16((int8_t *) &bRx[bStartIdxs[0]], (int16_t *) &pWaveFormBuffer[0]));      /* int16_t wFormat */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi16((int8_t *) &bRx[bStartIdxs[1]], (int16_t *) &pWaveFormBuffer[2]));      /* int16_t wType */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32((int8_t *) &bRx[bStartIdxs[2]], (int32_t *) &pWaveFormBuffer[4]));      /* int32_t dwPoints */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32((int8_t *) &bRx[bStartIdxs[3]], (int32_t *) &pWaveFormBuffer[8]));      /* int32_t dwCount */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64((int8_t *) &bRx[bStartIdxs[4]], (float64_t *) &pWaveFormBuffer[12]));   /* float64_t qwXIncrement */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64((int8_t *) &bRx[bStartIdxs[5]], (float64_t *) &pWaveFormBuffer[20]));   /* float64_t qwXOrigin */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32((int8_t *) &bRx[bStartIdxs[6]], (int32_t *) &pWaveFormBuffer[28]));     /* int32_t dwXReference */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof((int8_t *) &bRx[bStartIdxs[7]], (float32_t *) &pWaveFormBuffer[32]));     /* float32_t dwYIncrement */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof((int8_t *) &bRx[bStartIdxs[8]], (float32_t *) &pWaveFormBuffer[36]));     /* float32_t dwYOrigin */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32((int8_t *) &bRx[bStartIdxs[9]], (int32_t *) &pWaveFormBuffer[40]));     /* int32_t dwYReference */

    /* Data offset is coded just after the Preamble */
    pWaveFormBuffer[44] = (uint8_t)(PHDLOSCI_DSO7052A_PREAMBLE_LENGTH + 2 + bDataHeaderLength);

    /* the header offset is always zero */
    *pbHeaderOffset = 0;
    /* The preample also belongs to the waveForm so add also this length */
    *pdwWaveFormLength = *pdwWaveFormLength + PHDLOSCI_DSO7052A_PREAMBLE_LENGTH;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PCD(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe POSitive\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -10.2us\n", 26, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 2us\n", 20, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PICC_106_L3(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -40us\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 20us\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PICC_106_L4(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -110us\n", 25, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 20us\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PICC_212_L4(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -60us\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 20us\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PICC_424_L4(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -10us\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 20us\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigFDT_PICC_848_L4(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Trigger mode: Edge */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n", 19, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: positive */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger sweep/mode: normal */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Edge Source Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN4\n", 22, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 1V\n", 18, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set volts per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN4:SCAL 1V\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition -30us\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 10us\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigLive(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    /* Live config */
    phStatus_t statusTmp;

    /* set trigger auto, running mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":RUN\n", 5, 200));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE AUTO\n", 15, 200));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:POSition 0\n", 21, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Level in volts */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:LEVel 10mV\n", 20, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Trigger Slope: negative */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SLOPe NEGative\n", 24, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* Set Time per division */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TIMebase:SCALe 2us\n", 20, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigGlobal(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    /* global config */
    phStatus_t statusTmp;

    /* RESET */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"*RST\n", 6, 200));

    /* set triggermode according to reset */
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* GENERAL CONFIG */
    /* set up communication parameters */
    /* CLEAR_STATUS_BITS */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"*CLS\n", 6, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* VERNIER CHANNEL 1 ON */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN1:VERN ON\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* VERNIER CHANNEL 2 ON */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN2:VERN ON\n", 15, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* VERNIER CHANNEL 2 ON */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":CHAN2:IMPedance FIFTy\n", 23, 200));

    /* check command */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* put Oszi to LIVE mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ConfigLive(pDataParams));

    /* check command */
    return phdlOsci_DSO7052A_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigEmd(
    phdlOsci_DSO7052A_DataParams_t * pDataParams
    )
{
    /* EMD config */
    phStatus_t statusTmp;

    uint8_t bRx[40];
    uint16_t wRxlen;
    uint8_t abCmd[18] = ":CHANnel2:PROBe?\n";
    float32_t fProbeType;

    uint8_t triggerChannel = 4;
    uint8_t measureChannel = 2;

    /* RESET */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"*RST\n", 6, 200));

    /* set triggermode according to reset */
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* GENERAL CONFIG */
    /* set up communication parameters */
    /* CLEAR_STATUS_BITS */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"*CLS\n", 6, 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    /* CHANNELS CONFIG */
    /* Enable Channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChannelOnOff(pDataParams, PH_ON, measureChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChannelOnOff(pDataParams, PH_ON, triggerChannel));

/*    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetCoupling(pDataParams, sequenceChannel, PHDL_OSCI_DSO7052A_DC)); */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetCoupling(pDataParams, measureChannel, PHDL_OSCI_DSO7052A_AC));

    /* TRIGGER CONFIG */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_NORMAL));

    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":CHAN1:IMPedance FIFTy\n")); /* Coupling CHANNEL 1 */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":CHAN2:IMPedance ONEMeg\n"));/* Coupling CHANNEL 2 */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":CHAN2:VERN ON\n"));/* VERNIER CHANNEL 2 ON */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":CHANnel4:SCALe 5V\n"));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:HFReject ON\n")); */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:HOLDoff 60ns\n"));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:NREJect ON\n")); */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:MODE EDGE\n"));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:EDGE:SOURce CHANnel4\n"));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:EDGE:LEVel 2500mV\n"));/* Trigger Level in volts */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:EDGE:REJect HFReject\n")); */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TRIGger:EDGE:SLOPe NEGative\n"));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TIMebase:POSition -0.001\n"));/* Set trigger delay */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendStringCheckStatus(pDataParams, (uint8_t *)":TIMebase:SCALe 200us\n"));/* Set Time per division */

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *) abCmd, 17 ,40,(uint8_t *) bRx, &wRxlen));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx), &fProbeType));
    if (fProbeType != 10.0)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_OSCI);
    }

    /* Check Data Channel for Probe */
    /* check command */
    return statusTmp;
}

phStatus_t phdlOsci_DSO7052A_Int_SetTriggerMode(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint16_t wTriggerMode
    )
{
    phStatus_t statusTmp;

    switch (wTriggerMode)
    {
    case PHDL_OSCI_TRIGGER_AUTO:
        /* Trigger Edge Source Channel */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":STOP\n", 6, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIGger:SOURce CHAN1\n", 22, 200));
        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":RUN\n", 5, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE AUTO\n", 15, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));*/
        phdlOsci_DSO7052A_Int_ConfigLive(pDataParams);

        break;

    case PHDL_OSCI_TRIGGER_NORMAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":STOP\n", 6, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ClearArmed(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":RUN\n", 5, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_WaitArmed(pDataParams));
        break;

    case PHDL_OSCI_TRIGGER_SINGLE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":STOP\n", 6, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ClearArmed(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":SINGle\n", 8, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_WaitArmed(pDataParams));
        break;

    case PHDL_OSCI_TRIGGER_STOP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":STOP\n", 6, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        break;

    /* Wenn single ausreicht, dann kick it!
    case PHDL_OSCI_TRIGGER_ISOWAVER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":TRIG:SWE NORM\n", 15, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)"\n", 8, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, (uint8_t *)":SINGle\n", 8, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));
        break;
    */

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    pDataParams->wCurrentTriggerMode = wTriggerMode;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_DSO7052A_Int_ConfigOsci(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t bConfig
    )
{
    switch (bConfig)
    {
    case PHDL_OSCI_MODE_FDT_PCD:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PCD(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_106_L3:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PICC_106_L3(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_106_L4:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PICC_106_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_212_L4:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PICC_212_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_424_L4:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PICC_424_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_848_L4:
        return phdlOsci_DSO7052A_Int_ConfigFDT_PICC_848_L4(pDataParams);

    case PHDL_OSCI_MODE_LIVE:
        return phdlOsci_DSO7052A_Int_ConfigLive(pDataParams);

    case PHDL_OSCI_MODE_GLOBAL:
        return phdlOsci_DSO7052A_Int_ConfigGlobal(pDataParams);

    case PHDL_OSCI_MODE_EMD:
        return phdlOsci_DSO7052A_Int_ConfigEmd(pDataParams);

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }
}
phStatus_t phdlOsci_DSO7052A_Int_ConvertStringToInt64(uint8_t *abBuffer, int64_t qwMultiplier, int64_t *pqwValue)
{
    float32_t fTmpValue;

    if (sscanf((char *)abBuffer, "%f", &fTmpValue) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
    if (fTmpValue >= 0.0)
    {
        *pqwValue = (int64_t)(0.5 + ((float32_t)qwMultiplier * fTmpValue));
    }
    else
    {
        *pqwValue = (int64_t)(-0.5 + ((float32_t)qwMultiplier * fTmpValue));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_DSO7052A_Int_SendStringCheckStatus(
                             phdlOsci_DSO7052A_DataParams_t * pDataParams,
                             uint8_t * pTxBuffer
                             )
{
    phStatus_t statusTmp;

    /* Send command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_SendWait(pDataParams, pTxBuffer, (uint16_t)strlen((const char *)pTxBuffer), 200));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_DSO7052A_Int_ChkCmd(pDataParams));

    return statusTmp;
}
phStatus_t phdlOsci_DSO7052A_Int_Send(
                             phdlOsci_DSO7052A_DataParams_t * pDataParams,
                             uint8_t * pTxBuffer,
                             uint16_t wTxLength
                             )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 0));/*TESTME, was 100, now 0 like comment */

    /* send data */
    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return status;
}

phStatus_t phdlOsci_DSO7052A_Int_SendWait(
                                 phdlOsci_DSO7052A_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint16_t wWaittime
                                 )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to waittime */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /* send data */
    status = phhalReg_Visa_Cmd_Transmit(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength);
/*    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        0,
        &bRx,
        &wRxLen);*/


    /* restore IO timeout use statusTmp to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        /* there is a errorcode on stack because no answer is expected */
        status = phdlOsci_DSO7052A_Int_ChkLastCmd(pDataParams);

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return status;
}

phStatus_t phdlOsci_DSO7052A_Int_BalReg_VisaExchangeLarge(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,
    uint8_t * pTxBuffer,
    uint16_t wTxLength,
    uint32_t dwRxBufSize,
    uint8_t * pRxBuffer,
    uint32_t * pRxLength
    )
{
#ifdef _WIN32
    ViStatus statusVi;
    uint32_t dwBytesWritten, dwBytesRead;

    /* write the data */
    statusVi = (pDataParams->pBalRegDataParams->pviWrite)(pDataParams->pBalRegDataParams->instr, (ViBuf)pTxBuffer, (ViUInt32)wTxLength, &dwBytesWritten);
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    if (dwBytesWritten != wTxLength)
    {
        /* write error not all data send */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    /* read the data */

    statusVi = (pDataParams->pBalRegDataParams->pviRead)(pDataParams->pBalRegDataParams->instr, (ViPBuf)pRxBuffer, (ViUInt32)dwRxBufSize, &dwBytesRead);

    /* Return received length */
    *pRxLength = dwBytesRead;

    /* Check for timeout error */
    if (dwBytesRead == 0 || statusVi == VI_ERROR_TMO)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    pTxBuffer = NULL;
    wTxLength = 0;
    wRxBufSize = 0;
    pRxBuffer = NULL;
    pRxLength = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */
