/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 FPGA soft scope.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phhalHw.h> /* Softscope uses ISO3 HAL as its BAL.*/
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE

#include <phdlOsci.h>
#include "phdlOsci_ISO3SoftScope.h"
#include "phdlOsci_ISO3SoftScope_Int.h"
#include "phdlOsci_ISO3SoftScope_Cmd.h"
#include "../phdlOsci_Int.h"

#include "../../../../intfs/phhalHw_ISO3_cmd.h"             /* Requires Hal ISO3 interface function */
#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_reg.h"     /* Register definitions for ISO3 Reader */


phStatus_t phdlOsci_ISO3SoftScope_Init(
                                       phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                       uint16_t wSizeOfDataParams,
                                       phhalHw_ISO3_DataParams_t * pHalISO3DataParams
                                       )
{
    if (sizeof(phdlOsci_ISO3SoftScope_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalISO3DataParams);

    /* Check HAL data parameters */
    if (PH_GET_COMPCODE(pHalISO3DataParams) != PH_COMP_HAL ||
        PH_GET_COMPID(pHalISO3DataParams) != PHHAL_HW_ISO3_ID)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
    }


    pDataParams->wId                    = PH_COMP_DL_OSCI | PHDL_OSCI_ISO3SOFTSCOPE_ID;
    pDataParams->pHalISO3DataParams     = pHalISO3DataParams;
    pDataParams->dwLogCtrlReg           = (uint32_t) 0;
    pDataParams->dwLogSizeReg           = (uint32_t) PHDL_OSCI_ISO3SOFTSCOPE_MAX_BUFFER_SIZE;
    pDataParams->dwTriggerCtrl0Reg      = (uint32_t) 0;
    pDataParams->dwTriggerCtrl1Reg      = (uint32_t) 0;
    pDataParams->bWaveFormHeaderEnabled = (uint8_t) PH_ON;
    pDataParams->dwWaveFormSampleSize     = (uint32_t) 0;

    /* Init Trigger settings */
    ISO3_REG_TRIGGER_CTRL_0.conditionT1Rise = PHHAL_HW_ISO3_TRIG_MODE_TX_SOC;
    ISO3_REG_TRIGGER_CTRL_0.conditionT1Fall = PHHAL_HW_ISO3_TRIG_MODE_TX_LAST_BIT;
    ISO3_REG_TRIGGER_CTRL_0.conditionT2Rise = PHHAL_HW_ISO3_TRIG_MODE_RX_SOC;
    ISO3_REG_TRIGGER_CTRL_0.conditionT2Fall = PHHAL_HW_ISO3_TRIG_MODE_RX_EOC;
    ISO3_REG_TRIGGER_CTRL_0.externEnableT1 = PH_ON;
    ISO3_REG_TRIGGER_CTRL_0.externEnableT2 = PH_ON;
    ISO3_REG_TRIGGER_CTRL_0.signalIdT1 = PHHAL_HW_ISO3_TRIG1_SIGNAL_ADC_INPUT;
    ISO3_REG_TRIGGER_CTRL_0.signalIdT2 = PHHAL_HW_ISO3_TRIG2_SIGNAL_ADC_INPUT;
    ISO3_REG_TRIGGER_CTRL_0.fdtEnable = PH_OFF;

    ISO3_REG_TRIGGER_CTRL_1.triggerRiseOnBitNumber = 3;
    ISO3_REG_TRIGGER_CTRL_1.triggerFallOnBitNumber = 3;

    /* Init soft scope settings */
    ISO3_REG_LOG_CTRL.trig0     = 0;
    ISO3_REG_LOG_CTRL.trig1     = 0;
    ISO3_REG_LOG_CTRL.trigForce = 0;
    ISO3_REG_LOG_CTRL.trigMls   = 0;
    ISO3_REG_LOG_CTRL.rfu1      = 0;
    ISO3_REG_LOG_CTRL.logRx     = 0;
    ISO3_REG_LOG_CTRL.logCal    = 0;
    ISO3_REG_LOG_CTRL.logCos    = 0;
    ISO3_REG_LOG_CTRL.logSin    = 0;
    ISO3_REG_LOG_CTRL.logAgcO   = 0;
    ISO3_REG_LOG_CTRL.logIqO    = 0;
    ISO3_REG_LOG_CTRL.logDacO   = 0;
    ISO3_REG_LOG_CTRL.rfu2      = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_SetConfig(
                                            phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                            uint16_t wIdentifier,
                                            uint16_t wValue
                                            )
{
    /* Use SetConfig64 */
    if(pDataParams || wIdentifier || wValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_GetConfig(
                                     phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t * pwValue
                                     )
{
    /* Use SetConfig64 */
    if(pDataParams || wIdentifier || pwValue);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_GetConfig64(
                                            phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                            uint16_t wIdentifier,
                                            int64_t * pqValue
                                            )
{
    phhalHw_ISO3_TriggerCtrlRegister0_t* trigCtrl0;

    *pqValue = 0;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
    case PHDL_OSCI_CONFIG_MODE:
    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
    case PHDL_OSCI_CONFIG_RANGE_MIN:
    case PHDL_OSCI_CONFIG_RANGE_MAX:
    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
    case PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE:
    case PHDL_OSCI_CONFIG_NUMBER_CHANNELS:
    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES:
        *pqValue = pDataParams->dwLogSizeReg/PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG1_SIZE_NUM_SAMPLES:
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG2_SIZE_NUM_SAMPLES:
        *pqValue = PHDL_OSCI_ISO3SOFTSCOPE_RING_BUFFER_SAMPLE_COUNT;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_COND_FALL:
        trigCtrl0 = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
        *pqValue = trigCtrl0->conditionT1Fall;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_COND_FALL:
        trigCtrl0 = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
        *pqValue = trigCtrl0->conditionT2Fall;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG1_SIGNAL_ID:
        trigCtrl0 = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
        *pqValue = trigCtrl0->signalIdT1;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG2_SIGNAL_ID:
        trigCtrl0 = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
        *pqValue = trigCtrl0->signalIdT2;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED:
        *pqValue = pDataParams->bWaveFormHeaderEnabled;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_SAMPLE_SIZE:
        *pqValue = pDataParams->dwWaveFormSampleSize;
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_SetConfig64(
                                     phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t qwValue
                                     )
{
    phStatus_t statusTmp;
    uint64_t qwMax32 = 0xffffffff;
    uint64_t qwMax16 = 0xffff;
    uint64_t qwMax8 = 0xff;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
    case PHDL_OSCI_CONFIG_MODE:
    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
    case PHDL_OSCI_CONFIG_RANGE_MIN:
    case PHDL_OSCI_CONFIG_RANGE_MAX:
    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
    case PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE:
    case PHDL_OSCI_CONFIG_NUMBER_CHANNELS:
    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_TRIGGER_SRC:
        if((uint64_t)qwValue > qwMax32)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetBigBufferTriggerSource(pDataParams, (uint32_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_DATA_SRC:
        if((uint64_t)qwValue > qwMax32)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetDataSource(pDataParams, (uint32_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_COND_RISE:
        if((uint64_t)qwValue > qwMax16)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetTriggerRiseCondition(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1, (uint16_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_COND_FALL:
        if((uint64_t)qwValue > qwMax16)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetTriggerFallCondition(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1, (uint16_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_COND_RISE:
        if((uint64_t)qwValue > qwMax16)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetTriggerRiseCondition(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2, (uint16_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_COND_FALL:
        if((uint64_t)qwValue > qwMax16)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetTriggerFallCondition(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2, (uint16_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES:
        if((uint64_t)qwValue > qwMax32)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetBufferSize(pDataParams, (uint32_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_EXTERN_ENABLE:
        if((uint64_t)qwValue > qwMax8)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_EnableExternTrigger(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1, (uint8_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_EXTERN_ENABLE:
        if((uint64_t)qwValue > qwMax8)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_EnableExternTrigger(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2, (uint8_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_FDT_MEASUREMENT:
        if((uint64_t)qwValue > qwMax8)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_FDT_Measurement(pDataParams, (uint8_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_RISE_BIT_NUMB:
        if((uint64_t)qwValue > qwMax32)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_TriggerBitNumber(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_RISING, (uint32_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_FALL_BIT_NUMB:
        if((uint64_t)qwValue > qwMax32)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_TriggerBitNumber(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_FALLING, (uint32_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG1_SIGNAL_ID:
        if((uint64_t)qwValue > qwMax8)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetSignalId(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1, (uint8_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG2_SIGNAL_ID:
        if((uint64_t)qwValue > qwMax8)
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetSignalId(pDataParams, PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2, (uint8_t)qwValue));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG1_SIZE_NUM_SAMPLES:
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG2_SIZE_NUM_SAMPLES:
        /* size of these buffers cannot be changed */
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_OSCI);
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED:
        if(qwValue != PH_ON && qwValue != PH_OFF)
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        pDataParams->bWaveFormHeaderEnabled = (uint8_t)qwValue;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_SAMPLE_SIZE:
        if(pDataParams->dwLogSizeReg < qwValue * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        }
        pDataParams->dwWaveFormSampleSize = (uint32_t)qwValue;
        break;

    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_GetFieldStrength(
                                                   phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                                   uint8_t bChannel,
                                                   uint16_t * pwFieldStrength
                                                   )
{
    pDataParams;
    pwFieldStrength;
    bChannel;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_GetAmpl(
                                   phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwAmplitude
                                   )
{
    pDataParams;
    bChannel;
    pwAmplitude;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_GetPeakPeak(
                                   phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfPkPk
                                   )
{
    pDataParams;
    bChannel;
    pdfPkPk;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_GetRMS(
                                   phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwRMS
                                   )
{
    phStatus_t statusTmp;
    float64_t dfValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_GetRMSDouble(pDataParams, bChannel, &dfValue));

    *pwRMS = (uint16_t)(dfValue + 0.5);

    return statusTmp;
}

phStatus_t phdlOsci_ISO3SoftScope_GetRMSDouble(
                                   phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfRMS
                                   )
{
    pDataParams;
    bChannel;
    pdfRMS;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_SetCorrRange(
                                        phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                        uint8_t  bChannel
                                        )
{
    pDataParams;
    bChannel;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_GetWaveForm(
                                       phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       uint32_t dwWaveFormBufferSize,
                                       uint32_t * pdwWaveFormLength,
                                       uint8_t * pbWaveFormBuffer,
                                       uint8_t * pbHeaderOffset
                                       )
{
    phStatus_t statusTmp;
    /* Error checks */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    switch (bChannel)
    {
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER:
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE:
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE:
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL:
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_ReadBigBuffer(
            pDataParams,
            bChannel,
            pbWaveFormBuffer,
            dwWaveFormBufferSize,
            pdwWaveFormLength,
            pbHeaderOffset));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_RING_BUFFER_TRG1:
        /* Used for Ring Buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_ReadRingBuffer(
            pDataParams,
            PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1,
            pbWaveFormBuffer,
            dwWaveFormBufferSize,
            pdwWaveFormLength,
            pbHeaderOffset));
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_RING_BUFFER_TRG2:
        /* Used for Ring Buffer */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_ReadRingBuffer(
            pDataParams,
            PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2,
            pbWaveFormBuffer,
            dwWaveFormBufferSize,
            pdwWaveFormLength,
            pbHeaderOffset));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_InitOsci(
                                           phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
                                           )
{
    phStatus_t statusTmp;

    PH_ASSERT_NULL(pDataParams);

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Cmd_Reset(pDataParams));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */
