/*
 * Copyright 2019 - 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 FPGA soft scope.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <malloc.h> /* for malloc */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE

#include "phdlOsci_ISO3SoftScope.h"
#include "phdlOsci_ISO3SoftScope_Int.h"
#include "../phdlOsci_Int.h"
#include "../../../phhalHw/src/ISO3/phhalHw_ISO3_reg.h"
#include "../../../../intfs/phhalHw_ISO3_cmd.h" /* Requires Hal ISO3 interface function */


phStatus_t phdlOsci_ISO3SoftScope_Int_SetDataSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwDataSource
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pLogCtrl            = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    switch(dwDataSource)
    {
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_RX:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logRx = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_CAL:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logCal = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_COS:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logCos = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_SIN:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logSin = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_AGC:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logAgcO = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_IQ:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logIqO = PH_ON;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_DAC:
        phdlOsci_ISO3SoftScope_Int_ClearDataSources(pDataParams);
        pLogCtrl->logDacO = PH_ON;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_ResetSoftScope(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pLogCtrl            = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    statusTmp = phdlOsci_ISO3SoftScope_Int_StopAcquisition(pDataParams); /* Make sure that softscope finished capture */
    if (statusTmp != PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_OSCI)) /* Sometimes softscope triggers all the time, than WaitForBigBuffer returns TIMEOUT. In this case it is ok to do reset. */
    {
        PH_CHECK_SUCCESS(statusTmp);
    }

    pLogCtrl->reset = PH_ON;

    /* Toggle RESET-Bit of LogCtrl Reg (0x11) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    pLogCtrl->reset = PH_OFF;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_Int_TriggerBitNumber(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwConditionType,
    uint32_t dwBitNumber
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_TriggerCtrlRegister1_t* pTrigCtrl1;
    phhalHw_ISO3_DataParams_t* pHalParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pTrigCtrl1 = (phhalHw_ISO3_TriggerCtrlRegister1_t*) &(pDataParams->dwTriggerCtrl1Reg);
    pHalParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;
    switch (dwConditionType)
    {
    case PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_RISING:
        pTrigCtrl1->triggerRiseOnBitNumber = dwBitNumber;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_FALLING:
        pTrigCtrl1->triggerFallOnBitNumber = dwBitNumber;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL1,
        (uint8_t*)pTrigCtrl1,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_Int_SetBigBufferTriggerSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwLogTarget
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pLogCtrl            = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    phdlOsci_ISO3SoftScope_Int_ClearTriggerSource(pDataParams);

    if(dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_MLS)
    {
        pLogCtrl->trigMls = PH_ON;
    }
    else if(dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_FORCE)
    {
        pLogCtrl->trigForce = PH_ON;
    }
    else if(dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_1 ||
        dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_2)
    {
        if(dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_1)
            pLogCtrl->trig0 = PH_ON;
        if(dwLogTarget & PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_2)
            pLogCtrl->trig1 = PH_ON;
    }
    else
        PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_Int_ClearDataSources(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;

    PH_ASSERT_NULL(pDataParams);
    pLogCtrl = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);

    pLogCtrl->logRx = PH_OFF;
    pLogCtrl->logCal = PH_OFF;
    pLogCtrl->logCos = PH_OFF;
    pLogCtrl->logSin = PH_OFF;
    pLogCtrl->logAgcO = PH_OFF;
    pLogCtrl->logIqO = PH_OFF;
    pLogCtrl->logDacO = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_ClearTriggerSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;

    PH_ASSERT_NULL(pDataParams);
    pLogCtrl = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);

    pLogCtrl->trigMls = PH_OFF;
    pLogCtrl->trigForce = PH_OFF;
    pLogCtrl->trig0 = PH_OFF;
    pLogCtrl->trig1 = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_SetTriggerRiseCondition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint16_t wCondition
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_TriggerCtrlRegister0_t* trigCtrl0;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    trigCtrl0           = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1)
    {
        trigCtrl0->conditionT1Rise = 0xfu & wCondition;
    }

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2)
    {
        trigCtrl0->conditionT2Rise = 0xfu & wCondition;
    }

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*)trigCtrl0,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_SetTriggerFallCondition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint16_t wCondition
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_TriggerCtrlRegister0_t* trigCtrl0;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    trigCtrl0           = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1)
    {
        trigCtrl0->conditionT1Fall = 0xfu & wCondition;
    }

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2)
    {
        trigCtrl0->conditionT2Fall = 0xfu & wCondition;
    }

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*)trigCtrl0,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_Int_EnableExternTrigger(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint8_t bEnabled
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    phhalHw_ISO3_TriggerCtrlRegister0_t* trigCtrl0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;
    trigCtrl0           = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1)
    {
        trigCtrl0->externEnableT1 = PH_ON & bEnabled;
    }

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2)
    {
        trigCtrl0->externEnableT2 = PH_ON & bEnabled;
    }

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*)trigCtrl0,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_SetSignalId(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint8_t bId
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    phhalHw_ISO3_TriggerCtrlRegister0_t* trigCtrl0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;
    trigCtrl0           = (phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg);

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1)
    {
        trigCtrl0->signalIdT1 = (0x03u & bId);
    }

    if (bTriggerID == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2)
    {
        trigCtrl0->signalIdT2 = (0x03u & bId);
    }

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*)trigCtrl0,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_SetBufferSize(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwNumbSamples
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    uint32_t dwLogSizeBytes;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);
    PH_ASSERT_NULL_PARAM(dwNumbSamples, PH_COMP_DL_OSCI);

    /* check if buffer size is bigger than soft scope */
    if(dwNumbSamples > PHDL_OSCI_ISO3SOFTSCOPE_MAX_BUFFER_SIZE / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
    }

    dwLogSizeBytes = dwNumbSamples * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;

    /* if already set no update needed */
    if(pDataParams->dwLogSizeReg == dwLogSizeBytes)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_SIZE,
        (uint8_t*)&dwNumbSamples,
        sizeof(uint32_t)));

    pDataParams->dwLogSizeReg = dwLogSizeBytes;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_CalcBigBufferOffsetAndLength(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint32_t dwAvailableBufferSizeBytes,
    uint64_t* pqwByteOffset,
    uint64_t* pqwLengthBytes)
{
    phStatus_t statusTmp;
    uint32_t dwTrigRiseSampleCounter;
    uint32_t dwTrigFallSampleCounter;
    uint64_t qwBufferSizeNumSamples;

    PH_ASSERT_NULL_PARAM(pqwByteOffset, PH_COMP_DL_OSCI);
    PH_ASSERT_NULL_PARAM(pqwLengthBytes, PH_COMP_DL_OSCI);

    /* Reset data */
    *pqwByteOffset = 0;
    *pqwLengthBytes = 0;

    if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE ||
        bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL ||
        bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE ||
        bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL)
    {
        /* Get trigger sample counters (keyword offset is not included) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_GetTriggerCounter(
            pDataParams,
            bChannel,
            &dwTrigRiseSampleCounter,
            &dwTrigFallSampleCounter));

        if(dwTrigFallSampleCounter <= dwTrigRiseSampleCounter)
        {
            /* Workaround because trigger time will be overwritten after exchange in case trigger rise is set to TX_IDLE */
            if (((bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE || bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL) && ISO3_REG_TRIGGER_CTRL_0.conditionT1Rise == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_IDLE) ||
                ((bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE || bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL) && ISO3_REG_TRIGGER_CTRL_0.conditionT2Rise == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_IDLE))
            {
                if (dwTrigFallSampleCounter == 0) /* Still not fulfilled */
                {
                    return PH_ADD_COMPCODE(PHDL_OSCI_ERR_TRIGGER_NOT_FIRED, PH_COMP_DL_OSCI);
                }
                dwTrigRiseSampleCounter = 0;
            }
            else
            {
                return PH_ADD_COMPCODE(PHDL_OSCI_ERR_TRIGGER_NOT_FIRED, PH_COMP_DL_OSCI);
            }
        }

        /* Find length */
        if(((dwTrigFallSampleCounter - dwTrigRiseSampleCounter) * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE) >
            dwAvailableBufferSizeBytes)
        {
            *pqwLengthBytes = dwAvailableBufferSizeBytes;
        }
        else
        {
            *pqwLengthBytes = (dwTrigFallSampleCounter - dwTrigRiseSampleCounter) *
                PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
        }

        /* Find offset */
        if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE ||
            bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE)
        {
            *pqwByteOffset = dwTrigRiseSampleCounter * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
        }
        else if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL ||
            bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL)
        {
            *pqwByteOffset = (dwTrigFallSampleCounter * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE) - *pqwLengthBytes;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
        }

        /* Check Buffer overflow */
        if(*pqwLengthBytes > PHDL_OSCI_ISO3SOFTSCOPE_MAX_BUFFER_SIZE)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
        }
    }
    else if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER)
    {
        /* Check buffer size */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_GetConfig64(
            pDataParams,
            PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES,
            (int64_t*)&qwBufferSizeNumSamples));

        if(qwBufferSizeNumSamples > PHDL_OSCI_ISO3SOFTSCOPE_MAX_UINT32 / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }

        /* Find length */
        if(dwAvailableBufferSizeBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE > qwBufferSizeNumSamples)
        {
            *pqwLengthBytes = qwBufferSizeNumSamples * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
        }
        else
        {
            *pqwLengthBytes = dwAvailableBufferSizeBytes;
        }

        /* set offset */
        *pqwByteOffset = 0;
    }
    else
    {
        PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_ReadBigBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t * pWaveFormBuffer,
    uint32_t dwWaveFormBufferSizeBytes,
    uint32_t * pdwWaveFormLengthReceived,
    uint8_t * pbHeaderOffset
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    uint64_t pqwCmdParams[2];
    uint16_t wNumPackets;
    uint16_t wRestBytes;
    uint16_t i;
    uint64_t qwByteOffset;
    uint64_t qwLengthBytes;
    uint32_t dwReservedAreaSizeBytes;
    uint8_t* pbHeadRegion;
    uint8_t* pbDataRegion;
    int64_t bHeaderEnabled;
    uint64_t qwWaveformSampleSize;
    uint32_t dwWaveformSizeBytes;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Header enabled? */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_GetConfig64(
        pDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED,
        &bHeaderEnabled));

    /* Error checks */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL_PARAM(pWaveFormBuffer, PH_COMP_DL_OSCI);
    PH_ASSERT_NULL_PARAM(pdwWaveFormLengthReceived, PH_COMP_DL_OSCI);

    *pdwWaveFormLengthReceived = 0;

    if(!bHeaderEnabled)
    {
        /* No header included */
        if(dwWaveFormBufferSizeBytes == 0)
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);

        dwReservedAreaSizeBytes = 0;
        pbHeadRegion = NULL;
        pbDataRegion = (uint8_t*)pWaveFormBuffer;
    }
    else
    {
        /* Header included */
        PH_ASSERT_NULL_PARAM(pbHeaderOffset, PH_COMP_DL_OSCI);
        *pbHeaderOffset = 0;

        if(dwWaveFormBufferSizeBytes < PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET)
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);

        dwReservedAreaSizeBytes = PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
        pbHeadRegion = (uint8_t*)pWaveFormBuffer + PHDL_OSCI_ISO3SOFTSCOPE_HEADER_BYTE_OFFSET;
        pbDataRegion = (uint8_t*)pWaveFormBuffer + PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
    }

    /* Wait until Big Buffer is ready to be read */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_StopAcquisition(pDataParams));

    /* Get the Picc waveform samples count */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetConfig64(
        pDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_SAMPLE_SIZE,
        (int64_t*)&qwWaveformSampleSize));

    /* Concert the PICC waveform samples to bytes*/
    if(qwWaveformSampleSize > PHDL_OSCI_ISO3SOFTSCOPE_MAX_UINT32 / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);

    dwWaveformSizeBytes = (uint32_t)(qwWaveformSampleSize * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE);

    /* Calculate offset and length */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CalcBigBufferOffsetAndLength(
        pDataParams,
        bChannel,
        dwWaveformSizeBytes,
        &qwByteOffset,
        &qwLengthBytes));

    wNumPackets = (uint16_t)(qwLengthBytes / PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE);
    wRestBytes  = (uint16_t)(qwLengthBytes % PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE);

    /** Receive chunks */
    for(i = 0; i < wNumPackets; i++)
    {
        pqwCmdParams[0] = qwByteOffset;  /* from offset */
        pqwCmdParams[1] = PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE; /* numb bytes  */

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
            pHalISO3DataParams,
            PHHAL_HW_ISO3_SERVER_CMD_READ_SOFTSCOPE,
            (uint8_t*) pqwCmdParams,
            sizeof(pqwCmdParams)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_RecvServerResponse(
            pHalISO3DataParams,
            pbDataRegion + (i * PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE),
            PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE));

        qwByteOffset += PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE;
    }

    /** Receive rest */
    if(wRestBytes > 0)
    {
        pqwCmdParams[0] = qwByteOffset;  /* from offset */
        pqwCmdParams[1] = wRestBytes; /* numb bytes  */

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
            pHalISO3DataParams,
            PHHAL_HW_ISO3_SERVER_CMD_READ_SOFTSCOPE,
            (uint8_t*) pqwCmdParams,
            sizeof(pqwCmdParams)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_RecvServerResponse(
            pHalISO3DataParams,
            pbDataRegion + (i * PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE),
            wRestBytes));
    }

    /* Return number of valid bytes withing waveform buffer */
    *pdwWaveFormLengthReceived = (uint32_t)qwLengthBytes + dwReservedAreaSizeBytes;

    if(bHeaderEnabled)
    {
        *pbHeaderOffset = PHDL_OSCI_ISO3SOFTSCOPE_HEADER_BYTE_OFFSET;

        /* Create waveform header */
        switch (bChannel)
        {
        case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE:
        case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CreateHeader(
                pDataParams,
                pbHeadRegion,
                (uint32_t)(qwLengthBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE),    /* postEventCounter */
                0));                                                                /* preEventCounter */
            break;
        case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER:
        case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL:
        case PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CreateHeader(
                pDataParams,
                pbHeadRegion,
                0,                                                                  /* postEventCounter */
                (uint32_t)(qwLengthBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)));  /* preEventCounter */
            break;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_StoreBigBufferDataInRemoteBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bRemoteBufferId,
    uint32_t dwMaxBufferSizeBytes
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;
    uint64_t qwByteOffset;
    uint64_t qwLengthBytes;
    uint32_t pdwCmdParams[5];

    /* Error checks */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Wait until Big Buffer is ready to be read */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_StopAcquisition(pDataParams));

    /* Calculate offset and length */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CalcBigBufferOffsetAndLength(
        pDataParams,
        PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER,
        dwMaxBufferSizeBytes,
        &qwByteOffset,
        &qwLengthBytes));

    ((uint64_t*)pdwCmdParams)[0] = qwByteOffset;         /* from offset */
    ((uint64_t*)pdwCmdParams)[1] = qwLengthBytes;        /* numb bytes */
    pdwCmdParams[4] = bRemoteBufferId;                   /* remote buffer ID */

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_SendServerCommand(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_SERVER_CMD_STORE_SOFTSCOPE,
        (uint8_t*) pdwCmdParams,
        sizeof(pdwCmdParams)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_ReadRingBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTrigger,
    uint8_t * pbWaveFormBuffer,
    uint32_t dwWaveFormBufferSizeBytes,
    uint32_t * pdwWaveFormLengthReceived,
    uint8_t * pbHeaderByteOffset
    )
{
    phStatus_t statusTmp;
    uint16_t wRingBufferOffset;
    uint16_t wCounter;
    uint32_t dwWriteNumSamples;
    phhalHw_ISO3_DataParams_t * pHalDataParams;

    uint8_t* pbHeadRegion;
    uint8_t* pbDataRegion;

    /* Error checks */
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);
    PH_ASSERT_NULL_PARAM(pbWaveFormBuffer, PH_COMP_DL_OSCI);

    *pdwWaveFormLengthReceived = 0;
    pHalDataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Should wave form header be included? */
    if(pDataParams->bWaveFormHeaderEnabled == PH_ON)
    {
        if(dwWaveFormBufferSizeBytes < PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET ||
            pbHeaderByteOffset == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        dwWriteNumSamples = (dwWaveFormBufferSizeBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE) -
            PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
        pbHeadRegion = pbWaveFormBuffer + PHDL_OSCI_ISO3SOFTSCOPE_HEADER_BYTE_OFFSET;
        pbDataRegion = pbWaveFormBuffer + PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
    }
    else
    {
        dwWriteNumSamples = dwWaveFormBufferSizeBytes / PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
        pbHeadRegion = NULL;   /* not used */;
        pbDataRegion = (uint8_t*)pbWaveFormBuffer + 0;
    }

    /* Write up to PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_BUFFER_COUNT samples */
    if(dwWriteNumSamples > PHDL_OSCI_ISO3SOFTSCOPE_RING_BUFFER_SAMPLE_COUNT)
        dwWriteNumSamples = PHDL_OSCI_ISO3SOFTSCOPE_RING_BUFFER_SAMPLE_COUNT;

    /* Check if FDT_DATA did change since last ReadRingBuffer call */
    if(pHalDataParams->dwExchangeCounter != pDataParams->dwLastExchangeCounter)
    {
        pDataParams->dwLastExchangeCounter = pHalDataParams->dwExchangeCounter;

        /* Read 32-bit registers of FDT data into pDataParams */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(pHalDataParams,
            PHHAL_HW_ISO3_REG_FDT_DATA,
            (uint8_t*)&(pDataParams->pdwRingBuffer),
            sizeof(pDataParams->pdwRingBuffer)));
    }

    if (bTrigger == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1)
    {
        /* Copy data beginning from first sample to WaveFormPointer*/
        wRingBufferOffset = pHalDataParams->wLastFdtValueT1 % PHHAL_HW_ISO3_FDT_DATA_COUNT;
        for (wCounter = 0; wCounter < dwWriteNumSamples; wCounter++)
        {
            ((uint16_t*)pbDataRegion)[wCounter]  =
                (uint16_t)(pDataParams->pdwRingBuffer[(wRingBufferOffset + wCounter) % PHHAL_HW_ISO3_FDT_DATA_COUNT] & 0x0000FFFF);
        }
    }
    if (bTrigger == PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2)
    {
        /* Copy data beginning from offset */
        wRingBufferOffset = pHalDataParams->wLastFdtValueT2 % PHHAL_HW_ISO3_FDT_DATA_COUNT;
        for (wCounter = 0; wCounter < dwWriteNumSamples; wCounter++)
        {
            ((uint16_t*)pbDataRegion)[wCounter]  =
                (uint16_t)((pDataParams->pdwRingBuffer[(wRingBufferOffset + wCounter) % PHHAL_HW_ISO3_FDT_DATA_COUNT] & 0xFFFF0000) >> 16);
        }
    }

    /* Return number of valid bytes withing waveform buffer */
    *pdwWaveFormLengthReceived = dwWriteNumSamples * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;
    if(pDataParams->bWaveFormHeaderEnabled == PH_ON)
    {
        *pdwWaveFormLengthReceived += PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
        *pbHeaderByteOffset = PHDL_OSCI_ISO3SOFTSCOPE_HEADER_BYTE_OFFSET;

        /* Create wave form header */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_CreateHeader(
            pDataParams,
            pbHeadRegion,
            0,                              /* postCounter */
            dwWriteNumSamples));            /* preCounter */
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_CreateHeader(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t * pbHeaderArea,
    uint32_t postTriggerSampleCounter,
    uint32_t preTriggerSampleCounter
    )
{
    phdlOsci_ISO3SoftScope_WaveFormHeader_t waveFormHeader;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL_PARAM(pbHeaderArea, PH_COMP_DL_OSCI);

    /* Suppress unused warning for release build */
    PH_UNUSED_VARIABLE(pDataParams);

    memset(pbHeaderArea, 0, PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET);

    memcpy(waveFormHeader.pbInstrumentName,
        PHDL_OSCI_ISO3SOFTSCOPE_HEADER_INSTRUMENT_NAME,
        PHDL_OSCI_ISO3SOFTSCOPE_HEADER_INSTRUMENT_NAME_LENGTH);

    waveFormHeader.dwWaveFormOffset            = (uint32_t) PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET;
    waveFormHeader.dwHeaderVersion             = (uint32_t) PHDL_OSCI_ISO3SOFTSCOPE_HEADER_CURRENT_VERSION;
    waveFormHeader.dwGain                      = (uint32_t) PHDL_OSCI_ISO3SOFTSCOPE_HEADER_GAIN;
    waveFormHeader.dwPostTriggerSampleCounter  = (uint32_t) postTriggerSampleCounter;
    waveFormHeader.dwPreTriggerSampleCounter   = (uint32_t) preTriggerSampleCounter;
    waveFormHeader.qwHorizontalIntervalSec     = (float64_t) 1 / 135.6e6;

    memcpy(pbHeaderArea, &waveFormHeader, sizeof(phdlOsci_ISO3SoftScope_WaveFormHeader_t));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_ISO3SoftScope_Int_GetTriggerCounter(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint32_t * pdwRiseSampleCounter,
    uint32_t * pdwFallSampleCounter
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t * pHalDataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalDataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Get Trigger Counter */
    if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE ||
        bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHHAL_HW_ISO3_REG_TRIG1_RISE_COUNTER,
            (uint8_t*) pdwRiseSampleCounter,
            sizeof(uint32_t)));

         PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHHAL_HW_ISO3_REG_TRIG1_FALL_COUNTER,
            (uint8_t*) pdwFallSampleCounter,
            sizeof(uint32_t)));
    }
    else if(bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE ||
        bChannel == PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL)
    {
         PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHHAL_HW_ISO3_REG_TRIG2_RISE_COUNTER,
            (uint8_t*) pdwRiseSampleCounter,
            sizeof(uint32_t)));

         PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            PHHAL_HW_ISO3_REG_TRIG2_FALL_COUNTER,
            (uint8_t*) pdwFallSampleCounter,
            sizeof(uint32_t)));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_StopAcquisition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    phStatus_t statusTmp2;
    uint32_t dwNumbSamplesTmp = 0;

    /* Error checks */
    PH_ASSERT_NULL(pDataParams);

    /* Save current buffer size */
    dwNumbSamplesTmp = pDataParams->dwLogSizeReg/PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE;

    /* This should stop softscope acquisition  */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetBufferSize(pDataParams, 1));

    /* Wait for Softscope to complete acquiring data */
    statusTmp2 = phdlOsci_ISO3SoftScope_Int_CompleteAcquisition(pDataParams);

    /* Restoring buffer size */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_ISO3SoftScope_Int_SetBufferSize(pDataParams, dwNumbSamplesTmp));

    /* If softscope timeout occurs */
    if(statusTmp2 == PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_OSCI))
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_FDT_Measurement(phdlOsci_ISO3SoftScope_DataParams_t * pDataParams, uint8_t bEnabled)
{
    phStatus_t statusTmp;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    ((phhalHw_ISO3_TriggerCtrlRegister0_t*) &(pDataParams->dwTriggerCtrl0Reg))->fdtEnable = 0x01 & bEnabled;

    /* Write config to FPGA register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_TRIG1_CTRL0,
        (uint8_t*) &(pDataParams->dwTriggerCtrl0Reg),
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_ForceTrigger(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    phhalHw_ISO3_LogCtrlRegister_t *pLogCtrl;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pLogCtrl            = (phhalHw_ISO3_LogCtrlRegister_t*) &(pDataParams->dwLogCtrlReg);
    pHalISO3DataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    pLogCtrl->trigForce = PH_ON;

    /* Toggle RESET-Bit of LogCtrl Reg (0x11) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    pLogCtrl->trigForce = PH_OFF;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_WriteFpgaRegister(
        pHalISO3DataParams,
        PHHAL_HW_ISO3_REG_LOG_CTRL,
        (uint8_t*)pLogCtrl,
        sizeof(uint32_t)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_IsSignalInRange(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwDataSource,
    uint8_t * overflow
    )
{
    phStatus_t statusTmp;
    uint32_t signalValue = 0;
    phhalHw_ISO3_DataParams_t * pHalDataParams;
    uint8_t signalSrc;

    *overflow = 0;

    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalDataParams  = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    switch(dwDataSource)
    {
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_RX:
        signalSrc = PHHAL_HW_ISO3_REG_PEAK_SENS_SIGNAL;
        break;
    case PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_CAL:
        signalSrc = PHHAL_HW_ISO3_REG_PEAK_CAL_SIGNAL;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalDataParams,
            signalSrc,
            (uint8_t*) &signalValue,
            sizeof(uint32_t)));

    if (signalValue == PHDL_HW_ISO3_MAX_CAL_SENS_VALUE)
    {
        *overflow = 1;
    }
    else
    {
        *overflow = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_ISO3SoftScope_Int_CompleteAcquisition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint32_t dwScopeBusy = true;
    uint32_t dwTimePassed = 0;
    phhalHw_ISO3_DataParams_t* pHalISO3DataParams;

    /* Error checks */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHalISO3DataParams);

    pHalISO3DataParams = (phhalHw_ISO3_DataParams_t*) pDataParams->pHalISO3DataParams;

    /* Wait until softscope gets free  */
    while(dwScopeBusy && (dwTimePassed <= PHDL_OSCI_ISO3SOFTSCOPE_BIG_BUFFER_BUSY_TIMEOUT_MS))
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_ISO3_Cmd_ReadFpgaRegister(
            pHalISO3DataParams,
            PHHAL_HW_ISO3_REG_SOFT_SCOPE_BUSY,
            (uint8_t*) &dwScopeBusy,
            sizeof(uint32_t)));

        if(dwScopeBusy)
        {
            dwTimePassed += 100;
            phTools_Sleep(100);
        }
    }

    if(dwTimePassed >= PHDL_OSCI_ISO3SOFTSCOPE_BIG_BUFFER_BUSY_TIMEOUT_MS)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */
