/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 FPGA soft scope.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLOSCI_ISO3SOFTSCOPE_INT_H
#define PHDLOSCI_ISO3SOFTSCOPE_INT_H

#include <ph_Status.h>
#include <phdlOsci.h>
#include <math.h>

#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_1                   0x1     /**< Trigger 1 */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_2                   0x2     /**< Trigger 2 */
#define PHDL_OSCI_ISO3SOFTSCOPE_KEYWORD_SEARCH_AREA_SIZE    (100 * PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE)     /**< Search area in bytes */
#define PHDL_OSCI_ISO3SOFTSCOPE_KEYWORD                     0x4b1d  /**< 16-bit keyword in search area defining first valid sample of soft scope buffer */

#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_INSTRUMENT_NAME     "ISO3SoftScope\n"    /**< Name of this instrument specified in header struct */
#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_INSTRUMENT_NAME_LENGTH 14                /**< Length of instrument name */
#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_CHANEL_BIG_NAME     "BigS"               /**< Name of current chanel specified in header struct */
#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_CHANEL_RING1_NAME   "Rng1"               /**< Name of current chanel specified in header struct */
#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_CHANEL_RING2_NAME   "Rng2"               /**< Name of current chanel specified in header struct */

#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_CURRENT_VERSION     1                    /**< Current header version */
#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_GAIN                1                    /**< Current header version */

#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_RISING              1                   /**< */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_FALLING             2                   /**< */

#define PHDL_OSCI_ISO3SOFTSCOPE_BIG_BUFFER_BUSY_TIMEOUT_MS  5000                /**< Max time to wait until big buffer is ready to be read */

#define PHDL_OSCI_ISO3SOFTSCOPE_MAX_UINT32                                          0xffffffff
#define PHDL_OSCI_ISO3SOFTSCOPE_MAX_UINT16                                          0xffff
#define PHDL_OSCI_ISO3SOFTSCOPE_MAX_READ_SIZE                                       64240 /* changed to smaller than 0xFFFE because this resolves the network issues, this must be an even number because 1 sample is 2 bytes */

typedef struct ISO3SoftScopeWaveFormHeader
{
    int8_t pbInstrumentName[PHDL_OSCI_ISO3SOFTSCOPE_HEADER_INSTRUMENT_NAME_LENGTH];   /**< Name of instrument */
    uint32_t dwHeaderVersion;                                                         /**< Header version */
    uint32_t dwWaveFormOffset;                                                        /**< Start offset of waveform */
    uint32_t dwGain;                                                                  /**< Gain conversion factor */
    uint32_t dwPreTriggerSampleCounter;                                               /**< Number of samples before trigger event */
    uint32_t dwPostTriggerSampleCounter;                                              /**< Number of samples after trigger event */
    float64_t qwHorizontalIntervalSec;                                                /**< Time interval in between two samples */
} phdlOsci_ISO3SoftScope_WaveFormHeader_t;

#define PHDL_OSCI_ISO3SOFTSCOPE_HEADER_BYTE_OFFSET              0

#define PHDL_OSCI_ISO3SOFTSCOPE_WAVEFORM_BYTE_OFFSET            sizeof(phdlOsci_ISO3SoftScope_WaveFormHeader_t) + \
                                                                sizeof(phdlOsci_ISO3SoftScope_WaveFormHeader_t) % sizeof(uint16_t)   /**< Header size in bytes */
/**
* \brief Set data source of the next event. Sources are PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_RX,
* PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_CAL, PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_COS,
* PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_AGC, PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_DAC, ...
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetDataSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwDataSource
    );

/**
* \brief Set number of bit for trigger condition.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_TriggerBitNumber(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwConditionType,
    uint32_t dwBitNumber
    );

/**
* \brief Reset the soft scope interface registers on the FPGA.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ResetSoftScope(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );

/**
* \brief Set trigger source of the big buffer.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetBigBufferTriggerSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwLogTarget
    );

/**
* \brief Helper function to reset the data sources of the log ctrl register to 0.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ClearDataSources(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );

/**
* \brief Helper function to reset the log targets of the log ctrl register to 0.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ClearTriggerSource(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );

/**
* \brief Set the trigger condition for the rise event. This is used in the setConfig generic function.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetTriggerRiseCondition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint16_t wCondition
    );

/**
* \brief Set the trigger condition for the fall event. This is used in the setConfig generic function.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetTriggerFallCondition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint16_t wCondition
    );

/**
* \brief Enable to external output trigger event. This is used in the setConfig generic function.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_EnableExternTrigger(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint8_t bEnabled
    );

/**
* \brief Set the signal id for the specified trigger.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetSignalId(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bTriggerID,
    uint8_t bId
    );

/**
* \brief Set the buffer size of the soft scope buffer to a number of samples.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_SetBufferSize(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwNumbSamples
    );

/**
* \brief Read samples of big soft scope buffer.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ReadBigBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t * pWaveFormBuffer,
    uint32_t dwWaveFormBufferSize,
    uint32_t * pdwWaveformLength,
    uint8_t * pbHeaderOffset
    );

/**
* \brief Store samples of big soft scope buffer on PLServer.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_StoreBigBufferDataInRemoteBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bRemoteBufferId,
    uint32_t dwMaxBufferSizeBytes
    );

/**
* \brief Read samples of ring buffer of trigger 1 or 2.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ReadRingBuffer(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannelForTrigger,
    uint8_t * pWaveFormBuffer,
    uint32_t dwWaveFormBufferSize,
    uint32_t * pdwWaveFormLengthReceived,
    uint8_t * pbHeaderOffset
    );


/**
* \brief Create header struct for GetWaveForm function.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_CreateHeader(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t * pbHeaderArea,
    uint32_t postTriggerSampleCounter,
    uint32_t preTriggerSampleCounter
    );

/**
* \brief Get rise and fall counter offset within soft scope buffer for channels:
*   -> PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE
*   -> PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL
*   -> PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE
*   -> PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_GetTriggerCounter(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint32_t * dwRiseCounter,
    uint32_t * dwFallCounter
    );

/**
* \brief Stops data acquisition and waits for softscope to finish
* This is done by setting buffer size to very small value.
* Than function waits for trigger_counter to be 0.
* Which means that acquisition is finished.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_StopAcquisition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );

/**
* \brief Enables/Disable FDT measurement
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_FDT_Measurement(phdlOsci_ISO3SoftScope_DataParams_t * pDataParams, uint8_t bEnabled);


/**
* \brief Forces softscope to trigger capture
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_ForceTrigger(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );

/**
* \brief Checks if ADC signal value on CAL or on SENS is within range
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_IsSignalInRange(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,
    uint32_t dwDataSource,
    uint8_t * overflow
    );

/**
* \brief waits for softscope to finish fetching data
* This is done by setting a wait time in a loop.
* Than function waits for trigger_counter to be 0.
* Which means that acquisition is finished.
* \return Status code
* \retval #PH_ERR_SUCCESS Command successfully executed
*/
phStatus_t phdlOsci_ISO3SoftScope_Int_CompleteAcquisition(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams
    );



#endif /* PHDLOSCI_ISO3SOFTSCOPE_INT_H */
