/*
 * Copyright 2013 - 2014, 2017 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Oscilloscope Waverunner L64Xi Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#include <stdio.h>              /* PRQA S 5124 */

#ifdef NXPBUILD__PHDL_OSCI_LW64XI

/* for visa functionality displaced function exchange large */
#include "../../../phbalReg/src/Visa/external/visa.h"
#include "../../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"

#include <phdlOsci.h>
#include "phdlOsci_LW64Xi.h"
#include "phdlOsci_LW64Xi_Int.h"
#include "../phdlOsci_Int.h"

phStatus_t phdlOsci_LW64Xi_Init(
                                phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                uint16_t wSizeOfDataParams,
                                phbalReg_Visa_DataParams_t * pBalRegDataParams
                                )
{
    if (sizeof(phdlOsci_LW64Xi_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                 = PH_COMP_DL_OSCI | PHDL_OSCI_LW64XI_ID;
    pDataParams->pBalRegDataParams   = pBalRegDataParams;
    /* default value for standard antenna */
    pDataParams->wFieldStrengthMultiplier = 318;
    pDataParams->wAverageFact = 5;
    pDataParams->wRangeMin = 60;
    pDataParams->wRangeMax = 72;
    pDataParams->bMeasurementOptions = PHDL_OSCI_MEASUREMENT_ADJUST_TIME | PHDL_OSCI_MEASUREMENT_ADJUST_RANGE;
    pDataParams->bSelectedChannel = PHDL_OSCI_CHANNEL_1;
    pDataParams->bAutoAdjustChannels = PH_ON;
    pDataParams->dwMaxSamplesMaxValue = 0xffffffff;
    pDataParams->dwMaxSamplesMinValue = NULL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_SetConfig(
                                     phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t wValue
                                     )
{
    return phdlOsci_LW64Xi_SetConfig64(pDataParams, wIdentifier, wValue);
}

phStatus_t phdlOsci_LW64Xi_GetConfig(
                                     phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t * pwValue
                                     )
{
    phStatus_t statusTmp = 0;
    int64_t qwTempValue = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwTempValue));

    if (qwTempValue < 0 || qwTempValue > 65535)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }

    *pwValue = (uint16_t)qwTempValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_SetConfig64(
                                     phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t qwValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t abBuffer[75];
    int64_t qwCurrentConfigValue = 0;
    int64_t qwCurrentTriggerLevel = 0;
    int32_t dwPrintLength = 0;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, (uint16_t)qwValue));
        break;

    case PHDL_OSCI_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConfigOsci(pDataParams, (uint8_t)qwValue));
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        pDataParams->wFieldStrengthMultiplier = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wAverageFact = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        if (qwValue < 5 ||
            qwValue > 70 ||
            qwValue >= pDataParams->wRangeMax)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMin = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        if (qwValue < 10 ||
            qwValue > 75 ||
            qwValue <= pDataParams->wRangeMin)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMax = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        if ((qwValue & ~PHDL_OSCI_MEASUREMENT_MASK_LW64Xi) != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bMeasurementOptions = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        qwValue /= 10;
        dwPrintLength = sprintf((char *)abBuffer, "TDIV %lldE-6\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        if (qwValue < PHDL_OSCI_CHANNEL_1 ||
            qwValue > PHDL_OSCI_CHANNEL_LINE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bSelectedChannel = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"BWL", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        switch(qwValue)
        {
        case PHDL_OSCI_COUPLING_AC:
            dwPrintLength = sprintf((char *)abBuffer, "A1M");
            break;

        case PHDL_OSCI_COUPLING_DC:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE, &qwCurrentConfigValue));
            if (qwCurrentConfigValue == PHDL_OSCI_IMPEDANCE_ONE_MEGA)
            {
                dwPrintLength = sprintf((char *)abBuffer, "D1M");
            }
            else
            {
                dwPrintLength = sprintf((char *)abBuffer, "D50");
            }
            break;

        case PHDL_OSCI_COUPLING_GND:
            dwPrintLength = sprintf((char *)abBuffer, "GND");
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"CPL", abBuffer));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"TRA", (uint8_t *)((qwValue == PH_ON) ? "ON" : "OFF")));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        if (qwValue != PHDL_OSCI_IMPEDANCE_FIFTY && qwValue != PHDL_OSCI_IMPEDANCE_ONE_MEGA)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_COUPLING, &qwCurrentConfigValue));
        if(qwCurrentConfigValue == PHDL_OSCI_COUPLING_AC)
        {
            if(qwValue == PHDL_OSCI_IMPEDANCE_FIFTY)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
            }
            else
            {
                dwPrintLength = sprintf((char *)abBuffer, "A1M");
            }
        }
        else if(qwCurrentConfigValue == PHDL_OSCI_COUPLING_DC)
        {
            if(qwValue == PHDL_OSCI_IMPEDANCE_FIFTY)
            {
                dwPrintLength = sprintf((char *)abBuffer, "D50");
            }
            else
            {
                dwPrintLength = sprintf((char *)abBuffer, "D1M");
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"CPL", abBuffer));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        if(qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        dwPrintLength = sprintf((char *)abBuffer, "VBS 'app.Acquisition.C%d.Invert = %lld'\n", pDataParams->bSelectedChannel, qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));
        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lld", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"ATTN", abBuffer));
        break;

    /*case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
        break;*/

    /*case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        break;*/

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        switch(qwValue)
        {
        case PHDL_OSCI_TRIGGER_TYPE_EDGE:
            dwPrintLength = sprintf((char *)abBuffer, "TRSE EDGE\n");
            break;
        case PHDL_OSCI_TRIGGER_TYPE_WIDTH:
            dwPrintLength = sprintf((char *)abBuffer, "TRSE WIDTH\n");
            break;
        case PHDL_OSCI_TRIGGER_TYPE_TEQ:
            dwPrintLength = sprintf((char *)abBuffer, "TRSE TEQ\n");
            break;
        case PHDL_OSCI_TRIGGER_TYPE_PATTERN:
            dwPrintLength = sprintf((char *)abBuffer, "TRSE PA\n");
            break;
        case PHDL_OSCI_TRIGGER_TYPE_TV:
            dwPrintLength = sprintf((char *)abBuffer, "TRSE TV\n");
            break;
        case PHDL_OSCI_TRIGGER_TYPE_SMART:
            /* further configs needed for smart mode */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        if (qwValue < PHDL_OSCI_CHANNEL_1 ||
            qwValue > PHDL_OSCI_CHANNEL_LINE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        /* Get Slope and Trigger level as those parameters are depending on trigger source config */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE, &qwCurrentConfigValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV, &qwCurrentTriggerLevel));

        switch(qwValue)
        {
        case PHDL_OSCI_CHANNEL_1:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"C1"));
            break;
        case PHDL_OSCI_CHANNEL_2:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"C2"));
            break;
        case PHDL_OSCI_CHANNEL_3:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"C3"));
            break;
        case PHDL_OSCI_CHANNEL_4:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"C4"));
            break;
        case PHDL_OSCI_CHANNEL_EXTERNAL:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"EX"));
            break;
        case PHDL_OSCI_CHANNEL_LINE:
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerConfig(pDataParams, (uint8_t *)"LINE"));
            break;
        default:
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
            }
        }

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))

        /* reset configs as before */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE, qwCurrentConfigValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV, qwCurrentTriggerLevel));
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        if (qwValue != PHDL_OSCI_EDGE_SLOPE_POSITIVE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_NEGATIVE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_TRIGGER_SOURCE, &qwCurrentConfigValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            (uint8_t)qwCurrentConfigValue, (uint8_t *)"TRSL", (uint8_t *)(qwValue ? "NEG" : "POS")));
        break;

    /*case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        break;*/

    /*case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
        break;*/

    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldE-3", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"OFST", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        sprintf((char *)abBuffer, "%lldE-3", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VDIV", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        qwValue /= 8;
        sprintf((char *)abBuffer, "%lldE-3", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VDIV", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        dwPrintLength = sprintf((char *)abBuffer, "%lldE-3", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_TRIGGER_SOURCE, &qwCurrentConfigValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetChannelConfig(pDataParams,
            (uint8_t)qwCurrentConfigValue, (uint8_t *)"TRLV", abBuffer));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        /* invert to behave like other oscis */
        qwValue *= -1;
        dwPrintLength = sprintf((char *)abBuffer, "TRDL %lldE-9\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        dwPrintLength = sprintf((char *)abBuffer, "TDIV %lldE-9\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        qwValue /= 10;
        dwPrintLength = sprintf((char *)abBuffer, "TDIV %lldE-9\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_LW64XI_CONFIG_AUTOADJUSTCHANNELS:
        if(qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bAutoAdjustChannels = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_LW64XI_CONFIG_NOISE_FILTER:
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        if(qwValue < PHDL_OSCI_LW64XI_NOISE_FILTER_NONE || qwValue > PHDL_OSCI_LW64XI_NOISE_FILTER_THREE_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        dwPrintLength = sprintf((char *)abBuffer, "VBS 'app.Acquisition.C%d.EnhanceResType = %lld'\n", pDataParams->bSelectedChannel, qwValue);

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
        /* Check value range of MaxSamplePoints */
        if(qwValue < pDataParams->dwMaxSamplesMinValue || qwValue > pDataParams->dwMaxSamplesMaxValue)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        /* Set Osci to SetMaximumMemory Mode */
        dwPrintLength = sprintf((char *)abBuffer, "VBS 'app.Acquisition.Horizontal.SmartMemory = SetMaximumMemory'\n");
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));
        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))

        dwPrintLength = sprintf((char *)abBuffer, "VBS 'app.Acquisition.Horizontal.MaxSamples = %llu'\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));
        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    case PHDL_OSCI_LW64XI_CONFIG64_WAVEFORM_SAMPLE_SIZE:
        if(qwValue < 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        if(qwValue > 0xFFFFFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
        }

        /* Set the number of samples that should be transfered when getting the buffer */
        dwPrintLength = sprintf((char *)abBuffer, "WFSU NP,%llu\n", qwValue);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abBuffer, (uint16_t)dwPrintLength));
        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_GetConfig64(
                                     phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t * pqwValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t abBuffer[75];
    float64_t tmpFloatVal;
    uint16_t wTmpValue = 0;
    int32_t dwTmpValue = 0;
    uint8_t bTmpValue = 0;
    int64_t qwCurrentTriggerSource = 0;
    int32_t dwPrintLength = 0;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTmpValue));
        *pqwValue = wTmpValue;
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        *pqwValue = pDataParams->wFieldStrengthMultiplier;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        *pqwValue = pDataParams->wAverageFact;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        *pqwValue = pDataParams->wRangeMin;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        *pqwValue = pDataParams->wRangeMax;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        *pqwValue = pDataParams->bMeasurementOptions;
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TDIV?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000000 * 10);
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        *pqwValue = pDataParams->bSelectedChannel;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"BWL?\n", (uint16_t)5 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetParseChannelValue(
            pDataParams->bSelectedChannel, abBuffer, &bTmpValue));
        *pqwValue = bTmpValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"CPL", abBuffer, 50));
        if (strncmp((char *)abBuffer, "A", 1) == 0)
        {
            *pqwValue = PHDL_OSCI_COUPLING_AC;
        }
        else if (strncmp((char *)abBuffer, "D", 1) == 0)
        {
            *pqwValue = PHDL_OSCI_COUPLING_DC;
        }
        else if (strncmp((char *)abBuffer, "GND", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_COUPLING_GND;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"TRA", abBuffer, 50));
        if (strncmp((char *)abBuffer, PH_VALUE_ON, 2) == 0)
        {
            *pqwValue = PH_ON;
        }
        else if (strncmp((char *)abBuffer, PH_VALUE_OFF, 3) == 0)
        {
            *pqwValue = PH_OFF;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"CPL", abBuffer, 50));
        if (strncmp((char *)abBuffer + 1, "1M", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_IMPEDANCE_ONE_MEGA;
        }
        else if (strncmp((char *)abBuffer + 1, "50", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_IMPEDANCE_FIFTY;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        dwPrintLength = sprintf((char *)abBuffer, "VBS? 'Return=app.Acquisition.C%d.Invert'\n", pDataParams->bSelectedChannel);

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)abBuffer, (uint16_t)dwPrintLength , 50, (uint8_t *)abBuffer, &wTmpValue));

        if (strncmp((char *)abBuffer, "0", 1) == 0)
        {
            *pqwValue = PH_OFF;
        }
        else if (strncmp((char *)abBuffer, "1", 1) == 0)
        {
            *pqwValue = PH_ON;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"ATTN", abBuffer, 50));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConvertStringToInt64(
            abBuffer, 1, pqwValue));
        break;

    /*case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
        break;*/

    /*case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        break;*/

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TRSE?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        if (strncmp((char *)abBuffer, "EDGE", 4) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_EDGE;
        }
        else if (strncmp((char *)abBuffer, "WIDTH", 5) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_WIDTH;
        }
        else if (strncmp((char *)abBuffer, "TEQ", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_TEQ;
        }
        else if (strncmp((char *)abBuffer, "PA", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_PATTERN;
        }
        else if (strncmp((char *)abBuffer, "TV", 2) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_TV;
        }
        else if (strncmp((char *)abBuffer, "GLIT", 4) == 0 ||
                 strncmp((char *)abBuffer, "INTV", 4) == 0 ||
                 strncmp((char *)abBuffer, "DROP", 4) == 0 ||
                 strncmp((char *)abBuffer, "RUNT", 4) == 0 ||
                 strncmp((char *)abBuffer, "SLEWRATE", 8) == 0)
        {
            *pqwValue = PHDL_OSCI_TRIGGER_TYPE_SMART;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TRSE?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetParseTriggerSource(pDataParams,
            abBuffer, &bTmpValue));
        *pqwValue = bTmpValue;
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TRSE?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetParseTriggerSource(pDataParams,
            abBuffer, &bTmpValue));
        *pqwValue = bTmpValue;

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            (uint8_t)*pqwValue, (uint8_t *)"TRSL", abBuffer, 50));
        if (strncmp((char *)abBuffer, "NEG", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_NEGATIVE;
        }
        else if (strncmp((char *)abBuffer, "POS", 3) == 0)
        {
            *pqwValue = PHDL_OSCI_EDGE_SLOPE_POSITIVE;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        break;

    /*case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        break;*/

    /*case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
        break;*/

    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"OFST", abBuffer, 50));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000);
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VDIV", abBuffer, 50));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000);
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            pDataParams->bSelectedChannel, (uint8_t *)"VDIV", abBuffer, 50));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000 * 8);
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG_TRIGGER_SOURCE, &qwCurrentTriggerSource));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetChannelConfig(pDataParams,
            (uint8_t)qwCurrentTriggerSource, (uint8_t *)"TRLV", abBuffer, 50));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000);
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TRDL?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000000000);
        /* invert to behave like other oscis */
        *pqwValue *= -1;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TDIV?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000000000);
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TDIV?\n", (uint16_t)6 , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)(tmpFloatVal * 1000000000 * 10);
        break;

    case PHDL_OSCI_LW64XI_CONFIG_AUTOADJUSTCHANNELS:
        *pqwValue = pDataParams->bAutoAdjustChannels;
        break;

    case PHDL_OSCI_LW64XI_CONFIG_NOISE_FILTER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetNoiseFilterValue(pDataParams, &bTmpValue));
        *pqwValue = bTmpValue;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
        dwPrintLength = sprintf((char *)abBuffer, "VBS? 'Return=app.Acquisition.Horizontal.MaxSamples'\n");

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)abBuffer, (uint16_t)dwPrintLength , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)tmpFloatVal;
        break;

    case PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE:
        dwPrintLength = sprintf((char *)abBuffer, "VBS? 'Return=app.Acquisition.Horizontal.SamplingRate'\n");

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)abBuffer, (uint16_t)dwPrintLength , 50, (uint8_t *)abBuffer, &wTmpValue));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(
        (int8_t*)abBuffer, &tmpFloatVal));

        *pqwValue = (int64_t)tmpFloatVal;
        break;

    case PHDL_OSCI_CONFIG_NUMBER_CHANNELS:
        *pqwValue = 4;
        break;

    case PHDL_OSCI_LW64XI_CONFIG64_WAVEFORM_SAMPLE_SIZE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"WFSU?\n", (uint16_t)6 , 75, (uint8_t *)abBuffer, &wTmpValue));

        /* Find "NP" and extract the value between the next two ',' */
        for (bTmpValue = 0; bTmpValue < wTmpValue - 2; bTmpValue++)
        {
            if (strncmp((char *)(abBuffer + bTmpValue), "NP", 2) == 0)
            {
                break;
            }
        }
        if (bTmpValue >= wTmpValue - 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }
        for (; bTmpValue < wTmpValue - 2; bTmpValue++)
        {
            if (abBuffer[bTmpValue] == ',')
            {
                break;
            }
        }
        if (bTmpValue >= wTmpValue - 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }

        bTmpValue++;

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32(
            (int8_t*)(abBuffer + bTmpValue), &dwTmpValue));
        *pqwValue = dwTmpValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_GetFieldStrength(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint16_t * pwFieldStrength
    )
{
    phStatus_t statusTmp, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t abCh[3] = "C1";
    uint16_t wTriggerMode;
    int64_t qwTimeBase = 0;
    int64_t qwTimeBasePosition = 0;

    if (bChannel >= 1 && bChannel <= 4)
    {
        abCh[1] = (uint8_t)'0' + bChannel;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* get current trigger setting */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTriggerMode));

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* get current timebase setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, &qwTimeBase));
        /* get current timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwTimeBasePosition));

        /* set measurement settings */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, 0));
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, 2000));
    }

    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
    {
        /* get in correct range */
        statusTmp = phdlOsci_LW64Xi_Int_SetCorrRange(pDataParams, (uint8_t *) abCh);
    }

    /* Only perform command if previous was success */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* get the amplitude of channel */
        statusTmp = phdlOsci_LW64Xi_Int_GetRMS(pDataParams , (uint8_t *)abCh, &fTemp);
    }

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* Scale must be done before position because otherwise the position could be out of range it will also be adjusted when scale is changed */
        /* restore timebase setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, qwTimeBase));
        /* restore timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, qwTimeBasePosition));
    }

    /* restore trigger mode */
    PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, wTriggerMode));

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* modify RMS value to get fieldstrength */
    fTemp = (float32_t)((fTemp * (float32_t) PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR * (float32_t) PHDL_OSCI_FIELD_STRENGTH_DIVISOR )/ (float32_t) pDataParams->wFieldStrengthMultiplier);

    *pwFieldStrength = (uint16_t)fTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_GetAmpl(
                                   phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwAmplitude
                                   )
{
    phStatus_t statusTmp, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t abCh[3] = "C1";
    uint16_t wTriggerMode;
    int64_t qwTimeBase = 0;
    int64_t qwTimeBasePosition = 0;

    if ( bChannel >= 1 && bChannel <= 4 )
    {
        abCh[1] = (uint8_t)'0' + bChannel;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* get current trigger setting */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTriggerMode));

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* get current timebase setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, &qwTimeBase));
        /* get current timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwTimeBasePosition));

        /* set measurement settings */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, 0));
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, 2000));
    }

    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
    {
        /* get in correct range */
        statusTmp = phdlOsci_LW64Xi_Int_SetCorrRange(pDataParams, (uint8_t *) abCh);
    }

    /* Only perform command if previous was success */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* get the amplitude of channel */
        statusTmp = phdlOsci_LW64Xi_Int_GetAmpl(pDataParams , (uint8_t *)abCh, &fTemp);
    }

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* Scale must be done before position because otherwise the position could be out of range it will also be adjusted when scale is changed */
        /* restore timebase setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, qwTimeBase));
        /* restore timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, qwTimeBasePosition));
    }

    /* restore trigger mode */
    PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, wTriggerMode));

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pwAmplitude = (uint16_t) (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_GetPeakPeak(
                                   phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfPkPk
                                   )
{
    phStatus_t statusTmp, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t abCh[3] = "C1";
    uint16_t wTriggerMode;
    int64_t qwTimeBase = 0;
    int64_t qwTimeBasePosition = 0;

    if ( bChannel >= 1 && bChannel <= 4 )
    {
        abCh[1] = (uint8_t)'0' + bChannel;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* get current trigger setting */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTriggerMode));


    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* get current timebase setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, &qwTimeBase));
        /* get current timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwTimeBasePosition));

        /* set measurement settings */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, 0));
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, 2000));
    }

    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
    {
        /* get in correct range */
        statusTmp = phdlOsci_LW64Xi_Int_SetCorrRange(pDataParams, (uint8_t *) abCh);
    }

    /* Only perform command if previous was success */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* get the amplitude of channel */
        statusTmp = phdlOsci_LW64Xi_Int_GetPeakPeak(pDataParams , (uint8_t *)abCh, &fTemp);
    }

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* Scale must be done before position because otherwise the position could be out of range it will also be adjusted when scale is changed */
        /* restore timebase setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, qwTimeBase));
        /* restore timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, qwTimeBasePosition));
    }

    /* restore trigger mode */
    PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, wTriggerMode));

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pdfPkPk = (float64_t) (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_GetRMS(
                                   phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwRMS
                                   )
{
    phStatus_t statusTmp;
    float64_t dfValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetRMSDouble(pDataParams, bChannel, &dfValue));

    *pwRMS = (uint16_t)(dfValue + 0.5);

    return statusTmp;
}

phStatus_t phdlOsci_LW64Xi_GetRMSDouble(
                                   phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfRMS
                                   )
{
    phStatus_t statusTmp, statusRestore;
    float32_t fTemp = 0.0;
    uint8_t abCh[3] = "C1";
    uint16_t wTriggerMode;
    int64_t qwTimeBase = 0;
    int64_t qwTimeBasePosition = 0;

    if ( bChannel >= 1 && bChannel <= 4 )
    {
        abCh[1] = (uint8_t)'0' + bChannel;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* get current trigger setting */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTriggerMode));

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* get current timebase setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, &qwTimeBase));
        /* get current timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwTimeBasePosition));

        /* set measurement settings */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, 0));
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, 2000));
    }

    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
    {
        /* get in correct range */
        statusTmp = phdlOsci_LW64Xi_Int_SetCorrRange(pDataParams, (uint8_t *) abCh);
    }

    /* Only perform command if previous was success */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* get the amplitude of channel */
        statusTmp = phdlOsci_LW64Xi_Int_GetRMS(pDataParams , (uint8_t *)abCh, &fTemp);
    }

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* Scale must be done before position because otherwise the position could be out of range it will also be adjusted when scale is changed */
        /* restore timebase setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, qwTimeBase));
        /* restore timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, qwTimeBasePosition));
    }

    /* restore trigger mode */
    PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, wTriggerMode));

    /* Check the status */
    PH_CHECK_SUCCESS(statusTmp);

    /* assign value */
    *pdfRMS = (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_SetCorrRange(
                                        phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                        uint8_t  bChannel
                                        )
{
    phStatus_t statusTmp, statusRestore;
    uint8_t abCh[3] = "C1";
    uint16_t wTriggerMode;
    int64_t qwTimeBase = 0;
    int64_t qwTimeBasePosition = 0;

    if (bChannel >= 1 && bChannel <= 4)
    {
        abCh[1] = (uint8_t)'0' + bChannel;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* get current trigger setting */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetTriggerMode(pDataParams, &wTriggerMode));

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* get current timebase setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, &qwTimeBase));
        /* get current timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwTimeBasePosition));

        /* set measurement settings */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, 0));
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, 2000));
    }

    /* get in correct range */
    statusTmp = phdlOsci_LW64Xi_Int_SetCorrRange(pDataParams, (uint8_t *) abCh);

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_TIME) == PHDL_OSCI_MEASUREMENT_ADJUST_TIME)
    {
        /* Scale must be done before position because otherwise the position could be out of range it will also be adjusted when scale is changed */
        /* restore timebase setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS, qwTimeBase));
        /* restore timebase position setting */
        PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, qwTimeBasePosition));
    }

    /* restore trigger mode */
    PH_CHECK_SUCCESS_FCT(statusRestore, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, wTriggerMode));

    /* return the status */
    return statusTmp;
}

phStatus_t phdlOsci_LW64Xi_GetWaveForm(
                                       phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       uint32_t dwWaveFormBufferSize,
                                       uint32_t * pdwWaveFormLength,
                                       uint8_t * pWaveFormBuffer,
                                       uint8_t * pbHeaderOffset
                                       )
{
    uint8_t abCh[3] = "C1";

    if ((bChannel & (uint8_t)(~PHDL_OSCI_LW64XI_SELECT_FUNC_CHANNEL)) >= 1 && (bChannel & (uint8_t)(~PHDL_OSCI_LW64XI_SELECT_FUNC_CHANNEL)) <= 4)
    {
        abCh[1] = (uint8_t)'0' + (uint8_t)(bChannel & (uint8_t)(~PHDL_OSCI_LW64XI_SELECT_FUNC_CHANNEL));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }
    if ( (bChannel & PHDL_OSCI_LW64XI_SELECT_FUNC_CHANNEL) )
    {
        abCh[0] = (uint8_t)'F';
    }

    /* get waveform */
    return phdlOsci_LW64Xi_Int_GetWaveForm(pDataParams, (uint8_t *)abCh, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
}

phStatus_t phdlOsci_LW64Xi_InitOsci(
                                    phdlOsci_LW64Xi_DataParams_t * pDataParams
                                    )
{
    phStatus_t statusTmp;
    uint16_t wValue=0;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* To solve the problem with the FRTR in single mode after switching from Stoped (first time takes long) a timeout of at least 10seconds is needed */
    wValue = max(wValue, 10000);

    /* set IO timeout to 20 seconds to allow all the init calibration stuff */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 20000));

    /* set comm header to OFF */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"CHDR OFF\n", 9));

    /* clear all status bits */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"*CLS\n", 5));

    /* configure global */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConfigGlobal(pDataParams));

    /* set max samples range */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetMaxSamplesRange(pDataParams));

    /* restore IO timeout */
    return phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue);
}

#endif /* NXPBUILD__PHDL_OSCI_LW64Xi */
