/*
 * Copyright 2017, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Oscilloscope PicoScope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000

#include <phdlOsci.h>
#include "phdlOsci_PicoScope6000.h"
#include "phdlOsci_PicoScope6000_Int.h"
#include "../phdlOsci_Int.h"


phStatus_t phdlOsci_PicoScope6000_Init(
                                       phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                       uint16_t wSizeOfDataParams,
                                       phbalReg_Ps6000Usb_DataParams_t * pBalRegDataParams
                                       )
{
    if (sizeof(phdlOsci_PicoScope6000_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* PicoScope Channels are used as index and therefore have to start with zero */
    #pragma warning(suppress: 4127)
    if(PS6000_CHANNEL_A != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    /* init private data */
    pDataParams->wId                 = PH_COMP_DL_OSCI | PHDL_OSCI_PICOSCOPE6000_ID;
    pDataParams->pBalRegDataParams   = pBalRegDataParams;
    pDataParams->wAdditionalInfo     = 0;
    /* default value for cetecom antenna eg. RMS = FieldStrength * 0.318(318/1000) */
    pDataParams->wFieldStrengthMultiplier = 318;

    pDataParams->wTriggerAutoTriggerMilliseconds = 20;

    pDataParams->wAverageFact = 1;
    pDataParams->wRangeMin = 60;
    pDataParams->wRangeMax = 72;
    pDataParams->bMeasurementOptions = PHDL_OSCI_MEASUREMENT_ADJUST_TIME | PHDL_OSCI_MEASUREMENT_ADJUST_RANGE;

    return phdlOsci_PicoScope6000_Int_SetResetOsciConfig(pDataParams);
}

phStatus_t phdlOsci_PicoScope6000_SetConfig(
                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                            uint16_t wIdentifier,
                                            uint16_t wValue
                                            )
{
    return phdlOsci_PicoScope6000_SetConfig64(pDataParams, wIdentifier, wValue);
}


phStatus_t phdlOsci_PicoScope6000_GetConfig(
                                     phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t * pwValue
                                     )
{
    phStatus_t statusTmp = 0;
    int64_t qwTempValue = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig64(pDataParams, wIdentifier, &qwTempValue));
    if (qwTempValue < 0 || qwTempValue > 65535)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *pwValue = (uint16_t)qwTempValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_GetConfig64(
                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                            uint16_t wIdentifier,
                                            int64_t * pqValue
                                            )
{
    phStatus_t statusTmp;
    uint8_t bTmpValue;
    float64_t dfSamplingRate;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
        *pqValue = (uint16_t)pDataParams->wCurrentTriggerMode;
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        *pqValue = pDataParams->wFieldStrengthMultiplier;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        *pqValue = pDataParams->wAverageFact;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        *pqValue = pDataParams->wRangeMin;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        *pqValue = pDataParams->wRangeMax;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        *pqValue = pDataParams->bMeasurementOptions;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        /* Convert index to channel */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertIndexToChannel(pDataParams, pDataParams->bSelectedChannelIndex, &bTmpValue));
        *pqValue = bTmpValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        if (pDataParams->bChannelBWLimit[pDataParams->bSelectedChannelIndex] == PS6000_BW_FULL)
        {
            *pqValue = PH_OFF;
        }
        else
        {
            *pqValue = PH_ON;
        }
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        *pqValue = pDataParams->bChannelCoupling[pDataParams->bSelectedChannelIndex];
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        *pqValue = (pDataParams->bChannelEnabled[pDataParams->bSelectedChannelIndex] == PHDL_OSCI_PICOSCOPE6000_CHANNEL_ENABLED) ? PH_ON : PH_OFF;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        *pqValue = pDataParams->bChannelImpedance[pDataParams->bSelectedChannelIndex];
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        *pqValue = pDataParams->bChannelInvert[pDataParams->bSelectedChannelIndex];
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        *pqValue = pDataParams->wChannelProbeGain[pDataParams->bSelectedChannelIndex];
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        *pqValue = pDataParams->bTriggerType;
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        /* Convert index to channel */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertIndexToChannel(pDataParams, pDataParams->bTriggerSourceIndex,  &bTmpValue));
        *pqValue = bTmpValue;
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        *pqValue = pDataParams->bTriggerEdgeSlope;
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        *pqValue = PHDL_OSCI_TIMEBASE_MODE_MAIN;
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        phdlOsci_PicoScope6000_Int_ConvertRangeIndexToRange(
            pDataParams->bChannelRangeIndex[pDataParams->bSelectedChannelIndex],
            pDataParams->wChannelProbeGain[pDataParams->bSelectedChannelIndex],
            pqValue);
        *pqValue = (*pqValue)/8;
        break;

    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        phdlOsci_PicoScope6000_Int_ConvertRangeIndexToRange(
            pDataParams->bChannelRangeIndex[pDataParams->bSelectedChannelIndex],
            pDataParams->wChannelProbeGain[pDataParams->bSelectedChannelIndex],
            pqValue);
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        *pqValue = pDataParams->qwTriggerLevelmV;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        *pqValue = pDataParams->qwTimebasePositionNs;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        *pqValue = (int64_t)(0.5 + (pDataParams->qwTimebaseRangeNs / 10));
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        *pqValue = pDataParams->qwTimebaseRangeNs;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetCurrentTimingValues(pDataParams,
                                                            NULL,/*uint32_t *pdwTimebase,*/
                                                            NULL,/*float64_t *pdfOrigSampleRate,*/
                                                            NULL,/*int64_t *pqwOrigNoOfSamples,*/
                                                            NULL,/*float64_t *pdfOutputSampleRate,*/
                                                            pqValue,/*int64_t *pqwOutputNoOfSamples,*/
                                                            NULL/*uint32_t *pdwDownSampleRatio*/
                                                            ));
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
        *pqValue = (int64_t)(0.5 + (pDataParams->qwTimebaseRangeNs / 1000));
        break;

    case PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetCurrentTimingValues(pDataParams,
                                                            NULL,/*uint32_t *pdwTimebase,*/
                                                            NULL,/*float64_t *pdfOrigSampleRate,*/
                                                            NULL,/*int64_t *pqwOrigNoOfSamples,*/
                                                            &dfSamplingRate,/*float64_t *pdfOutputSampleRate,*/
                                                            NULL,/*int64_t *pqwOutputNoOfSamples,*/
                                                            NULL/*uint32_t *pdwDownSampleRatio*/
                                                            ));
        *pqValue = (int64_t)(dfSamplingRate + 0.5);
        break;

    case PHDL_OSCI_CONFIG_NUMBER_CHANNELS:
        *pqValue = pDataParams->bNumberOfChannels;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_SetConfig64(
                                     phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     int64_t qwValue
                                     )
{
    phStatus_t statusTmp;
    uint8_t bTmp;
    int64_t qwCurrentConfigValue = 0;
    uint16_t wVariant = 0;
    uint32_t dwTmp = 0;

    switch (wIdentifier)
    {
    case PHDL_OSCI_CONFIG_TRIGGER:
        pDataParams->wCurrentTriggerMode = (uint16_t)qwValue;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetTriggerMode(pDataParams));
        break;

    case PHDL_OSCI_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConfigOsci(pDataParams, (uint8_t)qwValue));
        break;

    case PHDL_OSCI_CONFIG_FIELD_MULTIPLIER:
        pDataParams->wFieldStrengthMultiplier = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_AVERAGE_FACT:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wAverageFact = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MIN:
        if (qwValue < 5 ||
            qwValue > 70 ||
            qwValue >= pDataParams->wRangeMax)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMin = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_RANGE_MAX:
        if (qwValue < 10 ||
            qwValue > 75 ||
            qwValue <= pDataParams->wRangeMin)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wRangeMax = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS:
        if ((qwValue & ~PHDL_OSCI_MEASUREMENT_MASK) != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bMeasurementOptions = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_ACTIVE:
        if (qwValue < PHDL_OSCI_CHANNEL_1 ||
            qwValue > PHDL_OSCI_CHANNEL_4 ||
            qwValue - PHDL_OSCI_CHANNEL_1 >= pDataParams->bNumberOfChannels)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        /* Convert channel to index */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, (uint8_t)qwValue, &bTmp));
        pDataParams->bSelectedChannelIndex = bTmp;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        wVariant = phdlOsci_PicoScope6000_Int_GetOsciVariant(pDataParams);
        if (qwValue == PH_OFF)
        {
            pDataParams->bChannelBWLimit[pDataParams->bSelectedChannelIndex] = PS6000_BW_FULL;
        }
        else if (wVariant == 6402 || wVariant == 6403)
        {
            pDataParams->bChannelBWLimit[pDataParams->bSelectedChannelIndex] = PS6000_BW_20MHZ;
        }
        else if (wVariant == 6404)
        {
            pDataParams->bChannelBWLimit[pDataParams->bSelectedChannelIndex] = PS6000_BW_25MHZ;
        }
        else
        {
            /* PS 6407 does not support BW Limit */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(pDataParams, pDataParams->bSelectedChannelIndex));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_COUPLING:
        if (qwValue != PHDL_OSCI_COUPLING_AC && qwValue != PHDL_OSCI_COUPLING_DC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        wVariant = phdlOsci_PicoScope6000_Int_GetOsciVariant(pDataParams);
        /* PS 6407 only support DC 50R */
        if (wVariant == 6407 && qwValue != PHDL_OSCI_COUPLING_DC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* If 50R is selected only DC is supported */
        if (pDataParams->bChannelImpedance[pDataParams->bSelectedChannelIndex] == PHDL_OSCI_IMPEDANCE_FIFTY &&
            qwValue != PHDL_OSCI_COUPLING_DC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        pDataParams->bChannelCoupling[pDataParams->bSelectedChannelIndex] = (uint8_t)qwValue;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(pDataParams, pDataParams->bSelectedChannelIndex));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_DISPLAY:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }
        pDataParams->bChannelEnabled[pDataParams->bSelectedChannelIndex] =
            (qwValue == PH_ON) ? PHDL_OSCI_PICOSCOPE6000_CHANNEL_ENABLED : PHDL_OSCI_PICOSCOPE6000_CHANNEL_DISABLED;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(pDataParams, pDataParams->bSelectedChannelIndex));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE:
        if (qwValue != PHDL_OSCI_IMPEDANCE_FIFTY && qwValue != PHDL_OSCI_IMPEDANCE_ONE_MEGA)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* First check if already set */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig64(pDataParams, wIdentifier, &qwCurrentConfigValue));
        if (qwCurrentConfigValue == qwValue)
        {
            break;
        }

        wVariant = phdlOsci_PicoScope6000_Int_GetOsciVariant(pDataParams);
        /* PS 6407 only support DC 50R */
        if (wVariant == 6407 && qwValue != PHDL_OSCI_IMPEDANCE_FIFTY)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* If AC is selected only 1M is supported */
        if (pDataParams->bChannelCoupling[pDataParams->bSelectedChannelIndex] == PHDL_OSCI_COUPLING_AC &&
            qwValue != PHDL_OSCI_IMPEDANCE_ONE_MEGA)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }

        pDataParams->bChannelImpedance[pDataParams->bSelectedChannelIndex] = (uint8_t)qwValue;

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(pDataParams, pDataParams->bSelectedChannelIndex));
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_INVERT:
        if (qwValue != PH_ON && qwValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bChannelInvert[pDataParams->bSelectedChannelIndex] = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_PROBE:
        if (qwValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->wChannelProbeGain[pDataParams->bSelectedChannelIndex] = (uint16_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_CHANNEL_UNITS:
    case PHDL_OSCI_CONFIG_CHANNEL_VERNIER:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);

    case PHDL_OSCI_CONFIG_TRIGGER_TYPE:
        if (qwValue != PHDL_OSCI_TRIGGER_TYPE_EDGE)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bTriggerType = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_SOURCE:
        if (qwValue > PHDL_OSCI_CHANNEL_EXTERNAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        /* Convert to Channel index */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, (uint8_t)qwValue, &bTmp));
        pDataParams->bTriggerSourceIndex = bTmp;
        break;

    case PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE:
        if (qwValue != PHDL_OSCI_EDGE_SLOPE_POSITIVE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_NEGATIVE &&
            qwValue != PHDL_OSCI_EDGE_SLOPE_EITHER)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->bTriggerEdgeSlope = (uint8_t)qwValue;
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_MODE:
        if (qwValue != PHDL_OSCI_TIMEBASE_MODE_MAIN)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        break;

    case PHDL_OSCI_CONFIG_TIMEBASE_VERNIER:
    case PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);

    case PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV:
        qwValue = 8 * qwValue;
    case PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV:
        /* Calc the osci range (divide by probe) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertRangeToRangeIndex(
            pDataParams,
            qwValue,
            pDataParams->wChannelProbeGain[pDataParams->bSelectedChannelIndex],
            &bTmp));

        pDataParams->bChannelRangeIndex[pDataParams->bSelectedChannelIndex] = bTmp;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(pDataParams, pDataParams->bSelectedChannelIndex));
        break;

    case PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV:
        /* Get Range and check if value is in range */
        phdlOsci_PicoScope6000_Int_ConvertRangeIndexToRange(
            pDataParams->bChannelRangeIndex[pDataParams->bTriggerSourceIndex],
            pDataParams->wChannelProbeGain[pDataParams->bTriggerSourceIndex],
            &qwCurrentConfigValue);
        if (qwValue < -qwCurrentConfigValue/2 ||
            qwValue > qwCurrentConfigValue/2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->qwTriggerLevelmV = qwValue;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS:
        /* Timebase position is possible from -10dif up to large values */
        if (qwValue < -pDataParams->qwTimebaseRangeNs)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcMinTimebase(pDataParams, pDataParams->qwTimebaseRangeNs, qwValue, &dwTmp));
        pDataParams->qwTimebasePositionNs = qwValue;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS:
        qwValue = qwValue * 10;
    case PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcMinTimebase(pDataParams, qwValue, pDataParams->qwTimebasePositionNs, &dwTmp));
        pDataParams->qwTimebaseRangeNs = qwValue;
        break;
    case PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcMinTimebase(pDataParams, qwValue * 1000, pDataParams->qwTimebasePositionNs, &dwTmp));
        pDataParams->qwTimebaseRangeNs = qwValue * 1000;
        break;

    case PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES:
        /* Check value range of MaxSamplePoints, 0 or -1 means not used */
        if(qwValue > pDataParams->wMaxMemoryMS * 1000000 || qwValue < -1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
        pDataParams->qwSamplePointsTarget = qwValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_PicoScope6000_GetFieldStrength(
                                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                   uint8_t bChannel,
                                                   uint16_t * pwFieldStrength
                                                   )
{
    phStatus_t statusTmp;
    float32_t fTemp;
    uint8_t bChannelIndex = 0;

    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));
    /* Enable channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, PH_ON));

    /* measure full scope and evaluate amplitude value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetRMS(pDataParams, &fTemp, bChannelIndex));

    fTemp = (float32_t)((fTemp * (float32_t) PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR * (float32_t) PHDL_OSCI_FIELD_STRENGTH_DIVISOR )/ (float32_t) pDataParams->wFieldStrengthMultiplier);

    *pwFieldStrength = (uint16_t)(fTemp + 0.5);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_GetAmpl(
                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwAmplitude
                                   )
{
    phStatus_t statusTmp;
    uint8_t bChannelIndex = 0;

    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));
    /* Enable channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, PH_ON));

    *pwAmplitude = 0;

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_GetPeakPeak(
                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfPkPk
                                   )
{
    phStatus_t statusTmp;
    uint8_t bChannelIndex = 0;

    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));
    /* Enable channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, PH_ON));

    *pdfPkPk = 0;

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_OSCI);
}


phStatus_t phdlOsci_PicoScope6000_GetRMS(
                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   uint16_t * pwRMS
                                   )
{
    phStatus_t statusTmp;
    float64_t dfValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetRMSDouble(pDataParams, bChannel, &dfValue));

    *pwRMS = (uint16_t)(dfValue + 0.5);

    return statusTmp;
}

phStatus_t phdlOsci_PicoScope6000_GetRMSDouble(
                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                   uint8_t  bChannel,
                                   float64_t * pdfRMS
                                   )
{
    phStatus_t statusTmp;
    float32_t fTemp;
    uint8_t bChannelIndex = 0;

    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));
    /* Enable channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, PH_ON));

    /* get the amplitude of channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetRMS(pDataParams , &fTemp, bChannelIndex));

    /* assign value */
    *pdfRMS = (fTemp * PHDL_OSCI_VOLTAGE_DIVISOR);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_SetCorrRange(
                                        phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                        uint8_t  bChannel
                                        )
{
    phStatus_t statusTmp;
    uint8_t bChannelIndex = 0;

    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));
    /* Enable channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, PH_ON));

    /* get in correct range */
    return phdlOsci_PicoScope6000_Int_SetCorrRange(pDataParams, bChannelIndex);
}

phStatus_t phdlOsci_PicoScope6000_GetWaveForm(
                                       phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                       uint8_t  bChannel,
                                       uint32_t dwWaveFormBufferSize,
                                       uint32_t * pdwWaveFormLength,
                                       uint8_t * pbWaveFormBuffer,
                                       uint8_t * pbHeaderOffset
                                       )
{
    phStatus_t statusTmp;
    int16_t overflow = 0;
    uint8_t  *pbData = &pbWaveFormBuffer[PICOSCOPE6000_HEADER_OFFSET];  /* Data starts at end of header */
    uint8_t bChannelIndex = 0;
    uint16_t wTemp = 0;
    int64_t qwOutputNoOfSample;
    uint32_t dwDownSampleRatio = 1;
    uint32_t dwDownSampleRatioMaxSamples = 1; /* The downsamling ratio that should be used accorting to the set max samples target */
    float64_t downSampleRatio = 1.0;
    int32_t dwDownSampleRatioMode = PS6000_RATIO_MODE_NONE;


    /* Convert channel to index */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(pDataParams, bChannel, &bChannelIndex));

    /* Check if channel is enabled */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_SetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_ACTIVE, bChannel));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_GetConfig(pDataParams, PHDL_OSCI_CONFIG_CHANNEL_DISPLAY, &wTemp));
    if (wTemp == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_OSCI);
    }

    /* check for enough memory to save all data */
    if (dwWaveFormBufferSize < PICOSCOPE6000_HEADER_OFFSET)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *pbHeaderOffset = PICOSCOPE6000_HEADER_OFFSET;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetCurrentTimingValues(pDataParams,
                                                            NULL,/*uint32_t *pdwTimebase,*/
                                                            NULL,/*float64_t *pdfOrigSampleRate,*/
                                                            NULL,/*int64_t *pqwOrigNoOfSamples,*/
                                                            NULL,/*float64_t *pdfOutputSampleRate,*/
                                                            &qwOutputNoOfSample,/*int64_t *pqwOutputNoOfSamples,*/
                                                            &dwDownSampleRatioMaxSamples/*uint32_t *pdwDownSampleRatio*/
                                                            ));
    if (dwDownSampleRatioMaxSamples > 1)
    {
        dwDownSampleRatioMode = PS6000_RATIO_MODE_AVERAGE;
    }

    /* If more samples than buffer use downsampling */
    if (qwOutputNoOfSample > (dwWaveFormBufferSize - *pbHeaderOffset) / 2)
    {
        downSampleRatio = (float64_t)(qwOutputNoOfSample) / (float64_t)((dwWaveFormBufferSize - *pbHeaderOffset) / 2);
        if (downSampleRatio == (float64_t)(int32_t)downSampleRatio)
        {
            dwDownSampleRatio = (int32_t)downSampleRatio;
        }
        else
        {
            dwDownSampleRatio = 1 + (int32_t)downSampleRatio;
        }
        dwDownSampleRatioMode = PS6000_RATIO_MODE_AVERAGE;
    }

    /* We use the max downsampling ratio */
    dwDownSampleRatio = max(dwDownSampleRatio, dwDownSampleRatioMaxSamples);

    /* measure the waveform */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetWaveForm(pDataParams,
        (dwWaveFormBufferSize - *pbHeaderOffset) / 2,
        (uint32_t)qwOutputNoOfSample,
        pdwWaveFormLength,
        (int16_t *)pbData, NULL,
        &overflow,
        0,/* Timebase is not used if just capturing pDataParams->dwTimebase, */
        dwDownSampleRatio,
        dwDownSampleRatioMode,
        0,
        bChannelIndex));

    /* create header of byte stream */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CreateHeader(pDataParams, (int8_t*)pbWaveFormBuffer, *pdwWaveFormLength, bChannelIndex, dwDownSampleRatio));

    *pdwWaveFormLength = (*pdwWaveFormLength * 2) + PICOSCOPE6000_HEADER_OFFSET;

    /* check for overflow in GetWaveForm */
    if(overflow != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_PicoScope6000_InitOsci(
                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                           )
{
    phStatus_t statusTmp;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetResetOsciConfig(pDataParams));

    /* Try to get info about the picoscope osci */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_IdentifyOscilloscope(pDataParams));

    /* configure global */
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ApplyChannelSettings(pDataParams));
    return phdlOsci_PicoScope6000_Int_ConfigGlobal(pDataParams);
}

#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */
