/*
 * Copyright 2017 - 2018, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Oscilloscope PicoScope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000

#include "phdlOsci_PicoScope6000.h"
#include "phdlOsci_PicoScope6000_Int.h"
#include "../phdlOsci_Int.h"

unsigned short inputRanges [PS6000_MAX_RANGES] = {  10,
                                                    20,
                                                    50,
                                                    100,
                                                    200,
                                                    500,
                                                    1000,
                                                    2000,
                                                    5000,
                                                    10000,
                                                    20000,
                                                    50000};

/* macros for conversion between adc counts and millivolts */
#define __MV_TO_ADC(mv, chInputRange) (((mv) * PS6000_MAX_VALUE)/ (inputRanges[(chInputRange)]))
#define __ADC_TO_MV(raw, chInputRange)(((raw) * inputRanges[(chInputRange)]) / PS6000_MAX_VALUE)

phStatus_t phdlOsci_PicoScope6000_Int_IdentifyOscilloscope(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           )
{
    phStatus_t statusTmp;
    int8_t pRxBuf[256];
    int16_t wRxBufLen = 256;
    int16_t pwRequiredSize;
    uint32_t dwPicoInfo = 3;
    uint8_t bStrFound;

    /* set device type */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_GetUnitInfo)(pDataParams->pBalRegDataParams->pDeviceHandle, pRxBuf, wRxBufLen, &pwRequiredSize, dwPicoInfo)));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6402C", (uint8_t)5, &bStrFound));
    /* If this osci was not found as "6402C" search only as "6402" */
    if (!bStrFound)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6402", (uint8_t)4, &bStrFound));
    }
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 250;
        pDataParams->wMaxMemoryMS = 256;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6402D", (uint8_t)5, &bStrFound));
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 250;
        pDataParams->wMaxMemoryMS = 512;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_ON;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6403C", (uint8_t)5, &bStrFound));
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 350;
        pDataParams->wMaxMemoryMS = 512;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6403D", (uint8_t)5, &bStrFound));
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 350;
        pDataParams->wMaxMemoryMS = 1024;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_ON;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6404C", (uint8_t)5, &bStrFound));
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 500;
        pDataParams->wMaxMemoryMS = 1024;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)pRxBuf, (int8_t *)"6404D", (uint8_t)5, &bStrFound));
    if (bStrFound)
    {
        pDataParams->bNumberOfChannels = 4;
        pDataParams->wBandwidthMHz = 500;
        pDataParams->wMaxMemoryMS = 2048;
        pDataParams->wSamplingRateMSs = 5000;
        pDataParams->bAWG = PH_ON;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_SetResetOsciConfig(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           )
{
    uint8_t bActChannel;
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;

    pDataParams->bNumberOfChannels = 0;
    pDataParams->wBandwidthMHz = 0;
    pDataParams->wMaxMemoryMS = 0;
    pDataParams->wSamplingRateMSs = 0;
    pDataParams->bAWG = PH_OFF;
    pDataParams->bSelectedChannelIndex = PS6000_CHANNEL_A;

    /* Deactivate all channels by default */
    for (bActChannel = 0; bActChannel < PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS; bActChannel++)
    {
        pDataParams->bChannelEnabled[bActChannel] = PHDL_OSCI_PICOSCOPE6000_CHANNEL_DISABLED;
        pDataParams->bChannelCoupling[bActChannel] = PHDL_OSCI_COUPLING_DC;
        pDataParams->bChannelImpedance[bActChannel] = PHDL_OSCI_IMPEDANCE_ONE_MEGA;
        pDataParams->bChannelBWLimit[bActChannel] = PS6000_BW_FULL;
        pDataParams->bChannelInvert[bActChannel] = PH_OFF;
        pDataParams->wChannelProbeGain[bActChannel] = 10;
        pDataParams->bChannelRangeIndex[bActChannel] = PS6000_1V; /* IMPORTANT: don't save inputRanges[dwRangeCh1] because driver expects index of enum! */
    }
    pDataParams->bChannelEnabled[pDataParams->bSelectedChannelIndex] = PHDL_OSCI_PICOSCOPE6000_CHANNEL_ENABLED;

    pDataParams->qwTimebaseRangeNs = 100000;
    pDataParams->qwTimebasePositionNs = 0;
    pDataParams->qwSamplePointsTarget = -1;

    pDataParams->bTriggerType = PHDL_OSCI_TRIGGER_TYPE_EDGE;
    pDataParams->bTriggerSourceIndex = PS6000_CHANNEL_A;
    pDataParams->bTriggerEdgeSlope = PHDL_OSCI_EDGE_SLOPE_EITHER;
    pDataParams->qwTriggerLevelmV = 0;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_GetOscilloscopeNameAndType(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint8_t * pbName,
                                                           uint16_t* pwNameLen,
                                                           uint8_t * pbType,
                                                           uint16_t* pwTypeLen
                                                           )
{
    phStatus_t statusTmp;
    int8_t pRxBuf[256];
    uint16_t wRxBufLen = 256;
    int16_t pwRequiredSize;
    uint32_t dwPicoInfo = 3;

    /* set device name */
    if (*pwNameLen < 9)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    memcpy(pbName, "PicoScope", 9);
    *pwNameLen = 9;

    /* set device type */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_GetUnitInfo)(pDataParams->pBalRegDataParams->pDeviceHandle, pRxBuf, wRxBufLen, &pwRequiredSize, dwPicoInfo)));
    if (pwRequiredSize > *pwTypeLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    memcpy(pbType, pRxBuf, pwRequiredSize);
    *pwTypeLen = pwRequiredSize;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_GetTriggerTimeOffset(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint64_t * pqwTrigTimeOffsetInFemtoSec
                                                           )
{
    phStatus_t statusTmp;
    uint32_t timeUnitsFemtoSecSelection;
    uint32_t segmentIndex = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_GetTriggerTimeOffset64)(pDataParams->pBalRegDataParams->pDeviceHandle, pqwTrigTimeOffsetInFemtoSec, &timeUnitsFemtoSecSelection, segmentIndex)));

    switch (timeUnitsFemtoSecSelection)
    {
    case 0: /* FemtoSec */
        /* DO NOTHING */
        break;

    case 1: /* PikoSec */
        *pqwTrigTimeOffsetInFemtoSec *= 1000;
        break;

    case 2: /* NanoSec */
        *pqwTrigTimeOffsetInFemtoSec *= 1000000;
        break;

    case 3: /* MicroSec */
        *pqwTrigTimeOffsetInFemtoSec *= 1000000000;
        break;

    case 4: /* MilliSec */
        *pqwTrigTimeOffsetInFemtoSec *= 1000000000000;
        break;

    case 5: /* Sec */
        *pqwTrigTimeOffsetInFemtoSec *= 1000000000000000;
        break;

    default:
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_CreateHeader(
                                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                   int8_t * pbWaveFormBuffer,
                                                   uint32_t dwNumberSamples,
                                                   uint8_t bChannelIndex,
                                                   uint32_t dwDownSampleRatio
                                                   )
{
    phStatus_t statusTmp;
    int8_t  *pbName = pbWaveFormBuffer;       /* 19 bytes */
    int8_t  *pbHeaderVersion = &pbWaveFormBuffer[19];  /* 1 byte */
    int8_t  *pbType = &pbWaveFormBuffer[20];  /* 10 bytes */
    uint64_t *pqwTrigTimeOffsetInFemtoSec = (uint64_t*)&pbWaveFormBuffer[30];
    float32_t *pdwTimebaseInMicroSec = (float32_t*)&pbWaveFormBuffer[38];
    float32_t *pdwTimeIntervalNanoseconds = (float32_t*)&pbWaveFormBuffer[42];
    float32_t *pdwSampleMultFactor = (float32_t*)&pbWaveFormBuffer[46];
    uint32_t *pdwNoOfSamples = (uint32_t*)&pbWaveFormBuffer[50];
    uint16_t *wRangeCh = (uint16_t*)&pbWaveFormBuffer[54];
    int64_t *pqwTimebasePositionNs = (int64_t*)&pbWaveFormBuffer[56];
    uint64_t *pqwTimebaseRangeNs = (uint64_t*)&pbWaveFormBuffer[64];
    uint16_t wNameLen = 18;/* last char is always '\0' */
    uint16_t wTypeLen = 9; /* last char is always '\0' */
    uint8_t  idx;

    /* get device name and type */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetOscilloscopeNameAndType(pDataParams, (uint8_t*)pbName, &wNameLen, (uint8_t*)pbType, &wTypeLen));

    /* fill name array with '\0' */
    for (idx = 20; idx > wNameLen; idx--)
    {
        pbName[idx-1] = (uint8_t)'\0';
    }
    /* fill type array with '\0' */
    for (idx = 10; idx > wTypeLen; idx--)
    {
        pbType[idx-1] = (uint8_t)'\0';
    }

    /* Version 0: Without version tag and without pqwTimebasePositionNs Header Size: 56 */
    /* Version 1: Add version tag + pqwTimebasePositionNs + pqwTimebaseRangeNs, Header size: 72 */
    *pbHeaderVersion = 1;

    /* get trigger time offset */
    /* This is the time between the trigger occured and the first post trigger sample.
     * The value will be zero for the highest sampling rate and can be zero or non zero for lower sampling rates.
     * See also https://www.picotech.com/support/topic17321.html Mon Jul 13, 2015 12:14 pm last paragraph */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetTriggerTimeOffset(pDataParams, pqwTrigTimeOffsetInFemtoSec));
    *pqwTimebasePositionNs = pDataParams->qwTimebasePositionNs;
    *pqwTimebaseRangeNs = pDataParams->qwTimebaseRangeNs;

    if (pDataParams->dwLastSamplingTimebase < 4)
    {
        *pdwTimebaseInMicroSec = (float32_t) pow(2.0F, pDataParams->dwLastSamplingTimebase) / 5000.0F;
    }
    else
    {
        *pdwTimebaseInMicroSec = (float32_t) (pDataParams->dwLastSamplingTimebase-4) / 156.250F;
    }
    *pdwTimebaseInMicroSec = *pdwTimebaseInMicroSec * (float32_t)dwDownSampleRatio;

    *pdwTimeIntervalNanoseconds = 0;/*dwTimeIntervalNanoseconds;*/

    *pdwSampleMultFactor = (float32_t) ((float32_t)inputRanges[(pDataParams->bChannelRangeIndex[bChannelIndex])] / ((float32_t)PS6000_MAX_VALUE)) * (float32_t)(pDataParams->wChannelProbeGain[bChannelIndex]) / (float32_t)(PHDL_OSCI_VOLTAGE_DIVISOR);
    if (pDataParams->bChannelInvert[bChannelIndex] == PH_ON)
    {
        *pdwSampleMultFactor = -(*pdwSampleMultFactor);
    }

    *pdwNoOfSamples = dwNumberSamples;


    /* also save pDataParams->dwRangeCh1 */
    *wRangeCh = inputRanges[pDataParams->bChannelRangeIndex[bChannelIndex]];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

uint16_t phdlOsci_PicoScope6000_Int_GetOsciVariant(phdlOsci_PicoScope6000_DataParams_t * pDataParams)
{
    if (strncmp((char *)pDataParams->pBalRegDataParams->pVariantInfo, "6402", 4) == 0)
    {
        return 6402;
    }
    if (strncmp((char *)pDataParams->pBalRegDataParams->pVariantInfo, "6403", 4) == 0)
    {
        return 6403;
    }
    if (strncmp((char *)pDataParams->pBalRegDataParams->pVariantInfo, "6404", 4) == 0)
    {
        return 6404;
    }
    if (strncmp((char *)pDataParams->pBalRegDataParams->pVariantInfo, "6407", 4) == 0)
    {
        return 6407;
    }
    return 0;
}

phStatus_t phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint8_t bChannel,
                                                            uint8_t * pbChannelIndex)
{
    if (bChannel < 1 || bChannel > pDataParams->bNumberOfChannels)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }
    switch(bChannel)
    {
    case PHDL_OSCI_CHANNEL_1:
        {
            *pbChannelIndex = PS6000_CHANNEL_A;
            break;
        }
    case PHDL_OSCI_CHANNEL_2:
        {
            *pbChannelIndex = PS6000_CHANNEL_B;
            break;
        }
    case PHDL_OSCI_CHANNEL_3:
        {
            *pbChannelIndex = PS6000_CHANNEL_C;
            break;
        }
    case PHDL_OSCI_CHANNEL_4:
        {
            *pbChannelIndex = PS6000_CHANNEL_D;
            break;
        }
    case PHDL_OSCI_CHANNEL_EXTERNAL:
        {
            *pbChannelIndex = PS6000_EXTERNAL;
            break;
        }

    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConvertIndexToChannel(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint8_t bChannelIndex,
                                                            uint8_t * pbChannel)
{
    if (pDataParams);

    switch(bChannelIndex)
    {
    case PS6000_CHANNEL_A:
        {
            *pbChannel = PHDL_OSCI_CHANNEL_1;
            break;
        }
    case PS6000_CHANNEL_B:
        {
            *pbChannel = PHDL_OSCI_CHANNEL_2;
            break;
        }
    case PS6000_CHANNEL_C:
        {
            *pbChannel = PHDL_OSCI_CHANNEL_3;
            break;
        }
    case PS6000_CHANNEL_D:
        {
            *pbChannel = PHDL_OSCI_CHANNEL_4;
            break;
        }
    case PS6000_EXTERNAL:
        {
            *pbChannel = PHDL_OSCI_CHANNEL_EXTERNAL;
            break;
        }

    default:
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConvertRangeToRangeIndex(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            int64_t qwRange,
                                                            uint16_t wProbeGain,
                                                            uint8_t *pbRangeIndex)
{
    uint8_t bMinPossibleRangeIndex = PS6000_50MV;
    uint8_t bMaxPossibleRangeIndex = PS6000_20V;
    int16_t wVariant;

    wVariant = phdlOsci_PicoScope6000_Int_GetOsciVariant(pDataParams);
    if (wVariant == 6402 || wVariant == 6403 || wVariant == 6404)
    {
        bMinPossibleRangeIndex = PS6000_50MV;
        if (pDataParams->bChannelImpedance[pDataParams->bSelectedChannelIndex] == PHDL_OSCI_IMPEDANCE_FIFTY &&
            pDataParams->bChannelCoupling[pDataParams->bSelectedChannelIndex] == PHDL_OSCI_COUPLING_DC)
        {
            bMaxPossibleRangeIndex = PS6000_5V;
        }
        else
        {
            bMaxPossibleRangeIndex = PS6000_20V;
        }
    }

    if (qwRange <= 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    for (*pbRangeIndex = bMinPossibleRangeIndex; *pbRangeIndex <= bMaxPossibleRangeIndex; (*pbRangeIndex)++)
    {
        /* The range is +/- the pico range so *2 */
        if (inputRanges[*pbRangeIndex] * 2 * wProbeGain >= qwRange)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConvertRangeIndexToRange(
                                                            uint8_t bRangeIndex,
                                                            uint16_t wProbeGain,
                                                            int64_t *pqwRange)
{
    *pqwRange = inputRanges[bRangeIndex] * 2 * wProbeGain;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_CalcMinTimebase(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            int64_t qwTimeRangeNs,
                                                            int64_t qwTimebasePositionNs,
                                                            uint32_t *pbTimebase)
{
    uint8_t bChannelsEnabled = 0;
    uint8_t bActChannel = 0;
    float64_t minSampleInterval = 0.0;
    float64_t calcSampleInterval = 0.0;
    float64_t calcTimeBase = 0.0;
    int64_t qwSamplingTimeNs = qwTimeRangeNs;

    if (qwTimeRangeNs <= 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* If we move out to the right side with the timebase position we need more pretrigger samples */
    if (pDataParams->qwTimebasePositionNs < -pDataParams->qwTimebaseRangeNs/2)
    {
        /* Sampling time is the time half time of the range + the delay */
        qwSamplingTimeNs = pDataParams->qwTimebaseRangeNs/2 - qwTimebasePositionNs;
    }

    for (bActChannel = 0; bActChannel < PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS; bActChannel++)
    {
        if (pDataParams->bChannelEnabled[bActChannel] == PH_ON)
        {
            bChannelsEnabled++;
        }
    }

    /* min Sample interval is the sampling time divided by memory that is available for one channel */
    minSampleInterval = ((float64_t)qwSamplingTimeNs / 1000000000) /
        (((float64_t)pDataParams->wMaxMemoryMS * 1000000) / (float64_t)bChannelsEnabled);

    /* Check min timebase if more channels are enabled */
    if (bChannelsEnabled == 1)
    {
        *pbTimebase = 0;
        calcSampleInterval = 1.0 / 5000000000.0;
    }
    else if (
        (pDataParams->bChannelEnabled[PS6000_CHANNEL_A] == PH_ON && pDataParams->bChannelEnabled[PS6000_CHANNEL_B] == PH_ON) ||
        (pDataParams->bChannelEnabled[PS6000_CHANNEL_C] == PH_ON && pDataParams->bChannelEnabled[PS6000_CHANNEL_D] == PH_ON)
        )
    {
        *pbTimebase = 2;
        calcSampleInterval = 4.0 / 5000000000.0;
    }
    else
    {
        *pbTimebase = 1;
        calcSampleInterval = 2.0 / 5000000000.0;
    }

    for (; (*pbTimebase) < 5; (*pbTimebase)++)
    {
        if (minSampleInterval <= calcSampleInterval)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        }
        calcSampleInterval = calcSampleInterval * 2.0;
    }

    calcTimeBase = (minSampleInterval * 156250000.0) + 4.0;
    if (calcTimeBase > 0xFFFFFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* Always round up */
    if (calcTimeBase == (float64_t)(uint32_t)calcTimeBase)
    {
        *pbTimebase = (uint32_t)calcTimeBase;
    }
    else
    {
        *pbTimebase = (uint32_t)calcTimeBase + 1;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_CalcTimingValues( phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                        uint32_t *pdwTimebase,
                                                        int32_t *pdwNoOfPreTriggerSamples,
                                                        int32_t *pdwNoOfPostTriggerSamples,
                                                        uint32_t *pdwTriggerDelay)
{
    phStatus_t statusTmp;
    float64_t sampleIntervalNs;
    uint32_t dwTimebase;
    uint32_t dwDummy; /* Dummy storage that is used of NULL is supplied as paramter */

    if (pdwTimebase == NULL)
    {
        pdwTimebase = &dwDummy;
    }
    if (pdwNoOfPreTriggerSamples == NULL)
    {
        pdwNoOfPreTriggerSamples = (int32_t *)&dwDummy;
    }
    if (pdwNoOfPostTriggerSamples == NULL)
    {
        pdwNoOfPostTriggerSamples = (int32_t *)&dwDummy;
    }
    if (pdwTriggerDelay == NULL)
    {
        pdwTriggerDelay = &dwDummy;
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcMinTimebase(pDataParams, pDataParams->qwTimebaseRangeNs, pDataParams->qwTimebasePositionNs, &dwTimebase));

    switch(dwTimebase)
    {
    case 0: sampleIntervalNs = 0.2; break;
    case 1: sampleIntervalNs = 0.4; break;
    case 2: sampleIntervalNs = 0.8; break;
    case 3: sampleIntervalNs = 1.6; break;
    case 4: sampleIntervalNs = 3.2; break;
    default:
        sampleIntervalNs = (float64_t)(dwTimebase - 4)/0.156250000; break;
    }

    *pdwTimebase = dwTimebase;

    /* If we move out to the left side with the timebase position we did not need pretrigger samples */
    if (pDataParams->qwTimebasePositionNs > pDataParams->qwTimebaseRangeNs/2)
    {
        *pdwNoOfPreTriggerSamples = 0;
        /* Full range post trigger samples */
        *pdwNoOfPostTriggerSamples = (uint32_t)((((float64_t)(pDataParams->qwTimebaseRangeNs)) / sampleIntervalNs) + 0.5);
        /* Delay after the trigger and the window left side */
        *pdwTriggerDelay = (uint32_t)((((float64_t)(pDataParams->qwTimebasePositionNs - pDataParams->qwTimebaseRangeNs/2)) / sampleIntervalNs) + 0.5);
    }
    /* If we move out to the right side with the timebase position we did not need posttrigger samples but more pretrigger */
    /* Post trigger samples will be negative to indicate that not all samples are used */
    else if (pDataParams->qwTimebasePositionNs < -pDataParams->qwTimebaseRangeNs/2)
    {
        /* More pre trigger samples are used */
        *pdwNoOfPreTriggerSamples = (uint32_t)((((float64_t)(pDataParams->qwTimebaseRangeNs/2 - pDataParams->qwTimebasePositionNs)) / sampleIntervalNs) + 0.5);
        /* Post trigger sample are negative because some samples has to be removed at the end */
        *pdwNoOfPostTriggerSamples = (uint32_t)((((float64_t)(pDataParams->qwTimebaseRangeNs/2 + pDataParams->qwTimebasePositionNs)) / sampleIntervalNs) + 0.5);
        *pdwTriggerDelay = 0;
    }
    else /* We are within the window range */
    {
        *pdwNoOfPreTriggerSamples = (uint32_t)((((float64_t)(pDataParams->qwTimebaseRangeNs/2 - pDataParams->qwTimebasePositionNs)) / sampleIntervalNs) + 0.5);
        *pdwNoOfPostTriggerSamples = (uint32_t)((((float64_t)(pDataParams->qwTimebaseRangeNs/2 + pDataParams->qwTimebasePositionNs)) / sampleIntervalNs) + 0.5);
        *pdwTriggerDelay = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_GetCurrentTimingValues( phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                        uint32_t *pdwTimebase,
                                                        float64_t *pdfOrigSampleRate,
                                                        int64_t *pqwOrigNoOfSamples,
                                                        float64_t *pdfOutputSampleRate,
                                                        int64_t *pqwOutputNoOfSamples,
                                                        uint32_t *pdwDownSampleRatio)
{
    phStatus_t statusTmp;

    uint32_t dwTimebase;
    float64_t dfOrigSampleRate;
    int64_t qwOrigNoOfSamples;
    float64_t dfOutputSampleRate;
    int64_t qwOutputNoOfSamples;
    uint32_t dwDownSampleRatio;
    int32_t dwNoOfPreTriggerSamples, dwNoOfPostTriggerSamples;
    float64_t sampleInterval;

    if (pdwTimebase == NULL)
    {
        pdwTimebase = &dwTimebase;
    }
    if (pdfOrigSampleRate == NULL)
    {
        pdfOrigSampleRate = &dfOrigSampleRate;
    }
    if (pqwOrigNoOfSamples == NULL)
    {
        pqwOrigNoOfSamples = &qwOrigNoOfSamples;
    }
    if (pdfOutputSampleRate == NULL)
    {
        pdfOutputSampleRate = &dfOutputSampleRate;
    }
    if (pqwOutputNoOfSamples == NULL)
    {
        pqwOutputNoOfSamples = &qwOutputNoOfSamples;
    }
    if (pdwDownSampleRatio == NULL)
    {
        pdwDownSampleRatio = &dwDownSampleRatio;
    }

    /* Calc the number of samples that would be captured */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcTimingValues(pDataParams, &dwTimebase, &dwNoOfPreTriggerSamples, &dwNoOfPostTriggerSamples, NULL));

    /* Check if the sample count is positiv */
    if (dwNoOfPreTriggerSamples + dwNoOfPostTriggerSamples < 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    switch(dwTimebase)
    {
    case 0: sampleInterval = 0.2e-9; break;
    case 1: sampleInterval = 0.4e-9; break;
    case 2: sampleInterval = 0.8e-9; break;
    case 3: sampleInterval = 1.6e-9; break;
    case 4: sampleInterval = 3.2e-9; break;
    default:
        sampleInterval = (float64_t)(dwTimebase - 4)/0.156250000e9; break;
    }

    *pqwOrigNoOfSamples = dwNoOfPreTriggerSamples + dwNoOfPostTriggerSamples;
    *pdfOrigSampleRate = 1.0 / sampleInterval;

    /* if target is not used just return the number of samples, qwSamplePointsTarget 0 or -1 means not used */
    if (pDataParams->qwSamplePointsTarget <= 0)
    {
        *pdwDownSampleRatio = 1;
        *pqwOutputNoOfSamples = *pqwOrigNoOfSamples;
        *pdfOutputSampleRate = *pdfOrigSampleRate;
    }
    else if (*pqwOrigNoOfSamples <= pDataParams->qwSamplePointsTarget)
    {   /* calculated samples already fit */
        *pdwDownSampleRatio = 1;
        *pqwOutputNoOfSamples = *pqwOrigNoOfSamples;
        *pdfOutputSampleRate = *pdfOrigSampleRate;
    }
    else
    {
        /* more samples than targed -> use downsampling */
        /* Always round down downSampleRatio because we should have at least the targed number of points */
        *pdwDownSampleRatio = (uint32_t)((float64_t)(*pqwOrigNoOfSamples) / (float64_t)(pDataParams->qwSamplePointsTarget));
        /* Round up number samples */
        *pqwOutputNoOfSamples = (*pqwOrigNoOfSamples + *pdwDownSampleRatio - 1) / *pdwDownSampleRatio;
        *pdfOutputSampleRate = *pdfOrigSampleRate / *pdwDownSampleRatio;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_SetChannel(
                                                    phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                    uint8_t bChannelIndex
                                                 )
{
    phStatus_t statusTmp;
    uint32_t dwCoupling = 0;

    if (pDataParams->bChannelCoupling[bChannelIndex] == PHDL_OSCI_COUPLING_AC && pDataParams->bChannelImpedance[bChannelIndex] == PHDL_OSCI_IMPEDANCE_FIFTY)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    if (pDataParams->bChannelCoupling[bChannelIndex] == PHDL_OSCI_COUPLING_AC)
    {
        dwCoupling = PS6000_AC;
    }
    else
    {
        if (pDataParams->bChannelImpedance[bChannelIndex] == PHDL_OSCI_IMPEDANCE_ONE_MEGA)
        {
            dwCoupling = PS6000_DC_1M;
        }
        else
        {
            dwCoupling = PS6000_DC_50R;
        }
    }

    /* Try to set the channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)
        pDataParams->pBalRegDataParams->pfPs6000Usb_SetChannel(pDataParams->pBalRegDataParams->pDeviceHandle,
        (uint32_t)bChannelIndex,
        (int16_t)pDataParams->bChannelEnabled[bChannelIndex],
        dwCoupling,
        (uint32_t)pDataParams->bChannelRangeIndex[bChannelIndex],
        (int32_t)0, (uint32_t)pDataParams->bChannelBWLimit[bChannelIndex])));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_SetTriggerMode(
                                                     phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                    )
{
    phStatus_t statusTmp;
    int16_t wThreshold;
    uint16_t wThresholdDirection;
    uint32_t dwTimebase;
    int32_t dwNoOfPreTriggerSamples;
    int32_t dwNoOfPostTriggerSamples;
    uint32_t dwTriggerDelay;

    wThreshold = __MV_TO_ADC(
        (int16_t)(pDataParams->qwTriggerLevelmV/pDataParams->wChannelProbeGain[pDataParams->bTriggerSourceIndex]),
        pDataParams->bChannelRangeIndex[pDataParams->bTriggerSourceIndex]);

    switch(pDataParams->bTriggerEdgeSlope)
    {
    case PHDL_OSCI_EDGE_SLOPE_POSITIVE:
        wThresholdDirection = PS6000_RISING;
        break;
    case PHDL_OSCI_EDGE_SLOPE_NEGATIVE:
        wThresholdDirection = PS6000_FALLING;
        break;
    case PHDL_OSCI_EDGE_SLOPE_EITHER:
        wThresholdDirection = PS6000_RISING_OR_FALLING;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_CalcTimingValues(pDataParams, &dwTimebase, &dwNoOfPreTriggerSamples, &dwNoOfPostTriggerSamples, &dwTriggerDelay));
    if (dwNoOfPostTriggerSamples < 0)
    {
        dwNoOfPostTriggerSamples = 0;
    }

    /* For each channel: PS6000_TRIGGER_CHANNEL_PROPERTIES trigProps[pDataParams->bNumberOfChannels]; */
    /*struct tPS6000TriggerChannelProperties trigProps;
    struct tPS6000TriggerConditions trigConds;
    struct tPS6000TriggerDirections trigDirections;

    trigProps.thresholdUpper = __MV_TO_ADC(pDataParams->wTriggerThresholdUpperMillivolts, pDataParams->bChannelRangeIndex[PHDL_OSCI_CHANNEL_1]);
    trigProps.thresholdLower = __MV_TO_ADC(pDataParams->wTriggerThresholdLowerMillivolts, pDataParams->bChannelRangeIndex[PHDL_OSCI_CHANNEL_1]);
    switch(PHDL_OSCI_CHANNEL_1)
    {
        case 0:
            trigProps.channel = PS6000_CHANNEL_A;
            trigConds.channelA = PS6000_CONDITION_TRUE;
            trigConds.channelB = PS6000_CONDITION_DONT_CARE;
            trigConds.channelC = PS6000_CONDITION_DONT_CARE;
            trigConds.channelD = PS6000_CONDITION_DONT_CARE;
            trigDirections.channelA = PS6000_FALLING;
            trigDirections.channelB = PS6000_NONE;
            trigDirections.channelC = PS6000_NONE;
            trigDirections.channelD = PS6000_NONE;
            break;
        case 1:
            trigProps.channel = PS6000_CHANNEL_B;
            trigConds.channelA = PS6000_CONDITION_DONT_CARE;
            trigConds.channelB = PS6000_CONDITION_TRUE;
            trigConds.channelC = PS6000_CONDITION_DONT_CARE;
            trigConds.channelD = PS6000_CONDITION_DONT_CARE;
            trigDirections.channelA = PS6000_NONE;
            trigDirections.channelB = PS6000_RISING;
            trigDirections.channelC = PS6000_NONE;
            trigDirections.channelD = PS6000_NONE;
            break;
        case 2:
            trigProps.channel = PS6000_CHANNEL_C;
            trigConds.channelA = PS6000_CONDITION_DONT_CARE;
            trigConds.channelB = PS6000_CONDITION_DONT_CARE;
            trigConds.channelC = PS6000_CONDITION_TRUE;
            trigConds.channelD = PS6000_CONDITION_DONT_CARE;
            trigDirections.channelA = PS6000_NONE;
            trigDirections.channelB = PS6000_NONE;
            trigDirections.channelC = PS6000_RISING;
            trigDirections.channelD = PS6000_NONE;
            break;
        case 3:
            trigProps.channel = PS6000_CHANNEL_D;
            trigConds.channelA = PS6000_CONDITION_DONT_CARE;
            trigConds.channelB = PS6000_CONDITION_DONT_CARE;
            trigConds.channelC = PS6000_CONDITION_DONT_CARE;
            trigConds.channelD = PS6000_CONDITION_TRUE;
            trigDirections.channelA = PS6000_NONE;
            trigDirections.channelB = PS6000_NONE;
            trigDirections.channelC = PS6000_NONE;
            trigDirections.channelD = PS6000_RISING;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    trigProps.hysteresisUpper = 0U;
    trigProps.hysteresisLower = 0U;
    trigProps.thresholdMode = PS6000_LEVEL;

    trigConds.external = PS6000_CONDITION_DONT_CARE;
    trigConds.aux      = PS6000_CONDITION_DONT_CARE;
    trigConds.pulseWidthQualifier = PS6000_CONDITION_DONT_CARE;*/

    switch (pDataParams->wCurrentTriggerMode)
    {
    case PHDL_OSCI_TRIGGER_AUTO:
        /* set trigger channel properties */
        /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelProperties)(pDataParams->pBalRegDataParams->pDeviceHandle, (uint32_t*)&trigProps, 1, 0, pDataParams->dwTriggerAutoTriggerMilliseconds)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelConditions)(pDataParams->pBalRegDataParams->pDeviceHandle, (uint32_t*)&trigConds, 1)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelDirections)(pDataParams->pBalRegDataParams->pDeviceHandle, trigDirections.channelA, trigDirections.channelB, trigDirections.channelC, trigDirections.channelD, PS6000_NONE, PS6000_NONE)));*/
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetSimpleTrigger)(
            pDataParams->pBalRegDataParams->pDeviceHandle,
            1,
            pDataParams->bTriggerSourceIndex,
            wThreshold,
            wThresholdDirection,
            dwTriggerDelay,
            pDataParams->wTriggerAutoTriggerMilliseconds)));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_RunBlock)(
            pDataParams->pBalRegDataParams->pDeviceHandle,
            dwNoOfPreTriggerSamples,
            dwNoOfPostTriggerSamples,
            dwTimebase,
            1,
            NULL, 0, 0, 0)));

        pDataParams->dwLastSamplingTimebase = dwTimebase;
        break;

    case PHDL_OSCI_TRIGGER_NORMAL:
        /* FIXXME: NOT SUPPORTED YET */
        /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelProperties)(pDataParams->pBalRegDataParams->pDeviceHandle, (uint32_t*)&trigProps, 1, 1, 0)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelConditions)(pDataParams->pBalRegDataParams->pDeviceHandle, (uint32_t*)&trigConds, 1)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelDirections)(pDataParams->pBalRegDataParams->pDeviceHandle, trigDirections.channelA, trigDirections.channelB, trigDirections.channelC, trigDirections.channelD, PS6000_NONE, PS6000_NONE)));*/
        break;

    case PHDL_OSCI_TRIGGER_SINGLE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
            (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetSimpleTrigger)(
                pDataParams->pBalRegDataParams->pDeviceHandle,
                1,
                pDataParams->bTriggerSourceIndex,
                wThreshold,
                wThresholdDirection,
                dwTriggerDelay,
                0
            )));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
            (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_RunBlock)(
                pDataParams->pBalRegDataParams->pDeviceHandle,
                dwNoOfPreTriggerSamples,
                dwNoOfPostTriggerSamples,
                dwTimebase,
                1,
                NULL, 0, 0, 0
            )));
        phTools_Sleep(50);
        pDataParams->dwLastSamplingTimebase = dwTimebase;
        break;

    case PHDL_OSCI_TRIGGER_STOP:
        /* FIXXME: NOT SUPPORTED YET */
        /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelProperties)(pDataParams->pBalRegDataParams->pDeviceHandle, 0, 0, 0, 0)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelConditions)(pDataParams->pBalRegDataParams->pDeviceHandle, (uint32_t*)&trigConds, 0)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetTriggerChannelDirections)(pDataParams->pBalRegDataParams->pDeviceHandle, PS6000_NONE, PS6000_NONE, PS6000_NONE, PS6000_NONE, PS6000_NONE, PS6000_NONE)));*/
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConfigGlobal(
    phdlOsci_PicoScope6000_DataParams_t * pDataParams
    )
{
    /* TODO: adapt settings as wished */
    /* global config */
    pDataParams->wTriggerAutoTriggerMilliseconds = 20;
    pDataParams->wCurrentTriggerMode = PHDL_OSCI_TRIGGER_AUTO;
    pDataParams->qwTriggerLevelmV = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConfigOsci(
    phdlOsci_PicoScope6000_DataParams_t * pDataParams,
    uint8_t bConfig
    )
{
    phStatus_t statusTmp;

    /* set parameters for configuration */
    switch (bConfig)
    {
    case PHDL_OSCI_MODE_GLOBAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConfigGlobal(pDataParams));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* return error code */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ConvertError(
                                 phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                 int16_t status)
{
    switch (status)
    {
    case PH_ERR_SUCCESS:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

        /*
    case 43:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

    case 12:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

    case 22:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

    case 41:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        */

    default:
        pDataParams->wAdditionalInfo = (uint16_t) status;
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_PICOSCOPE_DLL, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_PicoScope6000_Int_GetWaveForm (
                                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                   uint32_t dwWaveFormBufferSize,
                                                   uint32_t dwNumberOfSamplesRequired,
                                                   uint32_t * pdwWaveFormLength,
                                                   int16_t * pWaveFormBufferMax,
                                                   int16_t * pWaveFormBufferMin,
                                                   int16_t * pwOverflow,
                                                   uint32_t dwTimebase,
                                                   uint32_t dwDownSampleRatio,
                                                   int32_t dwDownSampleRatioMode,
                                                   uint8_t bStartCollection,
                                                   uint8_t bChannelIndex
                                                   )
{
    long timeIndisposed;
    phStatus_t statusTmp;
    uint32_t pollingCounter = 0;
    int16_t ready = 0;
    int16_t oversample = 1;
    uint32_t dwNumberRawSamples = dwWaveFormBufferSize*dwDownSampleRatio;
    uint8_t bActualChannel;

    /* Set Data buffer only for active channel */
    for (bActualChannel = 0; bActualChannel < pDataParams->bNumberOfChannels; bActualChannel++)
    {
        if (bChannelIndex == bActualChannel)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
                (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetDataBuffers)(pDataParams->pBalRegDataParams->pDeviceHandle, bActualChannel, pWaveFormBufferMax, pWaveFormBufferMin, dwWaveFormBufferSize, dwDownSampleRatioMode)));
        }
        else {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
                (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetDataBuffers)(pDataParams->pBalRegDataParams->pDeviceHandle, bActualChannel, NULL, NULL, 0, dwDownSampleRatioMode)));
        }
    }

    if (bStartCollection == 1)
    {
        /* Start it collecting, then wait for completion*/
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
            (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_RunBlock)(pDataParams->pBalRegDataParams->pDeviceHandle, 0, dwNumberRawSamples, dwTimebase, oversample, (int32_t *) &timeIndisposed, 0, 0, 0)));

        pDataParams->dwLastSamplingTimebase = dwTimebase;

        do
        {
            /* we do not use the CallBackBlock function since we don't want to have void* parameters */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
                (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_IsReady)(pDataParams->pBalRegDataParams->pDeviceHandle, &ready)));
            phTools_Sleep(10);
            pollingCounter++;
        } while (!ready && pollingCounter < 100); /* while status == 0 pico scope is collecting */
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
            (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_IsReady)(pDataParams->pBalRegDataParams->pDeviceHandle, &ready)));

        if (!ready)
        {
            phTools_Sleep(100);
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
                (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_IsReady)(pDataParams->pBalRegDataParams->pDeviceHandle, &ready)));
        }
    }

    /* if pico scope still not finished capturing, the data is invalid */
    if (!ready)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_TRIGGER_NOT_FIRED, PH_COMP_DL_OSCI);
    }

    /* transfer block of data from osci to our buffer and check for overflow at current range settings */
     PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams,
        (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_GetValues)(pDataParams->pBalRegDataParams->pDeviceHandle, 0, &dwNumberOfSamplesRequired, dwDownSampleRatio, dwDownSampleRatioMode, (uint32_t)0, pwOverflow)));

    *pdwWaveFormLength = dwNumberOfSamplesRequired;

    /* ALWAYS STOP capturing to free device for next call! */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_Stop)(pDataParams->pBalRegDataParams->pDeviceHandle)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_SetCorrRange(
                                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                   uint8_t bChannelIndex
                                                  )
{
    phStatus_t statusTmp;
    int16_t overflow = 1;
    uint8_t bUnterflow = 0;
    int16_t wMinValue = 0;
    int16_t wMaxValue = 0;
    uint32_t dwWaveFormLength;
    float64_t percentageOfMaxScale = 0.0;
    float64_t percentageOfMinScale = 0.0;
    uint32_t dwOldRangeChannel = 0;
    uint8_t bMinPossibleRangeIndex = PS6000_50MV;
    uint8_t bMaxPossibleRangeIndex = PS6000_20V;
    int16_t wVariant;

    wVariant = phdlOsci_PicoScope6000_Int_GetOsciVariant(pDataParams);
    if (wVariant == 6402 || wVariant == 6403 || wVariant == 6404)
    {
        bMinPossibleRangeIndex = PS6000_50MV;
        if (pDataParams->bChannelImpedance[bChannelIndex] == PHDL_OSCI_IMPEDANCE_FIFTY &&
            pDataParams->bChannelCoupling[bChannelIndex] == PHDL_OSCI_COUPLING_DC)
        {
            bMaxPossibleRangeIndex = PS6000_5V;
        }
        else
        {
            bMaxPossibleRangeIndex = PS6000_20V;
        }
    }
    else /* 6407 only support 100mV */
    {
        bMinPossibleRangeIndex = bMaxPossibleRangeIndex = PS6000_100MV;
        return phdlOsci_PicoScope6000_Int_SetChannel(
            pDataParams,
            bChannelIndex);
    }

    /* Run auto */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_ConvertError(pDataParams, (int16_t)(pDataParams->pBalRegDataParams->pfPs6000Usb_SetSimpleTrigger)(
            pDataParams->pBalRegDataParams->pDeviceHandle,
            1,
            bChannelIndex,
            0,
            PHDL_OSCI_EDGE_SLOPE_EITHER,
            0,
            20)));

    /* Scale until into limit */
    do
    {
        /* set scale to oszi */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(
            pDataParams,
            bChannelIndex));

        overflow = 0;
        bUnterflow = 0;

        /* Get Min/Max (just one point) but sample 50000 points */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetWaveForm (
                                                       pDataParams,
                                                       1,/*dwWaveFormBufferSize,*/
                                                       1,
                                                       &dwWaveFormLength,
                                                       (int16_t *)&wMaxValue,
                                                       (int16_t *)&wMinValue,
                                                       &overflow,
                                                       2, /* Timebase for meassureing is 2 -> 1.25GS/s */
                                                       50000,
                                                       /*PS6000_RATIO_MODE*/ PS6000_RATIO_MODE_AGGREGATE,
                                                       1,
                                                       bChannelIndex
                                                       ));

        percentageOfMaxScale = (double)max(wMaxValue, -wMinValue) / 32512.0 * (80.0 / (double)pDataParams->wRangeMax);
        percentageOfMinScale = (double)max(wMaxValue, -wMinValue) / 32512.0 * (80.0 / (double)pDataParams->wRangeMin);

        if (overflow != 0 || percentageOfMaxScale > 1.0) /* Overflow handling */
        {
            /* Already at the higest range */
            if (pDataParams->bChannelRangeIndex[bChannelIndex] == bMaxPossibleRangeIndex)
            {
                return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
            }

            pDataParams->bChannelRangeIndex[bChannelIndex]++;
        }
        else /* Unterflow handling */
        {
            dwOldRangeChannel = pDataParams->bChannelRangeIndex[bChannelIndex];
            if (percentageOfMinScale < 0.1) /* At least a factor of 10 can be used */
            {
                pDataParams->bChannelRangeIndex[bChannelIndex] -= 3;
            }
            else
            {
                if (pDataParams->bChannelRangeIndex[bChannelIndex] % PS6000_100MV == PS6000_10MV) /* Range is 10mV 100mV 1V 10V */
                {
                    if (percentageOfMinScale < 0.2)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 2; /*1 -> 0.2 */
                    }
                    else if (percentageOfMinScale < 0.5)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 1; /*1 -> 0.5 */
                    }
                }
                else if (pDataParams->bChannelRangeIndex[bChannelIndex] % PS6000_100MV == PS6000_20MV) /* Range is 20mV 200mV 2V 20V */
                {
                    if (percentageOfMinScale < 0.25)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 2; /*2 -> 0.5 */
                    }
                    else if (percentageOfMinScale < 0.5)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 1; /*2 -> 1 */
                    }
                }
                else if (pDataParams->bChannelRangeIndex[bChannelIndex] % PS6000_100MV == PS6000_50MV) /* Range is 50mV 500mV 5V 50V */
                {
                    if (percentageOfMinScale < 0.20)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 2; /*5 -> 1 */
                    }
                    else if (percentageOfMinScale < 0.4)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] -= 1; /*5 -> 2 */
                    }
                }
                else
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
                }
            }
            if (dwOldRangeChannel != pDataParams->bChannelRangeIndex[bChannelIndex])
            {
                /* Check if we reach the minimum range -> this is no underflow */
                if (dwOldRangeChannel == bMinPossibleRangeIndex)
                {
                    pDataParams->bChannelRangeIndex[bChannelIndex] = bMinPossibleRangeIndex;
                    bUnterflow = 0;
                }
                else
                {
                    if ((int8_t)pDataParams->bChannelRangeIndex[bChannelIndex] < bMinPossibleRangeIndex)
                    {
                        pDataParams->bChannelRangeIndex[bChannelIndex] = bMinPossibleRangeIndex;
                    }
                    bUnterflow = 1;
                }
            }
        }
    }while(overflow != 0 || percentageOfMaxScale > 1.0 || bUnterflow != 0);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_GetRMS(
                                                    phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                    float32_t *pdfRMS,
                                                    uint8_t bChannelIndex
                                                    )
{
    phStatus_t statusTmp;
    uint32_t dwWaveFormLength = 0;
    int16_t pWaveFormBuffer[BUFFER_SIZE];
    int16_t wOverflow = 0;
    uint16_t wActAverageRepetition;
    uint32_t dwCounter;
    float64_t dfSumValueSquare = .0F;
    float64_t dfRMSValue = .0F;

    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_ADJUST_RANGE) == PHDL_OSCI_MEASUREMENT_ADJUST_RANGE)
    {
        /* set correct range before starting the measurements */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetCorrRange(pDataParams, bChannelIndex));
    }

    /* collect oscilloscope data */
    for (wActAverageRepetition = 0; wActAverageRepetition < pDataParams->wAverageFact; wActAverageRepetition++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_GetWaveForm (
                                                       pDataParams,
                                                       BUFFER_SIZE,
                                                       BUFFER_SIZE,
                                                       &dwWaveFormLength,
                                                       pWaveFormBuffer,
                                                       NULL,
                                                       &wOverflow,
                                                       2, /* Timebase for meassureing is 2 -> 1.25GS/s */
                                                       1, /* Downsampling radio not used */
                                                       PS6000_RATIO_MODE_NONE,
                                                       1,
                                                       bChannelIndex
                                                       ));

        if (wOverflow != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
        }

        /* x_rms = sqrt(1/n * (x_1^2 + x_2^2 + ... + x_n^2)) */
        dfSumValueSquare = 0.0;
        for (dwCounter = 0; dwCounter < dwWaveFormLength; dwCounter++)
        {
            dfSumValueSquare += (double)(pWaveFormBuffer[dwCounter] * pWaveFormBuffer[dwCounter]);
        }
        dfRMSValue += sqrt(dfSumValueSquare / (double)dwWaveFormLength);
    }
    dfRMSValue /= pDataParams->wAverageFact;

    *pdfRMS = (float32_t)(__ADC_TO_MV(dfRMSValue, pDataParams->bChannelRangeIndex[bChannelIndex]) / 1000.0 * (double)pDataParams->wChannelProbeGain[bChannelIndex]);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_PicoScope6000_Int_ApplyChannelSettings(phdlOsci_PicoScope6000_DataParams_t * pDataParams)
{
    phStatus_t statusTmp;
    uint8_t bActualChannel;

    for (bActualChannel = 0; bActualChannel < pDataParams->bNumberOfChannels; bActualChannel++)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_PicoScope6000_Int_SetChannel(
            pDataParams,
            bActualChannel));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */
