/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Oscilloscope PicoScope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLOSCI_PICOSCOPE6000_INT_H
#define PHDLOSCI_PICOSCOPE6000_INT_H

#include <ph_Status.h>
#include <phdlOsci.h>
#include <math.h>

#define PICOSCOPE6000_HEADER_OFFSET 72

#define BUFFER_SIZE 10000
#define UINT32_MAX_VALUE 4294967295
#define PS6000_MAX_VALUE 32512

typedef enum enPS6000Coupling
{
    PS6000_AC,
    PS6000_DC_1M,
    PS6000_DC_50R,
} PS6000_COUPLING;

typedef enum enPS6000BandwidthLimiter
{
    PS6000_BW_FULL,
    PS6000_BW_20MHZ,
    PS6000_BW_25MHZ,
} PS6000_BANDWIDTH_LIMITER;

typedef enum enPS6000ThresholdDirection
{
	PS6000_ABOVE,
	PS6000_BELOW,
	PS6000_RISING,
	PS6000_FALLING,
	PS6000_RISING_OR_FALLING,
	PS6000_ABOVE_LOWER,
	PS6000_BELOW_LOWER,
	PS6000_RISING_LOWER,
	PS6000_FALLING_LOWER,
	/* Windowing using both thresholds */
	PS6000_INSIDE = PS6000_ABOVE,
	PS6000_OUTSIDE = PS6000_BELOW,
	PS6000_ENTER = PS6000_RISING,
	PS6000_EXIT = PS6000_FALLING,
	PS6000_ENTER_OR_EXIT = PS6000_RISING_OR_FALLING,
	PS6000_POSITIVE_RUNT = 9,
	PS6000_NEGATIVE_RUNT,
	/* no trigger set */
	PS6000_NONE = PS6000_RISING
} PS6000_THRESHOLD_DIRECTION;

typedef enum enPS6000Range
{
    PS6000_10MV,
    PS6000_20MV,
    PS6000_50MV,
    PS6000_100MV,
    PS6000_200MV,
    PS6000_500MV,
    PS6000_1V,
    PS6000_2V,
    PS6000_5V,
    PS6000_10V,
    PS6000_20V,
    PS6000_50V,
    PS6000_MAX_RANGES
} PS6000_RANGE;

typedef enum enPS6000Channel
{
    PS6000_CHANNEL_A,
    PS6000_CHANNEL_B,
    PS6000_CHANNEL_C,
    PS6000_CHANNEL_D,
    PS6000_EXTERNAL
}   PS6000_CHANNEL;

typedef enum enPS6000ThresholdMode
{
    PS6000_LEVEL,
    PS6000_WINDOW
} PS6000_THRESHOLD_MODE;

#pragma pack(1)
typedef struct tPS6000TriggerChannelProperties
{
  int16_t                 thresholdUpper;
  uint16_t                hysteresisUpper;
  int16_t                 thresholdLower;
  uint16_t                hysteresisLower;

  PS6000_CHANNEL        channel;
  PS6000_THRESHOLD_MODE thresholdMode;
} PS6000_TRIGGER_CHANNEL_PROPERTIES;
#pragma pack()

typedef enum enPS6000TriggerState
{
  PS6000_CONDITION_DONT_CARE,
  PS6000_CONDITION_TRUE,
  PS6000_CONDITION_FALSE,
  PS6000_CONDITION_MAX
} PS6000_TRIGGER_STATE;

typedef struct tPS6000TriggerConditions
{
  PS6000_TRIGGER_STATE channelA;
  PS6000_TRIGGER_STATE channelB;
  PS6000_TRIGGER_STATE channelC;
  PS6000_TRIGGER_STATE channelD;
  PS6000_TRIGGER_STATE external;
  PS6000_TRIGGER_STATE aux;
  PS6000_TRIGGER_STATE pulseWidthQualifier;
} PS6000_TRIGGER_CONDITIONS;
#pragma pack()

typedef struct tPS6000TriggerDirections
{
  PS6000_THRESHOLD_DIRECTION channelA;
  PS6000_THRESHOLD_DIRECTION channelB;
  PS6000_THRESHOLD_DIRECTION channelC;
  PS6000_THRESHOLD_DIRECTION channelD;
  PS6000_THRESHOLD_DIRECTION external;
  PS6000_THRESHOLD_DIRECTION aux;
} PS6000_TRIGGER_DIRECTIONS;
#pragma pack()

typedef enum enPS6000RatioMode
{
	PS6000_RATIO_MODE_NONE,
	PS6000_RATIO_MODE_AGGREGATE = 1,
	PS6000_RATIO_MODE_AVERAGE = 2,
	PS6000_RATIO_MODE_DECIMATE = 4,
	PS6000_RATIO_MODE_DISTRIBUTION = 8
} PS6000_RATIO_MODE;

phStatus_t phdlOsci_PicoScope6000_Int_IdentifyOscilloscope(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_SetResetOsciConfig(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_GetOscilloscopeNameAndType(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint8_t * pbName,
                                                           uint16_t* pwNameLen,
                                                           uint8_t * pbType,
                                                           uint16_t* pwTypeLen
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_CreateHeader         (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           int8_t * pbWaveFormBuffer,
                                                           uint32_t dwNumberSamples,
                                                           uint8_t bChannelIndex,
                                                           uint32_t dwDownSampleRatio
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_GetTriggerTimeOffset(
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint64_t * pqwTrigTimeOffsetInFemtoSec
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_SetChannel(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint8_t bChannelIndex
                                                 );

uint16_t phdlOsci_PicoScope6000_Int_GetOsciVariant(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                 );

phStatus_t phdlOsci_PicoScope6000_Int_ConvertChannelToIndex(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint8_t bChannel,
                                                            uint8_t * pbChannelIndex);

phStatus_t phdlOsci_PicoScope6000_Int_ConvertIndexToChannel(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint8_t bChannelIndex,
                                                            uint8_t * pbChannel);

phStatus_t phdlOsci_PicoScope6000_Int_ConvertRangeToRangeIndex(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            int64_t qwRange,
                                                            uint16_t wProbeGain,
                                                            uint8_t *pbRangeIndex);

phStatus_t phdlOsci_PicoScope6000_Int_ConvertRangeIndexToRange(
                                                            uint8_t bRangeIndex,
                                                            uint16_t wProbeGain,
                                                            int64_t *pqwRange);

phStatus_t phdlOsci_PicoScope6000_Int_CalcMinTimebase(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            int64_t qwTimeRangeNs,
                                                            int64_t qwTimebasePositionNs,
                                                            uint32_t *pbTimebase);

phStatus_t phdlOsci_PicoScope6000_Int_CalcTimingValues(
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint32_t *pdwTimebase,
                                                            int32_t *pdwNoOfPreTriggerSamples,
                                                            int32_t *pdwNoOfPostTriggerSamples,
                                                            uint32_t *pdwTriggerDelay);

phStatus_t phdlOsci_PicoScope6000_Int_GetCurrentTimingValues( phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            uint32_t *pdwTimebase,
                                                            float64_t *pdfOrigSampleRate,
                                                            int64_t *pqwOrigNoOfSamples,
                                                            float64_t *pdfOutputSampleRate,
                                                            int64_t *pqwOutputNoOfSamples,
                                                            uint32_t *pdwDownSampleRatio);

phStatus_t phdlOsci_PicoScope6000_Int_ConfigGlobal         (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_SetTriggerMode       (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_ConfigOsci           (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint8_t bConfig
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_ConvertError         (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           int16_t status
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_GetWaveForm (
                                                   phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                   uint32_t dwWaveFormBufferSize,
                                                   uint32_t dwNumberOfSamplesRequired,
                                                   uint32_t * pdwWaveFormLength,
                                                   int16_t * pWaveFormBufferMax,
                                                   int16_t * pWaveFormBufferMin,
                                                   int16_t * pwOverflow,
                                                   uint32_t dwTimebase,
                                                   uint32_t dwDownSampleRatio,
                                                   int32_t dwDownSampleRatioMode,
                                                   uint8_t bStartCollection,
                                                   uint8_t bChannelIndex
                                                   );

phStatus_t phdlOsci_PicoScope6000_Int_SetCorrRange         (
                                                           phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                           uint8_t bChannelIndex
                                                           );

phStatus_t phdlOsci_PicoScope6000_Int_GetRMS                (
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams,
                                                            float32_t *pdfRMS,
                                                            uint8_t bChannelIndex
                                                            );

phStatus_t phdlOsci_PicoScope6000_Int_ApplyChannelSettings  (
                                                            phdlOsci_PicoScope6000_DataParams_t * pDataParams
                                                            );

#endif /* PHDLOSCI_PICOSCOPE6000_INT_H */
