/*
 * Copyright 2013, 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Oscilloscope Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phdlOsci.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_OSCI_LW64XI
#include "LW64Xi/phdlOsci_LW64Xi.h"
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
#include "DSO7052A/phdlOsci_DSO7052A.h"
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
#include "PicoScope6000/phdlOsci_PicoScope6000.h"
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
#include "ISO3SoftScope/phdlOsci_ISO3SoftScope.h"
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

#ifdef NXPBUILD__PHDL_OSCI

phStatus_t phdlOsci_InitOsci(
                             void * pDataParams
                             )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_InitOsci");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_InitOsci((phdlOsci_LW64Xi_DataParams_t *)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_InitOsci((phdlOsci_DSO7052A_DataParams_t *)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_InitOsci((phdlOsci_PicoScope6000_DataParams_t *)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_InitOsci((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_SetConfig(
                              void * pDataParams,
                              uint16_t wIdentifier,
                              uint16_t wValue
                              )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_SetConfig");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIdentifier);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wIdentifier_log, &wIdentifier);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValue_log, &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_SetConfig((phdlOsci_LW64Xi_DataParams_t *)pDataParams, wIdentifier, wValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_SetConfig((phdlOsci_DSO7052A_DataParams_t *)pDataParams, wIdentifier, wValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_SetConfig((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, wIdentifier, wValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_SetConfig((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, wIdentifier, wValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetConfig(
                              void * pDataParams,
                              uint16_t wIdentifier,
                              uint16_t * pwValue
                              )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetConfig");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIdentifier);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pwValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wIdentifier_log, &wIdentifier);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetConfig((phdlOsci_LW64Xi_DataParams_t *)pDataParams, wIdentifier, pwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetConfig((phdlOsci_DSO7052A_DataParams_t *)pDataParams, wIdentifier, pwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetConfig((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, wIdentifier, pwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetConfig((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, wIdentifier, pwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pwValue_log, pwValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_SetConfig64(
                              void * pDataParams,
                              uint16_t wIdentifier,
                              int64_t qwValue
                              )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_SetConfig64");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIdentifier);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(qwValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wIdentifier_log, &wIdentifier);
    PH_LOG_HELPER_ADDPARAM_INT64(PH_LOG_LOGTYPE_DEBUG, qwValue_log, &qwValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_SetConfig64((phdlOsci_LW64Xi_DataParams_t *)pDataParams, wIdentifier, qwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_SetConfig64((phdlOsci_DSO7052A_DataParams_t *)pDataParams, wIdentifier, qwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_SetConfig64((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, wIdentifier, qwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_SetConfig64((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, wIdentifier, qwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetConfig64(
                              void * pDataParams,
                              uint16_t wIdentifier,
                              int64_t * pqwValue
                              )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetConfig64");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIdentifier);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pqwValue);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wIdentifier_log, &wIdentifier);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetConfig64((phdlOsci_LW64Xi_DataParams_t *)pDataParams, wIdentifier, pqwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetConfig64((phdlOsci_DSO7052A_DataParams_t *)pDataParams, wIdentifier, pqwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetConfig64((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, wIdentifier, pqwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetConfig64((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, wIdentifier, pqwValue);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_INT64(PH_LOG_LOGTYPE_DEBUG, pqwValue_log, pqwValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetFieldStrength(
                                     void * pDataParams,
                                     uint8_t bChannel,
                                     uint16_t * pwFieldStrength
                                     )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetFieldStrength");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pwFieldStrength);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetFieldStrength((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, pwFieldStrength);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetFieldStrength((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, pwFieldStrength);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetFieldStrength((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, pwFieldStrength);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetFieldStrength((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, pwFieldStrength);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pwFieldStrength_log, pwFieldStrength);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetAmpl(
                            void * pDataParams,
                            uint8_t bChannel,
                            uint16_t * pwAmplitude
                            )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetAmpl");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pwAmplitude);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetAmpl((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, pwAmplitude);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetAmpl((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, pwAmplitude);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetAmpl((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, pwAmplitude);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetAmpl((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, pwAmplitude);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pwAmplitude_log, pwAmplitude);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetPeakPeak(
                            void * pDataParams,
                            uint8_t bChannel,
                            float64_t * pdfPkPk
                            )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetPeakPeak");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pdfPkPk);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetPeakPeak((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, pdfPkPk);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetPeakPeak((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, pdfPkPk);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetPeakPeak((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, pdfPkPk);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetPeakPeak((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, pdfPkPk);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_FLOAT64(PH_LOG_LOGTYPE_DEBUG, pdfPkPk_log, pdfPkPk);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetRMS(
                            void * pDataParams,
                            uint8_t bChannel,
                            uint16_t * pwRMS
                            )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetRMS");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pwRMS);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetRMS((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, pwRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetRMS((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, pwRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetRMS((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, pwRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetRMS((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, pwRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pwRMS_log, pwRMS);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetRMSDouble(
                            void * pDataParams,
                            uint8_t bChannel,
                            float64_t * pdfRMS
                            )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetRMSDouble");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pdfRMS);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetRMSDouble((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, pdfRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetRMSDouble((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, pdfRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetRMSDouble((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, pdfRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetRMSDouble((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, pdfRMS);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_FLOAT64(PH_LOG_LOGTYPE_DEBUG, pdfRMS_log, pdfRMS);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_SetCorrRange(
                                 void * pDataParams,
                                 uint8_t bChannel
                                 )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_SetCorrRangel");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_SetCorrRange((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_SetCorrRange((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_SetCorrRange((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_SetCorrRange((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phdlOsci_GetWaveForm(
                                void * pDataParams,
                                uint8_t bChannel,
                                uint32_t dwWaveFormBufferSize,
                                uint32_t * pdwWaveFormLength,
                                uint8_t * pWaveFormBuffer,
                                uint8_t * pbHeaderOffset
                                )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phdlOsci_GetWaveForm");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bChannel);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(dwWaveFormBufferSize);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pdwWaveFormLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pWaveFormBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pbHeaderOffset);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bChannel_log, &bChannel);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, dwWaveFormBufferSize_log, &dwWaveFormBufferSize);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);
    PH_ASSERT_NULL (pDataParams);

    if (PH_GET_COMPCODE(pDataParams) != PH_COMP_DL_OSCI)
    {
		status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* perform operation on active layer */
    switch (PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHDL_OSCI_LW64XI
    case PHDL_OSCI_LW64XI_ID:
        status = phdlOsci_LW64Xi_GetWaveForm((phdlOsci_LW64Xi_DataParams_t *)pDataParams, bChannel, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
        break;
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI_DSO7052A
    case PHDL_OSCI_DSO7052A_ID:
        status = phdlOsci_DSO7052A_GetWaveForm((phdlOsci_DSO7052A_DataParams_t *)pDataParams, bChannel, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
        break;
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000
    case PHDL_OSCI_PICOSCOPE6000_ID:
        status = phdlOsci_PicoScope6000_GetWaveForm((phdlOsci_PicoScope6000_DataParams_t *)pDataParams, bChannel, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
        break;
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE
    case PHDL_OSCI_ISO3SOFTSCOPE_ID:
        status = phdlOsci_ISO3SoftScope_GetWaveForm((phdlOsci_ISO3SoftScope_DataParams_t *)pDataParams, bChannel, dwWaveFormBufferSize, pdwWaveFormLength, pWaveFormBuffer, pbHeaderOffset);
        break;
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

    default:
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_OSCI);
        break;
    }
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT32(PH_LOG_LOGTYPE_DEBUG, pdwWaveFormLength_log, pdwWaveFormLength);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pWaveFormBuffer_log, pWaveFormBuffer, (uint16_t) dwWaveFormBufferSize);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, pbHeaderOffset_log, pbHeaderOffset);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

#endif /* NXPBUILD__PHDL_OSCI */
