/*
 * Copyright 2013, 2017 - 2018, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Generic Osciloscope Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phTools.h>
#ifdef NXPBUILD__PHBAL_REG_VISA
#include "../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"
#endif /* NXPBUILD__PHBAL_REG_VISA */

#ifdef NXPBUILD__PHDL_OSCI_LW64XI
#include "LW64Xi/phdlOsci_LW64Xi.h"
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */

#ifdef NXPBUILD__PHDL_OSCI
#include "phdlOsci_Int.h"

phStatus_t phdlOsci_Int_Send(
                             phdlOsci_LW64Xi_DataParams_t * pDataParams,
                             uint8_t * pTxBuffer,
                             uint16_t wTxLength
                             )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

#ifdef NXPBUILD__PHBAL_REG_VISA
    //For a VISA interface we use the tranmit function and just wait the timeout
    if (PH_GET_COMPID(pDataParams->pBalRegDataParams) == PHBAL_REG_VISA_ID)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, pTxBuffer, wTxLength));
        phTools_Sleep(100);
        return statusTmp;
    }
#endif /* NXPBUILD__PHBAL_REG_VISA */

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 100));

    /* send data */
    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return phdlOsci_Int_CheckClearError(pDataParams, status);
}

phStatus_t phdlOsci_Int_SendWait(
                                 phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint16_t wWaittime
                                 )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

#ifdef NXPBUILD__PHBAL_REG_VISA
    //For a VISA interface we use the tranmit function and just wait the timeout
    if (PH_GET_COMPID(pDataParams->pBalRegDataParams) == PHBAL_REG_VISA_ID)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, pTxBuffer, wTxLength));
        phTools_Sleep(100);
        return statusTmp;
    }
#endif /* NXPBUILD__PHBAL_REG_VISA */

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /* send data */
    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return phdlOsci_Int_CheckClearError(pDataParams, status);
}

phStatus_t phdlOsci_Int_CheckClearError(
                                 phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                 phStatus_t status
                                 )
{
    phStatus_t statusTmp;
    uint16_t wCMRResp, wEXRResp;
    uint8_t bClearErrorCount;
    uint8_t bComErrorCount;
    uint8_t bRx[10];
    uint16_t wRxLen;

 /* If there was an error try to readout the error counter to clear a error
    but ignore this error */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        bClearErrorCount = 0;
        bComErrorCount = 0;
        do
        {
            /* read command errror register */
            wCMRResp = 1;
            statusTmp = phbalReg_Exchange(
                pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                (uint8_t *)"CMR?\n",
                5,
                10,
                (uint8_t*)bRx,
                &wRxLen);

            if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bComErrorCount++;
            }
            else
            {
                /* convert response */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui16((int8_t*)bRx, &wCMRResp));
            }

            /* read command execute register */
            wEXRResp = 1;
            statusTmp = phbalReg_Exchange(
                pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                (uint8_t*)"EXR?\n",
                5,
                10,
                (uint8_t*)bRx,
                &wRxLen);

            if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bComErrorCount++;
            }
            else
            {
                /* convert response */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui16((int8_t*)bRx, &wEXRResp));
            }
            bClearErrorCount++;
        } while(bClearErrorCount < 100 && bComErrorCount < 10 && (wCMRResp != 0 || wEXRResp != 0));
    }
    return status;
}

/* helperfunction asci to integer */
phStatus_t phdlOsci_Int_Atoui16(
                                int8_t * pAsci,
                                uint16_t * wURetVal
                                )
{
    phStatus_t statusTmp;
    uint32_t dwTemp = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui32(pAsci, &dwTemp));
    if (dwTemp > 65535)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *wURetVal = (uint16_t)dwTemp;
    return statusTmp;
}

/* helperfunction asci to integer */
phStatus_t phdlOsci_Int_Atoi16(
                               int8_t * pAsci,
                               int16_t * wIRetVal
                               )
{
    phStatus_t statusTmp;
    int32_t dwTemp = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi32(pAsci, &dwTemp));
    if (dwTemp < -32768 || dwTemp > 32767)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *wIRetVal = (int16_t)dwTemp;
    return statusTmp;
}

/* helperfunction asci to integer */
phStatus_t phdlOsci_Int_Atoui32(
                                int8_t * pAsci,
                                uint32_t * dwURetVal
                                )
{
    uint64_t qwTemp=0;

    if ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '+')
    {
        /* increse index, so that plus sign is ignored */
        if (*pAsci == '+')
            pAsci++;

        do
        {
            if (*pAsci != '.' )
            {
                qwTemp *= 10;
                qwTemp += (*pAsci-'0');
            }
            pAsci++;
            if (qwTemp > 4294967295LL)
            {
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
            }
        } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
    }
    *dwURetVal = (uint32_t)qwTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

/* helperfunction asci to integer */
phStatus_t phdlOsci_Int_Atoi32(
                               int8_t * pAsci,
                               int32_t * dwIRetVal
                               )
{
    int64_t qwTemp=0;

    if ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '-' || *pAsci == '+')
    {
        if (*pAsci == '-')
        {
            /* increse index, so that minus sign is ignored */
            pAsci++;
            do
            {
                if (*pAsci != '.')
                {
                    qwTemp *= 10;
                    qwTemp -= (*pAsci-'0');
                }
                pAsci++;
                if (qwTemp < -2147483648LL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
                }
            } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
        else
        {
            /* increse index, so that plus sign is ignored */
            if (*pAsci == '+')
                pAsci++;

            do
            {
                if (*pAsci != '.')
                {
                    qwTemp *= 10;
                    qwTemp += (*pAsci-'0');
                }
                pAsci++;
                if (qwTemp > 2147483647LL)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
                }
            }while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
    }
    *dwIRetVal = (int32_t)qwTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_Int_FindChar(
                                 int8_t * pAsci,
                                 int8_t bRef,
                                 uint8_t * bRetVal
                                 )
{
    uint8_t bCount=0;

    do
    {
        if (*pAsci == bRef)
        {
            break;
        }
        bCount++;
        pAsci++;
    } while (!(*pAsci == 0 || *pAsci == 10|| *pAsci == 13));

    *bRetVal= bCount;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_Int_FindStr(int8_t * pAsci,
                                int8_t * pRef,
                                uint8_t bRefSize,
                                uint8_t * pbStrFound
                                )
{
    uint8_t bI, bRefCount;

    *pbStrFound = 0;
    do
    {
        bRefCount = 0;
        /* check for string */
        for (bI = 0; bI<bRefSize ; bI++)
        {
            if (*(pAsci+bI) == *(pRef+bI))
            {
                bRefCount++;
            }
        }
        if (bRefCount == bRefSize)
        {
            *pbStrFound = 1;
            break;
        }
        /* increase index */
        pAsci++;
    } while (!(*(pAsci + bRefSize -1) == 0 || *(pAsci + bRefSize -1) == 10 || *(pAsci + bRefSize -1) == 13));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t  phdlOsci_Int_Atof(
                              int8_t * pAsci,
                              float32_t * fRetVal
                              )
{
    phStatus_t statusTmp;
    float64_t qwTemp = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof64(pAsci, &qwTemp));
    if (qwTemp < -FLT_MAX || qwTemp > FLT_MAX)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_DL_OSCI);
    }
    *fRetVal = (float32_t)qwTemp;
    return statusTmp;
}

phStatus_t  phdlOsci_Int_Atof64(
                              int8_t * pAsci,
                              float64_t * qwRetVal
                              )
{
    phStatus_t statusTmp;
    float64_t qwTemp = .0;
    uint8_t bDiv =0, bNeg = 0;
    int16_t iNegExp = 0;
    int16_t iExp = 0;
    int16_t iTemp;

    /* check if negative number */
    if (*pAsci == '-')
    {
        bNeg = 1;
        pAsci++;
    }
    /* check if positive number */
    if (*pAsci == '+')
    {
        bNeg = 0;
        pAsci++;
    }
    /* check for space */
    if (*pAsci == ' ')
    {
        pAsci++;
    }

    /* get significand */
    if (*pAsci >= '0' && *pAsci <= '9')
    {
        do
        {
            if (*pAsci != '.')
            {
                qwTemp *= 10;
                qwTemp += (float64_t) (*pAsci-'0');
                /* if decimal increase division */
                if (bDiv)
                {
                    iNegExp++;
                }
            }
            else
            {
                bDiv = 1;
            }
            /* increase pointer */
            pAsci++;
        } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
    }

    /* check for exponent */
    if (*pAsci == 'E' || *pAsci == 'e')
    {
        pAsci++;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoi16((int8_t*)pAsci, &iTemp));
        iExp = (int16_t) iTemp;
    }

    /* calculate exponent */
    iExp = iExp  - iNegExp;

    if (iExp>0)
    {
        while(iExp!=0)
        {
            qwTemp *= 10;
            iExp--;
        }
    }
    else if (iExp < 0)
    {
        while( iExp != 0)
        {
            qwTemp /= 10;
            iExp++;
        }
    }
    else
    {
        /* do nothing */
    }

    /* check for sign */
    if (bNeg)
    {
        qwTemp = (float64_t) -qwTemp;
    }
    *qwRetVal = qwTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

#endif /* NXPBUILD__PHDL_OSCI */
