/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Powersupply E3631A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_POWERSUPPLY_E3631A

#include "phdlPowerSupply_E3631A.h"
#include "phdlPowerSupply_E3631A_Int.h"
#include "../phdlPowerSupply_Int.h"

/* for visa functionality displaced function exchange large */
#include "../../../phbalReg/src/Visa/external/visa.h"
#include "../../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"

phStatus_t phdlPowerSupply_E3631A_Int_ChkLastCmd(
    phdlPowerSupply_E3631A_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[50];
    uint16_t wRxlen;
    int16_t wCmdCode;

    /* read command errror register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)":SYSTem:ERRor?\n", 15, 50, (uint8_t *)bRx, &wRxlen));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_Int_Atoi16((int8_t*)bRx, &wCmdCode));

    /* analyse response */

    if (wCmdCode == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_POWERSUPPLY);
    }
}

phStatus_t phdlPowerSupply_E3631A_Int_WaitExe(
                                       phdlPowerSupply_E3631A_DataParams_t * pDataParams
                                       )
{
    uint8_t bRx[10];
    uint16_t wRxlen;

    /* read operation complete, powersupply only responds if the last operation is completed */
    return phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)"*OPC?", 5, 10, (uint8_t *)bRx, &wRxlen);
}

phStatus_t phdlPowerSupply_E3631A_Int_ChkCmd(
                                      phdlPowerSupply_E3631A_DataParams_t * pDataParams
                                      )
{
    phStatus_t statusTmp;

    /* wait for command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_WaitExe(pDataParams));

    /* check if last command was interpreted correct */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_Int_PowerOn(
                                       phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                       uint8_t  bChannel
                                       )
{
    phStatus_t statusTmp;
    if (pDataParams->bStateOfChannels[bChannel] == PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SetVoltage(pDataParams, pDataParams->wMillivolts[bChannel]));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"OUTP ON\n", 8, 200));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));
    pDataParams->bStateOfChannels[bChannel] = PH_ON;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_Int_PowerOff(
                                       phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                       uint8_t  bChannel
                                       )
{
    phStatus_t statusTmp;
    if (pDataParams->bStateOfChannels[bChannel] == PH_OFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SetVoltage(pDataParams, 0));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"OUTP OFF\n", 9, 200));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));
    pDataParams->bStateOfChannels[bChannel] = PH_OFF;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_Int_SetVoltage(
                                                phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                                uint16_t bMilliVolts
                                                )
{
    phStatus_t statusTmp;
    uint8_t bCmdLength;
    uint8_t	abCmd[40];

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ConvertVoltageInCmd(bMilliVolts, &bCmdLength, (uint8_t *)abCmd));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, abCmd, bCmdLength, 200));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_Int_SetChannel(
                                                phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                                uint8_t bChannel
                                                )
{
    phStatus_t statusTmp;

    if (bChannel == PHDL_POWERSUPPLY_CHANNEL_ONE)
    {
        /* Setting 6volts positive channel for following Configs */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"INST:SEL P6V\n", 13, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));
    }
    else if(bChannel == PHDL_POWERSUPPLY_CHANNEL_TWO)
    {
        /* Setting 25 volts positive channel for following Configs */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"INST:SEL P25V\n", 14, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));
    }
    else if(bChannel == PHDL_POWERSUPPLY_CHANNEL_THREE)
    {
        /* Setting 25 volts negative channel for following Configs */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"INST:SEL N25V\n", 14, 200));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_POWERSUPPLY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_Int_ConvertVoltageInCmd(
    uint16_t wVoltage,
    uint8_t * pCmdLength,
    uint8_t * pCmd
    )
{

    /* prepare common section of command */
    pCmd[0] = 'V';
    pCmd[1] = 'O';
    pCmd[2] = 'L';
    pCmd[3] = 'T';
    pCmd[4] = ' ';

    if (wVoltage < 10000) /* Voltage between 0 and 9.999 Volts*/
    {
        pCmd[10] = '\0';
        pCmd[9] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[8] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[7] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[6] = '.';
        pCmd[5] = (uint8_t)'0' + (uint8_t)(wVoltage);

        *pCmdLength = 11;
    }
    else if ((wVoltage >= 10000) && (wVoltage <= 25000))
    {
        pCmd[11] = '\0';
        pCmd[10] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[9] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[8] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[7] = '.';
        pCmd[6] = (uint8_t)'0' + (uint8_t)(wVoltage%10);
        wVoltage =  wVoltage/10;
        pCmd[5] = (uint8_t)'0' + (uint8_t)(wVoltage);

        *pCmdLength = 12;
    }
    else /* voltage beyond 25 Volts not possible */
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlPowerSupply_E3631A_Int_SendWait(
                                 phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint16_t wWaittime
                                 )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    /*uint8_t bRx;*/
    /*uint16_t wRxLen;*/

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to waittime */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /* send data */
    status = phhalReg_Visa_Cmd_Transmit(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength);
/*    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        0,
        &bRx,
        &wRxLen);*/

    /* restore IO timeout use statusTmp to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
		/* there is a errorcode on stack because no answer is expected */
		status = phdlPowerSupply_E3631A_Int_ChkLastCmd(pDataParams);

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
    }

    return status;
}

phStatus_t phdlPowerSupply_E3631A_Int_ResetDevice(
                                                phdlPowerSupply_E3631A_DataParams_t * pDataParams
                                                )
{
    phStatus_t statusTmp;

    /* Reset E3631A to its standardvalues */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SendWait(pDataParams, (uint8_t *)"*RST\n", 5, 200));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ChkCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

#endif /* NXPBUILD__PHDL_POWERSUPPLY_E3631A */
