/*
 * Copyright 2013 - 2014, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * FPGA Box V6 Device Layer Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phdlRdFpga.h>

#ifdef NXPBUILD__PHDL_RDFPGA_V6

#include "phdlRdFpga_V6.h"
#include "../phdlRdFpga_Int.h"

#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#include "../../../phhalHw/src/RdFpgaV6/external/phdefines.h"
#pragma warning(pop)                    /* PRQA S 3116 */

phStatus_t phdlRdFpga_V6_Init(
                              phdlRdFpga_V6_DataParams_t * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                              uint16_t wSizeOfDataParams,							/**< [In] Specifies the size of the data parameter structure. */
                              phbalReg_RdFpgaV6_DataParams_t * pBalRegDataParams	/**< [In] Pointer to the parameter structure of the balReg component. */
                              )
{
    if (sizeof(phdlRdFpga_V6_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_RDFPGA);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID;
    pDataParams->pBalRegDataParams		= pBalRegDataParams;
	pDataParams->wAdditionalInfo		= 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_AMP);
}

phStatus_t phdlRdFpga_V6_SetPotiPos(
                                    phdlRdFpga_V6_DataParams_t * pDataParams,
                                    uint16_t wPos
                                    )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    /* set poti position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetPotiPos)(wPos)));

    /* wait for poti to be set */
    return phdlRdFpga_Int_Wait(PHDL_RDFPGA_V6_SET_POTI_TIME);
}

phStatus_t phdlRdFpga_V6_SetDefaultPotiPos(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint16_t wPos
    )
{
    /* set default poti position */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetDefaultPotiPos)(wPos));
}

phStatus_t phdlRdFpga_V6_GetPotiPos(
                                    phdlRdFpga_V6_DataParams_t * pDataParams,
                                    uint16_t * pwPos
                                    )
{
    /* get poti position */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetPotiPos)(pwPos));
}

phStatus_t phdlRdFpga_V6_IncPotiPos(
                                    phdlRdFpga_V6_DataParams_t * pDataParams
                                    )
{
    /* increase poti position */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_IncPotiPos)());
}

phStatus_t phdlRdFpga_V6_DecPotiPos(
                                    phdlRdFpga_V6_DataParams_t * pDataParams
                                    )
{
    /* decrease poti position */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_DecPotiPos)());
}

phStatus_t phdlRdFpga_V6_SetTimeOutVal(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint32_t dwTimeOutVal
                                       )
{
    /* set timeout value */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetTimeoutValue)(dwTimeOutVal));
}

phStatus_t phdlRdFpga_V6_GetTimeOutVal(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint32_t * pdwTimeOutVal
                                       )
{
    /* get timeout value */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetTimeoutValue)(pdwTimeOutVal));
}

phStatus_t phdlRdFpga_V6_LoadJoinerIniFile(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint8_t * pFilename
    )
{
    /* load INI file */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LoadJoinerIniFile)(pFilename));
}

phStatus_t phdlRdFpga_V6_LoadPattern(
                                     phdlRdFpga_V6_DataParams_t * pDataParams,
                                     uint8_t * pFilename,
                                     uint8_t bId
                                     )
{
    /* load pattern */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LoadPattern)(pFilename, bId));
}

phStatus_t phdlRdFpga_V6_PlayPattern(
                                     phdlRdFpga_V6_DataParams_t * pDataParams,
                                     uint8_t bId
                                     )
{
    /* play pattern */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_PlayPattern)(bId));
}

phStatus_t phdlRdFpga_V6_PlaySeqA(
                                  phdlRdFpga_V6_DataParams_t * pDataParams
                                  )
{
    /* play sequence A */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_PlaySeqA)());
}

phStatus_t phdlRdFpga_V6_PlaySeqB(
                                  phdlRdFpga_V6_DataParams_t * pDataParams
                                  )
{
    /* play sequence A */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_PlaySeqB)());
}

phStatus_t phdlRdFpga_V6_PlaySinus(
                                   phdlRdFpga_V6_DataParams_t * pDataParams
                                   )
{
    /* play sinus */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_PlaySinus)());
}

phStatus_t phdlRdFpga_V6_SelectTrigger(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint8_t bTriggerSource
                                       )
{
    uint16_t wTriggerSource;

    /* convert trigger variable, cast is not enough.... */
    wTriggerSource = (uint16_t) bTriggerSource;

    /* select trigger */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SelectTrigger)(wTriggerSource));
}

phStatus_t phdlRdFpga_V6_LabelReset(
                                    phdlRdFpga_V6_DataParams_t * pDataParams,
                                    uint16_t wRisingEdgeCnt,
                                    uint32_t dwSinus,
                                    uint32_t dwHfPause
                                    )
{
    /* label reset */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LabelReset)(wRisingEdgeCnt, dwSinus, dwHfPause));
}

phStatus_t phdlRdFpga_V6_LabelResetCounterReset(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint16_t wRisingEdgeCnt,
    uint32_t dwSinus,
    uint32_t dwHfPause
    )
{
    /* label reset counter reset */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LabelResetCounterReset)(wRisingEdgeCnt, dwSinus, dwHfPause));
}

phStatus_t phdlRdFpga_V6_LabelResetCounterResetSingleShot(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint16_t wRisingEdgeCnt,
    uint32_t dwSinus,
    uint32_t dwHfPause
    )
{
    /* label reset counter reset single shot */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LabelResetCounterResetSingleShot)(wRisingEdgeCnt, dwSinus, dwHfPause));
}

phStatus_t phdlRdFpga_V6_LabelResetSingleShot(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint16_t wRisingEdgeCnt,
    uint32_t dwSinus,
    uint32_t dwHfPause
    )
{
    /* label reset single shot */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LabelResetSingleShot)(wRisingEdgeCnt, dwSinus, dwHfPause));
}

phStatus_t phdlRdFpga_V6_LabelResetWithCommand(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwSinus
    )
{
    /* label reset with command */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LabelResetWithCommand)(dwSinus));
}

phStatus_t phdlRdFpga_V6_ActivateFDTMeasurement(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /* activate FDT measurement */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateFDTMeasurement)());
}

phStatus_t phdlRdFpga_V6_SetConfig32(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwIdentifier,
    uint32_t dwValue,
    uint8_t bParamSetMode
    )
{
    /*Set Param*/
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetParam) ( dwIdentifier ,dwValue , bParamSetMode));
}

phStatus_t phdlRdFpga_V6_ActivateFastExchange(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /*Set Fast Exchange Mode*/
     return phdlRdFpga_V6_SetConfig32(pDataParams,(uint32_t) PH_RDL_FPGA_FAST_EXCHANGE_MODE, (uint32_t) 0x01 , (uint8_t)PH_RDL_PARAM_SETMODE_NORMAL);
}

phStatus_t phdlRdFpga_V6_GetFDT(
                                phdlRdFpga_V6_DataParams_t * pDataParams,
                                uint32_t * pdwFdt
                                )
{
    /* get FDT */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetFDT)(pdwFdt));
}

phStatus_t phdlRdFpga_V6_GetFDTFast(
                                phdlRdFpga_V6_DataParams_t * pDataParams,
                                uint32_t * pdwFdt
                                )
{
    /* get FDT */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetFDTFast)(pdwFdt));
}

phStatus_t phdlRdFpga_V6_ActivateAntiTearingMode(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwSinus,
    uint32_t dwHfPause
    )
{
    /* activate Anti tearing mode */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateAntiTearingMode)(dwSinus, dwHfPause));
}

phStatus_t phdlRdFpga_V6_ResFreqTest(
                                     phdlRdFpga_V6_DataParams_t * pDataParams,
                                     uint32_t dwHfPause,
                                     uint8_t bRepeatFreqId
                                     )
{
    uint16_t   PH_MEMLOC_REM wRepeatFreqId;

    /* convert input, simple cast in function call is not enough.... */
    wRepeatFreqId = (uint16_t) bRepeatFreqId;

    /* resonance Freq Test */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ResonanceFrequenceTest)(dwHfPause, wRepeatFreqId));
}

phStatus_t phdlRdFpga_V6_ActivateFDTMeasurementPICCtoPCD(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwTime,
    uint8_t bCheckResp,
    uint16_t wRxByteCnt,
    uint8_t bRxBitCnt,
    uint8_t * pRxExpected
    )
{
    /* activate FDT measurement PICC to PCD */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateFDTMeasurementPICCtoPCD)(dwTime, bCheckResp, wRxByteCnt, bRxBitCnt, pRxExpected));
}

phStatus_t phdlRdFpga_V6_ActivateLMAMeasurement(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /* activate LMA measurement */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateLMAMeasurement)());
}

phStatus_t phdlRdFpga_V6_ActivateSequencerMode(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /* activate sequencer mode */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateSequencerMode)());
}


phStatus_t phdlRdFpga_V6_DeactivateModes(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /* deactivate modes */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_DeactivateModes)());
}


phStatus_t phdlRdFpga_V6_SetJoinerRxGain(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint8_t bValue
    )
{
    /* set joiner RX gain */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetJoinerRxGain)(bValue));
}

phStatus_t phdlRdFpga_V6_GetJoinerRxGain(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint8_t * pbValue
    )
{
    /* get joiner Rx gain */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetJoinerRxGain)(pbValue));
}

phStatus_t phdlRdFpga_V6_SetFDTPCDforActiveDS(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwValue
    )
{
    /* set FDT for active DS */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetFDTPCDforActiveDS)(dwValue));
}

phStatus_t phdlRdFpga_V6_SetFDTPCDforDS(
                                        phdlRdFpga_V6_DataParams_t * pDataParams,
                                        uint32_t dwValue,
                                        uint8_t bId
                                        )
{
    /* set FDT for DS */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetFDTPCDforDS)(dwValue, bId));
}

phStatus_t phdlRdFpga_V6_GetConfig(
                                   phdlRdFpga_V6_DataParams_t * pDataParams,
                                   uint16_t wIdentifier,
                                   uint16_t * pwValue
                                   )
{
    switch(wIdentifier)
    {
    case PHDL_RDFPGA_CONFIG_ADDITIONAL_INFO:
        *pwValue = pDataParams->wAdditionalInfo;
        break;

    case PHDL_RDFPGA_CONFIG_MIN_POTI:
        *pwValue = PHDL_RDFPGA_V6_POTI_VAL_MIN;
        break;

    case PHDL_RDFPGA_CONFIG_MAX_POTI:
        *pwValue = PHDL_RDFPGA_V6_POTI_VAL_MAX;
        break;

    case PHDL_RDFPGA_CONFIG_SNR_LIMIT_POTI:
        *pwValue = PHDL_RDFPGA_V6_POTI_VAL_SNR_LIMT;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_RDFPGA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_RDFPGA);
}

phStatus_t phdlRdFpga_V6_GetVersion(
                                    phdlRdFpga_V6_DataParams_t * pDataParams,
                                    uint8_t * pbValue
                                    )
{
    /* get FPGA version */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetVersion)(pbValue));
}

phStatus_t phdlRdFpga_V6_GetRevision(
                                     phdlRdFpga_V6_DataParams_t * pDataParams,
                                     uint8_t * pbValue
                                     )
{
    /* get FPGA revision */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetRevision)(pbValue));
}

phStatus_t phdlRdFpga_V6_GetErrorCodes(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint8_t * pbErrors,
                                       uint8_t bMaxErrorCnt,
                                       uint8_t * pbErrorCnt
                                       )
{
    /* get error codes from FPGA */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetErrorCodes)(pbErrors,bMaxErrorCnt, pbErrorCnt));
}

phStatus_t phdlRdFpga_V6_SetClockSource(
                                        phdlRdFpga_V6_DataParams_t * pDataParams,
                                        uint16_t wValue
                                        )
{
    /* set FPGA clock source */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetClockSource)(wValue));
}

phStatus_t phdlRdFpga_V6_GetFirstResponse(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint8_t * pbRxBuffer,
    uint16_t wRxBufferSize,
    uint16_t * pwRxBytes,
    uint8_t * pbNumValidRxBits,
    int16_t * pwError
    )
{
    /* get error first responce for FDT PICC to PCD */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetFirstResponse)(pbRxBuffer, wRxBufferSize, pwRxBytes, pbNumValidRxBits, pwError));
}

phStatus_t phdlRdFpga_V6_GetResponse(
                                     phdlRdFpga_V6_DataParams_t * pDataParams,
                                     uint8_t * pbRxBuffer,
                                     uint16_t wRxBufferSize,
                                     uint16_t * pwRxBytes,
                                     uint8_t * pbNumValidRxBits,
                                     int16_t * pwError
                                     )
{
    /* get error responce */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetResponse)(pbRxBuffer, wRxBufferSize, pwRxBytes, pbNumValidRxBits, pwError));
}

phStatus_t phdlRdFpga_V6_SetIQTimeoutValue(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t dwValue
    )
{
    /* set IQ timeout value */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_SetIQTimeoutValue)(dwValue));
}

phStatus_t phdlRdFpga_V6_GetIQTimeoutValue(
    phdlRdFpga_V6_DataParams_t * pDataParams,
    uint32_t * pdwValue
    )
{
    /* get IQ timeout value */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetIQTimeoutValue)(pdwValue));
}

phStatus_t phdlRdFpga_V6_GetTerminalId(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint8_t * pbValue
                                       )
{
    /* get terminal ID */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetTerminalId)(pbValue));
}

phStatus_t phdlRdFpga_V6_ResetJoiner(
                                     phdlRdFpga_V6_DataParams_t * pDataParams
                                     )
{
    /* reset joiner */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ResetJoiner)());
}

phStatus_t phdlRdFpga_V6_GetConfig32(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint32_t dwIdentifier,
                                       uint32_t * dwValue
                                       )
{
    /*GetParam */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_GetParam)(dwIdentifier,dwValue));
}

phStatus_t phdlRdFpga_V6_LoadFilterFile(
                                       phdlRdFpga_V6_DataParams_t * pDataParams,
                                       uint8_t * pFilename
                                       )
{
    /* load Filter File */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_LoadFilterFile)(pFilename));
}

phStatus_t phdlRdFpga_V6_ExchangeFPGACommand(
                                                phdlRdFpga_V6_DataParams_t * pDataParams,
                                                uint8_t * txData,
                                                uint32_t txLen,
                                                uint8_t * rxData,
                                                uint32_t maxRxLen,
                                                uint32_t * rxLen
                                                )
{
    /* Exchange FPGA Command */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ExchangeFPGACommand)(txData,txLen,rxData,maxRxLen,rxLen));
}

phStatus_t phdlRdFpga_V6_ActivateTypeBMode(
    phdlRdFpga_V6_DataParams_t * pDataParams
    )
{
    /* activate type B mode */
    return phdlRdFpga_Int_ConvertError(pDataParams, (pDataParams->pBalRegDataParams->pfRdFpga_ActivateTypeBMode)());
}

#endif /* NXPBUILD__PHDL_RDFPGA_HW */
