/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Spectrum Analyzer R&S FSV Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_SPA_FSV

#include <phdlSPA.h>
#include "phdlSPA_FSV.h"
#include "phdlSPA_FSV_Int.h"
#include "../phdlSPA_Int.h"

#include <stdio.h>

phStatus_t phdlSPA_FSV_Init(
                                phdlSPA_FSV_DataParams_t * pDataParams,
                                uint16_t wSizeOfDataParams,
                                phbalReg_Visa_DataParams_t * pBalRegDataParams
                                )
{
    if (sizeof(phdlSPA_FSV_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_SPA);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                 = PH_COMP_DL_SPA | PHDL_SPA_FSV_ID;
    pDataParams->pBalRegDataParams   = pBalRegDataParams;
    pDataParams->bMaxFreq            = 0;
    pDataParams->bMaxVideoBandwidth  = 0;
    pDataParams->bUseHeaderForTrace  = PH_OFF;
    memset(pDataParams->bDevId, 0x00, PHDLSPA_SERIAL_NUMBER_SIZE);
    memset(pDataParams->bSwVersion, 0x00, PHDLSPA_SOFTWARE_VERSION_SIZE);
    memset(pDataParams->bDevIdStr, 0x00, PHDLSPA_DEVICE_NUMBER_SIZE);
    pDataParams->wActiveProbeState   = PHDL_SPA_FSV_ACTIVE_PROBE_NOT_QUERIED;
    memset(pDataParams->bActiveProbeName, 0x00, PHDLSPA_ACTIVE_PROBE_NAME_SIZE);
    memset(pDataParams->bActiveProbeSerNum, 0x00, PHDLSPA_ACTIVE_PROBE_SERIAL_ID_SIZE);
    memset(pDataParams->bActiveProbePartId, 0x00, PHDLSPA_ACTIVE_PROBE_PART_ID_SIZE);
    pDataParams->wMaxRfAttValue      = 0;
    pDataParams->qMaxRefLevel        = 0;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_InitSPA(
                                phdlSPA_FSV_DataParams_t * pDataParams
                                )
{
    phStatus_t statusTmp, statusCal;
    uint8_t aRespBuff[100] = {0};
    uint8_t bCalTrue;
    uint32_t wSelfTestResult, wCalibrationRes;
    uint8_t aCalRespBuff[3800] = {'O'};
    uint16_t wReadTimeout = 0;

    /*Cancel all current measuremnts*/
    phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"ABORt", PH_ON);

    /* reset the instrument */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"SYSTem:ERRor:CLEar:ALL", PH_ON));

    /* reset the instrument */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"*RST\n", PH_ON));

    /* Activate Display Update*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"SYSTem:DISPlay:UPDate ON", PH_ON));

    /* Set attenuation and reference level to max*/
    phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *) "INPut:ATTenuation MAX", PH_ON);
    phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *) "DISPlay:TRACe:Y:RLEVel MAX", PH_ON);

    /* clear all status bits */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"*CLS\n", PH_ON));

    /* identify the instrument */
    /* Set the identify string to default output format */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"SYSTem:IDENtify:FACTory\n", PH_ON));

    /* Get instrument identification */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pDataParams->pBalRegDataParams, (uint8_t *)"*IDN?\n", aRespBuff, sizeof(aRespBuff)));
    if(strlen((const char *) aRespBuff) < 5)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_DL_SPA);
    }

    /* Get device information*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_FindInstrumentId(pDataParams, (int8_t *) aRespBuff, sizeof(aRespBuff), (int8_t *)","));

    /* Switches instrument to remote mode set to OFF for debugging*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Send(pDataParams->pBalRegDataParams, (uint8_t *)"SYSTem:KLOCk OFF", PH_ON));

    /* Perform instrument self test higher readtimeout for FSV series*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wReadTimeout));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 30000));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetUIntWithoutRespLeng(pDataParams->pBalRegDataParams, (uint8_t *)"*TST?\n", &wSelfTestResult));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wReadTimeout));
    if(wSelfTestResult != PH_OFF)
    {
        return PH_ADD_COMPCODE(PHDL_SPA_ERR_SELF_TEST_ERROR, PH_COMP_DL_SPA);
    }

    /* Read the calibration status and date*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_ExchangeGetResp(pDataParams->pBalRegDataParams, (uint8_t *)"CALibration:RESult?\n", aCalRespBuff, sizeof(aCalRespBuff)));
    /* Check calibration status*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetInstrumentCalData((uint8_t *)aCalRespBuff, (uint16_t)sizeof(aCalRespBuff), (uint8_t *)"PASSED", 9, (uint8_t *)",", &bCalTrue));
    if(bCalTrue != PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wReadTimeout));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 30000));
        statusCal = phdlSPA_Int_ExchangeGetUIntWithoutRespLeng(pDataParams->pBalRegDataParams, (uint8_t *)"CAL?\n", &wCalibrationRes);
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wReadTimeout));
        PH_CHECK_SUCCESS(statusCal);
    }

    /* configure global */
    /*return phdlSPA_FSV_Int_ConfigGlobal(pDataParams); */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_SetConfig(
                                phdlSPA_FSV_DataParams_t * pDataParams,
                                uint16_t wIdentifier,
                                uint16_t wValue
                                )
{
    phStatus_t statusTmp;

    switch (wIdentifier)
    {
    case PHDL_SPA_CONFIG_TRIGGER_SOURCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerSource(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_SLOPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerSlope(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_LEVEL_IFPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerLevelIfP(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_LEVEL_EXTERNAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetExtTriggerLevelMV(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_LEVEL_RFPOWER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerLevelRfPower(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_LEVEL_VIDEO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerLevelVideo(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRIGGER_CONDITION_MASKCOND:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerMaskCondition(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSPAMode(pDataParams->pBalRegDataParams, wValue, pDataParams->bDevIdStr));
        break;

    case PHDL_SPA_CONFIG_OUTPUT_FORMAT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetOutputFormat(pDataParams->pBalRegDataParams, wValue, &pDataParams->bUseHeaderForTrace));
        break;

    case PHDL_SPA_CONFIG_FREQUENCY_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetFreqMode(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSweepMode(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSweepType(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_COUNT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSweepCount(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_POINTS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSweepPoints(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_STATE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetAverageState(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_COUNT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetAverageCount(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetAverageType(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_INPUT_COUPLING:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetInputCoupling(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_INPUT_IMPEDANCE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetInputImpedance(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_INPUT_ATTENUATION_VALUE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetInputAttVal(pDataParams->pBalRegDataParams, wValue, pDataParams->wMaxRfAttValue));
        break;

    case PHDL_SPA_CONFIG_INPUT_ATTENUATION_AUTO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetInputAttAuto(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_BANDWIDTH_FILTER_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetBandwidthFilter(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRACE_DETECTOR:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTraceDetectorMode(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_TRACE_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTraceMode(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_POWER_UNIT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetPowerUnit(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_RUN_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetRunMode(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_CONTINUE_MEAS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetRunModeContinue(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_PRESET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_Preset(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_CONFIG_MEASUREMENT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetMeasuremntType(pDataParams->pBalRegDataParams, wValue));
        break;

    case PHDL_SPA_FSV_CONFIG_ACTIVE_PROBE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_FSV_Int_Set_Active_Probe(pDataParams, wValue));
        break;

    case PHDL_SPA_FSV_CONFIG_IQ_MEASUREMENT:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
        break;

    case PHDL_SPA_FSV_CONFIG_ACTIVE_TRACE:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_SPA);
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_GetConfig(
                                phdlSPA_FSV_DataParams_t * pDataParams,
                                uint16_t wIdentifier,
                                uint16_t * pwValue
                                )
{
    phStatus_t statusTmp;

    switch (wIdentifier)
    {
    case PHDL_SPA_CONFIG_TRIGGER_SLOPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetTriggerSlope(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetSPAMode(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_OUTPUT_FORMAT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetOutputFormat(pDataParams->pBalRegDataParams, pwValue, pDataParams->bUseHeaderForTrace));
        break;

    case PHDL_SPA_CONFIG_FREQUENCY_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetFreqMode(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_COUNT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetSweepCount(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_SWEEP_POINTS:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetSweepPoints(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_STATE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetAverageState(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_COUNT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetAverageCount(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_AVERAGE_TYPE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetAverageType(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_INPUT_ATTENUATION_VALUE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetInputAttVal(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_CONFIG_OPERATION_COMPLETED_STATE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetOperationCompleted(pDataParams->pBalRegDataParams, pwValue));
        break;

    case PHDL_SPA_FSV_CONFIG_ACTIVE_PROBE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_FSV_Int_Get_Active_Probe_Info(pDataParams));
        *pwValue = pDataParams->wActiveProbeState;
        break;

    case PHDL_SPA_CONFIG_POWER_UNIT:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetPowerUnit(pDataParams->pBalRegDataParams, pwValue));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_GetTrace(
                                phdlSPA_FSV_DataParams_t * pDataParams,
                                uint8_t  bChannel,
                                uint32_t dwTraceBufferSize,
                                uint32_t * pdwTraceLength,
                                uint8_t * pbTraceBuffer
                                )
{
    uint8_t bGetTraceCmd[14];
    phStatus_t statusTmp, statusGet;
    uint16_t wReadTimeout = 0;
    int64_t qwTriggerOffset;
    int64_t qwSweepTime;
    uint16_t wSweepPoints;
    int64_t * pConvertPointer;

    /* Write header and resize buffer */
    if (pDataParams->bUseHeaderForTrace == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_GetConfig64(pDataParams, PHDL_SPA_CONFIG64_TRIGGER_OFFSET, &qwTriggerOffset));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_GetConfig64(pDataParams, PHDL_SPA_CONFIG64_SWEEP_TIME, &qwSweepTime));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_GetConfig(pDataParams, PHDL_SPA_CONFIG_SWEEP_POINTS, &wSweepPoints));
        if (dwTraceBufferSize < 3*sizeof(int64_t))
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_DL_SPA);
        }
        pConvertPointer = (int64_t *)&pbTraceBuffer[0];
        *pConvertPointer = qwTriggerOffset;
        pConvertPointer++;
        *pConvertPointer = qwSweepTime;
        pConvertPointer++;
        *pConvertPointer = (int64_t)wSweepPoints;
        pbTraceBuffer += 3*sizeof(int64_t);
        dwTraceBufferSize -= 3*sizeof(int64_t);
    }

    sprintf_s((char *)bGetTraceCmd, sizeof(bGetTraceCmd), "TRACe? TRACE%d", bChannel);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wReadTimeout));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 10000));

    statusGet = phdlSPA_Int_BalReg_VisaExchangeLarge(pDataParams, (uint8_t *)bGetTraceCmd, (uint16_t)sizeof(bGetTraceCmd),
                                                                              dwTraceBufferSize, pbTraceBuffer, pdwTraceLength);
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wReadTimeout));

    PH_CHECK_SUCCESS(statusGet);

    /* Correct trace length */
    if (pDataParams->bUseHeaderForTrace == PH_ON)
    {
        *pdwTraceLength += 3*sizeof(int64_t);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}
/* Extended set config for center frequency settings */
phStatus_t phdlSPA_FSV_SetConfig64(
                                    phdlSPA_FSV_DataParams_t * pDataParams,
                                    uint16_t wIdentifier,
                                    int64_t qValue
                                   )
{
    phStatus_t statusTmp;
    uint64_t qMaxFreq = 0;

    switch (wIdentifier)
    {
    case PHDL_SPA_CONFIG64_CENTER_FREQUENCY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_FSV_Int_Get_Max_Frequency(pDataParams, &qMaxFreq));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetCenterFreq_HZ(pDataParams->pBalRegDataParams, qValue, qMaxFreq));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_SPAN:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetFreqSpan_HZ(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_OFFSET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetFreqOffSet_HZ(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_START:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetFreqStart_HZ(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_STOP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetFreqStop_HZ(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION_VIDEO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetBandwidthResVideo_HZ(pDataParams->pBalRegDataParams, qValue, pDataParams->bMaxVideoBandwidth));
        break;

    case PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetBandwidthRes_HZ(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_TRIGGER_OFFSET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetTriggerHoldoff_NS(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_SWEEP_TIME:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetSweepTime_NS(pDataParams->pBalRegDataParams, qValue));
        break;

    case PHDL_SPA_CONFIG64_REFERENCE_LEVEL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetReferenceLevel(pDataParams->pBalRegDataParams, qValue, pDataParams->qMaxRefLevel));
        break;

    case PHDL_SPA_CONFIG64_REFERENCE_LEVEL_OFFSET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_SetReferenceLevelOffset(pDataParams->pBalRegDataParams, qValue));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_GetConfig64(
                                phdlSPA_FSV_DataParams_t * pDataParams,
                                uint16_t wIdentifier,
                                int64_t * pqValue
                                )
{
    phStatus_t statusTmp;

    switch (wIdentifier)
    {
    case PHDL_SPA_CONFIG64_CENTER_FREQUENCY:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetCenterFreq_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_SPAN:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetFreqSpan_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_START:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetFreqStart_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_FREQUENCY_STOP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetFreqStop_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetBandwidthRes_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_BANDWIDTH_RESOLUTION_VIDEO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetBandwidthResVideo_HZ(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_TRIGGER_OFFSET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetTriggerHoldoff_NS(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_SWEEP_TIME:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetSweepTime_NS(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_REFERENCE_LEVEL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetReferenceLevel(pDataParams->pBalRegDataParams, pqValue));
        break;

    case PHDL_SPA_CONFIG64_REFERENCE_LEVEL_OFFSET:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlSPA_Int_GetReferenceLevelOffset(pDataParams->pBalRegDataParams, pqValue));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_SPA);
}

phStatus_t phdlSPA_FSV_GetMeasurement(
                                      phdlSPA_FSV_DataParams_t * pDataParams,
                                      uint16_t wMeasureSourceType,
                                      float64_t * pfValue
                                     )
{
    uint8_t bGetTraceCmd[47];
    char measureSourceType[5];

    switch (wMeasureSourceType)
    {
    case PHDL_SPA_MEASURE_TYPE_PEAK:
        sprintf(measureSourceType, "PPE");
        break;

    case PHDL_SPA_MEASURE_TYPE_MEAN:
        sprintf(measureSourceType, "MEAN");
        break;

    case PHDL_SPA_MEASURE_TYPE_RMS:
        sprintf(measureSourceType, "RMS");
        break;

    case PHDL_SPA_MEASURE_TYPE_SDEV:
        sprintf(measureSourceType, "SDEV");
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_SPA);
    }

    sprintf_s((char *)bGetTraceCmd, sizeof(bGetTraceCmd), "CALCulate:MARKer:FUNCtion:SUMMary:%s:RESult?", measureSourceType);

    return phdlSPA_Int_GetMeasurementData(pDataParams->pBalRegDataParams, bGetTraceCmd, pfValue);
}

#endif /* NXPBUILD__PHDL_SPA_FSV */
